#!/usr/bin/python
#
# This file is part of Ansible
#
# Ansible is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Ansible is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Ansible.  If not, see <http://www.gnu.org/licenses/>.
#

ANSIBLE_METADATA = {'metadata_version': '1.1',
                    'status': ['preview'],
                    'supported_by': 'network'}

DOCUMENTATION = '''
---
module: nxos_vrf_af
extends_documentation_fragment: nxos
version_added: "2.2"
short_description: Manages VRF AF.
description:
  - Manages VRF AF
author: Gabriele Gerbino (@GGabriele)
notes:
  - Tested against NXOSv 7.3.(0)D1(1) on VIRL
  - Default, where supported, restores params default value.
options:
  vrf:
    description:
      - Name of the VRF.
    required: true
  afi:
    description:
      - Address-Family Identifier (AFI).
    required: true
    choices: ['ipv4', 'ipv6']
    default: null
  safi:
    description:
      - Sub Address-Family Identifier (SAFI).
      - Deprecated in 2.4
    required: true
    choices: ['unicast', 'multicast']
    default: null
  route_target_both_auto_evpn:
    description:
      - Enable/Disable the EVPN route-target 'auto' setting for both
        import and export target communities.
    required: false
    choices: ['true', 'false']
    default: null
  state:
    description:
      - Determines whether the config should be present or
        not on the device.
    required: false
    default: present
    choices: ['present','absent']
'''

EXAMPLES = '''
- nxos_vrf_af:
    vrf: ntc
    afi: ipv4
    route_target_both_auto_evpn: True
    state: present
'''

RETURN = '''
commands:
    description: commands sent to the device
    returned: always
    type: list
    sample: ["vrf context ntc", "address-family ipv4 unicast"]
'''
from ansible.module_utils.nxos import get_config, load_config
from ansible.module_utils.nxos import nxos_argument_spec, check_args
from ansible.module_utils.basic import AnsibleModule
from ansible.module_utils.netcfg import NetworkConfig


def main():
    argument_spec = dict(
        vrf=dict(required=True),
        afi=dict(required=True, choices=['ipv4', 'ipv6']),
        route_target_both_auto_evpn=dict(required=False, type='bool'),
        state=dict(choices=['present', 'absent'], default='present'),

        m_facts=dict(default=False, type='bool', removed_in_version="2.4"),
        safi=dict(choices=['unicast', 'multicast'], removed_in_version="2.4"),
    )

    argument_spec.update(nxos_argument_spec)

    module = AnsibleModule(argument_spec=argument_spec, supports_check_mode=True)

    warnings = list()
    check_args(module, warnings)

    result = {'changed': False, 'warnings': warnings}

    config_text = get_config(module)
    config = NetworkConfig(indent=2, contents=config_text)

    path = ['vrf context %s' % module.params['vrf'],
            'address-family %s unicast' % module.params['afi']]

    try:
        current = config.get_block_config(path)
    except ValueError:
        current = None

    commands = list()
    if current and module.params['state'] == 'absent':
        commands.append('no address-family %s unicast' % module.params['afi'])

    elif module.params['state'] == 'present':

        if current:
            have = 'route-target both auto evpn' in current
            want = bool(module.params['route_target_both_auto_evpn'])

            if want and not have:
                commands.append('address-family %s unicast' % module.params['afi'])
                commands.append('route-target both auto evpn')
            elif have and not want:
                commands.append('address-family %s unicast' % module.params['afi'])
                commands.append('no route-target both auto evpn')

        else:
            commands.append('address-family %s unicast' % module.params['afi'])
            if module.params['route_target_both_auto_evpn']:
                commands.append('route-target both auto evpn')

    if commands:
        commands.insert(0, 'vrf context %s' % module.params['vrf'])
        if not module.check_mode:
            load_config(module, commands)
        result['changed'] = True

    result['commands'] = commands

    module.exit_json(**result)


if __name__ == '__main__':
    main()
