# -*- coding: utf-8 -*-

# Copyright (c) 2014 - 2018 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the translation engine base class.
"""

from __future__ import unicode_literals

from PyQt5.QtCore import QObject


class TranslationEngine(QObject):
    """
    Class implementing the translation engine base class containing
    default methods.
    """
    def __init__(self, plugin, parent=None):
        """
        Constructor
        
        @param plugin reference to the plugin object (TranslatorPlugin)
        @param parent reference to the parent object (QObject)
        """
        super(TranslationEngine, self).__init__(parent)
        
        self.plugin = plugin
    
    def engineName(self):
        """
        Public method to get the name of the engine.
        
        @return engine name (string)
        """
        return ""
    
    def supportedLanguages(self):
        """
        Public method to get the supported languages.
        
        @return list of supported language codes (list of string)
        """
        return []
    
    def hasTTS(self):
        """
        Public method indicating the Text-to-Speech capability.
        
        @return flag indicating the Text-to-Speech capability (boolean)
        """
        return False
    
    def getTextToSpeechData(self, requestObject, text, language):
        """
        Public method to pronounce the given text.
        
        @param requestObject reference to the request object
            (TranslatorRequest)
        @param text text to be pronounced (string)
        @param language language code of the text (string)
        @return tuple with pronounce data (QByteArray) or error string (string)
            and success flag (boolean)
        """
        return self.tr("No pronounce data available"), False
    
    def getTranslation(self, requestObject, text, originalLanguage,
                       translationLanguage):
        """
        Public method to translate the given text.
        
        @param requestObject reference to the request object
            (TranslatorRequest)
        @param text text to be translated (string)
        @param originalLanguage language code of the original (string)
        @param translationLanguage language code of the translation (string)
        @return tuple of translated text (string) and flag indicating
            success (boolean)
        """
        return self.tr("No translation available"), False
