/*
 * Copyright 2010-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package kotlin.reflect.jvm.internal.impl.builtins.functions

import kotlin.reflect.jvm.internal.impl.builtins.BuiltInsPackageFragment
import kotlin.reflect.jvm.internal.impl.builtins.functions.FunctionClassDescriptor.Kind
import kotlin.reflect.jvm.internal.impl.descriptors.ClassDescriptor
import kotlin.reflect.jvm.internal.impl.descriptors.ModuleDescriptor
import kotlin.reflect.jvm.internal.impl.descriptors.deserialization.ClassDescriptorFactory
import kotlin.reflect.jvm.internal.impl.name.ClassId
import kotlin.reflect.jvm.internal.impl.name.FqName
import kotlin.reflect.jvm.internal.impl.name.Name
import kotlin.reflect.jvm.internal.impl.storage.StorageManager

/**
 * Produces descriptors representing the fictitious classes for function types, such as kotlin.Function1 or kotlin.reflect.KFunction2.
 */
class BuiltInFictitiousFunctionClassFactory(
        private val storageManager: StorageManager,
        private val module: ModuleDescriptor
) : ClassDescriptorFactory {

    private data class KindWithArity(val kind: Kind, val arity: Int)

    companion object {
        private fun parseClassName(className: String, packageFqName: FqName): KindWithArity? {
            val kind = FunctionClassDescriptor.Kind.byClassNamePrefix(packageFqName, className) ?: return null

            val prefix = kind.classNamePrefix

            val arity = toInt(className.substring(prefix.length)) ?: return null

            // TODO: validate arity, should be <= 255
            return KindWithArity(kind, arity)
        }

        @JvmStatic
        fun getFunctionalClassKind(className: String, packageFqName: FqName) =
                parseClassName(className, packageFqName)?.kind

        private fun toInt(s: String): Int? {
            if (s.isEmpty()) return null

            var result = 0
            for (c in s) {
                val d = c - '0'
                if (d !in 0..9) return null
                result = result * 10 + d
            }
            return result
        }
    }

    override fun shouldCreateClass(packageFqName: FqName, name: Name): Boolean {
        val string = name.asString()
        return (string.startsWith("Function") || string.startsWith("KFunction")) // an optimization
               && parseClassName(string, packageFqName) != null
    }

    override fun createClass(classId: ClassId): ClassDescriptor? {
        if (classId.isLocal || classId.isNestedClass) return null

        val className = classId.relativeClassName.asString()
        if ("Function" !in className) return null // An optimization

        val packageFqName = classId.packageFqName
        val (kind, arity) = parseClassName(className, packageFqName) ?: return null

        // SuspendFunction$n can't be created by classId
        if (kind == Kind.SuspendFunction) return null

        val containingPackageFragment = module.getPackage(packageFqName).fragments.filterIsInstance<BuiltInsPackageFragment>().first()

        return FunctionClassDescriptor(storageManager, containingPackageFragment, kind, arity)
    }

    override fun getAllContributedClassesIfPossible(packageFqName: FqName): Collection<ClassDescriptor> {
        // We don't want to return 256 classes here since it would cause them to appear in every import list of every file
        // and likely slow down compilation very much
        return emptySet()
    }
}
