<?php

/**
 * @group Database
 * @covers Parser
 */
class ParserMethodsTest extends MediaWikiLangTestCase {

	public static function providePreSaveTransform() {
		return [
			[ 'hello this is ~~~',
				"hello this is [[Special:Contributions/127.0.0.1|127.0.0.1]]",
			],
			[ 'hello \'\'this\'\' is <nowiki>~~~</nowiki>',
				'hello \'\'this\'\' is <nowiki>~~~</nowiki>',
			],
		];
	}

	/**
	 * @dataProvider providePreSaveTransform
	 */
	public function testPreSaveTransform( $text, $expected ) {
		global $wgParser;

		$title = Title::newFromText( str_replace( '::', '__', __METHOD__ ) );
		$user = new User();
		$user->setName( "127.0.0.1" );
		$popts = ParserOptions::newFromUser( $user );
		$text = $wgParser->preSaveTransform( $text, $title, $user, $popts );

		$this->assertEquals( $expected, $text );
	}

	public static function provideStripOuterParagraph() {
		// This mimics the most common use case (stripping paragraphs generated by the parser).
		$message = new RawMessage( "Message text." );

		return [
			[
				"<p>Text.</p>",
				"Text.",
			],
			[
				"<p class='foo'>Text.</p>",
				"<p class='foo'>Text.</p>",
			],
			[
				"<p>Text.\n</p>\n",
				"Text.",
			],
			[
				"<p>Text.</p><p>More text.</p>",
				"<p>Text.</p><p>More text.</p>",
			],
			[
				$message->parse(),
				"Message text.",
			],
		];
	}

	/**
	 * @dataProvider provideStripOuterParagraph
	 */
	public function testStripOuterParagraph( $text, $expected ) {
		$this->assertEquals( $expected, Parser::stripOuterParagraph( $text ) );
	}

	/**
	 * @expectedException MWException
	 * @expectedExceptionMessage Parser state cleared while parsing.
	 *  Did you call Parser::parse recursively?
	 */
	public function testRecursiveParse() {
		global $wgParser;
		$title = Title::newFromText( 'foo' );
		$po = new ParserOptions;
		$wgParser->setHook( 'recursivecallparser', [ $this, 'helperParserFunc' ] );
		$wgParser->parse( '<recursivecallparser>baz</recursivecallparser>', $title, $po );
	}

	public function helperParserFunc( $input, $args, $parser ) {
		$title = Title::newFromText( 'foo' );
		$po = new ParserOptions;
		$parser->parse( $input, $title, $po );
		return 'bar';
	}

	public function testCallParserFunction() {
		global $wgParser;

		// Normal parses test passing PPNodes. Test passing an array.
		$title = Title::newFromText( str_replace( '::', '__', __METHOD__ ) );
		$wgParser->startExternalParse( $title, new ParserOptions(), Parser::OT_HTML );
		$frame = $wgParser->getPreprocessor()->newFrame();
		$ret = $wgParser->callParserFunction( $frame, '#tag',
			[ 'pre', 'foo', 'style' => 'margin-left: 1.6em' ]
		);
		$ret['text'] = $wgParser->mStripState->unstripBoth( $ret['text'] );
		$this->assertSame( [
			'found' => true,
			'text' => '<pre style="margin-left: 1.6em">foo</pre>',
		], $ret, 'callParserFunction works for {{#tag:pre|foo|style=margin-left: 1.6em}}' );
	}

	/**
	 * @covers Parser
	 * @covers ParserOutput::getSections
	 */
	public function testGetSections() {
		global $wgParser;

		$title = Title::newFromText( str_replace( '::', '__', __METHOD__ ) );
		$out = $wgParser->parse( "==foo==\n<h2>bar</h2>\n==baz==\n", $title, new ParserOptions() );
		$this->assertSame( [
			[
				'toclevel' => 1,
				'level' => '2',
				'line' => 'foo',
				'number' => '1',
				'index' => '1',
				'fromtitle' => $title->getPrefixedDBkey(),
				'byteoffset' => 0,
				'anchor' => 'foo',
			],
			[
				'toclevel' => 1,
				'level' => '2',
				'line' => 'bar',
				'number' => '2',
				'index' => '',
				'fromtitle' => false,
				'byteoffset' => null,
				'anchor' => 'bar',
			],
			[
				'toclevel' => 1,
				'level' => '2',
				'line' => 'baz',
				'number' => '3',
				'index' => '2',
				'fromtitle' => $title->getPrefixedDBkey(),
				'byteoffset' => 21,
				'anchor' => 'baz',
			],
		], $out->getSections(), 'getSections() with proper value when <h2> is used' );
	}

	/**
	 * @dataProvider provideNormalizeLinkUrl
	 */
	public function testNormalizeLinkUrl( $explanation, $url, $expected ) {
		$this->assertEquals( $expected, Parser::normalizeLinkUrl( $url ), $explanation );
	}

	public static function provideNormalizeLinkUrl() {
		return [
			[
				'Escaping of unsafe characters',
				'http://example.org/foo bar?param[]="value"&param[]=valüe',
				'http://example.org/foo%20bar?param%5B%5D=%22value%22&param%5B%5D=val%C3%BCe',
			],
			[
				'Case normalization of percent-encoded characters',
				'http://example.org/%ab%cD%Ef%FF',
				'http://example.org/%AB%CD%EF%FF',
			],
			[
				'Unescaping of safe characters',
				'http://example.org/%3C%66%6f%6F%3E?%3C%66%6f%6F%3E#%3C%66%6f%6F%3E',
				'http://example.org/%3Cfoo%3E?%3Cfoo%3E#%3Cfoo%3E',
			],
			[
				'Context-sensitive replacement of sometimes-safe characters',
				'http://example.org/%23%2F%3F%26%3D%2B%3B?%23%2F%3F%26%3D%2B%3B#%23%2F%3F%26%3D%2B%3B',
				'http://example.org/%23%2F%3F&=+;?%23/?%26%3D%2B%3B#%23/?&=+;',
			],
		];
	}

	// @todo Add tests for cleanSig() / cleanSigInSig(), getSection(),
	// replaceSection(), getPreloadText()
}
