#ifndef Magnum_Trade_AnyImageConverter_h
#define Magnum_Trade_AnyImageConverter_h
/*
    This file is part of Magnum.

    Copyright © 2010, 2011, 2012, 2013, 2014, 2015, 2016, 2017, 2018
              Vladimír Vondruš <mosra@centrum.cz>

    Permission is hereby granted, free of charge, to any person obtaining a
    copy of this software and associated documentation files (the "Software"),
    to deal in the Software without restriction, including without limitation
    the rights to use, copy, modify, merge, publish, distribute, sublicense,
    and/or sell copies of the Software, and to permit persons to whom the
    Software is furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included
    in all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
    THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
    DEALINGS IN THE SOFTWARE.
*/

/** @file
 * @brief Class @ref Magnum::Trade::AnyImageConverter
 */

#include <Magnum/Trade/AbstractImageConverter.h>

#include "MagnumPlugins/AnyImageConverter/configure.h"

#ifndef DOXYGEN_GENERATING_OUTPUT
#ifndef MAGNUM_ANYIMAGECONVERTER_BUILD_STATIC
    #if defined(AnyImageConverter_EXPORTS) || defined(AnyImageConverterObjects_EXPORTS)
        #define MAGNUM_ANYIMAGECONVERTER_EXPORT CORRADE_VISIBILITY_EXPORT
    #else
        #define MAGNUM_ANYIMAGECONVERTER_EXPORT CORRADE_VISIBILITY_IMPORT
    #endif
#else
    #define MAGNUM_ANYIMAGECONVERTER_EXPORT CORRADE_VISIBILITY_STATIC
#endif
#define MAGNUM_ANYIMAGECONVERTER_LOCAL CORRADE_VISIBILITY_LOCAL
#else
#define MAGNUM_ANYIMAGECONVERTER_EXPORT
#define MAGNUM_ANYIMAGECONVERTER_LOCAL
#endif

namespace Magnum { namespace Trade {

/**
@brief Any image converter plugin

Detects file type based on file extension, loads corresponding plugin and then
tries to convert the file with it.

This plugin depends on the @ref Trade library and is built if
`WITH_ANYIMAGECONVERTER` is enabled when building Magnum Plugins. To use as a
dynamic plugin, you need to load the @cpp "AnyImageConverter" @ce plugin from
`MAGNUM_PLUGINS_IMPORTER_DIR`. To use as a static plugin or as a dependency of
another plugin with CMake, you need to request the `AnyImageConverter`
component of the `MagnumPlugins` package and link to the
`MagnumPlugins::AnyImageConverter` target. See @ref building-plugins,
@ref cmake-plugins and @ref plugins for more information.

Supported formats for uncompressed data:

-   OpenEXR (`*.exr`), converted with any plugin that provides
    `OpenExrImageConverter`
-   Windows Bitmap (`*.bmp`), converted with any plugin that provides
    `BmpImageConverter`
-   Radiance HDR (`*.hdr`), converted with any plugin that provides
    `HdrImageConverter`
-   Portable Network Graphics (`*.png`), converted with @ref PngImageConverter
    or any other plugin that provides it
-   Truevision TGA (`*.tga`, `*.vda`, `*.icb`, `*.vst`), converted with
    @ref TgaImageConverter or any other plugin that provides it

No supported formats for compressed data yet.

Only exporting to files is supported.
*/
class MAGNUM_ANYIMAGECONVERTER_EXPORT AnyImageConverter: public AbstractImageConverter {
    public:
        /** @brief Constructor with access to plugin manager */
        explicit AnyImageConverter(PluginManager::Manager<AbstractImageConverter>& manager);

        /** @brief Plugin manager constructor */
        explicit AnyImageConverter(PluginManager::AbstractManager& manager, const std::string& plugin);

        ~AnyImageConverter();

    private:
        MAGNUM_ANYIMAGECONVERTER_LOCAL Features doFeatures() const override;
        MAGNUM_ANYIMAGECONVERTER_LOCAL bool doExportToFile(const ImageView2D& image, const std::string& filename) override;
        MAGNUM_ANYIMAGECONVERTER_LOCAL bool doExportToFile(const CompressedImageView2D& image, const std::string& filename) override;
};

}}

#endif
