/*
 * Copyright 2017 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.gradle.composite.internal;

import org.gradle.api.artifacts.ArtifactCollection;
import org.gradle.api.artifacts.Configuration;
import org.gradle.api.artifacts.component.BuildIdentifier;
import org.gradle.api.artifacts.component.ComponentArtifactIdentifier;
import org.gradle.api.artifacts.result.ResolvedArtifactResult;
import org.gradle.api.internal.initialization.ScriptClassPathInitializer;
import org.gradle.initialization.BuildIdentity;

import java.util.Set;

public class CompositeBuildClassPathInitializer implements ScriptClassPathInitializer {
    private final IncludedBuildTaskGraph includedBuildTaskGraph;
    private final BuildIdentifier currentBuild;

    public CompositeBuildClassPathInitializer(IncludedBuildTaskGraph includedBuildTaskGraph, BuildIdentity buildIdentity) {
        this.includedBuildTaskGraph = includedBuildTaskGraph;
        this.currentBuild = buildIdentity.getCurrentBuild();
    }

    @Override
    public void execute(Configuration classpath) {
        ArtifactCollection artifacts = classpath.getIncoming().getArtifacts();
        for (ResolvedArtifactResult artifactResult : artifacts.getArtifacts()) {
            ComponentArtifactIdentifier componentArtifactIdentifier = artifactResult.getId();
            build(currentBuild, componentArtifactIdentifier);
        }
    }

    private BuildIdentifier getBuildIdentifier(CompositeProjectComponentArtifactMetadata artifact) {
        return artifact.getComponentId().getBuild();
    }

    public void build(BuildIdentifier requestingBuild, ComponentArtifactIdentifier artifact) {
        if (artifact instanceof CompositeProjectComponentArtifactMetadata) {
            CompositeProjectComponentArtifactMetadata compositeBuildArtifact = (CompositeProjectComponentArtifactMetadata) artifact;
            BuildIdentifier targetBuild = getBuildIdentifier(compositeBuildArtifact);
            assert !requestingBuild.equals(targetBuild);
            Set<String> tasks = compositeBuildArtifact.getTasks();
            for (String taskName : tasks) {
                includedBuildTaskGraph.addTask(requestingBuild, targetBuild, taskName);
            }
            for (String taskName : tasks) {
                includedBuildTaskGraph.awaitCompletion(targetBuild, taskName);
            }
        }
    }
}
