var os = require('os')
var assert = require('assert')
var allocUnsafe = require('buffer-alloc-unsafe')

var fill = require('./')

var SIZE = 28
var EMPTY = '00000000000000000000000000000000000000000000000000000000'

// Check exceptions
var buf1 = allocUnsafe(SIZE)
assert.throws(function () { fill(buf1, 0, -1) })
assert.throws(function () { fill(buf1, 0, 0, buf1.length + 1) })
assert.throws(function () { fill(buf1, '', -1) })
assert.throws(function () { fill(buf1, '', 0, buf1.length + 1) })
assert.throws(function () { fill(buf1, 'a', 0, buf1.length, 'node rocks!') })
assert.throws(function () { fill(buf1, 'a', 0, 0, NaN) })
assert.throws(function () { fill(buf1, 'a', 0, 0, null) })
assert.throws(function () { fill(buf1, 'a', 0, 0, 'foo') })
assert.throws(function () { fill(buf1, '\u0222', 'hex') })

// Edge cases
var buf2 = allocUnsafe(1)
assert.equal(fill(buf2, 0)[0], 0x00)
assert.equal(fill(buf2, '\u0222')[0], 0xc8)

var buf3 = allocUnsafe(1)
assert.equal(fill(buf3, '\u0222', 'ucs2')[0], os.endianness() === 'LE' ? 0x22 : 0x02)

// Precomputed test cases
var testCases = [
  // Default encoding
  ['61626361626361626361626361626361626361626361626361626361', ['abc']],
  ['c8a26161c8a26161c8a26161c8a26161c8a26161c8a26161c8a26161', ['\u0222aa']],
  ['61c8b462c8b563c8b661c8b462c8b563c8b661c8b462c8b563c8b661', ['a\u0234b\u0235c\u0236']],
  ['00000000616263616263616263616263616263616263616263616263', ['abc', 4]],
  ['00000000006162636162636162636162636162636162636162636162', ['abc', 5]],
  ['00000000000000000000000000000000000000000000000000000000', ['abc', SIZE]],
  ['0000c8a26161c8a26161c8a26161c8a26161c8a26161c8a26161c8a2', ['\u0222aa', 2]],
  ['0000000000000000c8a26161c8a26161c8a26161c8a26161c8a26161', ['\u0222aa', 8]],
  ['0000000061c8b462c8b563c8b661c8b462c8b563c8b661c8b462c8b5', ['a\u0234b\u0235c\u0236', 4]],
  ['00000000000000000000000061c8b462c8b563c8b661c8b462c8b563', ['a\u0234b\u0235c\u0236', 12]],
  ['00000000000000000000000000000000000000000000000000000000', ['abc', 4, -1]],
  ['00000000000000000000000000000000000000000000000000000000', ['abc', 4, 1]],
  ['00000000000000000000000000000000000000000000000000000000', ['abc', 5, 1]],
  ['00000000000000000000000000000000000000000000000000000000', ['\u0222aa', 2, -1]],
  ['00000000000000000000000000000000000000000000000000000000', ['\u0222aa', 8, 1]],
  ['00000000000000000000000000000000000000000000000000000000', ['a\u0234b\u0235c\u0236', 4, -1]],
  ['00000000000000000000000000000000000000000000000000000000', ['a\u0234b\u0235c\u0236', 4, 1]],
  ['00000000000000000000000000000000000000000000000000000000', ['a\u0234b\u0235c\u0236', 12, 1]],

  // UTF8
  ['61626361626361626361626361626361626361626361626361626361', ['abc', 'utf8']],
  ['c8a26161c8a26161c8a26161c8a26161c8a26161c8a26161c8a26161', ['\u0222aa', 'utf8']],
  ['61c8b462c8b563c8b661c8b462c8b563c8b661c8b462c8b563c8b661', ['a\u0234b\u0235c\u0236', 'utf8']],
  ['00000000616263616263616263616263616263616263616263616263', ['abc', 4, 'utf8']],
  ['00000000006162636162636162636162636162636162636162636162', ['abc', 5, 'utf8']],
  ['00000000000000000000000000000000000000000000000000000000', ['abc', SIZE, 'utf8']],
  ['0000c8a26161c8a26161c8a26161c8a26161c8a26161c8a26161c8a2', ['\u0222aa', 2, 'utf8']],
  ['0000000000000000c8a26161c8a26161c8a26161c8a26161c8a26161', ['\u0222aa', 8, 'utf8']],
  ['0000000061c8b462c8b563c8b661c8b462c8b563c8b661c8b462c8b5', ['a\u0234b\u0235c\u0236', 4, 'utf8']],
  ['00000000000000000000000061c8b462c8b563c8b661c8b462c8b563', ['a\u0234b\u0235c\u0236', 12, 'utf8']],
  ['00000000000000000000000000000000000000000000000000000000', ['abc', 4, -1, 'utf8']],
  ['00000000000000000000000000000000000000000000000000000000', ['abc', 4, 1, 'utf8']],
  ['00000000000000000000000000000000000000000000000000000000', ['abc', 5, 1, 'utf8']],
  ['00000000000000000000000000000000000000000000000000000000', ['\u0222aa', 2, -1, 'utf8']],
  ['00000000000000000000000000000000000000000000000000000000', ['\u0222aa', 8, 1, 'utf8']],
  ['00000000000000000000000000000000000000000000000000000000', ['a\u0234b\u0235c\u0236', 4, -1, 'utf8']],
  ['00000000000000000000000000000000000000000000000000000000', ['a\u0234b\u0235c\u0236', 4, 1, 'utf8']],
  ['00000000000000000000000000000000000000000000000000000000', ['a\u0234b\u0235c\u0236', 12, 1, 'utf8']],

  // BINARY
  ['61626361626361626361626361626361626361626361626361626361', ['abc', 'binary']],
  ['22616122616122616122616122616122616122616122616122616122', ['\u0222aa', 'binary']],
  ['61346235633661346235633661346235633661346235633661346235', ['a\u0234b\u0235c\u0236', 'binary']],
  ['00000000616263616263616263616263616263616263616263616263', ['abc', 4, 'binary']],
  ['00000000006162636162636162636162636162636162636162636162', ['abc', 5, 'binary']],
  ['00000000000000000000000000000000000000000000000000000000', ['abc', SIZE, 'binary']],
  ['00002261612261612261612261612261612261612261612261612261', ['\u0222aa', 2, 'binary']],
  ['00000000000000002261612261612261612261612261612261612261', ['\u0222aa', 8, 'binary']],
  ['00000000613462356336613462356336613462356336613462356336', ['a\u0234b\u0235c\u0236', 4, 'binary']],
  ['00000000000000000000000061346235633661346235633661346235', ['a\u0234b\u0235c\u0236', 12, 'binary']],
  ['00000000000000000000000000000000000000000000000000000000', ['abc', 4, -1, 'binary']],
  ['00000000000000000000000000000000000000000000000000000000', ['abc', 4, 1, 'binary']],
  ['00000000000000000000000000000000000000000000000000000000', ['abc', 5, 1, 'binary']],
  ['00000000000000000000000000000000000000000000000000000000', ['\u0222aa', 2, -1, 'binary']],
  ['00000000000000000000000000000000000000000000000000000000', ['\u0222aa', 8, 1, 'binary']],
  ['00000000000000000000000000000000000000000000000000000000', ['a\u0234b\u0235c\u0236', 4, -1, 'binary']],
  ['00000000000000000000000000000000000000000000000000000000', ['a\u0234b\u0235c\u0236', 4, 1, 'binary']],
  ['00000000000000000000000000000000000000000000000000000000', ['a\u0234b\u0235c\u0236', 12, 1, 'binary']],

  // LATIN1
  ['61626361626361626361626361626361626361626361626361626361', ['abc', 'latin1']],
  ['22616122616122616122616122616122616122616122616122616122', ['\u0222aa', 'latin1']],
  ['61346235633661346235633661346235633661346235633661346235', ['a\u0234b\u0235c\u0236', 'latin1']],
  ['00000000616263616263616263616263616263616263616263616263', ['abc', 4, 'latin1']],
  ['00000000006162636162636162636162636162636162636162636162', ['abc', 5, 'latin1']],
  ['00000000000000000000000000000000000000000000000000000000', ['abc', SIZE, 'latin1']],
  ['00002261612261612261612261612261612261612261612261612261', ['\u0222aa', 2, 'latin1']],
  ['00000000000000002261612261612261612261612261612261612261', ['\u0222aa', 8, 'latin1']],
  ['00000000613462356336613462356336613462356336613462356336', ['a\u0234b\u0235c\u0236', 4, 'latin1']],
  ['00000000000000000000000061346235633661346235633661346235', ['a\u0234b\u0235c\u0236', 12, 'latin1']],
  ['00000000000000000000000000000000000000000000000000000000', ['abc', 4, -1, 'latin1']],
  ['00000000000000000000000000000000000000000000000000000000', ['abc', 4, 1, 'latin1']],
  ['00000000000000000000000000000000000000000000000000000000', ['abc', 5, 1, 'latin1']],
  ['00000000000000000000000000000000000000000000000000000000', ['\u0222aa', 2, -1, 'latin1']],
  ['00000000000000000000000000000000000000000000000000000000', ['\u0222aa', 8, 1, 'latin1']],
  ['00000000000000000000000000000000000000000000000000000000', ['a\u0234b\u0235c\u0236', 4, -1, 'latin1']],
  ['00000000000000000000000000000000000000000000000000000000', ['a\u0234b\u0235c\u0236', 4, 1, 'latin1']],
  ['00000000000000000000000000000000000000000000000000000000', ['a\u0234b\u0235c\u0236', 12, 1, 'latin1']],

  // UCS2
  ['61006200630061006200630061006200630061006200630061006200', ['abc', 'ucs2']],
  ['22026100610022026100610022026100610022026100610022026100', ['\u0222aa', 'ucs2']],
  ['61003402620035026300360261003402620035026300360261003402', ['a\u0234b\u0235c\u0236', 'ucs2']],
  ['00000000610062006300610062006300610062006300610062006300', ['abc', 4, 'ucs2']],
  ['00000000000000000000000000000000000000000000000000000000', ['abc', SIZE, 'ucs2']],
  ['00002202610061002202610061002202610061002202610061002202', ['\u0222aa', 2, 'ucs2']],
  ['00000000000000002202610061002202610061002202610061002202', ['\u0222aa', 8, 'ucs2']],
  ['00000000610034026200350263003602610034026200350263003602', ['a\u0234b\u0235c\u0236', 4, 'ucs2']],
  ['00000000000000000000000061003402620035026300360261003402', ['a\u0234b\u0235c\u0236', 12, 'ucs2']],
  ['00000000000000000000000000000000000000000000000000000000', ['abc', 4, -1, 'ucs2']],
  ['00000000000000000000000000000000000000000000000000000000', ['abc', 4, 1, 'ucs2']],
  ['00000000000000000000000000000000000000000000000000000000', ['abc', 5, 1, 'ucs2']],
  ['00000000000000000000000000000000000000000000000000000000', ['\u0222aa', 2, -1, 'ucs2']],
  ['00000000000000000000000000000000000000000000000000000000', ['\u0222aa', 8, 1, 'ucs2']],
  ['00000000000000000000000000000000000000000000000000000000', ['a\u0234b\u0235c\u0236', 4, -1, 'ucs2']],
  ['00000000000000000000000000000000000000000000000000000000', ['a\u0234b\u0235c\u0236', 4, 1, 'ucs2']],
  ['00000000000000000000000000000000000000000000000000000000', ['a\u0234b\u0235c\u0236', 12, 1, 'ucs2']],

  // HEX
  ['61626361626361626361626361626361626361626361626361626361', ['616263', 'hex']],
  ['c8a26161c8a26161c8a26161c8a26161c8a26161c8a26161c8a26161', ['c8a26161', 'hex']],
  ['61c8b462c8b563c8b661c8b462c8b563c8b661c8b462c8b563c8b661', ['61c8b462c8b563c8b6', 'hex']],
  ['00000000616263616263616263616263616263616263616263616263', ['616263', 4, 'hex']],
  ['00000000006162636162636162636162636162636162636162636162', ['616263', 5, 'hex']],
  ['00000000000000000000000000000000000000000000000000000000', ['616263', SIZE, 'hex']],
  ['0000c8a26161c8a26161c8a26161c8a26161c8a26161c8a26161c8a2', ['c8a26161', 2, 'hex']],
  ['0000000000000000c8a26161c8a26161c8a26161c8a26161c8a26161', ['c8a26161', 8, 'hex']],
  ['0000000061c8b462c8b563c8b661c8b462c8b563c8b661c8b462c8b5', ['61c8b462c8b563c8b6', 4, 'hex']],
  ['00000000000000000000000061c8b462c8b563c8b661c8b462c8b563', ['61c8b462c8b563c8b6', 12, 'hex']],
  ['00000000000000000000000000000000000000000000000000000000', ['616263', 4, -1, 'hex']],
  ['00000000000000000000000000000000000000000000000000000000', ['616263', 4, 1, 'hex']],
  ['00000000000000000000000000000000000000000000000000000000', ['616263', 5, 1, 'hex']],
  ['00000000000000000000000000000000000000000000000000000000', ['c8a26161', 2, -1, 'hex']],
  ['00000000000000000000000000000000000000000000000000000000', ['c8a26161', 8, 1, 'hex']],
  ['00000000000000000000000000000000000000000000000000000000', ['61c8b462c8b563c8b6', 4, -1, 'hex']],
  ['00000000000000000000000000000000000000000000000000000000', ['61c8b462c8b563c8b6', 4, 1, 'hex']],
  ['00000000000000000000000000000000000000000000000000000000', ['61c8b462c8b563c8b6', 12, 1, 'hex']],

  // BASE64
  ['590057004a006a00590057004a006a00590057004a006a0059005700', ['YWJj', 'ucs2']],
  ['79004b004a006800590051003d003d0079004b004a00680059005100', ['yKJhYQ==', 'ucs2']],
  ['59006300690030005900730069003100590038006900320059006300', ['Yci0Ysi1Y8i2', 'ucs2']],
  ['00000000590057004a006a00590057004a006a00590057004a006a00', ['YWJj', 4, 'ucs2']],
  ['00000000000000000000000000000000000000000000000000000000', ['YWJj', SIZE, 'ucs2']],
  ['000079004b004a006800590051003d003d0079004b004a0068005900', ['yKJhYQ==', 2, 'ucs2']],
  ['000000000000000079004b004a006800590051003d003d0079004b00', ['yKJhYQ==', 8, 'ucs2']],
  ['00000000590063006900300059007300690031005900380069003200', ['Yci0Ysi1Y8i2', 4, 'ucs2']],
  ['00000000000000000000000059006300690030005900730069003100', ['Yci0Ysi1Y8i2', 12, 'ucs2']],
  ['00000000000000000000000000000000000000000000000000000000', ['YWJj', 4, -1, 'ucs2']],
  ['00000000000000000000000000000000000000000000000000000000', ['YWJj', 4, 1, 'ucs2']],
  ['00000000000000000000000000000000000000000000000000000000', ['YWJj', 5, 1, 'ucs2']],
  ['00000000000000000000000000000000000000000000000000000000', ['yKJhYQ==', 2, -1, 'ucs2']],
  ['00000000000000000000000000000000000000000000000000000000', ['yKJhYQ==', 8, 1, 'ucs2']],
  ['00000000000000000000000000000000000000000000000000000000', ['Yci0Ysi1Y8i2', 4, -1, 'ucs2']],
  ['00000000000000000000000000000000000000000000000000000000', ['Yci0Ysi1Y8i2', 4, 1, 'ucs2']],
  ['00000000000000000000000000000000000000000000000000000000', ['Yci0Ysi1Y8i2', 12, 1, 'ucs2']]
]

var buf4 = allocUnsafe(SIZE)
for (var i = 0; i < testCases.length; i++) {
  assert.equal(fill(buf4, 0).toString('hex'), EMPTY)
  assert.equal(fill.apply(null, [buf4].concat(testCases[i][1])).toString('hex'), testCases[i][0])
}
