/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.env;

import java.io.Closeable;
import java.io.IOException;
import java.io.PrintStream;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileStore;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.apache.lucene.index.SegmentInfos;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.store.Lock;
import org.apache.lucene.store.LockFactory;
import org.apache.lucene.store.LockObtainFailedException;
import org.apache.lucene.store.NativeFSLockFactory;
import org.apache.lucene.store.SimpleFSDirectory;
import org.apache.lucene.util.IOUtils;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.common.Randomness;
import org.elasticsearch.common.SuppressForbidden;
import org.elasticsearch.common.UUIDs;
import org.elasticsearch.common.io.FileSystemUtils;
import org.elasticsearch.common.logging.ServerLoggers;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.xcontent.NamedXContentRegistry;
import org.elasticsearch.env.Environment;
import org.elasticsearch.env.NodeMetaData;
import org.elasticsearch.env.ShardLock;
import org.elasticsearch.env.ShardLockObtainFailedException;
import org.elasticsearch.index.Index;
import org.elasticsearch.index.IndexSettings;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.store.FsDirectoryService;
import org.elasticsearch.monitor.fs.FsInfo;
import org.elasticsearch.monitor.fs.FsProbe;
import org.elasticsearch.monitor.jvm.JvmInfo;
import org.elasticsearch.node.Node;

public final class NodeEnvironment
implements Closeable {
    private final Logger logger;
    private final NodePath[] nodePaths;
    private final Path sharedDataPath;
    private final Lock[] locks;
    private final int nodeLockId;
    private final AtomicBoolean closed = new AtomicBoolean(false);
    private final Map<ShardId, InternalShardLock> shardLocks = new HashMap<ShardId, InternalShardLock>();
    private final NodeMetaData nodeMetaData;
    public static final Setting<Integer> MAX_LOCAL_STORAGE_NODES_SETTING = Setting.intSetting("node.max_local_storage_nodes", 1, 1, Setting.Property.NodeScope);
    public static final Setting<Long> NODE_ID_SEED_SETTING = Setting.longSetting("node.id.seed", 0L, Long.MIN_VALUE, Setting.Property.NodeScope);
    public static final Setting<Boolean> ENABLE_LUCENE_SEGMENT_INFOS_TRACE_SETTING = Setting.boolSetting("node.enable_lucene_segment_infos_trace", false, Setting.Property.NodeScope);
    public static final String NODES_FOLDER = "nodes";
    public static final String INDICES_FOLDER = "indices";
    public static final String NODE_LOCK_FILENAME = "node.lock";
    static final String TEMP_FILE_NAME = ".es_temp_file";

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public NodeEnvironment(Settings settings, Environment environment) throws IOException {
        if (!DiscoveryNode.nodeRequiresLocalStorage(settings)) {
            this.nodePaths = null;
            this.sharedDataPath = null;
            this.locks = null;
            this.nodeLockId = -1;
            this.nodeMetaData = new NodeMetaData(NodeEnvironment.generateNodeId(settings));
            this.logger = ServerLoggers.getLogger(this.getClass(), Node.addNodeNameIfNeeded(settings, this.nodeMetaData.nodeId()), new String[0]);
            return;
        }
        NodePath[] nodePaths = new NodePath[environment.dataWithClusterFiles().length];
        Lock[] locks = new Lock[nodePaths.length];
        boolean success = false;
        Logger startupTraceLogger = ServerLoggers.getLogger(this.getClass(), settings, new String[0]);
        try {
            this.sharedDataPath = environment.sharedDataFile();
            int nodeLockId = -1;
            IOException lastException = null;
            int maxLocalStorageNodes = MAX_LOCAL_STORAGE_NODES_SETTING.get(settings);
            for (int possibleLockId = 0; possibleLockId < maxLocalStorageNodes; ++possibleLockId) {
                for (int dirIndex = 0; dirIndex < environment.dataFiles().length; ++dirIndex) {
                    Path dataDir = environment.dataFiles()[dirIndex];
                    Path dir = NodeEnvironment.resolveNodePath(dataDir, possibleLockId);
                    Files.createDirectories(dir, new FileAttribute[0]);
                    try {
                        FSDirectory luceneDir = FSDirectory.open((Path)dir, (LockFactory)NativeFSLockFactory.INSTANCE);
                        Throwable throwable = null;
                        try {
                            startupTraceLogger.trace("obtaining node lock on {} ...", (Object)dir.toAbsolutePath());
                            try {
                                locks[dirIndex] = luceneDir.obtainLock(NODE_LOCK_FILENAME);
                                nodePaths[dirIndex] = new NodePath(dir);
                                nodeLockId = possibleLockId;
                            }
                            catch (LockObtainFailedException ex) {
                                startupTraceLogger.trace((Message)new ParameterizedMessage("failed to obtain node lock on {}", (Object)dir.toAbsolutePath()), (Throwable)ex);
                                NodeEnvironment.releaseAndNullLocks(locks);
                                if (luceneDir == null) break;
                                NodeEnvironment.$closeResource(throwable, (AutoCloseable)luceneDir);
                                break;
                            }
                            if (luceneDir == null) continue;
                        }
                        catch (Throwable throwable2) {
                            try {
                                throwable = throwable2;
                                throw throwable2;
                            }
                            catch (Throwable throwable3) {
                                if (luceneDir != null) {
                                    NodeEnvironment.$closeResource(throwable, (AutoCloseable)luceneDir);
                                }
                                throw throwable3;
                            }
                        }
                        NodeEnvironment.$closeResource(throwable, (AutoCloseable)luceneDir);
                        continue;
                    }
                    catch (IOException e) {
                        startupTraceLogger.trace(() -> new ParameterizedMessage("failed to obtain node lock on {}", (Object)dir.toAbsolutePath()), (Throwable)e);
                        lastException = new IOException("failed to obtain lock on " + dir.toAbsolutePath(), e);
                        NodeEnvironment.releaseAndNullLocks(locks);
                        break;
                    }
                }
                if (locks[0] != null) break;
            }
            if (locks[0] == null) {
                String message = String.format(Locale.ROOT, "failed to obtain node locks, tried [%s] with lock id%s; maybe these locations are not writable or multiple nodes were started without increasing [%s] (was [%d])?", Arrays.toString(environment.dataWithClusterFiles()), maxLocalStorageNodes == 1 ? " [0]" : "s [0--" + (maxLocalStorageNodes - 1) + "]", MAX_LOCAL_STORAGE_NODES_SETTING.getKey(), maxLocalStorageNodes);
                throw new IllegalStateException(message, lastException);
            }
            this.nodeMetaData = NodeEnvironment.loadOrCreateNodeMetaData(settings, startupTraceLogger, nodePaths);
            this.logger = ServerLoggers.getLogger(this.getClass(), Node.addNodeNameIfNeeded(settings, this.nodeMetaData.nodeId()), new String[0]);
            this.nodeLockId = nodeLockId;
            this.locks = locks;
            this.nodePaths = nodePaths;
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("using node location [{}], local_lock_id [{}]", (Object)nodePaths, (Object)nodeLockId);
            }
            this.maybeLogPathDetails();
            this.maybeLogHeapDetails();
            NodeEnvironment.applySegmentInfosTrace(settings);
            this.assertCanWrite();
            success = true;
        }
        finally {
            if (!success) {
                IOUtils.closeWhileHandlingException((Closeable[])locks);
            }
        }
    }

    public static Path resolveNodePath(Path path, int nodeLockId) {
        return path.resolve(NODES_FOLDER).resolve(Integer.toString(nodeLockId));
    }

    private static boolean dirEmpty(Path path) throws IOException {
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(path);){
            boolean bl = !stream.iterator().hasNext();
            return bl;
        }
    }

    private static void releaseAndNullLocks(Lock[] locks) {
        for (int i = 0; i < locks.length; ++i) {
            if (locks[i] != null) {
                IOUtils.closeWhileHandlingException((Closeable[])new Closeable[]{locks[i]});
            }
            locks[i] = null;
        }
    }

    private void maybeLogPathDetails() throws IOException {
        if (this.logger.isDebugEnabled()) {
            StringBuilder sb = new StringBuilder();
            for (NodePath nodePath : this.nodePaths) {
                sb.append('\n').append(" -> ").append(nodePath.path.toAbsolutePath());
                FsInfo.Path fsPath = FsProbe.getFSInfo(nodePath);
                sb.append(", free_space [").append(fsPath.getFree()).append("], usable_space [").append(fsPath.getAvailable()).append("], total_space [").append(fsPath.getTotal()).append("], mount [").append(fsPath.getMount()).append("], type [").append(fsPath.getType()).append(']');
            }
            this.logger.debug("node data locations details:{}", (Object)sb);
        } else if (this.logger.isInfoEnabled()) {
            FsInfo.Path totFSPath = new FsInfo.Path();
            HashSet<String> allTypes = new HashSet<String>();
            HashSet<String> allMounts = new HashSet<String>();
            for (NodePath nodePath : this.nodePaths) {
                FsInfo.Path fsPath = FsProbe.getFSInfo(nodePath);
                String mount = fsPath.getMount();
                if (allMounts.contains(mount)) continue;
                allMounts.add(mount);
                String type = fsPath.getType();
                if (type != null) {
                    allTypes.add(type);
                }
                totFSPath.add(fsPath);
            }
            this.logger.info("using [{}] data paths, mounts [{}], net usable_space [{}], net total_space [{}], types [{}]", (Object)this.nodePaths.length, allMounts, (Object)totFSPath.getAvailable(), (Object)totFSPath.getTotal(), (Object)NodeEnvironment.toString(allTypes));
        }
    }

    private void maybeLogHeapDetails() {
        JvmInfo jvmInfo = JvmInfo.jvmInfo();
        ByteSizeValue maxHeapSize = jvmInfo.getMem().getHeapMax();
        String useCompressedOops = jvmInfo.useCompressedOops();
        this.logger.info("heap size [{}], compressed ordinary object pointers [{}]", (Object)maxHeapSize, (Object)useCompressedOops);
    }

    private static NodeMetaData loadOrCreateNodeMetaData(Settings settings, Logger logger, NodePath ... nodePaths) throws IOException {
        Path[] paths = (Path[])Arrays.stream(nodePaths).map(np -> np.path).toArray(Path[]::new);
        NodeMetaData metaData = NodeMetaData.FORMAT.loadLatestState(logger, NamedXContentRegistry.EMPTY, paths);
        if (metaData == null) {
            metaData = new NodeMetaData(NodeEnvironment.generateNodeId(settings));
        }
        NodeMetaData.FORMAT.write(metaData, paths);
        return metaData;
    }

    public static String generateNodeId(Settings settings) {
        Random random = Randomness.get(settings, NODE_ID_SEED_SETTING);
        return UUIDs.randomBase64UUID(random);
    }

    @SuppressForbidden(reason="System.out.*")
    static void applySegmentInfosTrace(Settings settings) {
        if (ENABLE_LUCENE_SEGMENT_INFOS_TRACE_SETTING.get(settings).booleanValue()) {
            SegmentInfos.setInfoStream((PrintStream)System.out);
        }
    }

    private static String toString(Collection<String> items) {
        StringBuilder b = new StringBuilder();
        for (String item : items) {
            if (b.length() > 0) {
                b.append(", ");
            }
            b.append(item);
        }
        return b.toString();
    }

    public void deleteShardDirectorySafe(ShardId shardId, IndexSettings indexSettings) throws IOException, ShardLockObtainFailedException {
        Path[] paths = this.availableShardPaths(shardId);
        this.logger.trace("deleting shard {} directory, paths: [{}]", (Object)shardId, (Object)paths);
        try (ShardLock lock = this.shardLock(shardId);){
            this.deleteShardDirectoryUnderLock(lock, indexSettings);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void acquireFSLockForPaths(IndexSettings indexSettings, Path ... shardPaths) throws IOException {
        Lock[] locks = new Lock[shardPaths.length];
        Directory[] dirs = new Directory[shardPaths.length];
        try {
            for (int i = 0; i < shardPaths.length; ++i) {
                Path p = shardPaths[i].resolve("index");
                dirs[i] = new SimpleFSDirectory(p, indexSettings.getValue(FsDirectoryService.INDEX_LOCK_FACTOR_SETTING));
                try {
                    locks[i] = dirs[i].obtainLock("write.lock");
                    continue;
                }
                catch (IOException ex) {
                    throw new LockObtainFailedException("unable to acquire write.lock for " + p, (Throwable)ex);
                }
            }
        }
        finally {
            IOUtils.closeWhileHandlingException((Closeable[])locks);
            IOUtils.closeWhileHandlingException((Closeable[])dirs);
        }
    }

    public void deleteShardDirectoryUnderLock(ShardLock lock, IndexSettings indexSettings) throws IOException {
        ShardId shardId = lock.getShardId();
        assert (this.isShardLocked(shardId)) : "shard " + shardId + " is not locked";
        Path[] paths = this.availableShardPaths(shardId);
        this.logger.trace("acquiring locks for {}, paths: [{}]", (Object)shardId, (Object)paths);
        NodeEnvironment.acquireFSLockForPaths(indexSettings, paths);
        IOUtils.rm((Path[])paths);
        if (indexSettings.hasCustomDataPath()) {
            Path customLocation = this.resolveCustomLocation(indexSettings, shardId);
            this.logger.trace("acquiring lock for {}, custom path: [{}]", (Object)shardId, (Object)customLocation);
            NodeEnvironment.acquireFSLockForPaths(indexSettings, customLocation);
            this.logger.trace("deleting custom shard {} directory [{}]", (Object)shardId, (Object)customLocation);
            IOUtils.rm((Path[])new Path[]{customLocation});
        }
        this.logger.trace("deleted shard {} directory, paths: [{}]", (Object)shardId, (Object)paths);
        assert (!FileSystemUtils.exists(paths));
    }

    private boolean isShardLocked(ShardId id) {
        try {
            this.shardLock(id, 0L).close();
            return false;
        }
        catch (ShardLockObtainFailedException ex) {
            return true;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deleteIndexDirectorySafe(Index index, long lockTimeoutMS, IndexSettings indexSettings) throws IOException, ShardLockObtainFailedException {
        List<ShardLock> locks = this.lockAllForIndex(index, indexSettings, lockTimeoutMS);
        try {
            this.deleteIndexDirectoryUnderLock(index, indexSettings);
        }
        finally {
            IOUtils.closeWhileHandlingException(locks);
        }
    }

    public void deleteIndexDirectoryUnderLock(Index index, IndexSettings indexSettings) throws IOException {
        Path[] indexPaths = this.indexPaths(index);
        this.logger.trace("deleting index {} directory, paths({}): [{}]", (Object)index, (Object)indexPaths.length, (Object)indexPaths);
        IOUtils.rm((Path[])indexPaths);
        if (indexSettings.hasCustomDataPath()) {
            Path customLocation = this.resolveIndexCustomLocation(indexSettings);
            this.logger.trace("deleting custom index {} directory [{}]", (Object)index, (Object)customLocation);
            IOUtils.rm((Path[])new Path[]{customLocation});
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<ShardLock> lockAllForIndex(Index index, IndexSettings settings, long lockTimeoutMS) throws IOException, ShardLockObtainFailedException {
        int numShards = settings.getNumberOfShards();
        if (numShards <= 0) {
            throw new IllegalArgumentException("settings must contain a non-null > 0 number of shards");
        }
        this.logger.trace("locking all shards for index {} - [{}]", (Object)index, (Object)numShards);
        ArrayList<ShardLock> allLocks = new ArrayList<ShardLock>(numShards);
        boolean success = false;
        long startTimeNS = System.nanoTime();
        try {
            for (int i = 0; i < numShards; ++i) {
                long timeoutLeftMS = Math.max(0L, lockTimeoutMS - TimeValue.nsecToMSec(System.nanoTime() - startTimeNS));
                allLocks.add(this.shardLock(new ShardId(index, i), timeoutLeftMS));
            }
            success = true;
        }
        finally {
            if (!success) {
                this.logger.trace("unable to lock all shards for index {}", (Object)index);
                IOUtils.closeWhileHandlingException(allLocks);
            }
        }
        return allLocks;
    }

    public ShardLock shardLock(ShardId id) throws ShardLockObtainFailedException {
        return this.shardLock(id, 0L);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ShardLock shardLock(final ShardId shardId, long lockTimeoutMS) throws ShardLockObtainFailedException {
        boolean acquired;
        InternalShardLock shardLock;
        this.logger.trace("acquiring node shardlock on [{}], timeout [{}]", (Object)shardId, (Object)lockTimeoutMS);
        Map<ShardId, InternalShardLock> map = this.shardLocks;
        synchronized (map) {
            if (this.shardLocks.containsKey(shardId)) {
                shardLock = this.shardLocks.get(shardId);
                shardLock.incWaitCount();
                acquired = false;
            } else {
                shardLock = new InternalShardLock(shardId);
                this.shardLocks.put(shardId, shardLock);
                acquired = true;
            }
        }
        if (!acquired) {
            boolean success = false;
            try {
                shardLock.acquire(lockTimeoutMS);
                success = true;
            }
            finally {
                if (!success) {
                    shardLock.decWaitCount();
                }
            }
        }
        this.logger.trace("successfully acquired shardlock for [{}]", (Object)shardId);
        return new ShardLock(shardId){

            @Override
            protected void closeInternal() {
                shardLock.release();
                NodeEnvironment.this.logger.trace("released shard lock for [{}]", (Object)shardId);
            }
        };
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Set<ShardId> lockedShards() {
        Map<ShardId, InternalShardLock> map = this.shardLocks;
        synchronized (map) {
            return Collections.unmodifiableSet(new HashSet<ShardId>(this.shardLocks.keySet()));
        }
    }

    public boolean hasNodeFile() {
        return this.nodePaths != null && this.locks != null;
    }

    public Path[] nodeDataPaths() {
        this.assertEnvIsLocked();
        Path[] paths = new Path[this.nodePaths.length];
        for (int i = 0; i < paths.length; ++i) {
            paths[i] = this.nodePaths[i].path;
        }
        return paths;
    }

    public String nodeId() {
        return this.nodeMetaData.nodeId();
    }

    public NodePath[] nodePaths() {
        this.assertEnvIsLocked();
        if (this.nodePaths == null || this.locks == null) {
            throw new IllegalStateException("node is not configured to store local location");
        }
        return this.nodePaths;
    }

    public int getNodeLockId() {
        this.assertEnvIsLocked();
        if (this.nodePaths == null || this.locks == null) {
            throw new IllegalStateException("node is not configured to store local location");
        }
        return this.nodeLockId;
    }

    public Path[] indexPaths(Index index) {
        this.assertEnvIsLocked();
        Path[] indexPaths = new Path[this.nodePaths.length];
        for (int i = 0; i < this.nodePaths.length; ++i) {
            indexPaths[i] = this.nodePaths[i].resolve(index);
        }
        return indexPaths;
    }

    public Path[] availableShardPaths(ShardId shardId) {
        this.assertEnvIsLocked();
        NodePath[] nodePaths = this.nodePaths();
        Path[] shardLocations = new Path[nodePaths.length];
        for (int i = 0; i < nodePaths.length; ++i) {
            shardLocations[i] = nodePaths[i].resolve(shardId);
        }
        return shardLocations;
    }

    public Set<String> availableIndexFolders() throws IOException {
        if (this.nodePaths == null || this.locks == null) {
            throw new IllegalStateException("node is not configured to store local location");
        }
        this.assertEnvIsLocked();
        HashSet<String> indexFolders = new HashSet<String>();
        for (NodePath nodePath : this.nodePaths) {
            indexFolders.addAll(this.availableIndexFoldersForPath(nodePath));
        }
        return indexFolders;
    }

    public Set<String> availableIndexFoldersForPath(NodePath nodePath) throws IOException {
        if (this.nodePaths == null || this.locks == null) {
            throw new IllegalStateException("node is not configured to store local location");
        }
        this.assertEnvIsLocked();
        HashSet<String> indexFolders = new HashSet<String>();
        Path indicesLocation = nodePath.indicesPath;
        if (Files.isDirectory(indicesLocation, new LinkOption[0])) {
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(indicesLocation);){
                for (Path index : stream) {
                    if (!Files.isDirectory(index, new LinkOption[0])) continue;
                    indexFolders.add(index.getFileName().toString());
                }
            }
        }
        return indexFolders;
    }

    public Path[] resolveIndexFolder(String indexFolderName) throws IOException {
        if (this.nodePaths == null || this.locks == null) {
            throw new IllegalStateException("node is not configured to store local location");
        }
        this.assertEnvIsLocked();
        ArrayList<Path> paths = new ArrayList<Path>(this.nodePaths.length);
        for (NodePath nodePath : this.nodePaths) {
            Path indexFolder = nodePath.indicesPath.resolve(indexFolderName);
            if (!Files.exists(indexFolder, new LinkOption[0])) continue;
            paths.add(indexFolder);
        }
        return paths.toArray(new Path[paths.size()]);
    }

    public Set<ShardId> findAllShardIds(Index index) throws IOException {
        assert (index != null);
        if (this.nodePaths == null || this.locks == null) {
            throw new IllegalStateException("node is not configured to store local location");
        }
        this.assertEnvIsLocked();
        HashSet<ShardId> shardIds = new HashSet<ShardId>();
        String indexUniquePathId = index.getUUID();
        for (NodePath nodePath : this.nodePaths) {
            Path location = nodePath.indicesPath;
            if (!Files.isDirectory(location, new LinkOption[0])) continue;
            try (DirectoryStream<Path> indexStream = Files.newDirectoryStream(location);){
                for (Path indexPath : indexStream) {
                    if (!indexUniquePathId.equals(indexPath.getFileName().toString())) continue;
                    shardIds.addAll(NodeEnvironment.findAllShardsForIndex(indexPath, index));
                }
            }
        }
        return shardIds;
    }

    public Map<NodePath, Long> shardCountPerPath(Index index) throws IOException {
        assert (index != null);
        if (this.nodePaths == null || this.locks == null) {
            throw new IllegalStateException("node is not configured to store local location");
        }
        this.assertEnvIsLocked();
        HashMap<NodePath, Long> shardCountPerPath = new HashMap<NodePath, Long>();
        String indexUniquePathId = index.getUUID();
        for (NodePath nodePath : this.nodePaths) {
            Path indexLocation = nodePath.indicesPath.resolve(indexUniquePathId);
            if (!Files.isDirectory(indexLocation, new LinkOption[0])) continue;
            shardCountPerPath.put(nodePath, Long.valueOf(NodeEnvironment.findAllShardsForIndex(indexLocation, index).size()));
        }
        return shardCountPerPath;
    }

    private static Set<ShardId> findAllShardsForIndex(Path indexPath, Index index) throws IOException {
        assert (indexPath.getFileName().toString().equals(index.getUUID()));
        HashSet<ShardId> shardIds = new HashSet<ShardId>();
        if (Files.isDirectory(indexPath, new LinkOption[0])) {
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(indexPath);){
                for (Path shardPath : stream) {
                    String fileName = shardPath.getFileName().toString();
                    if (!Files.isDirectory(shardPath, new LinkOption[0]) || !fileName.chars().allMatch(Character::isDigit)) continue;
                    int shardId = Integer.parseInt(fileName);
                    ShardId id = new ShardId(index, shardId);
                    shardIds.add(id);
                }
            }
        }
        return shardIds;
    }

    @Override
    public void close() {
        if (this.closed.compareAndSet(false, true) && this.locks != null) {
            for (Lock lock : this.locks) {
                try {
                    this.logger.trace("releasing lock [{}]", (Object)lock);
                    lock.close();
                }
                catch (IOException e) {
                    this.logger.trace(() -> new ParameterizedMessage("failed to release lock [{}]", (Object)lock), (Throwable)e);
                }
            }
        }
    }

    private void assertEnvIsLocked() {
        if (!this.closed.get() && this.locks != null) {
            for (Lock lock : this.locks) {
                try {
                    lock.ensureValid();
                }
                catch (IOException e) {
                    this.logger.warn("lock assertion failed", (Throwable)e);
                    throw new IllegalStateException("environment is not locked", e);
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void ensureAtomicMoveSupported() throws IOException {
        NodePath[] nodePaths;
        for (NodePath nodePath : nodePaths = this.nodePaths()) {
            assert (Files.isDirectory(nodePath.path, new LinkOption[0])) : nodePath.path + " is not a directory";
            Path src = nodePath.path.resolve(".es_temp_file.tmp");
            Path target = nodePath.path.resolve(".es_temp_file.final");
            try {
                Files.deleteIfExists(src);
                Files.createFile(src, new FileAttribute[0]);
                Files.move(src, target, StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
            }
            catch (AtomicMoveNotSupportedException ex) {
                throw new IllegalStateException("atomic_move is not supported by the filesystem on path [" + nodePath.path + "] atomic_move is required for elasticsearch to work correctly.", ex);
            }
            finally {
                try {
                    Files.deleteIfExists(src);
                }
                finally {
                    Files.deleteIfExists(target);
                }
            }
        }
    }

    public Path resolveBaseCustomLocation(IndexSettings indexSettings) {
        String customDataDir = indexSettings.customDataPath();
        if (customDataDir != null) {
            assert (this.sharedDataPath != null);
            return this.sharedDataPath.resolve(customDataDir).resolve(Integer.toString(this.nodeLockId));
        }
        throw new IllegalArgumentException("no custom index.data_path setting available");
    }

    private Path resolveIndexCustomLocation(IndexSettings indexSettings) {
        return this.resolveBaseCustomLocation(indexSettings).resolve(indexSettings.getUUID());
    }

    public Path resolveCustomLocation(IndexSettings indexSettings, ShardId shardId) {
        return this.resolveIndexCustomLocation(indexSettings).resolve(Integer.toString(shardId.id()));
    }

    public static Path shardStatePathToDataPath(Path shardPath) {
        int count = shardPath.getNameCount();
        assert (Integer.parseInt(shardPath.getName(count - 1).toString()) >= 0);
        assert (INDICES_FOLDER.equals(shardPath.getName(count - 3).toString()));
        return shardPath.getParent().getParent().getParent();
    }

    private void assertCanWrite() throws IOException {
        for (Path path : this.nodeDataPaths()) {
            NodeEnvironment.tryWriteTempFile(path);
        }
        for (String indexFolderName : this.availableIndexFolders()) {
            for (Path indexPath : this.resolveIndexFolder(indexFolderName)) {
                Path indexStatePath = indexPath.resolve("_state");
                NodeEnvironment.tryWriteTempFile(indexStatePath);
                NodeEnvironment.tryWriteTempFile(indexPath);
                try (DirectoryStream<Path> stream = Files.newDirectoryStream(indexPath);){
                    for (Path shardPath : stream) {
                        String fileName = shardPath.getFileName().toString();
                        if (!Files.isDirectory(shardPath, new LinkOption[0]) || !fileName.chars().allMatch(Character::isDigit)) continue;
                        Path indexDir = shardPath.resolve("index");
                        Path statePath = shardPath.resolve("_state");
                        Path translogDir = shardPath.resolve("translog");
                        NodeEnvironment.tryWriteTempFile(indexDir);
                        NodeEnvironment.tryWriteTempFile(translogDir);
                        NodeEnvironment.tryWriteTempFile(statePath);
                        NodeEnvironment.tryWriteTempFile(shardPath);
                    }
                }
            }
        }
    }

    private static void tryWriteTempFile(Path path) throws IOException {
        if (Files.exists(path, new LinkOption[0])) {
            Path resolve = path.resolve(TEMP_FILE_NAME);
            try {
                Files.deleteIfExists(resolve);
                Files.createFile(resolve, new FileAttribute[0]);
                Files.delete(resolve);
            }
            catch (IOException ex) {
                throw new IOException("failed to test writes in data directory [" + path + "] write permission is required", ex);
            }
        }
    }

    private final class InternalShardLock {
        private final Semaphore mutex = new Semaphore(1);
        private int waitCount = 1;
        private final ShardId shardId;

        InternalShardLock(ShardId shardId) {
            this.shardId = shardId;
            this.mutex.acquireUninterruptibly();
        }

        protected void release() {
            this.mutex.release();
            this.decWaitCount();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void incWaitCount() {
            Map map = NodeEnvironment.this.shardLocks;
            synchronized (map) {
                assert (this.waitCount > 0) : "waitCount is " + this.waitCount + " but should be > 0";
                ++this.waitCount;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void decWaitCount() {
            Map map = NodeEnvironment.this.shardLocks;
            synchronized (map) {
                assert (this.waitCount > 0) : "waitCount is " + this.waitCount + " but should be > 0";
                --this.waitCount;
                NodeEnvironment.this.logger.trace("shard lock wait count for {} is now [{}]", (Object)this.shardId, (Object)this.waitCount);
                if (this.waitCount == 0) {
                    NodeEnvironment.this.logger.trace("last shard lock wait decremented, removing lock for {}", (Object)this.shardId);
                    InternalShardLock remove = (InternalShardLock)NodeEnvironment.this.shardLocks.remove(this.shardId);
                    assert (remove != null) : "Removed lock was null";
                }
            }
        }

        void acquire(long timeoutInMillis) throws ShardLockObtainFailedException {
            try {
                if (!this.mutex.tryAcquire(timeoutInMillis, TimeUnit.MILLISECONDS)) {
                    throw new ShardLockObtainFailedException(this.shardId, "obtaining shard lock timed out after " + timeoutInMillis + "ms");
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new ShardLockObtainFailedException(this.shardId, "thread interrupted while trying to obtain shard lock", e);
            }
        }
    }

    @FunctionalInterface
    public static interface ShardLocker {
        public ShardLock lock(ShardId var1, long var2) throws ShardLockObtainFailedException;
    }

    public static class NodePath {
        public final Path path;
        public final Path indicesPath;
        public final FileStore fileStore;
        public final int majorDeviceNumber;
        public final int minorDeviceNumber;

        public NodePath(Path path) throws IOException {
            this.path = path;
            this.indicesPath = path.resolve(NodeEnvironment.INDICES_FOLDER);
            this.fileStore = Environment.getFileStore(path);
            if (this.fileStore.supportsFileAttributeView("lucene")) {
                this.majorDeviceNumber = (Integer)this.fileStore.getAttribute("lucene:major_device_number");
                this.minorDeviceNumber = (Integer)this.fileStore.getAttribute("lucene:minor_device_number");
            } else {
                this.majorDeviceNumber = -1;
                this.minorDeviceNumber = -1;
            }
        }

        public Path resolve(ShardId shardId) {
            return this.resolve(shardId.getIndex()).resolve(Integer.toString(shardId.id()));
        }

        public Path resolve(Index index) {
            return this.indicesPath.resolve(index.getUUID());
        }

        public String toString() {
            return "NodePath{path=" + this.path + ", indicesPath=" + this.indicesPath + ", fileStore=" + this.fileStore + ", majorDeviceNumber=" + this.majorDeviceNumber + ", minorDeviceNumber=" + this.minorDeviceNumber + '}';
        }
    }
}

