#ifndef Magnum_Primitives_UVSphere_h
#define Magnum_Primitives_UVSphere_h
/*
    This file is part of Magnum.

    Copyright © 2010, 2011, 2012, 2013, 2014, 2015, 2016, 2017, 2018
              Vladimír Vondruš <mosra@centrum.cz>

    Permission is hereby granted, free of charge, to any person obtaining a
    copy of this software and associated documentation files (the "Software"),
    to deal in the Software without restriction, including without limitation
    the rights to use, copy, modify, merge, publish, distribute, sublicense,
    and/or sell copies of the Software, and to permit persons to whom the
    Software is furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included
    in all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
    THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
    DEALINGS IN THE SOFTWARE.
*/

/** @file
 * @brief Class @ref Magnum::Primitives::uvSphereSolid(), @ref Magnum::Primitives::uvSphereWireframe()
 */

#include "Magnum/Trade/Trade.h"
#include "Magnum/Primitives/visibility.h"

#ifdef MAGNUM_BUILD_DEPRECATED
#include <Corrade/Utility/Macros.h>
#endif

namespace Magnum { namespace Primitives {

/**
@brief Whether to generate UV sphere texture coordinates

@see @ref uvSphereSolid()
*/
enum class UVSphereTextureCoords: UnsignedByte {
    DontGenerate,       /**< Don't generate texture coordinates */
    Generate            /**< Generate texture coordinates */
};

/**
@brief Solid 3D UV sphere
@param rings            Number of (face) rings. Must be larger or equal to
    @cpp 2 @ce.
@param segments         Number of (face) segments. Must be larger or
    equal to @cpp 3 @ce.
@param textureCoords    Whether to generate texture coordinates

Sphere with radius @cpp 1.0f @ce. Indexed @ref MeshPrimitive::Triangles with
normals and optional 2D texture coordinates. If texture coordinates are
generated, vertices of one segment are duplicated for texture wrapping.

@image html primitives-uvspheresolid.png

@see @ref icosphereSolid()
*/
MAGNUM_PRIMITIVES_EXPORT Trade::MeshData3D uvSphereSolid(UnsignedInt rings, UnsignedInt segments, UVSphereTextureCoords textureCoords = UVSphereTextureCoords::DontGenerate);

/**
@brief Wireframe 3D UV sphere
@param rings            Number of (line) rings. Must be larger or equal to
    @cpp 2 @ce and multiple of @cpp 2 @ce.
@param segments         Number of (line) segments. Must be larger or equal to
    @cpp 4 @ce and multiple of @cpp 4 @ce.

Sphere with radius @cpp 1.0f @ce. Indexed @ref MeshPrimitive::Lines.

@image html primitives-uvspherewireframe.png

@see @ref icosphereSolid()
*/
MAGNUM_PRIMITIVES_EXPORT Trade::MeshData3D uvSphereWireframe(UnsignedInt rings, UnsignedInt segments);

#ifdef MAGNUM_BUILD_DEPRECATED
/**
@brief 3D UV sphere
@deprecated Use @ref uvSphereSolid() or @ref uvSphereWireframe() instead.
*/
struct MAGNUM_PRIMITIVES_EXPORT UVSphere {
    /** @brief @copybrief UVSphereTextureCoords
     * @deprecated Use @ref UVSphereTextureCoords instead.
     */
    typedef CORRADE_DEPRECATED("use UVSphereTextureCoords instead") UVSphereTextureCoords TextureCoords;

    /** @brief @copybrief uvSphereSolid()
     * @deprecated Use @ref uvSphereSolid() instead.
     */
    CORRADE_DEPRECATED("use uvSphereSolid() instead") static Trade::MeshData3D solid(UnsignedInt rings, UnsignedInt segments, UVSphereTextureCoords textureCoords = UVSphereTextureCoords::DontGenerate);

    /** @brief @copybrief uvSphereWireframe()
     * @deprecated Use @ref uvSphereWireframe() instead.
     */
    CORRADE_DEPRECATED("use uvSphereWireframe() instead") static Trade::MeshData3D wireframe(UnsignedInt rings, UnsignedInt segments);
};
#endif

}}

#endif
