/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.rollup.job;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.log4j.Logger;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.search.SearchRequest;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.RangeQueryBuilder;
import org.elasticsearch.search.aggregations.AggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.composite.CompositeAggregation;
import org.elasticsearch.search.aggregations.bucket.composite.CompositeAggregationBuilder;
import org.elasticsearch.search.builder.SearchSourceBuilder;
import org.elasticsearch.xpack.core.rollup.job.DateHistoGroupConfig;
import org.elasticsearch.xpack.core.rollup.job.GroupConfig;
import org.elasticsearch.xpack.core.rollup.job.IndexerState;
import org.elasticsearch.xpack.core.rollup.job.RollupJob;
import org.elasticsearch.xpack.core.rollup.job.RollupJobConfig;
import org.elasticsearch.xpack.core.rollup.job.RollupJobStats;
import org.elasticsearch.xpack.rollup.job.IndexerUtils;

public abstract class RollupIndexer {
    private static final Logger logger = Logger.getLogger((String)RollupIndexer.class.getName());
    static final String AGGREGATION_NAME = "rollup";
    private final RollupJob job;
    private final RollupJobStats stats;
    private final AtomicReference<IndexerState> state;
    private final AtomicReference<Map<String, Object>> position;
    private final Executor executor;
    private final CompositeAggregationBuilder compositeBuilder;
    private long maxBoundary;

    RollupIndexer(Executor executor, RollupJob job, AtomicReference<IndexerState> initialState, Map<String, Object> initialPosition) {
        this.executor = executor;
        this.job = job;
        this.stats = new RollupJobStats();
        this.state = initialState;
        this.position = new AtomicReference<Map<String, Object>>(initialPosition);
        this.compositeBuilder = this.createCompositeBuilder(job.getConfig());
    }

    protected abstract void doNextSearch(SearchRequest var1, ActionListener<SearchResponse> var2);

    protected abstract void doNextBulk(BulkRequest var1, ActionListener<BulkResponse> var2);

    protected abstract void doSaveState(IndexerState var1, Map<String, Object> var2, Runnable var3);

    protected abstract void onFailure(Exception var1);

    protected abstract void onFinish();

    protected abstract void onAbort();

    public IndexerState getState() {
        return this.state.get();
    }

    public Map<String, Object> getPosition() {
        return this.position.get();
    }

    public RollupJobStats getStats() {
        return this.stats;
    }

    public synchronized IndexerState start() {
        this.state.compareAndSet(IndexerState.STOPPED, IndexerState.STARTED);
        return this.state.get();
    }

    public synchronized IndexerState stop() {
        IndexerState currentState = this.state.updateAndGet(previousState -> {
            if (previousState == IndexerState.INDEXING) {
                return IndexerState.STOPPING;
            }
            if (previousState == IndexerState.STARTED) {
                return IndexerState.STOPPED;
            }
            return previousState;
        });
        return currentState;
    }

    public synchronized boolean abort() {
        IndexerState prevState = this.state.getAndUpdate(prev -> IndexerState.ABORTING);
        return prevState == IndexerState.STOPPED || prevState == IndexerState.STARTED;
    }

    public synchronized boolean maybeTriggerAsyncJob(long now) {
        IndexerState currentState = this.state.get();
        switch (currentState) {
            case INDEXING: 
            case STOPPING: 
            case ABORTING: {
                logger.warn((Object)("Schedule was triggered for rollup job [" + this.job.getConfig().getId() + "], but prior indexer is still running."));
                return false;
            }
            case STOPPED: {
                logger.debug((Object)("Schedule was triggered for rollup job [" + this.job.getConfig().getId() + "] but job is stopped.  Ignoring trigger."));
                return false;
            }
            case STARTED: {
                logger.debug((Object)("Schedule was triggered for rollup job [" + this.job.getConfig().getId() + "], state: [" + currentState + "]"));
                this.stats.incrementNumInvocations(1L);
                DateHistoGroupConfig dateHisto = this.job.getConfig().getGroupConfig().getDateHisto();
                long rounded = dateHisto.createRounding().round(now);
                this.maxBoundary = dateHisto.getDelay() != null ? rounded - TimeValue.parseTimeValue((String)dateHisto.getDelay().toString(), (String)"").millis() : rounded;
                if (this.state.compareAndSet(IndexerState.STARTED, IndexerState.INDEXING)) {
                    this.executor.execute(() -> this.doNextSearch(this.buildSearchRequest(), (ActionListener<SearchResponse>)ActionListener.wrap(this::onSearchResponse, exc -> this.finishWithFailure((Exception)exc))));
                    logger.debug((Object)("Beginning to rollup [" + this.job.getConfig().getId() + "], state: [" + currentState + "]"));
                    return true;
                }
                logger.debug((Object)("Could not move from STARTED to INDEXING state because current state is [" + this.state.get() + "]"));
                return false;
            }
        }
        logger.warn((Object)("Encountered unexpected state [" + currentState + "] while indexing"));
        throw new IllegalStateException("Rollup job encountered an illegal state [" + currentState + "]");
    }

    private boolean checkState(IndexerState currentState) {
        switch (currentState) {
            case INDEXING: {
                return true;
            }
            case STOPPING: {
                logger.info((Object)("Rollup job encountered [" + IndexerState.STOPPING + "] state, halting indexer."));
                this.doSaveState(this.finishAndSetState(), this.getPosition(), () -> {});
                return false;
            }
            case STOPPED: {
                return false;
            }
            case ABORTING: {
                logger.info((Object)("Requested shutdown of indexer for job [" + this.job.getConfig().getId() + "]"));
                this.onAbort();
                return false;
            }
        }
        logger.warn((Object)("Encountered unexpected state [" + currentState + "] while indexing"));
        throw new IllegalStateException("Rollup job encountered an illegal state [" + currentState + "]");
    }

    private void onBulkResponse(BulkResponse response, Map<String, Object> after) {
        this.stats.incrementNumRollups((long)response.getItems().length);
        if (response.hasFailures()) {
            logger.warn((Object)("Error while attempting to bulk index rollup documents: " + response.buildFailureMessage()));
        }
        try {
            if (!this.checkState(this.getState())) {
                return;
            }
            this.position.set(after);
            ActionListener listener = ActionListener.wrap(this::onSearchResponse, this::finishWithFailure);
            if (this.stats.getNumPages() > 0L && this.stats.getNumPages() % 50L == 0L) {
                this.doSaveState(IndexerState.INDEXING, after, () -> this.doNextSearch(this.buildSearchRequest(), (ActionListener<SearchResponse>)listener));
            } else {
                this.doNextSearch(this.buildSearchRequest(), (ActionListener<SearchResponse>)listener);
            }
        }
        catch (Exception e) {
            this.finishWithFailure(e);
        }
    }

    private void onSearchResponse(SearchResponse searchResponse) {
        try {
            if (!this.checkState(this.getState())) {
                return;
            }
            if (searchResponse.getShardFailures().length != 0) {
                throw new RuntimeException("Shard failures encountered while running indexer for rollup job [" + this.job.getConfig().getId() + "]: " + Arrays.toString(searchResponse.getShardFailures()));
            }
            CompositeAggregation response = (CompositeAggregation)searchResponse.getAggregations().get(AGGREGATION_NAME);
            if (response == null) {
                throw new IllegalStateException("Missing composite response for query: " + this.compositeBuilder.toString());
            }
            this.stats.incrementNumPages(1L);
            if (response.getBuckets().isEmpty()) {
                logger.debug((Object)("Finished indexing for job [" + this.job.getConfig().getId() + "], saving state and shutting down."));
                this.doSaveState(this.finishAndSetState(), this.position.get(), this::onFinish);
                return;
            }
            BulkRequest bulkRequest = new BulkRequest();
            List<IndexRequest> docs = IndexerUtils.processBuckets(response, this.job.getConfig().getRollupIndex(), this.stats, this.job.getConfig().getGroupConfig(), this.job.getConfig().getId());
            docs.forEach(arg_0 -> ((BulkRequest)bulkRequest).add(arg_0));
            assert (bulkRequest.requests().size() > 0);
            this.doNextBulk(bulkRequest, (ActionListener<BulkResponse>)ActionListener.wrap(bulkResponse -> this.onBulkResponse((BulkResponse)bulkResponse, response.afterKey()), exc -> this.finishWithFailure((Exception)exc)));
        }
        catch (Exception e) {
            this.finishWithFailure(e);
        }
    }

    private void finishWithFailure(Exception exc) {
        this.doSaveState(this.finishAndSetState(), this.position.get(), () -> this.onFailure(exc));
    }

    private IndexerState finishAndSetState() {
        return this.state.updateAndGet(prev -> {
            switch (prev) {
                case INDEXING: {
                    return IndexerState.STARTED;
                }
                case STOPPING: {
                    return IndexerState.STOPPED;
                }
                case ABORTING: {
                    this.onAbort();
                    return IndexerState.ABORTING;
                }
                case STOPPED: {
                    return IndexerState.STOPPED;
                }
            }
            throw new IllegalStateException("Rollup job encountered an illegal state [" + prev + "]");
        });
    }

    private SearchRequest buildSearchRequest() {
        Map<String, Object> position = this.getPosition();
        SearchSourceBuilder searchSource = new SearchSourceBuilder().size(0).trackTotalHits(false).query(this.createBoundaryQuery(position)).aggregation((AggregationBuilder)this.compositeBuilder.aggregateAfter(position));
        return new SearchRequest(new String[]{this.job.getConfig().getIndexPattern()}).source(searchSource);
    }

    private CompositeAggregationBuilder createCompositeBuilder(RollupJobConfig config) {
        GroupConfig groupConfig = config.getGroupConfig();
        ArrayList builders = new ArrayList();
        HashMap metadata = new HashMap();
        if (groupConfig != null) {
            builders.addAll(groupConfig.getDateHisto().toBuilders());
            metadata.putAll(groupConfig.getDateHisto().getMetadata());
            if (groupConfig.getHisto() != null) {
                builders.addAll(groupConfig.getHisto().toBuilders());
                metadata.putAll(groupConfig.getHisto().getMetadata());
            }
            if (groupConfig.getTerms() != null) {
                builders.addAll(groupConfig.getTerms().toBuilders());
                metadata.putAll(groupConfig.getTerms().getMetadata());
            }
        }
        CompositeAggregationBuilder composite = new CompositeAggregationBuilder(AGGREGATION_NAME, builders);
        config.getMetricsConfig().forEach(m -> m.toBuilders().forEach(arg_0 -> ((CompositeAggregationBuilder)composite).subAggregation(arg_0)));
        if (!metadata.isEmpty()) {
            composite.setMetaData(metadata);
        }
        composite.size(config.getPageSize());
        return composite;
    }

    private QueryBuilder createBoundaryQuery(Map<String, Object> position) {
        long lowerBound;
        assert (this.maxBoundary < Long.MAX_VALUE);
        DateHistoGroupConfig dateHisto = this.job.getConfig().getGroupConfig().getDateHisto();
        String fieldName = dateHisto.getField();
        String rollupFieldName = fieldName + "." + "date_histogram";
        long l = lowerBound = position != null ? (Long)position.get(rollupFieldName) : 0L;
        assert (lowerBound <= this.maxBoundary);
        RangeQueryBuilder query = new RangeQueryBuilder(fieldName).gte((Object)lowerBound).lt((Object)this.maxBoundary);
        return query;
    }
}

