/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.watcher;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.elasticsearch.Version;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.block.ClusterBlockLevel;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.routing.AllocationId;
import org.elasticsearch.cluster.routing.RoutingNode;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.routing.ShardRoutingState;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.component.LifecycleListener;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.concurrent.AbstractRunnable;
import org.elasticsearch.common.util.concurrent.EsExecutors;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.gateway.GatewayService;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.xpack.core.upgrade.UpgradeField;
import org.elasticsearch.xpack.core.watcher.WatcherMetaData;
import org.elasticsearch.xpack.core.watcher.WatcherState;
import org.elasticsearch.xpack.watcher.WatcherService;
import org.elasticsearch.xpack.watcher.support.WatcherIndexTemplateRegistry;
import org.elasticsearch.xpack.watcher.watch.WatchStoreUtils;

public class WatcherLifeCycleService
extends AbstractComponent
implements ClusterStateListener {
    public static final Setting<Boolean> SETTING_REQUIRE_MANUAL_START = Setting.boolSetting((String)"xpack.watcher.require_manual_start", (boolean)false, (Setting.Property[])new Setting.Property[]{Setting.Property.NodeScope});
    private static final String LIFECYCLE_THREADPOOL_NAME = "watcher-lifecycle";
    private final WatcherService watcherService;
    private final ExecutorService executor;
    private AtomicReference<List<String>> previousAllocationIds = new AtomicReference(Collections.emptyList());
    private volatile boolean shutDown = false;
    private final boolean requireManualStart;

    WatcherLifeCycleService(Settings settings, ThreadPool threadPool, ClusterService clusterService, WatcherService watcherService) {
        this(settings, clusterService, watcherService, (ExecutorService)EsExecutors.newFixed((String)LIFECYCLE_THREADPOOL_NAME, (int)1, (int)1000, (ThreadFactory)EsExecutors.daemonThreadFactory((Settings)settings, (String)LIFECYCLE_THREADPOOL_NAME), (ThreadContext)threadPool.getThreadContext()));
    }

    WatcherLifeCycleService(Settings settings, ClusterService clusterService, WatcherService watcherService, ExecutorService executorService) {
        super(settings);
        this.executor = executorService;
        this.watcherService = watcherService;
        this.requireManualStart = (Boolean)SETTING_REQUIRE_MANUAL_START.get(settings);
        clusterService.addListener((ClusterStateListener)this);
        clusterService.addLifecycleListener(new LifecycleListener(){

            public void beforeStop() {
                WatcherLifeCycleService.this.shutDown();
            }
        });
    }

    public synchronized void stop(String reason) {
        this.watcherService.stop(reason);
    }

    synchronized void shutDown() {
        this.shutDown = true;
        this.stop("shutdown initiated");
        this.stopExecutor();
    }

    void stopExecutor() {
        ThreadPool.terminate((ExecutorService)this.executor, (long)10L, (TimeUnit)TimeUnit.SECONDS);
    }

    private synchronized void start(ClusterState state) {
        if (this.shutDown) {
            return;
        }
        WatcherState watcherState = this.watcherService.state();
        if (watcherState != WatcherState.STOPPED) {
            this.logger.debug("not starting watcher. watcher can only start if its current state is [{}], but its current state now is [{}]", (Object)WatcherState.STOPPED, (Object)watcherState);
            return;
        }
        WatcherMetaData watcherMetaData = (WatcherMetaData)state.getMetaData().custom("watcher");
        if (watcherMetaData != null && watcherMetaData.manuallyStopped()) {
            this.logger.debug("not starting watcher. watcher was stopped manually and therefore cannot be auto-started");
            return;
        }
        if (!WatcherIndexTemplateRegistry.validate(state)) {
            this.logger.debug("not starting watcher, watcher templates are missing in the cluster state");
            return;
        }
        if (this.watcherService.validate(state)) {
            this.logger.trace("starting... (based on cluster state version [{}])", (Object)state.getVersion());
            try {
                this.checkAndSetAllocationIds(state, false);
                this.watcherService.start(state);
            }
            catch (Exception e) {
                this.logger.warn("failed to start watcher. please wait for the cluster to become ready or try to start Watcher manually", (Throwable)e);
            }
        } else {
            this.logger.debug("not starting watcher. because the cluster isn't ready yet to run watcher");
        }
    }

    public void clusterChanged(ClusterChangedEvent event) {
        if (event.state().blocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK) || this.shutDown) {
            this.clearAllocationIds();
            return;
        }
        WatcherMetaData watcherMetaData = (WatcherMetaData)event.state().getMetaData().custom("watcher");
        if (watcherMetaData == null && this.requireManualStart) {
            this.clearAllocationIds();
            return;
        }
        if (Strings.isNullOrEmpty((String)event.state().nodes().getMasterNodeId())) {
            this.clearAllocationIds();
            this.executor.execute(() -> this.stop("no master node"));
            return;
        }
        if (event.state().getBlocks().hasGlobalBlock(ClusterBlockLevel.WRITE)) {
            this.clearAllocationIds();
            this.executor.execute(() -> this.stop("write level cluster block"));
            return;
        }
        if (this.isWatcherStoppedManually(event.state())) {
            this.clearAllocationIds();
            this.executor.execute(() -> this.stop("watcher manually marked to shutdown by cluster state update"));
        } else {
            WatcherState watcherState = this.watcherService.state();
            boolean isDistributedWatchExecutionEnabled = WatcherLifeCycleService.isWatchExecutionDistributed(event.state());
            if (isDistributedWatchExecutionEnabled) {
                if (watcherState == WatcherState.STARTED && event.state().nodes().getLocalNode().isDataNode()) {
                    this.checkAndSetAllocationIds(event.state(), true);
                } else if (watcherState != WatcherState.STARTED && watcherState != WatcherState.STARTING) {
                    boolean isIndexInternalFormatTriggeredWatchIndex;
                    IndexMetaData watcherIndexMetaData = WatchStoreUtils.getConcreteIndex(".watches", event.state().metaData());
                    IndexMetaData triggeredWatchesIndexMetaData = WatchStoreUtils.getConcreteIndex(".triggered_watches", event.state().metaData());
                    boolean isIndexInternalFormatWatchIndex = watcherIndexMetaData == null || UpgradeField.checkInternalIndexFormat((IndexMetaData)watcherIndexMetaData);
                    boolean bl = isIndexInternalFormatTriggeredWatchIndex = triggeredWatchesIndexMetaData == null || UpgradeField.checkInternalIndexFormat((IndexMetaData)triggeredWatchesIndexMetaData);
                    if (isIndexInternalFormatTriggeredWatchIndex && isIndexInternalFormatWatchIndex) {
                        this.checkAndSetAllocationIds(event.state(), false);
                        this.executor.execute(() -> this.start(event.state()));
                    } else {
                        this.logger.warn("not starting watcher, upgrade API run required: .watches[{}], .triggered_watches[{}]", (Object)isIndexInternalFormatWatchIndex, (Object)isIndexInternalFormatTriggeredWatchIndex);
                    }
                }
            } else if (event.localNodeMaster()) {
                if (watcherState != WatcherState.STARTED && watcherState != WatcherState.STARTING) {
                    this.executor.execute(() -> this.start(event.state()));
                }
            } else if (watcherState == WatcherState.STARTED || watcherState == WatcherState.STARTING) {
                this.executor.execute(() -> this.watcherService.pauseExecution("Pausing watcher, cluster contains old nodes not supporting distributed watch execution"));
            }
        }
    }

    public static boolean isWatchExecutionDistributed(ClusterState state) {
        return state.nodes().getMinNodeVersion().onOrAfter(Version.V_6_0_0_beta1);
    }

    private boolean isWatcherStoppedManually(ClusterState state) {
        WatcherMetaData watcherMetaData = (WatcherMetaData)state.getMetaData().custom("watcher");
        return watcherMetaData != null && watcherMetaData.manuallyStopped();
    }

    private void checkAndSetAllocationIds(ClusterState state, boolean callWatcherService) {
        IndexMetaData watcherIndexMetaData = WatchStoreUtils.getConcreteIndex(".watches", state.metaData());
        if (watcherIndexMetaData == null) {
            if (this.clearAllocationIds() && callWatcherService) {
                this.executor.execute((Runnable)WatcherLifeCycleService.wrapWatcherService(() -> this.watcherService.pauseExecution("no watcher index found"), e -> this.logger.error("error pausing watch execution", (Throwable)e)));
            }
            return;
        }
        DiscoveryNode localNode = state.nodes().getLocalNode();
        RoutingNode routingNode = state.getRoutingNodes().node(localNode.getId());
        if (routingNode == null) {
            if (this.clearAllocationIds() && callWatcherService) {
                this.executor.execute((Runnable)WatcherLifeCycleService.wrapWatcherService(() -> this.watcherService.pauseExecution("no routing node for local node found, network issue?"), e -> this.logger.error("error pausing watch execution", (Throwable)e)));
            }
            return;
        }
        String watchIndex = watcherIndexMetaData.getIndex().getName();
        List localShards = routingNode.shardsWithState(watchIndex, new ShardRoutingState[]{ShardRoutingState.RELOCATING, ShardRoutingState.STARTED});
        if (localShards.isEmpty()) {
            if (this.clearAllocationIds() && callWatcherService) {
                this.executor.execute((Runnable)WatcherLifeCycleService.wrapWatcherService(() -> this.watcherService.pauseExecution("no local watcher shards found"), e -> this.logger.error("error pausing watch execution", (Throwable)e)));
            }
            return;
        }
        List currentAllocationIds = localShards.stream().map(ShardRouting::allocationId).map(AllocationId::getId).collect(Collectors.toList());
        Collections.sort(currentAllocationIds);
        if (!this.previousAllocationIds.get().equals(currentAllocationIds)) {
            this.previousAllocationIds.set(Collections.unmodifiableList(currentAllocationIds));
            if (callWatcherService) {
                this.executor.execute((Runnable)WatcherLifeCycleService.wrapWatcherService(() -> this.watcherService.reload(state, "new local watcher shard allocation ids"), e -> this.logger.error("error reloading watcher", (Throwable)e)));
            }
        }
    }

    private boolean clearAllocationIds() {
        List previousIds = this.previousAllocationIds.getAndSet(Collections.emptyList());
        return !previousIds.equals(Collections.emptyList());
    }

    List<String> allocationIds() {
        return this.previousAllocationIds.get();
    }

    private static AbstractRunnable wrapWatcherService(final Runnable run, final Consumer<Exception> exceptionConsumer) {
        return new AbstractRunnable(){

            public void onFailure(Exception e) {
                exceptionConsumer.accept(e);
            }

            protected void doRun() throws Exception {
                run.run();
            }
        };
    }
}

