"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const util = require("util");
const semver = require("semver");
const json_1 = require("../utils/json");
const configuration_1 = require("../configuration");
const transport_1 = require("../utils/transport");
const debug_1 = require("debug");
const debug = debug_1.default('axm:notify');
class NotifyOptions {
}
exports.NotifyOptions = NotifyOptions;
exports.NotifyOptionsDefault = {
    level: 'info',
    catchExceptions: true
};
class NotifyFeature {
    constructor() {
        this.options = exports.NotifyOptionsDefault;
        this.levels = ['fatal', 'error', 'warn', 'info', 'debug', 'trace'];
    }
    init(options) {
        if (options) {
            this.options = options;
        }
        if (this.options && this.options.catchExceptions) {
            if (process.env.CATCH_CONTEXT_ON_ERROR === 'true' && semver.satisfies(process.version, '< 8.0.0')) {
                debug(`Inspector is not available on node version ${process.version} !`);
            }
            if (process.env.CATCH_CONTEXT_ON_ERROR === 'true' && semver.satisfies(process.version, '>= 8.0.0')) {
                const NotifyInspector = require('./notifyInspector').default;
                NotifyInspector.catchAllDebugger();
            }
            else {
                this.catchAll();
            }
        }
        return {
            notifyError: this.notifyError
        };
    }
    notifyError(err, level) {
        if (!(err instanceof Error)) {
            console.trace('[PM2-IO-APM] You should use notify with an Error object');
            return -1;
        }
        if (!level || this.levels.indexOf(level) === -1) {
            return transport_1.default.send({
                type: 'process:exception',
                data: json_1.default.jsonize(err)
            });
        }
        if (this.levels.indexOf(this.options.level) >= this.levels.indexOf(level)) {
            return transport_1.default.send({
                type: 'process:exception',
                data: json_1.default.jsonize(err)
            });
        }
        return null;
    }
    catchAll(opts) {
        if (opts === undefined) {
            opts = { errors: true };
        }
        configuration_1.default.configureModule({
            error: opts.errors
        });
        if (process.env.exec_mode === 'cluster_mode') {
            return false;
        }
        const self = this;
        function getUncaughtExceptionListener(listener) {
            return function uncaughtListener(err) {
                let error = err && err.stack ? err.stack : err;
                if (err && err.length) {
                    err._length = err.length;
                    delete err.length;
                }
                if (listener === 'unhandledRejection') {
                    console.log('You have triggered an unhandledRejection, you may have forgotten to catch a Promise rejection:');
                }
                console.error(error);
                let errObj;
                if (err) {
                    errObj = self._interpretError(err);
                }
                transport_1.default.send({
                    type: 'process:exception',
                    data: errObj !== undefined ? errObj : { message: 'No error but ' + listener + ' was caught!' }
                });
                if (!process.listeners(listener).filter(function (listener) {
                    return listener !== uncaughtListener;
                }).length) {
                    if (listener === 'uncaughtException') {
                        process.exit(1);
                    }
                }
            };
        }
        if (opts.errors === true && util.inspect(process.listeners('uncaughtException')).length === 2) {
            process.once('uncaughtException', getUncaughtExceptionListener('uncaughtException'));
            process.once('unhandledRejection', getUncaughtExceptionListener('unhandledRejection'));
        }
        else if (opts.errors === false
            && util.inspect(process.listeners('uncaughtException')).length !== 2) {
            process.removeAllListeners('uncaughtException');
            process.removeAllListeners('unhandledRejection');
        }
    }
    expressErrorHandler() {
        configuration_1.default.configureModule({
            error: true
        });
        return function errorHandler(err, req, res, next) {
            if (res.statusCode < 400)
                res.statusCode = 500;
            err.url = req.url;
            err.component = req.url;
            err.action = req.method;
            err.params = req.body;
            err.session = req.session;
            transport_1.default.send({
                type: 'process:exception',
                data: json_1.default.jsonize(err)
            });
            return next(err);
        };
    }
    _interpretError(err) {
        let sErr = {
            message: null,
            stack: null
        };
        if (err instanceof Error) {
            // Error object type processing
            sErr = err;
        }
        else {
            // JSON processing
            sErr.message = err;
            sErr.stack = err;
        }
        return json_1.default.jsonize(sErr);
    }
}
exports.NotifyFeature = NotifyFeature;
//# sourceMappingURL=data:application/json;base64,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