"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const domain = require("domain");
const debug_1 = require("debug");
const debug = debug_1.default('axm:actions');
const serviceManager_1 = require("../serviceManager");
const transport_1 = require("../utils/transport");
const actions_1 = require("../services/actions");
class ActionsFeature {
    constructor() {
        serviceManager_1.ServiceManager.set('actionsService', new actions_1.default(this));
        this.actionsService = serviceManager_1.ServiceManager.get('actionsService');
        process.on('message', this.listener);
    }
    listener(data) {
        if (!data)
            return false;
        const actionName = data.msg ? data.msg : data.action_name ? data.action_name : data;
        let actionData = serviceManager_1.ServiceManager.get('actions').get(actionName);
        let fn = actionData ? actionData.fn : null;
        const reply = actionData ? actionData.reply : null;
        if (actionData) {
            // In case 2 arguments has been set but no options has been transmitted
            if (fn.length === 2 && typeof (data) === 'string' && data === actionName) {
                return fn({}, reply);
            }
            // In case 1 arguments has been set but options has been transmitted
            if (fn.length === 1 && typeof (data) === 'object' && data.msg === actionName) {
                return fn(reply);
            }
            /**
             * Classical call
             */
            if (typeof (data) === 'string' && data === actionName) {
                return fn(reply);
            }
            /**
             * If data is an object == v2 protocol
             * Pass the opts as first argument
             */
            if (typeof (data) === 'object' && data.msg === actionName) {
                return fn(data.opts, reply);
            }
        }
        // -----------------------------------------------------------
        //                      Scoped actions
        // -----------------------------------------------------------
        if (data.uuid === undefined || data.action_name === undefined) {
            return false;
        }
        actionData = serviceManager_1.ServiceManager.get('actionsScoped').get(actionName);
        if (data.action_name === actionName) {
            const res = {
                send: (dt) => {
                    transport_1.default.send({
                        type: 'axm:scoped_action:stream',
                        data: {
                            data: dt,
                            uuid: data.uuid,
                            action_name: actionName
                        }
                    });
                },
                error: (dt) => {
                    transport_1.default.send({
                        type: 'axm:scoped_action:error',
                        data: {
                            data: dt,
                            uuid: data.uuid,
                            action_name: actionName
                        }
                    });
                },
                end: (dt) => {
                    transport_1.default.send({
                        type: 'axm:scoped_action:end',
                        data: {
                            data: dt,
                            uuid: data.uuid,
                            action_name: actionName
                        }
                    });
                }
            };
            const d = domain.create();
            d.on('error', function (err) {
                res.error(err.message || err.stack || err);
                setTimeout(function () {
                    process.exit(1);
                }, 300);
            });
            d.run(function () {
                actionData.fn(data.opts || null, res);
            });
        }
    }
    init(conf, force) {
        this.actionsService.init(conf, force);
        return {
            action: this.action
        };
    }
    destroy() {
        this.actionsService.destroy();
        serviceManager_1.ServiceManager.reset('actions');
        serviceManager_1.ServiceManager.reset('actionsScoped');
        process.removeListener('message', this.listener);
    }
    action(actionName, opts, fn) {
        if (!fn) {
            fn = opts;
            opts = null;
        }
        const check = this.check(actionName, fn);
        if (!check) {
            return check;
        }
        let type = 'custom';
        if (actionName.indexOf('km:') === 0 || actionName.indexOf('internal:') === 0) {
            type = 'internal';
        }
        // Notify the action
        transport_1.default.send({
            type: 'axm:action',
            data: {
                action_name: actionName,
                action_type: type,
                opts: opts,
                arity: fn.length
            }
        });
        const reply = (data) => {
            transport_1.default.send({
                type: 'axm:reply',
                data: {
                    return: data,
                    action_name: actionName
                }
            });
        };
        serviceManager_1.ServiceManager.get('actions').set(actionName, { fn: fn, reply: reply });
    }
    scopedAction(actionName, fn) {
        const check = this.check(actionName, fn);
        if (!check) {
            return check;
        }
        // Notify the action
        transport_1.default.send({
            type: 'axm:action',
            data: {
                action_name: actionName,
                action_type: 'scoped'
            }
        });
        serviceManager_1.ServiceManager.get('actionsScoped').set(actionName, { fn: fn });
    }
    check(actionName, fn) {
        if (!actionName) {
            console.error('[PMX] action.action_name is missing');
            return false;
        }
        if (!fn) {
            console.error('[PMX] callback is missing');
            return false;
        }
        if (!process.send) {
            debug('Process not running within PM2');
            return false;
        }
        return true;
    }
}
exports.default = ActionsFeature;
//# sourceMappingURL=data:application/json;base64,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