"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const meter_1 = require("../utils/metrics/meter");
const counter_1 = require("../utils/metrics/counter");
const histogram_1 = require("../utils/metrics/histogram");
const serviceManager_1 = require("../serviceManager");
const transport_1 = require("../utils/transport");
const constants_1 = require("../constants");
const metrics_1 = require("../services/metrics");
class MetricsFeature {
    constructor() {
        this._var = new Map();
        this.defaultAggregation = 'avg';
        this._started = false;
        this._alreadySentData = [];
        this.AVAILABLE_MEASUREMENTS = [
            'min',
            'max',
            'sum',
            'count',
            'variance',
            'mean',
            'stddev',
            'median',
            'p75',
            'p95',
            'p99',
            'p999'
        ];
        this._var = serviceManager_1.ServiceManager.get('metricsMap');
        serviceManager_1.ServiceManager.set('metricService', new metrics_1.default(this));
        this.metricService = serviceManager_1.ServiceManager.get('metricService');
    }
    init(config, force) {
        if (this._started === false) {
            this._started = true;
            const self = this;
            this.timer = setInterval(function () {
                const data = self._cookData(self._getVar());
                // don't send empty data
                if (Object.keys(data).length !== 0) {
                    transport_1.default.send({
                        type: 'axm:monitor',
                        data: data
                    });
                }
            }, constants_1.default.METRIC_INTERVAL);
        }
        this.metricService.init(config, force);
        return {
            histogram: this.histogram,
            meter: this.meter,
            counter: this.counter,
            metric: this.metric
        };
    }
    transpose(variableName, reporter) {
        if (typeof variableName === 'object') {
            reporter = variableName.data;
            variableName = variableName.name;
        }
        if (typeof reporter !== 'function') {
            console.error('[PM2 IO][Transpose] reporter is not a function');
            return undefined;
        }
        this._var.set(variableName, {
            value: reporter
        });
    }
    meter(opts) {
        if (!opts.name) {
            console.error('[PM2 IO][Meter] Name not defined');
            return undefined;
        }
        opts.unit = opts.unit || '';
        const meter = new meter_1.default(opts);
        this._var.set(opts.name, {
            value: function () {
                return meter.val() + `${opts.unit}`;
            },
            type: opts.type || opts.name,
            historic: this._historicEnabled(opts.historic),
            agg_type: opts.agg_type || this.defaultAggregation,
            unit: opts.unit
        });
        return meter;
    }
    counter(opts) {
        if (!opts.name) {
            console.error('[PM2 IO][Counter] Name not defined');
            return undefined;
        }
        const counter = new counter_1.default(opts);
        this._var.set(opts.name, {
            value: function () { return counter.val(); },
            type: opts.type || opts.name,
            historic: this._historicEnabled(opts.historic),
            agg_type: opts.agg_type || this.defaultAggregation,
            unit: opts.unit
        });
        return counter;
    }
    histogram(opts) {
        if (!opts.name) {
            console.error('[PM2 IO][Histogram] Name not defined');
            return undefined;
        }
        opts.measurement = opts.measurement || 'mean';
        opts.unit = opts.unit || '';
        if (this.AVAILABLE_MEASUREMENTS.indexOf(opts.measurement) === -1) {
            console.error('[PM2 IO][Histogram] Measure type %s does not exists', opts.measurement);
            return undefined;
        }
        const histogram = new histogram_1.default(opts);
        this._var.set(opts.name, {
            value: function () {
                return (Math.round(histogram.val() * 100) / 100) + `${opts.unit}`;
            },
            type: opts.type || opts.name,
            historic: this._historicEnabled(opts.historic),
            agg_type: opts.agg_type || this.defaultAggregation,
            unit: opts.unit
        });
        return histogram;
    }
    metric(opts) {
        if (!opts.name) {
            console.error('[PM2 IO][Metric] Name not defined');
            return undefined;
        }
        this._var.set(opts.name, {
            value: opts.value || 0,
            type: opts.type || opts.name,
            historic: this._historicEnabled(opts.historic),
            agg_type: opts.agg_type || this.defaultAggregation,
            unit: opts.unit
        });
        const self = this;
        return {
            val: function () {
                let value = self._var.get(opts.name).value;
                if (typeof (value) === 'function') {
                    value = value();
                }
                return value;
            },
            set: function (dt) {
                self._var.get(opts.name).value = dt;
            }
        };
    }
    deleteMetric(name) {
        this._var.delete(name);
        this._alreadySentData.splice(this._alreadySentData.indexOf(name), 1);
    }
    destroy() {
        clearInterval(this.timer);
        this._getVar().clear();
        this._started = false;
        this.metricService.destroyAll();
    }
    /** -----------------------------------------
     * Private Methods
     * ------------------------------------------
     */
    /**
     * Check if metric is historic or not
     *
     * @param historic
     * @returns {boolean}
     * @private
     */
    _historicEnabled(historic) {
        if (historic === false) {
            return false;
        }
        if (typeof (historic) === 'undefined') {
            return true;
        }
        return true;
    }
    /**
     * Only for tests
     *
     * @returns {Object}
     */
    _getVar() {
        return this._var;
    }
    /**
     * Data that will be sent to Keymetrics
     */
    _cookData(data) {
        const cookedData = {};
        const self = this;
        data.forEach(function (probe, probeName) {
            if (typeof (data.get(probeName).value) === 'undefined') {
                return false;
            }
            const value = self._getValue(data.get(probeName).value);
            // do not send data if this is always equal to 0
            // probably an initialized metric which is not "active"
            if ((self._alreadySentData.indexOf(probeName) === -1 && value !== 0) ||
                self._alreadySentData.indexOf(probeName) > -1) {
                if (self._alreadySentData.indexOf(probeName) === -1) {
                    self._alreadySentData.push(probeName);
                }
                cookedData[probeName] = {
                    value: value
                };
                /**
                 * Attach aggregation mode
                 */
                if (data.get(probeName).agg_type &&
                    data.get(probeName).agg_type !== 'none') {
                    cookedData[probeName].agg_type = data.get(probeName).agg_type;
                }
                cookedData[probeName].historic = data.get(probeName).historic;
                cookedData[probeName].type = data.get(probeName).type;
                cookedData[probeName].unit = data.get(probeName).unit;
            }
        });
        return cookedData;
    }
    _getValue(value) {
        if (typeof (value) === 'function') {
            return value();
        }
        return value;
    }
}
exports.default = MetricsFeature;
//# sourceMappingURL=data:application/json;base64,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