/* SPDX-License-Identifier: GPL-2.0
 *
 * Copyright (C) 2015-2018 Jason A. Donenfeld <Jason@zx2c4.com>. All Rights Reserved.
 */

#ifdef DEBUG
struct chacha20poly1305_testvec {
	u8 *key, *nonce, *assoc, *input, *result;
	size_t nlen, alen, ilen;
	bool failure;
};

/* The first of these are the ChaCha20-Poly1305 AEAD test vectors from RFC7539 2.8.2. After they
 * are generated by the below python program. And the final marked ones are taken from wycheproof,
 * but we only do these for the encrypt side, because mostly we're stressing the primitives rather
 * than the actual chapoly construction. This also requires adding a 96-bit nonce construction, just
 * for the purpose of the tests.
 *
 * #!/usr/bin/env python3
 *
 * from cryptography.hazmat.primitives.ciphers.aead import ChaCha20Poly1305
 * import os
 *
 * def encode_blob(blob):
 *     a = ""
 *     for i in blob:
 *         a += "\\x" + hex(i)[2:]
 *     return a
 *
 * enc = [ ]
 * dec = [ ]
 *
 * def make_vector(plen, adlen):
 *     key = os.urandom(32)
 *     nonce = os.urandom(8)
 *     p = os.urandom(plen)
 *     ad = os.urandom(adlen)
 *     c = ChaCha20Poly1305(key).encrypt(nonce=bytes(4) + nonce, data=p, associated_data=ad)
 *
 *     out = "{\n"
 *     out += "\t.key\t= \"" + encode_blob(key) + "\",\n"
 *     out += "\t.nonce\t= \"" + encode_blob(nonce) + "\",\n"
 *     out += "\t.assoc\t= \"" + encode_blob(ad) + "\",\n"
 *     out += "\t.alen\t= " + str(len(ad)) + ",\n"
 *     out += "\t.input\t= \"" + encode_blob(p) + "\",\n"
 *     out += "\t.ilen\t= " + str(len(p)) + ",\n"
 *     out += "\t.result\t= \"" + encode_blob(c) + "\"\n"
 *     out += "}"
 *     enc.append(out)
 *
 *
 *     out = "{\n"
 *     out += "\t.key\t= \"" + encode_blob(key) + "\",\n"
 *     out += "\t.nonce\t= \"" + encode_blob(nonce) + "\",\n"
 *     out += "\t.assoc\t= \"" + encode_blob(ad) + "\",\n"
 *     out += "\t.alen\t= " + str(len(ad)) + ",\n"
 *     out += "\t.input\t= \"" + encode_blob(c) + "\",\n"
 *     out += "\t.ilen\t= " + str(len(c)) + ",\n"
 *     out += "\t.result\t= \"" + encode_blob(p) + "\"\n"
 *     out += "}"
 *     dec.append(out)
 *
 *
 * make_vector(0, 0)
 * make_vector(0, 8)
 * make_vector(1, 8)
 * make_vector(1, 0)
 * make_vector(129, 7)
 * make_vector(256, 0)
 * make_vector(512, 0)
 * make_vector(513, 9)
 * make_vector(1024, 16)
 * make_vector(1933, 7)
 * make_vector(2011, 63)
 *
 * print("======== encryption vectors ========")
 * print(", ".join(enc))
 *
 * print("\n\n\n======== decryption vectors ========")
 * print(", ".join(dec))
 */

static const struct chacha20poly1305_testvec chacha20poly1305_enc_vectors[] __initconst = { {
	.key	= "\x1c\x92\x40\xa5\xeb\x55\xd3\x8a\xf3\x33\x88\x86\x04\xf6\xb5\xf0\x47\x39\x17\xc1\x40\x2b\x80\x09\x9d\xca\x5c\xbc\x20\x70\x75\xc0",
	.nonce	= "\x01\x02\x03\x04\x05\x06\x07\x08",
	.nlen	= 8,
	.assoc	= "\xf3\x33\x88\x86\x00\x00\x00\x00\x00\x00\x4e\x91",
	.alen	= 12,
	.input	= "\x49\x6e\x74\x65\x72\x6e\x65\x74\x2d\x44\x72\x61\x66\x74\x73\x20\x61\x72\x65\x20\x64\x72\x61\x66\x74\x20\x64\x6f\x63\x75\x6d\x65\x6e\x74\x73\x20\x76\x61\x6c\x69\x64\x20\x66\x6f\x72\x20\x61\x20\x6d\x61\x78\x69\x6d\x75\x6d\x20\x6f\x66\x20\x73\x69\x78\x20\x6d\x6f\x6e\x74\x68\x73\x20\x61\x6e\x64\x20\x6d\x61\x79\x20\x62\x65\x20\x75\x70\x64\x61\x74\x65\x64\x2c\x20\x72\x65\x70\x6c\x61\x63\x65\x64\x2c\x20\x6f\x72\x20\x6f\x62\x73\x6f\x6c\x65\x74\x65\x64\x20\x62\x79\x20\x6f\x74\x68\x65\x72\x20\x64\x6f\x63\x75\x6d\x65\x6e\x74\x73\x20\x61\x74\x20\x61\x6e\x79\x20\x74\x69\x6d\x65\x2e\x20\x49\x74\x20\x69\x73\x20\x69\x6e\x61\x70\x70\x72\x6f\x70\x72\x69\x61\x74\x65\x20\x74\x6f\x20\x75\x73\x65\x20\x49\x6e\x74\x65\x72\x6e\x65\x74\x2d\x44\x72\x61\x66\x74\x73\x20\x61\x73\x20\x72\x65\x66\x65\x72\x65\x6e\x63\x65\x20\x6d\x61\x74\x65\x72\x69\x61\x6c\x20\x6f\x72\x20\x74\x6f\x20\x63\x69\x74\x65\x20\x74\x68\x65\x6d\x20\x6f\x74\x68\x65\x72\x20\x74\x68\x61\x6e\x20\x61\x73\x20\x2f\xe2\x80\x9c\x77\x6f\x72\x6b\x20\x69\x6e\x20\x70\x72\x6f\x67\x72\x65\x73\x73\x2e\x2f\xe2\x80\x9d",
	.ilen	= 265,
	.result	= "\x64\xa0\x86\x15\x75\x86\x1a\xf4\x60\xf0\x62\xc7\x9b\xe6\x43\xbd\x5e\x80\x5c\xfd\x34\x5c\xf3\x89\xf1\x08\x67\x0a\xc7\x6c\x8c\xb2\x4c\x6c\xfc\x18\x75\x5d\x43\xee\xa0\x9e\xe9\x4e\x38\x2d\x26\xb0\xbd\xb7\xb7\x3c\x32\x1b\x01\x00\xd4\xf0\x3b\x7f\x35\x58\x94\xcf\x33\x2f\x83\x0e\x71\x0b\x97\xce\x98\xc8\xa8\x4a\xbd\x0b\x94\x81\x14\xad\x17\x6e\x00\x8d\x33\xbd\x60\xf9\x82\xb1\xff\x37\xc8\x55\x97\x97\xa0\x6e\xf4\xf0\xef\x61\xc1\x86\x32\x4e\x2b\x35\x06\x38\x36\x06\x90\x7b\x6a\x7c\x02\xb0\xf9\xf6\x15\x7b\x53\xc8\x67\xe4\xb9\x16\x6c\x76\x7b\x80\x4d\x46\xa5\x9b\x52\x16\xcd\xe7\xa4\xe9\x90\x40\xc5\xa4\x04\x33\x22\x5e\xe2\x82\xa1\xb0\xa0\x6c\x52\x3e\xaf\x45\x34\xd7\xf8\x3f\xa1\x15\x5b\x00\x47\x71\x8c\xbc\x54\x6a\x0d\x07\x2b\x04\xb3\x56\x4e\xea\x1b\x42\x22\x73\xf5\x48\x27\x1a\x0b\xb2\x31\x60\x53\xfa\x76\x99\x19\x55\xeb\xd6\x31\x59\x43\x4e\xce\xbb\x4e\x46\x6d\xae\x5a\x10\x73\xa6\x72\x76\x27\x09\x7a\x10\x49\xe6\x17\xd9\x1d\x36\x10\x94\xfa\x68\xf0\xff\x77\x98\x71\x30\x30\x5b\xea\xba\x2e\xda\x04\xdf\x99\x7b\x71\x4d\x6c\x6f\x2c\x29\xa6\xad\x5c\xb4\x02\x2b\x02\x70\x9b\xee\xad\x9d\x67\x89\x0c\xbb\x22\x39\x23\x36\xfe\xa1\x85\x1f\x38"
}, {
	.key	= "\x4c\xf5\x96\x83\x38\xe6\xae\x7f\x2d\x29\x25\x76\xd5\x75\x27\x86\x91\x9a\x27\x7a\xfb\x46\xc5\xef\x94\x81\x79\x57\x14\x59\x40\x68",
	.nonce	= "\xca\xbf\x33\x71\x32\x45\x77\x8e",
	.nlen	= 8,
	.assoc	= "",
	.alen	= 0,
	.input	= "",
	.ilen	= 0,
	.result	= "\xea\xe0\x1e\x9e\x2c\x91\xaa\xe1\xdb\x5d\x99\x3f\x8a\xf7\x69\x92"
}, {
	.key	= "\x2d\xb0\x5d\x40\xc8\xed\x44\x88\x34\xd1\x13\xaf\x57\xa1\xeb\x3a\x2a\x80\x51\x36\xec\x5b\xbc\x8\x93\x84\x21\xb5\x13\x88\x3c\xd",
	.nonce	= "\x3d\x86\xb5\x6b\xc8\xa3\x1f\x1d",
	.nlen	= 8,
	.assoc	= "\x33\x10\x41\x12\x1f\xf3\xd2\x6b",
	.alen	= 8,
	.input	= "",
	.ilen	= 0,
	.result	= "\xdd\x6b\x3b\x82\xce\x5a\xbd\xd6\xa9\x35\x83\xd8\x8c\x3d\x85\x77"
}, {
	.key	= "\x4b\x28\x4b\xa3\x7b\xbe\xe9\xf8\x31\x80\x82\xd7\xd8\xe8\xb5\xa1\xe2\x18\x18\x8a\x9c\xfa\xa3\x3d\x25\x71\x3e\x40\xbc\x54\x7a\x3e",
	.nonce	= "\xd2\x32\x1f\x29\x28\xc6\xc4\xc4",
	.nlen	= 8,
	.assoc	= "\x6a\xe2\xad\x3f\x88\x39\x5a\x40",
	.alen	= 8,
	.input	= "\xa4",
	.ilen	= 1,
	.result	= "\xb7\x1b\xb0\x73\x59\xb0\x84\xb2\x6d\x8e\xab\x94\x31\xa1\xae\xac\x89"
}, {
	.key	= "\x66\xca\x9c\x23\x2a\x4b\x4b\x31\xe\x92\x89\x8b\xf4\x93\xc7\x87\x98\xa3\xd8\x39\xf8\xf4\xa7\x1\xc0\x2e\xa\xa6\x7e\x5a\x78\x87",
	.nonce	= "\x20\x1c\xaa\x5f\x9c\xbf\x92\x30",
	.nlen	= 8,
	.assoc	= "",
	.alen	= 0,
	.input	= "\x2d",
	.ilen	= 1,
	.result	= "\xbf\xe1\x5b\xb\xdb\x6b\xf5\x5e\x6c\x5d\x84\x44\x39\x81\xc1\x9c\xac"
}, {
	.key	= "\x68\x7b\x8d\x8e\xe3\xc4\xdd\xae\xdf\x72\x7f\x53\x72\x25\x1e\x78\x91\xcb\x69\x76\x1f\x49\x93\xf9\x6f\x21\xcc\x39\x9c\xad\xb1\x1",
	.nonce	= "\xdf\x51\x84\x82\x42\xc\x75\x9c",
	.nlen	= 8,
	.assoc	= "\x70\xd3\x33\xf3\x8b\x18\xb",
	.alen	= 7,
	.input	= "\x33\x2f\x94\xc1\xa4\xef\xcc\x2a\x5b\xa6\xe5\x8f\x1d\x40\xf0\x92\x3c\xd9\x24\x11\xa9\x71\xf9\x37\x14\x99\xfa\xbe\xe6\x80\xde\x50\xc9\x96\xd4\xb0\xec\x9e\x17\xec\xd2\x5e\x72\x99\xfc\xa\xe1\xcb\x48\xd2\x85\xdd\x2f\x90\xe0\x66\x3b\xe6\x20\x74\xbe\x23\x8f\xcb\xb4\xe4\xda\x48\x40\xa6\xd1\x1b\xc7\x42\xce\x2f\xc\xa6\x85\x6e\x87\x37\x3\xb1\x7c\x25\x96\xa3\x5\xd8\xb0\xf4\xed\xea\xc2\xf0\x31\x98\x6c\xd1\x14\x25\xc0\xcb\x1\x74\xd0\x82\xf4\x36\xf5\x41\xd5\xdc\xca\xc5\xbb\x98\xfe\xfc\x69\x21\x70\xd8\xa4\x4b\xc8\xde\x8f",
	.ilen	= 129,
	.result	= "\x8b\x6\xd3\x31\xb0\x93\x45\xb1\x75\x6e\x26\xf9\x67\xbc\x90\x15\x81\x2c\xb5\xf0\xc6\x2b\xc7\x8c\x56\xd1\xbf\x69\x6c\x7\xa0\xda\x65\x27\xc9\x90\x3d\xef\x4b\x11\xf\x19\x7\xfd\x29\x92\xd9\xc8\xf7\x99\x2e\x4a\xd0\xb8\x2c\xdc\x93\xf5\x9e\x33\x78\xd1\x37\xc3\x66\xd7\x5e\xbc\x44\xbf\x53\xa5\xbc\xc4\xcb\x7b\x3a\x8e\x7f\x2\xbd\xbb\xe7\xca\xa6\x6c\x6b\x93\x21\x93\x10\x61\xe7\x69\xd0\x78\xf3\x7\x5a\x1a\x8f\x73\xaa\xb1\x4e\xd3\xda\x4f\xf3\x32\xe1\x66\x3e\x6c\xc6\x13\xba\x6\x5b\xfc\x6a\xe5\x6f\x60\xfb\x7\x40\xb0\x8c\x9d\x84\x43\x6b\xc1\xf7\x8d\x8d\x31\xf7\x7a\x39\x4d\x8f\x9a\xeb"
}, {
	.key	= "\x8d\xb8\x91\x48\xf0\xe7\xa\xbd\xf9\x3f\xcd\xd9\xa0\x1e\x42\x4c\xe7\xde\x25\x3d\xa3\xd7\x5\x80\x8d\xf2\x82\xac\x44\x16\x51\x1",
	.nonce	= "\xde\x7b\xef\xc3\x65\x1b\x68\xb0",
	.nlen	= 8,
	.assoc	= "",
	.alen	= 0,
	.input	= "\x9b\x18\xdb\xdd\x9a\xf\x3e\xa5\x15\x17\xde\xdf\x8\x9d\x65\xa\x67\x30\x12\xe2\x34\x77\x4b\xc1\xd9\xc6\x1f\xab\xc6\x18\x50\x17\xa7\x9d\x3c\xa6\xc5\x35\x8c\x1c\xc0\xa1\x7c\x9f\x3\x89\xca\xe1\xe6\xe9\xd4\xd3\x88\xdb\xb4\x51\x9d\xec\xb4\xfc\x52\xee\x6d\xf1\x75\x42\xc6\xfd\xbd\x7a\x8e\x86\xfc\x44\xb3\x4f\xf3\xea\x67\x5a\x41\x13\xba\xb0\xdc\xe1\xd3\x2a\x7c\x22\xb3\xca\xac\x6a\x37\x98\x3e\x1d\x40\x97\xf7\x9b\x1d\x36\x6b\xb3\x28\xbd\x60\x82\x47\x34\xaa\x2f\x7d\xe9\xa8\x70\x81\x57\xd4\xb9\x77\xa\x9d\x29\xa7\x84\x52\x4f\xc2\x4a\x40\x3b\x3c\xd4\xc9\x2a\xdb\x4a\x53\xc4\xbe\x80\xe9\x51\x7f\x8f\xc7\xa2\xce\x82\x5c\x91\x1e\x74\xd9\xd0\xbd\xd5\xf3\xfd\xda\x4d\x25\xb4\xbb\x2d\xac\x2f\x3d\x71\x85\x7b\xcf\x3c\x7b\x3e\xe\x22\x78\xc\x29\xbf\xe4\xf4\x57\xb3\xcb\x49\xa0\xfc\x1e\x5\x4e\x16\xbc\xd5\xa8\xa3\xee\x5\x35\xc6\x7c\xab\x60\x14\x55\x1a\x8e\xc5\x88\x5d\xd5\x81\xc2\x81\xa5\xc4\x60\xdb\xaf\x77\x91\xe1\xce\xa2\x7e\x7f\x42\xe3\xb0\x13\x1c\x1f\x25\x60\x21\xe2\x40\x5f\x99\xb7\x73\xec\x9b\x2b\xf0\x65\x11\xc8\xd0\xa\x9f\xd3",
	.ilen	= 256,
	.result	= "\x85\x4\xc2\xed\x8d\xfd\x97\x5c\xd2\xb7\xe2\xc1\x6b\xa3\xba\xf8\xc9\x50\xc3\xc6\xa5\xe3\xa4\x7c\xc3\x23\x49\x5e\xa9\xb9\x32\xeb\x8a\x7c\xca\xe5\xec\xfb\x7c\xc0\xcb\x7d\xdc\x2c\x9d\x92\x55\x21\xa\xc8\x43\x63\x59\xa\x31\x70\x82\x67\x41\x3\xf8\xdf\xf2\xac\xa7\x2\xd4\xd5\x8a\x2d\xc8\x99\x19\x66\xd0\xf6\x88\x2c\x77\xd9\xd4\xd\x6c\xbd\x98\xde\xe7\x7f\xad\x7e\x8a\xfb\xe9\x4b\xe5\xf7\xe5\x50\xa0\x90\x3f\xd6\x22\x53\xe3\xfe\x1b\xcc\x79\x3b\xec\x12\x47\x52\xa7\xd6\x4\xe3\x52\xe6\x93\x90\x91\x32\x73\x79\xb8\xd0\x31\xde\x1f\x9f\x2f\x5\x38\x54\x2f\x35\x4\x39\xe0\xa7\xba\xc6\x52\xf6\x37\x65\x4c\x7\xa9\x7e\xb3\x21\x6f\x74\x8c\xc9\xde\xdb\x65\x1b\x9b\xaa\x60\xb1\x3\x30\x6b\xb2\x3\xc4\x1c\x4\xf8\xf\x64\xaf\x46\xe4\x65\x99\x49\xe2\xea\xce\x78\x0\xd8\x8b\xd5\x2e\xcf\xfc\x40\x49\xe8\x58\xdc\x34\x9c\x8c\x61\xbf\xa\x8e\xec\x39\xa9\x30\x5\x5a\xd2\x56\x1\xc7\xda\x8f\x4e\xbb\x43\xa3\x3a\xf9\x15\x2a\xd0\xa0\x7a\x87\x34\x82\xfe\x8a\xd1\x2d\x5e\xc7\xbf\x4\x53\x5f\x3b\x36\xd4\x25\x5c\x34\x7a\x8d\xd5\x5\xce\x72\xca\xef\x7a\x4b\xbc\xb0\x10\x5c\x96\x42\x3a\x0\x98\xcd\x15\xe8\xb7\x53"
}, {
	.key	= "\xf2\xaa\x4f\x99\xfd\x3e\xa8\x53\xc1\x44\xe9\x81\x18\xdc\xf5\xf0\x3e\x44\x15\x59\xe0\xc5\x44\x86\xc3\x91\xa8\x75\xc0\x12\x46\xba",
	.nonce	= "\xe\xd\x57\xbb\x7b\x40\x54\x2",
	.nlen	= 8,
	.assoc	= "",
	.alen	= 0,
	.input	= "\xc3\x9\x94\x62\xe6\x46\x2e\x10\xbe\x0\xe4\xfc\xf3\x40\xa3\xe2\xf\xc2\x8b\x28\xdc\xba\xb4\x3c\xe4\x21\x58\x61\xcd\x8b\xcd\xfb\xac\x94\xa1\x45\xf5\x1c\xe1\x12\xe0\x3b\x67\x21\x54\x5e\x8c\xaa\xcf\xdb\xb4\x51\xd4\x13\xda\xe6\x83\x89\xb6\x92\xe9\x21\x76\xa4\x93\x7d\xe\xfd\x96\x36\x3\x91\x43\x5c\x92\x49\x62\x61\x7b\xeb\x43\x89\xb8\x12\x20\x43\xd4\x47\x6\x84\xee\x47\xe9\x8a\x73\x15\xf\x72\xcf\xed\xce\x96\xb2\x7f\x21\x45\x76\xeb\x26\x28\x83\x6a\xad\xaa\xa6\x81\xd8\x55\xb1\xa3\x85\xb3\xc\xdf\xf1\x69\x2d\x97\x5\x2a\xbc\x7c\x7b\x25\xf8\x80\x9d\x39\x25\xf3\x62\xf0\x66\x5e\xf4\xa0\xcf\xd8\xfd\x4f\xb1\x1f\x60\x3a\x8\x47\xaf\xe1\xf6\x10\x77\x9\xa7\x27\x8f\x9a\x97\x5a\x26\xfa\xfe\x41\x32\x83\x10\xe0\x1d\xbf\x64\xd\xf4\x1c\x32\x35\xe5\x1b\x36\xef\xd4\x4a\x93\x4d\x0\x7c\xec\x2\x7\x8b\x5d\x7d\x1b\xe\xd1\xa6\xa5\x5d\x7d\x57\x88\xa8\xcc\x81\xb4\x86\x4e\xb4\x40\xe9\x1d\xc3\xb1\x24\x3e\x7f\xcc\x8a\x24\x9b\xdf\x6d\xf0\x39\x69\x3e\x4c\xc0\x96\xe4\x13\xda\x90\xda\xf4\x95\x66\x8b\x17\x17\xfe\x39\x43\x25\xaa\xda\xa0\x43\x3c\xb1\x41\x2\xa3\xf0\xa7\x19\x59\xbc\x1d\x7d\x6c\x6d\x91\x9\x5c\xb7\x5b\x1\xd1\x6f\x17\x21\x97\xbf\x89\x71\xa5\xb0\x6e\x7\x45\xfd\x9d\xea\x7\xf6\x7a\x9f\x10\x18\x22\x30\x73\xac\xd4\x6b\x72\x44\xed\xd9\x19\x9b\x2d\x4a\x41\xdd\xd1\x85\x5e\x37\x19\xed\xd2\x15\x8f\x5e\x91\xdb\x33\xf2\xe4\xdb\xff\x98\xfb\xa3\xb5\xca\x21\x69\x8\xe7\x8a\xdf\x90\xff\x3e\xe9\x20\x86\x3c\xe9\xfc\xb\xfe\x5c\x61\xaa\x13\x92\x7f\x7b\xec\xe0\x6d\xa8\x23\x22\xf6\x6b\x77\xc4\xfe\x40\x7\x3b\xb6\xf6\x8e\x5f\xd4\xb9\xb7\xf\x21\x4\xef\x83\x63\x91\x69\x40\xa3\x48\x5c\xd2\x60\xf9\x4f\x6c\x47\x8b\x3b\xb1\x9f\x8e\xee\x16\x8a\x13\xfc\x46\x17\xc3\xc3\x32\x56\xf8\x3c\x85\x3a\xb6\x3e\xaa\x89\x4f\xb3\xdf\x38\xfd\xf1\xe4\x3a\xc0\xe6\x58\xb5\x8f\xc5\x29\xa2\x92\x4a\xb6\xa0\x34\x7f\xab\xb5\x8a\x90\xa1\xdb\x4d\xca\xb6\x2c\x41\x3c\xf7\x2b\x21\xc3\xfd\xf4\x17\x5c\xb5\x33\x17\x68\x2b\x8\x30\xf3\xf7\x30\x3c\x96\xe6\x6a\x20\x97\xe7\x4d\x10\x5f\x47\x5f\x49\x96\x9\xf0\x27\x91\xc8\xf8\x5a\x2e\x79\xb5\xe2\xb8\xe8\xb9\x7b\xd5\x10\xcb\xff\x5d\x14\x73\xf3",
	.ilen	= 512,
	.result	= "\x14\xf6\x41\x37\xa6\xd4\x27\xcd\xdb\x6\x3e\x9a\x4e\xab\xd5\xb1\x1e\x6b\xd2\xbc\x11\xf4\x28\x93\x63\x54\xef\xbb\x5e\x1d\x3a\x1d\x37\x3c\xa\x6c\x1e\xc2\xd1\x2c\xb5\xa3\xb5\x7b\xb8\x8f\x25\xa6\x1b\x61\x1c\xec\x28\x58\x26\xa4\xa8\x33\x28\x25\x5c\x45\x5\xe5\x6c\x99\xe5\x45\xc4\xa2\x3\x84\x3\x73\x1e\x8c\x49\xac\x20\xdd\x8d\xb3\xc4\xf5\xe7\x4f\xf1\xed\xa1\x98\xde\xa4\x96\xdd\x2f\xab\xab\x97\xcf\x3e\xd2\x9e\xb8\x13\x7\x28\x29\x19\xaf\xfd\xf2\x49\x43\xea\x49\x26\x91\xc1\x7\xd6\xbb\x81\x75\x35\xd\x24\x7f\xc8\xda\xd4\xb7\xeb\xe8\x5c\x9\xa2\x2f\xdc\x28\x7d\x3a\x3\xfa\x94\xb5\x1d\x17\x99\x36\xc3\x1c\x18\x34\xe3\x9f\xf5\x55\x7c\xb0\x60\x9d\xff\xac\xd4\x61\xf2\xad\xf8\xce\xc7\xbe\x5c\xd2\x95\xa8\x4b\x77\x13\x19\x59\x26\xc9\xb7\x8f\x6a\xcb\x2d\x37\x91\xea\x92\x9c\x94\x5b\xda\xb\xce\xfe\x30\x20\xf8\x51\xad\xf2\xbe\xe7\xc7\xff\xb3\x33\x91\x6a\xc9\x1a\x41\xc9\xf\xf3\x10\xe\xfd\x53\xff\x6c\x16\x52\xd9\xf3\xf7\x98\x2e\xc9\x7\x31\x2c\xc\x72\xd7\xc5\xc6\x8\x2a\x7b\xda\xbd\x7e\x2\xea\x1a\xbb\xf2\x4\x27\x61\x28\x8e\xf5\x4\x3\x1f\x4c\x7\x55\x82\xec\x1e\xd7\x8b\x2f\x65\x56\xd1\xd9\x1e\x3c\xe9\x1f\x5e\x98\x70\x38\x4a\x8c\x49\xc5\x43\xa0\xa1\x8b\x74\x9d\x4c\x62\xd\x10\xc\xf4\x6c\x8f\xe0\xaa\x9a\x8d\xb7\xe0\xbe\x4c\x87\xf1\x98\x2f\xcc\xed\xc0\x52\x29\xdc\x83\xf8\xfc\x2c\xe\xa8\x51\x4d\x80\xd\xa3\xfe\xd8\x37\xe7\x41\x24\xfc\xfb\x75\xe3\x71\x7b\x57\x45\xf5\x97\x73\x65\x63\x14\x74\xb8\x82\x9f\xf8\x60\x2f\x8a\xf2\x4e\xf1\x39\xda\x33\x91\xf8\x36\xe0\x8d\x3f\x1f\x3b\x56\xdc\xa0\x8f\x3c\x9d\x71\x52\xa7\xb8\xc0\xa5\xc6\xa2\x73\xda\xf4\x4b\x74\x5b\x0\x3d\x99\xd7\x96\xba\xe6\xe1\xa6\x96\x38\xad\xb3\xc0\xd2\xba\x91\x6b\xf9\x19\xdd\x3b\xbe\xbe\x9c\x20\x50\xba\xa1\xd0\xce\x11\xbd\x95\xd8\xd1\xdd\x33\x85\x74\xdc\xdb\x66\x76\x44\xdc\x3\x74\x48\x35\x98\xb1\x18\x47\x94\x7d\xff\x62\xe4\x58\x78\xab\xed\x95\x36\xd9\x84\x91\x82\x64\x41\xbb\x58\xe6\x1c\x20\x6d\x15\x6b\x13\x96\xe8\x35\x7f\xdc\x40\x2c\xe9\xbc\x8a\x4f\x92\xec\x6\x2d\x50\xdf\x93\x5d\x65\x5a\xa8\xfc\x20\x50\x14\xa9\x8a\x7e\x1d\x8\x1f\xe2\x99\xd0\xbe\xfb\x3a\x21\x9d\xad\x86\x54\xfd\xd\x98\x1c\x5a\x6f\x1f\x9a\x40\xcd\xa2\xff\x6a\xf1\x54"
}, {
	.key	= "\xea\xbc\x56\x99\xe3\x50\xff\xc5\xcc\x1a\xd7\xc1\x57\x72\xea\x86\x5b\x89\x88\x61\x3d\x2f\x9b\xb2\xe7\x9c\xec\x74\x6e\x3e\xf4\x3b",
	.nonce	= "\xef\x2d\x63\xee\x6b\x80\x8b\x78",
	.nlen	= 8,
	.assoc	= "\x5a\x27\xff\xeb\xdf\x84\xb2\x9e\xef",
	.alen	= 9,
	.input	= "\xe6\xc3\xdb\x63\x55\x15\xe3\x5b\xb7\x4b\x27\x8b\x5a\xdd\xc2\xe8\x3a\x6b\xd7\x81\x96\x35\x97\xca\xd7\x68\xe8\xef\xce\xab\xda\x9\x6e\xd6\x8e\xcb\x55\xb5\xe1\xe5\x57\xfd\xc4\xe3\xe0\x18\x4f\x85\xf5\x3f\x7e\x4b\x88\xc9\x52\x44\xf\xea\xaf\x1f\x71\x48\x9f\x97\x6d\xb9\x6f\x0\xa6\xde\x2b\x77\x8b\x15\xad\x10\xa0\x2b\x7b\x41\x90\x3\x2d\x69\xae\xcc\x77\x7c\xa5\x9d\x29\x22\xc2\xea\xb4\x0\x1a\xd2\x7a\x98\x8a\xf9\xf7\x82\xb0\xab\xd8\xa6\x94\x8d\x58\x2f\x1\x9e\x0\x20\xfc\x49\xdc\xe\x3\xe8\x45\x10\xd6\xa8\xda\x55\x10\x9a\xdf\x67\x22\x8b\x43\xab\x0\xbb\x2\xc8\xdd\x7b\x97\x17\xd7\x1d\x9e\x2\x5e\x48\xde\x8e\xcf\x99\x7\x95\x92\x3c\x5f\x9f\xc5\x8a\xc0\x23\xaa\xd5\x8c\x82\x6e\x16\x92\xb1\x12\x17\x7\xc3\xfb\x36\xf5\x6c\x35\xd6\x6\x1f\x9f\xa7\x94\xa2\x38\x63\x9c\xb0\x71\xb3\xa5\xd2\xd8\xba\x9f\x8\x1\xb3\xff\x4\x97\x73\x45\x1b\xd5\xa9\x9c\x80\xaf\x4\x9a\x85\xdb\x32\x5b\x5d\x1a\xc1\x36\x28\x10\x79\xf1\x3c\xbf\x1a\x41\x5c\x4e\xdf\xb2\x7c\x79\x3b\x7a\x62\x3d\x4b\xc9\x9b\x2a\x2e\x7c\xa2\xb1\x11\x98\xa7\x34\x1a\x0\xf3\xd1\xbc\x18\x22\xba\x2\x56\x62\x31\x10\x11\x6d\xe0\x54\x9d\x40\x1f\x26\x80\x41\xca\x3f\x68\xf\x32\x1d\xa\x8e\x79\xd8\xa4\x1b\x29\x1c\x90\x8e\xc5\xe3\xb4\x91\x37\x9a\x97\x86\x99\xd5\x9\xc5\xbb\xa3\x3f\x21\x29\x82\x14\x5c\xab\x25\xfb\xf2\x4f\x58\x26\xd4\x83\xaa\x66\x89\x67\x7e\xc0\x49\xe1\x11\x10\x7f\x7a\xda\x29\x4\xff\xf0\xcb\x9\x7c\x9d\xfa\x3\x6f\x81\x9\x31\x60\xfb\x8\xfa\x74\xd3\x64\x44\x7c\x55\x85\xec\x9c\x6e\x25\xb7\x6c\xc5\x37\xb6\x83\x87\x72\x95\x8b\x9d\xe1\x69\x5c\x31\x95\x42\xa6\x2c\xd1\x36\x47\x1f\xec\x54\xab\xa2\x1c\xd8\x0\xcc\xbc\xd\x65\xe2\x67\xbf\xbc\xea\xee\x9e\xe4\x36\x95\xbe\x73\xd9\xa6\xd9\xf\xa0\xcc\x82\x76\x26\xad\x5b\x58\x6c\x4e\xab\x29\x64\xd3\xd9\xa9\x8\x8c\x1d\xa1\x4f\x80\xd8\x3f\x94\xfb\xd3\x7b\xfc\xd1\x2b\xc3\x21\xeb\xe5\x1c\x84\x23\x7f\x4b\xfa\xdb\x34\x18\xa2\xc2\xe5\x13\xfe\x6c\x49\x81\xd2\x73\xe7\xe2\xd7\xe4\x4f\x4b\x8\x6e\xb1\x12\x22\x10\x9d\xac\x51\x1e\x17\xd9\x8a\xb\x42\x88\x16\x81\x37\x7c\x6a\xf7\xef\x2d\xe3\xd9\xf8\x5f\xe0\x53\x27\x74\xb9\xe2\xd6\x1c\x80\x2c\x52\x65",
	.ilen	= 513,
	.result	= "\xfd\x81\x8d\xd0\x3d\xb4\xd5\xdf\xd3\x42\x47\x5a\x6d\x19\x27\x66\x4b\x2e\xc\x27\x9c\x96\x4c\x72\x2\xa3\x65\xc3\xb3\x6f\x2e\xbd\x63\x8a\x4a\x5d\x29\xa2\xd0\x28\x48\xc5\x3d\x98\xa3\xbc\xe0\xbe\x3b\x3f\xe6\x8a\xa4\x7f\x53\x6\xfa\x7f\x27\x76\x72\x31\xa1\xf5\xd6\xc\x52\x47\xba\xcd\x4f\xd7\xeb\x5\x48\xd\x7c\x35\x4a\x9\xc9\x76\x71\x2\xa3\xfb\xb7\x1a\x65\xb7\xed\x98\xc6\x30\x8a\x0\xae\xa1\x31\xe5\xb5\x9e\x6d\x62\xda\xda\x7\xf\x38\x38\xd3\xcb\xc1\xb0\xad\xec\x72\xec\xb1\xa2\x7b\x59\xf3\x3d\x2b\xef\xcd\x28\x5b\x83\xcc\x18\x91\x88\xb0\x2e\xf9\x29\x31\x18\xf9\x4e\xe9\xa\x91\x92\x9f\xae\x2d\xad\xf4\xe6\x1a\xe2\xa4\xee\x47\x15\xbf\x83\x6e\xd7\x72\x12\x3b\x2d\x24\xe9\xb2\x55\xcb\x3c\x10\xf0\x24\x8a\x4a\x2\xea\x90\x25\xf0\xb4\x79\x3a\xef\x6e\xf5\x52\xdf\xb0\xa\xcd\x24\x1c\xd3\x2e\x22\x74\xea\x21\x6f\xe9\xbd\xc8\x3e\x36\x5b\x19\xf1\xca\x99\xa\xb4\xa7\x52\x1a\x4e\xf2\xad\x8d\x56\x85\xbb\x64\x89\xba\x26\xf9\xc7\xe1\x89\x19\x22\x77\xc3\xa8\xfc\xff\xad\xfe\xb9\x48\xae\x12\x30\x9f\x19\xfb\x1b\xef\x14\x87\x8a\x78\x71\xf3\xf4\xb7\x0\x9c\x1d\xb5\x3d\x49\x0\xc\x6\xd4\x50\xf9\x54\x45\xb2\x5b\x43\xdb\x6d\xcf\x1a\xe9\x7a\x7a\xcf\xfc\x8a\x4e\x4d\xb\x7\x63\x28\xd8\xe7\x8\x95\xdf\xa6\x72\x93\x2e\xbb\xa0\x42\x89\x16\xf1\xd9\xc\xf9\xa1\x16\xfd\xd9\x3\xb4\x3b\x8a\xf5\xf6\xe7\x6b\x2e\x8e\x4c\x3d\xe2\xaf\x8\x45\x3\xff\x9\xb6\xeb\x2d\xc6\x1b\x88\x94\xac\x3e\xf1\x9f\xe\xe\x2b\xd5\x0\x4d\x3f\x3b\x53\xae\xaf\x1c\x33\x5f\x55\x6e\x8d\xaf\x5\x7a\x10\x34\xc9\xf4\x66\xcb\x62\x12\xa6\xee\xe8\x1c\x5d\x12\x86\xdb\x6f\x1c\x33\xc4\x1c\xda\x82\x2d\x3b\x59\xfe\xb1\xa4\x59\x41\x86\xd0\xef\xae\xfb\xda\x6d\x11\xb8\xca\xe9\x6e\xff\xf7\xa9\xd9\x70\x30\xfc\x53\xe2\xd7\xa2\x4e\xc7\x91\xd9\x7\x6\xaa\xdd\xb0\x59\x28\x1d\x0\x66\xc5\x54\xc2\xfc\x6\xda\x5\x90\x52\x1d\x37\x66\xee\xf0\xb2\x55\x8a\x5d\xd2\x38\x86\x94\x9b\xfc\x10\x4c\xa1\xb9\x64\x3e\x44\xb8\x5f\xb0\xc\xec\xe0\xc9\xe5\x62\x75\x3f\x9\xd5\xf5\xd9\x26\xba\x9e\xd2\xf4\xb9\x48\xa\xbc\xa2\xd6\x7c\x36\x11\x7d\x26\x81\x89\xcf\xa4\xad\x73\xe\xee\xcc\x6\xa9\xdb\xb1\xfd\xfb\x9\x7f\x90\x42\x37\x2f\xe1\x9c\xf\x6f\xcf\x43\xb5\xd9\x90\xe1\x85\xf5\xa8\xae"
}, {
	.key	= "\x47\x11\xeb\x86\x2b\x2c\xab\x44\x34\xda\x7f\x57\x3\x39\xc\xaf\x2c\x14\xfd\x65\x23\xe9\x8e\x74\xd5\x8\x68\x8\xe7\xb4\x72\xd7",
	.nonce	= "\xdb\x92\xf\x7f\x17\x54\xc\x30",
	.nlen	= 8,
	.assoc	= "\xd2\xa1\x70\xdb\x7a\xf8\xfa\x27\xba\x73\xf\xbf\x3d\x1e\x82\xb2",
	.alen	= 16,
	.input	= "\x42\x93\xe4\xeb\x97\xb0\x57\xbf\x1a\x8b\x1f\xe4\x5f\x36\x20\x3c\xef\xa\xa9\x48\x5f\x5f\x37\x22\x3a\xde\xe3\xae\xbe\xad\x7\xcc\xb1\xf6\xf5\xf9\x56\xdd\xe7\x16\x1e\x7f\xdf\x7a\x9e\x75\xb7\xc7\xbe\xbe\x8a\x36\x4\xc0\x10\xf4\x95\x20\x3\xec\xdc\x5\xa1\x7d\xc4\xa9\x2c\x82\xd0\xbc\x8b\xc5\xc7\x45\x50\xf6\xa2\x1a\xb5\x46\x3b\x73\x2\xa6\x83\x4b\x73\x82\x58\x5e\x3b\x65\x2f\xe\xfd\x2b\x59\x16\xce\xa1\x60\x9c\xe8\x3a\x99\xed\x8d\x5a\xcf\xf6\x83\xaf\xba\xd7\x73\x73\x40\x97\x3d\xca\xef\x7\x57\xe6\xd9\x70\xe\x95\xae\xa6\x8d\x4\xcc\xee\xf7\x9\x31\x77\x12\xa3\x23\x97\x62\xb3\x7b\x32\xfb\x80\x14\x48\x81\xc3\xe5\xea\x91\x39\x52\x81\xa2\x4f\xe4\xb3\x9\xff\xde\x5e\xe9\x58\x84\x6e\xf9\x3d\xdf\x25\xea\xad\xae\xe6\x9a\xd1\x89\x55\xd3\xde\x6c\x52\xdb\x70\xfe\x37\xce\x44\xa\xa8\x25\x5f\x92\xc1\x33\x4a\x4f\x9b\x62\x35\xff\xce\xc0\xa9\x60\xce\x52\x0\x97\x51\x35\x26\x2e\xb9\x36\xa9\x87\x6e\x1e\xcc\x91\x78\x53\x98\x86\x5b\x9c\x74\x7d\x88\x33\xe1\xdf\x37\x69\x2b\xbb\xf1\x4d\xf4\xd1\xf1\x39\x93\x17\x51\x19\xe3\x19\x1e\x76\x37\x25\xfb\x9\x27\x6a\xab\x67\x6f\x14\x12\x64\xe7\xc4\x7\xdf\x4d\x17\xbb\x6d\xe0\xe9\xb9\xab\xca\x10\x68\xaf\x7e\xb7\x33\x54\x73\x7\x6e\xf7\x81\x97\x9c\x5\x6f\x84\x5f\xd2\x42\xfb\x38\xcf\xd1\x2f\x14\x30\x88\x98\x4d\x5a\xa9\x76\xd5\x4f\x3e\x70\x6c\x85\x76\xd7\x1\xa0\x1a\xc8\x4e\xaa\xac\x78\xfe\x46\xde\x6a\x5\x46\xa7\x43\xc\xb9\xde\xb9\x68\xfb\xce\x42\x99\x7\x4d\xb\x3b\x5a\x30\x35\xa8\xf9\x3a\x73\xef\xf\xdb\x1e\x16\x42\xc4\xba\xae\x58\xaa\xf8\xe5\x75\x2f\x1b\x15\x5c\xfd\xa\x97\xd0\xe4\x37\x83\x61\x5f\x43\xa6\xc7\x3f\x38\x59\xe6\xeb\xa3\x90\xc3\xaa\xaa\x5a\xd3\x34\xd4\x17\xc8\x65\x3e\x57\xbc\x5e\xdd\x9e\xb7\xf0\x2e\x5b\xb2\x1f\x8a\x8\xd\x45\x91\xb\x29\x53\x4f\x4c\x5a\x73\x56\xfe\xaf\x41\x1\x39\xa\x24\x3c\x7e\xbe\x4e\x53\xf3\xeb\x6\x66\x51\x28\x1d\xbd\x41\xa\x1\xab\x16\x47\x27\x47\x47\xf7\xcb\x46\xa\x70\x9e\x1\x9c\x9\xe1\x2a\x0\x1a\xd8\xd4\x79\x9d\x80\x15\x8e\x53\x2a\x65\x83\x78\x3e\x3\x0\x7\x12\x1f\x33\x3e\x7b\x13\x37\xf1\xc3\xef\xb7\xc1\x20\x3c\x3e\x67\x66\x5d\x88\xa7\x7d\x33\x50\x77\xb0\x28\x8e\xe7\x2c\x2e\x7a\xf4\x3c\x8d\x74\x83\xaf\x8e\x87\xf\xe4\x50\xff\x84\x5c\x47\xc\x6a\x49\xbf\x42\x86\x77\x15\x48\xa5\x90\x5d\x93\xd6\x2a\x11\xd5\xd5\x11\xaa\xce\xe7\x6f\xa5\xb0\x9\x2c\x8d\xd3\x92\xf0\x5a\x2a\xda\x5b\x1e\xd5\x9a\xc4\xc4\xf3\x49\x74\x41\xca\xe8\xc1\xf8\x44\xd6\x3c\xae\x6c\x1d\x9a\x30\x4\x4d\x27\xe\xb1\x5f\x59\xa2\x24\xe8\xe1\x98\xc5\x6a\x4c\xfe\x41\xd2\x27\x42\x52\xe1\xe9\x7d\x62\xe4\x88\xf\xad\xb2\x70\xcb\x9d\x4c\x27\x2e\x76\x1e\x1a\x63\x65\xf5\x3b\xf8\x57\x69\xeb\x5b\x38\x26\x39\x33\x25\x45\x3e\x91\xb8\xd8\xc7\xd5\x42\xc0\x22\x31\x74\xf4\xbc\xc\x23\xf1\xca\xc1\x8d\xd7\xbe\xc9\x62\xe4\x8\x1a\xcf\x36\xd5\xfe\x55\x21\x59\x91\x87\x87\xdf\x6\xdb\xdf\x96\x45\x58\xda\x5\xcd\x50\x4d\xd2\x7d\x5\x18\x73\x6a\x8d\x11\x85\xa6\x88\xe8\xda\xe6\x30\x33\xa4\x89\x31\x75\xbe\x69\x43\x84\x43\x50\x87\xdd\x71\x36\x83\xc3\x78\x74\x24\xa\xed\x7b\xdb\xa4\x24\xb\xb9\x7e\x5d\xff\xde\xb1\xef\x61\x5a\x45\x33\xf6\x17\x7\x8\x98\x83\x92\xf\x23\x6d\xe6\xaa\x17\x54\xad\x6a\xc8\xdb\x26\xbe\xb8\xb6\x8\xfa\x68\xf1\xd7\x79\x6f\x18\xb4\x9e\x2d\x3f\x1b\x64\xaf\x8d\x6\xe\x49\x28\xe0\x5d\x45\x68\x13\x87\xfa\xde\x40\x7b\xd2\xc3\x94\xd5\xe1\xd9\xc2\xaf\x55\x89\xeb\xb4\x12\x59\xa8\xd4\xc5\x29\x66\x38\xe6\xac\x22\x22\xd9\x64\x9b\x34\xa\x32\x9f\xc2\xbf\x17\x6c\x3f\x71\x7a\x38\x6b\x98\xfb\x49\x36\x89\xc9\xe2\xd6\xc7\x5d\xd0\x69\x5f\x23\x35\xc9\x30\xe2\xfd\x44\x58\x39\xd7\x97\xfb\x5c\x0\xd5\x4f\x7a\x1a\x95\x8b\x62\x4b\xce\xe5\x91\x21\x7b\x30\x0\xd6\xdd\x6d\x2\x86\x49\xf\x3c\x1a\x27\x3c\xd3\xe\x71\xf2\xff\xf5\x2f\x87\xac\x67\x59\x81\xa3\xf7\xf8\xd6\x11\xc\x84\xa9\x3\xee\x2a\xc4\xf3\x22\xab\x7c\xe2\x25\xf5\x67\xa3\xe4\x11\xe0\x59\xb3\xca\x87\xa0\xae\xc9\xa6\x62\x1b\x6e\x4d\x2\x6b\x7\x9d\xfd\xd0\x92\x6\xe1\xb2\x9a\x4a\x1f\x1f\x13\x49\x99\x97\x8\xde\x7f\x98\xaf\x51\x98\xee\x2c\xcb\xf0\xb\xc6\xb6\xb7\x2d\x9a\xb1\xac\xa6\xe3\x15\x77\x9d\x6b\x1a\xe4\xfc\x8b\xf2\x17\x59\x8\x4\x58\x81\x9d\x1b\x1b\x69\x55\xc2\xb4\x3c\x1f\x50\xf1\x7f\x77\x90\x4c\x66\x40\x5a\xc0\x33\x1f\xcb\x5\x6d\x5c\x6\x87\x52\xa2\x8f\x26\xd5\x4f",
	.ilen	= 1024,
	.result	= "\xe5\x26\xa4\x3d\xbd\x33\xd0\x4b\x6f\x5\xa7\x6e\x12\x7a\xd2\x74\xa6\xdd\xbd\x95\xeb\xf9\xa4\xf1\x59\x93\x91\x70\xd9\xfe\x9a\xcd\x53\x1f\x3a\xab\xa6\x7c\x9f\xa6\x9e\xbd\x99\xd9\xb5\x97\x44\xd5\x14\x48\x4d\x9d\xc0\xd0\x5\x96\xeb\x4c\x78\x55\x9\x8\x1\x2\x30\x90\x7b\x96\x7a\x7b\x5f\x30\x41\x24\xce\x68\x61\x49\x86\x57\x82\xdd\x53\x1c\x51\x28\x2b\x53\x6e\x2d\xc2\x20\x4c\xdd\x8f\x65\x10\x20\x50\xdd\x9d\x50\xe5\x71\x40\x53\x69\xfc\x77\x48\x11\xb9\xde\xa4\x8d\x58\xe4\xa6\x1a\x18\x47\x81\x7e\xfc\xdd\xf6\xef\xce\x2f\x43\x68\xd6\x6\xe2\x74\x6a\xad\x90\xf5\x37\xf3\x3d\x82\x69\x40\xe9\x6b\xa7\x3d\xa8\x1e\xd2\x2\x7c\xb7\x9b\xe4\xda\x8f\x95\x6\xc5\xdf\x73\xa3\x20\x9a\x49\xde\x9c\xbc\xee\x14\x3f\x81\x5e\xf8\x3b\x59\x3c\xe1\x68\x12\x5a\x3a\x76\x3a\x3f\xf7\x87\x33\xa\x1\xb8\xd4\xed\xb6\xbe\x94\x5e\x70\x40\x56\x67\x1f\x50\x44\x19\xce\x82\x70\x10\x87\x13\x20\xb\x4c\x5a\xb6\xf6\xa7\xae\x81\x75\x1\x81\xe6\x4b\x57\x7c\xdd\x6d\xf8\x1c\x29\x32\xf7\xda\x3c\x2d\xf8\x9b\x25\x6e\x0\xb4\xf7\x2f\xf7\x4\xf7\xa1\x56\xac\x4f\x1a\x64\xb8\x47\x55\x18\x7b\x7\x4d\xbd\x47\x24\x80\x5d\xa2\x70\xc5\xdd\x8e\x82\xd4\xeb\xec\xb2\xc\x39\xd2\x97\xc1\xcb\xeb\xf4\x77\x59\xb4\x87\xef\xcb\x43\x2d\x46\x54\xd1\xa7\xd7\x15\x99\xa\x43\xa1\xe0\x99\x33\x71\xc1\xed\xfe\x72\x46\x33\x8e\x91\x8\x9f\xc8\x2e\xca\xfa\xdc\x59\xd5\xc3\x76\x84\x9f\xa3\x37\x68\xc3\xf0\x47\x2c\x68\xdb\x5e\xc3\x49\x4c\xe8\x92\x85\xe2\x23\xd3\x3f\xad\x32\xe5\x2b\x82\xd7\x8f\x99\xa\x59\x5c\x45\xd9\xb4\x51\x52\xc2\xae\xbf\x80\xcf\xc9\xc9\x51\x24\x2a\x3b\x3a\x4d\xae\xeb\xbd\x22\xc3\xe\xf\x59\x25\x92\x17\xe9\x74\xc7\x8b\x70\x70\x36\x55\x95\x75\x4b\xad\x61\x2b\x9\xbc\x82\xf2\x6e\x94\x43\xae\xc3\xd5\xcd\x8e\xfe\x5b\x9a\x88\x43\x1\x75\xb2\x23\x9\xf7\x89\x83\xe7\xfa\xf9\xb4\x9b\xf8\xef\xbd\x1c\x92\xc1\xda\x7e\xfe\x5\xba\x5a\xcd\x7\x6a\x78\x9e\x5d\xfb\x11\x2f\x79\x38\xb6\xc2\x5b\x6b\x51\xb4\x71\xdd\xf7\x2a\xe4\xf4\x72\x76\xad\xc2\xdd\x64\x5d\x79\xb6\xf5\x7a\x77\x20\x5\x3d\x30\x6\xd4\x4c\xa\x2c\x98\x5a\xb9\xd4\x98\xa9\x3f\xc6\x12\xea\x3b\x4b\xc5\x79\x64\x63\x6b\x9\x54\x3b\x14\x27\xba\x99\x80\xc8\x72\xa8\x12\x90\x29\xba\x40\x54\x97\x2b\x7b\xfe\xeb\xcd\x1\x5\x44\x72\xdb\x99\xe4\x61\xc9\x69\xd6\xb9\x28\xd1\x5\x3e\xf9\xb\x49\xa\x49\xe9\x8d\xe\xa7\x4a\xf\xaf\x32\xd0\xe0\xb2\x3a\x55\x58\xfe\x5c\x28\x70\x51\x23\xb0\x7b\x6a\x5f\x1e\xb8\x17\xd7\x94\x15\x8f\xee\x20\xc7\x42\x25\x3e\x9a\x14\xd7\x60\x72\x39\x47\x48\xa9\xfe\xdd\x47\xa\xb1\xe6\x60\x28\x8c\x11\x68\xe1\xff\xd7\xce\xc8\xbe\xb3\xfe\x27\x30\x9\x70\xd7\xfa\x2\x33\x3a\x61\x2e\xc7\xff\xa4\x2a\xa8\x6e\xb4\x79\x35\x6d\x4c\x1e\x38\xf8\xee\xd4\x84\x4e\x6e\x28\xa7\xce\xc8\xc1\xcf\x80\x5\xf3\x4\xef\xc8\x18\x28\x2e\x8d\x5e\xc\xdf\xb8\x5f\x96\xe8\xc6\x9c\x2f\xe5\xa6\x44\xd7\xe7\x99\x44\xc\xec\xd7\x5\x60\x97\xbb\x74\x77\x58\xd5\xbb\x48\xde\x5a\xb2\x54\x7f\xe\x46\x70\x6a\x6f\x78\xa5\x8\x89\x5\x4e\x7e\xa0\x69\xb4\x40\x60\x55\x77\x75\x9b\x19\xf2\xd5\x13\x80\x77\xf9\x4b\x3f\x1e\xee\xe6\x76\x84\x7b\x8c\xe5\x27\xa8\xa\x91\x1\x68\x71\x8a\x3f\x6\xab\xf6\xa9\xa5\xe6\x72\x92\xe4\x67\xe2\xa2\x46\x35\x84\x55\x7d\xca\xa8\x85\xd0\xf1\x3f\xbe\xd7\x34\x64\xfc\xae\xe3\xe4\x4\x9f\x66\x2\xb9\x88\x10\xd9\xc4\x4c\x31\x43\x7a\x93\xe2\x9b\x56\x43\x84\xdc\xdc\xde\x1d\xa4\x2\xe\xc2\xef\xc3\xf8\x78\xd1\xb2\x6b\x63\x18\xc9\xa9\xe5\x72\xd8\xf3\xb9\xd1\x8a\xc7\x1a\x2\x27\x20\x77\x10\xe5\xc8\xd4\x4a\x47\xe5\xdf\x5f\x1\xaa\xb0\xd4\x10\xbb\x69\xe3\x36\xc8\xe1\x3d\x43\xfb\x86\xcd\xcc\xbf\xf4\x88\xe0\x20\xca\xb7\x1b\xf1\x2f\x5c\xee\xd4\xd3\xa3\xcc\xa4\x1e\x1c\x47\xfb\xbf\xfc\xa2\x41\x55\x9d\xf6\x5a\x5e\x65\x32\x34\x7b\x52\x8d\xd5\xd0\x20\x60\x3\xab\x3f\x8c\xd4\x21\xea\x2a\xd9\xc4\xd0\xd3\x65\xd8\x7a\x13\x28\x62\x32\x4b\x2c\x87\x93\xa8\xb4\x52\x45\x9\x44\xec\xec\xc3\x17\xdb\x9a\x4d\x5c\xa9\x11\xd4\x7d\xaf\x9e\xf1\x2d\xb2\x66\xc5\x1d\xed\xb7\xcd\xb\x25\x5e\x30\x47\x3f\x40\xf4\xa1\xa0\x0\x94\x10\xc5\x6a\x63\x1a\xd5\x88\x92\x8e\x82\x39\x87\x3c\x78\x65\x58\x42\x75\x5b\xdd\x77\x3e\x9\x4e\x76\x5b\xe6\xe\x4d\x38\xb2\xc0\xb8\x95\x1\x7a\x10\xe0\xfb\x7\xf2\xab\x2d\x8c\x32\xed\x2b\xc0\x46\xc2\xf5\x38\x83\xf0\x17\xec\xc1\x20\x6a\x9a\xb\x0\xa0\x98\x22\x50\x23\xd5\x80\x6b\xf6\x1f\xc3\xcc\x97\xc9\x24\x9f\xf3\xaf\x43\x14\xd5\xa0"
}, {
	.key	= "\x35\x4e\xb5\x70\x50\x42\x8a\x85\xf2\xfb\xed\x7b\xd0\x9e\x97\xca\xfa\x98\x66\x63\xee\x37\xcc\x52\xfe\xd1\xdf\x95\x15\x34\x29\x38",
	.nonce	= "\xfd\x87\xd4\xd8\x62\xfd\xec\xaa",
	.nlen	= 8,
	.assoc	= "\xd6\x31\xda\x5d\x42\x5e\xd7",
	.alen	= 7,
	.input	= "\x7a\x57\xf2\xc7\x6\x3f\x50\x7b\x36\x1a\x66\x5c\xb9\xe\x5e\x3b\x45\x60\xbe\x9a\x31\x9f\xff\x5d\x66\x34\xb4\xdc\xfb\x9d\x8e\xee\x6a\x33\xa4\x7\x3c\xf9\x4c\x30\xa1\x24\x52\xf9\x50\x46\x88\x20\x2\x32\x3a\xe\x99\x63\xaf\x1f\x15\x28\x2a\x5\xff\x57\x59\x5e\x18\xa1\x1f\xd0\x92\x5c\x88\x66\x1b\x0\x64\xa5\x93\x8d\x6\x46\xb0\x64\x8b\x8b\xef\x99\x5\x35\x85\xb3\xf3\x33\xbb\xec\x66\xb6\x3d\x57\x42\xe3\xb4\xc6\xaa\xb0\x41\x2a\xb9\x59\xa9\xf6\x3e\x15\x26\x12\x3\x21\x4c\x74\x43\x13\x2a\x3\x27\x9\xb4\xfb\xe7\xb7\x40\xff\x5e\xce\x48\x9a\x60\xe3\x8b\x80\x8c\x38\x2d\xcb\x93\x37\x74\x5\x52\x6f\x73\x3e\xc3\xbc\xca\x72\xa\xeb\xf1\x3b\xa0\x95\xdc\x8a\xc4\xa9\xdc\xca\x44\xd8\x8\x63\x6a\x36\xd3\x3c\xb8\xac\x46\x7d\xfd\xaa\xeb\x3e\xf\x45\x8f\x49\xda\x2b\xf2\x12\xbd\xaf\x67\x8a\x63\x48\x4b\x55\x5f\x6d\x8c\xb9\x76\x34\x84\xae\xc2\xfc\x52\x64\x82\xf7\xb0\x6\xf0\x45\x73\x12\x50\x30\x72\xea\x78\x9a\xa8\xaf\xb5\xe3\xbb\x77\x52\xec\x59\x84\xbf\x6b\x8f\xce\x86\x5e\x1f\x23\xe9\xfb\x8\x86\xf7\x10\xb9\xf2\x44\x96\x44\x63\xa9\xa8\x78\x0\x23\xd6\xc7\xe7\x6e\x66\x4f\xcc\xee\x15\xb3\xbd\x1d\xa0\xe5\x9c\x1b\x24\x2c\x4d\x3c\x62\x35\x9c\x88\x59\x9\xdd\x82\x1b\xcf\xa\x83\x6b\x3f\xae\x3\xc4\xb4\xdd\x7e\x5b\x28\x76\x25\x96\xd9\xc9\x9d\x5f\x86\xfa\xf6\xd7\xd2\xe6\x76\x1d\xf\xa1\xdc\x74\x5\x1b\x1d\xe0\xcd\x16\xb0\xa8\x8a\x34\x7b\x15\x11\x77\xe5\x7b\x7e\x20\xf7\xda\x38\xda\xce\x70\xe9\xf5\x6c\xd9\xbe\xc\x4c\x95\x4c\xc2\x9b\x34\x55\x55\xe1\xf3\x46\x8e\x48\x74\x14\x4f\x9d\xc9\xf5\xe8\x1a\xf0\x11\x4a\xc1\x8d\xe0\x93\xa0\xbe\x9\x1c\x2b\x4e\xf\xb2\x87\x8b\x84\xfe\x92\x32\x14\xd7\x93\xdf\xe7\x44\xbc\xc5\xae\x53\x69\xd8\xb3\x79\x37\x80\xe3\x17\x5c\xec\x53\x0\x9a\xe3\x8e\xdc\x38\xb8\x66\xf0\xd3\xad\x1d\x2\x96\x86\x3e\x9d\x3b\x5d\xa5\x7f\x21\x10\xf1\x1f\x13\x20\xf9\x57\x87\x20\xf5\x5f\xf1\x17\x48\xa\x51\x5a\xcd\x19\x3\xa6\x5a\xd1\x12\x97\xe9\x48\xe2\x1d\x83\x75\x50\xd9\x75\x7d\x6a\x82\xa1\xf9\x4e\x54\x87\x89\xc9\xc\xb7\x5b\x6a\x91\xc1\x9c\xb2\xa9\xdc\x9a\xa4\x49\xa\x6d\xd\xbb\xde\x86\x44\xdd\x5d\x89\x2b\x96\xf\x23\x95\xad\xcc\xa2\xb3\xb9\x7e\x74\x38\xba\x9f\x73\xae\x5f\xf8\x68\xa2\xe0\xa9\xce\xbd\x40\xd4\x4c\x6b\xd2\x56\x62\xb0\xcc\x63\x7e\x5b\xd3\xae\xd1\x75\xce\xbb\xb4\x5b\xa8\xf8\xb4\xac\x71\x75\xaa\xc9\x9f\xbb\x6c\xad\xf\x55\x5d\xe8\x85\x7d\xf9\x21\x35\xea\x92\x85\x2b\x0\xec\x84\x90\xa\x63\x96\xe4\x6b\xa9\x77\xb8\x91\xf8\x46\x15\x72\x63\x70\x1\x40\xa3\xa5\x76\x62\x2b\xbf\xf1\xe5\x8d\x9f\xa3\xfa\x9b\x3\xbe\xfe\x65\x6f\xa2\x29\xd\x54\xb4\x71\xce\xa9\xd6\x3d\x88\xf9\xaf\x6b\xa8\x9e\xf4\x16\x96\x36\xb9\x0\xdc\x10\xab\xb5\x8\x31\x1f\x0\xb1\x3c\xd9\x38\x3e\xc6\x4\xa7\x4e\xe8\xae\xed\x98\xc2\xf7\xb9\x0\x5f\x8c\x60\xd1\xe5\x15\xf7\xae\x1e\x84\x88\xd1\xf6\xbc\x3a\x89\x35\x22\x83\x7c\xca\xf0\x33\x82\x4c\x79\x3c\xfd\xb1\xae\x52\x62\x55\xd2\x41\x60\xc6\xbb\xfa\xe\x59\xd6\xa8\xfe\x5d\xed\x47\x3d\xe0\xea\x1f\x6e\x43\x51\xec\x10\x52\x56\x77\x42\x6b\x52\x87\xd8\xec\xe0\xaa\x76\xa5\x84\x2a\x22\x24\xfd\x92\x40\x88\xd5\x85\x1c\x1f\x6b\x47\xa0\xc4\xe4\xef\xf4\xea\xd7\x59\xac\x2a\x9e\x8c\xfa\x1f\x42\x8\xfe\x4f\x74\xa0\x26\xf5\xb3\x84\xf6\x58\x5f\x26\x66\x3e\xd7\xe4\x22\x91\x13\xc8\xac\x25\x96\x23\xd8\x9\xea\x45\x75\x23\xb8\x5f\xc2\x90\x8b\x9\xc4\xfc\x47\x6c\x6d\xa\xef\x69\xa4\x38\x19\xcf\x7d\xf9\x9\x73\x9b\x60\x5a\xf7\x37\xb5\xfe\x9f\xe3\x2b\x4c\xd\x6e\x19\xf1\xd6\xc0\x70\xf3\x9d\x22\x3c\xf9\x49\xce\x30\x8e\x44\xb5\x76\x15\x8f\x52\xfd\xa5\x4\xb8\x55\x6a\x36\x59\x7c\xc4\x48\xb8\xd7\xab\x5\x66\xe9\x5e\x21\x6f\x6b\x36\x29\xbb\xe9\xe3\xa2\x9a\xa8\xcd\x55\x25\x11\xba\x5a\x58\xa0\xde\xae\x19\x2a\x48\x5a\xff\x36\xcd\x6d\x16\x7a\x73\x38\x46\xe5\x47\x59\xc8\xa2\xf6\xe2\x6c\x83\xc5\x36\x2c\x83\x7d\xb4\x1\x5\x69\xe7\xaf\x5c\xc4\x64\x82\x12\x21\xef\xf7\xd1\x7d\xb8\x8d\x8c\x98\x7c\x5f\x7d\x92\x88\xb9\x94\x7\x9c\xd8\xe9\x9c\x17\x38\xe3\x57\x6c\xe0\xdc\xa5\x92\x42\xb3\xbd\x50\xa2\x7e\xb5\xb1\x52\x72\x3\x97\xd8\xaa\x9a\x1e\x75\x41\x11\xa3\x4f\xcc\xd4\xe3\x73\xad\x96\xdc\x47\x41\x9f\xb0\xbe\x79\x91\xf5\xb6\x18\xfe\xc2\x83\x18\x7d\x73\xd9\x4f\x83\x84\x3\xb3\xf0\x77\x66\x3d\x83\x63\x2e\x2c\xf9\xdd\xa6\x1f\x89\x82\xb8\x23\x42\xeb\xe2\xca\x70\x82\x61\x41\xa\x6d\x5f\x75\xc5\xe2\xc4\x91\x18\x44\x22\xfa\x34\x10\xf5\x20\xdc\xb7\xdd\x2a\x20\x77\xf5\xf9\xce\xdb\xa0\xa\x52\x2a\x4e\xdd\xcc\x97\xdf\x5\xe4\x5e\xb7\xaa\xf0\xe2\x80\xff\xba\x1a\xf\xac\xdf\x2\x32\xe6\xf7\xc7\x17\x13\xb7\xfc\x98\x48\x8c\xd\x82\xc9\x80\x7a\xe2\xa\xc5\xb4\xde\x7c\x3c\x79\x81\xe\x28\x65\x79\x67\x82\x69\x44\x66\x9\xf7\x16\x1a\xf9\x7d\x80\xa1\x79\x14\xa9\xc8\x20\xfb\xa2\x46\xbe\x8\x35\x17\x58\xc1\x1a\xda\x2a\x6b\x2e\x1e\xe6\x27\x55\x7b\x19\xe2\xfb\x64\xfc\x5e\x15\x54\x3c\xe7\xc2\x11\x50\x30\xb8\x72\x3\xb\x1a\x9f\x86\x27\x11\x5c\x6\x2b\xbd\x75\x1a\xa\xda\x1\xfa\x5c\x4a\xc1\x80\x3a\x6e\x30\xc8\x2c\xeb\x56\xec\x89\xfa\x35\x7b\xb2\xf0\x97\x8\x86\x53\xbe\xbd\x40\x41\x38\x1c\xb4\x8b\x79\x2e\x18\x96\x94\xde\xe8\xca\xe5\x9f\x92\x9f\x15\x5d\x56\x60\x5c\x9\xf9\x16\xf4\x17\xf\xf6\x4c\xda\xe6\x67\x89\x9f\xca\x6c\xe7\x9b\x4\x62\xe\x26\xa6\x52\xbd\x29\xff\xc7\xa4\x96\xe6\x6a\x2\xa5\x2e\x7b\xfe\x97\x68\x3e\x2e\x5f\x3b\xf\x36\xd6\x98\x19\x59\x48\xd2\xc6\xe1\x55\x1a\x6e\xd6\xed\x2c\xba\xc3\x9e\x64\xc9\x95\x86\x35\x5e\x3e\x88\x69\x99\x4b\xee\xbe\x9a\x99\xb5\x6e\x58\xae\xdd\x22\xdb\xdd\x6b\xfc\xaf\x90\xa3\x3d\xa4\xc1\x15\x92\x18\x8d\xd2\x4b\x7b\x6\xd1\x37\xb5\xe2\x7c\x2c\xf0\x25\xe4\x94\x2a\xbd\xe3\x82\x70\x78\xa3\x82\x10\x5a\x90\xd7\xa4\xfa\xaf\x1a\x88\x59\xdc\x74\x12\xb4\x8e\xd7\x19\x46\xf4\x84\x69\x9f\xbb\x70\xa8\x4c\x52\x81\xa9\xff\x76\x1c\xae\xd8\x11\x3d\x7f\x7d\xc5\x12\x59\x28\x18\xc2\xa2\xb7\x1c\x88\xf8\xd6\x1b\xa6\x7d\x9e\xde\x29\xf8\xed\xff\xeb\x92\x24\x4f\x5\xaa\xd9\x49\xba\x87\x59\x51\xc9\x20\x5c\x9b\x74\xcf\x3\xd9\x2d\x34\xc7\x5b\xa5\x40\xb2\x99\xf5\xcb\xb4\xf6\xb7\x72\x4a\xd6\xbd\xb0\xf3\x93\xe0\x1b\xa8\x4\x1e\x35\xd4\x80\x20\xf4\x9c\x31\x6b\x45\xb9\x15\xb0\x5e\xdd\xa\x33\x9c\x83\xcd\x58\x89\x50\x56\xbb\x81\x0\x91\x32\xf3\x1b\x3e\xcf\x45\xe1\xf9\xe1\x2c\x26\x78\x93\x9a\x60\x46\xc9\xb5\x5e\x6a\x28\x92\x87\x3f\x63\x7b\xdb\xf7\xd0\x13\x9d\x32\x40\x5e\xcf\xfb\x79\x68\x47\x4c\xfd\x1\x17\xe6\x97\x93\x78\xbb\xa6\x27\xa3\xe8\x1a\xe8\x94\x55\x7d\x8\xe5\xdc\x66\xa3\x69\xc8\xca\xc5\xa1\x84\x55\xde\x8\x91\x16\x3a\xc\x86\xab\x27\x2b\x64\x34\x2\x6c\x76\x8b\xc6\xaf\xcc\xe1\xd6\x8c\x2a\x18\x3d\xa6\x1b\x37\x75\x45\x73\xc2\x75\xd7\x53\x78\x3a\xd6\xe8\x29\xd2\x4a\xa8\x1e\x82\xf6\xb6\x81\xde\x21\xed\x2b\x56\xbb\xf2\xd0\x57\xc1\x7c\xd2\x6a\xd2\x56\xf5\x13\x5f\x1c\x6a\xb\x74\xfb\xe9\xfe\x9e\xea\x95\xb2\x46\xab\xa\xfc\xfd\xf3\xbb\x4\x2b\x76\x1b\xa4\x74\xb0\xc1\x78\xc3\x69\xe2\xb0\x1\xe1\xde\x32\x4c\x8d\x1a\xb3\x38\x8\xd5\xfc\x1f\xdc\xe\x2c\x9c\xb1\xa1\x63\x17\x22\xf5\x6c\x93\x70\x74\x0\xf8\x39\x1\x94\xd1\x32\x23\x56\x5d\xa6\x2\x76\x76\x93\xce\x2f\x19\xe9\x17\x52\xae\x6e\x2c\x6d\x61\x7f\x3b\xaa\xe0\x52\x85\xc5\x65\xc1\xbb\x8e\x5b\x21\xd5\xc9\x78\x83\x7\x97\x4c\x62\x61\x41\xd4\xfc\xc9\x39\xe3\x9b\xd0\xcc\x75\xc4\x97\xe6\xdd\x2a\x5f\xa6\xe8\x59\x6c\x98\xb9\x2\xe2\xa2\xd6\x68\xee\x3b\x1d\xe3\x4d\x5b\x30\xef\x3\xf2\xeb\x18\x57\x36\xe8\xa1\xf4\x47\xfb\xcb\x8f\xcb\xc8\xf3\x4f\x74\x9d\x9d\xb1\x8d\x14\x44\xd9\x19\xb4\x54\x4f\x75\x19\x9\xa0\x75\xbc\x3b\x82\xc6\x3f\xb8\x83\x19\x6e\xd6\x37\xfe\x6e\x8a\x4e\xe0\x4a\xab\x7b\xc8\xb4\x1d\xf4\xed\x27\x3\x65\xa2\xa1\xae\x11\xe7\x98\x78\x48\x91\xd2\xd2\xd4\x23\x78\x50\xb1\x5b\x85\x10\x8d\xca\x5f\xf\x71\xae\x72\x9a\xf6\x25\x19\x60\x6\xf7\x10\x34\x18\xd\xc9\x9f\x7b\xc\x9b\x8f\x91\x1b\x9f\xcd\x10\xee\x75\xf9\x97\x66\xfc\x4d\x33\x6e\x28\x2b\x92\x85\x4f\xab\x43\x8d\x8f\x7d\x86\xa7\xc7\xd8\xd3\xb\x8b\x57\xb6\x1d\x95\xd\xe9\xbc\xd9\x3\xd9\x10\x19\xc3\x46\x63\x55\x87\x61\x79\x6c\x95\xe\x9c\xdd\xca\xc3\xf3\x64\xf0\x7d\x76\xb7\x53\x67\x2b\x1e\x44\x56\x81\xea\x8f\x5c\x42\x16\xb8\x28\xeb\x1b\x61\x10\x1e\xbf\xec\xa8",
	.ilen	= 1933,
	.result	= "\x6a\xfc\x4b\x25\xdf\xc0\xe4\xe8\x17\x4d\x4c\xc9\x7e\xde\x3a\xcc\x3c\xba\x6a\x77\x47\xdb\xe3\x74\x7a\x4d\x5f\x8d\x37\x55\x80\x73\x90\x66\x5d\x3a\x7d\x5d\x86\x5e\x8d\xfd\x83\xff\x4e\x74\x6f\xf9\xe6\x70\x17\x70\x3e\x96\xa7\x7e\xcb\xab\x8f\x58\x24\x9b\x1\xfd\xcb\xe6\x4d\x9b\xf0\x88\x94\x57\x66\xef\x72\x4c\x42\x6e\x16\x19\x15\xea\x70\x5b\xac\x13\xdb\x9f\x18\xe2\x3c\x26\x97\xbc\xdc\x45\x8c\x6c\x24\x69\x9c\xf7\x65\x1e\x18\x59\x31\x7c\xe4\x73\xbc\x39\x62\xc6\x5c\x9f\xbf\xfa\x90\x3\xc9\x72\x26\xb6\x1b\xc2\xb7\x3f\xf2\x13\x77\xf2\x8d\xb9\x47\xd0\x53\xdd\xc8\x91\x83\x8b\xb1\xce\xa3\xfe\xcd\xd9\xdd\x92\x7b\xdb\xb8\xfb\xc9\x2d\x1\x59\x39\x52\xad\x1b\xec\xcf\xd7\x70\x13\x21\xf5\x47\xaa\x18\x21\x5c\xc9\x9a\xd2\x6b\x5\x9c\x1\xa1\xda\x35\x5d\xb3\x70\xe6\xa9\x80\x8b\x91\xb7\xb3\x5f\x24\x9a\xb7\xd1\x6b\xa1\x1c\x50\xba\x49\xe0\xee\x2e\x75\xac\x69\xc0\xeb\x3\xdd\x19\xe5\xf6\x6\xdd\xc3\xd7\x2b\x7\x7\x30\xa7\x19\xc\xbf\xe6\x18\xcc\xb1\x1\x11\x85\x77\x1d\x96\xa7\xa3\x0\x84\x2\xa2\x83\x68\xda\x17\x27\xc8\x7f\x23\xb7\xf4\x13\x85\xcf\xdd\x7a\x7d\x24\x57\xfe\x5\x93\xf5\x74\xce\xed\xc\x20\x98\x8d\x92\x30\xa1\x29\x23\x1a\xa0\x4f\x69\x56\x4c\xe1\xc8\xce\xf6\x9a\xc\xa4\xfa\x4\xf6\x62\x95\xf2\xfa\xc7\x40\x68\x40\x8f\x41\xda\xb4\x26\x6f\x70\xab\x40\x61\xa4\xe\x75\xfb\x86\xeb\x9d\x9a\x1f\xec\x76\x99\xe7\xea\xaa\x1e\x2d\xb5\xd4\xa6\x1a\xb8\x61\xa\x1d\x16\x5b\x98\xc2\x31\x40\xe7\x23\x1d\x66\x99\xc8\xc0\xd7\xce\xf3\x57\x40\x4\x3f\xfc\xea\xb3\xfc\xd2\xd3\x99\xa4\x94\x69\xa0\xef\xd1\x85\xb3\xa6\xb1\x28\xbf\x94\x67\x22\xc3\x36\x46\xf8\xd2\xf\x5f\xf4\x59\x80\xe6\x2d\x43\x8\x7d\x19\x9\x97\xa7\x4c\x3d\x8d\xba\x65\x62\xa3\x71\x33\x29\x62\xdb\xc1\x33\x34\x1a\x63\x33\x16\xb6\x64\x7e\xab\x33\xf0\xe6\x26\x68\xba\x1d\x2e\x38\x8\xe6\x2\xd3\x25\x2c\x47\x23\x58\x34\xf\x9d\x63\x4f\x63\xbb\x7f\x3b\x34\x38\xa7\xb5\x8d\x65\xd9\x9f\x79\x55\x3e\x4d\xe7\x73\xd8\xf6\x98\x97\x84\x60\x9c\xc8\xa9\x3c\xf6\xdc\x12\x5c\xe1\xbb\xb\x8b\x98\x9c\x9d\x26\x7c\x4a\xe6\x46\x36\x58\x21\x4a\xee\xca\xd7\x3b\xc2\x6c\x49\x2f\xe5\xd5\x3\x59\x84\x53\xcb\xfe\x92\x71\x2e\x7c\x21\xcc\x99\x85\x7f\xb8\x74\x90\x13\x42\x3f\xe0\x6b\x1d\xf2\x4d\x54\xd4\xfc\x3a\x5\xe6\x74\xaf\xa6\xa0\x2a\x20\x23\x5d\x34\x5c\xd9\x3e\x4e\xfa\x93\xe7\xaa\xe9\x6f\x8\x43\x67\x41\xc5\xad\xfb\x31\x95\x82\x73\x32\xd8\xa6\xa3\xed\xe\x2d\xf6\x5f\xfd\x80\xa6\x7a\xe0\xdf\x78\x15\x29\x74\x33\xd0\x9e\x83\x86\x72\x22\x57\x29\xb9\x9e\x5d\xd3\x1a\xb5\x96\x72\x41\x3d\xf1\x64\x43\x67\xee\xaa\x5c\xd3\x9a\x96\x13\x11\x5d\xf3\xc\x87\x82\x1e\x41\x9e\xd0\x27\xd7\x54\x3b\x67\x73\x9\x91\xe9\xd5\x36\xa7\xb5\x55\xe4\xf3\x21\x51\x49\x22\x7\x55\x4f\x44\x4b\xd2\x15\x93\x17\x2a\xfa\x4d\x4a\x57\xdb\x4c\xa6\xeb\xec\x53\x25\x6c\x21\xed\x0\x4c\x3b\xca\x14\x57\xa9\xd6\x6a\xcd\x8d\x5e\x74\xac\x72\xc1\x97\xe5\x1b\x45\x4e\xda\xfc\xcc\x40\xe8\x48\x88\xb\xa3\xe3\x8d\x83\x42\xc3\x23\xfd\x68\xb5\x8e\xf1\x9d\x63\x77\xe9\xa3\x8e\x8c\x26\x6b\xbd\x72\x73\x35\xc\x3\xf8\x43\x78\x52\x71\x15\x1f\x71\x5d\x6e\xed\xb9\xcc\x86\x30\xdb\x2b\xd3\x82\x88\x23\x71\x90\x53\x5c\xa9\x2f\x76\x1\xb7\x9a\xfe\x43\x55\xa3\x4\x9b\xe\xe4\x59\xdf\xc9\xe9\xb1\xea\x29\x28\x3c\x5c\xae\x72\x84\xb6\xc6\xeb\xc\x27\x7\x74\x90\xd\x31\xb0\x0\x77\xe9\x40\x70\x6f\x68\xa7\xfd\x6\xec\x4b\xc0\xb7\xac\xbc\x33\xb7\x6d\xa\xbd\x12\x1b\x59\xcb\xdd\x32\xf5\x1d\x94\x57\x76\x9e\xc\x18\x98\x71\xd7\x2a\xdb\xb\x7b\xa7\x71\xb7\x67\x81\x23\x96\xae\xb9\x7e\x32\x43\x92\x8a\x19\xa0\xc4\xd4\x3b\x57\xf9\x4a\x2c\xfb\x51\x46\xbb\xcb\x5d\xb3\xef\x13\x93\x6e\x68\x42\x54\x57\xd3\x6a\x3a\x8f\x9d\x66\xbf\xbd\x36\x23\xf5\x93\x83\x7b\x9c\xc0\xdd\xc5\x49\xc0\x64\xed\x7\x12\xb3\xe6\xe4\xe5\x38\x95\x23\xb1\xa0\x3b\x1a\x61\xda\x17\xac\xc3\x58\xdd\x74\x64\x22\x11\xe8\x32\x1d\x16\x93\x85\x99\xa5\x9c\x34\x55\xb1\xe9\x20\x72\xc9\x28\x7b\x79\x0\xa1\xa6\xa3\x27\x40\x18\x8a\x54\xe0\xcc\xe8\x4e\x8e\x43\x96\xe7\x3f\xc8\xe9\xb2\xf9\xc9\xda\x4\x71\x50\x47\xe4\xaa\xce\xa2\x30\xc8\xe4\xac\xc7\xd\x6\x2e\xe6\xe8\x80\x36\x29\x9e\x1\xb8\xc3\xf0\xa0\x5d\x7a\xca\x4d\xa0\x57\xbd\x2a\x45\xa7\x7f\x9c\x93\x7\x8f\x35\x67\x92\xe3\xe9\x7f\xa8\x61\x43\x9e\x25\x4f\x33\x76\x13\x6e\x12\xb9\xdd\xa4\x7c\x8\x9f\x7c\xe7\xa\x8d\x84\x6\xa4\x33\x17\x34\x5e\x10\x7c\xc0\xa8\x3d\x1f\x42\x20\x51\x65\x5d\x9\xc3\xaa\xc0\xc8\xd\xf0\x79\xbc\x20\x1b\x95\xe7\x6\x7d\x47\x20\x3\x1a\x74\xdd\xe2\xd4\xae\x38\x71\x9b\xf5\x80\xec\x8\x4e\x56\xba\x76\x12\x1a\xdf\x48\xf3\xae\xb3\xe6\xe6\xbe\xc0\x91\x2e\x1\xb3\x1\x86\xa2\xb9\x52\xd1\x21\xae\xd4\x97\x1d\xef\x41\x12\x95\x3d\x48\x45\x1c\x56\x32\x8f\xb8\x43\xbb\x19\xf3\xca\xe9\xeb\x6d\x84\xbe\x86\x6\xe2\x36\xb2\x62\x9d\xd3\x4c\x48\x18\x54\x13\x4e\xcf\xfd\xba\x84\xb9\x30\x53\xcf\xfb\xb9\x29\x8f\xdc\x9f\xef\x60\xb\x64\xf6\x8b\xee\xa6\x91\xc2\x41\x6c\xf6\xfa\x79\x67\x4b\xc1\x3f\xaf\x9\x81\xd4\x5d\xcb\x9\xdf\x36\x31\xc0\x14\x3c\x7c\xe\x65\x95\x99\x6d\xa3\xf4\xd7\x38\xee\x1a\x2b\x37\xe2\xa4\x3b\x4b\xd0\x65\xca\xf8\xc3\xe8\x15\x20\xef\xf2\x0\xfd\x1\x9\xc5\xc8\x17\x4\x93\xd0\x93\x3\x55\xc5\xfe\x32\xa3\x3e\x28\x2d\x3b\x93\x8a\xcc\x7\x72\x80\x8b\x74\x16\x24\xbb\xda\x94\x39\x30\x8f\xb1\xcd\x4a\x90\x92\x7c\x14\x8f\x95\x4e\xac\x9b\xd8\x8f\x1a\x87\xa4\x32\x27\x8a\xba\xf7\x41\xcf\x84\x37\x19\xe6\x6\xf5\xe\xcf\x36\xf5\x9e\x6c\xde\xbc\xff\x64\x7e\x4e\x59\x57\x48\xfe\x14\xf7\x9c\x93\x5d\x15\xad\xcc\x11\xb1\x17\x18\xb2\x7e\xcc\xab\xe9\xce\x7d\x77\x5b\x51\x1b\x1e\x20\xa8\x32\x6\xe\x75\x93\xac\xdb\x35\x37\x1f\xe9\x19\x1d\xb4\x71\x97\xd6\x4e\x2c\x8\xa5\x13\xf9\xe\x7e\x78\x6e\x14\xe0\xa9\xb9\x96\x4c\x80\x82\xba\x17\xb3\x9d\x69\xb0\x84\x46\xff\xf9\x52\x79\x94\x58\x3a\x62\x90\x15\x35\x71\x10\x37\xed\xa1\x8e\x53\x6e\xf4\x26\x57\x93\x15\x93\xf6\x81\x2c\x5a\x10\xda\x92\xad\x2f\xdb\x28\x31\x2d\x55\x4\xd2\x6\x28\x8c\x1e\xdc\xea\x54\xac\xff\xb7\x6c\x30\x15\xd4\xb4\xd\x0\x93\x57\xdd\xd2\x7\x7\x6\xd9\x43\x9b\xcd\x3a\xf4\x7d\x4c\x36\x5d\x23\xa2\xcc\x57\x40\x91\xe9\x2c\x2f\x2c\xd5\x30\x9b\x17\xb0\xc9\xf7\xa7\x2f\xd1\x93\x20\x6b\xc6\xc1\xe4\x6f\xcb\xd1\xe7\x9\xf\x9e\xdc\xaa\x9f\x2f\xdf\x56\x9f\xd4\x33\x4\xaf\xd3\x6c\x58\x61\xf0\x30\xec\xf2\x7f\xf2\x9c\xdf\x39\xbb\x6f\xa2\x8c\x7e\xc4\x22\x51\x71\xc0\x4d\x14\x1a\xc4\xcd\x4\xd9\x87\x8\x50\x5\xcc\xaf\xf6\xf0\x8f\x92\x54\x58\xc2\xc7\x9\x7a\x59\x2\x5\xe8\xb0\x86\xd9\xbf\x7b\x35\x51\x4d\xaf\x8\x97\x2c\x65\xda\x2a\x71\x3a\xa8\x51\xcc\xf2\x73\x27\xc3\xfd\x62\xcf\xe3\xb2\xca\xcb\xbe\x1a\xa\xa1\x34\x7b\x77\xc4\x62\x68\x78\x5f\x94\x7\x4\x65\x16\x4b\x61\xcb\xff\x75\x26\x50\x66\x1f\x6e\x93\xf8\xc5\x51\xeb\xa4\x4a\x48\x68\x6b\xe2\x5e\x44\xb2\x50\x2c\x6c\xae\x79\x4e\x66\x35\x81\x50\xac\xbc\x3f\xb1\xc\xf3\x5\x3c\x4a\xa3\x6c\x2a\x79\xb4\xb7\xab\xca\xc7\x9b\x8e\xcd\x5f\x11\x3\xcb\x30\xa3\xab\xda\xfe\x64\xb9\xbb\xd8\x5e\x3a\x1a\x56\xe5\x5\x48\x90\x1e\x61\x69\x1b\x22\xe6\x1a\x3c\x75\xad\x1f\x37\x28\xdc\xe4\x6d\xbd\x42\xdc\xd3\xc8\xb6\x1c\x48\xfe\x94\x77\x7f\xbd\x62\xac\xa3\x47\x27\xcf\x5f\xd9\xdb\xaf\xec\xf7\x5e\xc1\xb0\x9d\x1\x26\x99\x7e\x8f\x3\x70\xb5\x42\xbe\x67\x28\x1b\x7c\xbd\x61\x21\x97\xcc\x5c\xe1\x97\x8f\x8d\xde\x2b\xaa\xa7\x71\x1d\x1e\x2\x73\x70\x58\x32\x5b\x1d\x67\x3d\xe0\x74\x4f\x3\xf2\x70\x51\x79\xf1\x61\x70\x15\x74\x9d\x23\x89\xde\xac\xfd\xde\xd0\x1f\xc3\x87\x44\x35\x4b\xe5\xb0\x60\xc5\x22\xe4\x9e\xca\xeb\xd5\x3a\x9\x45\xa4\xdb\xfa\x3f\xeb\x1b\xc7\xc8\x14\x99\x51\x92\x10\xed\xed\x28\xe0\xa1\xf8\x26\xcf\xcd\xcb\x63\xa1\x3b\xe3\xdf\x7e\xfe\xa6\xf0\x81\x9a\xbf\x55\xde\x54\xd5\x56\x60\x98\x10\x68\xf4\x38\x96\x8e\x6f\x1d\x44\x7f\xd6\x2f\xfe\x55\xfb\xc\x7e\x67\xe2\x61\x44\xed\xf2\x35\x30\x5d\xe9\xc7\xd6\x6d\xe0\xa0\xed\xf3\xfc\xd8\x3e\xa\x7b\xcd\xaf\x65\x68\x18\xc0\xec\x4\x1c\x74\x6d\xe2\x6e\x79\xd4\x11\x2b\x62\xd5\x27\xad\x4f\x1\x59\x73\xcc\x6a\x53\xfb\x2d\xd5\x4e\x99\x21\x65\x4d\xf5\x82\xf7\xd8\x42\xce\x6f\x3d\x36\x47\xf1\x5\x16\xe8\x1b\x6a\x8f\x93\xf2\x8f\x37\x40\x12\x28\xa3\xe6\xb9\x17\x4a\x1f\xb1\xd1\x66\x69\x86\xc4\xfc\x97\xae\x3f\x8f\x1e\x2b\xdf\xcd\xf9\x3c"
}, {
	.key	= "\xb3\x35\x50\x3\x54\x2e\x40\x5e\x8f\x59\x8e\xc5\x90\xd5\x27\x2d\xba\x29\x2e\xcb\x1b\x70\x44\x1e\x65\x91\x6e\x2a\x79\x22\xda\x64",
	.nonce	= "\x5\xa3\x93\xed\x30\xc5\xa2\x6",
	.nlen	= 8,
	.assoc	= "\xb1\x69\x83\x87\x30\xaa\x5d\xb8\x77\xe8\x21\xff\x6\x59\x35\xce\x75\xfe\x38\xef\xb8\x91\x43\x8c\xcf\x70\xdd\xa\x68\xbf\xd4\xbc\x16\x76\x99\x36\x1e\x58\x79\x5e\xd4\x29\xf7\x33\x93\x48\xdb\x5f\x1\xae\x9c\xb6\xe4\x88\x6d\x2b\x76\x75\xe0\xf3\x74\xe2\xc9",
	.alen	= 63,
	.input	= "\x74\xa6\x3e\xe4\xb1\xcb\xaf\xb0\x40\xe5\xf\x9e\xf1\xf2\x89\xb5\x42\x34\x8a\xa1\x3\xb7\xe9\x57\x46\xbe\x20\xe4\x6e\xb0\xeb\xff\xea\x7\x7e\xef\xe2\x55\x9f\xe5\x78\x3a\xb7\x83\xc2\x18\x40\x7b\xeb\xcd\x81\xfb\x90\x12\x9e\x46\xa9\xd6\x4a\xba\xb0\x62\xdb\x6b\x99\xc4\xdb\x54\x4b\xb8\xa5\x71\xcb\xcd\x63\x32\x55\xfb\x31\xf0\x38\xf5\xbe\x78\xe4\x45\xce\x1b\x6a\x5b\xe\xf4\x16\xe4\xb1\x3d\xf6\x63\x7b\xa7\xc\xde\x6f\x8f\x74\xdf\xe0\x1e\x9d\xce\x8f\x24\xef\x23\x35\x33\x7b\x83\x34\x23\x58\x74\x14\x77\x1f\xc2\x4f\x4e\xc6\x89\xf9\x52\x9\x37\x64\x14\xc4\x1\x6b\x9d\x77\xe8\x90\x5d\xa8\x4a\x2a\xef\x5c\x7f\xeb\xbb\xb2\xc6\x93\x99\x66\xdc\x7f\xd4\x9e\x2a\xca\x8d\xdb\xe7\x20\xcf\xe4\x73\xae\x49\x7d\x64\xf\xe\x28\x46\xa9\xa8\x32\xe4\xe\xf6\x51\x53\xb8\x3c\xb1\xff\xa3\x33\x41\x75\xff\xf1\x6f\xf1\xfb\xbb\x83\x7f\x6\x9b\xe7\x1b\xa\xe0\x5c\x33\x60\x5b\xdb\x5b\xed\xfe\xa5\x16\x19\x72\xa3\x64\x23\x0\x2\xc7\xf3\x6a\x81\x3e\x44\x1d\x79\x15\x5f\x9a\xde\xe2\xfd\x1b\x73\xc1\xbc\x23\xba\x31\xd2\x50\xd5\xad\x7f\x74\xa7\xc9\xf8\x3e\x2b\x26\x10\xf6\x3\x36\x74\xe4\xe\x6a\x72\xb7\x73\xa\x42\x28\xc2\xad\x5e\x3\xbe\xb8\xb\xa8\x5b\xd4\xb8\xba\x52\x89\xb1\x9b\xc1\xc3\x65\x87\xed\xa5\xf4\x86\xfd\x41\x80\x91\x27\x59\x53\x67\x15\x78\x54\x8b\x2d\x3d\xc7\xff\x2\x92\x7\x5f\x7a\x4b\x60\x59\x3c\x6f\x5c\xd8\xec\x95\xd2\xfe\xa0\x3b\xd8\x3f\xd1\x69\xa6\xd6\x41\xb2\xf4\x4d\x12\xf4\x58\x3e\x66\x64\x80\x31\x9b\xa8\x4c\x8b\x7\xb2\xec\x66\x94\x66\x47\x50\x50\x5f\x18\xb\xe\xd6\xc0\x39\x21\x13\x9e\x33\xbc\x79\x36\x2\x96\x70\xf0\x48\x67\x2f\x26\xe9\x6d\x10\xbb\xd6\x3f\xd1\x64\x7a\x2e\xbe\xc\x61\xf0\x75\x42\x38\x23\xb1\x9e\x9f\x7c\x67\x66\xd9\x58\x9a\xf1\xbb\x41\x2a\x8d\x65\x84\x94\xfc\xdc\x6a\x50\x64\xdb\x56\x33\x76\x0\x10\xed\xbe\xd2\x12\xf6\xf6\x1b\xa2\x16\xde\xae\x31\x95\xdd\xb1\x8\x7e\x4e\xee\xe7\xf9\xa5\xfb\x5b\x61\x43\x0\x40\xf6\x7e\x2\x4\x32\x4e\xc\xe2\x66\xd\xd7\x7\x98\xe\xf8\x72\x34\x6d\x95\x86\xd7\xcb\x31\x54\x47\xd0\x38\x29\x9c\x5a\x68\xd4\x87\x76\xc9\xe7\x7e\xe3\xf4\x81\x6d\x18\xcb\xc9\x5\xaf\xa0\xfb\x66\xf7\xf1\x1c\xc6\x14\x11\x4f\x2b\x79\x42\x8b\xbc\xac\xe7\x6c\xfe\xf\x58\xe7\x7c\x78\x39\x30\xb0\x66\x2c\x9b\x6d\x3a\xe1\xcf\xc9\xa4\xe\x6d\x6d\x8a\xa1\x3a\xe7\x28\xd4\x78\x4c\xa6\xa2\x2a\xa6\x3\x30\xd7\xa8\x25\x66\x87\x2f\x69\x5c\x4e\xdd\xa5\x49\x5d\x37\x4a\x59\xc4\xaf\x1f\xa2\xe4\xf8\xa6\x12\x97\xd5\x79\xf5\xe2\x4a\x2b\x5f\x61\xe4\x9e\xe3\xee\xb8\xa7\x5b\x2f\xf4\x9e\x6c\xfb\xd1\xc6\x56\x77\xba\x75\xaa\x3d\x1a\xa8\xb\xb3\x68\x24\x0\x10\x7f\xfd\xd7\xa1\x8d\x83\x54\x4f\x1f\xd8\x2a\xbe\x8a\xc\x87\xab\xa2\xde\xc3\x39\xbf\x9\x3\xa5\xf3\x5\x28\xe1\xe1\xee\x39\x70\x9c\xd8\x81\x12\x1e\x2\x40\xd2\x6e\xf0\xeb\x1b\x3d\x22\xc6\xe5\xe3\xb4\x5a\x98\xbb\xf0\x22\x28\x8d\xe5\xd3\x16\x48\x24\xa5\xe6\x66\xc\xf9\x8\xf9\x7e\x1e\xe1\x28\x26\x22\xc7\xc7\xa\x32\x47\xfa\xa3\xbe\x3c\xc4\xc5\x53\xa\xd5\x94\x4a\xd7\x93\xd8\x42\x99\xb9\xa\xdb\x56\xf7\xb9\x1c\x53\x4f\xfa\xd3\x74\xad\xd9\x68\xf1\x1b\xdf\x61\xc6\x5e\xa8\x48\xfc\xd4\x4a\x4c\x3c\x32\xf7\x1c\x96\x21\x9b\xf9\xa3\xcc\x5a\xce\xd5\xd7\x8\x24\xf6\x1c\xfd\xdd\x38\xc2\x32\xe9\xb8\xe7\xb6\xfa\x9d\x45\x13\x2c\x83\xfd\x4a\x69\x82\xcd\xdc\xb3\x76\xc\x9e\xd8\xf4\x1b\x45\x15\xb4\x97\xe7\x58\x34\xe2\x3\x29\x5a\xbf\xb6\xe0\x5d\x13\xd9\x2b\xb4\x80\xb2\x45\x81\x6a\x2e\x6c\x89\x7d\xee\xbb\x52\xdd\x1f\x18\xe7\x13\x6b\x33\xe\xea\x36\x92\x77\x7b\x6d\x9c\x5a\x5f\x45\x7b\x7b\x35\x62\x23\xd1\xbf\xf\xd0\x8\x1b\x2b\x80\x6b\x7e\xf1\x21\x47\xb0\x57\xd1\x98\x72\x90\x34\x1c\x20\x4\xff\x3d\x5c\xee\xe\x57\x5f\x6f\x24\x4e\x3c\xea\xfc\xa5\xa9\x83\xc9\x61\xb4\x51\x24\xf8\x27\x5e\x46\x8c\xb1\x53\x2\x96\x35\xba\xb8\x4c\x71\xd3\x15\x59\x35\x22\x20\xad\x3\x9f\x66\x44\x3b\x9c\x35\x37\x1f\x9b\xbb\xf3\xdb\x35\x63\x30\x64\xaa\xa2\x6\xa8\x5d\xbb\xe1\x9f\x70\xec\x82\x11\x6\x36\xec\x8b\x69\x66\x24\x44\xc9\x4a\x57\xbb\x9b\x78\x13\xce\x9c\xc\xba\x92\x93\x63\xb8\xe2\x95\xf\xf\x16\x39\x52\xfd\x3a\x6d\x2\x4b\xdf\x13\xd3\x2a\x22\xb4\x3\x7c\x54\x49\x96\x68\x54\x10\xfa\xef\xaa\x6c\xe8\x22\xdc\x71\x16\x13\x1a\xf6\x28\xe5\x6d\x77\x3d\xcd\x30\x63\xb1\x70\x52\xa1\xc5\x94\x5f\xcf\xe8\xb8\x26\x98\xf7\x6\xa0\xa\x70\xfa\x3\x80\xac\xc1\xec\xd6\x4c\x54\xd7\xfe\x47\xb6\x88\x4a\xf7\x71\x24\xee\xf3\xd2\xc2\x4a\x7f\xfe\x61\xc7\x35\xc9\x37\x67\xcb\x24\x35\xda\x7e\xca\x5f\xf3\x8d\xd4\x13\x8e\xd6\xcb\x4d\x53\x8f\x53\x1f\xc0\x74\xf7\x53\xb9\x5e\x23\x37\xba\x6e\xe3\x9d\x7\x55\x25\x7b\xe6\x2a\x64\xd1\x32\xdd\x54\x1b\x4b\xc0\xe1\xd7\x69\x58\xf8\x93\x29\xc4\xdd\x23\x2f\xa5\xfc\x9d\x7e\xf8\xd4\x90\xcd\x82\x55\xdc\x16\x16\x9f\x7\x52\x9b\x9d\x25\xed\x32\xc5\x7b\xdf\xf6\x83\x46\x3d\x65\xb7\xef\x87\x7a\x12\x69\x8f\x6\x7c\x51\x15\x4a\x8\xe8\xac\x9a\xc\x24\xa7\x27\xd8\x46\x2f\xe7\x1\xe\x1c\xc6\x91\xb0\x6e\x85\x65\xf0\x29\xd\x2e\x6b\x3b\xfb\x4b\xdf\xe4\x80\x93\x3\x66\x46\x3e\x8a\x6e\xf3\x5e\x4d\x62\xe\x49\x5\xaf\xd4\xf8\x21\x20\x61\x1d\x39\x17\xf4\x61\x47\x95\xfb\x15\x2e\xb3\x4f\xd0\x5d\xf5\x7d\x40\xda\x90\x3c\x6b\xcb\x17\x0\x13\x3b\x64\x34\x1b\xf0\xf2\xe5\x3b\xb2\xc7\xd3\x5f\x3a\x44\xa6\x9b\xb7\x78\xe\x42\x5d\x4c\xc1\xe9\xd2\xcb\xb7\x78\xd1\xfe\x9a\xb5\x7\xe9\xe0\xbe\xe2\x8a\xa7\x1\x83\x0\x8c\x5c\x8\xe6\x63\x12\x92\xb7\xb7\xa6\x19\x7d\x38\x13\x38\x92\x87\x24\xf9\x48\xb3\x5e\x87\x6a\x40\x39\x5c\x3f\xed\x8f\xee\xdb\x15\x82\x6\xda\x49\x21\x2b\xb5\xbf\x32\x7c\x9f\x42\x28\x63\xcf\xaf\x1e\xf8\xc6\xa0\xd1\x2\x43\x57\x62\xec\x9b\xf\x1\x9e\x71\xd8\x87\x9d\x1\xc1\x58\x77\xd9\xaf\xb1\x10\x7e\xdd\xa6\x50\x96\xe5\xf0\x72\x0\x6d\x4b\xf8\x2a\x8f\x19\xf3\x22\x88\x11\x4a\x8b\x7c\xfd\xb7\xed\xe1\xf6\x40\x39\xe0\xe9\xf6\x3d\x25\xe6\x74\x3c\x58\x57\x7f\xe1\x22\x96\x47\x31\x91\xba\x70\x85\x28\x6b\x9f\x6e\x25\xac\x23\x66\x2f\x29\x88\x28\xce\x8c\x5c\x88\x53\xd1\x3b\xcc\x6a\x51\xb2\xe1\x28\x3f\x91\xb4\xd\x0\x3a\xe3\xf8\xc3\x8f\xd7\x96\x62\xe\x2e\xfc\xc8\x6c\x77\xa6\x1d\x22\xc1\xb8\xe6\x61\xd7\x67\x36\x13\x7b\xbb\x9b\x59\x9\xa6\xdf\xf7\x6b\xa3\x40\x1a\xf5\x4f\xb4\xda\xd3\xf3\x81\x93\xc6\x18\xd9\x26\xee\xac\xf0\xaa\xdf\xc5\x9c\xca\xc2\xa2\xcc\x7b\x5c\x24\xb0\xbc\xd0\x6a\x4d\x89\x9\xb8\x7\xfe\x87\xad\xa\xea\xb8\x42\xf9\x5e\xb3\x3e\x36\x4c\xaf\x75\x9e\x1c\xeb\xbd\xbc\xbb\x80\x40\xa7\x3a\x30\xbf\xa8\x44\xf4\xeb\x38\xad\x29\xba\x23\xed\x41\xc\xea\xd2\xbb\x41\x18\xd6\xb9\xba\x65\x2b\xa3\x91\x6d\x1f\xa9\xf4\xd1\x25\x8d\x4d\x38\xff\x64\xa0\xec\xde\xa6\xb6\x79\xab\x8e\x33\x6c\x47\xde\xaf\x94\xa4\xa5\x86\x77\x55\x9\x92\x81\x31\x76\xc7\x34\x22\x89\x8e\x3d\x26\x26\xd7\xfc\x1e\x16\x72\x13\x33\x63\xd5\x22\xbe\xb8\x4\x34\x84\x41\xbb\x80\xd0\x9f\x46\x48\x7\xa7\xfc\x2b\x3a\x75\x55\x8c\xc7\x6a\xbd\x7e\x46\x8\x84\xf\xd5\x74\xc0\x82\x8e\xaa\x61\x5\x1\xb2\x47\x6e\x20\x6a\x2d\x58\x70\x48\x32\xa7\x37\xd2\xb8\x82\x1a\x51\xb9\x61\xdd\xfd\x9d\x6b\xe\x18\x97\xf8\x45\x5f\x87\x10\xcf\x34\x72\x45\x26\x49\x70\xe7\xa3\x78\xe0\x52\x89\x84\x94\x83\x82\xc2\x69\x8f\xe3\xe1\x3f\x60\x74\x88\xc4\xf7\x75\x2c\xfb\xbd\xb6\xc4\x7e\x10\xa\x6c\x90\x4\x9e\xc3\x3f\x59\x7c\xce\x31\x18\x60\x57\x73\x46\x94\x7d\x6\xa0\x6d\x44\xec\xa2\xa\x9e\x5\x15\xef\xca\x5c\xbf\x0\xeb\xf7\x3d\x32\xd4\xa5\xef\x49\x89\x5e\x46\xb0\xa6\x63\x5b\x8a\x73\xae\x6f\xd5\x9d\xf8\x4f\x40\xb5\xb2\x6e\xd3\xb6\x1\xa9\x26\xa2\x21\xcf\x33\x7a\x3a\xa4\x23\x13\xb0\x69\x6a\xee\xce\xd8\x9d\x1\x1d\x50\xc1\x30\x6c\xb1\xcd\xa0\xf0\xf0\xa2\x64\x6f\xbb\xbf\x5e\xe6\xab\x87\xb4\xf\x4f\x15\xaf\xb5\x25\xa1\xb2\xd0\x80\x2c\xfb\xf9\xfe\xd2\x33\xbb\x76\xfe\x7c\xa8\x66\xf7\xe7\x85\x9f\x1f\x85\x57\x88\xe1\xe9\x63\xe4\xd8\x1c\xa1\xfb\xda\x44\x5\x2e\x1d\x3a\x1c\xff\xc8\x3b\xc0\xfe\xda\x22\xb\x43\xd6\x88\x39\x4c\x4a\xa6\x69\x18\x93\x42\x4e\xb5\xcc\x66\xd\x9\xf8\x1e\x7c\xd3\x3c\x99\xd\x50\x1d\x62\xe9\x57\x6\xbf\x19\x88\xdd\xad\x7b\x4f\xf9\xc7\x82\x6d\x8d\xc8\xc4\xc5\x78\x17\x20\x15\xc5\x52\x41\xcf\x5b\xd6\x7f\x94\x2\x41\xe0\x40\x22\x3\x5e\xd1\x53\xd4\x86\xd3\x2c\x9f\xf\x96\xe3\x6b\x9a\x76\x32\x6\x47\x4b\x11\xb3\xdd\x3\x65\xbd\x9b\x1\xda\x9c\xb9\x7e\x3f\x6a\xc4\x7b\xea\xd4\x3c\xb9\xfb\x5c\x6b\x64\x33\x52\xba\x64\x78\x8f\xa4\xaf\x7a\x61\x8d\xbc\xc5\x73\xe9\x6b\x58\x97\x4b\xbf\x63\x22\xd3\x37\x2\x54\xc5\xb9\x16\x4a\xf0\x19\xd8\x94\x57\xb8\x8a\xb3\x16\x3b\xd0\x84\x8e\x67\xa6\xa3\x7d\x78\xec\x0",
	.ilen	= 2011,
	.result	= "\x52\x34\xb3\x65\x3b\xb7\xe5\xd3\xab\x49\x17\x60\xd2\x52\x56\xdf\xdf\x34\x56\x82\xe2\xbe\xe5\xe1\x28\xd1\x4e\x5f\x4f\x1\x7d\x3f\x99\x6b\x30\x6e\x1a\x7c\x4c\x8e\x62\x81\xae\x86\x3f\x6b\xd0\xb5\xa9\xcf\x50\xf1\x2\x12\xa0\xb\x24\xe9\xe6\x72\x89\x2c\x52\x1b\x34\x38\xf8\x75\x5f\xa0\x74\xe2\x99\xdd\xa6\x4b\x14\x50\x4e\xf1\xbe\xd6\x9e\xdb\xb2\x24\x27\x74\x12\x4a\x78\x78\x17\xa5\x58\x8e\x2f\xf9\xf4\x8d\xee\x3\x88\xae\xb8\x29\xa1\x2f\x4b\xee\x92\xbd\x87\xb3\xce\x34\x21\x57\x46\x4\x49\xc\x80\xf2\x1\x13\xa1\x55\xb3\xff\x44\x30\x3c\x1c\xd0\xef\xbc\x18\x74\x26\xad\x41\x5b\x5b\x3e\x9a\x7a\x46\x4f\x16\xd6\x74\x5a\xb7\x3a\x28\x31\xd8\xae\x26\xac\x50\x53\x86\xf2\x56\xd7\x3f\x29\xbc\x45\x68\x8e\xcb\x98\x64\xdd\xc9\xba\xb8\x4b\x7b\x82\xdd\x14\xa7\xcb\x71\x72\x0\x5c\xad\x7b\x6a\x89\xa4\x3d\xbf\xb5\x4b\x3e\x7c\x5a\xcf\xb8\xa1\xc5\x6e\xc8\xb6\x31\x57\x7b\xdf\xa5\x7e\xb1\xd6\x42\x2a\x31\x36\xd1\xd0\x3f\x7a\xe5\x94\xd6\x36\xa0\x6f\xb7\x40\x7d\x37\xc6\x55\x7c\x50\x40\x6d\x29\x89\xe3\x5a\xae\x97\xe7\x44\x49\x6e\xbd\x81\x3d\x3\x93\x6\x12\x6\xe2\x41\x12\x4a\xf1\x6a\xa4\x58\xa2\xfb\xd2\x15\xba\xc9\x79\xc9\xce\x5e\x13\xbb\xf1\x9\x4\xcc\xfd\xe8\x51\x34\x6a\xe8\x61\x88\xda\xed\x1\x47\x84\xf5\x73\x25\xf9\x1c\x42\x86\x7\xf3\x5b\x1a\x1\xb3\xeb\x24\x32\x8d\xf6\xed\x7c\x4b\xeb\x3c\x36\x42\x28\xdf\xdf\xb6\xbe\xd9\x8c\x52\xd3\x2b\x8\x90\x8c\xe7\x98\x31\xe2\x32\x8e\xfc\x11\x48\x0\xa8\x6a\x42\x4a\x2\xc6\x4b\x9\xf1\xe3\x49\xf3\x45\x1f\xe\xbc\x56\xe2\xe4\xdf\xfb\xeb\x61\xfa\x24\xc1\x63\x75\xbb\x47\x75\xaf\xe1\x53\x16\x96\x21\x85\x26\x11\xb3\x76\xe3\x23\xa1\x6b\x74\x37\xd0\xde\x6\x90\x71\x5d\x43\x88\x9b\x0\x54\xa6\x75\x2f\xa1\xc2\xb\x73\x20\x1d\xb6\x21\x79\x57\x3f\xfa\x9\xbe\x8a\x33\xc3\x52\xf0\x1d\x82\x31\xd1\x55\xb5\x6c\x99\x25\xcf\x5c\x32\xce\xe9\xd\xfa\x69\x2c\xd5\xd\xc5\x6d\x86\xd0\xc\x3b\x6\x50\x79\xe8\xc3\xae\x4\xe6\xcd\x51\xe4\x26\x9b\x4f\x7e\xa6\xf\xab\xd8\xe5\xde\xa9\x0\x95\xbe\xa3\x9d\x5d\xb2\x9\x70\x18\x1c\xf0\xac\x29\x23\x2\x29\x28\xd2\x74\x35\x57\x62\xf\x24\xea\x5e\x33\xc2\x92\xf3\x78\x4d\x30\x1e\xa1\x99\xa9\x82\xb0\x42\x31\x8d\xad\x8a\xbc\xfc\xd4\x57\x47\x3e\xb4\x50\xdd\x6e\x2c\x80\x4d\x22\xf1\xfb\x57\xc4\xdd\x17\xe1\x8a\x36\x4a\xb3\x37\xca\xc9\x4e\xab\xd5\x69\xc4\xf4\xbc\xb\x3b\x44\x4b\x29\x9c\xee\xd4\x35\x22\x21\xb0\x1f\x27\x64\xa8\x51\x1b\xf0\x9f\x19\x5c\xfb\x5a\x64\x74\x70\x45\x9\xf5\x64\xfe\x1a\x2d\xc9\x14\x4\x14\xcf\xd5\x7d\x60\xaf\x94\x39\x94\xe2\x7d\x79\x82\xd0\x65\x3b\x6b\x9c\x19\x84\xb4\x6d\xb3\xc\x99\xc0\x56\xa8\xbd\x73\xce\x5\x84\x3e\x30\xaa\xc4\x9b\x1b\x4\x2a\x9f\xd7\x43\x2b\x23\xdf\xbf\xaa\xd5\xc2\x43\x2d\x70\xab\xdc\x75\xad\xac\xf7\xc0\xbe\x67\xb2\x74\xed\x67\x10\x4a\x92\x60\xc1\x40\x50\x19\x8a\x8a\x8c\x9\xe\x72\xe1\x73\x5e\xe8\x41\x85\x63\x9f\x3f\xd7\x7d\xc4\xfb\x22\x5d\x92\x6c\xb3\x1e\xe2\x50\x2f\x82\xa8\x28\xc0\xb5\xd7\x5f\x68\xd\x2c\x2d\xaf\x7e\xfa\x2e\x8\xf\x1f\x70\x9f\xe9\x19\x72\x55\xf8\xfb\x51\xd2\x33\x5d\xa0\xd3\x2b\xa\x6c\xbc\x4e\xcf\x36\x4d\xdc\x3b\xe9\x3e\x81\x7c\x61\xdb\x20\x2d\x3a\xc3\xb3\xc\x1e\x0\xb9\x7c\xf5\xca\x10\x5f\x3a\x71\xb3\xe4\x20\xdb\xc\x2a\x98\x63\x45\x0\x58\xf6\x68\xe4\xb\xda\x13\x3b\x60\x5c\x76\xdb\xb9\x97\x71\xe4\xd9\xb7\xdb\xbd\x68\xc7\x84\x84\xaa\x7c\x68\x62\x5e\x16\xfc\xba\x72\xaa\x9a\xa9\xeb\x7c\x75\x47\x97\x7e\xad\xe2\xd9\x91\xe8\xe4\xa5\x31\xd7\x1\x8e\xa2\x11\x88\x95\xb9\xf2\x9b\xd3\x7f\x1b\x81\x22\xf7\x98\x60\xa\x64\xa6\xc1\xf6\x49\xc7\xe3\x7\x4d\x94\x7a\xcf\x6e\x68\xc\x1b\x3f\x6e\x2e\xee\x92\xfa\x52\xb3\x59\xf8\xf1\x8f\x6a\x66\xa3\x82\x76\x4a\x7\x1a\xc7\xdd\xf5\xda\x9c\x3c\x24\xbf\xfd\x42\xa1\x10\x64\x6a\xf\x89\xee\x36\xa5\xce\x99\x48\x6a\xf0\x9f\x9e\x69\xa4\x40\x20\xe9\x16\x15\xf7\xdb\x75\x2\xcb\xe9\x73\x8b\x3b\x49\x2f\xf0\xaf\x51\x6\x5c\xdf\x27\x27\x49\x6a\xd1\xcc\xc7\xb5\x63\xb5\xfc\xb8\x5c\x87\x7f\x84\xb4\xcc\x14\xa9\x53\xda\xa4\x56\xf8\xb6\x1b\xcc\x40\x27\x52\x6\x5a\x13\x81\xd7\x3a\xd4\x3b\xfb\x49\x65\x31\x33\xb2\xfa\xcd\xad\x58\x4e\x2b\xae\xd2\x20\xfb\x1a\x48\xb4\x3f\x9a\xd8\x7a\x35\x4a\xc8\xee\x88\x5e\x7\x66\x54\xb9\xec\x9f\xa3\xe3\xb9\x37\xaa\x49\x76\x31\xda\x74\x2d\x3c\xa4\x65\x10\x32\x38\xf0\xde\xd3\x99\x17\xaa\x71\xaa\x8f\xf\x8c\xaf\xa2\xf8\x5d\x64\xba\x1d\xa3\xef\x96\x73\xe8\xa1\x2\x8d\xc\x6d\xb8\x6\x90\xb8\x8\x56\x2c\xa7\x6\xc9\xc2\x38\xdb\x7c\x63\xb1\x57\x8e\xea\x7c\x79\xf3\x49\x1d\xfe\x9f\xf3\x6e\xb1\x1d\xba\x19\x80\x1a\xa\xd3\xb0\x26\x21\x40\xb1\x7c\xf9\x4d\x8d\x10\xc1\x7e\xf4\xf6\x3c\xa8\xfd\x7c\xa3\x92\xb2\xf\xaa\xcc\xa6\x11\xfe\x4\xe3\xd1\x7a\x32\x89\xdf\xd\xc4\x8f\x79\x6b\xca\x16\x7c\x6e\xf9\xad\xf\xf6\xfe\x27\xdb\xc4\x13\x70\xf1\x62\x1a\x4f\x79\x40\xc9\x9b\x8b\x21\xea\x84\xfa\xf5\xf1\x89\xce\xb7\x55\xa\x80\x39\x2f\x55\x36\x16\x9c\x7b\x8\xbd\x87\xd\xa5\x32\xf1\x52\x7c\xe8\x55\x60\x5b\xd7\x69\xe4\xfc\xfa\x12\x85\x96\xea\x50\x28\xab\x8a\xf7\xbb\xe\x53\x74\xca\xa6\x27\x9\xc2\xb5\xde\x18\x14\xd9\xea\xe5\x29\x1c\x40\x56\xcf\xd7\xae\x5\x3f\x65\xaf\x5\x73\xe2\x35\x96\x27\x7\x14\xc0\xad\x33\xf1\xdc\x44\x7a\x89\x17\x77\xd2\x9c\x58\x60\xf0\x3f\x7b\x2d\x2e\x57\x95\x54\x87\xed\xf2\xc7\x4c\xf0\xae\x56\x29\x19\x7d\x66\x4b\x9b\x83\x84\x42\x3b\x1\x25\x66\x8e\x2\xde\xb9\x83\x54\x19\xf6\x9f\x79\xd\x67\xc5\x1d\x7a\x44\x2\x98\xa7\x16\x1c\x29\xd\x74\xff\x85\x40\x6\xef\x2c\xa9\xc6\xf5\x53\x7\x6\xae\xe4\xfa\x5f\xd8\x39\x4d\xf1\x9b\x6b\xd9\x24\x84\xfe\x3\x4c\xb2\x3f\xdf\xa1\x5\x9e\x50\x14\x5a\xd9\x1a\xa2\xa7\xfa\xfa\x17\xf7\x78\xd6\xb5\x92\x61\x91\xac\x36\xfa\x56\xd\x38\x32\x18\x85\x8\x58\x37\xf0\x4b\xdb\x59\xe7\xa4\x34\xc0\x1b\x1\xaf\x2d\xde\xa1\xaa\x5d\xd3\xec\xe1\xd4\xf7\xe6\x54\x68\xf0\x51\x97\xa7\x89\xea\x24\xad\xd3\x6e\x47\x93\x8b\x4b\xb4\xf7\x1c\x42\x6\x67\xe8\x99\xf6\xf5\x7b\x85\xb5\x65\xb5\xb5\xd2\x37\xf5\xf3\x2\xa6\x4d\x11\xa7\xdc\x51\x9\x7f\xa0\xd8\x88\x1c\x13\x71\xae\x9c\xb7\x7b\x34\xd6\x4e\x68\x26\x83\x51\xaf\x1d\xee\x8b\xbb\x69\x43\x2b\x9e\x8a\xbc\x2\xe\xa0\x1b\xe0\xa8\x5f\x6f\xaf\x1b\x8f\xe7\x64\x71\x74\x11\x7e\xa8\xd8\xf9\x97\x6\xc3\xb6\xfb\xfb\xb7\x3d\x35\x9d\x3b\x52\xed\x54\xca\xf4\x81\x1\x2d\x1b\xc3\xa7\x0\x3d\x1a\x39\x54\xe1\xf6\xff\xed\x6f\xb\x5a\x68\xda\x58\xdd\xa9\xcf\x5c\x4a\xe5\x9\x4e\xde\x9d\xbc\x3e\xee\x5a\x0\x3b\x2c\x87\x10\x65\x60\xdd\xd7\x56\xd1\x4c\x64\x45\xe4\x21\xec\x78\xf8\x25\x7a\x3e\x16\x5d\x9\x53\x14\xbe\x4f\xae\x87\xd8\xd1\xaa\x3c\xf6\x3e\xa4\x70\x8c\x5e\x70\xa4\xb3\x6b\x66\x73\xd3\xbf\x31\x6\x19\x62\x93\x15\xf2\x86\xe4\x52\x7e\x53\x4c\x12\x38\xcc\x34\x7d\x57\xf6\x42\x93\x8a\xc4\xee\x5c\x8a\xe1\x52\x8f\x56\x64\xf6\xa6\xd1\x91\x57\x70\xcd\x11\x76\xf5\x59\x60\x60\x3c\xc1\xc3\xb\x7f\x58\x1a\x50\x91\xf1\x68\x8f\x6e\x74\x74\xa8\x51\xb\xf7\x7a\x98\x37\xf2\xa\xe\xa4\x97\x4\xb8\x9b\xfd\xa0\xea\xf7\xd\xe1\xdb\x3\xf0\x31\x29\xf8\xdd\x6b\x8b\x5d\xd8\x59\xa9\x29\xcf\x9a\x79\x89\x19\x63\x46\x9\x79\x6a\x11\xda\x63\x68\x48\x77\x23\xfb\x7d\x3a\x43\xcb\x2\x3b\x7a\x6d\x10\x2a\x9e\xac\xf1\xd4\x19\xf8\x23\x64\x1d\x2c\x5f\xf2\xb0\x5c\x23\x27\xf7\x27\x30\x16\x37\xb1\x90\xab\x38\xfb\x55\xcd\x78\x58\xd4\x7d\x43\xf6\x45\x5e\x55\x8d\xb1\x2\x65\x58\xb4\x13\x4b\x36\xf7\xcc\xfe\x3d\xb\x82\xe2\x12\x11\xbb\xe6\xb8\x3a\x48\x71\xc7\x50\x6\x16\x3a\xe6\x7c\x5\xc7\xc8\x4d\x2f\x8\x6a\x17\x9a\x95\x97\x50\x68\xdc\x28\x18\xc4\x61\x38\xb9\xe0\x3e\x78\xdb\x29\xe0\x9f\x52\xdd\xf8\x4f\x91\xc1\xd0\x33\xa1\x7a\x8e\x30\x13\x82\x7\x9f\xd3\x31\xf\x23\xbe\x32\x5a\x75\xcf\x96\xb2\xec\xb5\x32\xac\x21\xd1\x82\x33\xd3\x15\x74\xbd\x90\xf1\x2c\xe6\x5f\x8d\xe3\x2\xe8\xe9\xc4\xca\x96\xeb\xe\xbc\x91\xf4\xb9\xea\xd9\x1b\x75\xbd\xe1\xac\x2a\x5\x37\x52\x9b\x1b\x3f\x5a\xdc\x21\xc3\x98\xbb\xaf\xa3\xf2\x0\xbf\xd\x30\x89\x5\xcc\xa5\x76\xf5\x6\xf0\xc6\x54\x8a\x5d\xd4\x1e\xc1\xf2\xce\xb0\x62\xc8\xfc\x59\x42\x9a\x90\x60\x55\xfe\x88\xa5\x8b\xb8\x33\xc\x23\x24\xd\x15\x70\x37\x1e\x3d\xf6\xd2\xea\x92\x10\xb2\xc4\x51\xac\xf2\xac\xf3\x6b\x6c\xaa\xcf\x12\xc5\x6c\x90\x50\xb5\xc\xfc\x1a\x15\x52\xe9\x26\xc6\x52\xa4\xe7\x81\x69\xe1\xe7\x9e\x30\x1\xec\x84\x89\xb2\xd\x66\xdd\xce\x28\x5c\xec\x98\x46\x68\x21\x9f\x88\x3f\x1f\x42\x77\xce\xd0\x61\xd4\x20\xa7\xff\x53\xad\x37\xd0\x17\x35\xc9\xfc\xba\xa\x78\x3f\xf2\xcc\x86\x89\xe8\x4b\x3c\x48\x33\x9\x7f\xc6\xc0\xdd\xb8\xfd\x7a\x66\x66\x65\xeb\x47\xa7\x4\x28\xa3\x19\x8e\xa9\xb1\x13\x67\x62\x70\xcf\xd6"
}, { /* wycheproof - rfc7539 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x07\x00\x00\x00\x40\x41\x42\x43\x44\x45\x46\x47",
	.nlen	= 12,
	.assoc	= "\x50\x51\x52\x53\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7",
	.alen	= 12,
	.input	= "\x4c\x61\x64\x69\x65\x73\x20\x61\x6e\x64\x20\x47\x65\x6e\x74\x6c\x65\x6d\x65\x6e\x20\x6f\x66\x20\x74\x68\x65\x20\x63\x6c\x61\x73\x73\x20\x6f\x66\x20\x27\x39\x39\x3a\x20\x49\x66\x20\x49\x20\x63\x6f\x75\x6c\x64\x20\x6f\x66\x66\x65\x72\x20\x79\x6f\x75\x20\x6f\x6e\x6c\x79\x20\x6f\x6e\x65\x20\x74\x69\x70\x20\x66\x6f\x72\x20\x74\x68\x65\x20\x66\x75\x74\x75\x72\x65\x2c\x20\x73\x75\x6e\x73\x63\x72\x65\x65\x6e\x20\x77\x6f\x75\x6c\x64\x20\x62\x65\x20\x69\x74\x2e",
	.ilen	= 114,
	.result	= "\xd3\x1a\x8d\x34\x64\x8e\x60\xdb\x7b\x86\xaf\xbc\x53\xef\x7e\xc2\xa4\xad\xed\x51\x29\x6e\x08\xfe\xa9\xe2\xb5\xa7\x36\xee\x62\xd6\x3d\xbe\xa4\x5e\x8c\xa9\x67\x12\x82\xfa\xfb\x69\xda\x92\x72\x8b\x1a\x71\xde\x0a\x9e\x06\x0b\x29\x05\xd6\xa5\xb6\x7e\xcd\x3b\x36\x92\xdd\xbd\x7f\x2d\x77\x8b\x8c\x98\x03\xae\xe3\x28\x09\x1b\x58\xfa\xb3\x24\xe4\xfa\xd6\x75\x94\x55\x85\x80\x8b\x48\x31\xd7\xbc\x3f\xf4\xde\xf0\x8e\x4b\x7a\x9d\xe5\x76\xd2\x65\x86\xce\xc6\x4b\x61\x16\x1a\xe1\x0b\x59\x4f\x09\xe2\x6a\x7e\x90\x2e\xcb\xd0\x60\x06\x91"
}, { /* wycheproof - misc */
	.key	= "\x80\xba\x31\x92\xc8\x03\xce\x96\x5e\xa3\x71\xd5\xff\x07\x3c\xf0\xf4\x3b\x6a\x2a\xb5\x76\xb2\x08\x42\x6e\x11\x40\x9c\x09\xb9\xb0",
	.nonce	= "\x4d\xa5\xbf\x8d\xfd\x58\x52\xc1\xea\x12\x37\x9d",
	.nlen	= 12,
	.assoc	= "",
	.alen	= 0,
	.input	= "",
	.ilen	= 0,
	.result	= "\x76\xac\xb3\x42\xcf\x31\x66\xa5\xb6\x3c\x0c\x0e\xa1\x38\x3c\x8d"
}, { /* wycheproof - misc */
	.key	= "\x7a\x4c\xd7\x59\x17\x2e\x02\xeb\x20\x4d\xb2\xc3\xf5\xc7\x46\x22\x7d\xf5\x84\xfc\x13\x45\x19\x63\x91\xdb\xb9\x57\x7a\x25\x07\x42",
	.nonce	= "\xa9\x2e\xf0\xac\x99\x1d\xd5\x16\xa3\xc6\xf6\x89",
	.nlen	= 12,
	.assoc	= "\xbd\x50\x67\x64\xf2\xd2\xc4\x10",
	.alen	= 8,
	.input	= "",
	.ilen	= 0,
	.result	= "\x90\x6f\xa6\x28\x4b\x52\xf8\x7b\x73\x59\xcb\xaa\x75\x63\xc7\x09"
}, { /* wycheproof - misc */
	.key	= "\xcc\x56\xb6\x80\x55\x2e\xb7\x50\x08\xf5\x48\x4b\x4c\xb8\x03\xfa\x50\x63\xeb\xd6\xea\xb9\x1f\x6a\xb6\xae\xf4\x91\x6a\x76\x62\x73",
	.nonce	= "\x99\xe2\x3e\xc4\x89\x85\xbc\xcd\xee\xab\x60\xf1",
	.nlen	= 12,
	.assoc	= "",
	.alen	= 0,
	.input	= "\x2a",
	.ilen	= 1,
	.result	= "\x3a\xca\xc2\x7d\xec\x09\x68\x80\x1e\x9f\x6e\xde\xd6\x9d\x80\x75\x22"
}, { /* wycheproof - misc */
	.key	= "\x46\xf0\x25\x49\x65\xf7\x69\xd5\x2b\xdb\x4a\x70\xb4\x43\x19\x9f\x8e\xf2\x07\x52\x0d\x12\x20\xc5\x5e\x4b\x70\xf0\xfd\xa6\x20\xee",
	.nonce	= "\xab\x0d\xca\x71\x6e\xe0\x51\xd2\x78\x2f\x44\x03",
	.nlen	= 12,
	.assoc	= "\x91\xca\x6c\x59\x2c\xbc\xca\x53",
	.alen	= 8,
	.input	= "\x51",
	.ilen	= 1,
	.result	= "\xc4\x16\x83\x10\xca\x45\xb1\xf7\xc6\x6c\xad\x4e\x99\xe4\x3f\x72\xb9"
}, { /* wycheproof - misc */
	.key	= "\x2f\x7f\x7e\x4f\x59\x2b\xb3\x89\x19\x49\x89\x74\x35\x07\xbf\x3e\xe9\xcb\xde\x17\x86\xb6\x69\x5f\xe6\xc0\x25\xfd\x9b\xa4\xc1\x00",
	.nonce	= "\x46\x1a\xf1\x22\xe9\xf2\xe0\x34\x7e\x03\xf2\xdb",
	.nlen	= 12,
	.assoc	= "",
	.alen	= 0,
	.input	= "\x5c\x60",
	.ilen	= 2,
	.result	= "\x4d\x13\x91\xe8\xb6\x1e\xfb\x39\xc1\x22\x19\x54\x53\x07\x7b\x22\xe5\xe2"
}, { /* wycheproof - misc */
	.key	= "\xc8\x83\x3d\xce\x5e\xa9\xf2\x48\xaa\x20\x30\xea\xcf\xe7\x2b\xff\xe6\x9a\x62\x0c\xaf\x79\x33\x44\xe5\x71\x8f\xe0\xd7\xab\x1a\x58",
	.nonce	= "\x61\x54\x6b\xa5\xf1\x72\x05\x90\xb6\x04\x0a\xc6",
	.nlen	= 12,
	.assoc	= "\x88\x36\x4f\xc8\x06\x05\x18\xbf",
	.alen	= 8,
	.input	= "\xdd\xf2",
	.ilen	= 2,
	.result	= "\xb6\x0d\xea\xd0\xfd\x46\x97\xec\x2e\x55\x58\x23\x77\x19\xd0\x24\x37\xa2"
}, { /* wycheproof - misc */
	.key	= "\x55\x56\x81\x58\xd3\xa6\x48\x3f\x1f\x70\x21\xea\xb6\x9b\x70\x3f\x61\x42\x51\xca\xdc\x1a\xf5\xd3\x4a\x37\x4f\xdb\xfc\x5a\xda\xc7",
	.nonce	= "\x3c\x4e\x65\x4d\x66\x3f\xa4\x59\x6d\xc5\x5b\xb7",
	.nlen	= 12,
	.assoc	= "",
	.alen	= 0,
	.input	= "\xab\x85\xe9\xc1\x57\x17\x31",
	.ilen	= 7,
	.result	= "\x5d\xfe\x34\x40\xdb\xb3\xc3\xed\x7a\x43\x4e\x26\x02\xd3\x94\x28\x1e\x0a\xfa\x9f\xb7\xaa\x42"
}, { /* wycheproof - misc */
	.key	= "\xe3\xc0\x9e\x7f\xab\x1a\xef\xb5\x16\xda\x6a\x33\x02\x2a\x1d\xd4\xeb\x27\x2c\x80\xd5\x40\xc5\xda\x52\xa7\x30\xf3\x4d\x84\x0d\x7f",
	.nonce	= "\x58\x38\x93\x75\xc6\x9e\xe3\x98\xde\x94\x83\x96",
	.nlen	= 12,
	.assoc	= "\x84\xe4\x6b\xe8\xc0\x91\x90\x53",
	.alen	= 8,
	.input	= "\x4e\xe5\xcd\xa2\x0d\x42\x90",
	.ilen	= 7,
	.result	= "\x4b\xd4\x72\x12\x94\x1c\xe3\x18\x5f\x14\x08\xee\x7f\xbf\x18\xf5\xab\xad\x6e\x22\x53\xa1\xba"
}, { /* wycheproof - misc */
	.key	= "\x51\xe4\xbf\x2b\xad\x92\xb7\xaf\xf1\xa4\xbc\x05\x55\x0b\xa8\x1d\xf4\xb9\x6f\xab\xf4\x1c\x12\xc7\xb0\x0e\x60\xe4\x8d\xb7\xe1\x52",
	.nonce	= "\x4f\x07\xaf\xed\xfd\xc3\xb6\xc2\x36\x18\x23\xd3",
	.nlen	= 12,
	.assoc	= "",
	.alen	= 0,
	.input	= "\xbe\x33\x08\xf7\x2a\x2c\x6a\xed",
	.ilen	= 8,
	.result	= "\x8e\x94\x39\xa5\x6e\xee\xc8\x17\xfb\xe8\xa6\xed\x8f\xab\xb1\x93\x75\x39\xdd\x6c\x00\xe9\x00\x21"
}, { /* wycheproof - misc */
	.key	= "\x11\x31\xc1\x41\x85\x77\xa0\x54\xde\x7a\x4a\xc5\x51\x95\x0f\x1a\x05\x3f\x9a\xe4\x6e\x5b\x75\xfe\x4a\xbd\x56\x08\xd7\xcd\xda\xdd",
	.nonce	= "\xb4\xea\x66\x6e\xe1\x19\x56\x33\x66\x48\x4a\x78",
	.nlen	= 12,
	.assoc	= "\x66\xc0\xae\x70\x07\x6c\xb1\x4d",
	.alen	= 8,
	.input	= "\xa4\xc9\xc2\x80\x1b\x71\xf7\xdf",
	.ilen	= 8,
	.result	= "\xb9\xb9\x10\x43\x3a\xf0\x52\xb0\x45\x30\xf5\x1a\xee\xe0\x24\xe0\xa4\x45\xa6\x32\x8f\xa6\x7a\x18"
}, { /* wycheproof - misc */
	.key	= "\x99\xb6\x2b\xd5\xaf\xbe\x3f\xb0\x15\xbd\xe9\x3f\x0a\xbf\x48\x39\x57\xa1\xc3\xeb\x3c\xa5\x9c\xb5\x0b\x39\xf7\xf8\xa9\xcc\x51\xbe",
	.nonce	= "\x9a\x59\xfc\xe2\x6d\xf0\x00\x5e\x07\x53\x86\x56",
	.nlen	= 12,
	.assoc	= "",
	.alen	= 0,
	.input	= "\x42\xba\xae\x59\x78\xfe\xaf\x5c\x36\x8d\x14\xe0",
	.ilen	= 12,
	.result	= "\xff\x7d\xc2\x03\xb2\x6c\x46\x7a\x6b\x50\xdb\x33\x57\x8c\x0f\x27\x58\xc2\xe1\x4e\x36\xd4\xfc\x10\x6d\xcb\x29\xb4"
}, { /* wycheproof - misc */
	.key	= "\x85\xf3\x5b\x62\x82\xcf\xf4\x40\xbc\x10\x20\xc8\x13\x6f\xf2\x70\x31\x11\x0f\xa6\x3e\xc1\x6f\x1e\x82\x51\x18\xb0\x06\xb9\x12\x57",
	.nonce	= "\x58\xdb\xd4\xad\x2c\x4a\xd3\x5d\xd9\x06\xe9\xce",
	.nlen	= 12,
	.assoc	= "\xa5\x06\xe1\xa5\xc6\x90\x93\xf9",
	.alen	= 8,
	.input	= "\xfd\xc8\x5b\x94\xa4\xb2\xa6\xb7\x59\xb1\xa0\xda",
	.ilen	= 12,
	.result	= "\x9f\x88\x16\xde\x09\x94\xe9\x38\xd9\xe5\x3f\x95\xd0\x86\xfc\x6c\x9d\x8f\xa9\x15\xfd\x84\x23\xa7\xcf\x05\x07\x2f"
}, { /* wycheproof - misc */
	.key	= "\x67\x11\x96\x27\xbd\x98\x8e\xda\x90\x62\x19\xe0\x8c\x0d\x0d\x77\x9a\x07\xd2\x08\xce\x8a\x4f\xe0\x70\x9a\xf7\x55\xee\xec\x6d\xcb",
	.nonce	= "\x68\xab\x7f\xdb\xf6\x19\x01\xda\xd4\x61\xd2\x3c",
	.nlen	= 12,
	.assoc	= "",
	.alen	= 0,
	.input	= "\x51\xf8\xc1\xf7\x31\xea\x14\xac\xdb\x21\x0a\x6d\x97\x3e\x07",
	.ilen	= 15,
	.result	= "\x0b\x29\x63\x8e\x1f\xbd\xd6\xdf\x53\x97\x0b\xe2\x21\x00\x42\x2a\x91\x34\x08\x7d\x67\xa4\x6e\x79\x17\x8d\x0a\x93\xf5\xe1\xd2"
}, { /* wycheproof - misc */
	.key	= "\xe6\xf1\x11\x8d\x41\xe4\xb4\x3f\xb5\x82\x21\xb7\xed\x79\x67\x38\x34\xe0\xd8\xac\x5c\x4f\xa6\x0b\xbc\x8b\xc4\x89\x3a\x58\x89\x4d",
	.nonce	= "\xd9\x5b\x32\x43\xaf\xae\xf7\x14\xc5\x03\x5b\x6a",
	.nlen	= 12,
	.assoc	= "\x64\x53\xa5\x33\x84\x63\x22\x12",
	.alen	= 8,
	.input	= "\x97\x46\x9d\xa6\x67\xd6\x11\x0f\x9c\xbd\xa1\xd1\xa2\x06\x73",
	.ilen	= 15,
	.result	= "\x32\xdb\x66\xc4\xa3\x81\x9d\x81\x55\x74\x55\xe5\x98\x0f\xed\xfe\xae\x30\xde\xc9\x4e\x6a\xd3\xa9\xee\xa0\x6a\x0d\x70\x39\x17"
}, { /* wycheproof - misc */
	.key	= "\x59\xd4\xea\xfb\x4d\xe0\xcf\xc7\xd3\xdb\x99\xa8\xf5\x4b\x15\xd7\xb3\x9f\x0a\xcc\x8d\xa6\x97\x63\xb0\x19\xc1\x69\x9f\x87\x67\x4a",
	.nonce	= "\x2f\xcb\x1b\x38\xa9\x9e\x71\xb8\x47\x40\xad\x9b",
	.nlen	= 12,
	.assoc	= "",
	.alen	= 0,
	.input	= "\x54\x9b\x36\x5a\xf9\x13\xf3\xb0\x81\x13\x1c\xcb\x6b\x82\x55\x88",
	.ilen	= 16,
	.result	= "\xe9\x11\x0e\x9f\x56\xab\x3c\xa4\x83\x50\x0c\xea\xba\xb6\x7a\x13\x83\x6c\xca\xbf\x15\xa6\xa2\x2a\x51\xc1\x07\x1c\xfa\x68\xfa\x0c"
}, { /* wycheproof - misc */
	.key	= "\xb9\x07\xa4\x50\x75\x51\x3f\xe8\xa8\x01\x9e\xde\xe3\xf2\x59\x14\x87\xb2\xa0\x30\xb0\x3c\x6e\x1d\x77\x1c\x86\x25\x71\xd2\xea\x1e",
	.nonce	= "\x11\x8a\x69\x64\xc2\xd3\xe3\x80\x07\x1f\x52\x66",
	.nlen	= 12,
	.assoc	= "\x03\x45\x85\x62\x1a\xf8\xd7\xff",
	.alen	= 8,
	.input	= "\x55\xa4\x65\x64\x4f\x5b\x65\x09\x28\xcb\xee\x7c\x06\x32\x14\xd6",
	.ilen	= 16,
	.result	= "\xe4\xb1\x13\xcb\x77\x59\x45\xf3\xd3\xa8\xae\x9e\xc1\x41\xc0\x0c\x7c\x43\xf1\x6c\xe0\x96\xd0\xdc\x27\xc9\x58\x49\xdc\x38\x3b\x7d"
}, { /* wycheproof - misc */
	.key	= "\x3b\x24\x58\xd8\x17\x6e\x16\x21\xc0\xcc\x24\xc0\xc0\xe2\x4c\x1e\x80\xd7\x2f\x7e\xe9\x14\x9a\x4b\x16\x61\x76\x62\x96\x16\xd0\x11",
	.nonce	= "\x45\xaa\xa3\xe5\xd1\x6d\x2d\x42\xdc\x03\x44\x5d",
	.nlen	= 12,
	.assoc	= "",
	.alen	= 0,
	.input	= "\x3f\xf1\x51\x4b\x1c\x50\x39\x15\x91\x8f\x0c\x0c\x31\x09\x4a\x6e\x1f",
	.ilen	= 17,
	.result	= "\x02\xcc\x3a\xcb\x5e\xe1\xfc\xdd\x12\xa0\x3b\xb8\x57\x97\x64\x74\xd3\xd8\x3b\x74\x63\xa2\xc3\x80\x0f\xe9\x58\xc2\x8e\xaa\x29\x08\x13"
}, { /* wycheproof - misc */
	.key	= "\xf6\x0c\x6a\x1b\x62\x57\x25\xf7\x6c\x70\x37\xb4\x8f\xe3\x57\x7f\xa7\xf7\xb8\x7b\x1b\xd5\xa9\x82\x17\x6d\x18\x23\x06\xff\xb8\x70",
	.nonce	= "\xf0\x38\x4f\xb8\x76\x12\x14\x10\x63\x3d\x99\x3d",
	.nlen	= 12,
	.assoc	= "\x9a\xaf\x29\x9e\xee\xa7\x8f\x79",
	.alen	= 8,
	.input	= "\x63\x85\x8c\xa3\xe2\xce\x69\x88\x7b\x57\x8a\x3c\x16\x7b\x42\x1c\x9c",
	.ilen	= 17,
	.result	= "\x35\x76\x64\x88\xd2\xbc\x7c\x2b\x8d\x17\xcb\xbb\x9a\xbf\xad\x9e\x6d\x1f\x39\x1e\x65\x7b\x27\x38\xdd\xa0\x84\x48\xcb\xa2\x81\x1c\xeb"
}, { /* wycheproof - misc */
	.key	= "\x02\x12\xa8\xde\x50\x07\xed\x87\xb3\x3f\x1a\x70\x90\xb6\x11\x4f\x9e\x08\xce\xfd\x96\x07\xf2\xc2\x76\xbd\xcf\xdb\xc5\xce\x9c\xd7",
	.nonce	= "\xe6\xb1\xad\xf2\xfd\x58\xa8\x76\x2c\x65\xf3\x1b",
	.nlen	= 12,
	.assoc	= "",
	.alen	= 0,
	.input	= "\x10\xf1\xec\xf9\xc6\x05\x84\x66\x5d\x9a\xe5\xef\xe2\x79\xe7\xf7\x37\x7e\xea\x69\x16\xd2\xb1\x11",
	.ilen	= 24,
	.result	= "\x42\xf2\x6c\x56\xcb\x4b\xe2\x1d\x9d\x8d\x0c\x80\xfc\x99\xdd\xe0\x0d\x75\xf3\x80\x74\xbf\xe7\x64\x54\xaa\x7e\x13\xd4\x8f\xff\x7d\x75\x57\x03\x94\x57\x04\x0a\x3a"
}, { /* wycheproof - misc */
	.key	= "\xc5\xbc\x09\x56\x56\x46\xe7\xed\xda\x95\x4f\x1f\x73\x92\x23\xda\xda\x20\xb9\x5c\x44\xab\x03\x3d\x0f\xae\x4b\x02\x83\xd1\x8b\xe3",
	.nonce	= "\x6b\x28\x2e\xbe\xcc\x54\x1b\xcd\x78\x34\xed\x55",
	.nlen	= 12,
	.assoc	= "\x3e\x8b\xc5\xad\xe1\x82\xff\x08",
	.alen	= 8,
	.input	= "\x92\x22\xf9\x01\x8e\x54\xfd\x6d\xe1\x20\x08\x06\xa9\xee\x8e\x4c\xc9\x04\xd2\x9f\x25\xcb\xa1\x93",
	.ilen	= 24,
	.result	= "\x12\x30\x32\x43\x7b\x4b\xfd\x69\x20\xe8\xf7\xe7\xe0\x08\x7a\xe4\x88\x9e\xbe\x7a\x0a\xd0\xe9\x00\x3c\xf6\x8f\x17\x95\x50\xda\x63\xd3\xb9\x6c\x2d\x55\x41\x18\x65"
}, { /* wycheproof - misc */
	.key	= "\x2e\xb5\x1c\x46\x9a\xa8\xeb\x9e\x6c\x54\xa8\x34\x9b\xae\x50\xa2\x0f\x0e\x38\x27\x11\xbb\xa1\x15\x2c\x42\x4f\x03\xb6\x67\x1d\x71",
	.nonce	= "\x04\xa9\xbe\x03\x50\x8a\x5f\x31\x37\x1a\x6f\xd2",
	.nlen	= 12,
	.assoc	= "",
	.alen	= 0,
	.input	= "\xb0\x53\x99\x92\x86\xa2\x82\x4f\x42\xcc\x8c\x20\x3a\xb2\x4e\x2c\x97\xa6\x85\xad\xcc\x2a\xd3\x26\x62\x55\x8e\x55\xa5\xc7\x29",
	.ilen	= 31,
	.result	= "\x45\xc7\xd6\xb5\x3a\xca\xd4\xab\xb6\x88\x76\xa6\xe9\x6a\x48\xfb\x59\x52\x4d\x2c\x92\xc9\xd8\xa1\x89\xc9\xfd\x2d\xb9\x17\x46\x56\x6d\x3c\xa1\x0e\x31\x1b\x69\x5f\x3e\xae\x15\x51\x65\x24\x93"
}, { /* wycheproof - misc */
	.key	= "\x7f\x5b\x74\xc0\x7e\xd1\xb4\x0f\xd1\x43\x58\xfe\x2f\xf2\xa7\x40\xc1\x16\xc7\x70\x65\x10\xe6\xa4\x37\xf1\x9e\xa4\x99\x11\xce\xc4",
	.nonce	= "\x47\x0a\x33\x9e\xcb\x32\x19\xb8\xb8\x1a\x1f\x8b",
	.nlen	= 12,
	.assoc	= "\x37\x46\x18\xa0\x6e\xa9\x8a\x48",
	.alen	= 8,
	.input	= "\xf4\x52\x06\xab\xc2\x55\x52\xb2\xab\xc9\xab\x7f\xa2\x43\x03\x5f\xed\xaa\xdd\xc3\xb2\x29\x39\x56\xf1\xea\x6e\x71\x56\xe7\xeb",
	.ilen	= 31,
	.result	= "\x46\xa8\x0c\x41\x87\x02\x47\x20\x08\x46\x27\x58\x00\x80\xdd\xe5\xa3\xf4\xa1\x10\x93\xa7\x07\x6e\xd6\xf3\xd3\x26\xbc\x7b\x70\x53\x4d\x4a\xa2\x83\x5a\x52\xe7\x2d\x14\xdf\x0e\x4f\x47\xf2\x5f"
}, { /* wycheproof - misc */
	.key	= "\xe1\x73\x1d\x58\x54\xe1\xb7\x0c\xb3\xff\xe8\xb7\x86\xa2\xb3\xeb\xf0\x99\x43\x70\x95\x47\x57\xb9\xdc\x8c\x7b\xc5\x35\x46\x34\xa3",
	.nonce	= "\x72\xcf\xd9\x0e\xf3\x02\x6c\xa2\x2b\x7e\x6e\x6a",
	.nlen	= 12,
	.assoc	= "",
	.alen	= 0,
	.input	= "\xb9\xc5\x54\xcb\xc3\x6a\xc1\x8a\xe8\x97\xdf\x7b\xee\xca\xc1\xdb\xeb\x4e\xaf\xa1\x56\xbb\x60\xce\x2e\x5d\x48\xf0\x57\x15\xe6\x78",
	.ilen	= 32,
	.result	= "\xea\x29\xaf\xa4\x9d\x36\xe8\x76\x0f\x5f\xe1\x97\x23\xb9\x81\x1e\xd5\xd5\x19\x93\x4a\x44\x0f\x50\x81\xac\x43\x0b\x95\x3b\x0e\x21\x22\x25\x41\xaf\x46\xb8\x65\x33\xc6\xb6\x8d\x2f\xf1\x08\xa7\xea"
}, { /* wycheproof - misc */
	.key	= "\x27\xd8\x60\x63\x1b\x04\x85\xa4\x10\x70\x2f\xea\x61\xbc\x87\x3f\x34\x42\x26\x0c\xad\xed\x4a\xbd\xe2\x5b\x78\x6a\x2d\x97\xf1\x45",
	.nonce	= "\x26\x28\x80\xd4\x75\xf3\xda\xc5\x34\x0d\xd1\xb8",
	.nlen	= 12,
	.assoc	= "\x23\x33\xe5\xce\x0f\x93\xb0\x59",
	.alen	= 8,
	.input	= "\x6b\x26\x04\x99\x6c\xd3\x0c\x14\xa1\x3a\x52\x57\xed\x6c\xff\xd3\xbc\x5e\x29\xd6\xb9\x7e\xb1\x79\x9e\xb3\x35\xe2\x81\xea\x45\x1e",
	.ilen	= 32,
	.result	= "\x6d\xad\x63\x78\x97\x54\x4d\x8b\xf6\xbe\x95\x07\xed\x4d\x1b\xb2\xe9\x54\xbc\x42\x7e\x5d\xe7\x29\xda\xf5\x07\x62\x84\x6f\xf2\xf4\x7b\x99\x7d\x93\xc9\x82\x18\x9d\x70\x95\xdc\x79\x4c\x74\x62\x32"
}, { /* wycheproof - misc */
	.key	= "\xcf\x0d\x40\xa4\x64\x4e\x5f\x51\x81\x51\x65\xd5\x30\x1b\x22\x63\x1f\x45\x44\xc4\x9a\x18\x78\xe3\xa0\xa5\xe8\xe1\xaa\xe0\xf2\x64",
	.nonce	= "\xe7\x4a\x51\x5e\x7e\x21\x02\xb9\x0b\xef\x55\xd2",
	.nlen	= 12,
	.assoc	= "",
	.alen	= 0,
	.input	= "\x97\x3d\x0c\x75\x38\x26\xba\xe4\x66\xcf\x9a\xbb\x34\x93\x15\x2e\x9d\xe7\x81\x9e\x2b\xd0\xc7\x11\x71\x34\x6b\x4d\x2c\xeb\xf8\x04\x1a\xa3\xce\xdc\x0d\xfd\x7b\x46\x7e\x26\x22\x8b\xc8\x6c\x9a",
	.ilen	= 47,
	.result	= "\xfb\xa7\x8a\xe4\xf9\xd8\x08\xa6\x2e\x3d\xa4\x0b\xe2\xcb\x77\x00\xc3\x61\x3d\x9e\xb2\xc5\x29\xc6\x52\xe7\x6a\x43\x2c\x65\x8d\x27\x09\x5f\x0e\xb8\xf9\x40\xc3\x24\x98\x1e\xa9\x35\xe5\x07\xf9\x8f\x04\x69\x56\xdb\x3a\x51\x29\x08\xbd\x7a\xfc\x8f\x2a\xb0\xa9"
}, { /* wycheproof - misc */
	.key	= "\x6c\xbf\xd7\x1c\x64\x5d\x18\x4c\xf5\xd2\x3c\x40\x2b\xdb\x0d\x25\xec\x54\x89\x8c\x8a\x02\x73\xd4\x2e\xb5\xbe\x10\x9f\xdc\xb2\xac",
	.nonce	= "\xd4\xd8\x07\x34\x16\x83\x82\x5b\x31\xcd\x4d\x95",
	.nlen	= 12,
	.assoc	= "\xb3\xe4\x06\x46\x83\xb0\x2d\x84",
	.alen	= 8,
	.input	= "\xa9\x89\x95\x50\x4d\xf1\x6f\x74\x8b\xfb\x77\x85\xff\x91\xee\xb3\xb6\x60\xea\x9e\xd3\x45\x0c\x3d\x5e\x7b\x0e\x79\xef\x65\x36\x59\xa9\x97\x8d\x75\x54\x2e\xf9\x1c\x45\x67\x62\x21\x56\x40\xb9",
	.ilen	= 47,
	.result	= "\xa1\xff\xed\x80\x76\x18\x29\xec\xce\x24\x2e\x0e\x88\xb1\x38\x04\x90\x16\xbc\xa0\x18\xda\x2b\x6e\x19\x98\x6b\x3e\x31\x8c\xae\x8d\x80\x61\x98\xfb\x4c\x52\x7c\xc3\x93\x50\xeb\xdd\xea\xc5\x73\xc4\xcb\xf0\xbe\xfd\xa0\xb7\x02\x42\xc6\x40\xd7\xcd\x02\xd7\xa3"
}, { /* wycheproof - misc */
	.key	= "\x5b\x1d\x10\x35\xc0\xb1\x7e\xe0\xb0\x44\x47\x67\xf8\x0a\x25\xb8\xc1\xb7\x41\xf4\xb5\x0a\x4d\x30\x52\x22\x6b\xaa\x1c\x6f\xb7\x01",
	.nonce	= "\xd6\x10\x40\xa3\x13\xed\x49\x28\x23\xcc\x06\x5b",
	.nlen	= 12,
	.assoc	= "",
	.alen	= 0,
	.input	= "\xd0\x96\x80\x31\x81\xbe\xef\x9e\x00\x8f\xf8\x5d\x5d\xdc\x38\xdd\xac\xf0\xf0\x9e\xe5\xf7\xe0\x7f\x1e\x40\x79\xcb\x64\xd0\xdc\x8f\x5e\x67\x11\xcd\x49\x21\xa7\x88\x7d\xe7\x6e\x26\x78\xfd\xc6\x76\x18\xf1\x18\x55\x86\xbf\xea\x9d\x4c\x68\x5d\x50\xe4\xbb\x9a\x82",
	.ilen	= 64,
	.result	= "\x9a\x4e\xf2\x2b\x18\x16\x77\xb5\x75\x5c\x08\xf7\x47\xc0\xf8\xd8\xe8\xd4\xc1\x8a\x9c\xc2\x40\x5c\x12\xbb\x51\xbb\x18\x72\xc8\xe8\xb8\x77\x67\x8b\xec\x44\x2c\xfc\xbb\x0f\xf4\x64\xa6\x4b\x74\x33\x2c\xf0\x72\x89\x8c\x7e\x0e\xdd\xf6\x23\x2e\xa6\xe2\x7e\xfe\x50\x9f\xf3\x42\x7a\x0f\x32\xfa\x56\x6d\x9c\xa0\xa7\x8a\xef\xc0\x13"
}, { /* wycheproof - misc */
	.key	= "\x97\xd6\x35\xc4\xf4\x75\x74\xd9\x99\x8a\x90\x87\x5d\xa1\xd3\xa2\x84\xb7\x55\xb2\xd3\x92\x97\xa5\x72\x52\x35\x19\x0e\x10\xa9\x7e",
	.nonce	= "\xd3\x1c\x21\xab\xa1\x75\xb7\x0d\xe4\xeb\xb1\x9c",
	.nlen	= 12,
	.assoc	= "\x71\x93\xf6\x23\x66\x33\x21\xa2",
	.alen	= 8,
	.input	= "\x94\xee\x16\x6d\x6d\x6e\xcf\x88\x32\x43\x71\x36\xb4\xae\x80\x5d\x42\x88\x64\x35\x95\x86\xd9\x19\x3a\x25\x01\x62\x93\xed\xba\x44\x3c\x58\xe0\x7e\x7b\x71\x95\xec\x5b\xd8\x45\x82\xa9\xd5\x6c\x8d\x4a\x10\x8c\x7d\x7c\xe3\x4e\x6c\x6f\x8e\xa1\xbe\xc0\x56\x73\x17",
	.ilen	= 64,
	.result	= "\x5f\xbb\xde\xcc\x34\xbe\x20\x16\x14\xf6\x36\x03\x1e\xeb\x42\xf1\xca\xce\x3c\x79\xa1\x2c\xff\xd8\x71\xee\x8e\x73\x82\x0c\x82\x97\x49\xf1\xab\xb4\x29\x43\x67\x84\x9f\xb6\xc2\xaa\x56\xbd\xa8\xa3\x07\x8f\x72\x3d\x7c\x1c\x85\x20\x24\xb0\x17\xb5\x89\x73\xfb\x1e\x09\x26\x3d\xa7\xb4\xcb\x92\x14\x52\xf9\x7d\xca\x40\xf5\x80\xec"
}, { /* wycheproof - misc */
	.key	= "\xfe\x6e\x55\xbd\xae\xd1\xf7\x28\x4c\xa5\xfc\x0f\x8c\x5f\x2b\x8d\xf5\x6d\xc0\xf4\x9e\x8c\xa6\x6a\x41\x99\x5e\x78\x33\x51\xf9\x01",
	.nonce	= "\x17\xc8\x6a\x8a\xbb\xb7\xe0\x03\xac\xde\x27\x99",
	.nlen	= 12,
	.assoc	= "",
	.alen	= 0,
	.input	= "\xb4\x29\xeb\x80\xfb\x8f\xe8\xba\xed\xa0\xc8\x5b\x9c\x33\x34\x58\xe7\xc2\x99\x2e\x55\x84\x75\x06\x9d\x12\xd4\x5c\x22\x21\x75\x64\x12\x15\x88\x03\x22\x97\xef\xf5\x67\x83\x74\x2a\x5f\xc2\x2d\x74\x10\xff\xb2\x9d\x66\x09\x86\x61\xd7\x6f\x12\x6c\x3c\x27\x68\x9e\x43\xb3\x72\x67\xca\xc5\xa3\xa6\xd3\xab\x49\xe3\x91\xda\x29\xcd\x30\x54\xa5\x69\x2e\x28\x07\xe4\xc3\xea\x46\xc8\x76\x1d\x50\xf5\x92",
	.ilen	= 97,
	.result	= "\xd0\x10\x2f\x6c\x25\x8b\xf4\x97\x42\xce\xc3\x4c\xf2\xd0\xfe\xdf\x23\xd1\x05\xfb\x4c\x84\xcf\x98\x51\x5e\x1b\xc9\xa6\x4f\x8a\xd5\xbe\x8f\x07\x21\xbd\xe5\x06\x45\xd0\x00\x83\xc3\xa2\x63\xa3\x10\x53\xb7\x60\x24\x5f\x52\xae\x28\x66\xa5\xec\x83\xb1\x9f\x61\xbe\x1d\x30\xd5\xc5\xd9\xfe\xcc\x4c\xbb\xe0\x8f\xd3\x85\x81\x3a\x2a\xa3\x9a\x00\xff\x9c\x10\xf7\xf2\x37\x02\xad\xd1\xe4\xb2\xff\xa3\x1c\x41\x86\x5f\xc7\x1d\xe1\x2b\x19\x61\x21\x27\xce\x49\x99\x3b\xb0"
}, { /* wycheproof - misc */
	.key	= "\xaa\xbc\x06\x34\x74\xe6\x5c\x4c\x3e\x9b\xdc\x48\x0d\xea\x97\xb4\x51\x10\xc8\x61\x88\x46\xff\x6b\x15\xbd\xd2\xa4\xa5\x68\x2c\x4e",
	.nonce	= "\x46\x36\x2f\x45\xd6\x37\x9e\x63\xe5\x22\x94\x60",
	.nlen	= 12,
	.assoc	= "\xa1\x1c\x40\xb6\x03\x76\x73\x30",
	.alen	= 8,
	.input	= "\xce\xb5\x34\xce\x50\xdc\x23\xff\x63\x8a\xce\x3e\xf6\x3a\xb2\xcc\x29\x73\xee\xad\xa8\x07\x85\xfc\x16\x5d\x06\xc2\xf5\x10\x0f\xf5\xe8\xab\x28\x82\xc4\x75\xaf\xcd\x05\xcc\xd4\x9f\x2e\x7d\x8f\x55\xef\x3a\x72\xe3\xdc\x51\xd6\x85\x2b\x8e\x6b\x9e\x7a\xec\xe5\x7b\xe6\x55\x6b\x0b\x6d\x94\x13\xe3\x3f\xc5\xfc\x24\xa9\xa2\x05\xad\x59\x57\x4b\xb3\x9d\x94\x4a\x92\xdc\x47\x97\x0d\x84\xa6\xad\x31\x76",
	.ilen	= 97,
	.result	= "\x75\x45\x39\x1b\x51\xde\x01\xd5\xc5\x3d\xfa\xca\x77\x79\x09\x06\x3e\x58\xed\xee\x4b\xb1\x22\x7e\x71\x10\xac\x4d\x26\x20\xc2\xae\xc2\xf8\x48\xf5\x6d\xee\xb0\x37\xa8\xdc\xed\x75\xaf\xa8\xa6\xc8\x90\xe2\xde\xe4\x2f\x95\x0b\xb3\x3d\x9e\x24\x24\xd0\x8a\x50\x5d\x89\x95\x63\x97\x3e\xd3\x88\x70\xf3\xde\x6e\xe2\xad\xc7\xfe\x07\x2c\x36\x6c\x14\xe2\xcf\x7c\xa6\x2f\xb3\xd3\x6b\xee\x11\x68\x54\x61\xb7\x0d\x44\xef\x8c\x66\xc5\xc7\xbb\xf1\x0d\xca\xdd\x7f\xac\xf6"
}, { /* wycheproof - misc */
	.key	= "\x7d\x00\xb4\x80\x95\xad\xfa\x32\x72\x05\x06\x07\xb2\x64\x18\x50\x02\xba\x99\x95\x7c\x49\x8b\xe0\x22\x77\x0f\x2c\xe2\xf3\x14\x3c",
	.nonce	= "\x87\x34\x5f\x10\x55\xfd\x9e\x21\x02\xd5\x06\x56",
	.nlen	= 12,
	.assoc	= "\x02",
	.alen	= 1,
	.input	= "\xe5\xcc\xaa\x44\x1b\xc8\x14\x68\x8f\x8f\x6e\x8f\x28\xb5\x00\xb2",
	.ilen	= 16,
	.result	= "\x7e\x72\xf5\xa1\x85\xaf\x16\xa6\x11\x92\x1b\x43\x8f\x74\x9f\x0b\x12\x42\xc6\x70\x73\x23\x34\x02\x9a\xdf\xe1\xc5\x00\x16\x51\xe4"
}, { /* wycheproof - misc */
	.key	= "\x64\x32\x71\x7f\x1d\xb8\x5e\x41\xac\x78\x36\xbc\xe2\x51\x85\xa0\x80\xd5\x76\x2b\x9e\x2b\x18\x44\x4b\x6e\xc7\x2c\x3b\xd8\xe4\xdc",
	.nonce	= "\x87\xa3\x16\x3e\xc0\x59\x8a\xd9\x5b\x3a\xa7\x13",
	.nlen	= 12,
	.assoc	= "\xb6\x48",
	.alen	= 2,
	.input	= "\x02\xcd\xe1\x68\xfb\xa3\xf5\x44\xbb\xd0\x33\x2f\x7a\xde\xad\xa8",
	.ilen	= 16,
	.result	= "\x85\xf2\x9a\x71\x95\x57\xcd\xd1\x4d\x1f\x8f\xff\xab\x6d\x9e\x60\x73\x2c\xa3\x2b\xec\xd5\x15\xa1\xed\x35\x3f\x54\x2e\x99\x98\x58"
}, { /* wycheproof - misc */
	.key	= "\x8e\x34\xcf\x73\xd2\x45\xa1\x08\x2a\x92\x0b\x86\x36\x4e\xb8\x96\xc4\x94\x64\x67\xbc\xb3\xd5\x89\x29\xfc\xb3\x66\x90\xe6\x39\x4f",
	.nonce	= "\x6f\x57\x3a\xa8\x6b\xaa\x49\x2b\xa4\x65\x96\xdf",
	.nlen	= 12,
	.assoc	= "\xbd\x4c\xd0\x2f\xc7\x50\x2b\xbd\xbd\xf6\xc9\xa3\xcb\xe8\xf0",
	.alen	= 15,
	.input	= "\x16\xdd\xd2\x3f\xf5\x3f\x3d\x23\xc0\x63\x34\x48\x70\x40\xeb\x47",
	.ilen	= 16,
	.result	= "\xc1\xb2\x95\x93\x6d\x56\xfa\xda\xc0\x3e\x5f\x74\x2b\xff\x73\xa1\x39\xc4\x57\xdb\xab\x66\x38\x2b\xab\xb3\xb5\x58\x00\xcd\xa5\xb8"
}, { /* wycheproof - misc */
	.key	= "\xcb\x55\x75\xf5\xc7\xc4\x5c\x91\xcf\x32\x0b\x13\x9f\xb5\x94\x23\x75\x60\xd0\xa3\xe6\xf8\x65\xa6\x7d\x4f\x63\x3f\x2c\x08\xf0\x16",
	.nonce	= "\x1a\x65\x18\xf0\x2e\xde\x1d\xa6\x80\x92\x66\xd9",
	.nlen	= 12,
	.assoc	= "\x89\xcc\xe9\xfb\x47\x44\x1d\x07\xe0\x24\x5a\x66\xfe\x8b\x77\x8b",
	.alen	= 16,
	.input	= "\x62\x3b\x78\x50\xc3\x21\xe2\xcf\x0c\x6f\xbc\xc8\xdf\xd1\xaf\xf2",
	.ilen	= 16,
	.result	= "\xc8\x4c\x9b\xb7\xc6\x1c\x1b\xcb\x17\x77\x2a\x1c\x50\x0c\x50\x95\xdb\xad\xf7\xa5\x13\x8c\xa0\x34\x59\xa2\xcd\x65\x83\x1e\x09\x2f"
}, { /* wycheproof - misc */
	.key	= "\xa5\x56\x9e\x72\x9a\x69\xb2\x4b\xa6\xe0\xff\x15\xc4\x62\x78\x97\x43\x68\x24\xc9\x41\xe9\xd0\x0b\x2e\x93\xfd\xdc\x4b\xa7\x76\x57",
	.nonce	= "\x56\x4d\xee\x49\xab\x00\xd2\x40\xfc\x10\x68\xc3",
	.nlen	= 12,
	.assoc	= "\xd1\x9f\x2d\x98\x90\x95\xf7\xab\x03\xa5\xfd\xe8\x44\x16\xe0\x0c\x0e",
	.alen	= 17,
	.input	= "\x87\xb3\xa4\xd7\xb2\x6d\x8d\x32\x03\xa0\xde\x1d\x64\xef\x82\xe3",
	.ilen	= 16,
	.result	= "\x94\xbc\x80\x62\x1e\xd1\xe7\x1b\x1f\xd2\xb5\xc3\xa1\x5e\x35\x68\x33\x35\x11\x86\x17\x96\x97\x84\x01\x59\x8b\x96\x37\x22\xf5\xb3"
}, { /* wycheproof - misc */
	.key	= "\x56\x20\x74\x65\xb4\xe4\x8e\x6d\x04\x63\x0f\x4a\x42\xf3\x5c\xfc\x16\x3a\xb2\x89\xc2\x2a\x2b\x47\x84\xf6\xf9\x29\x03\x30\xbe\xe0",
	.nonce	= "\xdf\x87\x13\xe8\x7e\xc3\xdb\xcf\xad\x14\xd5\x3e",
	.nlen	= 12,
	.assoc	= "\x5e\x64\x70\xfa\xcd\x99\xc1\xd8\x1e\x37\xcd\x44\x01\x5f\xe1\x94\x80\xa2\xa4\xd3\x35\x2a\x4f\xf5\x60\xc0\x64\x0f\xdb\xda",
	.alen	= 30,
	.input	= "\xe6\x01\xb3\x85\x57\x79\x7d\xa2\xf8\xa4\x10\x6a\x08\x9d\x1d\xa6",
	.ilen	= 16,
	.result	= "\x29\x9b\x5d\x3f\x3d\x03\xc0\x87\x20\x9a\x16\xe2\x85\x14\x31\x11\x4b\x45\x4e\xd1\x98\xde\x11\x7e\x83\xec\x49\xfa\x8d\x85\x08\xd6"
}, { /* wycheproof - misc */
	.key	= "\x39\x37\x98\x6a\xf8\x6d\xaf\xc1\xba\x0c\x46\x72\xd8\xab\xc4\x6c\x20\x70\x62\x68\x2d\x9c\x26\x4a\xb0\x6d\x6c\x58\x07\x20\x51\x30",
	.nonce	= "\x8d\xf4\xb1\x5a\x88\x8c\x33\x28\x6a\x7b\x76\x51",
	.nlen	= 12,
	.assoc	= "\xba\x44\x6f\x6f\x9a\x0c\xed\x22\x45\x0f\xeb\x10\x73\x7d\x90\x07\xfd\x69\xab\xc1\x9b\x1d\x4d\x90\x49\xa5\x55\x1e\x86\xec\x2b\x37",
	.alen	= 32,
	.input	= "\xdc\x9e\x9e\xaf\x11\xe3\x14\x18\x2d\xf6\xa4\xeb\xa1\x7a\xec\x9c",
	.ilen	= 16,
	.result	= "\x60\x5b\xbf\x90\xae\xb9\x74\xf6\x60\x2b\xc7\x78\x05\x6f\x0d\xca\x38\xea\x23\xd9\x90\x54\xb4\x6b\x42\xff\xe0\x04\x12\x9d\x22\x04"
}, { /* wycheproof - misc */
	.key	= "\x36\x37\x2a\xbc\xdb\x78\xe0\x27\x96\x46\xac\x3d\x17\x6b\x96\x74\xe9\x15\x4e\xec\xf0\xd5\x46\x9c\x65\x1e\xc7\xe1\x6b\x4c\x11\x99",
	.nonce	= "\xbe\x40\xe5\xf1\xa1\x18\x17\xa0\xa8\xfa\x89\x49",
	.nlen	= 12,
	.assoc	= "\xd4\x1a\x82\x8d\x5e\x71\x82\x92\x47\x02\x19\x05\x40\x2e\xa2\x57\xdc\xcb\xc3\xb8\x0f\xcd\x56\x75\x05\x6b\x68\xbb\x59\xe6\x2e\x88\x73",
	.alen	= 33,
	.input	= "\x81\xce\x84\xed\xe9\xb3\x58\x59\xcc\x8c\x49\xa8\xf6\xbe\x7d\xc6",
	.ilen	= 16,
	.result	= "\x7b\x7c\xe0\xd8\x24\x80\x9a\x70\xde\x32\x56\x2c\xcf\x2c\x2b\xbd\x15\xd4\x4a\x00\xce\x0d\x19\xb4\x23\x1f\x92\x1e\x22\xbc\x0a\x43"
}, { /* wycheproof - misc */
	.key	= "\x9f\x14\x79\xed\x09\x7d\x7f\xe5\x29\xc1\x1f\x2f\x5a\xdd\x9a\xaf\xf4\xa1\xca\x0b\x68\x99\x7a\x2c\xb7\xf7\x97\x49\xbd\x90\xaa\xf4",
	.nonce	= "\x84\xc8\x7d\xae\x4e\xee\x27\x73\x0e\xc3\x5d\x12",
	.nlen	= 12,
	.assoc	= "\x3f\x2d\xd4\x9b\xbf\x09\xd6\x9a\x78\xa3\xd8\x0e\xa2\x56\x66\x14\xfc\x37\x94\x74\x19\x6c\x1a\xae\x84\x58\x3d\xa7\x3d\x7f\xf8\x5c\x6f\x42\xca\x42\x05\x6a\x97\x92\xcc\x1b\x9f\xb3\xc7\xd2\x61",
	.alen	= 47,
	.input	= "\xa6\x67\x47\xc8\x9e\x85\x7a\xf3\xa1\x8e\x2c\x79\x50\x00\x87\xed",
	.ilen	= 16,
	.result	= "\xca\x82\xbf\xf3\xe2\xf3\x10\xcc\xc9\x76\x67\x2c\x44\x15\xe6\x9b\x57\x63\x8c\x62\xa5\xd8\x5d\xed\x77\x4f\x91\x3c\x81\x3e\xa0\x32"
}, { /* wycheproof - misc */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x01\xee\x32\x00",
	.nlen	= 8,
	.assoc	= "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00",
	.alen	= 16,
	.input	= "\x25\x6d\x40\x88\x80\x94\x17\x83\x55\xd3\x04\x84\x64\x43\xfe\xe8\xdf\x99\x47\x03\x03\xfb\x3b\x7b\x80\xe0\x30\xbe\xeb\xd3\x29\xbe",
	.ilen	= 32,
	.result	= "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe6\xd3\xd7\x32\x4a\x1c\xbb\xa7\x77\xbb\xb0\xec\xdd\xa3\x78\x07"
}, { /* wycheproof - misc */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x01\xee\x32\x00",
	.nlen	= 8,
	.assoc	= "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00",
	.alen	= 16,
	.input	= "\x25\x6d\x40\x88\x80\x94\x17\x83\x55\xd3\x04\x84\x64\x43\xfe\xe8\xdf\x99\x47\x03\x03\xfb\x3b\x7b\x80\xe0\x30\xbe\xeb\xd3\x29\xbe\xe3\xbc\xdb\x5b\x1e\xde\xfc\xfe\x8b\xcd\xa1\xb6\xa1\x5c\x8c\x2b\x08\x69\xff\xd2\xec\x5e\x26\xe5\x53\xb7\xb2\x27\xfe\x87\xfd\xbd",
	.ilen	= 64,
	.result	= "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x06\x2d\xe6\x79\x5f\x27\x4f\xd2\xa3\x05\xd7\x69\x80\xbc\x9c\xce"
}, { /* wycheproof - misc */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x01\xee\x32\x00",
	.nlen	= 8,
	.assoc	= "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00",
	.alen	= 16,
	.input	= "\x25\x6d\x40\x88\x80\x94\x17\x83\x55\xd3\x04\x84\x64\x43\xfe\xe8\xdf\x99\x47\x03\x03\xfb\x3b\x7b\x80\xe0\x30\xbe\xeb\xd3\x29\xbe\xe3\xbc\xdb\x5b\x1e\xde\xfc\xfe\x8b\xcd\xa1\xb6\xa1\x5c\x8c\x2b\x08\x69\xff\xd2\xec\x5e\x26\xe5\x53\xb7\xb2\x27\xfe\x87\xfd\xbd\x7a\xda\x44\x42\x42\x69\xbf\xfa\x55\x27\xf2\x70\xac\xf6\x85\x02\xb7\x4c\x5a\xe2\xe6\x0c\x05\x80\x98\x1a\x49\x38\x45\x93\x92\xc4\x9b\xb2\xf2\x84\xb6\x46\xef\xc7\xf3\xf0\xb1\x36\x1d\xc3\x48\xed\x77\xd3\x0b\xc5\x76\x92\xed\x38\xfb\xac\x01\x88\x38\x04\x88\xc7",
	.ilen	= 128,
	.result	= "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xb4\x79\x02\xba\xae\xaf\xb3\x42\x03\x05\x15\x29\xaf\x28\x2e"
}, { /* wycheproof - misc */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x01\xee\x32\x00",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff",
	.alen	= 16,
	.input	= "\xda\x92\xbf\x77\x7f\x6b\xe8\x7c\xaa\x2c\xfb\x7b\x9b\xbc\x01\x17\x20\x66\xb8\xfc\xfc\x04\xc4\x84\x7f\x1f\xcf\x41\x14\x2c\xd6\x41",
	.ilen	= 32,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xb3\x89\x1c\x84\x9c\xb5\x2c\x27\x74\x7e\xdf\xcf\x31\x21\x3b\xb6"
}, { /* wycheproof - misc */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x01\xee\x32\x00",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff",
	.alen	= 16,
	.input	= "\xda\x92\xbf\x77\x7f\x6b\xe8\x7c\xaa\x2c\xfb\x7b\x9b\xbc\x01\x17\x20\x66\xb8\xfc\xfc\x04\xc4\x84\x7f\x1f\xcf\x41\x14\x2c\xd6\x41\x1c\x43\x24\xa4\xe1\x21\x03\x01\x74\x32\x5e\x49\x5e\xa3\x73\xd4\xf7\x96\x00\x2d\x13\xa1\xd9\x1a\xac\x48\x4d\xd8\x01\x78\x02\x42",
	.ilen	= 64,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xf0\xc1\x2d\x26\xef\x03\x02\x9b\x62\xc0\x08\xda\x27\xc5\xdc\x68"
}, { /* wycheproof - misc */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x01\xee\x32\x00",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff",
	.alen	= 16,
	.input	= "\xda\x92\xbf\x77\x7f\x6b\xe8\x7c\xaa\x2c\xfb\x7b\x9b\xbc\x01\x17\x20\x66\xb8\xfc\xfc\x04\xc4\x84\x7f\x1f\xcf\x41\x14\x2c\xd6\x41\x1c\x43\x24\xa4\xe1\x21\x03\x01\x74\x32\x5e\x49\x5e\xa3\x73\xd4\xf7\x96\x00\x2d\x13\xa1\xd9\x1a\xac\x48\x4d\xd8\x01\x78\x02\x42\x85\x25\xbb\xbd\xbd\x96\x40\x05\xaa\xd8\x0d\x8f\x53\x09\x7a\xfd\x48\xb3\xa5\x1d\x19\xf3\xfa\x7f\x67\xe5\xb6\xc7\xba\x6c\x6d\x3b\x64\x4d\x0d\x7b\x49\xb9\x10\x38\x0c\x0f\x4e\xc9\xe2\x3c\xb7\x12\x88\x2c\xf4\x3a\x89\x6d\x12\xc7\x04\x53\xfe\x77\xc7\xfb\x77\x38",
	.ilen	= 128,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xee\x65\x78\x30\x01\xc2\x56\x91\xfa\x28\xd0\xf5\xf1\xc1\xd7\x62"
}, { /* wycheproof - misc */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x01\xee\x32\x00",
	.nlen	= 8,
	.assoc	= "\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80",
	.alen	= 16,
	.input	= "\x25\x6d\x40\x08\x80\x94\x17\x03\x55\xd3\x04\x04\x64\x43\xfe\x68\xdf\x99\x47\x83\x03\xfb\x3b\xfb\x80\xe0\x30\x3e\xeb\xd3\x29\x3e",
	.ilen	= 32,
	.result	= "\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x79\xba\x7a\x29\xf5\xa7\xbb\x75\x79\x7a\xf8\x7a\x61\x01\x29\xa4"
}, { /* wycheproof - misc */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x01\xee\x32\x00",
	.nlen	= 8,
	.assoc	= "\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80",
	.alen	= 16,
	.input	= "\x25\x6d\x40\x08\x80\x94\x17\x03\x55\xd3\x04\x04\x64\x43\xfe\x68\xdf\x99\x47\x83\x03\xfb\x3b\xfb\x80\xe0\x30\x3e\xeb\xd3\x29\x3e\xe3\xbc\xdb\xdb\x1e\xde\xfc\x7e\x8b\xcd\xa1\x36\xa1\x5c\x8c\xab\x08\x69\xff\x52\xec\x5e\x26\x65\x53\xb7\xb2\xa7\xfe\x87\xfd\x3d",
	.ilen	= 64,
	.result	= "\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x36\xb1\x74\x38\x19\xe1\xb9\xba\x15\x51\xe8\xed\x92\x2a\x95\x9a"
}, { /* wycheproof - misc */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x01\xee\x32\x00",
	.nlen	= 8,
	.assoc	= "\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80",
	.alen	= 16,
	.input	= "\x25\x6d\x40\x08\x80\x94\x17\x03\x55\xd3\x04\x04\x64\x43\xfe\x68\xdf\x99\x47\x83\x03\xfb\x3b\xfb\x80\xe0\x30\x3e\xeb\xd3\x29\x3e\xe3\xbc\xdb\xdb\x1e\xde\xfc\x7e\x8b\xcd\xa1\x36\xa1\x5c\x8c\xab\x08\x69\xff\x52\xec\x5e\x26\x65\x53\xb7\xb2\xa7\xfe\x87\xfd\x3d\x7a\xda\x44\xc2\x42\x69\xbf\x7a\x55\x27\xf2\xf0\xac\xf6\x85\x82\xb7\x4c\x5a\x62\xe6\x0c\x05\x00\x98\x1a\x49\xb8\x45\x93\x92\x44\x9b\xb2\xf2\x04\xb6\x46\xef\x47\xf3\xf0\xb1\xb6\x1d\xc3\x48\x6d\x77\xd3\x0b\x45\x76\x92\xed\xb8\xfb\xac\x01\x08\x38\x04\x88\x47",
	.ilen	= 128,
	.result	= "\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\xfe\xac\x49\x55\x55\x4e\x80\x6f\x3a\x19\x02\xe2\x44\x32\xc0\x8a"
}, { /* wycheproof - misc */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x01\xee\x32\x00",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f",
	.alen	= 16,
	.input	= "\xda\x92\xbf\xf7\x7f\x6b\xe8\xfc\xaa\x2c\xfb\xfb\x9b\xbc\x01\x97\x20\x66\xb8\x7c\xfc\x04\xc4\x04\x7f\x1f\xcf\xc1\x14\x2c\xd6\xc1",
	.ilen	= 32,
	.result	= "\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\x20\xa3\x79\x8d\xf1\x29\x2c\x59\x72\xbf\x97\x41\xae\xc3\x8a\x19"
}, { /* wycheproof - misc */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x01\xee\x32\x00",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f",
	.alen	= 16,
	.input	= "\xda\x92\xbf\xf7\x7f\x6b\xe8\xfc\xaa\x2c\xfb\xfb\x9b\xbc\x01\x97\x20\x66\xb8\x7c\xfc\x04\xc4\x04\x7f\x1f\xcf\xc1\x14\x2c\xd6\xc1\x1c\x43\x24\x24\xe1\x21\x03\x81\x74\x32\x5e\xc9\x5e\xa3\x73\x54\xf7\x96\x00\xad\x13\xa1\xd9\x9a\xac\x48\x4d\x58\x01\x78\x02\xc2",
	.ilen	= 64,
	.result	= "\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xc0\x3d\x9f\x67\x35\x4a\x97\xb2\xf0\x74\xf7\x55\x15\x57\xe4\x9c"
}, { /* wycheproof - misc */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x01\xee\x32\x00",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f",
	.alen	= 16,
	.input	= "\xda\x92\xbf\xf7\x7f\x6b\xe8\xfc\xaa\x2c\xfb\xfb\x9b\xbc\x01\x97\x20\x66\xb8\x7c\xfc\x04\xc4\x04\x7f\x1f\xcf\xc1\x14\x2c\xd6\xc1\x1c\x43\x24\x24\xe1\x21\x03\x81\x74\x32\x5e\xc9\x5e\xa3\x73\x54\xf7\x96\x00\xad\x13\xa1\xd9\x9a\xac\x48\x4d\x58\x01\x78\x02\xc2\x85\x25\xbb\x3d\xbd\x96\x40\x85\xaa\xd8\x0d\x0f\x53\x09\x7a\x7d\x48\xb3\xa5\x9d\x19\xf3\xfa\xff\x67\xe5\xb6\x47\xba\x6c\x6d\xbb\x64\x4d\x0d\xfb\x49\xb9\x10\xb8\x0c\x0f\x4e\x49\xe2\x3c\xb7\x92\x88\x2c\xf4\xba\x89\x6d\x12\x47\x04\x53\xfe\xf7\xc7\xfb\x77\xb8",
	.ilen	= 128,
	.result	= "\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xc8\x6d\xa8\xdd\x65\x22\x86\xd5\x02\x13\xd3\x28\xd6\x3e\x40\x06"
}, { /* wycheproof - misc */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x01\xee\x32\x00",
	.nlen	= 8,
	.assoc	= "\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff",
	.alen	= 16,
	.input	= "\x5a\x92\xbf\x77\xff\x6b\xe8\x7c\x2a\x2c\xfb\x7b\x1b\xbc\x01\x17\xa0\x66\xb8\xfc\x7c\x04\xc4\x84\xff\x1f\xcf\x41\x94\x2c\xd6\x41",
	.ilen	= 32,
	.result	= "\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\xbe\xde\x90\x83\xce\xb3\x6d\xdf\xe5\xfa\x81\x1f\x95\x47\x1c\x67"
}, { /* wycheproof - misc */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x01\xee\x32\x00",
	.nlen	= 8,
	.assoc	= "\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff",
	.alen	= 16,
	.input	= "\x5a\x92\xbf\x77\xff\x6b\xe8\x7c\x2a\x2c\xfb\x7b\x1b\xbc\x01\x17\xa0\x66\xb8\xfc\x7c\x04\xc4\x84\xff\x1f\xcf\x41\x94\x2c\xd6\x41\x9c\x43\x24\xa4\x61\x21\x03\x01\xf4\x32\x5e\x49\xde\xa3\x73\xd4\x77\x96\x00\x2d\x93\xa1\xd9\x1a\x2c\x48\x4d\xd8\x81\x78\x02\x42",
	.ilen	= 64,
	.result	= "\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x30\x08\x74\xbb\x06\x92\xb6\x89\xde\xad\x9a\xe1\x5b\x06\x73\x90"
}, { /* wycheproof - misc */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x01\xee\x32\x00",
	.nlen	= 8,
	.assoc	= "\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff",
	.alen	= 16,
	.input	= "\x5a\x92\xbf\x77\xff\x6b\xe8\x7c\x2a\x2c\xfb\x7b\x1b\xbc\x01\x17\xa0\x66\xb8\xfc\x7c\x04\xc4\x84\xff\x1f\xcf\x41\x94\x2c\xd6\x41\x9c\x43\x24\xa4\x61\x21\x03\x01\xf4\x32\x5e\x49\xde\xa3\x73\xd4\x77\x96\x00\x2d\x93\xa1\xd9\x1a\x2c\x48\x4d\xd8\x81\x78\x02\x42\x05\x25\xbb\xbd\x3d\x96\x40\x05\x2a\xd8\x0d\x8f\xd3\x09\x7a\xfd\xc8\xb3\xa5\x1d\x99\xf3\xfa\x7f\xe7\xe5\xb6\xc7\x3a\x6c\x6d\x3b\xe4\x4d\x0d\x7b\xc9\xb9\x10\x38\x8c\x0f\x4e\xc9\x62\x3c\xb7\x12\x08\x2c\xf4\x3a\x09\x6d\x12\xc7\x84\x53\xfe\x77\x47\xfb\x77\x38",
	.ilen	= 128,
	.result	= "\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x99\xca\xd8\x5f\x45\xca\x40\x94\x2d\x0d\x4d\x5e\x95\x0a\xde\x22"
}, { /* wycheproof - misc */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x01\xee\x32\x00",
	.nlen	= 8,
	.assoc	= "\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff",
	.alen	= 16,
	.input	= "\x25\x6d\x40\x88\x7f\x6b\xe8\x7c\x55\xd3\x04\x84\x9b\xbc\x01\x17\xdf\x99\x47\x03\xfc\x04\xc4\x84\x80\xe0\x30\xbe\x14\x2c\xd6\x41",
	.ilen	= 32,
	.result	= "\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x8b\xbe\x14\x52\x72\xe7\xc2\xd9\xa1\x89\x1a\x3a\xb0\x98\x3d\x9d"
}, { /* wycheproof - misc */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x01\xee\x32\x00",
	.nlen	= 8,
	.assoc	= "\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff",
	.alen	= 16,
	.input	= "\x25\x6d\x40\x88\x7f\x6b\xe8\x7c\x55\xd3\x04\x84\x9b\xbc\x01\x17\xdf\x99\x47\x03\xfc\x04\xc4\x84\x80\xe0\x30\xbe\x14\x2c\xd6\x41\xe3\xbc\xdb\x5b\xe1\x21\x03\x01\x8b\xcd\xa1\xb6\x5e\xa3\x73\xd4\x08\x69\xff\xd2\x13\xa1\xd9\x1a\x53\xb7\xb2\x27\x01\x78\x02\x42",
	.ilen	= 64,
	.result	= "\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x3b\x41\x86\x19\x13\xa8\xf6\xde\x7f\x61\xe2\x25\x63\x1b\xc3\x82"
}, { /* wycheproof - misc */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x01\xee\x32\x00",
	.nlen	= 8,
	.assoc	= "\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff",
	.alen	= 16,
	.input	= "\x25\x6d\x40\x88\x7f\x6b\xe8\x7c\x55\xd3\x04\x84\x9b\xbc\x01\x17\xdf\x99\x47\x03\xfc\x04\xc4\x84\x80\xe0\x30\xbe\x14\x2c\xd6\x41\xe3\xbc\xdb\x5b\xe1\x21\x03\x01\x8b\xcd\xa1\xb6\x5e\xa3\x73\xd4\x08\x69\xff\xd2\x13\xa1\xd9\x1a\x53\xb7\xb2\x27\x01\x78\x02\x42\x7a\xda\x44\x42\xbd\x96\x40\x05\x55\x27\xf2\x70\x53\x09\x7a\xfd\xb7\x4c\x5a\xe2\x19\xf3\xfa\x7f\x98\x1a\x49\x38\xba\x6c\x6d\x3b\x9b\xb2\xf2\x84\x49\xb9\x10\x38\xf3\xf0\xb1\x36\xe2\x3c\xb7\x12\x77\xd3\x0b\xc5\x89\x6d\x12\xc7\xfb\xac\x01\x88\xc7\xfb\x77\x38",
	.ilen	= 128,
	.result	= "\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x84\x28\xbc\xf0\x23\xec\x6b\xf3\x1f\xd9\xef\xb2\x03\xff\x08\x71"
}, { /* wycheproof - misc */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x01\xee\x32\x00",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00",
	.alen	= 16,
	.input	= "\xda\x92\xbf\x77\x80\x94\x17\x83\xaa\x2c\xfb\x7b\x64\x43\xfe\xe8\x20\x66\xb8\xfc\x03\xfb\x3b\x7b\x7f\x1f\xcf\x41\xeb\xd3\x29\xbe",
	.ilen	= 32,
	.result	= "\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\x13\x9f\xdf\x64\x74\xea\x24\xf5\x49\xb0\x75\x82\x5f\x2c\x76\x20"
}, { /* wycheproof - misc */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x01\xee\x32\x00",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00",
	.alen	= 16,
	.input	= "\xda\x92\xbf\x77\x80\x94\x17\x83\xaa\x2c\xfb\x7b\x64\x43\xfe\xe8\x20\x66\xb8\xfc\x03\xfb\x3b\x7b\x7f\x1f\xcf\x41\xeb\xd3\x29\xbe\x1c\x43\x24\xa4\x1e\xde\xfc\xfe\x74\x32\x5e\x49\xa1\x5c\x8c\x2b\xf7\x96\x00\x2d\xec\x5e\x26\xe5\xac\x48\x4d\xd8\xfe\x87\xfd\xbd",
	.ilen	= 64,
	.result	= "\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xbb\xad\x8d\x86\x3b\x83\x5a\x8e\x86\x64\xfd\x1d\x45\x66\xb6\xb4"
}, { /* wycheproof - misc */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x01\xee\x32\x00",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00",
	.alen	= 16,
	.input	= "\xda\x92\xbf\x77\x80\x94\x17\x83\xaa\x2c\xfb\x7b\x64\x43\xfe\xe8\x20\x66\xb8\xfc\x03\xfb\x3b\x7b\x7f\x1f\xcf\x41\xeb\xd3\x29\xbe\x1c\x43\x24\xa4\x1e\xde\xfc\xfe\x74\x32\x5e\x49\xa1\x5c\x8c\x2b\xf7\x96\x00\x2d\xec\x5e\x26\xe5\xac\x48\x4d\xd8\xfe\x87\xfd\xbd\x85\x25\xbb\xbd\x42\x69\xbf\xfa\xaa\xd8\x0d\x8f\xac\xf6\x85\x02\x48\xb3\xa5\x1d\xe6\x0c\x05\x80\x67\xe5\xb6\xc7\x45\x93\x92\xc4\x64\x4d\x0d\x7b\xb6\x46\xef\xc7\x0c\x0f\x4e\xc9\x1d\xc3\x48\xed\x88\x2c\xf4\x3a\x76\x92\xed\x38\x04\x53\xfe\x77\x38\x04\x88\xc7",
	.ilen	= 128,
	.result	= "\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\x42\xf2\x35\x42\x97\x84\x9a\x51\x1d\x53\xe5\x57\x17\x72\xf7\x1f"
}, { /* wycheproof - checking for int overflows */
	.key	= "\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30",
	.nonce	= "\x30\x30\x30\x30\x30\x30\x30\x30\x00\x02\x50\x6e",
	.nlen	= 12,
	.assoc	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff",
	.alen	= 64,
	.input	= "\xd4\x50\x0b\xf0\x09\x49\x35\x51\xc3\x80\xad\xf5\x2c\x57\x3a\x69\xdf\x7e\x8b\x76\x24\x63\x33\x0f\xac\xc1\x6a\x57\x26\xbe\x71\x90\xc6\x3c\x5a\x1c\x92\x65\x84\xa0\x96\x75\x68\x28\xdc\xdc\x64\xac\xdf\x96\x3d\x93\x1b\xf1\xda\xe2\x38\xf3\xf1\x57\x22\x4a\xc4\xb5\x42\xd7\x85\xb0\xdd\x84\xdb\x6b\xe3\xbc\x5a\x36\x63\xe8\x41\x49\xff\xbe\xd0\x9e\x54\xf7\x8f\x16\xa8\x22\x3b\x24\xcb\x01\x9f\x58\xb2\x1b\x0e\x55\x1e\x7a\xa0\x73\x27\x62\x95\x51\x37\x6c\xcb\xc3\x93\x76\x71\xa0\x62\x9b\xd9\x5c\x99\x15\xc7\x85\x55\x77\x1e\x7a",
	.ilen	= 128,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x0b\x30\x0d\x8d\xa5\x6c\x21\x85\x75\x52\x79\x55\x3c\x4c\x82\xca"
}, { /* wycheproof - checking for int overflows */
	.key	= "\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30",
	.nonce	= "\x30\x30\x30\x30\x30\x30\x30\x30\x00\x03\x18\xa5",
	.nlen	= 12,
	.assoc	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff",
	.alen	= 64,
	.input	= "\x7d\xe8\x7f\x67\x29\x94\x52\x75\xd0\x65\x5d\xa4\xc7\xfd\xe4\x56\x9e\x16\xf1\x11\xb5\xeb\x26\xc2\x2d\x85\x9e\x3f\xf8\x22\xec\xed\x3a\x6d\xd9\xa6\x0f\x22\x95\x7f\x7b\x7c\x85\x7e\x88\x22\xeb\x9f\xe0\xb8\xd7\x02\x21\x41\xf2\xd0\xb4\x8f\x4b\x56\x12\xd3\x22\xa8\x8d\xd0\xfe\x0b\x4d\x91\x79\x32\x4f\x7c\x6c\x9e\x99\x0e\xfb\xd8\x0e\x5e\xd6\x77\x58\x26\x49\x8b\x1e\xfe\x0f\x71\xa0\xf3\xec\x5b\x29\xcb\x28\xc2\x54\x0a\x7d\xcd\x51\xb7\xda\xae\xe0\xff\x4a\x7f\x3a\xc1\xee\x54\xc2\x9e\xe4\xc1\x70\xde\x40\x8f\x66\x69\x21\x94",
	.ilen	= 128,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xc5\x78\xe2\xaa\x44\xd3\x09\xb7\xb6\xa5\x19\x3b\xdc\x61\x18\xf5"
}, { /* wycheproof - checking for int overflows */
	.key	= "\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30",
	.nonce	= "\x00\x00\x00\x00\x00\x07\xb4\xf0",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff",
	.alen	= 64,
	.input	= "\x1b\x99\x6f\x9a\x3c\xcc\x67\x85\xde\x22\xff\x5b\x8a\xdd\x95\x02\xce\x03\xa0\xfa\xf5\x99\x2a\x09\x52\x2c\xdd\x12\x06\xd2\x20\xb8\xf8\xbd\x07\xd1\xf1\xf5\xa1\xbd\x9a\x71\xd1\x1c\x7f\x57\x9b\x85\x58\x18\xc0\x8d\x4d\xe0\x36\x39\x31\x83\xb7\xf5\x90\xb3\x35\xae\xd8\xde\x5b\x57\xb1\x3c\x5f\xed\xe2\x44\x1c\x3e\x18\x4a\xa9\xd4\x6e\x61\x59\x85\x06\xb3\xe1\x1c\x43\xc6\x2c\xbc\xac\xec\xed\x33\x19\x08\x75\xb0\x12\x21\x8b\x19\x30\xfb\x7c\x38\xec\x45\xac\x11\xc3\x53\xd0\xcf\x93\x8d\xcc\xb9\xef\xad\x8f\xed\xbe\x46\xda\xa5",
	.ilen	= 128,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x4b\x0b\xda\x8a\xd0\x43\x83\x0d\x83\x19\xab\x82\xc5\x0c\x76\x63"
}, { /* wycheproof - checking for int overflows */
	.key	= "\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30",
	.nonce	= "\x00\x00\x00\x00\x00\x20\xfb\x66",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff",
	.alen	= 64,
	.input	= "\x86\xcb\xac\xae\x4d\x3f\x74\xae\x01\x21\x3e\x05\x51\xcc\x15\x16\x0e\xa1\xbe\x84\x08\xe3\xd5\xd7\x4f\x01\x46\x49\x95\xa6\x9e\x61\x76\xcb\x9e\x02\xb2\x24\x7e\xd2\x99\x89\x2f\x91\x82\xa4\x5c\xaf\x4c\x69\x40\x56\x11\x76\x6e\xdf\xaf\xdc\x28\x55\x19\xea\x30\x48\x0c\x44\xf0\x5e\x78\x1e\xac\xf8\xfc\xec\xc7\x09\x0a\xbb\x28\xfa\x5f\xd5\x85\xac\x8c\xda\x7e\x87\x72\xe5\x94\xe4\xce\x6c\x88\x32\x81\x93\x2e\x0f\x89\xf8\x77\xa1\xf0\x4d\x9c\x32\xb0\x6c\xf9\x0b\x0e\x76\x2b\x43\x0c\x4d\x51\x7c\x97\x10\x70\x68\xf4\x98\xef\x7f",
	.ilen	= 128,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x4b\xc9\x8f\x72\xc4\x94\xc2\xa4\x3c\x2b\x15\xa1\x04\x3f\x1c\xfa"
}, { /* wycheproof - checking for int overflows */
	.key	= "\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30",
	.nonce	= "\x00\x00\x00\x00\x00\x38\xbb\x90",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff",
	.alen	= 64,
	.input	= "\xfa\xb1\xcd\xdf\x4f\xe1\x98\xef\x63\xad\xd8\x81\xd6\xea\xd6\xc5\x76\x37\xbb\xe9\x20\x18\xca\x7c\x0b\x96\xfb\xa0\x87\x1e\x93\x2d\xb1\xfb\xf9\x07\x61\xbe\x25\xdf\x8d\xfa\xf9\x31\xce\x57\x57\xe6\x17\xb3\xd7\xa9\xf0\xbf\x0f\xfe\x5d\x59\x1a\x33\xc1\x43\xb8\xf5\x3f\xd0\xb5\xa1\x96\x09\xfd\x62\xe5\xc2\x51\xa4\x28\x1a\x20\x0c\xfd\xc3\x4f\x28\x17\x10\x40\x6f\x4e\x37\x62\x54\x46\xff\x6e\xf2\x24\x91\x3d\xeb\x0d\x89\xaf\x33\x71\x28\xe3\xd1\x55\xd1\x6d\x3e\xc3\x24\x60\x41\x43\x21\x43\xe9\xab\x3a\x6d\x2c\xcc\x2f\x4d\x62",
	.ilen	= 128,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xf7\xe9\xe1\x51\xb0\x25\x33\xc7\x46\x58\xbf\xc7\x73\x7c\x68\x0d"
}, { /* wycheproof - checking for int overflows */
	.key	= "\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30",
	.nonce	= "\x00\x00\x00\x00\x00\x70\x48\x4a",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff",
	.alen	= 64,
	.input	= "\x22\x72\x02\xbe\x7f\x35\x15\xe9\xd1\xc0\x2e\xea\x2f\x19\x50\xb6\x48\x1b\x04\x8a\x4c\x91\x50\x6c\xb4\x0d\x50\x4e\x6c\x94\x9f\x82\xd1\x97\xc2\x5a\xd1\x7d\xc7\x21\x65\x11\x25\x78\x2a\xc7\xa7\x12\x47\xfe\xae\xf3\x2f\x1f\x25\x0c\xe4\xbb\x8f\x79\xac\xaa\x17\x9d\x45\xa7\xb0\x54\x5f\x09\x24\x32\x5e\xfa\x87\xd5\xe4\x41\xd2\x84\x78\xc6\x1f\x22\x23\xee\x67\xc3\xb4\x1f\x43\x94\x53\x5e\x2a\x24\x36\x9a\x2e\x16\x61\x3c\x45\x94\x90\xc1\x4f\xb1\xd7\x55\xfe\x53\xfb\xe1\xee\x45\xb1\xb2\x1f\x71\x62\xe2\xfc\xaa\x74\x2a\xbe\xfd",
	.ilen	= 128,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x79\x5b\xcf\xf6\x47\xc5\x53\xc2\xe4\xeb\x6e\x0e\xaf\xd9\xe0\x4e"
}, { /* wycheproof - checking for int overflows */
	.key	= "\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30",
	.nonce	= "\x00\x00\x00\x00\x00\x93\x2f\x40",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff",
	.alen	= 64,
	.input	= "\xfa\xe5\x83\x45\xc1\x6c\xb0\xf5\xcc\x53\x7f\x2b\x1b\x34\x69\xc9\x69\x46\x3b\x3e\xa7\x1b\xcf\x6b\x98\xd6\x69\xa8\xe6\x0e\x04\xfc\x08\xd5\xfd\x06\x9c\x36\x26\x38\xe3\x40\x0e\xf4\xcb\x24\x2e\x27\xe2\x24\x5e\x68\xcb\x9e\xc5\x83\xda\x53\x40\xb1\x2e\xdf\x42\x3b\x73\x26\xad\x20\xfe\xeb\x57\xda\xca\x2e\x04\x67\xa3\x28\x99\xb4\x2d\xf8\xe5\x6d\x84\xe0\x06\xbc\x8a\x7a\xcc\x73\x1e\x7c\x1f\x6b\xec\xb5\x71\x9f\x70\x77\xf0\xd4\xf4\xc6\x1a\xb1\x1e\xba\xc1\x00\x18\x01\xce\x33\xc4\xe4\xa7\x7d\x83\x1d\x3c\xe3\x4e\x84\x10\xe1",
	.ilen	= 128,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x19\x46\xd6\x53\x96\x0f\x94\x7a\x74\xd3\xe8\x09\x3c\xf4\x85\x02"
}, { /* wycheproof - checking for int overflows */
	.key	= "\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30\x30",
	.nonce	= "\x00\x00\x00\x00\x00\xe2\x93\x35",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff",
	.alen	= 64,
	.input	= "\xeb\xb2\x16\xdd\xd7\xca\x70\x92\x15\xf5\x03\xdf\x9c\xe6\x3c\x5c\xd2\x19\x4e\x7d\x90\x99\xe8\xa9\x0b\x2a\xfa\xad\x5e\xba\x35\x06\x99\x25\xa6\x03\xfd\xbc\x34\x1a\xae\xd4\x15\x05\xb1\x09\x41\xfa\x38\x56\xa7\xe2\x47\xb1\x04\x07\x09\x74\x6c\xfc\x20\x96\xca\xa6\x31\xb2\xff\xf4\x1c\x25\x05\x06\xd8\x89\xc1\xc9\x06\x71\xad\xe8\x53\xee\x63\x94\xc1\x91\x92\xa5\xcf\x37\x10\xd1\x07\x30\x99\xe5\xbc\x94\x65\x82\xfc\x0f\xab\x9f\x54\x3c\x71\x6a\xe2\x48\x6a\x86\x83\xfd\xca\x39\xd2\xe1\x4f\x23\xd0\x0a\x58\x26\x64\xf4\xec\xb1",
	.ilen	= 128,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x36\xc3\x00\x29\x85\xdd\x21\xba\xf8\x95\xd6\x33\x57\x3f\x12\xc0"
}, { /* wycheproof - checking for int overflows */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x00\x0e\xf7\xd5",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff",
	.alen	= 64,
	.input	= "\x40\x8a\xe6\xef\x1c\x7e\xf0\xfb\x2c\x2d\x61\x08\x16\xfc\x78\x49\xef\xa5\x8f\x78\x27\x3f\x5f\x16\x6e\xa6\x5f\x81\xb5\x75\x74\x7d\x03\x5b\x30\x40\xfe\xde\x1e\xb9\x45\x97\x88\x66\x97\x88\x40\x8e\x00\x41\x3b\x3e\x37\x6d\x15\x2d\x20\x4a\xa2\xb7\xa8\x35\x58\xfc\xd4\x8a\x0e\xf7\xa2\x6b\x1c\xd6\xd3\x5d\x23\xb3\xf5\xdf\xe0\xca\x77\xa4\xce\x32\xb9\x4a\xbf\x83\xda\x2a\xef\xca\xf0\x68\x38\x08\x79\xe8\x9f\xb0\xa3\x82\x95\x95\xcf\x44\xc3\x85\x2a\xe2\xcc\x66\x2b\x68\x9f\x93\x55\xd9\xc1\x83\x80\x1f\x6a\xcc\x31\x3f\x89\x07",
	.ilen	= 128,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x65\x14\x51\x8e\x0a\x26\x41\x42\xe0\xb7\x35\x1f\x96\x7f\xc2\xae"
}, { /* wycheproof - checking for int overflows */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x00\x3d\xfc\xe4",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff",
	.alen	= 64,
	.input	= "\x0a\x0a\x24\x49\x9b\xca\xde\x58\xcf\x15\x76\xc3\x12\xac\xa9\x84\x71\x8c\xb4\xcc\x7e\x01\x53\xf5\xa9\x01\x58\x10\x85\x96\x44\xdf\xc0\x21\x17\x4e\x0b\x06\x0a\x39\x74\x48\xde\x8b\x48\x4a\x86\x03\xbe\x68\x0a\x69\x34\xc0\x90\x6f\x30\xdd\x17\xea\xe2\xd4\xc5\xfa\xa7\x77\xf8\xca\x53\x37\x0e\x08\x33\x1b\x88\xc3\x42\xba\xc9\x59\x78\x7b\xbb\x33\x93\x0e\x3b\x56\xbe\x86\xda\x7f\x2a\x6e\xb1\xf9\x40\x89\xd1\xd1\x81\x07\x4d\x43\x02\xf8\xe0\x55\x2d\x0d\xe1\xfa\xb3\x06\xa2\x1b\x42\xd4\xc3\xba\x6e\x6f\x0c\xbc\xc8\x1e\x87\x7a",
	.ilen	= 128,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x4c\x19\x4d\xa6\xa9\x9f\xd6\x5b\x40\xe9\xca\xd7\x98\xf4\x4b\x19"
}, { /* wycheproof - checking for int overflows */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x01\x84\x86\xa8",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff",
	.alen	= 64,
	.input	= "\x4a\x0a\xaf\xf8\x49\x47\x29\x18\x86\x91\x70\x13\x40\xf3\xce\x2b\x8a\x78\xee\xd3\xa0\xf0\x65\x99\x4b\x72\x48\x4e\x79\x91\xd2\x5c\x29\xaa\x07\x5e\xb1\xfc\x16\xde\x93\xfe\x06\x90\x58\x11\x2a\xb2\x84\xa3\xed\x18\x78\x03\x26\xd1\x25\x8a\x47\x22\x2f\xa6\x33\xd8\xb2\x9f\x3b\xd9\x15\x0b\x23\x9b\x15\x46\xc2\xbb\x9b\x9f\x41\x0f\xeb\xea\xd3\x96\x00\x0e\xe4\x77\x70\x15\x32\xc3\xd0\xf5\xfb\xf8\x95\xd2\x80\x19\x6d\x2f\x73\x7c\x5e\x9f\xec\x50\xd9\x2b\xb0\xdf\x5d\x7e\x51\x3b\xe5\xb8\xea\x97\x13\x10\xd5\xbf\x16\xba\x7a\xee",
	.ilen	= 128,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xc8\xae\x77\x88\xcd\x28\x74\xab\xc1\x38\x54\x1e\x11\xfd\x05\x87"
}, { /* wycheproof - checking for int overflows */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff",
	.alen	= 64,
	.input	= "\xff\x94\x28\xd0\x79\x35\x1f\x66\x5c\xd0\x01\x35\x43\x19\x87\x5c\x78\x3d\x35\xf6\x13\xe6\xd9\x09\x3d\x38\xe9\x75\xc3\x8f\xe3\xb8\x9f\x7a\xed\x35\xcb\x5a\x2f\xca\xa0\x34\x6e\xfb\x93\x65\x54\x64\x9c\xf6\x37\x81\x71\xea\xe4\x39\x6e\xa1\x5d\xc2\x40\xd1\xab\xf4\x47\x2d\x90\x96\x52\x4f\xa1\xb2\xb0\x23\xb8\xb2\x88\x22\x27\x73\xd4\xd2\x06\x61\x6f\x92\x93\xf6\x5b\x45\xdb\xbc\x74\xe7\xc2\xed\xfb\xcb\xbf\x1c\xfb\x67\x9b\xb7\x39\xa5\x86\x2d\xe2\xbc\xb9\x37\xf7\x4d\x5b\xf8\x67\x1c\x5a\x8a\x50\x92\xf6\x1d\x54\xc9\xaa\x5b",
	.ilen	= 128,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x93\x3a\x51\x63\xc7\xf6\x23\x68\x32\x7b\x3f\xbc\x10\x36\xc9\x43"
}, { /* wycheproof - special case tag */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b",
	.nlen	= 12,
	.assoc	= "\x85\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xa6\x90\x2f\xcb\xc8\x83\xbb\xc1\x80\xb2\x56\xae\x34\xad\x7f\x00",
	.alen	= 32,
	.input	= "\x9a\x49\xc4\x0f\x8b\x48\xd7\xc6\x6d\x1d\xb4\xe5\x3f\x20\xf2\xdd\x4a\xaa\x24\x1d\xda\xb2\x6b\x5b\xc0\xe2\x18\xb7\x2c\x33\x90\xf2\xdf\x3e\xbd\x01\x76\x70\x44\x19\x97\x2b\xcd\xbc\x6b\xbc\xb3\xe4\xe7\x4a\x71\x52\x8e\xf5\x12\x63\xce\x24\xe0\xd5\x75\xe0\xe4\x4d",
	.ilen	= 64,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
}, { /* wycheproof - special case tag */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b",
	.nlen	= 12,
	.assoc	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x24\x7e\x50\x64\x2a\x1c\x0a\x2f\x8f\x77\x21\x96\x09\xdb\xa9\x58",
	.alen	= 32,
	.input	= "\x9a\x49\xc4\x0f\x8b\x48\xd7\xc6\x6d\x1d\xb4\xe5\x3f\x20\xf2\xdd\x4a\xaa\x24\x1d\xda\xb2\x6b\x5b\xc0\xe2\x18\xb7\x2c\x33\x90\xf2\xdf\x3e\xbd\x01\x76\x70\x44\x19\x97\x2b\xcd\xbc\x6b\xbc\xb3\xe4\xe7\x4a\x71\x52\x8e\xf5\x12\x63\xce\x24\xe0\xd5\x75\xe0\xe4\x4d",
	.ilen	= 64,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
}, { /* wycheproof - special case tag */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b",
	.nlen	= 12,
	.assoc	= "\x7c\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xd9\xe7\x2c\x06\x4a\xc8\x96\x1f\x3f\xa5\x85\xe0\xe2\xab\xd6\x00",
	.alen	= 32,
	.input	= "\x9a\x49\xc4\x0f\x8b\x48\xd7\xc6\x6d\x1d\xb4\xe5\x3f\x20\xf2\xdd\x4a\xaa\x24\x1d\xda\xb2\x6b\x5b\xc0\xe2\x18\xb7\x2c\x33\x90\xf2\xdf\x3e\xbd\x01\x76\x70\x44\x19\x97\x2b\xcd\xbc\x6b\xbc\xb3\xe4\xe7\x4a\x71\x52\x8e\xf5\x12\x63\xce\x24\xe0\xd5\x75\xe0\xe4\x4d",
	.ilen	= 64,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
}, { /* wycheproof - special case tag */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b",
	.nlen	= 12,
	.assoc	= "\x65\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x95\xaf\x0f\x4d\x0b\x68\x6e\xae\xcc\xca\x43\x07\xd5\x96\xf5\x02",
	.alen	= 32,
	.input	= "\x9a\x49\xc4\x0f\x8b\x48\xd7\xc6\x6d\x1d\xb4\xe5\x3f\x20\xf2\xdd\x4a\xaa\x24\x1d\xda\xb2\x6b\x5b\xc0\xe2\x18\xb7\x2c\x33\x90\xf2\xdf\x3e\xbd\x01\x76\x70\x44\x19\x97\x2b\xcd\xbc\x6b\xbc\xb3\xe4\xe7\x4a\x71\x52\x8e\xf5\x12\x63\xce\x24\xe0\xd5\x75\xe0\xe4\x4d",
	.ilen	= 64,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80\x00\x00\x00\x80"
}, { /* wycheproof - special case tag */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b",
	.nlen	= 12,
	.assoc	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x85\x40\xb4\x64\x35\x77\x07\xbe\x3a\x39\xd5\x5c\x34\xf8\xbc\xb3",
	.alen	= 32,
	.input	= "\x9a\x49\xc4\x0f\x8b\x48\xd7\xc6\x6d\x1d\xb4\xe5\x3f\x20\xf2\xdd\x4a\xaa\x24\x1d\xda\xb2\x6b\x5b\xc0\xe2\x18\xb7\x2c\x33\x90\xf2\xdf\x3e\xbd\x01\x76\x70\x44\x19\x97\x2b\xcd\xbc\x6b\xbc\xb3\xe4\xe7\x4a\x71\x52\x8e\xf5\x12\x63\xce\x24\xe0\xd5\x75\xe0\xe4\x4d",
	.ilen	= 64,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f\xff\xff\xff\x7f"
}, { /* wycheproof - special case tag */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b",
	.nlen	= 12,
	.assoc	= "\x4f\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x66\x23\xd9\x90\xb8\x98\xd8\x30\xd2\x12\xaf\x23\x83\x33\x07\x01",
	.alen	= 32,
	.input	= "\x9a\x49\xc4\x0f\x8b\x48\xd7\xc6\x6d\x1d\xb4\xe5\x3f\x20\xf2\xdd\x4a\xaa\x24\x1d\xda\xb2\x6b\x5b\xc0\xe2\x18\xb7\x2c\x33\x90\xf2\xdf\x3e\xbd\x01\x76\x70\x44\x19\x97\x2b\xcd\xbc\x6b\xbc\xb3\xe4\xe7\x4a\x71\x52\x8e\xf5\x12\x63\xce\x24\xe0\xd5\x75\xe0\xe4\x4d",
	.ilen	= 64,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x01\x00\x00\x00\x01\x00\x00\x00\x01\x00\x00\x00\x01\x00\x00\x00"
}, { /* wycheproof - special case tag */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b",
	.nlen	= 12,
	.assoc	= "\x83\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x5f\x16\xd0\x9f\x17\x78\x72\x11\xb7\xd4\x84\xe0\x24\xf8\x97\x01",
	.alen	= 32,
	.input	= "\x9a\x49\xc4\x0f\x8b\x48\xd7\xc6\x6d\x1d\xb4\xe5\x3f\x20\xf2\xdd\x4a\xaa\x24\x1d\xda\xb2\x6b\x5b\xc0\xe2\x18\xb7\x2c\x33\x90\xf2\xdf\x3e\xbd\x01\x76\x70\x44\x19\x97\x2b\xcd\xbc\x6b\xbc\xb3\xe4\xe7\x4a\x71\x52\x8e\xf5\x12\x63\xce\x24\xe0\xd5\x75\xe0\xe4\x4d",
	.ilen	= 64,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
}, { /* wycheproof - edge case intermediate sums in poly1305 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff",
	.alen	= 4,
	.input	= "\x00\x52\x35\xd2\xa9\x19\xf2\x8d\x3d\xb7\x66\x4a\x34\xae\x6b\x44\x4d\x3d\x35\xf6\x13\xe6\xd9\x09\x3d\x38\xe9\x75\xc3\x8f\xe3\xb8\x5b\x8b\x94\x50\x9e\x2b\x74\xa3\x6d\x34\x6e\x33\xd5\x72\x65\x9b\xa9\xf6\x37\x81\x71\xea\xe4\x39\x6e\xa1\x5d\xc2\x40\xd1\xab\xf4\x83\xdc\xe9\xf3\x07\x3e\xfa\xdb\x7d\x23\xb8\x7a\xce\x35\x16\x8c",
	.ilen	= 80,
	.result	= "\x00\x39\xe2\xfd\x2f\xd3\x12\x14\x9e\x98\x98\x80\x88\x48\x13\xe7\xca\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x3b\x0e\x86\x9a\xaa\x8e\xa4\x96\x32\xff\xff\x37\xb9\xe8\xce\x00\xca\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x3b\x0e\x86\x9a\xaa\x8e\xa4\x96\x32\xff\xff\x37\xb9\xe8\xce\x00\xa5\x19\xac\x1a\x35\xb4\xa5\x77\x87\x51\x0a\xf7\x8d\x8d\x20\x0a"
}, { /* wycheproof - edge case intermediate sums in poly1305 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff",
	.alen	= 4,
	.input	= "\xd3\x94\x28\xd0\x79\x35\x1f\x66\x5c\xd0\x01\x35\x43\x19\x87\x5c\xe5\xda\x78\x76\x6f\xa1\x92\x90\xc0\x31\xf7\x52\x08\x50\x67\x45\xae\x7a\xed\x35\xcb\x5a\x2f\xca\xa0\x34\x6e\xfb\x93\x65\x54\x64\x49\x6d\xde\xb0\x55\x09\xc6\xef\xff\xab\x75\xeb\x2d\xf4\xab\x09\x76\x2d\x90\x96\x52\x4f\xa1\xb2\xb0\x23\xb8\xb2\x88\x22\x27\x73\x01\x49\xef\x50\x4b\x71\xb1\x20\xca\x4f\xf3\x95\x19\xc2\xc2\x10",
	.ilen	= 96,
	.result	= "\xd3\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x62\x18\xb2\x7f\x83\xb8\xb4\x66\x02\xf6\xe1\xd8\x34\x20\x7b\x02\xce\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x2a\x64\x16\xce\xdb\x1c\xdd\x29\x6e\xf5\xd7\xd6\x92\xda\xff\x02\xce\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x2a\x64\x16\xce\xdb\x1c\xdd\x29\x6e\xf5\xd7\xd6\x92\xda\xff\x02\x30\x2f\xe8\x2a\xb0\xa0\x9a\xf6\x44\x00\xd0\x15\xae\x83\xd9\xcc"
}, { /* wycheproof - edge case intermediate sums in poly1305 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff",
	.alen	= 4,
	.input	= "\xe9\x94\x28\xd0\x79\x35\x1f\x66\x5c\xd0\x01\x35\x43\x19\x87\x5c\x6d\xf1\x39\x4e\xdc\x53\x9b\x5b\x3a\x09\x57\xbe\x0f\xb8\x59\x46\x80\x7a\xed\x35\xcb\x5a\x2f\xca\xa0\x34\x6e\xfb\x93\x65\x54\x64\xd1\x76\x9f\xe8\x06\xbb\xfe\xb6\xf5\x90\x95\x0f\x2e\xac\x9e\x0a\x58\x2d\x90\x96\x52\x4f\xa1\xb2\xb0\x23\xb8\xb2\x88\x22\x27\x73\x99\x52\xae\x08\x18\xc3\x89\x79\xc0\x74\x13\x71\x1a\x9a\xf7\x13",
	.ilen	= 96,
	.result	= "\xe9\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xea\x33\xf3\x47\x30\x4a\xbd\xad\xf8\xce\x41\x34\x33\xc8\x45\x01\xe0\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xb2\x7f\x57\x96\x88\xae\xe5\x70\x64\xce\x37\x32\x91\x82\xca\x01\xe0\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xb2\x7f\x57\x96\x88\xae\xe5\x70\x64\xce\x37\x32\x91\x82\xca\x01\x98\xa7\xe8\x36\xe0\xee\x4d\x02\x35\x00\xd0\x55\x7e\xc2\xcb\xe0"
}, { /* wycheproof - edge case intermediate sums in poly1305 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff",
	.alen	= 4,
	.input	= "\xff\x94\x28\xd0\x79\x35\x1f\x66\x5c\xd0\x01\x35\x43\x19\x87\x5c\x64\xf9\x0f\x5b\x26\x92\xb8\x60\xd4\x59\x6f\xf4\xb3\x40\x2c\x5c\x00\xb9\xbb\x53\x70\x7a\xa6\x67\xd3\x56\xfe\x50\xc7\x19\x96\x94\x03\x35\x61\xe7\xca\xca\x6d\x94\x1d\xc3\xcd\x69\x14\xad\x69\x04",
	.ilen	= 64,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xe3\x3b\xc5\x52\xca\x8b\x9e\x96\x16\x9e\x79\x7e\x8f\x30\x30\x1b\x60\x3c\xa9\x99\x44\xdf\x76\x52\x8c\x9d\x6f\x54\xab\x83\x3d\x0f\x60\x3c\xa9\x99\x44\xdf\x76\x52\x8c\x9d\x6f\x54\xab\x83\x3d\x0f\x6a\xb8\xdc\xe2\xc5\x9d\xa4\x73\x71\x30\xb0\x25\x2f\x68\xa8\xd8"
}, { /* wycheproof - edge case intermediate sums in poly1305 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff",
	.alen	= 4,
	.input	= "\x68\x94\x28\xd0\x79\x35\x1f\x66\x5c\xd0\x01\x35\x43\x19\x87\x5c\xb0\x8f\x25\x67\x5b\x9b\xcb\xf6\xe3\x84\x07\xde\x2e\xc7\x5a\x47\x9f\x7a\xed\x35\xcb\x5a\x2f\xca\xa0\x34\x6e\xfb\x93\x65\x54\x64\x2d\x2a\xf7\xcd\x6b\x08\x05\x01\xd3\x1b\xa5\x4f\xb2\xeb\x75\x96\x47\x2d\x90\x96\x52\x4f\xa1\xb2\xb0\x23\xb8\xb2\x88\x22\x27\x73\x65\x0e\xc6\x2d\x75\x70\x72\xce\xe6\xff\x23\x31\x86\xdd\x1c\x8f",
	.ilen	= 96,
	.result	= "\x68\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x37\x4d\xef\x6e\xb7\x82\xed\x00\x21\x43\x11\x54\x12\xb7\x46\x00\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x4e\x23\x3f\xb3\xe5\x1d\x1e\xc7\x42\x45\x07\x72\x0d\xc5\x21\x9d\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x4e\x23\x3f\xb3\xe5\x1d\x1e\xc7\x42\x45\x07\x72\x0d\xc5\x21\x9d\x04\x4d\xea\x60\x88\x80\x41\x2b\xfd\xff\xcf\x35\x57\x9e\x9b\x26"
}, { /* wycheproof - edge case intermediate sums in poly1305 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff",
	.alen	= 4,
	.input	= "\x6d\x94\x28\xd0\x79\x35\x1f\x66\x5c\xd0\x01\x35\x43\x19\x87\x5c\xa1\x61\xb5\xab\x04\x09\x00\x62\x9e\xfe\xff\x78\xd7\xd8\x6b\x45\x9f\x7a\xed\x35\xcb\x5a\x2f\xca\xa0\x34\x6e\xfb\x93\x65\x54\x64\xc6\xf8\x07\x8c\xc8\xef\x12\xa0\xff\x65\x7d\x6d\x08\xdb\x10\xb8\x47\x2d\x90\x96\x52\x4f\xa1\xb2\xb0\x23\xb8\xb2\x88\x22\x27\x73\x8e\xdc\x36\x6c\xd6\x97\x65\x6f\xca\x81\xfb\x13\x3c\xed\x79\xa1",
	.ilen	= 96,
	.result	= "\x6d\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x26\xa3\x7f\xa2\xe8\x10\x26\x94\x5c\x39\xe9\xf2\xeb\xa8\x77\x02\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xa5\xf1\xcf\xf2\x46\xfa\x09\x66\x6e\x3b\xdf\x50\xb7\xf5\x44\xb3\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xa5\xf1\xcf\xf2\x46\xfa\x09\x66\x6e\x3b\xdf\x50\xb7\xf5\x44\xb3\x1e\x6b\xea\x63\x14\x54\x2e\x2e\xf9\xff\xcf\x45\x0b\x2e\x98\x2b"
}, { /* wycheproof - edge case intermediate sums in poly1305 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff",
	.alen	= 4,
	.input	= "\xff\x94\x28\xd0\x79\x35\x1f\x66\x5c\xd0\x01\x35\x43\x19\x87\x5c\xfc\x01\xb8\x91\xe5\xf0\xf9\x12\x8d\x7d\x1c\x57\x91\x92\xb6\x98\x63\x41\x44\x15\xb6\x99\x68\x95\x9a\x72\x91\xb7\xa5\xaf\x13\x48\x60\xcd\x9e\xa1\x0c\x29\xa3\x66\x54\xe7\xa2\x8e\x76\x1b\xec\xd8",
	.ilen	= 64,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x7b\xc3\x72\x98\x09\xe9\xdf\xe4\x4f\xba\x0a\xdd\xad\xe2\xaa\xdf\x03\xc4\x56\xdf\x82\x3c\xb8\xa0\xc5\xb9\x00\xb3\xc9\x35\xb8\xd3\x03\xc4\x56\xdf\x82\x3c\xb8\xa0\xc5\xb9\x00\xb3\xc9\x35\xb8\xd3\xed\x20\x17\xc8\xdb\xa4\x77\x56\x29\x04\x9d\x78\x6e\x3b\xce\xb1"
}, { /* wycheproof - edge case intermediate sums in poly1305 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff",
	.alen	= 4,
	.input	= "\xff\x94\x28\xd0\x79\x35\x1f\x66\x5c\xd0\x01\x35\x43\x19\x87\x5c\x6b\x6d\xc9\xd2\x1a\x81\x9e\x70\xb5\x77\xf4\x41\x37\xd3\xd6\xbd\x13\x35\xf5\xeb\x44\x49\x40\x77\xb2\x64\x49\xa5\x4b\x6c\x7c\x75\x10\xb9\x2f\x5f\xfe\xf9\x8b\x84\x7c\xf1\x7a\x9c\x98\xd8\x83\xe5",
	.ilen	= 64,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xec\xaf\x03\xdb\xf6\x98\xb8\x86\x77\xb0\xe2\xcb\x0b\xa3\xca\xfa\x73\xb0\xe7\x21\x70\xec\x90\x42\xed\xaf\xd8\xa1\x27\xf6\xd7\xee\x73\xb0\xe7\x21\x70\xec\x90\x42\xed\xaf\xd8\xa1\x27\xf6\xd7\xee\x07\x3f\x17\xcb\x67\x78\x64\x59\x25\x04\x9d\x88\x22\xcb\xca\xb6"
}, { /* wycheproof - edge case intermediate sums in poly1305 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff",
	.alen	= 4,
	.input	= "\xff\xcb\x2b\x11\x06\xf8\x23\x4c\x5e\x99\xd4\xdb\x4c\x70\x48\xde\x32\x3d\x35\xf6\x13\xe6\xd9\x09\x3d\x38\xe9\x75\xc3\x8f\xe3\xb8\x16\xe9\x88\x4a\x11\x4f\x0e\x92\x66\xce\xa3\x88\x5f\xe3\x6b\x9f\xd6\xf6\x37\x81\x71\xea\xe4\x39\x6e\xa1\x5d\xc2\x40\xd1\xab\xf4\xce\xbe\xf5\xe9\x88\x5a\x80\xea\x76\xd9\x75\xc1\x44\xa4\x18\x88",
	.ilen	= 80,
	.result	= "\xff\xa0\xfc\x3e\x80\x32\xc3\xd5\xfd\xb6\x2a\x11\xf0\x96\x30\x7d\xb5\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x76\x6c\x9a\x80\x25\xea\xde\xa7\x39\x05\x32\x8c\x33\x79\xc0\x04\xb5\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x76\x6c\x9a\x80\x25\xea\xde\xa7\x39\x05\x32\x8c\x33\x79\xc0\x04\x8b\x9b\xb4\xb4\x86\x12\x89\x65\x8c\x69\x6a\x83\x40\x15\x04\x05"
}, { /* wycheproof - edge case intermediate sums in poly1305 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff",
	.alen	= 4,
	.input	= "\x6f\x9e\x70\xed\x3b\x8b\xac\xa0\x26\xe4\x6a\x5a\x09\x43\x15\x8d\x21\x3d\x35\xf6\x13\xe6\xd9\x09\x3d\x38\xe9\x75\xc3\x8f\xe3\xb8\x0c\x61\x2c\x5e\x8d\x89\xa8\x73\xdb\xca\xad\x5b\x73\x46\x42\x9b\xc5\xf6\x37\x81\x71\xea\xe4\x39\x6e\xa1\x5d\xc2\x40\xd1\xab\xf4\xd4\x36\x51\xfd\x14\x9c\x26\x0b\xcb\xdd\x7b\x12\x68\x01\x31\x8c",
	.ilen	= 80,
	.result	= "\x6f\xf5\xa7\xc2\xbd\x41\x4c\x39\x85\xcb\x94\x90\xb5\xa5\x6d\x2e\xa6\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x6c\xe4\x3e\x94\xb9\x2c\x78\x46\x84\x01\x3c\x5f\x1f\xdc\xe9\x00\xa6\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x6c\xe4\x3e\x94\xb9\x2c\x78\x46\x84\x01\x3c\x5f\x1f\xdc\xe9\x00\x8b\x3b\xbd\x51\x64\x44\x59\x56\x8d\x81\xca\x1f\xa7\x2c\xe4\x04"
}, { /* wycheproof - edge case intermediate sums in poly1305 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff",
	.alen	= 4,
	.input	= "\x41\x2b\x08\x0a\x3e\x19\xc1\x0d\x44\xa1\xaf\x1e\xab\xde\xb4\xce\x35\x3d\x35\xf6\x13\xe6\xd9\x09\x3d\x38\xe9\x75\xc3\x8f\xe3\xb8\x6b\x83\x94\x33\x09\x21\x48\x6c\xa1\x1d\x29\x1c\x3e\x97\xee\x9a\xd1\xf6\x37\x81\x71\xea\xe4\x39\x6e\xa1\x5d\xc2\x40\xd1\xab\xf4\xb3\xd4\xe9\x90\x90\x34\xc6\x14\xb1\x0a\xff\x55\x25\xd0\x9d\x8d",
	.ilen	= 80,
	.result	= "\x41\x40\xdf\x25\xb8\xd3\x21\x94\xe7\x8e\x51\xd4\x17\x38\xcc\x6d\xb2\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x0b\x06\x86\xf9\x3d\x84\x98\x59\xfe\xd6\xb8\x18\x52\x0d\x45\x01\xb2\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x0b\x06\x86\xf9\x3d\x84\x98\x59\xfe\xd6\xb8\x18\x52\x0d\x45\x01\x86\xfb\xab\x2b\x4a\x94\xf4\x7a\xa5\x6f\x0a\xea\x65\xd1\x10\x08"
}, { /* wycheproof - edge case intermediate sums in poly1305 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff",
	.alen	= 4,
	.input	= "\xb2\x47\xa7\x47\x23\x49\x1a\xac\xac\xaa\xd7\x09\xc9\x1e\x93\x2b\x31\x3d\x35\xf6\x13\xe6\xd9\x09\x3d\x38\xe9\x75\xc3\x8f\xe3\xb8\x9a\xde\x04\xe7\x5b\xb7\x01\xd9\x66\x06\x01\xb3\x47\x65\xde\x98\xd5\xf6\x37\x81\x71\xea\xe4\x39\x6e\xa1\x5d\xc2\x40\xd1\xab\xf4\x42\x89\x79\x44\xc2\xa2\x8f\xa1\x76\x11\xd7\xfa\x5c\x22\xad\x8f",
	.ilen	= 80,
	.result	= "\xb2\x2c\x70\x68\xa5\x83\xfa\x35\x0f\x85\x29\xc3\x75\xf8\xeb\x88\xb6\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xfa\x5b\x16\x2d\x6f\x12\xd1\xec\x39\xcd\x90\xb7\x2b\xff\x75\x03\xb6\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xfa\x5b\x16\x2d\x6f\x12\xd1\xec\x39\xcd\x90\xb7\x2b\xff\x75\x03\xa0\x19\xac\x2e\xd6\x67\xe1\x7d\xa1\x6f\x0a\xfa\x19\x61\x0d\x0d"
}, { /* wycheproof - edge case intermediate sums in poly1305 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff",
	.alen	= 4,
	.input	= "\x74\x0f\x9e\x49\xf6\x10\xef\xa5\x85\xb6\x59\xca\x6e\xd8\xb4\x99\x2d\x3d\x35\xf6\x13\xe6\xd9\x09\x3d\x38\xe9\x75\xc3\x8f\xe3\xb8\x41\x2d\x96\xaf\xbe\x80\xec\x3e\x79\xd4\x51\xb0\x0a\x2d\xb2\x9a\xc9\xf6\x37\x81\x71\xea\xe4\x39\x6e\xa1\x5d\xc2\x40\xd1\xab\xf4\x99\x7a\xeb\x0c\x27\x95\x62\x46\x69\xc3\x87\xf9\x11\x6a\xc1\x8d",
	.ilen	= 80,
	.result	= "\x74\x64\x49\x66\x70\xda\x0f\x3c\x26\x99\xa7\x00\xd2\x3e\xcc\x3a\xaa\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x21\xa8\x84\x65\x8a\x25\x3c\x0b\x26\x1f\xc0\xb4\x66\xb7\x19\x01\xaa\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x21\xa8\x84\x65\x8a\x25\x3c\x0b\x26\x1f\xc0\xb4\x66\xb7\x19\x01\x73\x6e\x18\x18\x16\x96\xa5\x88\x9c\x31\x59\xfa\xab\xab\x20\xfd"
}, { /* wycheproof - edge case intermediate sums in poly1305 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff",
	.alen	= 4,
	.input	= "\xad\xba\x5d\x10\x5b\xc8\xaa\x06\x2c\x23\x36\xcb\x88\x9d\xdb\xd5\x37\x3d\x35\xf6\x13\xe6\xd9\x09\x3d\x38\xe9\x75\xc3\x8f\xe3\xb8\x17\x7c\x5f\xfe\x28\x75\xf4\x68\xf6\xc2\x96\x57\x48\xf3\x59\x9a\xd3\xf6\x37\x81\x71\xea\xe4\x39\x6e\xa1\x5d\xc2\x40\xd1\xab\xf4\xcf\x2b\x22\x5d\xb1\x60\x7a\x10\xe6\xd5\x40\x1e\x53\xb4\x2a\x8d",
	.ilen	= 80,
	.result	= "\xad\xd1\x8a\x3f\xdd\x02\x4a\x9f\x8f\x0c\xc8\x01\x34\x7b\xa3\x76\xb0\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x77\xf9\x4d\x34\x1c\xd0\x24\x5d\xa9\x09\x07\x53\x24\x69\xf2\x01\xb0\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x77\xf9\x4d\x34\x1c\xd0\x24\x5d\xa9\x09\x07\x53\x24\x69\xf2\x01\xba\xd5\x8f\x10\xa9\x1e\x6a\x88\x9a\xba\x32\xfd\x17\xd8\x33\x1a"
}, { /* wycheproof - edge case intermediate sums in poly1305 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff",
	.alen	= 4,
	.input	= "\xfe\x94\x28\xd0\x79\x35\x1f\x66\x5c\xd0\x01\x35\x43\x19\x87\x5c\xc0\x01\xed\xc5\xda\x44\x2e\x71\x9b\xce\x9a\xbe\x27\x3a\xf1\x44\xb4\x7a\xed\x35\xcb\x5a\x2f\xca\xa0\x34\x6e\xfb\x93\x65\x54\x64\x48\x02\x5f\x41\xfa\x4e\x33\x6c\x78\x69\x57\xa2\xa7\xc4\x93\x0a\x6c\x2d\x90\x96\x52\x4f\xa1\xb2\xb0\x23\xb8\xb2\x88\x22\x27\x73\x00\x26\x6e\xa1\xe4\x36\x44\xa3\x4d\x8d\xd1\xdc\x93\xf2\xfa\x13",
	.ilen	= 96,
	.result	= "\xfe\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x47\xc3\x27\xcc\x36\x5d\x08\x87\x59\x09\x8c\x34\x1b\x4a\xed\x03\xd4\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x2b\x0b\x97\x3f\x74\x5b\x28\xaa\xe9\x37\xf5\x9f\x18\xea\xc7\x01\xd4\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x2b\x0b\x97\x3f\x74\x5b\x28\xaa\xe9\x37\xf5\x9f\x18\xea\xc7\x01\xd6\x8c\xe1\x74\x07\x9a\xdd\x02\x8d\xd0\x5c\xf8\x14\x63\x04\x88"
}, { /* wycheproof - edge case intermediate sums in poly1305 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff",
	.alen	= 4,
	.input	= "\xb5\x13\xb0\x6a\xb9\xac\x14\x43\x5a\xcb\x8a\xa3\xa3\x7a\xfd\xb6\x54\x3d\x35\xf6\x13\xe6\xd9\x09\x3d\x38\xe9\x75\xc3\x8f\xe3\xb8\x61\x95\x01\x93\xb1\xbf\x03\x11\xff\x11\x79\x89\xae\xd9\xa9\x99\xb0\xf6\x37\x81\x71\xea\xe4\x39\x6e\xa1\x5d\xc2\x40\xd1\xab\xf4\xb9\xc2\x7c\x30\x28\xaa\x8d\x69\xef\x06\xaf\xc0\xb5\x9e\xda\x8e",
	.ilen	= 80,
	.result	= "\xb5\x78\x67\x45\x3f\x66\xf4\xda\xf9\xe4\x74\x69\x1f\x9c\x85\x15\xd3\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x01\x10\x13\x59\x85\x1a\xd3\x24\xa0\xda\xe8\x8d\xc2\x43\x02\x02\xd3\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x01\x10\x13\x59\x85\x1a\xd3\x24\xa0\xda\xe8\x8d\xc2\x43\x02\x02\xaa\x48\xa3\x88\x7d\x4b\x05\x96\x99\xc2\xfd\xf9\xc6\x78\x7e\x0a"
}, { /* wycheproof - edge case intermediate sums in poly1305 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff",
	.alen	= 4,
	.input	= "\xff\x94\x28\xd0\x79\x35\x1f\x66\x5c\xd0\x01\x35\x43\x19\x87\x5c\xd4\xf1\x09\xe8\x14\xce\xa8\x5a\x08\xc0\x11\xd8\x50\xdd\x1d\xcb\xcf\x7a\xed\x35\xcb\x5a\x2f\xca\xa0\x34\x6e\xfb\x93\x65\x54\x64\x53\x40\xb8\x5a\x9a\xa0\x82\x96\xb7\x7a\x5f\xc3\x96\x1f\x66\x0f\x17\x2d\x90\x96\x52\x4f\xa1\xb2\xb0\x23\xb8\xb2\x88\x22\x27\x73\x1b\x64\x89\xba\x84\xd8\xf5\x59\x82\x9e\xd9\xbd\xa2\x29\x0f\x16",
	.ilen	= 96,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x53\x33\xc3\xe1\xf8\xd7\x8e\xac\xca\x07\x07\x52\x6c\xad\x01\x8c\xaf\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x30\x49\x70\x24\x14\xb5\x99\x50\x26\x24\xfd\xfe\x29\x31\x32\x04\xaf\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x30\x49\x70\x24\x14\xb5\x99\x50\x26\x24\xfd\xfe\x29\x31\x32\x04\xb9\x36\xa8\x17\xf2\x21\x1a\xf1\x29\xe2\xcf\x16\x0f\xd4\x2b\xcb"
}, { /* wycheproof - edge case intermediate sums in poly1305 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff",
	.alen	= 4,
	.input	= "\xff\x94\x28\xd0\x79\x35\x1f\x66\x5c\xd0\x01\x35\x43\x19\x87\x5c\xdf\x4c\x62\x03\x2d\x41\x19\xb5\x88\x47\x7e\x99\x92\x5a\x56\xd9\xd6\x7a\xed\x35\xcb\x5a\x2f\xca\xa0\x34\x6e\xfb\x93\x65\x54\x64\xfa\x84\xf0\x64\x55\x36\x42\x1b\x2b\xb9\x24\x6e\xc2\x19\xed\x0b\x0e\x2d\x90\x96\x52\x4f\xa1\xb2\xb0\x23\xb8\xb2\x88\x22\x27\x73\xb2\xa0\xc1\x84\x4b\x4e\x35\xd4\x1e\x5d\xa2\x10\xf6\x2f\x84\x12",
	.ilen	= 96,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x58\x8e\xa8\x0a\xc1\x58\x3f\x43\x4a\x80\x68\x13\xae\x2a\x4a\x9e\xb6\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x99\x8d\x38\x1a\xdb\x23\x59\xdd\xba\xe7\x86\x53\x7d\x37\xb9\x00\xb6\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x99\x8d\x38\x1a\xdb\x23\x59\xdd\xba\xe7\x86\x53\x7d\x37\xb9\x00\x9f\x7a\xc4\x35\x1f\x6b\x91\xe6\x30\x97\xa7\x13\x11\x5d\x05\xbe"
}, { /* wycheproof - edge case intermediate sums in poly1305 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff",
	.alen	= 4,
	.input	= "\xff\x94\x28\xd0\x79\x35\x1f\x66\x5c\xd0\x01\x35\x43\x19\x87\x5c\x13\xf8\x0a\x00\x6d\xc1\xbb\xda\xd6\x39\xa9\x2f\xc7\xec\xa6\x55\xf7\x7a\xed\x35\xcb\x5a\x2f\xca\xa0\x34\x6e\xfb\x93\x65\x54\x64\x63\x48\xb8\xfd\x29\xbf\x96\xd5\x63\xa5\x17\xe2\x7d\x7b\xfc\x0f\x2f\x2d\x90\x96\x52\x4f\xa1\xb2\xb0\x23\xb8\xb2\x88\x22\x27\x73\x2b\x6c\x89\x1d\x37\xc7\xe1\x1a\x56\x41\x91\x9c\x49\x4d\x95\x16",
	.ilen	= 96,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x94\x3a\xc0\x09\x81\xd8\x9d\x2c\x14\xfe\xbf\xa5\xfb\x9c\xba\x12\x97\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00\x41\x70\x83\xa7\xaa\x8d\x13\xf2\xfb\xb5\xdf\xc2\x55\xa8\x04\x97\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00\x41\x70\x83\xa7\xaa\x8d\x13\xf2\xfb\xb5\xdf\xc2\x55\xa8\x04\x9a\x18\xa8\x28\x07\x02\x69\xf4\x47\x00\xd0\x09\xe7\x17\x1c\xc9"
}, { /* wycheproof - edge case intermediate sums in poly1305 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff",
	.alen	= 4,
	.input	= "\xff\x94\x28\xd0\x79\x35\x1f\x66\x5c\xd0\x01\x35\x43\x19\x87\x5c\x82\xe5\x9b\x45\x82\x91\x50\x38\xf9\x33\x81\x1e\x65\x2d\xc6\x6a\xfc\x7a\xed\x35\xcb\x5a\x2f\xca\xa0\x34\x6e\xfb\x93\x65\x54\x64\xb6\x71\xc8\xca\xc2\x70\xc2\x65\xa0\xac\x2f\x53\x57\x99\x88\x0a\x24\x2d\x90\x96\x52\x4f\xa1\xb2\xb0\x23\xb8\xb2\x88\x22\x27\x73\xfe\x55\xf9\x2a\xdc\x08\xb5\xaa\x95\x48\xa9\x2d\x63\xaf\xe1\x13",
	.ilen	= 96,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x05\x27\x51\x4c\x6e\x88\x76\xce\x3b\xf4\x97\x94\x59\x5d\xda\x2d\x9c\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xd5\x78\x00\xb4\x4c\x65\xd9\xa3\x31\xf2\x8d\x6e\xe8\xb7\xdc\x01\x9c\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xd5\x78\x00\xb4\x4c\x65\xd9\xa3\x31\xf2\x8d\x6e\xe8\xb7\xdc\x01\xb4\x36\xa8\x2b\x93\xd5\x55\xf7\x43\x00\xd0\x19\x9b\xa7\x18\xce"
}, { /* wycheproof - edge case intermediate sums in poly1305 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff",
	.alen	= 4,
	.input	= "\xff\x94\x28\xd0\x79\x35\x1f\x66\x5c\xd0\x01\x35\x43\x19\x87\x5c\xf1\xd1\x28\x87\xb7\x21\x69\x86\xa1\x2d\x79\x09\x8b\x6d\xe6\x0f\xc0\x7a\xed\x35\xcb\x5a\x2f\xca\xa0\x34\x6e\xfb\x93\x65\x54\x64\xa7\xc7\x58\x99\xf3\xe6\x0a\xf1\xfc\xb6\xc7\x30\x7d\x87\x59\x0f\x18\x2d\x90\x96\x52\x4f\xa1\xb2\xb0\x23\xb8\xb2\x88\x22\x27\x73\xef\xe3\x69\x79\xed\x9e\x7d\x3e\xc9\x52\x41\x4e\x49\xb1\x30\x16",
	.ilen	= 96,
	.result	= "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x76\x13\xe2\x8e\x5b\x38\x4f\x70\x63\xea\x6f\x83\xb7\x1d\xfa\x48\xa0\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xc4\xce\x90\xe7\x7d\xf3\x11\x37\x6d\xe8\x65\x0d\xc2\xa9\x0d\x04\xa0\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xc4\xce\x90\xe7\x7d\xf3\x11\x37\x6d\xe8\x65\x0d\xc2\xa9\x0d\x04\xce\x54\xa8\x2e\x1f\xa9\x42\xfa\x3f\x00\xd0\x29\x4f\x37\x15\xd3"
}, { /* wycheproof - edge case intermediate sums in poly1305 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff",
	.alen	= 4,
	.input	= "\xcb\xf1\xda\x9e\x0b\xa9\x37\x73\x74\xe6\x9e\x1c\x0e\x60\x0c\xfc\x34\x3d\x35\xf6\x13\xe6\xd9\x09\x3d\x38\xe9\x75\xc3\x8f\xe3\xb8\xbe\x3f\xa6\x6b\x6c\xe7\x80\x8a\xa3\xe4\x59\x49\xf9\x44\x64\x9f\xd0\xf6\x37\x81\x71\xea\xe4\x39\x6e\xa1\x5d\xc2\x40\xd1\xab\xf4\x66\x68\xdb\xc8\xf5\xf2\x0e\xf2\xb3\xf3\x8f\x00\xe2\x03\x17\x88",
	.ilen	= 80,
	.result	= "\xcb\x9a\x0d\xb1\x8d\x63\xd7\xea\xd7\xc9\x60\xd6\xb2\x86\x74\x5f\xb3\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xde\xba\xb4\xa1\x58\x42\x50\xbf\xfc\x2f\xc8\x4d\x95\xde\xcf\x04\xb3\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xde\xba\xb4\xa1\x58\x42\x50\xbf\xfc\x2f\xc8\x4d\x95\xde\xcf\x04\x23\x83\xab\x0b\x79\x92\x05\x69\x9b\x51\x0a\xa7\x09\xbf\x31\xf1"
}, { /* wycheproof - edge case intermediate sums in poly1305 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff",
	.alen	= 4,
	.input	= "\x8f\x27\x86\x94\xc4\xe9\xda\xeb\xd5\x8d\x3e\x5b\x96\x6e\x8b\x68\x42\x3d\x35\xf6\x13\xe6\xd9\x09\x3d\x38\xe9\x75\xc3\x8f\xe3\xb8\x06\x53\xe7\xa3\x31\x71\x88\x33\xac\xc3\xb9\xad\xff\x1c\x31\x98\xa6\xf6\x37\x81\x71\xea\xe4\x39\x6e\xa1\x5d\xc2\x40\xd1\xab\xf4\xde\x04\x9a\x00\xa8\x64\x06\x4b\xbc\xd4\x6f\xe4\xe4\x5b\x42\x8f",
	.ilen	= 80,
	.result	= "\x8f\x4c\x51\xbb\x42\x23\x3a\x72\x76\xa2\xc0\x91\x2a\x88\xf3\xcb\xc5\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x66\xd6\xf5\x69\x05\xd4\x58\x06\xf3\x08\x28\xa9\x93\x86\x9a\x03\xc5\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x66\xd6\xf5\x69\x05\xd4\x58\x06\xf3\x08\x28\xa9\x93\x86\x9a\x03\x8b\xfb\xab\x17\xa9\xe0\xb8\x74\x8b\x51\x0a\xe7\xd9\xfd\x23\x05"
}, { /* wycheproof - edge case intermediate sums in poly1305 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff",
	.alen	= 4,
	.input	= "\xd5\x94\x28\xd0\x79\x35\x1f\x66\x5c\xd0\x01\x35\x43\x19\x87\x5c\x9a\x22\xd7\x0a\x48\xe2\x4f\xdd\xcd\xd4\x41\x9d\xe6\x4c\x8f\x44\xfc\x7a\xed\x35\xcb\x5a\x2f\xca\xa0\x34\x6e\xfb\x93\x65\x54\x64\x77\xb5\xc9\x07\xd9\xc9\xe1\xea\x51\x85\x1a\x20\x4a\xad\x9f\x0a\x24\x2d\x90\x96\x52\x4f\xa1\xb2\xb0\x23\xb8\xb2\x88\x22\x27\x73\x3f\x91\xf8\xe7\xc7\xb1\x96\x25\x64\x61\x9c\x5e\x7e\x9b\xf6\x13",
	.ilen	= 96,
	.result	= "\xd5\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x1d\xe0\x1d\x03\xa4\xfb\x69\x2b\x0f\x13\x57\x17\xda\x3c\x93\x03\x9c\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x14\xbc\x01\x79\x57\xdc\xfa\x2c\xc0\xdb\xb8\x1d\xf5\x83\xcb\x01\x9c\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x14\xbc\x01\x79\x57\xdc\xfa\x2c\xc0\xdb\xb8\x1d\xf5\x83\xcb\x01\x49\xbc\x6e\x9f\xc5\x1c\x4d\x50\x30\x36\x64\x4d\x84\x27\x73\xd2"
}, { /* wycheproof - edge case intermediate sums in poly1305 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff",
	.alen	= 4,
	.input	= "\xdb\x94\x28\xd0\x79\x35\x1f\x66\x5c\xd0\x01\x35\x43\x19\x87\x5c\x75\xd5\x64\x3a\xa5\xaf\x93\x4d\x8c\xce\x39\x2c\xc3\xee\xdb\x47\xc0\x7a\xed\x35\xcb\x5a\x2f\xca\xa0\x34\x6e\xfb\x93\x65\x54\x64\x60\x1b\x5a\xd2\x06\x7f\x28\x06\x6a\x8f\x32\x81\x71\x5b\xa8\x08\x18\x2d\x90\x96\x52\x4f\xa1\xb2\xb0\x23\xb8\xb2\x88\x22\x27\x73\x28\x3f\x6b\x32\x18\x07\x5f\xc9\x5f\x6b\xb4\xff\x45\x6d\xc1\x11",
	.ilen	= 96,
	.result	= "\xdb\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xf2\x17\xae\x33\x49\xb6\xb5\xbb\x4e\x09\x2f\xa6\xff\x9e\xc7\x00\xa0\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x03\x12\x92\xac\x88\x6a\x33\xc0\xfb\xd1\x90\xbc\xce\x75\xfc\x03\xa0\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x03\x12\x92\xac\x88\x6a\x33\xc0\xfb\xd1\x90\xbc\xce\x75\xfc\x03\x63\xda\x6e\xa2\x51\xf0\x39\x53\x2c\x36\x64\x5d\x38\xb7\x6f\xd7"
}, { /* wycheproof - edge case intermediate sums in poly1305 */
	.key	= "\x80\x81\x82\x83\x84\x85\x86\x87\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f\x90\x91\x92\x93\x94\x95\x96\x97\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f",
	.nonce	= "\x00\x00\x00\x00\x06\x4c\x2d\x52",
	.nlen	= 8,
	.assoc	= "\xff\xff\xff\xff",
	.alen	= 4,
	.input	= "\x93\x94\x28\xd0\x79\x35\x1f\x66\x5c\xd0\x01\x35\x43\x19\x87\x5c\x62\x48\x39\x60\x42\x16\xe4\x03\xeb\xcc\x6a\xf5\x59\xec\x8b\x43\x97\x7a\xed\x35\xcb\x5a\x2f\xca\xa0\x34\x6e\xfb\x93\x65\x54\x64\xd8\xc8\xc3\xfa\x1a\x9e\x47\x4a\xbe\x52\xd0\x2c\x81\x87\xe9\x0f\x4f\x2d\x90\x96\x52\x4f\xa1\xb2\xb0\x23\xb8\xb2\x88\x22\x27\x73\x90\xec\xf2\x1a\x04\xe6\x30\x85\x8b\xb6\x56\x52\xb5\xb1\x80\x16",
	.ilen	= 96,
	.result	= "\x93\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xe5\x8a\xf3\x69\xae\x0f\xc2\xf5\x29\x0b\x7c\x7f\x65\x9c\x97\x04\xf7\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xbb\xc1\x0b\x84\x94\x8b\x5c\x8c\x2f\x0c\x72\x11\x3e\xa9\xbd\x04\xf7\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xbb\xc1\x0b\x84\x94\x8b\x5c\x8c\x2f\x0c\x72\x11\x3e\xa9\xbd\x04\x73\xeb\x27\x24\xb5\xc4\x05\xf0\x4d\x00\xd0\xf1\x58\x40\xa1\xc1"
} };
static const struct chacha20poly1305_testvec chacha20poly1305_dec_vectors[] __initconst = { {
	.key	= "\x1c\x92\x40\xa5\xeb\x55\xd3\x8a\xf3\x33\x88\x86\x04\xf6\xb5\xf0\x47\x39\x17\xc1\x40\x2b\x80\x09\x9d\xca\x5c\xbc\x20\x70\x75\xc0",
	.nonce	= "\x01\x02\x03\x04\x05\x06\x07\x08",
	.nlen	= 8,
	.assoc	= "\xf3\x33\x88\x86\x00\x00\x00\x00\x00\x00\x4e\x91",
	.alen	= 12,
	.input	= "\x64\xa0\x86\x15\x75\x86\x1a\xf4\x60\xf0\x62\xc7\x9b\xe6\x43\xbd\x5e\x80\x5c\xfd\x34\x5c\xf3\x89\xf1\x08\x67\x0a\xc7\x6c\x8c\xb2\x4c\x6c\xfc\x18\x75\x5d\x43\xee\xa0\x9e\xe9\x4e\x38\x2d\x26\xb0\xbd\xb7\xb7\x3c\x32\x1b\x01\x00\xd4\xf0\x3b\x7f\x35\x58\x94\xcf\x33\x2f\x83\x0e\x71\x0b\x97\xce\x98\xc8\xa8\x4a\xbd\x0b\x94\x81\x14\xad\x17\x6e\x00\x8d\x33\xbd\x60\xf9\x82\xb1\xff\x37\xc8\x55\x97\x97\xa0\x6e\xf4\xf0\xef\x61\xc1\x86\x32\x4e\x2b\x35\x06\x38\x36\x06\x90\x7b\x6a\x7c\x02\xb0\xf9\xf6\x15\x7b\x53\xc8\x67\xe4\xb9\x16\x6c\x76\x7b\x80\x4d\x46\xa5\x9b\x52\x16\xcd\xe7\xa4\xe9\x90\x40\xc5\xa4\x04\x33\x22\x5e\xe2\x82\xa1\xb0\xa0\x6c\x52\x3e\xaf\x45\x34\xd7\xf8\x3f\xa1\x15\x5b\x00\x47\x71\x8c\xbc\x54\x6a\x0d\x07\x2b\x04\xb3\x56\x4e\xea\x1b\x42\x22\x73\xf5\x48\x27\x1a\x0b\xb2\x31\x60\x53\xfa\x76\x99\x19\x55\xeb\xd6\x31\x59\x43\x4e\xce\xbb\x4e\x46\x6d\xae\x5a\x10\x73\xa6\x72\x76\x27\x09\x7a\x10\x49\xe6\x17\xd9\x1d\x36\x10\x94\xfa\x68\xf0\xff\x77\x98\x71\x30\x30\x5b\xea\xba\x2e\xda\x04\xdf\x99\x7b\x71\x4d\x6c\x6f\x2c\x29\xa6\xad\x5c\xb4\x02\x2b\x02\x70\x9b\xee\xad\x9d\x67\x89\x0c\xbb\x22\x39\x23\x36\xfe\xa1\x85\x1f\x38",
	.ilen	= 281,
	.result	= "\x49\x6e\x74\x65\x72\x6e\x65\x74\x2d\x44\x72\x61\x66\x74\x73\x20\x61\x72\x65\x20\x64\x72\x61\x66\x74\x20\x64\x6f\x63\x75\x6d\x65\x6e\x74\x73\x20\x76\x61\x6c\x69\x64\x20\x66\x6f\x72\x20\x61\x20\x6d\x61\x78\x69\x6d\x75\x6d\x20\x6f\x66\x20\x73\x69\x78\x20\x6d\x6f\x6e\x74\x68\x73\x20\x61\x6e\x64\x20\x6d\x61\x79\x20\x62\x65\x20\x75\x70\x64\x61\x74\x65\x64\x2c\x20\x72\x65\x70\x6c\x61\x63\x65\x64\x2c\x20\x6f\x72\x20\x6f\x62\x73\x6f\x6c\x65\x74\x65\x64\x20\x62\x79\x20\x6f\x74\x68\x65\x72\x20\x64\x6f\x63\x75\x6d\x65\x6e\x74\x73\x20\x61\x74\x20\x61\x6e\x79\x20\x74\x69\x6d\x65\x2e\x20\x49\x74\x20\x69\x73\x20\x69\x6e\x61\x70\x70\x72\x6f\x70\x72\x69\x61\x74\x65\x20\x74\x6f\x20\x75\x73\x65\x20\x49\x6e\x74\x65\x72\x6e\x65\x74\x2d\x44\x72\x61\x66\x74\x73\x20\x61\x73\x20\x72\x65\x66\x65\x72\x65\x6e\x63\x65\x20\x6d\x61\x74\x65\x72\x69\x61\x6c\x20\x6f\x72\x20\x74\x6f\x20\x63\x69\x74\x65\x20\x74\x68\x65\x6d\x20\x6f\x74\x68\x65\x72\x20\x74\x68\x61\x6e\x20\x61\x73\x20\x2f\xe2\x80\x9c\x77\x6f\x72\x6b\x20\x69\x6e\x20\x70\x72\x6f\x67\x72\x65\x73\x73\x2e\x2f\xe2\x80\x9d"
}, {
	.key	= "\x4c\xf5\x96\x83\x38\xe6\xae\x7f\x2d\x29\x25\x76\xd5\x75\x27\x86\x91\x9a\x27\x7a\xfb\x46\xc5\xef\x94\x81\x79\x57\x14\x59\x40\x68",
	.nonce	= "\xca\xbf\x33\x71\x32\x45\x77\x8e",
	.nlen	= 8,
	.assoc	= "",
	.alen	= 0,
	.input	= "\xea\xe0\x1e\x9e\x2c\x91\xaa\xe1\xdb\x5d\x99\x3f\x8a\xf7\x69\x92",
	.ilen	= 16,
	.result	= ""
}, {
	.key	= "\x2d\xb0\x5d\x40\xc8\xed\x44\x88\x34\xd1\x13\xaf\x57\xa1\xeb\x3a\x2a\x80\x51\x36\xec\x5b\xbc\x8\x93\x84\x21\xb5\x13\x88\x3c\xd",
	.nonce	= "\x3d\x86\xb5\x6b\xc8\xa3\x1f\x1d",
	.nlen	= 8,
	.assoc	= "\x33\x10\x41\x12\x1f\xf3\xd2\x6b",
	.alen	= 8,
	.input	= "\xdd\x6b\x3b\x82\xce\x5a\xbd\xd6\xa9\x35\x83\xd8\x8c\x3d\x85\x77",
	.ilen	= 16,
	.result	= ""
}, {
	.key	= "\x4b\x28\x4b\xa3\x7b\xbe\xe9\xf8\x31\x80\x82\xd7\xd8\xe8\xb5\xa1\xe2\x18\x18\x8a\x9c\xfa\xa3\x3d\x25\x71\x3e\x40\xbc\x54\x7a\x3e",
	.nonce	= "\xd2\x32\x1f\x29\x28\xc6\xc4\xc4",
	.nlen	= 8,
	.assoc	= "\x6a\xe2\xad\x3f\x88\x39\x5a\x40",
	.alen	= 8,
	.input	= "\xb7\x1b\xb0\x73\x59\xb0\x84\xb2\x6d\x8e\xab\x94\x31\xa1\xae\xac\x89",
	.ilen	= 17,
	.result	= "\xa4"
}, {
	.key	= "\x66\xca\x9c\x23\x2a\x4b\x4b\x31\xe\x92\x89\x8b\xf4\x93\xc7\x87\x98\xa3\xd8\x39\xf8\xf4\xa7\x1\xc0\x2e\xa\xa6\x7e\x5a\x78\x87",
	.nonce	= "\x20\x1c\xaa\x5f\x9c\xbf\x92\x30",
	.nlen	= 8,
	.assoc	= "",
	.alen	= 0,
	.input	= "\xbf\xe1\x5b\xb\xdb\x6b\xf5\x5e\x6c\x5d\x84\x44\x39\x81\xc1\x9c\xac",
	.ilen	= 17,
	.result	= "\x2d"
}, {
	.key	= "\x68\x7b\x8d\x8e\xe3\xc4\xdd\xae\xdf\x72\x7f\x53\x72\x25\x1e\x78\x91\xcb\x69\x76\x1f\x49\x93\xf9\x6f\x21\xcc\x39\x9c\xad\xb1\x1",
	.nonce	= "\xdf\x51\x84\x82\x42\xc\x75\x9c",
	.nlen	= 8,
	.assoc	= "\x70\xd3\x33\xf3\x8b\x18\xb",
	.alen	= 7,
	.input	= "\x8b\x6\xd3\x31\xb0\x93\x45\xb1\x75\x6e\x26\xf9\x67\xbc\x90\x15\x81\x2c\xb5\xf0\xc6\x2b\xc7\x8c\x56\xd1\xbf\x69\x6c\x7\xa0\xda\x65\x27\xc9\x90\x3d\xef\x4b\x11\xf\x19\x7\xfd\x29\x92\xd9\xc8\xf7\x99\x2e\x4a\xd0\xb8\x2c\xdc\x93\xf5\x9e\x33\x78\xd1\x37\xc3\x66\xd7\x5e\xbc\x44\xbf\x53\xa5\xbc\xc4\xcb\x7b\x3a\x8e\x7f\x2\xbd\xbb\xe7\xca\xa6\x6c\x6b\x93\x21\x93\x10\x61\xe7\x69\xd0\x78\xf3\x7\x5a\x1a\x8f\x73\xaa\xb1\x4e\xd3\xda\x4f\xf3\x32\xe1\x66\x3e\x6c\xc6\x13\xba\x6\x5b\xfc\x6a\xe5\x6f\x60\xfb\x7\x40\xb0\x8c\x9d\x84\x43\x6b\xc1\xf7\x8d\x8d\x31\xf7\x7a\x39\x4d\x8f\x9a\xeb",
	.ilen	= 145,
	.result	= "\x33\x2f\x94\xc1\xa4\xef\xcc\x2a\x5b\xa6\xe5\x8f\x1d\x40\xf0\x92\x3c\xd9\x24\x11\xa9\x71\xf9\x37\x14\x99\xfa\xbe\xe6\x80\xde\x50\xc9\x96\xd4\xb0\xec\x9e\x17\xec\xd2\x5e\x72\x99\xfc\xa\xe1\xcb\x48\xd2\x85\xdd\x2f\x90\xe0\x66\x3b\xe6\x20\x74\xbe\x23\x8f\xcb\xb4\xe4\xda\x48\x40\xa6\xd1\x1b\xc7\x42\xce\x2f\xc\xa6\x85\x6e\x87\x37\x3\xb1\x7c\x25\x96\xa3\x5\xd8\xb0\xf4\xed\xea\xc2\xf0\x31\x98\x6c\xd1\x14\x25\xc0\xcb\x1\x74\xd0\x82\xf4\x36\xf5\x41\xd5\xdc\xca\xc5\xbb\x98\xfe\xfc\x69\x21\x70\xd8\xa4\x4b\xc8\xde\x8f"
}, {
	.key	= "\x8d\xb8\x91\x48\xf0\xe7\xa\xbd\xf9\x3f\xcd\xd9\xa0\x1e\x42\x4c\xe7\xde\x25\x3d\xa3\xd7\x5\x80\x8d\xf2\x82\xac\x44\x16\x51\x1",
	.nonce	= "\xde\x7b\xef\xc3\x65\x1b\x68\xb0",
	.nlen	= 8,
	.assoc	= "",
	.alen	= 0,
	.input	= "\x85\x4\xc2\xed\x8d\xfd\x97\x5c\xd2\xb7\xe2\xc1\x6b\xa3\xba\xf8\xc9\x50\xc3\xc6\xa5\xe3\xa4\x7c\xc3\x23\x49\x5e\xa9\xb9\x32\xeb\x8a\x7c\xca\xe5\xec\xfb\x7c\xc0\xcb\x7d\xdc\x2c\x9d\x92\x55\x21\xa\xc8\x43\x63\x59\xa\x31\x70\x82\x67\x41\x3\xf8\xdf\xf2\xac\xa7\x2\xd4\xd5\x8a\x2d\xc8\x99\x19\x66\xd0\xf6\x88\x2c\x77\xd9\xd4\xd\x6c\xbd\x98\xde\xe7\x7f\xad\x7e\x8a\xfb\xe9\x4b\xe5\xf7\xe5\x50\xa0\x90\x3f\xd6\x22\x53\xe3\xfe\x1b\xcc\x79\x3b\xec\x12\x47\x52\xa7\xd6\x4\xe3\x52\xe6\x93\x90\x91\x32\x73\x79\xb8\xd0\x31\xde\x1f\x9f\x2f\x5\x38\x54\x2f\x35\x4\x39\xe0\xa7\xba\xc6\x52\xf6\x37\x65\x4c\x7\xa9\x7e\xb3\x21\x6f\x74\x8c\xc9\xde\xdb\x65\x1b\x9b\xaa\x60\xb1\x3\x30\x6b\xb2\x3\xc4\x1c\x4\xf8\xf\x64\xaf\x46\xe4\x65\x99\x49\xe2\xea\xce\x78\x0\xd8\x8b\xd5\x2e\xcf\xfc\x40\x49\xe8\x58\xdc\x34\x9c\x8c\x61\xbf\xa\x8e\xec\x39\xa9\x30\x5\x5a\xd2\x56\x1\xc7\xda\x8f\x4e\xbb\x43\xa3\x3a\xf9\x15\x2a\xd0\xa0\x7a\x87\x34\x82\xfe\x8a\xd1\x2d\x5e\xc7\xbf\x4\x53\x5f\x3b\x36\xd4\x25\x5c\x34\x7a\x8d\xd5\x5\xce\x72\xca\xef\x7a\x4b\xbc\xb0\x10\x5c\x96\x42\x3a\x0\x98\xcd\x15\xe8\xb7\x53",
	.ilen	= 272,
	.result	= "\x9b\x18\xdb\xdd\x9a\xf\x3e\xa5\x15\x17\xde\xdf\x8\x9d\x65\xa\x67\x30\x12\xe2\x34\x77\x4b\xc1\xd9\xc6\x1f\xab\xc6\x18\x50\x17\xa7\x9d\x3c\xa6\xc5\x35\x8c\x1c\xc0\xa1\x7c\x9f\x3\x89\xca\xe1\xe6\xe9\xd4\xd3\x88\xdb\xb4\x51\x9d\xec\xb4\xfc\x52\xee\x6d\xf1\x75\x42\xc6\xfd\xbd\x7a\x8e\x86\xfc\x44\xb3\x4f\xf3\xea\x67\x5a\x41\x13\xba\xb0\xdc\xe1\xd3\x2a\x7c\x22\xb3\xca\xac\x6a\x37\x98\x3e\x1d\x40\x97\xf7\x9b\x1d\x36\x6b\xb3\x28\xbd\x60\x82\x47\x34\xaa\x2f\x7d\xe9\xa8\x70\x81\x57\xd4\xb9\x77\xa\x9d\x29\xa7\x84\x52\x4f\xc2\x4a\x40\x3b\x3c\xd4\xc9\x2a\xdb\x4a\x53\xc4\xbe\x80\xe9\x51\x7f\x8f\xc7\xa2\xce\x82\x5c\x91\x1e\x74\xd9\xd0\xbd\xd5\xf3\xfd\xda\x4d\x25\xb4\xbb\x2d\xac\x2f\x3d\x71\x85\x7b\xcf\x3c\x7b\x3e\xe\x22\x78\xc\x29\xbf\xe4\xf4\x57\xb3\xcb\x49\xa0\xfc\x1e\x5\x4e\x16\xbc\xd5\xa8\xa3\xee\x5\x35\xc6\x7c\xab\x60\x14\x55\x1a\x8e\xc5\x88\x5d\xd5\x81\xc2\x81\xa5\xc4\x60\xdb\xaf\x77\x91\xe1\xce\xa2\x7e\x7f\x42\xe3\xb0\x13\x1c\x1f\x25\x60\x21\xe2\x40\x5f\x99\xb7\x73\xec\x9b\x2b\xf0\x65\x11\xc8\xd0\xa\x9f\xd3"
}, {
	.key	= "\xf2\xaa\x4f\x99\xfd\x3e\xa8\x53\xc1\x44\xe9\x81\x18\xdc\xf5\xf0\x3e\x44\x15\x59\xe0\xc5\x44\x86\xc3\x91\xa8\x75\xc0\x12\x46\xba",
	.nonce	= "\xe\xd\x57\xbb\x7b\x40\x54\x2",
	.nlen	= 8,
	.assoc	= "",
	.alen	= 0,
	.input	= "\x14\xf6\x41\x37\xa6\xd4\x27\xcd\xdb\x6\x3e\x9a\x4e\xab\xd5\xb1\x1e\x6b\xd2\xbc\x11\xf4\x28\x93\x63\x54\xef\xbb\x5e\x1d\x3a\x1d\x37\x3c\xa\x6c\x1e\xc2\xd1\x2c\xb5\xa3\xb5\x7b\xb8\x8f\x25\xa6\x1b\x61\x1c\xec\x28\x58\x26\xa4\xa8\x33\x28\x25\x5c\x45\x5\xe5\x6c\x99\xe5\x45\xc4\xa2\x3\x84\x3\x73\x1e\x8c\x49\xac\x20\xdd\x8d\xb3\xc4\xf5\xe7\x4f\xf1\xed\xa1\x98\xde\xa4\x96\xdd\x2f\xab\xab\x97\xcf\x3e\xd2\x9e\xb8\x13\x7\x28\x29\x19\xaf\xfd\xf2\x49\x43\xea\x49\x26\x91\xc1\x7\xd6\xbb\x81\x75\x35\xd\x24\x7f\xc8\xda\xd4\xb7\xeb\xe8\x5c\x9\xa2\x2f\xdc\x28\x7d\x3a\x3\xfa\x94\xb5\x1d\x17\x99\x36\xc3\x1c\x18\x34\xe3\x9f\xf5\x55\x7c\xb0\x60\x9d\xff\xac\xd4\x61\xf2\xad\xf8\xce\xc7\xbe\x5c\xd2\x95\xa8\x4b\x77\x13\x19\x59\x26\xc9\xb7\x8f\x6a\xcb\x2d\x37\x91\xea\x92\x9c\x94\x5b\xda\xb\xce\xfe\x30\x20\xf8\x51\xad\xf2\xbe\xe7\xc7\xff\xb3\x33\x91\x6a\xc9\x1a\x41\xc9\xf\xf3\x10\xe\xfd\x53\xff\x6c\x16\x52\xd9\xf3\xf7\x98\x2e\xc9\x7\x31\x2c\xc\x72\xd7\xc5\xc6\x8\x2a\x7b\xda\xbd\x7e\x2\xea\x1a\xbb\xf2\x4\x27\x61\x28\x8e\xf5\x4\x3\x1f\x4c\x7\x55\x82\xec\x1e\xd7\x8b\x2f\x65\x56\xd1\xd9\x1e\x3c\xe9\x1f\x5e\x98\x70\x38\x4a\x8c\x49\xc5\x43\xa0\xa1\x8b\x74\x9d\x4c\x62\xd\x10\xc\xf4\x6c\x8f\xe0\xaa\x9a\x8d\xb7\xe0\xbe\x4c\x87\xf1\x98\x2f\xcc\xed\xc0\x52\x29\xdc\x83\xf8\xfc\x2c\xe\xa8\x51\x4d\x80\xd\xa3\xfe\xd8\x37\xe7\x41\x24\xfc\xfb\x75\xe3\x71\x7b\x57\x45\xf5\x97\x73\x65\x63\x14\x74\xb8\x82\x9f\xf8\x60\x2f\x8a\xf2\x4e\xf1\x39\xda\x33\x91\xf8\x36\xe0\x8d\x3f\x1f\x3b\x56\xdc\xa0\x8f\x3c\x9d\x71\x52\xa7\xb8\xc0\xa5\xc6\xa2\x73\xda\xf4\x4b\x74\x5b\x0\x3d\x99\xd7\x96\xba\xe6\xe1\xa6\x96\x38\xad\xb3\xc0\xd2\xba\x91\x6b\xf9\x19\xdd\x3b\xbe\xbe\x9c\x20\x50\xba\xa1\xd0\xce\x11\xbd\x95\xd8\xd1\xdd\x33\x85\x74\xdc\xdb\x66\x76\x44\xdc\x3\x74\x48\x35\x98\xb1\x18\x47\x94\x7d\xff\x62\xe4\x58\x78\xab\xed\x95\x36\xd9\x84\x91\x82\x64\x41\xbb\x58\xe6\x1c\x20\x6d\x15\x6b\x13\x96\xe8\x35\x7f\xdc\x40\x2c\xe9\xbc\x8a\x4f\x92\xec\x6\x2d\x50\xdf\x93\x5d\x65\x5a\xa8\xfc\x20\x50\x14\xa9\x8a\x7e\x1d\x8\x1f\xe2\x99\xd0\xbe\xfb\x3a\x21\x9d\xad\x86\x54\xfd\xd\x98\x1c\x5a\x6f\x1f\x9a\x40\xcd\xa2\xff\x6a\xf1\x54",
	.ilen	= 528,
	.result	= "\xc3\x9\x94\x62\xe6\x46\x2e\x10\xbe\x0\xe4\xfc\xf3\x40\xa3\xe2\xf\xc2\x8b\x28\xdc\xba\xb4\x3c\xe4\x21\x58\x61\xcd\x8b\xcd\xfb\xac\x94\xa1\x45\xf5\x1c\xe1\x12\xe0\x3b\x67\x21\x54\x5e\x8c\xaa\xcf\xdb\xb4\x51\xd4\x13\xda\xe6\x83\x89\xb6\x92\xe9\x21\x76\xa4\x93\x7d\xe\xfd\x96\x36\x3\x91\x43\x5c\x92\x49\x62\x61\x7b\xeb\x43\x89\xb8\x12\x20\x43\xd4\x47\x6\x84\xee\x47\xe9\x8a\x73\x15\xf\x72\xcf\xed\xce\x96\xb2\x7f\x21\x45\x76\xeb\x26\x28\x83\x6a\xad\xaa\xa6\x81\xd8\x55\xb1\xa3\x85\xb3\xc\xdf\xf1\x69\x2d\x97\x5\x2a\xbc\x7c\x7b\x25\xf8\x80\x9d\x39\x25\xf3\x62\xf0\x66\x5e\xf4\xa0\xcf\xd8\xfd\x4f\xb1\x1f\x60\x3a\x8\x47\xaf\xe1\xf6\x10\x77\x9\xa7\x27\x8f\x9a\x97\x5a\x26\xfa\xfe\x41\x32\x83\x10\xe0\x1d\xbf\x64\xd\xf4\x1c\x32\x35\xe5\x1b\x36\xef\xd4\x4a\x93\x4d\x0\x7c\xec\x2\x7\x8b\x5d\x7d\x1b\xe\xd1\xa6\xa5\x5d\x7d\x57\x88\xa8\xcc\x81\xb4\x86\x4e\xb4\x40\xe9\x1d\xc3\xb1\x24\x3e\x7f\xcc\x8a\x24\x9b\xdf\x6d\xf0\x39\x69\x3e\x4c\xc0\x96\xe4\x13\xda\x90\xda\xf4\x95\x66\x8b\x17\x17\xfe\x39\x43\x25\xaa\xda\xa0\x43\x3c\xb1\x41\x2\xa3\xf0\xa7\x19\x59\xbc\x1d\x7d\x6c\x6d\x91\x9\x5c\xb7\x5b\x1\xd1\x6f\x17\x21\x97\xbf\x89\x71\xa5\xb0\x6e\x7\x45\xfd\x9d\xea\x7\xf6\x7a\x9f\x10\x18\x22\x30\x73\xac\xd4\x6b\x72\x44\xed\xd9\x19\x9b\x2d\x4a\x41\xdd\xd1\x85\x5e\x37\x19\xed\xd2\x15\x8f\x5e\x91\xdb\x33\xf2\xe4\xdb\xff\x98\xfb\xa3\xb5\xca\x21\x69\x8\xe7\x8a\xdf\x90\xff\x3e\xe9\x20\x86\x3c\xe9\xfc\xb\xfe\x5c\x61\xaa\x13\x92\x7f\x7b\xec\xe0\x6d\xa8\x23\x22\xf6\x6b\x77\xc4\xfe\x40\x7\x3b\xb6\xf6\x8e\x5f\xd4\xb9\xb7\xf\x21\x4\xef\x83\x63\x91\x69\x40\xa3\x48\x5c\xd2\x60\xf9\x4f\x6c\x47\x8b\x3b\xb1\x9f\x8e\xee\x16\x8a\x13\xfc\x46\x17\xc3\xc3\x32\x56\xf8\x3c\x85\x3a\xb6\x3e\xaa\x89\x4f\xb3\xdf\x38\xfd\xf1\xe4\x3a\xc0\xe6\x58\xb5\x8f\xc5\x29\xa2\x92\x4a\xb6\xa0\x34\x7f\xab\xb5\x8a\x90\xa1\xdb\x4d\xca\xb6\x2c\x41\x3c\xf7\x2b\x21\xc3\xfd\xf4\x17\x5c\xb5\x33\x17\x68\x2b\x8\x30\xf3\xf7\x30\x3c\x96\xe6\x6a\x20\x97\xe7\x4d\x10\x5f\x47\x5f\x49\x96\x9\xf0\x27\x91\xc8\xf8\x5a\x2e\x79\xb5\xe2\xb8\xe8\xb9\x7b\xd5\x10\xcb\xff\x5d\x14\x73\xf3"
}, {
	.key	= "\xea\xbc\x56\x99\xe3\x50\xff\xc5\xcc\x1a\xd7\xc1\x57\x72\xea\x86\x5b\x89\x88\x61\x3d\x2f\x9b\xb2\xe7\x9c\xec\x74\x6e\x3e\xf4\x3b",
	.nonce	= "\xef\x2d\x63\xee\x6b\x80\x8b\x78",
	.nlen	= 8,
	.assoc	= "\x5a\x27\xff\xeb\xdf\x84\xb2\x9e\xef",
	.alen	= 9,
	.input	= "\xfd\x81\x8d\xd0\x3d\xb4\xd5\xdf\xd3\x42\x47\x5a\x6d\x19\x27\x66\x4b\x2e\xc\x27\x9c\x96\x4c\x72\x2\xa3\x65\xc3\xb3\x6f\x2e\xbd\x63\x8a\x4a\x5d\x29\xa2\xd0\x28\x48\xc5\x3d\x98\xa3\xbc\xe0\xbe\x3b\x3f\xe6\x8a\xa4\x7f\x53\x6\xfa\x7f\x27\x76\x72\x31\xa1\xf5\xd6\xc\x52\x47\xba\xcd\x4f\xd7\xeb\x5\x48\xd\x7c\x35\x4a\x9\xc9\x76\x71\x2\xa3\xfb\xb7\x1a\x65\xb7\xed\x98\xc6\x30\x8a\x0\xae\xa1\x31\xe5\xb5\x9e\x6d\x62\xda\xda\x7\xf\x38\x38\xd3\xcb\xc1\xb0\xad\xec\x72\xec\xb1\xa2\x7b\x59\xf3\x3d\x2b\xef\xcd\x28\x5b\x83\xcc\x18\x91\x88\xb0\x2e\xf9\x29\x31\x18\xf9\x4e\xe9\xa\x91\x92\x9f\xae\x2d\xad\xf4\xe6\x1a\xe2\xa4\xee\x47\x15\xbf\x83\x6e\xd7\x72\x12\x3b\x2d\x24\xe9\xb2\x55\xcb\x3c\x10\xf0\x24\x8a\x4a\x2\xea\x90\x25\xf0\xb4\x79\x3a\xef\x6e\xf5\x52\xdf\xb0\xa\xcd\x24\x1c\xd3\x2e\x22\x74\xea\x21\x6f\xe9\xbd\xc8\x3e\x36\x5b\x19\xf1\xca\x99\xa\xb4\xa7\x52\x1a\x4e\xf2\xad\x8d\x56\x85\xbb\x64\x89\xba\x26\xf9\xc7\xe1\x89\x19\x22\x77\xc3\xa8\xfc\xff\xad\xfe\xb9\x48\xae\x12\x30\x9f\x19\xfb\x1b\xef\x14\x87\x8a\x78\x71\xf3\xf4\xb7\x0\x9c\x1d\xb5\x3d\x49\x0\xc\x6\xd4\x50\xf9\x54\x45\xb2\x5b\x43\xdb\x6d\xcf\x1a\xe9\x7a\x7a\xcf\xfc\x8a\x4e\x4d\xb\x7\x63\x28\xd8\xe7\x8\x95\xdf\xa6\x72\x93\x2e\xbb\xa0\x42\x89\x16\xf1\xd9\xc\xf9\xa1\x16\xfd\xd9\x3\xb4\x3b\x8a\xf5\xf6\xe7\x6b\x2e\x8e\x4c\x3d\xe2\xaf\x8\x45\x3\xff\x9\xb6\xeb\x2d\xc6\x1b\x88\x94\xac\x3e\xf1\x9f\xe\xe\x2b\xd5\x0\x4d\x3f\x3b\x53\xae\xaf\x1c\x33\x5f\x55\x6e\x8d\xaf\x5\x7a\x10\x34\xc9\xf4\x66\xcb\x62\x12\xa6\xee\xe8\x1c\x5d\x12\x86\xdb\x6f\x1c\x33\xc4\x1c\xda\x82\x2d\x3b\x59\xfe\xb1\xa4\x59\x41\x86\xd0\xef\xae\xfb\xda\x6d\x11\xb8\xca\xe9\x6e\xff\xf7\xa9\xd9\x70\x30\xfc\x53\xe2\xd7\xa2\x4e\xc7\x91\xd9\x7\x6\xaa\xdd\xb0\x59\x28\x1d\x0\x66\xc5\x54\xc2\xfc\x6\xda\x5\x90\x52\x1d\x37\x66\xee\xf0\xb2\x55\x8a\x5d\xd2\x38\x86\x94\x9b\xfc\x10\x4c\xa1\xb9\x64\x3e\x44\xb8\x5f\xb0\xc\xec\xe0\xc9\xe5\x62\x75\x3f\x9\xd5\xf5\xd9\x26\xba\x9e\xd2\xf4\xb9\x48\xa\xbc\xa2\xd6\x7c\x36\x11\x7d\x26\x81\x89\xcf\xa4\xad\x73\xe\xee\xcc\x6\xa9\xdb\xb1\xfd\xfb\x9\x7f\x90\x42\x37\x2f\xe1\x9c\xf\x6f\xcf\x43\xb5\xd9\x90\xe1\x85\xf5\xa8\xae",
	.ilen	= 529,
	.result	= "\xe6\xc3\xdb\x63\x55\x15\xe3\x5b\xb7\x4b\x27\x8b\x5a\xdd\xc2\xe8\x3a\x6b\xd7\x81\x96\x35\x97\xca\xd7\x68\xe8\xef\xce\xab\xda\x9\x6e\xd6\x8e\xcb\x55\xb5\xe1\xe5\x57\xfd\xc4\xe3\xe0\x18\x4f\x85\xf5\x3f\x7e\x4b\x88\xc9\x52\x44\xf\xea\xaf\x1f\x71\x48\x9f\x97\x6d\xb9\x6f\x0\xa6\xde\x2b\x77\x8b\x15\xad\x10\xa0\x2b\x7b\x41\x90\x3\x2d\x69\xae\xcc\x77\x7c\xa5\x9d\x29\x22\xc2\xea\xb4\x0\x1a\xd2\x7a\x98\x8a\xf9\xf7\x82\xb0\xab\xd8\xa6\x94\x8d\x58\x2f\x1\x9e\x0\x20\xfc\x49\xdc\xe\x3\xe8\x45\x10\xd6\xa8\xda\x55\x10\x9a\xdf\x67\x22\x8b\x43\xab\x0\xbb\x2\xc8\xdd\x7b\x97\x17\xd7\x1d\x9e\x2\x5e\x48\xde\x8e\xcf\x99\x7\x95\x92\x3c\x5f\x9f\xc5\x8a\xc0\x23\xaa\xd5\x8c\x82\x6e\x16\x92\xb1\x12\x17\x7\xc3\xfb\x36\xf5\x6c\x35\xd6\x6\x1f\x9f\xa7\x94\xa2\x38\x63\x9c\xb0\x71\xb3\xa5\xd2\xd8\xba\x9f\x8\x1\xb3\xff\x4\x97\x73\x45\x1b\xd5\xa9\x9c\x80\xaf\x4\x9a\x85\xdb\x32\x5b\x5d\x1a\xc1\x36\x28\x10\x79\xf1\x3c\xbf\x1a\x41\x5c\x4e\xdf\xb2\x7c\x79\x3b\x7a\x62\x3d\x4b\xc9\x9b\x2a\x2e\x7c\xa2\xb1\x11\x98\xa7\x34\x1a\x0\xf3\xd1\xbc\x18\x22\xba\x2\x56\x62\x31\x10\x11\x6d\xe0\x54\x9d\x40\x1f\x26\x80\x41\xca\x3f\x68\xf\x32\x1d\xa\x8e\x79\xd8\xa4\x1b\x29\x1c\x90\x8e\xc5\xe3\xb4\x91\x37\x9a\x97\x86\x99\xd5\x9\xc5\xbb\xa3\x3f\x21\x29\x82\x14\x5c\xab\x25\xfb\xf2\x4f\x58\x26\xd4\x83\xaa\x66\x89\x67\x7e\xc0\x49\xe1\x11\x10\x7f\x7a\xda\x29\x4\xff\xf0\xcb\x9\x7c\x9d\xfa\x3\x6f\x81\x9\x31\x60\xfb\x8\xfa\x74\xd3\x64\x44\x7c\x55\x85\xec\x9c\x6e\x25\xb7\x6c\xc5\x37\xb6\x83\x87\x72\x95\x8b\x9d\xe1\x69\x5c\x31\x95\x42\xa6\x2c\xd1\x36\x47\x1f\xec\x54\xab\xa2\x1c\xd8\x0\xcc\xbc\xd\x65\xe2\x67\xbf\xbc\xea\xee\x9e\xe4\x36\x95\xbe\x73\xd9\xa6\xd9\xf\xa0\xcc\x82\x76\x26\xad\x5b\x58\x6c\x4e\xab\x29\x64\xd3\xd9\xa9\x8\x8c\x1d\xa1\x4f\x80\xd8\x3f\x94\xfb\xd3\x7b\xfc\xd1\x2b\xc3\x21\xeb\xe5\x1c\x84\x23\x7f\x4b\xfa\xdb\x34\x18\xa2\xc2\xe5\x13\xfe\x6c\x49\x81\xd2\x73\xe7\xe2\xd7\xe4\x4f\x4b\x8\x6e\xb1\x12\x22\x10\x9d\xac\x51\x1e\x17\xd9\x8a\xb\x42\x88\x16\x81\x37\x7c\x6a\xf7\xef\x2d\xe3\xd9\xf8\x5f\xe0\x53\x27\x74\xb9\xe2\xd6\x1c\x80\x2c\x52\x65"
}, {
	.key	= "\x47\x11\xeb\x86\x2b\x2c\xab\x44\x34\xda\x7f\x57\x3\x39\xc\xaf\x2c\x14\xfd\x65\x23\xe9\x8e\x74\xd5\x8\x68\x8\xe7\xb4\x72\xd7",
	.nonce	= "\xdb\x92\xf\x7f\x17\x54\xc\x30",
	.nlen	= 8,
	.assoc	= "\xd2\xa1\x70\xdb\x7a\xf8\xfa\x27\xba\x73\xf\xbf\x3d\x1e\x82\xb2",
	.alen	= 16,
	.input	= "\xe5\x26\xa4\x3d\xbd\x33\xd0\x4b\x6f\x5\xa7\x6e\x12\x7a\xd2\x74\xa6\xdd\xbd\x95\xeb\xf9\xa4\xf1\x59\x93\x91\x70\xd9\xfe\x9a\xcd\x53\x1f\x3a\xab\xa6\x7c\x9f\xa6\x9e\xbd\x99\xd9\xb5\x97\x44\xd5\x14\x48\x4d\x9d\xc0\xd0\x5\x96\xeb\x4c\x78\x55\x9\x8\x1\x2\x30\x90\x7b\x96\x7a\x7b\x5f\x30\x41\x24\xce\x68\x61\x49\x86\x57\x82\xdd\x53\x1c\x51\x28\x2b\x53\x6e\x2d\xc2\x20\x4c\xdd\x8f\x65\x10\x20\x50\xdd\x9d\x50\xe5\x71\x40\x53\x69\xfc\x77\x48\x11\xb9\xde\xa4\x8d\x58\xe4\xa6\x1a\x18\x47\x81\x7e\xfc\xdd\xf6\xef\xce\x2f\x43\x68\xd6\x6\xe2\x74\x6a\xad\x90\xf5\x37\xf3\x3d\x82\x69\x40\xe9\x6b\xa7\x3d\xa8\x1e\xd2\x2\x7c\xb7\x9b\xe4\xda\x8f\x95\x6\xc5\xdf\x73\xa3\x20\x9a\x49\xde\x9c\xbc\xee\x14\x3f\x81\x5e\xf8\x3b\x59\x3c\xe1\x68\x12\x5a\x3a\x76\x3a\x3f\xf7\x87\x33\xa\x1\xb8\xd4\xed\xb6\xbe\x94\x5e\x70\x40\x56\x67\x1f\x50\x44\x19\xce\x82\x70\x10\x87\x13\x20\xb\x4c\x5a\xb6\xf6\xa7\xae\x81\x75\x1\x81\xe6\x4b\x57\x7c\xdd\x6d\xf8\x1c\x29\x32\xf7\xda\x3c\x2d\xf8\x9b\x25\x6e\x0\xb4\xf7\x2f\xf7\x4\xf7\xa1\x56\xac\x4f\x1a\x64\xb8\x47\x55\x18\x7b\x7\x4d\xbd\x47\x24\x80\x5d\xa2\x70\xc5\xdd\x8e\x82\xd4\xeb\xec\xb2\xc\x39\xd2\x97\xc1\xcb\xeb\xf4\x77\x59\xb4\x87\xef\xcb\x43\x2d\x46\x54\xd1\xa7\xd7\x15\x99\xa\x43\xa1\xe0\x99\x33\x71\xc1\xed\xfe\x72\x46\x33\x8e\x91\x8\x9f\xc8\x2e\xca\xfa\xdc\x59\xd5\xc3\x76\x84\x9f\xa3\x37\x68\xc3\xf0\x47\x2c\x68\xdb\x5e\xc3\x49\x4c\xe8\x92\x85\xe2\x23\xd3\x3f\xad\x32\xe5\x2b\x82\xd7\x8f\x99\xa\x59\x5c\x45\xd9\xb4\x51\x52\xc2\xae\xbf\x80\xcf\xc9\xc9\x51\x24\x2a\x3b\x3a\x4d\xae\xeb\xbd\x22\xc3\xe\xf\x59\x25\x92\x17\xe9\x74\xc7\x8b\x70\x70\x36\x55\x95\x75\x4b\xad\x61\x2b\x9\xbc\x82\xf2\x6e\x94\x43\xae\xc3\xd5\xcd\x8e\xfe\x5b\x9a\x88\x43\x1\x75\xb2\x23\x9\xf7\x89\x83\xe7\xfa\xf9\xb4\x9b\xf8\xef\xbd\x1c\x92\xc1\xda\x7e\xfe\x5\xba\x5a\xcd\x7\x6a\x78\x9e\x5d\xfb\x11\x2f\x79\x38\xb6\xc2\x5b\x6b\x51\xb4\x71\xdd\xf7\x2a\xe4\xf4\x72\x76\xad\xc2\xdd\x64\x5d\x79\xb6\xf5\x7a\x77\x20\x5\x3d\x30\x6\xd4\x4c\xa\x2c\x98\x5a\xb9\xd4\x98\xa9\x3f\xc6\x12\xea\x3b\x4b\xc5\x79\x64\x63\x6b\x9\x54\x3b\x14\x27\xba\x99\x80\xc8\x72\xa8\x12\x90\x29\xba\x40\x54\x97\x2b\x7b\xfe\xeb\xcd\x1\x5\x44\x72\xdb\x99\xe4\x61\xc9\x69\xd6\xb9\x28\xd1\x5\x3e\xf9\xb\x49\xa\x49\xe9\x8d\xe\xa7\x4a\xf\xaf\x32\xd0\xe0\xb2\x3a\x55\x58\xfe\x5c\x28\x70\x51\x23\xb0\x7b\x6a\x5f\x1e\xb8\x17\xd7\x94\x15\x8f\xee\x20\xc7\x42\x25\x3e\x9a\x14\xd7\x60\x72\x39\x47\x48\xa9\xfe\xdd\x47\xa\xb1\xe6\x60\x28\x8c\x11\x68\xe1\xff\xd7\xce\xc8\xbe\xb3\xfe\x27\x30\x9\x70\xd7\xfa\x2\x33\x3a\x61\x2e\xc7\xff\xa4\x2a\xa8\x6e\xb4\x79\x35\x6d\x4c\x1e\x38\xf8\xee\xd4\x84\x4e\x6e\x28\xa7\xce\xc8\xc1\xcf\x80\x5\xf3\x4\xef\xc8\x18\x28\x2e\x8d\x5e\xc\xdf\xb8\x5f\x96\xe8\xc6\x9c\x2f\xe5\xa6\x44\xd7\xe7\x99\x44\xc\xec\xd7\x5\x60\x97\xbb\x74\x77\x58\xd5\xbb\x48\xde\x5a\xb2\x54\x7f\xe\x46\x70\x6a\x6f\x78\xa5\x8\x89\x5\x4e\x7e\xa0\x69\xb4\x40\x60\x55\x77\x75\x9b\x19\xf2\xd5\x13\x80\x77\xf9\x4b\x3f\x1e\xee\xe6\x76\x84\x7b\x8c\xe5\x27\xa8\xa\x91\x1\x68\x71\x8a\x3f\x6\xab\xf6\xa9\xa5\xe6\x72\x92\xe4\x67\xe2\xa2\x46\x35\x84\x55\x7d\xca\xa8\x85\xd0\xf1\x3f\xbe\xd7\x34\x64\xfc\xae\xe3\xe4\x4\x9f\x66\x2\xb9\x88\x10\xd9\xc4\x4c\x31\x43\x7a\x93\xe2\x9b\x56\x43\x84\xdc\xdc\xde\x1d\xa4\x2\xe\xc2\xef\xc3\xf8\x78\xd1\xb2\x6b\x63\x18\xc9\xa9\xe5\x72\xd8\xf3\xb9\xd1\x8a\xc7\x1a\x2\x27\x20\x77\x10\xe5\xc8\xd4\x4a\x47\xe5\xdf\x5f\x1\xaa\xb0\xd4\x10\xbb\x69\xe3\x36\xc8\xe1\x3d\x43\xfb\x86\xcd\xcc\xbf\xf4\x88\xe0\x20\xca\xb7\x1b\xf1\x2f\x5c\xee\xd4\xd3\xa3\xcc\xa4\x1e\x1c\x47\xfb\xbf\xfc\xa2\x41\x55\x9d\xf6\x5a\x5e\x65\x32\x34\x7b\x52\x8d\xd5\xd0\x20\x60\x3\xab\x3f\x8c\xd4\x21\xea\x2a\xd9\xc4\xd0\xd3\x65\xd8\x7a\x13\x28\x62\x32\x4b\x2c\x87\x93\xa8\xb4\x52\x45\x9\x44\xec\xec\xc3\x17\xdb\x9a\x4d\x5c\xa9\x11\xd4\x7d\xaf\x9e\xf1\x2d\xb2\x66\xc5\x1d\xed\xb7\xcd\xb\x25\x5e\x30\x47\x3f\x40\xf4\xa1\xa0\x0\x94\x10\xc5\x6a\x63\x1a\xd5\x88\x92\x8e\x82\x39\x87\x3c\x78\x65\x58\x42\x75\x5b\xdd\x77\x3e\x9\x4e\x76\x5b\xe6\xe\x4d\x38\xb2\xc0\xb8\x95\x1\x7a\x10\xe0\xfb\x7\xf2\xab\x2d\x8c\x32\xed\x2b\xc0\x46\xc2\xf5\x38\x83\xf0\x17\xec\xc1\x20\x6a\x9a\xb\x0\xa0\x98\x22\x50\x23\xd5\x80\x6b\xf6\x1f\xc3\xcc\x97\xc9\x24\x9f\xf3\xaf\x43\x14\xd5\xa0",
	.ilen	= 1040,
	.result	= "\x42\x93\xe4\xeb\x97\xb0\x57\xbf\x1a\x8b\x1f\xe4\x5f\x36\x20\x3c\xef\xa\xa9\x48\x5f\x5f\x37\x22\x3a\xde\xe3\xae\xbe\xad\x7\xcc\xb1\xf6\xf5\xf9\x56\xdd\xe7\x16\x1e\x7f\xdf\x7a\x9e\x75\xb7\xc7\xbe\xbe\x8a\x36\x4\xc0\x10\xf4\x95\x20\x3\xec\xdc\x5\xa1\x7d\xc4\xa9\x2c\x82\xd0\xbc\x8b\xc5\xc7\x45\x50\xf6\xa2\x1a\xb5\x46\x3b\x73\x2\xa6\x83\x4b\x73\x82\x58\x5e\x3b\x65\x2f\xe\xfd\x2b\x59\x16\xce\xa1\x60\x9c\xe8\x3a\x99\xed\x8d\x5a\xcf\xf6\x83\xaf\xba\xd7\x73\x73\x40\x97\x3d\xca\xef\x7\x57\xe6\xd9\x70\xe\x95\xae\xa6\x8d\x4\xcc\xee\xf7\x9\x31\x77\x12\xa3\x23\x97\x62\xb3\x7b\x32\xfb\x80\x14\x48\x81\xc3\xe5\xea\x91\x39\x52\x81\xa2\x4f\xe4\xb3\x9\xff\xde\x5e\xe9\x58\x84\x6e\xf9\x3d\xdf\x25\xea\xad\xae\xe6\x9a\xd1\x89\x55\xd3\xde\x6c\x52\xdb\x70\xfe\x37\xce\x44\xa\xa8\x25\x5f\x92\xc1\x33\x4a\x4f\x9b\x62\x35\xff\xce\xc0\xa9\x60\xce\x52\x0\x97\x51\x35\x26\x2e\xb9\x36\xa9\x87\x6e\x1e\xcc\x91\x78\x53\x98\x86\x5b\x9c\x74\x7d\x88\x33\xe1\xdf\x37\x69\x2b\xbb\xf1\x4d\xf4\xd1\xf1\x39\x93\x17\x51\x19\xe3\x19\x1e\x76\x37\x25\xfb\x9\x27\x6a\xab\x67\x6f\x14\x12\x64\xe7\xc4\x7\xdf\x4d\x17\xbb\x6d\xe0\xe9\xb9\xab\xca\x10\x68\xaf\x7e\xb7\x33\x54\x73\x7\x6e\xf7\x81\x97\x9c\x5\x6f\x84\x5f\xd2\x42\xfb\x38\xcf\xd1\x2f\x14\x30\x88\x98\x4d\x5a\xa9\x76\xd5\x4f\x3e\x70\x6c\x85\x76\xd7\x1\xa0\x1a\xc8\x4e\xaa\xac\x78\xfe\x46\xde\x6a\x5\x46\xa7\x43\xc\xb9\xde\xb9\x68\xfb\xce\x42\x99\x7\x4d\xb\x3b\x5a\x30\x35\xa8\xf9\x3a\x73\xef\xf\xdb\x1e\x16\x42\xc4\xba\xae\x58\xaa\xf8\xe5\x75\x2f\x1b\x15\x5c\xfd\xa\x97\xd0\xe4\x37\x83\x61\x5f\x43\xa6\xc7\x3f\x38\x59\xe6\xeb\xa3\x90\xc3\xaa\xaa\x5a\xd3\x34\xd4\x17\xc8\x65\x3e\x57\xbc\x5e\xdd\x9e\xb7\xf0\x2e\x5b\xb2\x1f\x8a\x8\xd\x45\x91\xb\x29\x53\x4f\x4c\x5a\x73\x56\xfe\xaf\x41\x1\x39\xa\x24\x3c\x7e\xbe\x4e\x53\xf3\xeb\x6\x66\x51\x28\x1d\xbd\x41\xa\x1\xab\x16\x47\x27\x47\x47\xf7\xcb\x46\xa\x70\x9e\x1\x9c\x9\xe1\x2a\x0\x1a\xd8\xd4\x79\x9d\x80\x15\x8e\x53\x2a\x65\x83\x78\x3e\x3\x0\x7\x12\x1f\x33\x3e\x7b\x13\x37\xf1\xc3\xef\xb7\xc1\x20\x3c\x3e\x67\x66\x5d\x88\xa7\x7d\x33\x50\x77\xb0\x28\x8e\xe7\x2c\x2e\x7a\xf4\x3c\x8d\x74\x83\xaf\x8e\x87\xf\xe4\x50\xff\x84\x5c\x47\xc\x6a\x49\xbf\x42\x86\x77\x15\x48\xa5\x90\x5d\x93\xd6\x2a\x11\xd5\xd5\x11\xaa\xce\xe7\x6f\xa5\xb0\x9\x2c\x8d\xd3\x92\xf0\x5a\x2a\xda\x5b\x1e\xd5\x9a\xc4\xc4\xf3\x49\x74\x41\xca\xe8\xc1\xf8\x44\xd6\x3c\xae\x6c\x1d\x9a\x30\x4\x4d\x27\xe\xb1\x5f\x59\xa2\x24\xe8\xe1\x98\xc5\x6a\x4c\xfe\x41\xd2\x27\x42\x52\xe1\xe9\x7d\x62\xe4\x88\xf\xad\xb2\x70\xcb\x9d\x4c\x27\x2e\x76\x1e\x1a\x63\x65\xf5\x3b\xf8\x57\x69\xeb\x5b\x38\x26\x39\x33\x25\x45\x3e\x91\xb8\xd8\xc7\xd5\x42\xc0\x22\x31\x74\xf4\xbc\xc\x23\xf1\xca\xc1\x8d\xd7\xbe\xc9\x62\xe4\x8\x1a\xcf\x36\xd5\xfe\x55\x21\x59\x91\x87\x87\xdf\x6\xdb\xdf\x96\x45\x58\xda\x5\xcd\x50\x4d\xd2\x7d\x5\x18\x73\x6a\x8d\x11\x85\xa6\x88\xe8\xda\xe6\x30\x33\xa4\x89\x31\x75\xbe\x69\x43\x84\x43\x50\x87\xdd\x71\x36\x83\xc3\x78\x74\x24\xa\xed\x7b\xdb\xa4\x24\xb\xb9\x7e\x5d\xff\xde\xb1\xef\x61\x5a\x45\x33\xf6\x17\x7\x8\x98\x83\x92\xf\x23\x6d\xe6\xaa\x17\x54\xad\x6a\xc8\xdb\x26\xbe\xb8\xb6\x8\xfa\x68\xf1\xd7\x79\x6f\x18\xb4\x9e\x2d\x3f\x1b\x64\xaf\x8d\x6\xe\x49\x28\xe0\x5d\x45\x68\x13\x87\xfa\xde\x40\x7b\xd2\xc3\x94\xd5\xe1\xd9\xc2\xaf\x55\x89\xeb\xb4\x12\x59\xa8\xd4\xc5\x29\x66\x38\xe6\xac\x22\x22\xd9\x64\x9b\x34\xa\x32\x9f\xc2\xbf\x17\x6c\x3f\x71\x7a\x38\x6b\x98\xfb\x49\x36\x89\xc9\xe2\xd6\xc7\x5d\xd0\x69\x5f\x23\x35\xc9\x30\xe2\xfd\x44\x58\x39\xd7\x97\xfb\x5c\x0\xd5\x4f\x7a\x1a\x95\x8b\x62\x4b\xce\xe5\x91\x21\x7b\x30\x0\xd6\xdd\x6d\x2\x86\x49\xf\x3c\x1a\x27\x3c\xd3\xe\x71\xf2\xff\xf5\x2f\x87\xac\x67\x59\x81\xa3\xf7\xf8\xd6\x11\xc\x84\xa9\x3\xee\x2a\xc4\xf3\x22\xab\x7c\xe2\x25\xf5\x67\xa3\xe4\x11\xe0\x59\xb3\xca\x87\xa0\xae\xc9\xa6\x62\x1b\x6e\x4d\x2\x6b\x7\x9d\xfd\xd0\x92\x6\xe1\xb2\x9a\x4a\x1f\x1f\x13\x49\x99\x97\x8\xde\x7f\x98\xaf\x51\x98\xee\x2c\xcb\xf0\xb\xc6\xb6\xb7\x2d\x9a\xb1\xac\xa6\xe3\x15\x77\x9d\x6b\x1a\xe4\xfc\x8b\xf2\x17\x59\x8\x4\x58\x81\x9d\x1b\x1b\x69\x55\xc2\xb4\x3c\x1f\x50\xf1\x7f\x77\x90\x4c\x66\x40\x5a\xc0\x33\x1f\xcb\x5\x6d\x5c\x6\x87\x52\xa2\x8f\x26\xd5\x4f"
}, {
	.key	= "\x35\x4e\xb5\x70\x50\x42\x8a\x85\xf2\xfb\xed\x7b\xd0\x9e\x97\xca\xfa\x98\x66\x63\xee\x37\xcc\x52\xfe\xd1\xdf\x95\x15\x34\x29\x38",
	.nonce	= "\xfd\x87\xd4\xd8\x62\xfd\xec\xaa",
	.nlen	= 8,
	.assoc	= "\xd6\x31\xda\x5d\x42\x5e\xd7",
	.alen	= 7,
	.input	= "\x6a\xfc\x4b\x25\xdf\xc0\xe4\xe8\x17\x4d\x4c\xc9\x7e\xde\x3a\xcc\x3c\xba\x6a\x77\x47\xdb\xe3\x74\x7a\x4d\x5f\x8d\x37\x55\x80\x73\x90\x66\x5d\x3a\x7d\x5d\x86\x5e\x8d\xfd\x83\xff\x4e\x74\x6f\xf9\xe6\x70\x17\x70\x3e\x96\xa7\x7e\xcb\xab\x8f\x58\x24\x9b\x1\xfd\xcb\xe6\x4d\x9b\xf0\x88\x94\x57\x66\xef\x72\x4c\x42\x6e\x16\x19\x15\xea\x70\x5b\xac\x13\xdb\x9f\x18\xe2\x3c\x26\x97\xbc\xdc\x45\x8c\x6c\x24\x69\x9c\xf7\x65\x1e\x18\x59\x31\x7c\xe4\x73\xbc\x39\x62\xc6\x5c\x9f\xbf\xfa\x90\x3\xc9\x72\x26\xb6\x1b\xc2\xb7\x3f\xf2\x13\x77\xf2\x8d\xb9\x47\xd0\x53\xdd\xc8\x91\x83\x8b\xb1\xce\xa3\xfe\xcd\xd9\xdd\x92\x7b\xdb\xb8\xfb\xc9\x2d\x1\x59\x39\x52\xad\x1b\xec\xcf\xd7\x70\x13\x21\xf5\x47\xaa\x18\x21\x5c\xc9\x9a\xd2\x6b\x5\x9c\x1\xa1\xda\x35\x5d\xb3\x70\xe6\xa9\x80\x8b\x91\xb7\xb3\x5f\x24\x9a\xb7\xd1\x6b\xa1\x1c\x50\xba\x49\xe0\xee\x2e\x75\xac\x69\xc0\xeb\x3\xdd\x19\xe5\xf6\x6\xdd\xc3\xd7\x2b\x7\x7\x30\xa7\x19\xc\xbf\xe6\x18\xcc\xb1\x1\x11\x85\x77\x1d\x96\xa7\xa3\x0\x84\x2\xa2\x83\x68\xda\x17\x27\xc8\x7f\x23\xb7\xf4\x13\x85\xcf\xdd\x7a\x7d\x24\x57\xfe\x5\x93\xf5\x74\xce\xed\xc\x20\x98\x8d\x92\x30\xa1\x29\x23\x1a\xa0\x4f\x69\x56\x4c\xe1\xc8\xce\xf6\x9a\xc\xa4\xfa\x4\xf6\x62\x95\xf2\xfa\xc7\x40\x68\x40\x8f\x41\xda\xb4\x26\x6f\x70\xab\x40\x61\xa4\xe\x75\xfb\x86\xeb\x9d\x9a\x1f\xec\x76\x99\xe7\xea\xaa\x1e\x2d\xb5\xd4\xa6\x1a\xb8\x61\xa\x1d\x16\x5b\x98\xc2\x31\x40\xe7\x23\x1d\x66\x99\xc8\xc0\xd7\xce\xf3\x57\x40\x4\x3f\xfc\xea\xb3\xfc\xd2\xd3\x99\xa4\x94\x69\xa0\xef\xd1\x85\xb3\xa6\xb1\x28\xbf\x94\x67\x22\xc3\x36\x46\xf8\xd2\xf\x5f\xf4\x59\x80\xe6\x2d\x43\x8\x7d\x19\x9\x97\xa7\x4c\x3d\x8d\xba\x65\x62\xa3\x71\x33\x29\x62\xdb\xc1\x33\x34\x1a\x63\x33\x16\xb6\x64\x7e\xab\x33\xf0\xe6\x26\x68\xba\x1d\x2e\x38\x8\xe6\x2\xd3\x25\x2c\x47\x23\x58\x34\xf\x9d\x63\x4f\x63\xbb\x7f\x3b\x34\x38\xa7\xb5\x8d\x65\xd9\x9f\x79\x55\x3e\x4d\xe7\x73\xd8\xf6\x98\x97\x84\x60\x9c\xc8\xa9\x3c\xf6\xdc\x12\x5c\xe1\xbb\xb\x8b\x98\x9c\x9d\x26\x7c\x4a\xe6\x46\x36\x58\x21\x4a\xee\xca\xd7\x3b\xc2\x6c\x49\x2f\xe5\xd5\x3\x59\x84\x53\xcb\xfe\x92\x71\x2e\x7c\x21\xcc\x99\x85\x7f\xb8\x74\x90\x13\x42\x3f\xe0\x6b\x1d\xf2\x4d\x54\xd4\xfc\x3a\x5\xe6\x74\xaf\xa6\xa0\x2a\x20\x23\x5d\x34\x5c\xd9\x3e\x4e\xfa\x93\xe7\xaa\xe9\x6f\x8\x43\x67\x41\xc5\xad\xfb\x31\x95\x82\x73\x32\xd8\xa6\xa3\xed\xe\x2d\xf6\x5f\xfd\x80\xa6\x7a\xe0\xdf\x78\x15\x29\x74\x33\xd0\x9e\x83\x86\x72\x22\x57\x29\xb9\x9e\x5d\xd3\x1a\xb5\x96\x72\x41\x3d\xf1\x64\x43\x67\xee\xaa\x5c\xd3\x9a\x96\x13\x11\x5d\xf3\xc\x87\x82\x1e\x41\x9e\xd0\x27\xd7\x54\x3b\x67\x73\x9\x91\xe9\xd5\x36\xa7\xb5\x55\xe4\xf3\x21\x51\x49\x22\x7\x55\x4f\x44\x4b\xd2\x15\x93\x17\x2a\xfa\x4d\x4a\x57\xdb\x4c\xa6\xeb\xec\x53\x25\x6c\x21\xed\x0\x4c\x3b\xca\x14\x57\xa9\xd6\x6a\xcd\x8d\x5e\x74\xac\x72\xc1\x97\xe5\x1b\x45\x4e\xda\xfc\xcc\x40\xe8\x48\x88\xb\xa3\xe3\x8d\x83\x42\xc3\x23\xfd\x68\xb5\x8e\xf1\x9d\x63\x77\xe9\xa3\x8e\x8c\x26\x6b\xbd\x72\x73\x35\xc\x3\xf8\x43\x78\x52\x71\x15\x1f\x71\x5d\x6e\xed\xb9\xcc\x86\x30\xdb\x2b\xd3\x82\x88\x23\x71\x90\x53\x5c\xa9\x2f\x76\x1\xb7\x9a\xfe\x43\x55\xa3\x4\x9b\xe\xe4\x59\xdf\xc9\xe9\xb1\xea\x29\x28\x3c\x5c\xae\x72\x84\xb6\xc6\xeb\xc\x27\x7\x74\x90\xd\x31\xb0\x0\x77\xe9\x40\x70\x6f\x68\xa7\xfd\x6\xec\x4b\xc0\xb7\xac\xbc\x33\xb7\x6d\xa\xbd\x12\x1b\x59\xcb\xdd\x32\xf5\x1d\x94\x57\x76\x9e\xc\x18\x98\x71\xd7\x2a\xdb\xb\x7b\xa7\x71\xb7\x67\x81\x23\x96\xae\xb9\x7e\x32\x43\x92\x8a\x19\xa0\xc4\xd4\x3b\x57\xf9\x4a\x2c\xfb\x51\x46\xbb\xcb\x5d\xb3\xef\x13\x93\x6e\x68\x42\x54\x57\xd3\x6a\x3a\x8f\x9d\x66\xbf\xbd\x36\x23\xf5\x93\x83\x7b\x9c\xc0\xdd\xc5\x49\xc0\x64\xed\x7\x12\xb3\xe6\xe4\xe5\x38\x95\x23\xb1\xa0\x3b\x1a\x61\xda\x17\xac\xc3\x58\xdd\x74\x64\x22\x11\xe8\x32\x1d\x16\x93\x85\x99\xa5\x9c\x34\x55\xb1\xe9\x20\x72\xc9\x28\x7b\x79\x0\xa1\xa6\xa3\x27\x40\x18\x8a\x54\xe0\xcc\xe8\x4e\x8e\x43\x96\xe7\x3f\xc8\xe9\xb2\xf9\xc9\xda\x4\x71\x50\x47\xe4\xaa\xce\xa2\x30\xc8\xe4\xac\xc7\xd\x6\x2e\xe6\xe8\x80\x36\x29\x9e\x1\xb8\xc3\xf0\xa0\x5d\x7a\xca\x4d\xa0\x57\xbd\x2a\x45\xa7\x7f\x9c\x93\x7\x8f\x35\x67\x92\xe3\xe9\x7f\xa8\x61\x43\x9e\x25\x4f\x33\x76\x13\x6e\x12\xb9\xdd\xa4\x7c\x8\x9f\x7c\xe7\xa\x8d\x84\x6\xa4\x33\x17\x34\x5e\x10\x7c\xc0\xa8\x3d\x1f\x42\x20\x51\x65\x5d\x9\xc3\xaa\xc0\xc8\xd\xf0\x79\xbc\x20\x1b\x95\xe7\x6\x7d\x47\x20\x3\x1a\x74\xdd\xe2\xd4\xae\x38\x71\x9b\xf5\x80\xec\x8\x4e\x56\xba\x76\x12\x1a\xdf\x48\xf3\xae\xb3\xe6\xe6\xbe\xc0\x91\x2e\x1\xb3\x1\x86\xa2\xb9\x52\xd1\x21\xae\xd4\x97\x1d\xef\x41\x12\x95\x3d\x48\x45\x1c\x56\x32\x8f\xb8\x43\xbb\x19\xf3\xca\xe9\xeb\x6d\x84\xbe\x86\x6\xe2\x36\xb2\x62\x9d\xd3\x4c\x48\x18\x54\x13\x4e\xcf\xfd\xba\x84\xb9\x30\x53\xcf\xfb\xb9\x29\x8f\xdc\x9f\xef\x60\xb\x64\xf6\x8b\xee\xa6\x91\xc2\x41\x6c\xf6\xfa\x79\x67\x4b\xc1\x3f\xaf\x9\x81\xd4\x5d\xcb\x9\xdf\x36\x31\xc0\x14\x3c\x7c\xe\x65\x95\x99\x6d\xa3\xf4\xd7\x38\xee\x1a\x2b\x37\xe2\xa4\x3b\x4b\xd0\x65\xca\xf8\xc3\xe8\x15\x20\xef\xf2\x0\xfd\x1\x9\xc5\xc8\x17\x4\x93\xd0\x93\x3\x55\xc5\xfe\x32\xa3\x3e\x28\x2d\x3b\x93\x8a\xcc\x7\x72\x80\x8b\x74\x16\x24\xbb\xda\x94\x39\x30\x8f\xb1\xcd\x4a\x90\x92\x7c\x14\x8f\x95\x4e\xac\x9b\xd8\x8f\x1a\x87\xa4\x32\x27\x8a\xba\xf7\x41\xcf\x84\x37\x19\xe6\x6\xf5\xe\xcf\x36\xf5\x9e\x6c\xde\xbc\xff\x64\x7e\x4e\x59\x57\x48\xfe\x14\xf7\x9c\x93\x5d\x15\xad\xcc\x11\xb1\x17\x18\xb2\x7e\xcc\xab\xe9\xce\x7d\x77\x5b\x51\x1b\x1e\x20\xa8\x32\x6\xe\x75\x93\xac\xdb\x35\x37\x1f\xe9\x19\x1d\xb4\x71\x97\xd6\x4e\x2c\x8\xa5\x13\xf9\xe\x7e\x78\x6e\x14\xe0\xa9\xb9\x96\x4c\x80\x82\xba\x17\xb3\x9d\x69\xb0\x84\x46\xff\xf9\x52\x79\x94\x58\x3a\x62\x90\x15\x35\x71\x10\x37\xed\xa1\x8e\x53\x6e\xf4\x26\x57\x93\x15\x93\xf6\x81\x2c\x5a\x10\xda\x92\xad\x2f\xdb\x28\x31\x2d\x55\x4\xd2\x6\x28\x8c\x1e\xdc\xea\x54\xac\xff\xb7\x6c\x30\x15\xd4\xb4\xd\x0\x93\x57\xdd\xd2\x7\x7\x6\xd9\x43\x9b\xcd\x3a\xf4\x7d\x4c\x36\x5d\x23\xa2\xcc\x57\x40\x91\xe9\x2c\x2f\x2c\xd5\x30\x9b\x17\xb0\xc9\xf7\xa7\x2f\xd1\x93\x20\x6b\xc6\xc1\xe4\x6f\xcb\xd1\xe7\x9\xf\x9e\xdc\xaa\x9f\x2f\xdf\x56\x9f\xd4\x33\x4\xaf\xd3\x6c\x58\x61\xf0\x30\xec\xf2\x7f\xf2\x9c\xdf\x39\xbb\x6f\xa2\x8c\x7e\xc4\x22\x51\x71\xc0\x4d\x14\x1a\xc4\xcd\x4\xd9\x87\x8\x50\x5\xcc\xaf\xf6\xf0\x8f\x92\x54\x58\xc2\xc7\x9\x7a\x59\x2\x5\xe8\xb0\x86\xd9\xbf\x7b\x35\x51\x4d\xaf\x8\x97\x2c\x65\xda\x2a\x71\x3a\xa8\x51\xcc\xf2\x73\x27\xc3\xfd\x62\xcf\xe3\xb2\xca\xcb\xbe\x1a\xa\xa1\x34\x7b\x77\xc4\x62\x68\x78\x5f\x94\x7\x4\x65\x16\x4b\x61\xcb\xff\x75\x26\x50\x66\x1f\x6e\x93\xf8\xc5\x51\xeb\xa4\x4a\x48\x68\x6b\xe2\x5e\x44\xb2\x50\x2c\x6c\xae\x79\x4e\x66\x35\x81\x50\xac\xbc\x3f\xb1\xc\xf3\x5\x3c\x4a\xa3\x6c\x2a\x79\xb4\xb7\xab\xca\xc7\x9b\x8e\xcd\x5f\x11\x3\xcb\x30\xa3\xab\xda\xfe\x64\xb9\xbb\xd8\x5e\x3a\x1a\x56\xe5\x5\x48\x90\x1e\x61\x69\x1b\x22\xe6\x1a\x3c\x75\xad\x1f\x37\x28\xdc\xe4\x6d\xbd\x42\xdc\xd3\xc8\xb6\x1c\x48\xfe\x94\x77\x7f\xbd\x62\xac\xa3\x47\x27\xcf\x5f\xd9\xdb\xaf\xec\xf7\x5e\xc1\xb0\x9d\x1\x26\x99\x7e\x8f\x3\x70\xb5\x42\xbe\x67\x28\x1b\x7c\xbd\x61\x21\x97\xcc\x5c\xe1\x97\x8f\x8d\xde\x2b\xaa\xa7\x71\x1d\x1e\x2\x73\x70\x58\x32\x5b\x1d\x67\x3d\xe0\x74\x4f\x3\xf2\x70\x51\x79\xf1\x61\x70\x15\x74\x9d\x23\x89\xde\xac\xfd\xde\xd0\x1f\xc3\x87\x44\x35\x4b\xe5\xb0\x60\xc5\x22\xe4\x9e\xca\xeb\xd5\x3a\x9\x45\xa4\xdb\xfa\x3f\xeb\x1b\xc7\xc8\x14\x99\x51\x92\x10\xed\xed\x28\xe0\xa1\xf8\x26\xcf\xcd\xcb\x63\xa1\x3b\xe3\xdf\x7e\xfe\xa6\xf0\x81\x9a\xbf\x55\xde\x54\xd5\x56\x60\x98\x10\x68\xf4\x38\x96\x8e\x6f\x1d\x44\x7f\xd6\x2f\xfe\x55\xfb\xc\x7e\x67\xe2\x61\x44\xed\xf2\x35\x30\x5d\xe9\xc7\xd6\x6d\xe0\xa0\xed\xf3\xfc\xd8\x3e\xa\x7b\xcd\xaf\x65\x68\x18\xc0\xec\x4\x1c\x74\x6d\xe2\x6e\x79\xd4\x11\x2b\x62\xd5\x27\xad\x4f\x1\x59\x73\xcc\x6a\x53\xfb\x2d\xd5\x4e\x99\x21\x65\x4d\xf5\x82\xf7\xd8\x42\xce\x6f\x3d\x36\x47\xf1\x5\x16\xe8\x1b\x6a\x8f\x93\xf2\x8f\x37\x40\x12\x28\xa3\xe6\xb9\x17\x4a\x1f\xb1\xd1\x66\x69\x86\xc4\xfc\x97\xae\x3f\x8f\x1e\x2b\xdf\xcd\xf9\x3c",
	.ilen	= 1949,
	.result	= "\x7a\x57\xf2\xc7\x6\x3f\x50\x7b\x36\x1a\x66\x5c\xb9\xe\x5e\x3b\x45\x60\xbe\x9a\x31\x9f\xff\x5d\x66\x34\xb4\xdc\xfb\x9d\x8e\xee\x6a\x33\xa4\x7\x3c\xf9\x4c\x30\xa1\x24\x52\xf9\x50\x46\x88\x20\x2\x32\x3a\xe\x99\x63\xaf\x1f\x15\x28\x2a\x5\xff\x57\x59\x5e\x18\xa1\x1f\xd0\x92\x5c\x88\x66\x1b\x0\x64\xa5\x93\x8d\x6\x46\xb0\x64\x8b\x8b\xef\x99\x5\x35\x85\xb3\xf3\x33\xbb\xec\x66\xb6\x3d\x57\x42\xe3\xb4\xc6\xaa\xb0\x41\x2a\xb9\x59\xa9\xf6\x3e\x15\x26\x12\x3\x21\x4c\x74\x43\x13\x2a\x3\x27\x9\xb4\xfb\xe7\xb7\x40\xff\x5e\xce\x48\x9a\x60\xe3\x8b\x80\x8c\x38\x2d\xcb\x93\x37\x74\x5\x52\x6f\x73\x3e\xc3\xbc\xca\x72\xa\xeb\xf1\x3b\xa0\x95\xdc\x8a\xc4\xa9\xdc\xca\x44\xd8\x8\x63\x6a\x36\xd3\x3c\xb8\xac\x46\x7d\xfd\xaa\xeb\x3e\xf\x45\x8f\x49\xda\x2b\xf2\x12\xbd\xaf\x67\x8a\x63\x48\x4b\x55\x5f\x6d\x8c\xb9\x76\x34\x84\xae\xc2\xfc\x52\x64\x82\xf7\xb0\x6\xf0\x45\x73\x12\x50\x30\x72\xea\x78\x9a\xa8\xaf\xb5\xe3\xbb\x77\x52\xec\x59\x84\xbf\x6b\x8f\xce\x86\x5e\x1f\x23\xe9\xfb\x8\x86\xf7\x10\xb9\xf2\x44\x96\x44\x63\xa9\xa8\x78\x0\x23\xd6\xc7\xe7\x6e\x66\x4f\xcc\xee\x15\xb3\xbd\x1d\xa0\xe5\x9c\x1b\x24\x2c\x4d\x3c\x62\x35\x9c\x88\x59\x9\xdd\x82\x1b\xcf\xa\x83\x6b\x3f\xae\x3\xc4\xb4\xdd\x7e\x5b\x28\x76\x25\x96\xd9\xc9\x9d\x5f\x86\xfa\xf6\xd7\xd2\xe6\x76\x1d\xf\xa1\xdc\x74\x5\x1b\x1d\xe0\xcd\x16\xb0\xa8\x8a\x34\x7b\x15\x11\x77\xe5\x7b\x7e\x20\xf7\xda\x38\xda\xce\x70\xe9\xf5\x6c\xd9\xbe\xc\x4c\x95\x4c\xc2\x9b\x34\x55\x55\xe1\xf3\x46\x8e\x48\x74\x14\x4f\x9d\xc9\xf5\xe8\x1a\xf0\x11\x4a\xc1\x8d\xe0\x93\xa0\xbe\x9\x1c\x2b\x4e\xf\xb2\x87\x8b\x84\xfe\x92\x32\x14\xd7\x93\xdf\xe7\x44\xbc\xc5\xae\x53\x69\xd8\xb3\x79\x37\x80\xe3\x17\x5c\xec\x53\x0\x9a\xe3\x8e\xdc\x38\xb8\x66\xf0\xd3\xad\x1d\x2\x96\x86\x3e\x9d\x3b\x5d\xa5\x7f\x21\x10\xf1\x1f\x13\x20\xf9\x57\x87\x20\xf5\x5f\xf1\x17\x48\xa\x51\x5a\xcd\x19\x3\xa6\x5a\xd1\x12\x97\xe9\x48\xe2\x1d\x83\x75\x50\xd9\x75\x7d\x6a\x82\xa1\xf9\x4e\x54\x87\x89\xc9\xc\xb7\x5b\x6a\x91\xc1\x9c\xb2\xa9\xdc\x9a\xa4\x49\xa\x6d\xd\xbb\xde\x86\x44\xdd\x5d\x89\x2b\x96\xf\x23\x95\xad\xcc\xa2\xb3\xb9\x7e\x74\x38\xba\x9f\x73\xae\x5f\xf8\x68\xa2\xe0\xa9\xce\xbd\x40\xd4\x4c\x6b\xd2\x56\x62\xb0\xcc\x63\x7e\x5b\xd3\xae\xd1\x75\xce\xbb\xb4\x5b\xa8\xf8\xb4\xac\x71\x75\xaa\xc9\x9f\xbb\x6c\xad\xf\x55\x5d\xe8\x85\x7d\xf9\x21\x35\xea\x92\x85\x2b\x0\xec\x84\x90\xa\x63\x96\xe4\x6b\xa9\x77\xb8\x91\xf8\x46\x15\x72\x63\x70\x1\x40\xa3\xa5\x76\x62\x2b\xbf\xf1\xe5\x8d\x9f\xa3\xfa\x9b\x3\xbe\xfe\x65\x6f\xa2\x29\xd\x54\xb4\x71\xce\xa9\xd6\x3d\x88\xf9\xaf\x6b\xa8\x9e\xf4\x16\x96\x36\xb9\x0\xdc\x10\xab\xb5\x8\x31\x1f\x0\xb1\x3c\xd9\x38\x3e\xc6\x4\xa7\x4e\xe8\xae\xed\x98\xc2\xf7\xb9\x0\x5f\x8c\x60\xd1\xe5\x15\xf7\xae\x1e\x84\x88\xd1\xf6\xbc\x3a\x89\x35\x22\x83\x7c\xca\xf0\x33\x82\x4c\x79\x3c\xfd\xb1\xae\x52\x62\x55\xd2\x41\x60\xc6\xbb\xfa\xe\x59\xd6\xa8\xfe\x5d\xed\x47\x3d\xe0\xea\x1f\x6e\x43\x51\xec\x10\x52\x56\x77\x42\x6b\x52\x87\xd8\xec\xe0\xaa\x76\xa5\x84\x2a\x22\x24\xfd\x92\x40\x88\xd5\x85\x1c\x1f\x6b\x47\xa0\xc4\xe4\xef\xf4\xea\xd7\x59\xac\x2a\x9e\x8c\xfa\x1f\x42\x8\xfe\x4f\x74\xa0\x26\xf5\xb3\x84\xf6\x58\x5f\x26\x66\x3e\xd7\xe4\x22\x91\x13\xc8\xac\x25\x96\x23\xd8\x9\xea\x45\x75\x23\xb8\x5f\xc2\x90\x8b\x9\xc4\xfc\x47\x6c\x6d\xa\xef\x69\xa4\x38\x19\xcf\x7d\xf9\x9\x73\x9b\x60\x5a\xf7\x37\xb5\xfe\x9f\xe3\x2b\x4c\xd\x6e\x19\xf1\xd6\xc0\x70\xf3\x9d\x22\x3c\xf9\x49\xce\x30\x8e\x44\xb5\x76\x15\x8f\x52\xfd\xa5\x4\xb8\x55\x6a\x36\x59\x7c\xc4\x48\xb8\xd7\xab\x5\x66\xe9\x5e\x21\x6f\x6b\x36\x29\xbb\xe9\xe3\xa2\x9a\xa8\xcd\x55\x25\x11\xba\x5a\x58\xa0\xde\xae\x19\x2a\x48\x5a\xff\x36\xcd\x6d\x16\x7a\x73\x38\x46\xe5\x47\x59\xc8\xa2\xf6\xe2\x6c\x83\xc5\x36\x2c\x83\x7d\xb4\x1\x5\x69\xe7\xaf\x5c\xc4\x64\x82\x12\x21\xef\xf7\xd1\x7d\xb8\x8d\x8c\x98\x7c\x5f\x7d\x92\x88\xb9\x94\x7\x9c\xd8\xe9\x9c\x17\x38\xe3\x57\x6c\xe0\xdc\xa5\x92\x42\xb3\xbd\x50\xa2\x7e\xb5\xb1\x52\x72\x3\x97\xd8\xaa\x9a\x1e\x75\x41\x11\xa3\x4f\xcc\xd4\xe3\x73\xad\x96\xdc\x47\x41\x9f\xb0\xbe\x79\x91\xf5\xb6\x18\xfe\xc2\x83\x18\x7d\x73\xd9\x4f\x83\x84\x3\xb3\xf0\x77\x66\x3d\x83\x63\x2e\x2c\xf9\xdd\xa6\x1f\x89\x82\xb8\x23\x42\xeb\xe2\xca\x70\x82\x61\x41\xa\x6d\x5f\x75\xc5\xe2\xc4\x91\x18\x44\x22\xfa\x34\x10\xf5\x20\xdc\xb7\xdd\x2a\x20\x77\xf5\xf9\xce\xdb\xa0\xa\x52\x2a\x4e\xdd\xcc\x97\xdf\x5\xe4\x5e\xb7\xaa\xf0\xe2\x80\xff\xba\x1a\xf\xac\xdf\x2\x32\xe6\xf7\xc7\x17\x13\xb7\xfc\x98\x48\x8c\xd\x82\xc9\x80\x7a\xe2\xa\xc5\xb4\xde\x7c\x3c\x79\x81\xe\x28\x65\x79\x67\x82\x69\x44\x66\x9\xf7\x16\x1a\xf9\x7d\x80\xa1\x79\x14\xa9\xc8\x20\xfb\xa2\x46\xbe\x8\x35\x17\x58\xc1\x1a\xda\x2a\x6b\x2e\x1e\xe6\x27\x55\x7b\x19\xe2\xfb\x64\xfc\x5e\x15\x54\x3c\xe7\xc2\x11\x50\x30\xb8\x72\x3\xb\x1a\x9f\x86\x27\x11\x5c\x6\x2b\xbd\x75\x1a\xa\xda\x1\xfa\x5c\x4a\xc1\x80\x3a\x6e\x30\xc8\x2c\xeb\x56\xec\x89\xfa\x35\x7b\xb2\xf0\x97\x8\x86\x53\xbe\xbd\x40\x41\x38\x1c\xb4\x8b\x79\x2e\x18\x96\x94\xde\xe8\xca\xe5\x9f\x92\x9f\x15\x5d\x56\x60\x5c\x9\xf9\x16\xf4\x17\xf\xf6\x4c\xda\xe6\x67\x89\x9f\xca\x6c\xe7\x9b\x4\x62\xe\x26\xa6\x52\xbd\x29\xff\xc7\xa4\x96\xe6\x6a\x2\xa5\x2e\x7b\xfe\x97\x68\x3e\x2e\x5f\x3b\xf\x36\xd6\x98\x19\x59\x48\xd2\xc6\xe1\x55\x1a\x6e\xd6\xed\x2c\xba\xc3\x9e\x64\xc9\x95\x86\x35\x5e\x3e\x88\x69\x99\x4b\xee\xbe\x9a\x99\xb5\x6e\x58\xae\xdd\x22\xdb\xdd\x6b\xfc\xaf\x90\xa3\x3d\xa4\xc1\x15\x92\x18\x8d\xd2\x4b\x7b\x6\xd1\x37\xb5\xe2\x7c\x2c\xf0\x25\xe4\x94\x2a\xbd\xe3\x82\x70\x78\xa3\x82\x10\x5a\x90\xd7\xa4\xfa\xaf\x1a\x88\x59\xdc\x74\x12\xb4\x8e\xd7\x19\x46\xf4\x84\x69\x9f\xbb\x70\xa8\x4c\x52\x81\xa9\xff\x76\x1c\xae\xd8\x11\x3d\x7f\x7d\xc5\x12\x59\x28\x18\xc2\xa2\xb7\x1c\x88\xf8\xd6\x1b\xa6\x7d\x9e\xde\x29\xf8\xed\xff\xeb\x92\x24\x4f\x5\xaa\xd9\x49\xba\x87\x59\x51\xc9\x20\x5c\x9b\x74\xcf\x3\xd9\x2d\x34\xc7\x5b\xa5\x40\xb2\x99\xf5\xcb\xb4\xf6\xb7\x72\x4a\xd6\xbd\xb0\xf3\x93\xe0\x1b\xa8\x4\x1e\x35\xd4\x80\x20\xf4\x9c\x31\x6b\x45\xb9\x15\xb0\x5e\xdd\xa\x33\x9c\x83\xcd\x58\x89\x50\x56\xbb\x81\x0\x91\x32\xf3\x1b\x3e\xcf\x45\xe1\xf9\xe1\x2c\x26\x78\x93\x9a\x60\x46\xc9\xb5\x5e\x6a\x28\x92\x87\x3f\x63\x7b\xdb\xf7\xd0\x13\x9d\x32\x40\x5e\xcf\xfb\x79\x68\x47\x4c\xfd\x1\x17\xe6\x97\x93\x78\xbb\xa6\x27\xa3\xe8\x1a\xe8\x94\x55\x7d\x8\xe5\xdc\x66\xa3\x69\xc8\xca\xc5\xa1\x84\x55\xde\x8\x91\x16\x3a\xc\x86\xab\x27\x2b\x64\x34\x2\x6c\x76\x8b\xc6\xaf\xcc\xe1\xd6\x8c\x2a\x18\x3d\xa6\x1b\x37\x75\x45\x73\xc2\x75\xd7\x53\x78\x3a\xd6\xe8\x29\xd2\x4a\xa8\x1e\x82\xf6\xb6\x81\xde\x21\xed\x2b\x56\xbb\xf2\xd0\x57\xc1\x7c\xd2\x6a\xd2\x56\xf5\x13\x5f\x1c\x6a\xb\x74\xfb\xe9\xfe\x9e\xea\x95\xb2\x46\xab\xa\xfc\xfd\xf3\xbb\x4\x2b\x76\x1b\xa4\x74\xb0\xc1\x78\xc3\x69\xe2\xb0\x1\xe1\xde\x32\x4c\x8d\x1a\xb3\x38\x8\xd5\xfc\x1f\xdc\xe\x2c\x9c\xb1\xa1\x63\x17\x22\xf5\x6c\x93\x70\x74\x0\xf8\x39\x1\x94\xd1\x32\x23\x56\x5d\xa6\x2\x76\x76\x93\xce\x2f\x19\xe9\x17\x52\xae\x6e\x2c\x6d\x61\x7f\x3b\xaa\xe0\x52\x85\xc5\x65\xc1\xbb\x8e\x5b\x21\xd5\xc9\x78\x83\x7\x97\x4c\x62\x61\x41\xd4\xfc\xc9\x39\xe3\x9b\xd0\xcc\x75\xc4\x97\xe6\xdd\x2a\x5f\xa6\xe8\x59\x6c\x98\xb9\x2\xe2\xa2\xd6\x68\xee\x3b\x1d\xe3\x4d\x5b\x30\xef\x3\xf2\xeb\x18\x57\x36\xe8\xa1\xf4\x47\xfb\xcb\x8f\xcb\xc8\xf3\x4f\x74\x9d\x9d\xb1\x8d\x14\x44\xd9\x19\xb4\x54\x4f\x75\x19\x9\xa0\x75\xbc\x3b\x82\xc6\x3f\xb8\x83\x19\x6e\xd6\x37\xfe\x6e\x8a\x4e\xe0\x4a\xab\x7b\xc8\xb4\x1d\xf4\xed\x27\x3\x65\xa2\xa1\xae\x11\xe7\x98\x78\x48\x91\xd2\xd2\xd4\x23\x78\x50\xb1\x5b\x85\x10\x8d\xca\x5f\xf\x71\xae\x72\x9a\xf6\x25\x19\x60\x6\xf7\x10\x34\x18\xd\xc9\x9f\x7b\xc\x9b\x8f\x91\x1b\x9f\xcd\x10\xee\x75\xf9\x97\x66\xfc\x4d\x33\x6e\x28\x2b\x92\x85\x4f\xab\x43\x8d\x8f\x7d\x86\xa7\xc7\xd8\xd3\xb\x8b\x57\xb6\x1d\x95\xd\xe9\xbc\xd9\x3\xd9\x10\x19\xc3\x46\x63\x55\x87\x61\x79\x6c\x95\xe\x9c\xdd\xca\xc3\xf3\x64\xf0\x7d\x76\xb7\x53\x67\x2b\x1e\x44\x56\x81\xea\x8f\x5c\x42\x16\xb8\x28\xeb\x1b\x61\x10\x1e\xbf\xec\xa8"
}, {
	.key	= "\xb3\x35\x50\x3\x54\x2e\x40\x5e\x8f\x59\x8e\xc5\x90\xd5\x27\x2d\xba\x29\x2e\xcb\x1b\x70\x44\x1e\x65\x91\x6e\x2a\x79\x22\xda\x64",
	.nonce	= "\x5\xa3\x93\xed\x30\xc5\xa2\x6",
	.nlen	= 8,
	.assoc	= "\xb1\x69\x83\x87\x30\xaa\x5d\xb8\x77\xe8\x21\xff\x6\x59\x35\xce\x75\xfe\x38\xef\xb8\x91\x43\x8c\xcf\x70\xdd\xa\x68\xbf\xd4\xbc\x16\x76\x99\x36\x1e\x58\x79\x5e\xd4\x29\xf7\x33\x93\x48\xdb\x5f\x1\xae\x9c\xb6\xe4\x88\x6d\x2b\x76\x75\xe0\xf3\x74\xe2\xc9",
	.alen	= 63,
	.input	= "\x52\x34\xb3\x65\x3b\xb7\xe5\xd3\xab\x49\x17\x60\xd2\x52\x56\xdf\xdf\x34\x56\x82\xe2\xbe\xe5\xe1\x28\xd1\x4e\x5f\x4f\x1\x7d\x3f\x99\x6b\x30\x6e\x1a\x7c\x4c\x8e\x62\x81\xae\x86\x3f\x6b\xd0\xb5\xa9\xcf\x50\xf1\x2\x12\xa0\xb\x24\xe9\xe6\x72\x89\x2c\x52\x1b\x34\x38\xf8\x75\x5f\xa0\x74\xe2\x99\xdd\xa6\x4b\x14\x50\x4e\xf1\xbe\xd6\x9e\xdb\xb2\x24\x27\x74\x12\x4a\x78\x78\x17\xa5\x58\x8e\x2f\xf9\xf4\x8d\xee\x3\x88\xae\xb8\x29\xa1\x2f\x4b\xee\x92\xbd\x87\xb3\xce\x34\x21\x57\x46\x4\x49\xc\x80\xf2\x1\x13\xa1\x55\xb3\xff\x44\x30\x3c\x1c\xd0\xef\xbc\x18\x74\x26\xad\x41\x5b\x5b\x3e\x9a\x7a\x46\x4f\x16\xd6\x74\x5a\xb7\x3a\x28\x31\xd8\xae\x26\xac\x50\x53\x86\xf2\x56\xd7\x3f\x29\xbc\x45\x68\x8e\xcb\x98\x64\xdd\xc9\xba\xb8\x4b\x7b\x82\xdd\x14\xa7\xcb\x71\x72\x0\x5c\xad\x7b\x6a\x89\xa4\x3d\xbf\xb5\x4b\x3e\x7c\x5a\xcf\xb8\xa1\xc5\x6e\xc8\xb6\x31\x57\x7b\xdf\xa5\x7e\xb1\xd6\x42\x2a\x31\x36\xd1\xd0\x3f\x7a\xe5\x94\xd6\x36\xa0\x6f\xb7\x40\x7d\x37\xc6\x55\x7c\x50\x40\x6d\x29\x89\xe3\x5a\xae\x97\xe7\x44\x49\x6e\xbd\x81\x3d\x3\x93\x6\x12\x6\xe2\x41\x12\x4a\xf1\x6a\xa4\x58\xa2\xfb\xd2\x15\xba\xc9\x79\xc9\xce\x5e\x13\xbb\xf1\x9\x4\xcc\xfd\xe8\x51\x34\x6a\xe8\x61\x88\xda\xed\x1\x47\x84\xf5\x73\x25\xf9\x1c\x42\x86\x7\xf3\x5b\x1a\x1\xb3\xeb\x24\x32\x8d\xf6\xed\x7c\x4b\xeb\x3c\x36\x42\x28\xdf\xdf\xb6\xbe\xd9\x8c\x52\xd3\x2b\x8\x90\x8c\xe7\x98\x31\xe2\x32\x8e\xfc\x11\x48\x0\xa8\x6a\x42\x4a\x2\xc6\x4b\x9\xf1\xe3\x49\xf3\x45\x1f\xe\xbc\x56\xe2\xe4\xdf\xfb\xeb\x61\xfa\x24\xc1\x63\x75\xbb\x47\x75\xaf\xe1\x53\x16\x96\x21\x85\x26\x11\xb3\x76\xe3\x23\xa1\x6b\x74\x37\xd0\xde\x6\x90\x71\x5d\x43\x88\x9b\x0\x54\xa6\x75\x2f\xa1\xc2\xb\x73\x20\x1d\xb6\x21\x79\x57\x3f\xfa\x9\xbe\x8a\x33\xc3\x52\xf0\x1d\x82\x31\xd1\x55\xb5\x6c\x99\x25\xcf\x5c\x32\xce\xe9\xd\xfa\x69\x2c\xd5\xd\xc5\x6d\x86\xd0\xc\x3b\x6\x50\x79\xe8\xc3\xae\x4\xe6\xcd\x51\xe4\x26\x9b\x4f\x7e\xa6\xf\xab\xd8\xe5\xde\xa9\x0\x95\xbe\xa3\x9d\x5d\xb2\x9\x70\x18\x1c\xf0\xac\x29\x23\x2\x29\x28\xd2\x74\x35\x57\x62\xf\x24\xea\x5e\x33\xc2\x92\xf3\x78\x4d\x30\x1e\xa1\x99\xa9\x82\xb0\x42\x31\x8d\xad\x8a\xbc\xfc\xd4\x57\x47\x3e\xb4\x50\xdd\x6e\x2c\x80\x4d\x22\xf1\xfb\x57\xc4\xdd\x17\xe1\x8a\x36\x4a\xb3\x37\xca\xc9\x4e\xab\xd5\x69\xc4\xf4\xbc\xb\x3b\x44\x4b\x29\x9c\xee\xd4\x35\x22\x21\xb0\x1f\x27\x64\xa8\x51\x1b\xf0\x9f\x19\x5c\xfb\x5a\x64\x74\x70\x45\x9\xf5\x64\xfe\x1a\x2d\xc9\x14\x4\x14\xcf\xd5\x7d\x60\xaf\x94\x39\x94\xe2\x7d\x79\x82\xd0\x65\x3b\x6b\x9c\x19\x84\xb4\x6d\xb3\xc\x99\xc0\x56\xa8\xbd\x73\xce\x5\x84\x3e\x30\xaa\xc4\x9b\x1b\x4\x2a\x9f\xd7\x43\x2b\x23\xdf\xbf\xaa\xd5\xc2\x43\x2d\x70\xab\xdc\x75\xad\xac\xf7\xc0\xbe\x67\xb2\x74\xed\x67\x10\x4a\x92\x60\xc1\x40\x50\x19\x8a\x8a\x8c\x9\xe\x72\xe1\x73\x5e\xe8\x41\x85\x63\x9f\x3f\xd7\x7d\xc4\xfb\x22\x5d\x92\x6c\xb3\x1e\xe2\x50\x2f\x82\xa8\x28\xc0\xb5\xd7\x5f\x68\xd\x2c\x2d\xaf\x7e\xfa\x2e\x8\xf\x1f\x70\x9f\xe9\x19\x72\x55\xf8\xfb\x51\xd2\x33\x5d\xa0\xd3\x2b\xa\x6c\xbc\x4e\xcf\x36\x4d\xdc\x3b\xe9\x3e\x81\x7c\x61\xdb\x20\x2d\x3a\xc3\xb3\xc\x1e\x0\xb9\x7c\xf5\xca\x10\x5f\x3a\x71\xb3\xe4\x20\xdb\xc\x2a\x98\x63\x45\x0\x58\xf6\x68\xe4\xb\xda\x13\x3b\x60\x5c\x76\xdb\xb9\x97\x71\xe4\xd9\xb7\xdb\xbd\x68\xc7\x84\x84\xaa\x7c\x68\x62\x5e\x16\xfc\xba\x72\xaa\x9a\xa9\xeb\x7c\x75\x47\x97\x7e\xad\xe2\xd9\x91\xe8\xe4\xa5\x31\xd7\x1\x8e\xa2\x11\x88\x95\xb9\xf2\x9b\xd3\x7f\x1b\x81\x22\xf7\x98\x60\xa\x64\xa6\xc1\xf6\x49\xc7\xe3\x7\x4d\x94\x7a\xcf\x6e\x68\xc\x1b\x3f\x6e\x2e\xee\x92\xfa\x52\xb3\x59\xf8\xf1\x8f\x6a\x66\xa3\x82\x76\x4a\x7\x1a\xc7\xdd\xf5\xda\x9c\x3c\x24\xbf\xfd\x42\xa1\x10\x64\x6a\xf\x89\xee\x36\xa5\xce\x99\x48\x6a\xf0\x9f\x9e\x69\xa4\x40\x20\xe9\x16\x15\xf7\xdb\x75\x2\xcb\xe9\x73\x8b\x3b\x49\x2f\xf0\xaf\x51\x6\x5c\xdf\x27\x27\x49\x6a\xd1\xcc\xc7\xb5\x63\xb5\xfc\xb8\x5c\x87\x7f\x84\xb4\xcc\x14\xa9\x53\xda\xa4\x56\xf8\xb6\x1b\xcc\x40\x27\x52\x6\x5a\x13\x81\xd7\x3a\xd4\x3b\xfb\x49\x65\x31\x33\xb2\xfa\xcd\xad\x58\x4e\x2b\xae\xd2\x20\xfb\x1a\x48\xb4\x3f\x9a\xd8\x7a\x35\x4a\xc8\xee\x88\x5e\x7\x66\x54\xb9\xec\x9f\xa3\xe3\xb9\x37\xaa\x49\x76\x31\xda\x74\x2d\x3c\xa4\x65\x10\x32\x38\xf0\xde\xd3\x99\x17\xaa\x71\xaa\x8f\xf\x8c\xaf\xa2\xf8\x5d\x64\xba\x1d\xa3\xef\x96\x73\xe8\xa1\x2\x8d\xc\x6d\xb8\x6\x90\xb8\x8\x56\x2c\xa7\x6\xc9\xc2\x38\xdb\x7c\x63\xb1\x57\x8e\xea\x7c\x79\xf3\x49\x1d\xfe\x9f\xf3\x6e\xb1\x1d\xba\x19\x80\x1a\xa\xd3\xb0\x26\x21\x40\xb1\x7c\xf9\x4d\x8d\x10\xc1\x7e\xf4\xf6\x3c\xa8\xfd\x7c\xa3\x92\xb2\xf\xaa\xcc\xa6\x11\xfe\x4\xe3\xd1\x7a\x32\x89\xdf\xd\xc4\x8f\x79\x6b\xca\x16\x7c\x6e\xf9\xad\xf\xf6\xfe\x27\xdb\xc4\x13\x70\xf1\x62\x1a\x4f\x79\x40\xc9\x9b\x8b\x21\xea\x84\xfa\xf5\xf1\x89\xce\xb7\x55\xa\x80\x39\x2f\x55\x36\x16\x9c\x7b\x8\xbd\x87\xd\xa5\x32\xf1\x52\x7c\xe8\x55\x60\x5b\xd7\x69\xe4\xfc\xfa\x12\x85\x96\xea\x50\x28\xab\x8a\xf7\xbb\xe\x53\x74\xca\xa6\x27\x9\xc2\xb5\xde\x18\x14\xd9\xea\xe5\x29\x1c\x40\x56\xcf\xd7\xae\x5\x3f\x65\xaf\x5\x73\xe2\x35\x96\x27\x7\x14\xc0\xad\x33\xf1\xdc\x44\x7a\x89\x17\x77\xd2\x9c\x58\x60\xf0\x3f\x7b\x2d\x2e\x57\x95\x54\x87\xed\xf2\xc7\x4c\xf0\xae\x56\x29\x19\x7d\x66\x4b\x9b\x83\x84\x42\x3b\x1\x25\x66\x8e\x2\xde\xb9\x83\x54\x19\xf6\x9f\x79\xd\x67\xc5\x1d\x7a\x44\x2\x98\xa7\x16\x1c\x29\xd\x74\xff\x85\x40\x6\xef\x2c\xa9\xc6\xf5\x53\x7\x6\xae\xe4\xfa\x5f\xd8\x39\x4d\xf1\x9b\x6b\xd9\x24\x84\xfe\x3\x4c\xb2\x3f\xdf\xa1\x5\x9e\x50\x14\x5a\xd9\x1a\xa2\xa7\xfa\xfa\x17\xf7\x78\xd6\xb5\x92\x61\x91\xac\x36\xfa\x56\xd\x38\x32\x18\x85\x8\x58\x37\xf0\x4b\xdb\x59\xe7\xa4\x34\xc0\x1b\x1\xaf\x2d\xde\xa1\xaa\x5d\xd3\xec\xe1\xd4\xf7\xe6\x54\x68\xf0\x51\x97\xa7\x89\xea\x24\xad\xd3\x6e\x47\x93\x8b\x4b\xb4\xf7\x1c\x42\x6\x67\xe8\x99\xf6\xf5\x7b\x85\xb5\x65\xb5\xb5\xd2\x37\xf5\xf3\x2\xa6\x4d\x11\xa7\xdc\x51\x9\x7f\xa0\xd8\x88\x1c\x13\x71\xae\x9c\xb7\x7b\x34\xd6\x4e\x68\x26\x83\x51\xaf\x1d\xee\x8b\xbb\x69\x43\x2b\x9e\x8a\xbc\x2\xe\xa0\x1b\xe0\xa8\x5f\x6f\xaf\x1b\x8f\xe7\x64\x71\x74\x11\x7e\xa8\xd8\xf9\x97\x6\xc3\xb6\xfb\xfb\xb7\x3d\x35\x9d\x3b\x52\xed\x54\xca\xf4\x81\x1\x2d\x1b\xc3\xa7\x0\x3d\x1a\x39\x54\xe1\xf6\xff\xed\x6f\xb\x5a\x68\xda\x58\xdd\xa9\xcf\x5c\x4a\xe5\x9\x4e\xde\x9d\xbc\x3e\xee\x5a\x0\x3b\x2c\x87\x10\x65\x60\xdd\xd7\x56\xd1\x4c\x64\x45\xe4\x21\xec\x78\xf8\x25\x7a\x3e\x16\x5d\x9\x53\x14\xbe\x4f\xae\x87\xd8\xd1\xaa\x3c\xf6\x3e\xa4\x70\x8c\x5e\x70\xa4\xb3\x6b\x66\x73\xd3\xbf\x31\x6\x19\x62\x93\x15\xf2\x86\xe4\x52\x7e\x53\x4c\x12\x38\xcc\x34\x7d\x57\xf6\x42\x93\x8a\xc4\xee\x5c\x8a\xe1\x52\x8f\x56\x64\xf6\xa6\xd1\x91\x57\x70\xcd\x11\x76\xf5\x59\x60\x60\x3c\xc1\xc3\xb\x7f\x58\x1a\x50\x91\xf1\x68\x8f\x6e\x74\x74\xa8\x51\xb\xf7\x7a\x98\x37\xf2\xa\xe\xa4\x97\x4\xb8\x9b\xfd\xa0\xea\xf7\xd\xe1\xdb\x3\xf0\x31\x29\xf8\xdd\x6b\x8b\x5d\xd8\x59\xa9\x29\xcf\x9a\x79\x89\x19\x63\x46\x9\x79\x6a\x11\xda\x63\x68\x48\x77\x23\xfb\x7d\x3a\x43\xcb\x2\x3b\x7a\x6d\x10\x2a\x9e\xac\xf1\xd4\x19\xf8\x23\x64\x1d\x2c\x5f\xf2\xb0\x5c\x23\x27\xf7\x27\x30\x16\x37\xb1\x90\xab\x38\xfb\x55\xcd\x78\x58\xd4\x7d\x43\xf6\x45\x5e\x55\x8d\xb1\x2\x65\x58\xb4\x13\x4b\x36\xf7\xcc\xfe\x3d\xb\x82\xe2\x12\x11\xbb\xe6\xb8\x3a\x48\x71\xc7\x50\x6\x16\x3a\xe6\x7c\x5\xc7\xc8\x4d\x2f\x8\x6a\x17\x9a\x95\x97\x50\x68\xdc\x28\x18\xc4\x61\x38\xb9\xe0\x3e\x78\xdb\x29\xe0\x9f\x52\xdd\xf8\x4f\x91\xc1\xd0\x33\xa1\x7a\x8e\x30\x13\x82\x7\x9f\xd3\x31\xf\x23\xbe\x32\x5a\x75\xcf\x96\xb2\xec\xb5\x32\xac\x21\xd1\x82\x33\xd3\x15\x74\xbd\x90\xf1\x2c\xe6\x5f\x8d\xe3\x2\xe8\xe9\xc4\xca\x96\xeb\xe\xbc\x91\xf4\xb9\xea\xd9\x1b\x75\xbd\xe1\xac\x2a\x5\x37\x52\x9b\x1b\x3f\x5a\xdc\x21\xc3\x98\xbb\xaf\xa3\xf2\x0\xbf\xd\x30\x89\x5\xcc\xa5\x76\xf5\x6\xf0\xc6\x54\x8a\x5d\xd4\x1e\xc1\xf2\xce\xb0\x62\xc8\xfc\x59\x42\x9a\x90\x60\x55\xfe\x88\xa5\x8b\xb8\x33\xc\x23\x24\xd\x15\x70\x37\x1e\x3d\xf6\xd2\xea\x92\x10\xb2\xc4\x51\xac\xf2\xac\xf3\x6b\x6c\xaa\xcf\x12\xc5\x6c\x90\x50\xb5\xc\xfc\x1a\x15\x52\xe9\x26\xc6\x52\xa4\xe7\x81\x69\xe1\xe7\x9e\x30\x1\xec\x84\x89\xb2\xd\x66\xdd\xce\x28\x5c\xec\x98\x46\x68\x21\x9f\x88\x3f\x1f\x42\x77\xce\xd0\x61\xd4\x20\xa7\xff\x53\xad\x37\xd0\x17\x35\xc9\xfc\xba\xa\x78\x3f\xf2\xcc\x86\x89\xe8\x4b\x3c\x48\x33\x9\x7f\xc6\xc0\xdd\xb8\xfd\x7a\x66\x66\x65\xeb\x47\xa7\x4\x28\xa3\x19\x8e\xa9\xb1\x13\x67\x62\x70\xcf\xd6",
	.ilen	= 2027,
	.result	= "\x74\xa6\x3e\xe4\xb1\xcb\xaf\xb0\x40\xe5\xf\x9e\xf1\xf2\x89\xb5\x42\x34\x8a\xa1\x3\xb7\xe9\x57\x46\xbe\x20\xe4\x6e\xb0\xeb\xff\xea\x7\x7e\xef\xe2\x55\x9f\xe5\x78\x3a\xb7\x83\xc2\x18\x40\x7b\xeb\xcd\x81\xfb\x90\x12\x9e\x46\xa9\xd6\x4a\xba\xb0\x62\xdb\x6b\x99\xc4\xdb\x54\x4b\xb8\xa5\x71\xcb\xcd\x63\x32\x55\xfb\x31\xf0\x38\xf5\xbe\x78\xe4\x45\xce\x1b\x6a\x5b\xe\xf4\x16\xe4\xb1\x3d\xf6\x63\x7b\xa7\xc\xde\x6f\x8f\x74\xdf\xe0\x1e\x9d\xce\x8f\x24\xef\x23\x35\x33\x7b\x83\x34\x23\x58\x74\x14\x77\x1f\xc2\x4f\x4e\xc6\x89\xf9\x52\x9\x37\x64\x14\xc4\x1\x6b\x9d\x77\xe8\x90\x5d\xa8\x4a\x2a\xef\x5c\x7f\xeb\xbb\xb2\xc6\x93\x99\x66\xdc\x7f\xd4\x9e\x2a\xca\x8d\xdb\xe7\x20\xcf\xe4\x73\xae\x49\x7d\x64\xf\xe\x28\x46\xa9\xa8\x32\xe4\xe\xf6\x51\x53\xb8\x3c\xb1\xff\xa3\x33\x41\x75\xff\xf1\x6f\xf1\xfb\xbb\x83\x7f\x6\x9b\xe7\x1b\xa\xe0\x5c\x33\x60\x5b\xdb\x5b\xed\xfe\xa5\x16\x19\x72\xa3\x64\x23\x0\x2\xc7\xf3\x6a\x81\x3e\x44\x1d\x79\x15\x5f\x9a\xde\xe2\xfd\x1b\x73\xc1\xbc\x23\xba\x31\xd2\x50\xd5\xad\x7f\x74\xa7\xc9\xf8\x3e\x2b\x26\x10\xf6\x3\x36\x74\xe4\xe\x6a\x72\xb7\x73\xa\x42\x28\xc2\xad\x5e\x3\xbe\xb8\xb\xa8\x5b\xd4\xb8\xba\x52\x89\xb1\x9b\xc1\xc3\x65\x87\xed\xa5\xf4\x86\xfd\x41\x80\x91\x27\x59\x53\x67\x15\x78\x54\x8b\x2d\x3d\xc7\xff\x2\x92\x7\x5f\x7a\x4b\x60\x59\x3c\x6f\x5c\xd8\xec\x95\xd2\xfe\xa0\x3b\xd8\x3f\xd1\x69\xa6\xd6\x41\xb2\xf4\x4d\x12\xf4\x58\x3e\x66\x64\x80\x31\x9b\xa8\x4c\x8b\x7\xb2\xec\x66\x94\x66\x47\x50\x50\x5f\x18\xb\xe\xd6\xc0\x39\x21\x13\x9e\x33\xbc\x79\x36\x2\x96\x70\xf0\x48\x67\x2f\x26\xe9\x6d\x10\xbb\xd6\x3f\xd1\x64\x7a\x2e\xbe\xc\x61\xf0\x75\x42\x38\x23\xb1\x9e\x9f\x7c\x67\x66\xd9\x58\x9a\xf1\xbb\x41\x2a\x8d\x65\x84\x94\xfc\xdc\x6a\x50\x64\xdb\x56\x33\x76\x0\x10\xed\xbe\xd2\x12\xf6\xf6\x1b\xa2\x16\xde\xae\x31\x95\xdd\xb1\x8\x7e\x4e\xee\xe7\xf9\xa5\xfb\x5b\x61\x43\x0\x40\xf6\x7e\x2\x4\x32\x4e\xc\xe2\x66\xd\xd7\x7\x98\xe\xf8\x72\x34\x6d\x95\x86\xd7\xcb\x31\x54\x47\xd0\x38\x29\x9c\x5a\x68\xd4\x87\x76\xc9\xe7\x7e\xe3\xf4\x81\x6d\x18\xcb\xc9\x5\xaf\xa0\xfb\x66\xf7\xf1\x1c\xc6\x14\x11\x4f\x2b\x79\x42\x8b\xbc\xac\xe7\x6c\xfe\xf\x58\xe7\x7c\x78\x39\x30\xb0\x66\x2c\x9b\x6d\x3a\xe1\xcf\xc9\xa4\xe\x6d\x6d\x8a\xa1\x3a\xe7\x28\xd4\x78\x4c\xa6\xa2\x2a\xa6\x3\x30\xd7\xa8\x25\x66\x87\x2f\x69\x5c\x4e\xdd\xa5\x49\x5d\x37\x4a\x59\xc4\xaf\x1f\xa2\xe4\xf8\xa6\x12\x97\xd5\x79\xf5\xe2\x4a\x2b\x5f\x61\xe4\x9e\xe3\xee\xb8\xa7\x5b\x2f\xf4\x9e\x6c\xfb\xd1\xc6\x56\x77\xba\x75\xaa\x3d\x1a\xa8\xb\xb3\x68\x24\x0\x10\x7f\xfd\xd7\xa1\x8d\x83\x54\x4f\x1f\xd8\x2a\xbe\x8a\xc\x87\xab\xa2\xde\xc3\x39\xbf\x9\x3\xa5\xf3\x5\x28\xe1\xe1\xee\x39\x70\x9c\xd8\x81\x12\x1e\x2\x40\xd2\x6e\xf0\xeb\x1b\x3d\x22\xc6\xe5\xe3\xb4\x5a\x98\xbb\xf0\x22\x28\x8d\xe5\xd3\x16\x48\x24\xa5\xe6\x66\xc\xf9\x8\xf9\x7e\x1e\xe1\x28\x26\x22\xc7\xc7\xa\x32\x47\xfa\xa3\xbe\x3c\xc4\xc5\x53\xa\xd5\x94\x4a\xd7\x93\xd8\x42\x99\xb9\xa\xdb\x56\xf7\xb9\x1c\x53\x4f\xfa\xd3\x74\xad\xd9\x68\xf1\x1b\xdf\x61\xc6\x5e\xa8\x48\xfc\xd4\x4a\x4c\x3c\x32\xf7\x1c\x96\x21\x9b\xf9\xa3\xcc\x5a\xce\xd5\xd7\x8\x24\xf6\x1c\xfd\xdd\x38\xc2\x32\xe9\xb8\xe7\xb6\xfa\x9d\x45\x13\x2c\x83\xfd\x4a\x69\x82\xcd\xdc\xb3\x76\xc\x9e\xd8\xf4\x1b\x45\x15\xb4\x97\xe7\x58\x34\xe2\x3\x29\x5a\xbf\xb6\xe0\x5d\x13\xd9\x2b\xb4\x80\xb2\x45\x81\x6a\x2e\x6c\x89\x7d\xee\xbb\x52\xdd\x1f\x18\xe7\x13\x6b\x33\xe\xea\x36\x92\x77\x7b\x6d\x9c\x5a\x5f\x45\x7b\x7b\x35\x62\x23\xd1\xbf\xf\xd0\x8\x1b\x2b\x80\x6b\x7e\xf1\x21\x47\xb0\x57\xd1\x98\x72\x90\x34\x1c\x20\x4\xff\x3d\x5c\xee\xe\x57\x5f\x6f\x24\x4e\x3c\xea\xfc\xa5\xa9\x83\xc9\x61\xb4\x51\x24\xf8\x27\x5e\x46\x8c\xb1\x53\x2\x96\x35\xba\xb8\x4c\x71\xd3\x15\x59\x35\x22\x20\xad\x3\x9f\x66\x44\x3b\x9c\x35\x37\x1f\x9b\xbb\xf3\xdb\x35\x63\x30\x64\xaa\xa2\x6\xa8\x5d\xbb\xe1\x9f\x70\xec\x82\x11\x6\x36\xec\x8b\x69\x66\x24\x44\xc9\x4a\x57\xbb\x9b\x78\x13\xce\x9c\xc\xba\x92\x93\x63\xb8\xe2\x95\xf\xf\x16\x39\x52\xfd\x3a\x6d\x2\x4b\xdf\x13\xd3\x2a\x22\xb4\x3\x7c\x54\x49\x96\x68\x54\x10\xfa\xef\xaa\x6c\xe8\x22\xdc\x71\x16\x13\x1a\xf6\x28\xe5\x6d\x77\x3d\xcd\x30\x63\xb1\x70\x52\xa1\xc5\x94\x5f\xcf\xe8\xb8\x26\x98\xf7\x6\xa0\xa\x70\xfa\x3\x80\xac\xc1\xec\xd6\x4c\x54\xd7\xfe\x47\xb6\x88\x4a\xf7\x71\x24\xee\xf3\xd2\xc2\x4a\x7f\xfe\x61\xc7\x35\xc9\x37\x67\xcb\x24\x35\xda\x7e\xca\x5f\xf3\x8d\xd4\x13\x8e\xd6\xcb\x4d\x53\x8f\x53\x1f\xc0\x74\xf7\x53\xb9\x5e\x23\x37\xba\x6e\xe3\x9d\x7\x55\x25\x7b\xe6\x2a\x64\xd1\x32\xdd\x54\x1b\x4b\xc0\xe1\xd7\x69\x58\xf8\x93\x29\xc4\xdd\x23\x2f\xa5\xfc\x9d\x7e\xf8\xd4\x90\xcd\x82\x55\xdc\x16\x16\x9f\x7\x52\x9b\x9d\x25\xed\x32\xc5\x7b\xdf\xf6\x83\x46\x3d\x65\xb7\xef\x87\x7a\x12\x69\x8f\x6\x7c\x51\x15\x4a\x8\xe8\xac\x9a\xc\x24\xa7\x27\xd8\x46\x2f\xe7\x1\xe\x1c\xc6\x91\xb0\x6e\x85\x65\xf0\x29\xd\x2e\x6b\x3b\xfb\x4b\xdf\xe4\x80\x93\x3\x66\x46\x3e\x8a\x6e\xf3\x5e\x4d\x62\xe\x49\x5\xaf\xd4\xf8\x21\x20\x61\x1d\x39\x17\xf4\x61\x47\x95\xfb\x15\x2e\xb3\x4f\xd0\x5d\xf5\x7d\x40\xda\x90\x3c\x6b\xcb\x17\x0\x13\x3b\x64\x34\x1b\xf0\xf2\xe5\x3b\xb2\xc7\xd3\x5f\x3a\x44\xa6\x9b\xb7\x78\xe\x42\x5d\x4c\xc1\xe9\xd2\xcb\xb7\x78\xd1\xfe\x9a\xb5\x7\xe9\xe0\xbe\xe2\x8a\xa7\x1\x83\x0\x8c\x5c\x8\xe6\x63\x12\x92\xb7\xb7\xa6\x19\x7d\x38\x13\x38\x92\x87\x24\xf9\x48\xb3\x5e\x87\x6a\x40\x39\x5c\x3f\xed\x8f\xee\xdb\x15\x82\x6\xda\x49\x21\x2b\xb5\xbf\x32\x7c\x9f\x42\x28\x63\xcf\xaf\x1e\xf8\xc6\xa0\xd1\x2\x43\x57\x62\xec\x9b\xf\x1\x9e\x71\xd8\x87\x9d\x1\xc1\x58\x77\xd9\xaf\xb1\x10\x7e\xdd\xa6\x50\x96\xe5\xf0\x72\x0\x6d\x4b\xf8\x2a\x8f\x19\xf3\x22\x88\x11\x4a\x8b\x7c\xfd\xb7\xed\xe1\xf6\x40\x39\xe0\xe9\xf6\x3d\x25\xe6\x74\x3c\x58\x57\x7f\xe1\x22\x96\x47\x31\x91\xba\x70\x85\x28\x6b\x9f\x6e\x25\xac\x23\x66\x2f\x29\x88\x28\xce\x8c\x5c\x88\x53\xd1\x3b\xcc\x6a\x51\xb2\xe1\x28\x3f\x91\xb4\xd\x0\x3a\xe3\xf8\xc3\x8f\xd7\x96\x62\xe\x2e\xfc\xc8\x6c\x77\xa6\x1d\x22\xc1\xb8\xe6\x61\xd7\x67\x36\x13\x7b\xbb\x9b\x59\x9\xa6\xdf\xf7\x6b\xa3\x40\x1a\xf5\x4f\xb4\xda\xd3\xf3\x81\x93\xc6\x18\xd9\x26\xee\xac\xf0\xaa\xdf\xc5\x9c\xca\xc2\xa2\xcc\x7b\x5c\x24\xb0\xbc\xd0\x6a\x4d\x89\x9\xb8\x7\xfe\x87\xad\xa\xea\xb8\x42\xf9\x5e\xb3\x3e\x36\x4c\xaf\x75\x9e\x1c\xeb\xbd\xbc\xbb\x80\x40\xa7\x3a\x30\xbf\xa8\x44\xf4\xeb\x38\xad\x29\xba\x23\xed\x41\xc\xea\xd2\xbb\x41\x18\xd6\xb9\xba\x65\x2b\xa3\x91\x6d\x1f\xa9\xf4\xd1\x25\x8d\x4d\x38\xff\x64\xa0\xec\xde\xa6\xb6\x79\xab\x8e\x33\x6c\x47\xde\xaf\x94\xa4\xa5\x86\x77\x55\x9\x92\x81\x31\x76\xc7\x34\x22\x89\x8e\x3d\x26\x26\xd7\xfc\x1e\x16\x72\x13\x33\x63\xd5\x22\xbe\xb8\x4\x34\x84\x41\xbb\x80\xd0\x9f\x46\x48\x7\xa7\xfc\x2b\x3a\x75\x55\x8c\xc7\x6a\xbd\x7e\x46\x8\x84\xf\xd5\x74\xc0\x82\x8e\xaa\x61\x5\x1\xb2\x47\x6e\x20\x6a\x2d\x58\x70\x48\x32\xa7\x37\xd2\xb8\x82\x1a\x51\xb9\x61\xdd\xfd\x9d\x6b\xe\x18\x97\xf8\x45\x5f\x87\x10\xcf\x34\x72\x45\x26\x49\x70\xe7\xa3\x78\xe0\x52\x89\x84\x94\x83\x82\xc2\x69\x8f\xe3\xe1\x3f\x60\x74\x88\xc4\xf7\x75\x2c\xfb\xbd\xb6\xc4\x7e\x10\xa\x6c\x90\x4\x9e\xc3\x3f\x59\x7c\xce\x31\x18\x60\x57\x73\x46\x94\x7d\x6\xa0\x6d\x44\xec\xa2\xa\x9e\x5\x15\xef\xca\x5c\xbf\x0\xeb\xf7\x3d\x32\xd4\xa5\xef\x49\x89\x5e\x46\xb0\xa6\x63\x5b\x8a\x73\xae\x6f\xd5\x9d\xf8\x4f\x40\xb5\xb2\x6e\xd3\xb6\x1\xa9\x26\xa2\x21\xcf\x33\x7a\x3a\xa4\x23\x13\xb0\x69\x6a\xee\xce\xd8\x9d\x1\x1d\x50\xc1\x30\x6c\xb1\xcd\xa0\xf0\xf0\xa2\x64\x6f\xbb\xbf\x5e\xe6\xab\x87\xb4\xf\x4f\x15\xaf\xb5\x25\xa1\xb2\xd0\x80\x2c\xfb\xf9\xfe\xd2\x33\xbb\x76\xfe\x7c\xa8\x66\xf7\xe7\x85\x9f\x1f\x85\x57\x88\xe1\xe9\x63\xe4\xd8\x1c\xa1\xfb\xda\x44\x5\x2e\x1d\x3a\x1c\xff\xc8\x3b\xc0\xfe\xda\x22\xb\x43\xd6\x88\x39\x4c\x4a\xa6\x69\x18\x93\x42\x4e\xb5\xcc\x66\xd\x9\xf8\x1e\x7c\xd3\x3c\x99\xd\x50\x1d\x62\xe9\x57\x6\xbf\x19\x88\xdd\xad\x7b\x4f\xf9\xc7\x82\x6d\x8d\xc8\xc4\xc5\x78\x17\x20\x15\xc5\x52\x41\xcf\x5b\xd6\x7f\x94\x2\x41\xe0\x40\x22\x3\x5e\xd1\x53\xd4\x86\xd3\x2c\x9f\xf\x96\xe3\x6b\x9a\x76\x32\x6\x47\x4b\x11\xb3\xdd\x3\x65\xbd\x9b\x1\xda\x9c\xb9\x7e\x3f\x6a\xc4\x7b\xea\xd4\x3c\xb9\xfb\x5c\x6b\x64\x33\x52\xba\x64\x78\x8f\xa4\xaf\x7a\x61\x8d\xbc\xc5\x73\xe9\x6b\x58\x97\x4b\xbf\x63\x22\xd3\x37\x2\x54\xc5\xb9\x16\x4a\xf0\x19\xd8\x94\x57\xb8\x8a\xb3\x16\x3b\xd0\x84\x8e\x67\xa6\xa3\x7d\x78\xec\x0"
}, {
	.key	= "\xb3\x35\x50\x3\x54\x2e\x40\x5e\x8f\x59\x8e\xc5\x90\xd5\x27\x2d\xba\x29\x2e\xcb\x1b\x70\x44\x1e\x65\x91\x6e\x2a\x79\x22\xda\x64",
	.nonce	= "\x5\xa3\x93\xed\x30\xc5\xa2\x6",
	.nlen	= 8,
	.assoc	= "\xb1\x69\x83\x87\x30\xaa\x5d\xb8\x77\xe8\x21\xff\x6\x59\x35\xce\x75\xfe\x38\xef\xb8\x91\x43\x8c\xcf\x70\xdd\xa\x68\xbf\xd4\xbc\x16\x76\x99\x36\x1e\x58\x79\x5e\xd4\x29\xf7\x33\x93\x48\xdb\x5f\x1\xae\x9c\xb6\xe4\x88\x6d\x2b\x76\x75\xe0\xf3\x74\xe2\xc9",
	.alen	= 63,
	.input	= "\x52\x34\xb3\x65\x3b\xb7\xe5\xd3\xab\x49\x17\x60\xd2\x52\x56\xdf\xdf\x34\x56\x82\xe2\xbe\xe5\xe1\x28\xd1\x4e\x5f\x4f\x1\x7d\x3f\x99\x6b\x30\x6e\x1a\x7c\x4c\x8e\x62\x81\xae\x86\x3f\x6b\xd0\xb5\xa9\xcf\x50\xf1\x2\x12\xa0\xb\x24\xe9\xe6\x72\x89\x2c\x52\x1b\x34\x38\xf8\x75\x5f\xa0\x74\xe2\x99\xdd\xa6\x4b\x14\x50\x4e\xf1\xbe\xd6\x9e\xdb\xb2\x24\x27\x74\x12\x4a\x78\x78\x17\xa5\x58\x8e\x2f\xf9\xf4\x8d\xee\x3\x88\xae\xb8\x29\xa1\x2f\x4b\xee\x92\xbd\x87\xb3\xce\x34\x21\x57\x46\x4\x49\xc\x80\xf2\x1\x13\xa1\x55\xb3\xff\x44\x30\x3c\x1c\xd0\xef\xbc\x18\x74\x26\xad\x41\x5b\x5b\x3e\x9a\x7a\x46\x4f\x16\xd6\x74\x5a\xb7\x3a\x28\x31\xd8\xae\x26\xac\x50\x53\x86\xf2\x56\xd7\x3f\x29\xbc\x45\x68\x8e\xcb\x98\x64\xdd\xc9\xba\xb8\x4b\x7b\x82\xdd\x14\xa7\xcb\x71\x72\x0\x5c\xad\x7b\x6a\x89\xa4\x3d\xbf\xb5\x4b\x3e\x7c\x5a\xcf\xb8\xa1\xc5\x6e\xc8\xb6\x31\x57\x7b\xdf\xa5\x7e\xb1\xd6\x42\x2a\x31\x36\xd1\xd0\x3f\x7a\xe5\x94\xd6\x36\xa0\x6f\xb7\x40\x7d\x37\xc6\x55\x7c\x50\x40\x6d\x29\x89\xe3\x5a\xae\x97\xe7\x44\x49\x6e\xbd\x81\x3d\x3\x93\x6\x12\x6\xe2\x41\x12\x4a\xf1\x6a\xa4\x58\xa2\xfb\xd2\x15\xba\xc9\x79\xc9\xce\x5e\x13\xbb\xf1\x9\x4\xcc\xfd\xe8\x51\x34\x6a\xe8\x61\x88\xda\xed\x1\x47\x84\xf5\x73\x25\xf9\x1c\x42\x86\x7\xf3\x5b\x1a\x1\xb3\xeb\x24\x32\x8d\xf6\xed\x7c\x4b\xeb\x3c\x36\x42\x28\xdf\xdf\xb6\xbe\xd9\x8c\x52\xd3\x2b\x8\x90\x8c\xe7\x98\x31\xe2\x32\x8e\xfc\x11\x48\x0\xa8\x6a\x42\x4a\x2\xc6\x4b\x9\xf1\xe3\x49\xf3\x45\x1f\xe\xbc\x56\xe2\xe4\xdf\xfb\xeb\x61\xfa\x24\xc1\x63\x75\xbb\x47\x75\xaf\xe1\x53\x16\x96\x21\x85\x26\x11\xb3\x76\xe3\x23\xa1\x6b\x74\x37\xd0\xde\x6\x90\x71\x5d\x43\x88\x9b\x0\x54\xa6\x75\x2f\xa1\xc2\xb\x73\x20\x1d\xb6\x21\x79\x57\x3f\xfa\x9\xbe\x8a\x33\xc3\x52\xf0\x1d\x82\x31\xd1\x55\xb5\x6c\x99\x25\xcf\x5c\x32\xce\xe9\xd\xfa\x69\x2c\xd5\xd\xc5\x6d\x86\xd0\xc\x3b\x6\x50\x79\xe8\xc3\xae\x4\xe6\xcd\x51\xe4\x26\x9b\x4f\x7e\xa6\xf\xab\xd8\xe5\xde\xa9\x0\x95\xbe\xa3\x9d\x5d\xb2\x9\x70\x18\x1c\xf0\xac\x29\x23\x2\x29\x28\xd2\x74\x35\x57\x62\xf\x24\xea\x5e\x33\xc2\x92\xf3\x78\x4d\x30\x1e\xa1\x99\xa9\x82\xb0\x42\x31\x8d\xad\x8a\xbc\xfc\xd4\x57\x47\x3e\xb4\x50\xdd\x6e\x2c\x80\x4d\x22\xf1\xfb\x57\xc4\xdd\x17\xe1\x8a\x36\x4a\xb3\x37\xca\xc9\x4e\xab\xd5\x69\xc4\xf4\xbc\xb\x3b\x44\x4b\x29\x9c\xee\xd4\x35\x22\x21\xb0\x1f\x27\x64\xa8\x51\x1b\xf0\x9f\x19\x5c\xfb\x5a\x64\x74\x70\x45\x9\xf5\x64\xfe\x1a\x2d\xc9\x14\x4\x14\xcf\xd5\x7d\x60\xaf\x94\x39\x94\xe2\x7d\x79\x82\xd0\x65\x3b\x6b\x9c\x19\x84\xb4\x6d\xb3\xc\x99\xc0\x56\xa8\xbd\x73\xce\x5\x84\x3e\x30\xaa\xc4\x9b\x1b\x4\x2a\x9f\xd7\x43\x2b\x23\xdf\xbf\xaa\xd5\xc2\x43\x2d\x70\xab\xdc\x75\xad\xac\xf7\xc0\xbe\x67\xb2\x74\xed\x67\x10\x4a\x92\x60\xc1\x40\x50\x19\x8a\x8a\x8c\x9\xe\x72\xe1\x73\x5e\xe8\x41\x85\x63\x9f\x3f\xd7\x7d\xc4\xfb\x22\x5d\x92\x6c\xb3\x1e\xe2\x50\x2f\x82\xa8\x28\xc0\xb5\xd7\x5f\x68\xd\x2c\x2d\xaf\x7e\xfa\x2e\x8\xf\x1f\x70\x9f\xe9\x19\x72\x55\xf8\xfb\x51\xd2\x33\x5d\xa0\xd3\x2b\xa\x6c\xbc\x4e\xcf\x36\x4d\xdc\x3b\xe9\x3e\x81\x7c\x61\xdb\x20\x2d\x3a\xc3\xb3\xc\x1e\x0\xb9\x7c\xf5\xca\x10\x5f\x3a\x71\xb3\xe4\x20\xdb\xc\x2a\x98\x63\x45\x0\x58\xf6\x68\xe4\xb\xda\x13\x3b\x60\x5c\x76\xdb\xb9\x97\x71\xe4\xd9\xb7\xdb\xbd\x68\xc7\x84\x84\xaa\x7c\x68\x62\x5e\x16\xfc\xba\x72\xaa\x9a\xa9\xeb\x7c\x75\x47\x97\x7e\xad\xe2\xd9\x91\xe8\xe4\xa5\x31\xd7\x1\x8e\xa2\x11\x88\x95\xb9\xf2\x9b\xd3\x7f\x1b\x81\x22\xf7\x98\x60\xa\x64\xa6\xc1\xf6\x49\xc7\xe3\x7\x4d\x94\x7a\xcf\x6e\x68\xc\x1b\x3f\x6e\x2e\xee\x92\xfa\x52\xb3\x59\xf8\xf1\x8f\x6a\x66\xa3\x82\x76\x4a\x7\x1a\xc7\xdd\xf5\xda\x9c\x3c\x24\xbf\xfd\x42\xa1\x10\x64\x6a\xf\x89\xee\x36\xa5\xce\x99\x48\x6a\xf0\x9f\x9e\x69\xa4\x40\x20\xe9\x16\x15\xf7\xdb\x75\x2\xcb\xe9\x73\x8b\x3b\x49\x2f\xf0\xaf\x51\x6\x5c\xdf\x27\x27\x49\x6a\xd1\xcc\xc7\xb5\x63\xb5\xfc\xb8\x5c\x87\x7f\x84\xb4\xcc\x14\xa9\x53\xda\xa4\x56\xf8\xb6\x1b\xcc\x40\x27\x52\x6\x5a\x13\x81\xd7\x3a\xd4\x3b\xfb\x49\x65\x31\x33\xb2\xfa\xcd\xad\x58\x4e\x2b\xae\xd2\x20\xfb\x1a\x48\xb4\x3f\x9a\xd8\x7a\x35\x4a\xc8\xee\x88\x5e\x7\x66\x54\xb9\xec\x9f\xa3\xe3\xb9\x37\xaa\x49\x76\x31\xda\x74\x2d\x3c\xa4\x65\x10\x32\x38\xf0\xde\xd3\x99\x17\xaa\x71\xaa\x8f\xf\x8c\xaf\xa2\xf8\x5d\x64\xba\x1d\xa3\xef\x96\x73\xe8\xa1\x2\x8d\xc\x6d\xb8\x6\x90\xb8\x8\x56\x2c\xa7\x6\xc9\xc2\x38\xdb\x7c\x63\xb1\x57\x8e\xea\x7c\x79\xf3\x49\x1d\xfe\x9f\xf3\x6e\xb1\x1d\xba\x19\x80\x1a\xa\xd3\xb0\x26\x21\x40\xb1\x7c\xf9\x4d\x8d\x10\xc1\x7e\xf4\xf6\x3c\xa8\xfd\x7c\xa3\x92\xb2\xf\xaa\xcc\xa6\x11\xfe\x4\xe3\xd1\x7a\x32\x89\xdf\xd\xc4\x8f\x79\x6b\xca\x16\x7c\x6e\xf9\xad\xf\xf6\xfe\x27\xdb\xc4\x13\x70\xf1\x62\x1a\x4f\x79\x40\xc9\x9b\x8b\x21\xea\x84\xfa\xf5\xf1\x89\xce\xb7\x55\xa\x80\x39\x2f\x55\x36\x16\x9c\x7b\x8\xbd\x87\xd\xa5\x32\xf1\x52\x7c\xe8\x55\x60\x5b\xd7\x69\xe4\xfc\xfa\x12\x85\x96\xea\x50\x28\xab\x8a\xf7\xbb\xe\x53\x74\xca\xa6\x27\x9\xc2\xb5\xde\x18\x14\xd9\xea\xe5\x29\x1c\x40\x56\xcf\xd7\xae\x5\x3f\x65\xaf\x5\x73\xe2\x35\x96\x27\x7\x14\xc0\xad\x33\xf1\xdc\x44\x7a\x89\x17\x77\xd2\x9c\x58\x60\xf0\x3f\x7b\x2d\x2e\x57\x95\x54\x87\xed\xf2\xc7\x4c\xf0\xae\x56\x29\x19\x7d\x66\x4b\x9b\x83\x84\x42\x3b\x1\x25\x66\x8e\x2\xde\xb9\x83\x54\x19\xf6\x9f\x79\xd\x67\xc5\x1d\x7a\x44\x2\x98\xa7\x16\x1c\x29\xd\x74\xff\x85\x40\x6\xef\x2c\xa9\xc6\xf5\x53\x7\x6\xae\xe4\xfa\x5f\xd8\x39\x4d\xf1\x9b\x6b\xd9\x24\x84\xfe\x3\x4c\xb2\x3f\xdf\xa1\x5\x9e\x50\x14\x5a\xd9\x1a\xa2\xa7\xfa\xfa\x17\xf7\x78\xd6\xb5\x92\x61\x91\xac\x36\xfa\x56\xd\x38\x32\x18\x85\x8\x58\x37\xf0\x4b\xdb\x59\xe7\xa4\x34\xc0\x1b\x1\xaf\x2d\xde\xa1\xaa\x5d\xd3\xec\xe1\xd4\xf7\xe6\x54\x68\xf0\x51\x97\xa7\x89\xea\x24\xad\xd3\x6e\x47\x93\x8b\x4b\xb4\xf7\x1c\x42\x6\x67\xe8\x99\xf6\xf5\x7b\x85\xb5\x65\xb5\xb5\xd2\x37\xf5\xf3\x2\xa6\x4d\x11\xa7\xdc\x51\x9\x7f\xa0\xd8\x88\x1c\x13\x71\xae\x9c\xb7\x7b\x34\xd6\x4e\x68\x26\x83\x51\xaf\x1d\xee\x8b\xbb\x69\x43\x2b\x9e\x8a\xbc\x2\xe\xa0\x1b\xe0\xa8\x5f\x6f\xaf\x1b\x8f\xe7\x64\x71\x74\x11\x7e\xa8\xd8\xf9\x97\x6\xc3\xb6\xfb\xfb\xb7\x3d\x35\x9d\x3b\x52\xed\x54\xca\xf4\x81\x1\x2d\x1b\xc3\xa7\x0\x3d\x1a\x39\x54\xe1\xf6\xff\xed\x6f\xb\x5a\x68\xda\x58\xdd\xa9\xcf\x5c\x4a\xe5\x9\x4e\xde\x9d\xbc\x3e\xee\x5a\x0\x3b\x2c\x87\x10\x65\x60\xdd\xd7\x56\xd1\x4c\x64\x45\xe4\x21\xec\x78\xf8\x25\x7a\x3e\x16\x5d\x9\x53\x14\xbe\x4f\xae\x87\xd8\xd1\xaa\x3c\xf6\x3e\xa4\x70\x8c\x5e\x70\xa4\xb3\x6b\x66\x73\xd3\xbf\x31\x6\x19\x62\x93\x15\xf2\x86\xe4\x52\x7e\x53\x4c\x12\x38\xcc\x34\x7d\x57\xf6\x42\x93\x8a\xc4\xee\x5c\x8a\xe1\x52\x8f\x56\x64\xf6\xa6\xd1\x91\x57\x70\xcd\x11\x76\xf5\x59\x60\x60\x3c\xc1\xc3\xb\x7f\x58\x1a\x50\x91\xf1\x68\x8f\x6e\x74\x74\xa8\x51\xb\xf7\x7a\x98\x37\xf2\xa\xe\xa4\x97\x4\xb8\x9b\xfd\xa0\xea\xf7\xd\xe1\xdb\x3\xf0\x31\x29\xf8\xdd\x6b\x8b\x5d\xd8\x59\xa9\x29\xcf\x9a\x79\x89\x19\x63\x46\x9\x79\x6a\x11\xda\x63\x68\x48\x77\x23\xfb\x7d\x3a\x43\xcb\x2\x3b\x7a\x6d\x10\x2a\x9e\xac\xf1\xd4\x19\xf8\x23\x64\x1d\x2c\x5f\xf2\xb0\x5c\x23\x27\xf7\x27\x30\x16\x37\xb1\x90\xab\x38\xfb\x55\xcd\x78\x58\xd4\x7d\x43\xf6\x45\x5e\x55\x8d\xb1\x2\x65\x58\xb4\x13\x4b\x36\xf7\xcc\xfe\x3d\xb\x82\xe2\x12\x11\xbb\xe6\xb8\x3a\x48\x71\xc7\x50\x6\x16\x3a\xe6\x7c\x5\xc7\xc8\x4d\x2f\x8\x6a\x17\x9a\x95\x97\x50\x68\xdc\x28\x18\xc4\x61\x38\xb9\xe0\x3e\x78\xdb\x29\xe0\x9f\x52\xdd\xf8\x4f\x91\xc1\xd0\x33\xa1\x7a\x8e\x30\x13\x82\x7\x9f\xd3\x31\xf\x23\xbe\x32\x5a\x75\xcf\x96\xb2\xec\xb5\x32\xac\x21\xd1\x82\x33\xd3\x15\x74\xbd\x90\xf1\x2c\xe6\x5f\x8d\xe3\x2\xe8\xe9\xc4\xca\x96\xeb\xe\xbc\x91\xf4\xb9\xea\xd9\x1b\x75\xbd\xe1\xac\x2a\x5\x37\x52\x9b\x1b\x3f\x5a\xdc\x21\xc3\x98\xbb\xaf\xa3\xf2\x0\xbf\xd\x30\x89\x5\xcc\xa5\x76\xf5\x6\xf0\xc6\x54\x8a\x5d\xd4\x1e\xc1\xf2\xce\xb0\x62\xc8\xfc\x59\x42\x9a\x90\x60\x55\xfe\x88\xa5\x8b\xb8\x33\xc\x23\x24\xd\x15\x70\x37\x1e\x3d\xf6\xd2\xea\x92\x10\xb2\xc4\x51\xac\xf2\xac\xf3\x6b\x6c\xaa\xcf\x12\xc5\x6c\x90\x50\xb5\xc\xfc\x1a\x15\x52\xe9\x26\xc6\x52\xa4\xe7\x81\x69\xe1\xe7\x9e\x30\x1\xec\x84\x89\xb2\xd\x66\xdd\xce\x28\x5c\xec\x98\x46\x68\x21\x9f\x88\x3f\x1f\x42\x77\xce\xd0\x61\xd4\x20\xa7\xff\x53\xad\x37\xd0\x17\x35\xc9\xfc\xba\xa\x78\x3f\xf2\xcc\x86\x89\xe8\x4b\x3c\x48\x33\x9\x7f\xc6\xc0\xdd\xb8\xfd\x7a\x66\x66\x65\xeb\x47\xa7\x4\x28\xa3\x19\x8e\xa9\xb1\x13\x67\x62\x70\xcf\xd7",
	.ilen	= 2027,
	.result	= "\x74\xa6\x3e\xe4\xb1\xcb\xaf\xb0\x40\xe5\xf\x9e\xf1\xf2\x89\xb5\x42\x34\x8a\xa1\x3\xb7\xe9\x57\x46\xbe\x20\xe4\x6e\xb0\xeb\xff\xea\x7\x7e\xef\xe2\x55\x9f\xe5\x78\x3a\xb7\x83\xc2\x18\x40\x7b\xeb\xcd\x81\xfb\x90\x12\x9e\x46\xa9\xd6\x4a\xba\xb0\x62\xdb\x6b\x99\xc4\xdb\x54\x4b\xb8\xa5\x71\xcb\xcd\x63\x32\x55\xfb\x31\xf0\x38\xf5\xbe\x78\xe4\x45\xce\x1b\x6a\x5b\xe\xf4\x16\xe4\xb1\x3d\xf6\x63\x7b\xa7\xc\xde\x6f\x8f\x74\xdf\xe0\x1e\x9d\xce\x8f\x24\xef\x23\x35\x33\x7b\x83\x34\x23\x58\x74\x14\x77\x1f\xc2\x4f\x4e\xc6\x89\xf9\x52\x9\x37\x64\x14\xc4\x1\x6b\x9d\x77\xe8\x90\x5d\xa8\x4a\x2a\xef\x5c\x7f\xeb\xbb\xb2\xc6\x93\x99\x66\xdc\x7f\xd4\x9e\x2a\xca\x8d\xdb\xe7\x20\xcf\xe4\x73\xae\x49\x7d\x64\xf\xe\x28\x46\xa9\xa8\x32\xe4\xe\xf6\x51\x53\xb8\x3c\xb1\xff\xa3\x33\x41\x75\xff\xf1\x6f\xf1\xfb\xbb\x83\x7f\x6\x9b\xe7\x1b\xa\xe0\x5c\x33\x60\x5b\xdb\x5b\xed\xfe\xa5\x16\x19\x72\xa3\x64\x23\x0\x2\xc7\xf3\x6a\x81\x3e\x44\x1d\x79\x15\x5f\x9a\xde\xe2\xfd\x1b\x73\xc1\xbc\x23\xba\x31\xd2\x50\xd5\xad\x7f\x74\xa7\xc9\xf8\x3e\x2b\x26\x10\xf6\x3\x36\x74\xe4\xe\x6a\x72\xb7\x73\xa\x42\x28\xc2\xad\x5e\x3\xbe\xb8\xb\xa8\x5b\xd4\xb8\xba\x52\x89\xb1\x9b\xc1\xc3\x65\x87\xed\xa5\xf4\x86\xfd\x41\x80\x91\x27\x59\x53\x67\x15\x78\x54\x8b\x2d\x3d\xc7\xff\x2\x92\x7\x5f\x7a\x4b\x60\x59\x3c\x6f\x5c\xd8\xec\x95\xd2\xfe\xa0\x3b\xd8\x3f\xd1\x69\xa6\xd6\x41\xb2\xf4\x4d\x12\xf4\x58\x3e\x66\x64\x80\x31\x9b\xa8\x4c\x8b\x7\xb2\xec\x66\x94\x66\x47\x50\x50\x5f\x18\xb\xe\xd6\xc0\x39\x21\x13\x9e\x33\xbc\x79\x36\x2\x96\x70\xf0\x48\x67\x2f\x26\xe9\x6d\x10\xbb\xd6\x3f\xd1\x64\x7a\x2e\xbe\xc\x61\xf0\x75\x42\x38\x23\xb1\x9e\x9f\x7c\x67\x66\xd9\x58\x9a\xf1\xbb\x41\x2a\x8d\x65\x84\x94\xfc\xdc\x6a\x50\x64\xdb\x56\x33\x76\x0\x10\xed\xbe\xd2\x12\xf6\xf6\x1b\xa2\x16\xde\xae\x31\x95\xdd\xb1\x8\x7e\x4e\xee\xe7\xf9\xa5\xfb\x5b\x61\x43\x0\x40\xf6\x7e\x2\x4\x32\x4e\xc\xe2\x66\xd\xd7\x7\x98\xe\xf8\x72\x34\x6d\x95\x86\xd7\xcb\x31\x54\x47\xd0\x38\x29\x9c\x5a\x68\xd4\x87\x76\xc9\xe7\x7e\xe3\xf4\x81\x6d\x18\xcb\xc9\x5\xaf\xa0\xfb\x66\xf7\xf1\x1c\xc6\x14\x11\x4f\x2b\x79\x42\x8b\xbc\xac\xe7\x6c\xfe\xf\x58\xe7\x7c\x78\x39\x30\xb0\x66\x2c\x9b\x6d\x3a\xe1\xcf\xc9\xa4\xe\x6d\x6d\x8a\xa1\x3a\xe7\x28\xd4\x78\x4c\xa6\xa2\x2a\xa6\x3\x30\xd7\xa8\x25\x66\x87\x2f\x69\x5c\x4e\xdd\xa5\x49\x5d\x37\x4a\x59\xc4\xaf\x1f\xa2\xe4\xf8\xa6\x12\x97\xd5\x79\xf5\xe2\x4a\x2b\x5f\x61\xe4\x9e\xe3\xee\xb8\xa7\x5b\x2f\xf4\x9e\x6c\xfb\xd1\xc6\x56\x77\xba\x75\xaa\x3d\x1a\xa8\xb\xb3\x68\x24\x0\x10\x7f\xfd\xd7\xa1\x8d\x83\x54\x4f\x1f\xd8\x2a\xbe\x8a\xc\x87\xab\xa2\xde\xc3\x39\xbf\x9\x3\xa5\xf3\x5\x28\xe1\xe1\xee\x39\x70\x9c\xd8\x81\x12\x1e\x2\x40\xd2\x6e\xf0\xeb\x1b\x3d\x22\xc6\xe5\xe3\xb4\x5a\x98\xbb\xf0\x22\x28\x8d\xe5\xd3\x16\x48\x24\xa5\xe6\x66\xc\xf9\x8\xf9\x7e\x1e\xe1\x28\x26\x22\xc7\xc7\xa\x32\x47\xfa\xa3\xbe\x3c\xc4\xc5\x53\xa\xd5\x94\x4a\xd7\x93\xd8\x42\x99\xb9\xa\xdb\x56\xf7\xb9\x1c\x53\x4f\xfa\xd3\x74\xad\xd9\x68\xf1\x1b\xdf\x61\xc6\x5e\xa8\x48\xfc\xd4\x4a\x4c\x3c\x32\xf7\x1c\x96\x21\x9b\xf9\xa3\xcc\x5a\xce\xd5\xd7\x8\x24\xf6\x1c\xfd\xdd\x38\xc2\x32\xe9\xb8\xe7\xb6\xfa\x9d\x45\x13\x2c\x83\xfd\x4a\x69\x82\xcd\xdc\xb3\x76\xc\x9e\xd8\xf4\x1b\x45\x15\xb4\x97\xe7\x58\x34\xe2\x3\x29\x5a\xbf\xb6\xe0\x5d\x13\xd9\x2b\xb4\x80\xb2\x45\x81\x6a\x2e\x6c\x89\x7d\xee\xbb\x52\xdd\x1f\x18\xe7\x13\x6b\x33\xe\xea\x36\x92\x77\x7b\x6d\x9c\x5a\x5f\x45\x7b\x7b\x35\x62\x23\xd1\xbf\xf\xd0\x8\x1b\x2b\x80\x6b\x7e\xf1\x21\x47\xb0\x57\xd1\x98\x72\x90\x34\x1c\x20\x4\xff\x3d\x5c\xee\xe\x57\x5f\x6f\x24\x4e\x3c\xea\xfc\xa5\xa9\x83\xc9\x61\xb4\x51\x24\xf8\x27\x5e\x46\x8c\xb1\x53\x2\x96\x35\xba\xb8\x4c\x71\xd3\x15\x59\x35\x22\x20\xad\x3\x9f\x66\x44\x3b\x9c\x35\x37\x1f\x9b\xbb\xf3\xdb\x35\x63\x30\x64\xaa\xa2\x6\xa8\x5d\xbb\xe1\x9f\x70\xec\x82\x11\x6\x36\xec\x8b\x69\x66\x24\x44\xc9\x4a\x57\xbb\x9b\x78\x13\xce\x9c\xc\xba\x92\x93\x63\xb8\xe2\x95\xf\xf\x16\x39\x52\xfd\x3a\x6d\x2\x4b\xdf\x13\xd3\x2a\x22\xb4\x3\x7c\x54\x49\x96\x68\x54\x10\xfa\xef\xaa\x6c\xe8\x22\xdc\x71\x16\x13\x1a\xf6\x28\xe5\x6d\x77\x3d\xcd\x30\x63\xb1\x70\x52\xa1\xc5\x94\x5f\xcf\xe8\xb8\x26\x98\xf7\x6\xa0\xa\x70\xfa\x3\x80\xac\xc1\xec\xd6\x4c\x54\xd7\xfe\x47\xb6\x88\x4a\xf7\x71\x24\xee\xf3\xd2\xc2\x4a\x7f\xfe\x61\xc7\x35\xc9\x37\x67\xcb\x24\x35\xda\x7e\xca\x5f\xf3\x8d\xd4\x13\x8e\xd6\xcb\x4d\x53\x8f\x53\x1f\xc0\x74\xf7\x53\xb9\x5e\x23\x37\xba\x6e\xe3\x9d\x7\x55\x25\x7b\xe6\x2a\x64\xd1\x32\xdd\x54\x1b\x4b\xc0\xe1\xd7\x69\x58\xf8\x93\x29\xc4\xdd\x23\x2f\xa5\xfc\x9d\x7e\xf8\xd4\x90\xcd\x82\x55\xdc\x16\x16\x9f\x7\x52\x9b\x9d\x25\xed\x32\xc5\x7b\xdf\xf6\x83\x46\x3d\x65\xb7\xef\x87\x7a\x12\x69\x8f\x6\x7c\x51\x15\x4a\x8\xe8\xac\x9a\xc\x24\xa7\x27\xd8\x46\x2f\xe7\x1\xe\x1c\xc6\x91\xb0\x6e\x85\x65\xf0\x29\xd\x2e\x6b\x3b\xfb\x4b\xdf\xe4\x80\x93\x3\x66\x46\x3e\x8a\x6e\xf3\x5e\x4d\x62\xe\x49\x5\xaf\xd4\xf8\x21\x20\x61\x1d\x39\x17\xf4\x61\x47\x95\xfb\x15\x2e\xb3\x4f\xd0\x5d\xf5\x7d\x40\xda\x90\x3c\x6b\xcb\x17\x0\x13\x3b\x64\x34\x1b\xf0\xf2\xe5\x3b\xb2\xc7\xd3\x5f\x3a\x44\xa6\x9b\xb7\x78\xe\x42\x5d\x4c\xc1\xe9\xd2\xcb\xb7\x78\xd1\xfe\x9a\xb5\x7\xe9\xe0\xbe\xe2\x8a\xa7\x1\x83\x0\x8c\x5c\x8\xe6\x63\x12\x92\xb7\xb7\xa6\x19\x7d\x38\x13\x38\x92\x87\x24\xf9\x48\xb3\x5e\x87\x6a\x40\x39\x5c\x3f\xed\x8f\xee\xdb\x15\x82\x6\xda\x49\x21\x2b\xb5\xbf\x32\x7c\x9f\x42\x28\x63\xcf\xaf\x1e\xf8\xc6\xa0\xd1\x2\x43\x57\x62\xec\x9b\xf\x1\x9e\x71\xd8\x87\x9d\x1\xc1\x58\x77\xd9\xaf\xb1\x10\x7e\xdd\xa6\x50\x96\xe5\xf0\x72\x0\x6d\x4b\xf8\x2a\x8f\x19\xf3\x22\x88\x11\x4a\x8b\x7c\xfd\xb7\xed\xe1\xf6\x40\x39\xe0\xe9\xf6\x3d\x25\xe6\x74\x3c\x58\x57\x7f\xe1\x22\x96\x47\x31\x91\xba\x70\x85\x28\x6b\x9f\x6e\x25\xac\x23\x66\x2f\x29\x88\x28\xce\x8c\x5c\x88\x53\xd1\x3b\xcc\x6a\x51\xb2\xe1\x28\x3f\x91\xb4\xd\x0\x3a\xe3\xf8\xc3\x8f\xd7\x96\x62\xe\x2e\xfc\xc8\x6c\x77\xa6\x1d\x22\xc1\xb8\xe6\x61\xd7\x67\x36\x13\x7b\xbb\x9b\x59\x9\xa6\xdf\xf7\x6b\xa3\x40\x1a\xf5\x4f\xb4\xda\xd3\xf3\x81\x93\xc6\x18\xd9\x26\xee\xac\xf0\xaa\xdf\xc5\x9c\xca\xc2\xa2\xcc\x7b\x5c\x24\xb0\xbc\xd0\x6a\x4d\x89\x9\xb8\x7\xfe\x87\xad\xa\xea\xb8\x42\xf9\x5e\xb3\x3e\x36\x4c\xaf\x75\x9e\x1c\xeb\xbd\xbc\xbb\x80\x40\xa7\x3a\x30\xbf\xa8\x44\xf4\xeb\x38\xad\x29\xba\x23\xed\x41\xc\xea\xd2\xbb\x41\x18\xd6\xb9\xba\x65\x2b\xa3\x91\x6d\x1f\xa9\xf4\xd1\x25\x8d\x4d\x38\xff\x64\xa0\xec\xde\xa6\xb6\x79\xab\x8e\x33\x6c\x47\xde\xaf\x94\xa4\xa5\x86\x77\x55\x9\x92\x81\x31\x76\xc7\x34\x22\x89\x8e\x3d\x26\x26\xd7\xfc\x1e\x16\x72\x13\x33\x63\xd5\x22\xbe\xb8\x4\x34\x84\x41\xbb\x80\xd0\x9f\x46\x48\x7\xa7\xfc\x2b\x3a\x75\x55\x8c\xc7\x6a\xbd\x7e\x46\x8\x84\xf\xd5\x74\xc0\x82\x8e\xaa\x61\x5\x1\xb2\x47\x6e\x20\x6a\x2d\x58\x70\x48\x32\xa7\x37\xd2\xb8\x82\x1a\x51\xb9\x61\xdd\xfd\x9d\x6b\xe\x18\x97\xf8\x45\x5f\x87\x10\xcf\x34\x72\x45\x26\x49\x70\xe7\xa3\x78\xe0\x52\x89\x84\x94\x83\x82\xc2\x69\x8f\xe3\xe1\x3f\x60\x74\x88\xc4\xf7\x75\x2c\xfb\xbd\xb6\xc4\x7e\x10\xa\x6c\x90\x4\x9e\xc3\x3f\x59\x7c\xce\x31\x18\x60\x57\x73\x46\x94\x7d\x6\xa0\x6d\x44\xec\xa2\xa\x9e\x5\x15\xef\xca\x5c\xbf\x0\xeb\xf7\x3d\x32\xd4\xa5\xef\x49\x89\x5e\x46\xb0\xa6\x63\x5b\x8a\x73\xae\x6f\xd5\x9d\xf8\x4f\x40\xb5\xb2\x6e\xd3\xb6\x1\xa9\x26\xa2\x21\xcf\x33\x7a\x3a\xa4\x23\x13\xb0\x69\x6a\xee\xce\xd8\x9d\x1\x1d\x50\xc1\x30\x6c\xb1\xcd\xa0\xf0\xf0\xa2\x64\x6f\xbb\xbf\x5e\xe6\xab\x87\xb4\xf\x4f\x15\xaf\xb5\x25\xa1\xb2\xd0\x80\x2c\xfb\xf9\xfe\xd2\x33\xbb\x76\xfe\x7c\xa8\x66\xf7\xe7\x85\x9f\x1f\x85\x57\x88\xe1\xe9\x63\xe4\xd8\x1c\xa1\xfb\xda\x44\x5\x2e\x1d\x3a\x1c\xff\xc8\x3b\xc0\xfe\xda\x22\xb\x43\xd6\x88\x39\x4c\x4a\xa6\x69\x18\x93\x42\x4e\xb5\xcc\x66\xd\x9\xf8\x1e\x7c\xd3\x3c\x99\xd\x50\x1d\x62\xe9\x57\x6\xbf\x19\x88\xdd\xad\x7b\x4f\xf9\xc7\x82\x6d\x8d\xc8\xc4\xc5\x78\x17\x20\x15\xc5\x52\x41\xcf\x5b\xd6\x7f\x94\x2\x41\xe0\x40\x22\x3\x5e\xd1\x53\xd4\x86\xd3\x2c\x9f\xf\x96\xe3\x6b\x9a\x76\x32\x6\x47\x4b\x11\xb3\xdd\x3\x65\xbd\x9b\x1\xda\x9c\xb9\x7e\x3f\x6a\xc4\x7b\xea\xd4\x3c\xb9\xfb\x5c\x6b\x64\x33\x52\xba\x64\x78\x8f\xa4\xaf\x7a\x61\x8d\xbc\xc5\x73\xe9\x6b\x58\x97\x4b\xbf\x63\x22\xd3\x37\x2\x54\xc5\xb9\x16\x4a\xf0\x19\xd8\x94\x57\xb8\x8a\xb3\x16\x3b\xd0\x84\x8e\x67\xa6\xa3\x7d\x78\xec\x0",
	.failure = true
} };

static const struct chacha20poly1305_testvec xchacha20poly1305_enc_vectors[] __initconst = { {
	.key	= "\x1c\x92\x40\xa5\xeb\x55\xd3\x8a\xf3\x33\x88\x86\x04\xf6\xb5\xf0\x47\x39\x17\xc1\x40\x2b\x80\x09\x9d\xca\x5c\xbc\x20\x70\x75\xc0",
	.nonce	= "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10\x11\x12\x13\x14\x15\x16\x17",
	.nlen	= 8,
	.assoc	= "\xf3\x33\x88\x86\x00\x00\x00\x00\x00\x00\x4e\x91",
	.alen	= 12,
	.input	= "\x49\x6e\x74\x65\x72\x6e\x65\x74\x2d\x44\x72\x61\x66\x74\x73\x20\x61\x72\x65\x20\x64\x72\x61\x66\x74\x20\x64\x6f\x63\x75\x6d\x65\x6e\x74\x73\x20\x76\x61\x6c\x69\x64\x20\x66\x6f\x72\x20\x61\x20\x6d\x61\x78\x69\x6d\x75\x6d\x20\x6f\x66\x20\x73\x69\x78\x20\x6d\x6f\x6e\x74\x68\x73\x20\x61\x6e\x64\x20\x6d\x61\x79\x20\x62\x65\x20\x75\x70\x64\x61\x74\x65\x64\x2c\x20\x72\x65\x70\x6c\x61\x63\x65\x64\x2c\x20\x6f\x72\x20\x6f\x62\x73\x6f\x6c\x65\x74\x65\x64\x20\x62\x79\x20\x6f\x74\x68\x65\x72\x20\x64\x6f\x63\x75\x6d\x65\x6e\x74\x73\x20\x61\x74\x20\x61\x6e\x79\x20\x74\x69\x6d\x65\x2e\x20\x49\x74\x20\x69\x73\x20\x69\x6e\x61\x70\x70\x72\x6f\x70\x72\x69\x61\x74\x65\x20\x74\x6f\x20\x75\x73\x65\x20\x49\x6e\x74\x65\x72\x6e\x65\x74\x2d\x44\x72\x61\x66\x74\x73\x20\x61\x73\x20\x72\x65\x66\x65\x72\x65\x6e\x63\x65\x20\x6d\x61\x74\x65\x72\x69\x61\x6c\x20\x6f\x72\x20\x74\x6f\x20\x63\x69\x74\x65\x20\x74\x68\x65\x6d\x20\x6f\x74\x68\x65\x72\x20\x74\x68\x61\x6e\x20\x61\x73\x20\x2f\xe2\x80\x9c\x77\x6f\x72\x6b\x20\x69\x6e\x20\x70\x72\x6f\x67\x72\x65\x73\x73\x2e\x2f\xe2\x80\x9d",
	.ilen	= 265,
	.result	= "\x1a\x6e\x3a\xd9\xfd\x41\x3f\x77\x54\x72\x0a\x70\x9a\xa0\x29\x92\x2e\xed\x93\xcf\x0f\x71\x88\x18\x7a\x9d\x2d\x24\xe0\xf5\xea\x3d\x55\x64\xd7\xad\x2a\x1a\x1f\x7e\x86\x6d\xb0\xce\x80\x41\x72\x86\x26\xee\x84\xd7\xef\x82\x9e\xe2\x60\x9d\x5a\xfc\xf0\xe4\x19\x85\xea\x09\xc6\xfb\xb3\xa9\x50\x09\xec\x5e\x11\x90\xa1\xc5\x4e\x49\xef\x50\xd8\x8f\xe0\x78\xd7\xfd\xb9\x3b\xc9\xf2\x91\xc8\x25\xc8\xa7\x63\x60\xce\x10\xcd\xc6\x7f\xf8\x16\xf8\xe1\x0a\xd9\xde\x79\x50\x33\xf2\x16\x0f\x17\xba\xb8\x5d\xd8\xdf\x4e\x51\xa8\x39\xd0\x85\xca\x46\x6a\x10\xa7\xa3\x88\xef\x79\xb9\xf8\x24\xf3\xe0\x71\x7b\x76\x28\x46\x3a\x3a\x1b\x91\xb6\xd4\x3e\x23\xe5\x44\x15\xbf\x60\x43\x9d\xa4\xbb\xd5\x5f\x89\xeb\xef\x8e\xfd\xdd\xb4\x0d\x46\xf0\x69\x23\x63\xae\x94\xf5\x5e\xa5\xad\x13\x1c\x41\x76\xe6\x90\xd6\x6d\xa2\x8f\x97\x4c\xa8\x0b\xcf\x8d\x43\x2b\x9c\x9b\xc5\x58\xa5\xb6\x95\x9a\xbf\x81\xc6\x54\xc9\x66\x0c\xe5\x4f\x6a\x53\xa1\xe5\x0c\xba\x31\xde\x34\x64\x73\x8a\x3b\xbd\x92\x01\xdb\x71\x69\xf3\x58\x99\xbc\xd1\xcb\x4a\x05\xe2\x58\x9c\x25\x17\xcd\xdc\x83\xb7\xff\xfb\x09\x61\xad\xbf\x13\x5b\x5e\xed\x46\x82\x6f\x22\xd8\x93\xa6\x85\x5b\x40\x39\x5c\xc5\x9c"
} };
static const struct chacha20poly1305_testvec xchacha20poly1305_dec_vectors[] __initconst = {	{
	.key	= "\x1c\x92\x40\xa5\xeb\x55\xd3\x8a\xf3\x33\x88\x86\x04\xf6\xb5\xf0\x47\x39\x17\xc1\x40\x2b\x80\x09\x9d\xca\x5c\xbc\x20\x70\x75\xc0",
	.nonce	= "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f\x10\x11\x12\x13\x14\x15\x16\x17",
	.nlen	= 8,
	.assoc	= "\xf3\x33\x88\x86\x00\x00\x00\x00\x00\x00\x4e\x91",
	.alen	= 12,
	.input	= "\x1a\x6e\x3a\xd9\xfd\x41\x3f\x77\x54\x72\x0a\x70\x9a\xa0\x29\x92\x2e\xed\x93\xcf\x0f\x71\x88\x18\x7a\x9d\x2d\x24\xe0\xf5\xea\x3d\x55\x64\xd7\xad\x2a\x1a\x1f\x7e\x86\x6d\xb0\xce\x80\x41\x72\x86\x26\xee\x84\xd7\xef\x82\x9e\xe2\x60\x9d\x5a\xfc\xf0\xe4\x19\x85\xea\x09\xc6\xfb\xb3\xa9\x50\x09\xec\x5e\x11\x90\xa1\xc5\x4e\x49\xef\x50\xd8\x8f\xe0\x78\xd7\xfd\xb9\x3b\xc9\xf2\x91\xc8\x25\xc8\xa7\x63\x60\xce\x10\xcd\xc6\x7f\xf8\x16\xf8\xe1\x0a\xd9\xde\x79\x50\x33\xf2\x16\x0f\x17\xba\xb8\x5d\xd8\xdf\x4e\x51\xa8\x39\xd0\x85\xca\x46\x6a\x10\xa7\xa3\x88\xef\x79\xb9\xf8\x24\xf3\xe0\x71\x7b\x76\x28\x46\x3a\x3a\x1b\x91\xb6\xd4\x3e\x23\xe5\x44\x15\xbf\x60\x43\x9d\xa4\xbb\xd5\x5f\x89\xeb\xef\x8e\xfd\xdd\xb4\x0d\x46\xf0\x69\x23\x63\xae\x94\xf5\x5e\xa5\xad\x13\x1c\x41\x76\xe6\x90\xd6\x6d\xa2\x8f\x97\x4c\xa8\x0b\xcf\x8d\x43\x2b\x9c\x9b\xc5\x58\xa5\xb6\x95\x9a\xbf\x81\xc6\x54\xc9\x66\x0c\xe5\x4f\x6a\x53\xa1\xe5\x0c\xba\x31\xde\x34\x64\x73\x8a\x3b\xbd\x92\x01\xdb\x71\x69\xf3\x58\x99\xbc\xd1\xcb\x4a\x05\xe2\x58\x9c\x25\x17\xcd\xdc\x83\xb7\xff\xfb\x09\x61\xad\xbf\x13\x5b\x5e\xed\x46\x82\x6f\x22\xd8\x93\xa6\x85\x5b\x40\x39\x5c\xc5\x9c",
	.ilen	= 281,
	.result	= "\x49\x6e\x74\x65\x72\x6e\x65\x74\x2d\x44\x72\x61\x66\x74\x73\x20\x61\x72\x65\x20\x64\x72\x61\x66\x74\x20\x64\x6f\x63\x75\x6d\x65\x6e\x74\x73\x20\x76\x61\x6c\x69\x64\x20\x66\x6f\x72\x20\x61\x20\x6d\x61\x78\x69\x6d\x75\x6d\x20\x6f\x66\x20\x73\x69\x78\x20\x6d\x6f\x6e\x74\x68\x73\x20\x61\x6e\x64\x20\x6d\x61\x79\x20\x62\x65\x20\x75\x70\x64\x61\x74\x65\x64\x2c\x20\x72\x65\x70\x6c\x61\x63\x65\x64\x2c\x20\x6f\x72\x20\x6f\x62\x73\x6f\x6c\x65\x74\x65\x64\x20\x62\x79\x20\x6f\x74\x68\x65\x72\x20\x64\x6f\x63\x75\x6d\x65\x6e\x74\x73\x20\x61\x74\x20\x61\x6e\x79\x20\x74\x69\x6d\x65\x2e\x20\x49\x74\x20\x69\x73\x20\x69\x6e\x61\x70\x70\x72\x6f\x70\x72\x69\x61\x74\x65\x20\x74\x6f\x20\x75\x73\x65\x20\x49\x6e\x74\x65\x72\x6e\x65\x74\x2d\x44\x72\x61\x66\x74\x73\x20\x61\x73\x20\x72\x65\x66\x65\x72\x65\x6e\x63\x65\x20\x6d\x61\x74\x65\x72\x69\x61\x6c\x20\x6f\x72\x20\x74\x6f\x20\x63\x69\x74\x65\x20\x74\x68\x65\x6d\x20\x6f\x74\x68\x65\x72\x20\x74\x68\x61\x6e\x20\x61\x73\x20\x2f\xe2\x80\x9c\x77\x6f\x72\x6b\x20\x69\x6e\x20\x70\x72\x6f\x67\x72\x65\x73\x73\x2e\x2f\xe2\x80\x9d"
} };

static inline void chacha20poly1305_selftest_encrypt_bignonce(u8 *dst, const u8 *src, const size_t src_len, const u8 *ad, const size_t ad_len, const u8 nonce[12], const u8 key[CHACHA20POLY1305_KEYLEN])
{
	bool have_simd = simd_get();
	struct poly1305_ctx poly1305_state;
	struct chacha20_ctx chacha20_state;
	union {
		u8 block0[POLY1305_KEY_SIZE];
		__le64 lens[2];
	} b = {{ 0 }};

	chacha20_init(&chacha20_state, key, 0);
	chacha20_state.counter[1] = le32_to_cpu(*(__le32 *)(nonce + 0));
	chacha20_state.counter[2] = le32_to_cpu(*(__le32 *)(nonce + 4));
	chacha20_state.counter[3] = le32_to_cpu(*(__le32 *)(nonce + 8));
	chacha20(&chacha20_state, b.block0, b.block0, sizeof(b.block0), have_simd);
	poly1305_init(&poly1305_state, b.block0, have_simd);
	poly1305_update(&poly1305_state, ad, ad_len, have_simd);
	poly1305_update(&poly1305_state, pad0, (0x10 - ad_len) & 0xf, have_simd);
	chacha20(&chacha20_state, dst, src, src_len, have_simd);
	poly1305_update(&poly1305_state, dst, src_len, have_simd);
	poly1305_update(&poly1305_state, pad0, (0x10 - src_len) & 0xf, have_simd);
	b.lens[0] = cpu_to_le64(ad_len);
	b.lens[1] = cpu_to_le64(src_len);
	poly1305_update(&poly1305_state, (u8 *)b.lens, sizeof(b.lens), have_simd);
	poly1305_finish(&poly1305_state, dst + src_len, have_simd);
	simd_put(have_simd);
	memzero_explicit(&chacha20_state, sizeof(chacha20_state));
	memzero_explicit(&b, sizeof(b));
}

static inline void chacha20poly1305_selftest_encrypt(u8 *dst, const u8 *src, const size_t src_len, const u8 *ad, const size_t ad_len, const u8 *nonce, const size_t nonce_len, const u8 key[CHACHA20POLY1305_KEYLEN])
{
	if (nonce_len == 8)
		chacha20poly1305_encrypt(dst, src, src_len, ad, ad_len, le64_to_cpup((__force __le64 *)nonce), key);
	else if (nonce_len == 12)
		chacha20poly1305_selftest_encrypt_bignonce(dst, src, src_len, ad, ad_len, nonce, key);
	else
		BUG();
}

static inline bool decryption_success(bool func_ret, bool expect_failure, int memcmp_result)
{
	if (expect_failure)
		return !func_ret;
	return func_ret && !memcmp_result;
}

enum { MAXIMUM_TEST_BUFFER_LEN = 3000 };

bool __init chacha20poly1305_selftest(void)
{
	size_t i;
	u8 computed_result[MAXIMUM_TEST_BUFFER_LEN], *heap_src, *heap_dst;
	bool success = true, ret, have_simd;
	struct scatterlist sg_src, sg_dst;

	heap_src = kmalloc(MAXIMUM_TEST_BUFFER_LEN, GFP_KERNEL);
	heap_dst = kmalloc(MAXIMUM_TEST_BUFFER_LEN, GFP_KERNEL);
	if (!heap_src || !heap_dst) {
		kfree(heap_src);
		kfree(heap_dst);
		pr_info("chacha20poly1305 self-test malloc: FAIL\n");
		return false;
	}

	for (i = 0; i < ARRAY_SIZE(chacha20poly1305_enc_vectors); ++i) {
		memset(computed_result, 0, sizeof(computed_result));
		chacha20poly1305_selftest_encrypt(computed_result, chacha20poly1305_enc_vectors[i].input, chacha20poly1305_enc_vectors[i].ilen, chacha20poly1305_enc_vectors[i].assoc, chacha20poly1305_enc_vectors[i].alen, chacha20poly1305_enc_vectors[i].nonce, chacha20poly1305_enc_vectors[i].nlen, chacha20poly1305_enc_vectors[i].key);
		if (memcmp(computed_result, chacha20poly1305_enc_vectors[i].result, chacha20poly1305_enc_vectors[i].ilen + POLY1305_MAC_SIZE)) {
			pr_info("chacha20poly1305 encryption self-test %zu: FAIL\n", i + 1);
			success = false;
		}
	}
	have_simd = simd_get();
	for (i = 0; i < ARRAY_SIZE(chacha20poly1305_enc_vectors); ++i) {
		if (chacha20poly1305_enc_vectors[i].nlen != 8)
			continue;
		memset(heap_dst, 0, MAXIMUM_TEST_BUFFER_LEN);
		memcpy(heap_src, chacha20poly1305_enc_vectors[i].input, chacha20poly1305_enc_vectors[i].ilen);
		sg_init_one(&sg_src, heap_src, chacha20poly1305_enc_vectors[i].ilen);
		sg_init_one(&sg_dst, heap_dst, chacha20poly1305_enc_vectors[i].ilen + POLY1305_MAC_SIZE);
		ret = chacha20poly1305_encrypt_sg(&sg_dst, &sg_src, chacha20poly1305_enc_vectors[i].ilen, chacha20poly1305_enc_vectors[i].assoc, chacha20poly1305_enc_vectors[i].alen, le64_to_cpup((__force __le64 *)chacha20poly1305_enc_vectors[i].nonce), chacha20poly1305_enc_vectors[i].key, have_simd);
		if (!ret || memcmp(heap_dst, chacha20poly1305_enc_vectors[i].result, chacha20poly1305_enc_vectors[i].ilen + POLY1305_MAC_SIZE)) {
			pr_info("chacha20poly1305 sg encryption self-test %zu: FAIL\n", i + 1);
			success = false;
		}
	}
	simd_put(have_simd);
	for (i = 0; i < ARRAY_SIZE(chacha20poly1305_dec_vectors); ++i) {
		memset(computed_result, 0, sizeof(computed_result));
		ret = chacha20poly1305_decrypt(computed_result, chacha20poly1305_dec_vectors[i].input, chacha20poly1305_dec_vectors[i].ilen, chacha20poly1305_dec_vectors[i].assoc, chacha20poly1305_dec_vectors[i].alen, le64_to_cpu(*(__force __le64 *)chacha20poly1305_dec_vectors[i].nonce), chacha20poly1305_dec_vectors[i].key);
		if (!decryption_success(ret, chacha20poly1305_dec_vectors[i].failure, memcmp(computed_result, chacha20poly1305_dec_vectors[i].result, chacha20poly1305_dec_vectors[i].ilen - POLY1305_MAC_SIZE))) {
			pr_info("chacha20poly1305 decryption self-test %zu: FAIL\n", i + 1);
			success = false;
		}
	}
	have_simd = simd_get();
	for (i = 0; i < ARRAY_SIZE(chacha20poly1305_dec_vectors); ++i) {
		memset(heap_dst, 0, MAXIMUM_TEST_BUFFER_LEN);
		memcpy(heap_src, chacha20poly1305_dec_vectors[i].input, chacha20poly1305_dec_vectors[i].ilen);
		sg_init_one(&sg_src, heap_src, chacha20poly1305_dec_vectors[i].ilen);
		sg_init_one(&sg_dst, heap_dst, chacha20poly1305_dec_vectors[i].ilen - POLY1305_MAC_SIZE);
		ret = chacha20poly1305_decrypt_sg(&sg_dst, &sg_src, chacha20poly1305_dec_vectors[i].ilen, chacha20poly1305_dec_vectors[i].assoc, chacha20poly1305_dec_vectors[i].alen, le64_to_cpup((__force __le64 *)chacha20poly1305_dec_vectors[i].nonce), chacha20poly1305_dec_vectors[i].key, have_simd);
		if (!decryption_success(ret, chacha20poly1305_dec_vectors[i].failure, memcmp(heap_dst, chacha20poly1305_dec_vectors[i].result, chacha20poly1305_dec_vectors[i].ilen - POLY1305_MAC_SIZE))) {
			pr_info("chacha20poly1305 sg decryption self-test %zu: FAIL\n", i + 1);
			success = false;
		}
	}
	simd_put(have_simd);
	for (i = 0; i < ARRAY_SIZE(xchacha20poly1305_enc_vectors); ++i) {
		memset(computed_result, 0, sizeof(computed_result));
		xchacha20poly1305_encrypt(computed_result, xchacha20poly1305_enc_vectors[i].input, xchacha20poly1305_enc_vectors[i].ilen, xchacha20poly1305_enc_vectors[i].assoc, xchacha20poly1305_enc_vectors[i].alen, xchacha20poly1305_enc_vectors[i].nonce, xchacha20poly1305_enc_vectors[i].key);
		if (memcmp(computed_result, xchacha20poly1305_enc_vectors[i].result, xchacha20poly1305_enc_vectors[i].ilen + POLY1305_MAC_SIZE)) {
			pr_info("xchacha20poly1305 encryption self-test %zu: FAIL\n", i + 1);
			success = false;
		}
	}
	for (i = 0; i < ARRAY_SIZE(xchacha20poly1305_dec_vectors); ++i) {
		memset(computed_result, 0, sizeof(computed_result));
		ret = xchacha20poly1305_decrypt(computed_result, xchacha20poly1305_dec_vectors[i].input, xchacha20poly1305_dec_vectors[i].ilen, xchacha20poly1305_dec_vectors[i].assoc, xchacha20poly1305_dec_vectors[i].alen, xchacha20poly1305_dec_vectors[i].nonce, xchacha20poly1305_dec_vectors[i].key);
		if (!decryption_success(ret, xchacha20poly1305_dec_vectors[i].failure, memcmp(computed_result, xchacha20poly1305_dec_vectors[i].result, xchacha20poly1305_dec_vectors[i].ilen - POLY1305_MAC_SIZE))) {
			pr_info("xchacha20poly1305 decryption self-test %zu: FAIL\n", i + 1);
			success = false;
		}
	}
	if (success)
		pr_info("chacha20poly1305 self-tests: pass\n");
	kfree(heap_src);
	kfree(heap_dst);
	return success;
}
#endif
