/*
Copyright 2015, 2016 OpenMarket Ltd
Copyright 2017 Vector Creations Ltd
Copyright 2018 New Vector Ltd

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
"use strict";

/*
 * TODO:
 * This class mainly serves to take all the syncing logic out of client.js and
 * into a separate file. It's all very fluid, and this class gut wrenches a lot
 * of MatrixClient props (e.g. _http). Given we want to support WebSockets as
 * an alternative syncing API, we may want to have a proper syncing interface
 * for HTTP and WS at some point.
 */

var _stringify = require("babel-runtime/core-js/json/stringify");

var _stringify2 = _interopRequireDefault(_stringify);

var _keys = require("babel-runtime/core-js/object/keys");

var _keys2 = _interopRequireDefault(_keys);

var _getIterator2 = require("babel-runtime/core-js/get-iterator");

var _getIterator3 = _interopRequireDefault(_getIterator2);

var _regenerator = require("babel-runtime/regenerator");

var _regenerator2 = _interopRequireDefault(_regenerator);

var _bluebird = require("bluebird");

var _bluebird2 = _interopRequireDefault(_bluebird);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

var User = require("./models/user");
var Room = require("./models/room");
var Group = require('./models/group');
var utils = require("./utils");
var Filter = require("./filter");
var EventTimeline = require("./models/event-timeline");

var DEBUG = true;

// /sync requests allow you to set a timeout= but the request may continue
// beyond that and wedge forever, so we need to track how long we are willing
// to keep open the connection. This constant is *ADDED* to the timeout= value
// to determine the max time we're willing to wait.
var BUFFER_PERIOD_MS = 80 * 1000;

// Number of consecutive failed syncs that will lead to a syncState of ERROR as opposed
// to RECONNECTING. This is needed to inform the client of server issues when the
// keepAlive is successful but the server /sync fails.
var FAILED_SYNC_ERROR_THRESHOLD = 3;

function getFilterName(userId, suffix) {
    // scope this on the user ID because people may login on many accounts
    // and they all need to be stored!
    return "FILTER_SYNC_" + userId + (suffix ? "_" + suffix : "");
}

function debuglog() {
    var _console;

    if (!DEBUG) {
        return;
    }
    (_console = console).log.apply(_console, arguments);
}

/**
 * <b>Internal class - unstable.</b>
 * Construct an entity which is able to sync with a homeserver.
 * @constructor
 * @param {MatrixClient} client The matrix client instance to use.
 * @param {Object} opts Config options
 * @param {module:crypto=} opts.crypto Crypto manager
 * @param {Function=} opts.canResetEntireTimeline A function which is called
 * with a room ID and returns a boolean. It should return 'true' if the SDK can
 * SAFELY remove events from this room. It may not be safe to remove events if
 * there are other references to the timelines for this room.
 * Default: returns false.
 * @param {Boolean=} opts.disablePresence True to perform syncing without automatically
 * updating presence.
 */
function SyncApi(client, opts) {
    this.client = client;
    opts = opts || {};
    opts.initialSyncLimit = opts.initialSyncLimit === undefined ? 8 : opts.initialSyncLimit;
    opts.resolveInvitesToProfiles = opts.resolveInvitesToProfiles || false;
    opts.pollTimeout = opts.pollTimeout || 30 * 1000;
    opts.pendingEventOrdering = opts.pendingEventOrdering || "chronological";
    if (!opts.canResetEntireTimeline) {
        opts.canResetEntireTimeline = function (roomId) {
            return false;
        };
    }
    this.opts = opts;
    this._peekRoomId = null;
    this._currentSyncRequest = null;
    this._syncState = null;
    this._catchingUp = false;
    this._running = false;
    this._keepAliveTimer = null;
    this._connectionReturnedDefer = null;
    this._notifEvents = []; // accumulator of sync events in the current sync response
    this._failedSyncCount = 0; // Number of consecutive failed /sync requests

    if (client.getNotifTimelineSet()) {
        client.reEmitter.reEmit(client.getNotifTimelineSet(), ["Room.timeline", "Room.timelineReset"]);
    }
}

/**
 * @param {string} roomId
 * @return {Room}
 */
SyncApi.prototype.createRoom = function (roomId) {
    var client = this.client;
    var room = new Room(roomId, {
        pendingEventOrdering: this.opts.pendingEventOrdering,
        timelineSupport: client.timelineSupport
    });
    client.reEmitter.reEmit(room, ["Room.name", "Room.timeline", "Room.redaction", "Room.receipt", "Room.tags", "Room.timelineReset", "Room.localEchoUpdated", "Room.accountData"]);
    this._registerStateListeners(room);
    return room;
};

/**
 * @param {string} groupId
 * @return {Group}
 */
SyncApi.prototype.createGroup = function (groupId) {
    var client = this.client;
    var group = new Group(groupId);
    client.reEmitter.reEmit(group, ["Group.profile", "Group.myMembership"]);
    client.store.storeGroup(group);
    return group;
};

/**
 * @param {Room} room
 * @private
 */
SyncApi.prototype._registerStateListeners = function (room) {
    var client = this.client;
    // we need to also re-emit room state and room member events, so hook it up
    // to the client now. We need to add a listener for RoomState.members in
    // order to hook them correctly. (TODO: find a better way?)
    client.reEmitter.reEmit(room.currentState, ["RoomState.events", "RoomState.members", "RoomState.newMember"]);
    room.currentState.on("RoomState.newMember", function (event, state, member) {
        member.user = client.getUser(member.userId);
        client.reEmitter.reEmit(member, ["RoomMember.name", "RoomMember.typing", "RoomMember.powerLevel", "RoomMember.membership"]);
    });
};

/**
 * @param {Room} room
 * @private
 */
SyncApi.prototype._deregisterStateListeners = function (room) {
    // could do with a better way of achieving this.
    room.currentState.removeAllListeners("RoomState.events");
    room.currentState.removeAllListeners("RoomState.members");
    room.currentState.removeAllListeners("RoomState.newMember");
};

/**
 * Sync rooms the user has left.
 * @return {Promise} Resolved when they've been added to the store.
 */
SyncApi.prototype.syncLeftRooms = function () {
    var client = this.client;
    var self = this;

    // grab a filter with limit=1 and include_leave=true
    var filter = new Filter(this.client.credentials.userId);
    filter.setTimelineLimit(1);
    filter.setIncludeLeaveRooms(true);

    var localTimeoutMs = this.opts.pollTimeout + BUFFER_PERIOD_MS;
    var qps = {
        timeout: 0 };

    return client.getOrCreateFilter(getFilterName(client.credentials.userId, "LEFT_ROOMS"), filter).then(function (filterId) {
        qps.filter = filterId;
        return client._http.authedRequest(undefined, "GET", "/sync", qps, undefined, localTimeoutMs);
    }).then(function (data) {
        var leaveRooms = [];
        if (data.rooms && data.rooms.leave) {
            leaveRooms = self._mapSyncResponseToRoomArray(data.rooms.leave);
        }
        var rooms = [];
        leaveRooms.forEach(function (leaveObj) {
            var room = leaveObj.room;
            rooms.push(room);
            if (!leaveObj.isBrandNewRoom) {
                // the intention behind syncLeftRooms is to add in rooms which were
                // *omitted* from the initial /sync. Rooms the user were joined to
                // but then left whilst the app is running will appear in this list
                // and we do not want to bother with them since they will have the
                // current state already (and may get dupe messages if we add
                // yet more timeline events!), so skip them.
                // NB: When we persist rooms to localStorage this will be more
                //     complicated...
                return;
            }
            leaveObj.timeline = leaveObj.timeline || {};
            var timelineEvents = self._mapSyncEventsFormat(leaveObj.timeline, room);
            var stateEvents = self._mapSyncEventsFormat(leaveObj.state, room);

            // set the back-pagination token. Do this *before* adding any
            // events so that clients can start back-paginating.
            room.getLiveTimeline().setPaginationToken(leaveObj.timeline.prev_batch, EventTimeline.BACKWARDS);

            self._processRoomEvents(room, stateEvents, timelineEvents);

            room.recalculate(client.credentials.userId);
            client.store.storeRoom(room);
            client.emit("Room", room);

            self._processEventsForNotifs(room, timelineEvents);
        });
        return rooms;
    });
};

/**
 * Peek into a room. This will result in the room in question being synced so it
 * is accessible via getRooms(). Live updates for the room will be provided.
 * @param {string} roomId The room ID to peek into.
 * @return {Promise} A promise which resolves once the room has been added to the
 * store.
 */
SyncApi.prototype.peek = function (roomId) {
    var self = this;
    var client = this.client;
    this._peekRoomId = roomId;
    return this.client.roomInitialSync(roomId, 20).then(function (response) {
        // make sure things are init'd
        response.messages = response.messages || {};
        response.messages.chunk = response.messages.chunk || [];
        response.state = response.state || [];

        var peekRoom = self.createRoom(roomId);

        // FIXME: Mostly duplicated from _processRoomEvents but not entirely
        // because "state" in this API is at the BEGINNING of the chunk
        var oldStateEvents = utils.map(utils.deepCopy(response.state), client.getEventMapper());
        var stateEvents = utils.map(response.state, client.getEventMapper());
        var messages = utils.map(response.messages.chunk, client.getEventMapper());

        // XXX: copypasted from /sync until we kill off this
        // minging v1 API stuff)
        // handle presence events (User objects)
        if (response.presence && utils.isArray(response.presence)) {
            response.presence.map(client.getEventMapper()).forEach(function (presenceEvent) {
                var user = client.store.getUser(presenceEvent.getContent().user_id);
                if (user) {
                    user.setPresenceEvent(presenceEvent);
                } else {
                    user = createNewUser(client, presenceEvent.getContent().user_id);
                    user.setPresenceEvent(presenceEvent);
                    client.store.storeUser(user);
                }
                client.emit("event", presenceEvent);
            });
        }

        // set the pagination token before adding the events in case people
        // fire off pagination requests in response to the Room.timeline
        // events.
        if (response.messages.start) {
            peekRoom.oldState.paginationToken = response.messages.start;
        }

        // set the state of the room to as it was after the timeline executes
        peekRoom.oldState.setStateEvents(oldStateEvents);
        peekRoom.currentState.setStateEvents(stateEvents);

        self._resolveInvites(peekRoom);
        peekRoom.recalculate(self.client.credentials.userId);

        // roll backwards to diverge old state. addEventsToTimeline
        // will overwrite the pagination token, so make sure it overwrites
        // it with the right thing.
        peekRoom.addEventsToTimeline(messages.reverse(), true, peekRoom.getLiveTimeline(), response.messages.start);

        client.store.storeRoom(peekRoom);
        client.emit("Room", peekRoom);

        self._peekPoll(peekRoom);
        return peekRoom;
    });
};

/**
 * Stop polling for updates in the peeked room. NOPs if there is no room being
 * peeked.
 */
SyncApi.prototype.stopPeeking = function () {
    this._peekRoomId = null;
};

/**
 * Do a peek room poll.
 * @param {Room} peekRoom
 * @param {string} token from= token
 */
SyncApi.prototype._peekPoll = function (peekRoom, token) {
    if (this._peekRoomId !== peekRoom.roomId) {
        debuglog("Stopped peeking in room %s", peekRoom.roomId);
        return;
    }

    var self = this;
    // FIXME: gut wrenching; hard-coded timeout values
    this.client._http.authedRequest(undefined, "GET", "/events", {
        room_id: peekRoom.roomId,
        timeout: 30 * 1000,
        from: token
    }, undefined, 50 * 1000).done(function (res) {
        if (self._peekRoomId !== peekRoom.roomId) {
            debuglog("Stopped peeking in room %s", peekRoom.roomId);
            return;
        }
        // We have a problem that we get presence both from /events and /sync
        // however, /sync only returns presence for users in rooms
        // you're actually joined to.
        // in order to be sure to get presence for all of the users in the
        // peeked room, we handle presence explicitly here. This may result
        // in duplicate presence events firing for some users, which is a
        // performance drain, but such is life.
        // XXX: copypasted from /sync until we can kill this minging v1 stuff.

        res.chunk.filter(function (e) {
            return e.type === "m.presence";
        }).map(self.client.getEventMapper()).forEach(function (presenceEvent) {
            var user = self.client.store.getUser(presenceEvent.getContent().user_id);
            if (user) {
                user.setPresenceEvent(presenceEvent);
            } else {
                user = createNewUser(self.client, presenceEvent.getContent().user_id);
                user.setPresenceEvent(presenceEvent);
                self.client.store.storeUser(user);
            }
            self.client.emit("event", presenceEvent);
        });

        // strip out events which aren't for the given room_id (e.g presence)
        var events = res.chunk.filter(function (e) {
            return e.room_id === peekRoom.roomId;
        }).map(self.client.getEventMapper());

        peekRoom.addLiveEvents(events);
        self._peekPoll(peekRoom, res.end);
    }, function (err) {
        console.error("[%s] Peek poll failed: %s", peekRoom.roomId, err);
        setTimeout(function () {
            self._peekPoll(peekRoom, token);
        }, 30 * 1000);
    });
};

/**
 * Returns the current state of this sync object
 * @see module:client~MatrixClient#event:"sync"
 * @return {?String}
 */
SyncApi.prototype.getSyncState = function () {
    return this._syncState;
};

SyncApi.prototype.recoverFromSyncStartupError = function () {
    var _ref = (0, _bluebird.coroutine)(_regenerator2.default.mark(function _callee(savedSyncPromise, err) {
        var keepaliveProm;
        return _regenerator2.default.wrap(function _callee$(_context) {
            while (1) {
                switch (_context.prev = _context.next) {
                    case 0:
                        _context.next = 2;
                        return (0, _bluebird.resolve)(savedSyncPromise);

                    case 2:
                        keepaliveProm = this._startKeepAlives();

                        this._updateSyncState("ERROR", { error: err });
                        _context.next = 6;
                        return (0, _bluebird.resolve)(keepaliveProm);

                    case 6:
                    case "end":
                        return _context.stop();
                }
            }
        }, _callee, this);
    }));

    return function (_x, _x2) {
        return _ref.apply(this, arguments);
    };
}();

/**
 * Main entry point
 */
SyncApi.prototype.sync = function () {

    // We need to do one-off checks before we can begin the /sync loop.
    // These are:
    //   1) We need to get push rules so we can check if events should bing as we get
    //      them from /sync.
    //   2) We need to get/create a filter which we can use for /sync.

    var getPushRules = function () {
        var _ref2 = (0, _bluebird.coroutine)(_regenerator2.default.mark(function _callee2() {
            var result;
            return _regenerator2.default.wrap(function _callee2$(_context2) {
                while (1) {
                    switch (_context2.prev = _context2.next) {
                        case 0:
                            _context2.prev = 0;
                            _context2.next = 3;
                            return (0, _bluebird.resolve)(client.getPushRules());

                        case 3:
                            result = _context2.sent;

                            debuglog("Got push rules");

                            client.pushRules = result;
                            _context2.next = 14;
                            break;

                        case 8:
                            _context2.prev = 8;
                            _context2.t0 = _context2["catch"](0);
                            _context2.next = 12;
                            return (0, _bluebird.resolve)(self.recoverFromSyncStartupError(savedSyncPromise, _context2.t0));

                        case 12:
                            getPushRules();
                            return _context2.abrupt("return");

                        case 14:
                            getFilter(); // Now get the filter and start syncing

                        case 15:
                        case "end":
                            return _context2.stop();
                    }
                }
            }, _callee2, this, [[0, 8]]);
        }));

        return function getPushRules() {
            return _ref2.apply(this, arguments);
        };
    }();

    var getFilter = function () {
        var _ref3 = (0, _bluebird.coroutine)(_regenerator2.default.mark(function _callee3() {
            var filter, filterId;
            return _regenerator2.default.wrap(function _callee3$(_context3) {
                while (1) {
                    switch (_context3.prev = _context3.next) {
                        case 0:
                            filter = void 0;

                            if (self.opts.filter) {
                                filter = self.opts.filter;
                            } else {
                                filter = new Filter(client.credentials.userId);
                                filter.setTimelineLimit(self.opts.initialSyncLimit);
                            }

                            filterId = void 0;
                            _context3.prev = 3;
                            _context3.next = 6;
                            return (0, _bluebird.resolve)(client.getOrCreateFilter(getFilterName(client.credentials.userId), filter));

                        case 6:
                            filterId = _context3.sent;
                            _context3.next = 15;
                            break;

                        case 9:
                            _context3.prev = 9;
                            _context3.t0 = _context3["catch"](3);
                            _context3.next = 13;
                            return (0, _bluebird.resolve)(self.recoverFromSyncStartupError(savedSyncPromise, _context3.t0));

                        case 13:
                            getFilter();
                            return _context3.abrupt("return");

                        case 15:
                            // reset the notifications timeline to prepare it to paginate from
                            // the current point in time.
                            // The right solution would be to tie /sync pagination tokens into
                            // /notifications API somehow.
                            client.resetNotifTimelineSet();

                            if (self._currentSyncRequest === null) {
                                // Send this first sync request here so we can then wait for the saved
                                // sync data to finish processing before we process the results of this one.
                                console.log("Sending first sync request...");
                                self._currentSyncRequest = self._doSyncRequest({ filterId: filterId }, savedSyncToken);
                            }

                            // Now wait for the saved sync to finish...
                            _context3.next = 19;
                            return (0, _bluebird.resolve)(savedSyncPromise);

                        case 19:
                            self._sync({ filterId: filterId });

                        case 20:
                        case "end":
                            return _context3.stop();
                    }
                }
            }, _callee3, this, [[3, 9]]);
        }));

        return function getFilter() {
            return _ref3.apply(this, arguments);
        };
    }();

    var client = this.client;
    var self = this;

    this._running = true;

    if (global.document) {
        this._onOnlineBound = this._onOnline.bind(this);
        global.document.addEventListener("online", this._onOnlineBound, false);
    }

    var savedSyncPromise = _bluebird2.default.resolve();
    var savedSyncToken = null;

    if (client.isGuest()) {
        // no push rules for guests, no access to POST filter for guests.
        self._sync({});
    } else {
        // Pull the saved sync token out first, before the worker starts sending
        // all the sync data which could take a while. This will let us send our
        // first incremental sync request before we've processed our saved data.
        savedSyncPromise = client.store.getSavedSyncToken().then(function (tok) {
            savedSyncToken = tok;
            return client.store.getSavedSync();
        }).then(function (savedSync) {
            if (savedSync) {
                return self._syncFromCache(savedSync);
            }
        });
        // Now start the first incremental sync request: this can also
        // take a while so if we set it going now, we can wait for it
        // to finish while we process our saved sync data.
        getPushRules();
    }
};

/**
 * Stops the sync object from syncing.
 */
SyncApi.prototype.stop = function () {
    debuglog("SyncApi.stop");
    if (global.document) {
        global.document.removeEventListener("online", this._onOnlineBound, false);
        this._onOnlineBound = undefined;
    }
    this._running = false;
    if (this._currentSyncRequest) {
        this._currentSyncRequest.abort();
    }
    if (this._keepAliveTimer) {
        clearTimeout(this._keepAliveTimer);
        this._keepAliveTimer = null;
    }
};

/**
 * Retry a backed off syncing request immediately. This should only be used when
 * the user <b>explicitly</b> attempts to retry their lost connection.
 * @return {boolean} True if this resulted in a request being retried.
 */
SyncApi.prototype.retryImmediately = function () {
    if (!this._connectionReturnedDefer) {
        return false;
    }
    this._startKeepAlives(0);
    return true;
};
/**
 * Process a single set of cached sync data.
 * @param {Object} savedSync a saved sync that was persisted by a store. This
 * should have been acquired via client.store.getSavedSync().
 */
SyncApi.prototype._syncFromCache = function () {
    var _ref4 = (0, _bluebird.coroutine)(_regenerator2.default.mark(function _callee4(savedSync) {
        var nextSyncToken, syncEventData, data;
        return _regenerator2.default.wrap(function _callee4$(_context4) {
            while (1) {
                switch (_context4.prev = _context4.next) {
                    case 0:
                        debuglog("sync(): not doing HTTP hit, instead returning stored /sync data");

                        nextSyncToken = savedSync.nextBatch;

                        // Set sync token for future incremental syncing

                        this.client.store.setSyncToken(nextSyncToken);

                        // No previous sync, set old token to null
                        syncEventData = {
                            oldSyncToken: null,
                            nextSyncToken: nextSyncToken,
                            catchingUp: false
                        };
                        data = {
                            next_batch: nextSyncToken,
                            rooms: savedSync.roomsData,
                            groups: savedSync.groupsData,
                            account_data: {
                                events: savedSync.accountData
                            }
                        };
                        _context4.prev = 5;
                        _context4.next = 8;
                        return (0, _bluebird.resolve)(this._processSyncResponse(syncEventData, data));

                    case 8:
                        _context4.next = 13;
                        break;

                    case 10:
                        _context4.prev = 10;
                        _context4.t0 = _context4["catch"](5);

                        console.error("Error processing cached sync", _context4.t0.stack || _context4.t0);

                    case 13:

                        this._updateSyncState("PREPARED", syncEventData);

                    case 14:
                    case "end":
                        return _context4.stop();
                }
            }
        }, _callee4, this, [[5, 10]]);
    }));

    return function (_x3) {
        return _ref4.apply(this, arguments);
    };
}();

/**
 * Invoke me to do /sync calls
 * @param {Object} syncOptions
 * @param {string} syncOptions.filterId
 * @param {boolean} syncOptions.hasSyncedBefore
 */
SyncApi.prototype._sync = function () {
    var _ref5 = (0, _bluebird.coroutine)(_regenerator2.default.mark(function _callee5(syncOptions) {
        var client, syncToken, data, syncEventData;
        return _regenerator2.default.wrap(function _callee5$(_context5) {
            while (1) {
                switch (_context5.prev = _context5.next) {
                    case 0:
                        client = this.client;

                        if (this._running) {
                            _context5.next = 6;
                            break;
                        }

                        debuglog("Sync no longer running: exiting.");
                        if (this._connectionReturnedDefer) {
                            this._connectionReturnedDefer.reject();
                            this._connectionReturnedDefer = null;
                        }
                        this._updateSyncState("STOPPED");
                        return _context5.abrupt("return");

                    case 6:
                        syncToken = client.store.getSyncToken();
                        data = void 0;
                        _context5.prev = 8;

                        //debuglog('Starting sync since=' + syncToken);
                        if (this._currentSyncRequest === null) {
                            this._currentSyncRequest = this._doSyncRequest(syncOptions, syncToken);
                        }
                        _context5.next = 12;
                        return (0, _bluebird.resolve)(this._currentSyncRequest);

                    case 12:
                        data = _context5.sent;
                        _context5.next = 19;
                        break;

                    case 15:
                        _context5.prev = 15;
                        _context5.t0 = _context5["catch"](8);

                        this._onSyncError(_context5.t0, syncOptions);
                        return _context5.abrupt("return");

                    case 19:
                        _context5.prev = 19;

                        this._currentSyncRequest = null;
                        return _context5.finish(19);

                    case 22:

                        //debuglog('Completed sync, next_batch=' + data.next_batch);

                        // set the sync token NOW *before* processing the events. We do this so
                        // if something barfs on an event we can skip it rather than constantly
                        // polling with the same token.
                        client.store.setSyncToken(data.next_batch);

                        // Reset after a successful sync
                        this._failedSyncCount = 0;

                        _context5.next = 26;
                        return (0, _bluebird.resolve)(client.store.setSyncData(data));

                    case 26:
                        syncEventData = {
                            oldSyncToken: syncToken,
                            nextSyncToken: data.next_batch,
                            catchingUp: this._catchingUp
                        };

                        if (!this.opts.crypto) {
                            _context5.next = 30;
                            break;
                        }

                        _context5.next = 30;
                        return (0, _bluebird.resolve)(this.opts.crypto.onSyncWillProcess(syncEventData));

                    case 30:
                        _context5.prev = 30;
                        _context5.next = 33;
                        return (0, _bluebird.resolve)(this._processSyncResponse(syncEventData, data));

                    case 33:
                        _context5.next = 38;
                        break;

                    case 35:
                        _context5.prev = 35;
                        _context5.t1 = _context5["catch"](30);

                        // log the exception with stack if we have it, else fall back
                        // to the plain description
                        console.error("Caught /sync error", _context5.t1.stack || _context5.t1);

                    case 38:

                        // update this as it may have changed
                        syncEventData.catchingUp = this._catchingUp;

                        // emit synced events
                        if (!syncOptions.hasSyncedBefore) {
                            this._updateSyncState("PREPARED", syncEventData);
                            syncOptions.hasSyncedBefore = true;
                        }

                        // tell the crypto module to do its processing. It may block (to do a
                        // /keys/changes request).

                        if (!this.opts.crypto) {
                            _context5.next = 43;
                            break;
                        }

                        _context5.next = 43;
                        return (0, _bluebird.resolve)(this.opts.crypto.onSyncCompleted(syncEventData));

                    case 43:

                        // keep emitting SYNCING -> SYNCING for clients who want to do bulk updates
                        this._updateSyncState("SYNCING", syncEventData);

                        if (!client.store.wantsSave()) {
                            _context5.next = 49;
                            break;
                        }

                        if (!this.opts.crypto) {
                            _context5.next = 48;
                            break;
                        }

                        _context5.next = 48;
                        return (0, _bluebird.resolve)(this.opts.crypto.saveDeviceList(0));

                    case 48:

                        // tell databases that everything is now in a consistent state and can be saved.
                        client.store.save();

                    case 49:

                        // Begin next sync
                        this._sync(syncOptions);

                    case 50:
                    case "end":
                        return _context5.stop();
                }
            }
        }, _callee5, this, [[8, 15, 19, 22], [30, 35]]);
    }));

    return function (_x4) {
        return _ref5.apply(this, arguments);
    };
}();

SyncApi.prototype._doSyncRequest = function (syncOptions, syncToken) {
    var qps = this._getSyncParams(syncOptions, syncToken);
    return this.client._http.authedRequest(undefined, "GET", "/sync", qps, undefined, qps.timeout + BUFFER_PERIOD_MS);
};

SyncApi.prototype._getSyncParams = function (syncOptions, syncToken) {
    var pollTimeout = this.opts.pollTimeout;

    if (this.getSyncState() !== 'SYNCING' || this._catchingUp) {
        // unless we are happily syncing already, we want the server to return
        // as quickly as possible, even if there are no events queued. This
        // serves two purposes:
        //
        // * When the connection dies, we want to know asap when it comes back,
        //   so that we can hide the error from the user. (We don't want to
        //   have to wait for an event or a timeout).
        //
        // * We want to know if the server has any to_device messages queued up
        //   for us. We do that by calling it with a zero timeout until it
        //   doesn't give us any more to_device messages.
        this._catchingUp = true;
        pollTimeout = 0;
    }

    var filterId = syncOptions.filterId;
    if (this.client.isGuest() && !filterId) {
        filterId = this._getGuestFilter();
    }

    var qps = {
        filter: filterId,
        timeout: pollTimeout
    };

    if (this.opts.disablePresence) {
        qps.set_presence = "offline";
    }

    if (syncToken) {
        qps.since = syncToken;
    } else {
        // use a cachebuster for initialsyncs, to make sure that
        // we don't get a stale sync
        // (https://github.com/vector-im/vector-web/issues/1354)
        qps._cacheBuster = Date.now();
    }

    if (this.getSyncState() == 'ERROR' || this.getSyncState() == 'RECONNECTING') {
        // we think the connection is dead. If it comes back up, we won't know
        // about it till /sync returns. If the timeout= is high, this could
        // be a long time. Set it to 0 when doing retries so we don't have to wait
        // for an event or a timeout before emiting the SYNCING event.
        qps.timeout = 0;
    }

    return qps;
};

SyncApi.prototype._onSyncError = function (err, syncOptions) {
    var _this = this;

    if (!this._running) {
        debuglog("Sync no longer running: exiting");
        if (this._connectionReturnedDefer) {
            this._connectionReturnedDefer.reject();
            this._connectionReturnedDefer = null;
        }
        this._updateSyncState("STOPPED");
        return;
    }

    console.error("/sync error %s", err);
    console.error(err);

    this._failedSyncCount++;
    console.log('Number of consecutive failed sync requests:', this._failedSyncCount);

    debuglog("Starting keep-alive");
    // Note that we do *not* mark the sync connection as
    // lost yet: we only do this if a keepalive poke
    // fails, since long lived HTTP connections will
    // go away sometimes and we shouldn't treat this as
    // erroneous. We set the state to 'reconnecting'
    // instead, so that clients can onserve this state
    // if they wish.
    this._startKeepAlives().then(function () {
        _this._sync(syncOptions);
    });

    this._currentSyncRequest = null;
    // Transition from RECONNECTING to ERROR after a given number of failed syncs
    this._updateSyncState(this._failedSyncCount >= FAILED_SYNC_ERROR_THRESHOLD ? "ERROR" : "RECONNECTING");
};

/**
 * Process data returned from a sync response and propagate it
 * into the model objects
 *
 * @param {Object} syncEventData Object containing sync tokens associated with this sync
 * @param {Object} data The response from /sync
 */
SyncApi.prototype._processSyncResponse = function () {
    var _ref6 = (0, _bluebird.coroutine)(_regenerator2.default.mark(function _callee8(syncEventData, data) {
        var client, self, events, inviteRooms, joinRooms, leaveRooms, currentCount;
        return _regenerator2.default.wrap(function _callee8$(_context8) {
            while (1) {
                switch (_context8.prev = _context8.next) {
                    case 0:
                        client = this.client;
                        self = this;

                        // data looks like:
                        // {
                        //    next_batch: $token,
                        //    presence: { events: [] },
                        //    account_data: { events: [] },
                        //    device_lists: { changed: ["@user:server", ... ]},
                        //    to_device: { events: [] },
                        //    device_one_time_keys_count: { signed_curve25519: 42 },
                        //    rooms: {
                        //      invite: {
                        //        $roomid: {
                        //          invite_state: { events: [] }
                        //        }
                        //      },
                        //      join: {
                        //        $roomid: {
                        //          state: { events: [] },
                        //          timeline: { events: [], prev_batch: $token, limited: true },
                        //          ephemeral: { events: [] },
                        //          account_data: { events: [] },
                        //          unread_notifications: {
                        //              highlight_count: 0,
                        //              notification_count: 0,
                        //          }
                        //        }
                        //      },
                        //      leave: {
                        //        $roomid: {
                        //          state: { events: [] },
                        //          timeline: { events: [], prev_batch: $token }
                        //        }
                        //      }
                        //    },
                        //    groups: {
                        //        invite: {
                        //            $groupId: {
                        //                inviter: $inviter,
                        //                profile: {
                        //                    avatar_url: $avatarUrl,
                        //                    name: $groupName,
                        //                },
                        //            },
                        //        },
                        //        join: {},
                        //        leave: {},
                        //    },
                        // }

                        // TODO-arch:
                        // - Each event we pass through needs to be emitted via 'event', can we
                        //   do this in one place?
                        // - The isBrandNewRoom boilerplate is boilerplatey.

                        // handle presence events (User objects)

                        if (data.presence && utils.isArray(data.presence.events)) {
                            data.presence.events.map(client.getEventMapper()).forEach(function (presenceEvent) {
                                var user = client.store.getUser(presenceEvent.getSender());
                                if (user) {
                                    user.setPresenceEvent(presenceEvent);
                                } else {
                                    user = createNewUser(client, presenceEvent.getSender());
                                    user.setPresenceEvent(presenceEvent);
                                    client.store.storeUser(user);
                                }
                                client.emit("event", presenceEvent);
                            });
                        }

                        // handle non-room account_data
                        if (data.account_data && utils.isArray(data.account_data.events)) {
                            events = data.account_data.events.map(client.getEventMapper());

                            client.store.storeAccountDataEvents(events);
                            events.forEach(function (accountDataEvent) {
                                // Honour push rules that come down the sync stream but also
                                // honour push rules that were previously cached. Base rules
                                // will be updated when we recieve push rules via getPushRules
                                // (see SyncApi.prototype.sync) before syncing over the network.
                                if (accountDataEvent.getType() == 'm.push_rules') {
                                    client.pushRules = accountDataEvent.getContent();
                                }
                                client.emit("accountData", accountDataEvent);
                                return accountDataEvent;
                            });
                        }

                        // handle to-device events
                        if (data.to_device && utils.isArray(data.to_device.events) && data.to_device.events.length > 0) {
                            data.to_device.events.map(client.getEventMapper()).forEach(function (toDeviceEvent) {
                                var content = toDeviceEvent.getContent();
                                if (toDeviceEvent.getType() == "m.room.message" && content.msgtype == "m.bad.encrypted") {
                                    // the mapper already logged a warning.
                                    console.log('Ignoring undecryptable to-device event from ' + toDeviceEvent.getSender());
                                    return;
                                }

                                client.emit("toDeviceEvent", toDeviceEvent);
                            });
                        } else {
                            // no more to-device events: we can stop polling with a short timeout.
                            this._catchingUp = false;
                        }

                        if (data.groups) {
                            if (data.groups.invite) {
                                this._processGroupSyncEntry(data.groups.invite, 'invite');
                            }

                            if (data.groups.join) {
                                this._processGroupSyncEntry(data.groups.join, 'join');
                            }

                            if (data.groups.leave) {
                                this._processGroupSyncEntry(data.groups.leave, 'leave');
                            }
                        }

                        // the returned json structure is a bit crap, so make it into a
                        // nicer form (array) after applying sanity to make sure we don't fail
                        // on missing keys (on the off chance)
                        inviteRooms = [];
                        joinRooms = [];
                        leaveRooms = [];


                        if (data.rooms) {
                            if (data.rooms.invite) {
                                inviteRooms = this._mapSyncResponseToRoomArray(data.rooms.invite);
                            }
                            if (data.rooms.join) {
                                joinRooms = this._mapSyncResponseToRoomArray(data.rooms.join);
                            }
                            if (data.rooms.leave) {
                                leaveRooms = this._mapSyncResponseToRoomArray(data.rooms.leave);
                            }
                        }

                        this._notifEvents = [];

                        // Handle invites
                        inviteRooms.forEach(function (inviteObj) {
                            var room = inviteObj.room;
                            var stateEvents = self._mapSyncEventsFormat(inviteObj.invite_state, room);
                            self._processRoomEvents(room, stateEvents);
                            if (inviteObj.isBrandNewRoom) {
                                room.recalculate(client.credentials.userId);
                                client.store.storeRoom(room);
                                client.emit("Room", room);
                            }
                            stateEvents.forEach(function (e) {
                                client.emit("event", e);
                            });
                        });

                        // Handle joins
                        _context8.next = 14;
                        return (0, _bluebird.resolve)(_bluebird2.default.mapSeries(joinRooms, function () {
                            var _ref7 = (0, _bluebird.coroutine)(_regenerator2.default.mark(function _callee7(joinObj) {
                                var processRoomEvent = function () {
                                    var _ref8 = (0, _bluebird.coroutine)(_regenerator2.default.mark(function _callee6(e) {
                                        return _regenerator2.default.wrap(function _callee6$(_context6) {
                                            while (1) {
                                                switch (_context6.prev = _context6.next) {
                                                    case 0:
                                                        client.emit("event", e);

                                                        if (!(e.isState() && e.getType() == "m.room.encryption" && self.opts.crypto)) {
                                                            _context6.next = 4;
                                                            break;
                                                        }

                                                        _context6.next = 4;
                                                        return (0, _bluebird.resolve)(self.opts.crypto.onCryptoEvent(e));

                                                    case 4:
                                                    case "end":
                                                        return _context6.stop();
                                                }
                                            }
                                        }, _callee6, this);
                                    }));

                                    return function processRoomEvent(_x8) {
                                        return _ref8.apply(this, arguments);
                                    };
                                }();

                                var room, stateEvents, timelineEvents, ephemeralEvents, accountDataEvents, limited, i, eventId;
                                return _regenerator2.default.wrap(function _callee7$(_context7) {
                                    while (1) {
                                        switch (_context7.prev = _context7.next) {
                                            case 0:
                                                room = joinObj.room;
                                                stateEvents = self._mapSyncEventsFormat(joinObj.state, room);
                                                timelineEvents = self._mapSyncEventsFormat(joinObj.timeline, room);
                                                ephemeralEvents = self._mapSyncEventsFormat(joinObj.ephemeral);
                                                accountDataEvents = self._mapSyncEventsFormat(joinObj.account_data);

                                                // we do this first so it's correct when any of the events fire

                                                if (joinObj.unread_notifications) {
                                                    room.setUnreadNotificationCount('total', joinObj.unread_notifications.notification_count);
                                                    room.setUnreadNotificationCount('highlight', joinObj.unread_notifications.highlight_count);
                                                }

                                                joinObj.timeline = joinObj.timeline || {};

                                                if (!joinObj.isBrandNewRoom) {
                                                    _context7.next = 11;
                                                    break;
                                                }

                                                // set the back-pagination token. Do this *before* adding any
                                                // events so that clients can start back-paginating.
                                                room.getLiveTimeline().setPaginationToken(joinObj.timeline.prev_batch, EventTimeline.BACKWARDS);
                                                _context7.next = 25;
                                                break;

                                            case 11:
                                                if (!joinObj.timeline.limited) {
                                                    _context7.next = 25;
                                                    break;
                                                }

                                                limited = true;

                                                // we've got a limited sync, so we *probably* have a gap in the
                                                // timeline, so should reset. But we might have been peeking or
                                                // paginating and already have some of the events, in which
                                                // case we just want to append any subsequent events to the end
                                                // of the existing timeline.
                                                //
                                                // This is particularly important in the case that we already have
                                                // *all* of the events in the timeline - in that case, if we reset
                                                // the timeline, we'll end up with an entirely empty timeline,
                                                // which we'll try to paginate but not get any new events (which
                                                // will stop us linking the empty timeline into the chain).
                                                //

                                                i = timelineEvents.length - 1;

                                            case 14:
                                                if (!(i >= 0)) {
                                                    _context7.next = 24;
                                                    break;
                                                }

                                                eventId = timelineEvents[i].getId();

                                                if (!room.getTimelineForEvent(eventId)) {
                                                    _context7.next = 21;
                                                    break;
                                                }

                                                debuglog("Already have event " + eventId + " in limited " + "sync - not resetting");
                                                limited = false;

                                                // we might still be missing some of the events before i;
                                                // we don't want to be adding them to the end of the
                                                // timeline because that would put them out of order.
                                                timelineEvents.splice(0, i);

                                                // XXX: there's a problem here if the skipped part of the
                                                // timeline modifies the state set in stateEvents, because
                                                // we'll end up using the state from stateEvents rather
                                                // than the later state from timelineEvents. We probably
                                                // need to wind stateEvents forward over the events we're
                                                // skipping.

                                                return _context7.abrupt("break", 24);

                                            case 21:
                                                i--;
                                                _context7.next = 14;
                                                break;

                                            case 24:

                                                if (limited) {
                                                    self._deregisterStateListeners(room);
                                                    room.resetLiveTimeline(joinObj.timeline.prev_batch, self.opts.canResetEntireTimeline(room.roomId) ? null : syncEventData.oldSyncToken);

                                                    // We have to assume any gap in any timeline is
                                                    // reason to stop incrementally tracking notifications and
                                                    // reset the timeline.
                                                    client.resetNotifTimelineSet();

                                                    self._registerStateListeners(room);
                                                }

                                            case 25:

                                                self._processRoomEvents(room, stateEvents, timelineEvents);

                                                // XXX: should we be adding ephemeralEvents to the timeline?
                                                // It feels like that for symmetry with room.addAccountData()
                                                // there should be a room.addEphemeralEvents() or similar.
                                                room.addLiveEvents(ephemeralEvents);

                                                // we deliberately don't add accountData to the timeline
                                                room.addAccountData(accountDataEvents);

                                                room.recalculate(client.credentials.userId);
                                                if (joinObj.isBrandNewRoom) {
                                                    client.store.storeRoom(room);
                                                    client.emit("Room", room);
                                                }

                                                self._processEventsForNotifs(room, timelineEvents);

                                                _context7.next = 33;
                                                return (0, _bluebird.resolve)(_bluebird2.default.mapSeries(stateEvents, processRoomEvent));

                                            case 33:
                                                _context7.next = 35;
                                                return (0, _bluebird.resolve)(_bluebird2.default.mapSeries(timelineEvents, processRoomEvent));

                                            case 35:
                                                ephemeralEvents.forEach(function (e) {
                                                    client.emit("event", e);
                                                });
                                                accountDataEvents.forEach(function (e) {
                                                    client.emit("event", e);
                                                });

                                            case 37:
                                            case "end":
                                                return _context7.stop();
                                        }
                                    }
                                }, _callee7, this);
                            }));

                            return function (_x7) {
                                return _ref7.apply(this, arguments);
                            };
                        }()));

                    case 14:

                        // Handle leaves (e.g. kicked rooms)
                        leaveRooms.forEach(function (leaveObj) {
                            var room = leaveObj.room;
                            var stateEvents = self._mapSyncEventsFormat(leaveObj.state, room);
                            var timelineEvents = self._mapSyncEventsFormat(leaveObj.timeline, room);
                            var accountDataEvents = self._mapSyncEventsFormat(leaveObj.account_data);

                            self._processRoomEvents(room, stateEvents, timelineEvents);
                            room.addAccountData(accountDataEvents);

                            room.recalculate(client.credentials.userId);
                            if (leaveObj.isBrandNewRoom) {
                                client.store.storeRoom(room);
                                client.emit("Room", room);
                            }

                            self._processEventsForNotifs(room, timelineEvents);

                            stateEvents.forEach(function (e) {
                                client.emit("event", e);
                            });
                            timelineEvents.forEach(function (e) {
                                client.emit("event", e);
                            });
                            accountDataEvents.forEach(function (e) {
                                client.emit("event", e);
                            });
                        });

                        // update the notification timeline, if appropriate.
                        // we only do this for live events, as otherwise we can't order them sanely
                        // in the timeline relative to ones paginated in by /notifications.
                        // XXX: we could fix this by making EventTimeline support chronological
                        // ordering... but it doesn't, right now.
                        if (syncEventData.oldSyncToken && this._notifEvents.length) {
                            this._notifEvents.sort(function (a, b) {
                                return a.getTs() - b.getTs();
                            });
                            this._notifEvents.forEach(function (event) {
                                client.getNotifTimelineSet().addLiveEvent(event);
                            });
                        }

                        // Handle device list updates

                        if (!data.device_lists) {
                            _context8.next = 22;
                            break;
                        }

                        if (!this.opts.crypto) {
                            _context8.next = 22;
                            break;
                        }

                        _context8.next = 20;
                        return (0, _bluebird.resolve)(this.opts.crypto.handleDeviceListChanges(syncEventData, data.device_lists));

                    case 20:
                        _context8.next = 22;
                        break;

                    case 22:

                        // Handle one_time_keys_count
                        if (this.opts.crypto && data.device_one_time_keys_count) {
                            currentCount = data.device_one_time_keys_count.signed_curve25519 || 0;

                            this.opts.crypto.updateOneTimeKeyCount(currentCount);
                        }

                    case 23:
                    case "end":
                        return _context8.stop();
                }
            }
        }, _callee8, this);
    }));

    return function (_x5, _x6) {
        return _ref6.apply(this, arguments);
    };
}();

/**
 * Starts polling the connectivity check endpoint
 * @param {number} delay How long to delay until the first poll.
 *        defaults to a short, randomised interval (to prevent
 *        tightlooping if /versions succeeds but /sync etc. fail).
 * @return {promise} which resolves once the connection returns
 */
SyncApi.prototype._startKeepAlives = function (delay) {
    if (delay === undefined) {
        delay = 2000 + Math.floor(Math.random() * 5000);
    }

    if (this._keepAliveTimer !== null) {
        clearTimeout(this._keepAliveTimer);
    }
    var self = this;
    if (delay > 0) {
        self._keepAliveTimer = setTimeout(self._pokeKeepAlive.bind(self), delay);
    } else {
        self._pokeKeepAlive();
    }
    if (!this._connectionReturnedDefer) {
        this._connectionReturnedDefer = _bluebird2.default.defer();
    }
    return this._connectionReturnedDefer.promise;
};

/**
 * Make a dummy call to /_matrix/client/versions, to see if the HS is
 * reachable.
 *
 * On failure, schedules a call back to itself. On success, resolves
 * this._connectionReturnedDefer.
 */
SyncApi.prototype._pokeKeepAlive = function () {
    var self = this;
    function success() {
        clearTimeout(self._keepAliveTimer);
        if (self._connectionReturnedDefer) {
            self._connectionReturnedDefer.resolve();
            self._connectionReturnedDefer = null;
        }
    }

    this.client._http.request(undefined, // callback
    "GET", "/_matrix/client/versions", undefined, // queryParams
    undefined, // data
    {
        prefix: '',
        localTimeoutMs: 15 * 1000
    }).done(function () {
        success();
    }, function (err) {
        if (err.httpStatus == 400) {
            // treat this as a success because the server probably just doesn't
            // support /versions: point is, we're getting a response.
            // We wait a short time though, just in case somehow the server
            // is in a mode where it 400s /versions responses and sync etc.
            // responses fail, this will mean we don't hammer in a loop.
            self._keepAliveTimer = setTimeout(success, 2000);
        } else {
            self._keepAliveTimer = setTimeout(self._pokeKeepAlive.bind(self), 5000 + Math.floor(Math.random() * 5000));
            // A keepalive has failed, so we emit the
            // error state (whether or not this is the
            // first failure).
            // Note we do this after setting the timer:
            // this lets the unit tests advance the mock
            // clock when the get the error.
            self._updateSyncState("ERROR", { error: err });
        }
    });
};

/**
 * @param {Object} groupsSection Groups section object, eg. response.groups.invite
 * @param {string} sectionName Which section this is ('invite', 'join' or 'leave')
 */
SyncApi.prototype._processGroupSyncEntry = function (groupsSection, sectionName) {
    // Processes entries from 'groups' section of the sync stream
    var _iteratorNormalCompletion = true;
    var _didIteratorError = false;
    var _iteratorError = undefined;

    try {
        for (var _iterator = (0, _getIterator3.default)((0, _keys2.default)(groupsSection)), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
            var groupId = _step.value;

            var groupInfo = groupsSection[groupId];
            var group = this.client.store.getGroup(groupId);
            var isBrandNew = group === null;
            if (group === null) {
                group = this.createGroup(groupId);
            }
            if (groupInfo.profile) {
                group.setProfile(groupInfo.profile.name, groupInfo.profile.avatar_url);
            }
            if (groupInfo.inviter) {
                group.setInviter({ userId: groupInfo.inviter });
            }
            group.setMyMembership(sectionName);
            if (isBrandNew) {
                // Now we've filled in all the fields, emit the Group event
                this.client.emit("Group", group);
            }
        }
    } catch (err) {
        _didIteratorError = true;
        _iteratorError = err;
    } finally {
        try {
            if (!_iteratorNormalCompletion && _iterator.return) {
                _iterator.return();
            }
        } finally {
            if (_didIteratorError) {
                throw _iteratorError;
            }
        }
    }
};

/**
 * @param {Object} obj
 * @return {Object[]}
 */
SyncApi.prototype._mapSyncResponseToRoomArray = function (obj) {
    // Maps { roomid: {stuff}, roomid: {stuff} }
    // to
    // [{stuff+Room+isBrandNewRoom}, {stuff+Room+isBrandNewRoom}]
    var client = this.client;
    var self = this;
    return utils.keys(obj).map(function (roomId) {
        var arrObj = obj[roomId];
        var room = client.store.getRoom(roomId);
        var isBrandNewRoom = false;
        if (!room) {
            room = self.createRoom(roomId);
            isBrandNewRoom = true;
        }
        arrObj.room = room;
        arrObj.isBrandNewRoom = isBrandNewRoom;
        return arrObj;
    });
};

/**
 * @param {Object} obj
 * @param {Room} room
 * @return {MatrixEvent[]}
 */
SyncApi.prototype._mapSyncEventsFormat = function (obj, room) {
    if (!obj || !utils.isArray(obj.events)) {
        return [];
    }
    var mapper = this.client.getEventMapper();
    return obj.events.map(function (e) {
        if (room) {
            e.room_id = room.roomId;
        }
        return mapper(e);
    });
};

/**
 * @param {Room} room
 */
SyncApi.prototype._resolveInvites = function (room) {
    if (!room || !this.opts.resolveInvitesToProfiles) {
        return;
    }
    var client = this.client;
    // For each invited room member we want to give them a displayname/avatar url
    // if they have one (the m.room.member invites don't contain this).
    room.getMembersWithMembership("invite").forEach(function (member) {
        if (member._requestedProfileInfo) {
            return;
        }
        member._requestedProfileInfo = true;
        // try to get a cached copy first.
        var user = client.getUser(member.userId);
        var promise = void 0;
        if (user) {
            promise = _bluebird2.default.resolve({
                avatar_url: user.avatarUrl,
                displayname: user.displayName
            });
        } else {
            promise = client.getProfileInfo(member.userId);
        }
        promise.done(function (info) {
            // slightly naughty by doctoring the invite event but this means all
            // the code paths remain the same between invite/join display name stuff
            // which is a worthy trade-off for some minor pollution.
            var inviteEvent = member.events.member;
            if (inviteEvent.getContent().membership !== "invite") {
                // between resolving and now they have since joined, so don't clobber
                return;
            }
            inviteEvent.getContent().avatar_url = info.avatar_url;
            inviteEvent.getContent().displayname = info.displayname;
            // fire listeners
            member.setMembershipEvent(inviteEvent, room.currentState);
        }, function (err) {
            // OH WELL.
        });
    });
};

/**
 * @param {Room} room
 * @param {MatrixEvent[]} stateEventList A list of state events. This is the state
 * at the *START* of the timeline list if it is supplied.
 * @param {MatrixEvent[]} [timelineEventList] A list of timeline events. Lower index
 * is earlier in time. Higher index is later.
 */
SyncApi.prototype._processRoomEvents = function (room, stateEventList, timelineEventList) {
    // If there are no events in the timeline yet, initialise it with
    // the given state events
    var liveTimeline = room.getLiveTimeline();
    var timelineWasEmpty = liveTimeline.getEvents().length == 0;
    if (timelineWasEmpty) {
        // Passing these events into initialiseState will freeze them, so we need
        // to compute and cache the push actions for them now, otherwise sync dies
        // with an attempt to assign to read only property.
        // XXX: This is pretty horrible and is assuming all sorts of behaviour from
        // these functions that it shouldn't be. We should probably either store the
        // push actions cache elsewhere so we can freeze MatrixEvents, or otherwise
        // find some solution where MatrixEvents are immutable but allow for a cache
        // field.
        var _iteratorNormalCompletion2 = true;
        var _didIteratorError2 = false;
        var _iteratorError2 = undefined;

        try {
            for (var _iterator2 = (0, _getIterator3.default)(stateEventList), _step2; !(_iteratorNormalCompletion2 = (_step2 = _iterator2.next()).done); _iteratorNormalCompletion2 = true) {
                var ev = _step2.value;

                this.client.getPushActionsForEvent(ev);
            }
        } catch (err) {
            _didIteratorError2 = true;
            _iteratorError2 = err;
        } finally {
            try {
                if (!_iteratorNormalCompletion2 && _iterator2.return) {
                    _iterator2.return();
                }
            } finally {
                if (_didIteratorError2) {
                    throw _iteratorError2;
                }
            }
        }

        liveTimeline.initialiseState(stateEventList);
    }

    this._resolveInvites(room);

    // recalculate the room name at this point as adding events to the timeline
    // may make notifications appear which should have the right name.
    // XXX: This looks suspect: we'll end up recalculating the room once here
    // and then again after adding events (_processSyncResponse calls it after
    // calling us) even if no state events were added. It also means that if
    // one of the room events in timelineEventList is something that needs
    // a recalculation (like m.room.name) we won't recalculate until we've
    // finished adding all the events, which will cause the notification to have
    // the old room name rather than the new one.
    room.recalculate(this.client.credentials.userId);

    // If the timeline wasn't empty, we process the state events here: they're
    // defined as updates to the state before the start of the timeline, so this
    // starts to roll the state forward.
    // XXX: That's what we *should* do, but this can happen if we were previously
    // peeking in a room, in which case we obviously do *not* want to add the
    // state events here onto the end of the timeline. Historically, the js-sdk
    // has just set these new state events on the old and new state. This seems
    // very wrong because there could be events in the timeline that diverge the
    // state, in which case this is going to leave things out of sync. However,
    // for now I think it;s best to behave the same as the code has done previously.
    if (!timelineWasEmpty) {
        // XXX: As above, don't do this...
        //room.addLiveEvents(stateEventList || []);
        // Do this instead...
        room.oldState.setStateEvents(stateEventList || []);
        room.currentState.setStateEvents(stateEventList || []);
    }
    // execute the timeline events. This will continue to diverge the current state
    // if the timeline has any state events in it.
    // This also needs to be done before running push rules on the events as they need
    // to be decorated with sender etc.
    room.addLiveEvents(timelineEventList || []);
};

/**
 * Takes a list of timelineEvents and adds and adds to _notifEvents
 * as appropriate.
 * This must be called after the room the events belong to has been stored.
 *
 * @param {Room} room
 * @param {MatrixEvent[]} [timelineEventList] A list of timeline events. Lower index
 * is earlier in time. Higher index is later.
 */
SyncApi.prototype._processEventsForNotifs = function (room, timelineEventList) {
    // gather our notifications into this._notifEvents
    if (this.client.getNotifTimelineSet()) {
        for (var i = 0; i < timelineEventList.length; i++) {
            var pushActions = this.client.getPushActionsForEvent(timelineEventList[i]);
            if (pushActions && pushActions.notify && pushActions.tweaks && pushActions.tweaks.highlight) {
                this._notifEvents.push(timelineEventList[i]);
            }
        }
    }
};

/**
 * @return {string}
 */
SyncApi.prototype._getGuestFilter = function () {
    var guestRooms = this.client._guestRooms; // FIXME: horrible gut-wrenching
    if (!guestRooms) {
        return "{}";
    }
    // we just need to specify the filter inline if we're a guest because guests
    // can't create filters.
    return (0, _stringify2.default)({
        room: {
            timeline: {
                limit: 20
            }
        }
    });
};

/**
 * Sets the sync state and emits an event to say so
 * @param {String} newState The new state string
 * @param {Object} data Object of additional data to emit in the event
 */
SyncApi.prototype._updateSyncState = function (newState, data) {
    var old = this._syncState;
    this._syncState = newState;
    this.client.emit("sync", this._syncState, old, data);
};

/**
 * Event handler for the 'online' event
 * This event is generally unreliable and precise behaviour
 * varies between browsers, so we poll for connectivity too,
 * but this might help us reconnect a little faster.
 */
SyncApi.prototype._onOnline = function () {
    debuglog("Browser thinks we are back online");
    this._startKeepAlives(0);
};

function createNewUser(client, userId) {
    var user = new User(userId);
    client.reEmitter.reEmit(user, ["User.avatarUrl", "User.displayName", "User.presence", "User.currentlyActive", "User.lastPresenceTs"]);
    return user;
}

/** */
module.exports = SyncApi;
//# sourceMappingURL=sync.js.map