#############################################################################
##
#W  ideals-def.gd           Manuel Delgado <mdelgado@fc.up.pt>
#W                          Pedro A. Garcia-Sanchez <pedro@ugr.es>
#W                          Jose Morais <josejoao@fc.up.pt>
##
##
#Y  Copyright 2005 by Manuel Delgado,
#Y  Pedro Garcia-Sanchez and Jose Joao Morais
#Y  We adopt the copyright regulations of GAP as detailed in the
#Y  copyright notice in the GAP manual.
##
#############################################################################

#############################################################################
##
#R  IsIdealOfNumericalSemigroupRep
##
##  The representation of an ideal of a numerical semigroup.
##
#############################################################################
DeclareRepresentation( "IsIdealOfNumericalSemigroupRep", IsAttributeStoringRep,
        ["UnderlyingNSIdeal", "Generators"] );


#############################################################################
##
#C  IsIdealOfNumericalSemigroup
##
##  The category of ideals of numerical semigroups.
##
#############################################################################
DeclareCategory( "IsIdealOfNumericalSemigroup", IsAdditiveMagma and IsIdealOfNumericalSemigroupRep);


# Elements of ideals of numerical semigroups are integers, so ideals of
# numerical semigroups are collections of integers.
BindGlobal( "IdealsOfNumericalSemigroupsType",
        NewType( CollectionsFamily(CyclotomicsFamily),
                 IsIdealOfNumericalSemigroup));


#############################################################################
##
#F IdealOfNumericalSemigroup(l,S)
##
## l is a list of integers and S a numerical semigroup
##
## returns the ideal of S generated by l.
##
#############################################################################
DeclareGlobalFunction("IdealOfNumericalSemigroup");
#A
DeclareAttribute( "UnderlyingNSIdeal", IsNumericalSemigroup);


#############################################################################
##
#A  Generators(I)
#A  GeneratorsOfIdealOfNumericalSemigroup(I)
##
##  Returns a set of generators of the ideal I.
##  If a minimal generating system has already been computed, this
##  is the set returned.
############################################################################
##DeclareGlobalFunction("GeneratorsOfIdealOfNumericalSemigroup");
#A
DeclareAttribute( "Generators", IsIdealOfNumericalSemigroup);
DeclareSynonymAttr( "GeneratorsOfIdealOfNumericalSemigroup", Generators);


#############################################################################
##
#F  GeneratorsOfIdealOfNumericalSemigroupNC(I)
## just for compatibility of code...
##  Returns a set of generators of the ideal I.
############################################################################
DeclareSynonym("GeneratorsOfIdealOfNumericalSemigroupNC",GeneratorsOfIdealOfNumericalSemigroup);



#############################################################################
##
#F AmbientNumericalSemigroupOfIdeal(I)
##
##  Returns the ambient semigroup of the ideal I.
############################################################################
DeclareGlobalFunction("AmbientNumericalSemigroupOfIdeal");

#############################################################################
##
#P  IsIntegralIdealOfNumericalSemigroup(i)
##
##  Detects if the ideal i is contained in its ambient semigroup
##
#############################################################################
DeclareProperty("IsIntegral", IsIdealOfNumericalSemigroup);
DeclareSynonym("IsIntegralIdealOfNumericalSemigroup", IsIntegral);


#############################################################################
##
#F SumIdealsOfNumericalSemigroup(I,J)
##
## returns the sum of the ideals I and J (in the same ambient semigroup)
#############################################################################
DeclareGlobalFunction("SumIdealsOfNumericalSemigroup");



#############################################################################
##
#F SubtractIdealsOfNumericalSemigroup(I,J)
##
## returns the ideal I - J
#############################################################################
DeclareGlobalFunction("SubtractIdealsOfNumericalSemigroup");


#############################################################################
##
#F  BelongsToIdealOfNumericalSemigroup(n,I)
##
##  Tests if the integer n belongs to the ideal I.
##
#############################################################################
DeclareGlobalFunction("BelongsToIdealOfNumericalSemigroup");



#############################################################################
##
#F DifferenceOfdealsOfNumericalSemigroup(I,J)
##
## returns the set difference I\J
#############################################################################
DeclareGlobalFunction("DifferenceOfIdealsOfNumericalSemigroup");



#############################################################################
##
#F MultipleOfIdealOfNumericalSemigroup(n,I)
##
## n is a non negative integer and I is an ideal
## returns the multiple nI (I+...+I n times) of I
#############################################################################
DeclareGlobalFunction("MultipleOfIdealOfNumericalSemigroup");



#############################################################################
##
#F HilbertFunctionOfIdealOfNumericalSemigroup(n,I)
##
## returns the value of the Hilbert function associated to I in n,
## that is, nI\(n+1)I. I must be an ideal included in its ambient semigroup.
#############################################################################
DeclareGlobalFunction("HilbertFunctionOfIdealOfNumericalSemigroup");



#############################################################################
##
#F BlowUpIdealOfNumericalSemigroup(I)
##
##  Computes the Blow Up (of the maximal ideal) of
##  the numerical semigroup <s>.
#############################################################################
DeclareGlobalFunction("BlowUpIdealOfNumericalSemigroup");


#############################################################################
##
#A MinimalGenerators(I)
#A MinimalGeneratingSystem(I)
#A MinimalGeneratingSystemOfIdealOfNumericalSemigroup(I)
##
## The argument I is an ideal of a numerical semigroup
## returns the minimal generating system of I.
##
#############################################################################
DeclareAttribute( "MinimalGenerators", IsIdealOfNumericalSemigroup);
#DeclareSynonymAttr("MinimalGeneratingSystem", MinimalGenerators);
DeclareSynonymAttr("MinimalGeneratingSystemOfIdealOfNumericalSemigroup", MinimalGenerators);


#############################################################################
##
#A SmallElements
#A SmallElementsOfIdealOfNumericalSemigroup
##
##  Returns the list of elements in the ideal I up to the last gap + 1.
##
#############################################################################
DeclareAttribute("SmallElements", IsIdealOfNumericalSemigroup);
DeclareSynonymAttr( "SmallElementsOfIdealOfNumericalSemigroup", SmallElements);

#############################################################################
##
#A  ConductorOfIdealOfNumericalSemigroup(I)
##
##  Returns the conductor of I, the largest element in SmallElements(I)
##
#############################################################################
DeclareAttribute("Conductor", IsIdealOfNumericalSemigroup);
DeclareSynonymAttr("ConductorOfIdealOfNumericalSemigroup", Conductor);


#############################################################################
##
#F ElementNumber_IdealOfNumericalSemigroup(S,n)
# Given an ideal I of a numerical semigroup and an integer n, returns 
# the nth element of I
#############################################################################
DeclareGlobalFunction("ElementNumber_IdealOfNumericalSemigroup");

#############################################################################
##
#F NumberElement_IdealOfNumericalSemigroup(S,n)
# Given an ideal of a numerical semigroup I and an integer n, returns the 
# position of n in I
#############################################################################
DeclareGlobalFunction("NumberElement_IdealOfNumericalSemigroup");



#############################################################################
##
#F  MaximalIdealOfNumericalSemigroup(S)
##
##  Returns the maximal ideal of S.
##
#############################################################################
DeclareGlobalFunction("MaximalIdealOfNumericalSemigroup");

#############################################################################
##
#P IsMonomialNumericalSemigroup
## Tests if a numerical semigroup is a monomial semigroup ring
##
#############################################################################
DeclareProperty("IsMonomialNumericalSemigroup", IsNumericalSemigroup);

#############################################################################
##
#F  BlowUpOfNumericalSemigroup(s)
##
##  Computes the Blow Up (of the maximal ideal) of
##  the numerical semigroup <s>.
##
#############################################################################
DeclareGlobalFunction("BlowUpOfNumericalSemigroup");
DeclareSynonym("LipmanSemigroup",BlowUpOfNumericalSemigroup);
#############################################################################
##
#F MultiplicitySequenceOfNumericalSemigroup(s)
##
##  Computes the multiplicity sequence of the numerical semigroup <s>.
##
#############################################################################
DeclareGlobalFunction("MultiplicitySequenceOfNumericalSemigroup");

#############################################################################
##
#F  MicroInvariantsOfNumericalSemigroup(s)
##
##  Computes the microinvariants of the numerical semigroup <s>
##  using the formula given by Valentina and Ralf [BF06]. The
##  microinvariants of a numerial semigroup where introduced
##  by J. Elias in [E01].
##
#############################################################################
DeclareGlobalFunction("MicroInvariantsOfNumericalSemigroup");


#############################################################################
##
#P  IsGradedAssociatedRingNumericalSemigroupCM(s)
##
##  Returns true if the associated graded ring of
##  the semigroup ring algebra k[[s]] is Cohen-Macaulay.
##  This function implements the algorithm given in [BF06].
##
#############################################################################
DeclareProperty("IsGradedAssociatedRingNumericalSemigroupCM", IsNumericalSemigroup);


#############################################################################
##
#F  CanonicalIdealOfNumericalSemigroup(s)
##
##  Computes a canonical ideal of <s> ([B06]):
##      { x in Z | g-x not in S}
##
#############################################################################
DeclareGlobalFunction("CanonicalIdealOfNumericalSemigroup");

#############################################################################
##
#P  IsCanonicalIdealOfNumericalSemigroup(e)
##
##  Detects if the ideal e is a translation of the canonical ideal of its
##  ambient semigroup
##
#############################################################################
DeclareProperty("IsCanonicalIdeal",IsIdealOfNumericalSemigroup);
DeclareSynonymAttr("IsCanonicalIdealOfNumericalSemigroup", IsCanonicalIdeal);


#############################################################################
##
#A  ReductionNumberIdealNumericalSemigroup(I)
##
##  Returns the least nonnegative integer such that
##  nI-nI=(n+1)I-(n+1)I, see [B06].
##
#############################################################################
DeclareAttribute("ReductionNumber",IsIdealOfNumericalSemigroup);
DeclareSynonymAttr("ReductionNumberIdealNumericalSemigroup",ReductionNumber);

#############################################################################
##
#F  RatliffRushClosureOfIdealOfNumericalSemigroup(I)
##
##  Returns the the union of all (n+1)I-nI with n nonnegative integers
##
#############################################################################
DeclareGlobalFunction("RatliffRushClosureOfIdealOfNumericalSemigroup");

#############################################################################
##
#F  RatliffRushNumberOfIdealOfNumericalSemigroup(I)
##
##  Returns the least nonnegative integer such that
##  (n+1)I-nI is the Ratliff-Rush closure of I, see [DA-G-H].
##
#############################################################################
DeclareGlobalFunction("RatliffRushNumberOfIdealOfNumericalSemigroup");

#############################################################################
##
#F  AsymptoticRatliffRushNumberOfIdealOfNumericalSemigroup(I)
##
##  Returns the least nonnegative integer n such that
##  mI equals the Ratliff-Rush closure of mI for all m>=n,  see [DA-G-H].
##
#############################################################################
DeclareGlobalFunction("AsymptoticRatliffRushNumberOfIdealOfNumericalSemigroup");


#############################################################################
##
#F  TranslationOfIdealOfNumericalSemigroup(k,I)
##
##  Given an ideal <I> of a numerical semigroup S and an integer <k>
##  returns an ideal of the numerical semigroup S generated by
##  {i1+k,...,in+k} where {i1,...,in} is the system of generators of <I>.
##
#############################################################################
DeclareGlobalFunction("TranslationOfIdealOfNumericalSemigroup");


#############################################################################
##
#F  IntersectionIdealsOfNumericalSemigroup(I,J)
##
##  Given two ideals <I> and <J> of a numerical semigroup S
##  returns the ideal of the numerical semigroup S which is the
##  intersection of the ideals <I> and <J>.
##
#############################################################################
DeclareGlobalFunction("IntersectionIdealsOfNumericalSemigroup");

########################################################################
##
#F AperyListOfIdealOfNumericalSemigroupWRTElement(I,n)
##
##  Computes the sets of elements x of I such that x-n not in I,
##  where n is supposed to be in the ambient semigroup of I.
##  The element in the i-th position of the output list (starting in 0)
##  is congruent with i modulo n
########################################################################
DeclareGlobalFunction("AperyListOfIdealOfNumericalSemigroupWRTElement");

########################################################################
##
#F AperyTableOfNumericalSemigroup(S)
##
##  Computes the Apéry table associated to S as
## explained in [CJZ],
##  that is, a list containing the Apéry list of S with respect to
## its multiplicity and the Apéry lists of kM (with M the maximal
##  ideal of S) with respect to the multiplicity of S, for k=1..r,
##  where r is the reduction number of M
##  (see ReductionNumberIdealNumericalSemigroup).
########################################################################
DeclareGlobalFunction("AperyTableOfNumericalSemigroup");

########################################################################
##
#F StarClosureOfIdealOfNumericalSemigroup(i,is)
##  i is an ideal and is is a set of ideals (all from the same
##	numerical semigroup). The output is i^{*_is}, where
## *_is is the star operation generated by is
## The implementation uses Section 3 of
##  -D. Spirito, Star Operations on Numerical Semigroups
########################################################################

DeclareGlobalFunction("StarClosureOfIdealOfNumericalSemigroup");
