# -*- coding: utf-8 -*-

# Copyright (c) 2015 - 2018 Detlev Offenbach <detlev@die-offenbachs.de>
#


"""
Module implementing a dialog to enter a file to be processed.
"""

from __future__ import unicode_literals

import os

from PyQt5.QtCore import pyqtSlot
from PyQt5.QtWidgets import QDialog, QDialogButtonBox

from E5Gui.E5PathPicker import E5PathPickerModes

from .Ui_PipFileSelectionDialog import Ui_PipFileSelectionDialog

import Utilities


# TODO: add checkbox to select an installation into 'user site'
class PipFileSelectionDialog(QDialog, Ui_PipFileSelectionDialog):
    """
    Class implementing a dialog to enter a file to be processed.
    """
    def __init__(self, plugin, mode, parent=None):
        """
        Constructor
        
        @param plugin reference to the plugin object
        @type PipInterfacePlugin
        @param mode mode of the dialog
        @type str
        @param parent reference to the parent widget
        @type QWidget
        """
        super(PipFileSelectionDialog, self).__init__(parent)
        self.setupUi(self)
        
        if mode == "requirements":
            self.fileLabel.setText(self.tr("Enter requirements file:"))
            self.filePicker.setMode(E5PathPickerModes.OpenFileMode)
            self.filePicker.setToolTip(self.tr(
                "Press to select the requirements file through a file"
                " selection dialog."))
            self.filePicker.setFilters(
                self.tr("Text Files (*.txt);;All Files (*)"))
        elif mode == "package":
            self.fileLabel.setText(self.tr("Enter package file:"))
            self.filePicker.setMode(E5PathPickerModes.OpenFileMode)
            self.filePicker.setToolTip(self.tr(
                "Press to select the package file through a file"
                " selection dialog."))
            self.filePicker.setFilters(
                self.tr("Python Wheel (*.whl);;"
                        "Archive Files (*.tar.gz *.zip);;"
                        "All Files (*)"))
        else:
            self.fileLabel.setText(self.tr("Enter file name:"))
            self.filePicker.setMode(E5PathPickerModes.OpenFileMode)
            self.filePicker.setToolTip(self.tr(
                "Press to select a file through a file selection dialog."))
            self.filePicker.setFilters(self.tr("All Files (*)"))
        self.filePicker.setDefaultDirectory(os.path.expanduser("~"))
        
        self.buttonBox.button(QDialogButtonBox.Ok).setEnabled(False)
        
        self.__default = self.tr("<Default>")
        pipExecutables = sorted(plugin.getPreferences("PipExecutables"))
        self.pipComboBox.addItem(self.__default)
        self.pipComboBox.addItems(pipExecutables)
        
        msh = self.minimumSizeHint()
        self.resize(max(self.width(), msh.width()), msh.height())
    
    @pyqtSlot(str)
    def on_filePicker_textChanged(self, txt):
        """
        Private slot to handle entering the name of a file.
        
        @param txt name of the file
        @type str
        """
        self.buttonBox.button(QDialogButtonBox.Ok).setEnabled(
            bool(txt) and
            os.path.exists(Utilities.toNativeSeparators(txt))
        )
    
    def getData(self):
        """
        Public method to get the entered data.
        
        @return tuple with the pip command and the name of the
            selected file
        @rtype tuple of (str, str)
        """
        command = self.pipComboBox.currentText()
        if command == self.__default:
            command = ""
        
        return command, self.filePicker.text()
