/****************************************************************************
 *   Copyright (C) 2015-2016 by Savoir-faire Linux                               *
 *   Author : Emmanuel Lepage Vallee <emmanuel.lepage@savoirfairelinux.com> *
 *                                                                          *
 *   This library is free software; you can redistribute it and/or          *
 *   modify it under the terms of the GNU Lesser General Public             *
 *   License as published by the Free Software Foundation; either           *
 *   version 2.1 of the License, or (at your option) any later version.     *
 *                                                                          *
 *   This library is distributed in the hope that it will be useful,        *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *   Lesser General Public License for more details.                        *
 *                                                                          *
 *   You should have received a copy of the GNU General Public License      *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.  *
 ***************************************************************************/
#pragma once

#include <QtCore/QObject>

#include "typedefs.h"
#include <libcard/flagutils.h>

class Call;
class CallModelPrivate;

namespace Media {
   class MediaPrivate;
}

namespace Media {

class LIB_EXPORT Media : public QObject
{
   Q_OBJECT
   friend class ::CallModelPrivate;
public:
   Q_PROPERTY(Direction direction READ direction)
   Q_PROPERTY(Call*     call      READ call)

   enum class Type {
      AUDIO = 0, /*!< */
      VIDEO = 1, /*!< */
      TEXT  = 2, /*!< */
      FILE  = 3, /*!< */
      COUNT__
   };
   Q_ENUMS(Type)

   /// Media flags to be able to easily add and remove media from builders.
   enum class TypeFlags {
      NONE  = 0x0 << 0, /*!< */
      AUDIO = 0x1 << 0, /*!< */
      VIDEO = 0x1 << 1, /*!< */
      TEXT  = 0x1 << 2, /*!< */
      FILE  = 0x1 << 3, /*!< */
      COUNT__
   };

   enum class State {
      ACTIVE = 0, /*!< The media is currently in progress       */
      MUTED  = 1, /*!< The media has been paused                */
      IDLE   = 2, /*!< The media is passive, but in progress    */
      OVER   = 3, /*!< The media is terminated                  */
      COUNT__
   };
   Q_ENUMS(State)

   enum class Direction {
      IN , /*!< The media is coming from the peer   */
      OUT, /*!< The media is going to the peer      */
      BOTH, /*!< This is a purely interactive media */
      COUNT__
   };
   Q_ENUMS(Direction)

   enum class Action {
      MUTE     , /*!< Mute this media   */
      UNMUTE   , /*!< Unmute this media */
      TERMINATE, /*!< End this media    */
      COUNT__
   };
   Q_ENUMS(Action)

   //Getter
   virtual Media::Type type() = 0;
   Call* call() const;
   Direction direction() const;

   //Getters
   Media::Media::State state() const;
   bool performAction(const Media::Action);

   //TODO add an abstract history getter with specialisation per media

   virtual ~Media();

   static TypeFlags toFlags(Type t);

protected:

   //Protected mutators
   virtual bool mute();
   virtual bool unmute();
   virtual bool terminate();

   Media(Call* parent, const Direction direction);
   explicit Media(QObject* parent = nullptr);

Q_SIGNALS:
   void stateChanged(const Media::State state, const Media::State previous);

private:
   MediaPrivate* d_ptr;
   Q_DECLARE_PRIVATE(Media)
};

}

Q_DECLARE_METATYPE(Media::Media*)
Q_DECLARE_METATYPE(Media::Media::Direction)
Q_DECLARE_METATYPE(Media::Media::Type)

DECLARE_ENUM_FLAGS(Media::Media::TypeFlags)

Media::Media* operator<<(Media::Media* m, Media::Media::Action a);

