/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.ml.job;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.logging.log4j.util.Supplier;
import org.elasticsearch.ResourceNotFoundException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.support.WriteRequest;
import org.elasticsearch.client.Client;
import org.elasticsearch.cluster.AckedClusterStateUpdateTask;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateTaskConfig;
import org.elasticsearch.cluster.ClusterStateUpdateTask;
import org.elasticsearch.cluster.ack.AckedRequest;
import org.elasticsearch.cluster.metadata.MetaData;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.CheckedConsumer;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.logging.DeprecationLogger;
import org.elasticsearch.common.logging.Loggers;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.xcontent.ToXContent;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.XContentFactory;
import org.elasticsearch.env.Environment;
import org.elasticsearch.index.analysis.AnalysisRegistry;
import org.elasticsearch.persistent.PersistentTasksCustomMetaData;
import org.elasticsearch.xpack.core.XPackPlugin;
import org.elasticsearch.xpack.core.ml.MachineLearningField;
import org.elasticsearch.xpack.core.ml.MlMetadata;
import org.elasticsearch.xpack.core.ml.MlTasks;
import org.elasticsearch.xpack.core.ml.action.DeleteJobAction;
import org.elasticsearch.xpack.core.ml.action.PutJobAction;
import org.elasticsearch.xpack.core.ml.action.RevertModelSnapshotAction;
import org.elasticsearch.xpack.core.ml.action.UpdateJobAction;
import org.elasticsearch.xpack.core.ml.action.util.QueryPage;
import org.elasticsearch.xpack.core.ml.job.config.AnalysisLimits;
import org.elasticsearch.xpack.core.ml.job.config.CategorizationAnalyzerConfig;
import org.elasticsearch.xpack.core.ml.job.config.DataDescription;
import org.elasticsearch.xpack.core.ml.job.config.Job;
import org.elasticsearch.xpack.core.ml.job.config.JobState;
import org.elasticsearch.xpack.core.ml.job.config.JobUpdate;
import org.elasticsearch.xpack.core.ml.job.config.MlFilter;
import org.elasticsearch.xpack.core.ml.job.messages.Messages;
import org.elasticsearch.xpack.core.ml.job.persistence.JobStorageDeletionTask;
import org.elasticsearch.xpack.core.ml.job.process.autodetect.state.ModelSizeStats;
import org.elasticsearch.xpack.core.ml.job.process.autodetect.state.ModelSnapshot;
import org.elasticsearch.xpack.core.ml.utils.ExceptionsHelper;
import org.elasticsearch.xpack.ml.job.UpdateJobProcessNotifier;
import org.elasticsearch.xpack.ml.job.categorization.CategorizationAnalyzer;
import org.elasticsearch.xpack.ml.job.persistence.JobProvider;
import org.elasticsearch.xpack.ml.job.persistence.JobResultsPersister;
import org.elasticsearch.xpack.ml.job.process.autodetect.UpdateParams;
import org.elasticsearch.xpack.ml.notifications.Auditor;
import org.elasticsearch.xpack.ml.utils.ChainTaskExecutor;

public class JobManager
extends AbstractComponent {
    private static final DeprecationLogger DEPRECATION_LOGGER = new DeprecationLogger(Loggers.getLogger(JobManager.class));
    private final Environment environment;
    private final JobProvider jobProvider;
    private final ClusterService clusterService;
    private final Auditor auditor;
    private final Client client;
    private final UpdateJobProcessNotifier updateJobProcessNotifier;
    private volatile ByteSizeValue maxModelMemoryLimit;

    public JobManager(Environment environment, Settings settings, JobProvider jobProvider, ClusterService clusterService, Auditor auditor, Client client, UpdateJobProcessNotifier updateJobProcessNotifier) {
        super(settings);
        this.environment = environment;
        this.jobProvider = Objects.requireNonNull(jobProvider);
        this.clusterService = Objects.requireNonNull(clusterService);
        this.auditor = Objects.requireNonNull(auditor);
        this.client = Objects.requireNonNull(client);
        this.updateJobProcessNotifier = updateJobProcessNotifier;
        this.maxModelMemoryLimit = (ByteSizeValue)MachineLearningField.MAX_MODEL_MEMORY_LIMIT.get(settings);
        clusterService.getClusterSettings().addSettingsUpdateConsumer(MachineLearningField.MAX_MODEL_MEMORY_LIMIT, this::setMaxModelMemoryLimit);
    }

    private void setMaxModelMemoryLimit(ByteSizeValue maxModelMemoryLimit) {
        this.maxModelMemoryLimit = maxModelMemoryLimit;
    }

    public Job getJobOrThrowIfUnknown(String jobId) {
        return JobManager.getJobOrThrowIfUnknown(jobId, this.clusterService.state());
    }

    public static Job getJobOrThrowIfUnknown(String jobId, ClusterState clusterState) {
        Job job = (Job)MlMetadata.getMlMetadata((ClusterState)clusterState).getJobs().get(jobId);
        if (job == null) {
            throw ExceptionsHelper.missingJobException((String)jobId);
        }
        return job;
    }

    private Set<String> expandJobIds(String expression, boolean allowNoJobs, ClusterState clusterState) {
        return MlMetadata.getMlMetadata((ClusterState)clusterState).expandJobIds(expression, allowNoJobs);
    }

    public QueryPage<Job> expandJobs(String expression, boolean allowNoJobs, ClusterState clusterState) {
        Set<String> expandedJobIds = this.expandJobIds(expression, allowNoJobs, clusterState);
        MlMetadata mlMetadata = MlMetadata.getMlMetadata((ClusterState)clusterState);
        ArrayList<Job> jobs = new ArrayList<Job>();
        for (String expandedJobId : expandedJobIds) {
            jobs.add((Job)mlMetadata.getJobs().get(expandedJobId));
        }
        this.logger.debug("Returning jobs matching [" + expression + "]");
        return new QueryPage(jobs, (long)jobs.size(), Job.RESULTS_FIELD);
    }

    public JobState getJobState(String jobId) {
        PersistentTasksCustomMetaData tasks = (PersistentTasksCustomMetaData)this.clusterService.state().getMetaData().custom("persistent_tasks");
        return MlTasks.getJobState((String)jobId, (PersistentTasksCustomMetaData)tasks);
    }

    static void validateCategorizationAnalyzer(Job.Builder jobBuilder, AnalysisRegistry analysisRegistry, Environment environment) throws IOException {
        CategorizationAnalyzerConfig categorizationAnalyzerConfig = jobBuilder.getAnalysisConfig().getCategorizationAnalyzerConfig();
        if (categorizationAnalyzerConfig != null) {
            CategorizationAnalyzer.verifyConfigBuilder(new CategorizationAnalyzerConfig.Builder(categorizationAnalyzerConfig), analysisRegistry, environment);
        }
    }

    public void putJob(final PutJobAction.Request request, AnalysisRegistry analysisRegistry, ClusterState state, final ActionListener<PutJobAction.Response> actionListener) throws IOException {
        request.getJobBuilder().validateAnalysisLimitsAndSetDefaults(this.maxModelMemoryLimit);
        JobManager.validateCategorizationAnalyzer(request.getJobBuilder(), analysisRegistry, this.environment);
        final Job job = request.getJobBuilder().build(new Date());
        if (job.getDataDescription() != null && job.getDataDescription().getFormat() == DataDescription.DataFormat.DELIMITED) {
            DEPRECATION_LOGGER.deprecated("Creating jobs with delimited data format is deprecated. Please use xcontent instead.", new Object[0]);
        }
        XPackPlugin.checkReadyForXPackCustomMetadata((ClusterState)state);
        MlMetadata currentMlMetadata = MlMetadata.getMlMetadata((ClusterState)state);
        if (currentMlMetadata.getJobs().containsKey(job.getId())) {
            actionListener.onFailure((Exception)ExceptionsHelper.jobAlreadyExists((String)job.getId()));
            return;
        }
        ActionListener<Boolean> putJobListener = new ActionListener<Boolean>(){

            public void onResponse(Boolean indicesCreated) {
                JobManager.this.clusterService.submitStateUpdateTask("put-job-" + job.getId(), (ClusterStateTaskConfig)new AckedClusterStateUpdateTask<PutJobAction.Response>((AckedRequest)request, actionListener){

                    protected PutJobAction.Response newResponse(boolean acknowledged) {
                        JobManager.this.auditor.info(job.getId(), Messages.getMessage((String)"Job created"));
                        return new PutJobAction.Response(job);
                    }

                    public ClusterState execute(ClusterState currentState) {
                        return JobManager.this.updateClusterState(job, false, currentState);
                    }
                });
            }

            public void onFailure(Exception e) {
                Matcher matcher;
                if (e instanceof IllegalArgumentException && (matcher = Pattern.compile("(?:mapper|Can't merge a non object mapping) \\[(.*)\\] (?:of different type, current_type \\[.*\\], merged_type|with an object mapping) \\[.*\\]").matcher(e.getMessage())).matches()) {
                    String msg = Messages.getMessage((String)"This job would cause a mapping clash with existing field [{0}] - avoid the clash by assigning a dedicated results index", (Object[])new Object[]{matcher.group(1)});
                    actionListener.onFailure((Exception)ExceptionsHelper.badRequestException((String)msg, (Throwable)e, (Object[])new Object[0]));
                    return;
                }
                actionListener.onFailure(e);
            }
        };
        ActionListener checkForLeftOverDocs = ActionListener.wrap(arg_0 -> this.lambda$putJob$0(job, state, (ActionListener)putJobListener, arg_0), arg_0 -> actionListener.onFailure(arg_0));
        this.jobProvider.checkForLeftOverDocuments(job, (ActionListener<Boolean>)checkForLeftOverDocs);
    }

    public void updateJob(UpdateJobAction.Request request, ActionListener<PutJobAction.Response> actionListener) {
        Job job = this.getJobOrThrowIfUnknown(request.getJobId());
        this.validate(request.getJobUpdate(), job, (ActionListener<Void>)ActionListener.wrap(nullValue -> this.internalJobUpdate(request, actionListener), arg_0 -> actionListener.onFailure(arg_0)));
    }

    private void validate(JobUpdate jobUpdate, Job job, ActionListener<Void> handler) {
        ChainTaskExecutor chainTaskExecutor = new ChainTaskExecutor(this.client.threadPool().executor("ml_utility"), true);
        this.validateModelSnapshotIdUpdate(job, jobUpdate.getModelSnapshotId(), chainTaskExecutor);
        this.validateAnalysisLimitsUpdate(job, jobUpdate.getAnalysisLimits(), chainTaskExecutor);
        chainTaskExecutor.execute(handler);
    }

    private void validateModelSnapshotIdUpdate(Job job, String modelSnapshotId, ChainTaskExecutor chainTaskExecutor) {
        if (modelSnapshotId != null) {
            chainTaskExecutor.add(listener -> this.jobProvider.getModelSnapshot(job.getId(), modelSnapshotId, newModelSnapshot -> {
                if (newModelSnapshot == null) {
                    String message = Messages.getMessage((String)"No model snapshot with id [{0}] exists for job [{1}]", (Object[])new Object[]{modelSnapshotId, job.getId()});
                    listener.onFailure((Exception)new ResourceNotFoundException(message, new Object[0]));
                    return;
                }
                this.jobProvider.getModelSnapshot(job.getId(), job.getModelSnapshotId(), oldModelSnapshot -> {
                    if (oldModelSnapshot != null && ((ModelSnapshot)newModelSnapshot.result).getTimestamp().before(((ModelSnapshot)oldModelSnapshot.result).getTimestamp())) {
                        String message = "Job [" + job.getId() + "] has a more recent model snapshot [" + ((ModelSnapshot)oldModelSnapshot.result).getSnapshotId() + "]";
                        listener.onFailure((Exception)new IllegalArgumentException(message));
                    }
                    listener.onResponse(null);
                }, arg_0 -> ((ActionListener)listener).onFailure(arg_0));
            }, arg_0 -> ((ActionListener)listener).onFailure(arg_0)));
        }
    }

    private void validateAnalysisLimitsUpdate(Job job, AnalysisLimits newLimits, ChainTaskExecutor chainTaskExecutor) {
        if (newLimits == null || newLimits.getModelMemoryLimit() == null) {
            return;
        }
        Long newModelMemoryLimit = newLimits.getModelMemoryLimit();
        chainTaskExecutor.add(listener -> {
            if (this.isJobOpen(this.clusterService.state(), job.getId())) {
                listener.onFailure((Exception)ExceptionsHelper.badRequestException((String)("Cannot update " + Job.ANALYSIS_LIMITS.getPreferredName() + " while the job is open"), (Object[])new Object[0]));
                return;
            }
            this.jobProvider.modelSizeStats(job.getId(), modelSizeStats -> {
                if (modelSizeStats != null) {
                    ByteSizeValue modelSize = new ByteSizeValue(modelSizeStats.getModelBytes(), ByteSizeUnit.BYTES);
                    if (newModelMemoryLimit < modelSize.getMb()) {
                        listener.onFailure((Exception)ExceptionsHelper.badRequestException((String)Messages.getMessage((String)"Invalid update value for analysis_limits: model_memory_limit cannot be decreased below current usage; current usage [{0}], update had [{1}]", (Object[])new Object[]{new ByteSizeValue(modelSize.getMb(), ByteSizeUnit.MB), new ByteSizeValue(newModelMemoryLimit.longValue(), ByteSizeUnit.MB)}), (Object[])new Object[0]));
                        return;
                    }
                }
                listener.onResponse(null);
            }, arg_0 -> ((ActionListener)listener).onFailure(arg_0));
        });
    }

    private void internalJobUpdate(final UpdateJobAction.Request request, final ActionListener<PutJobAction.Response> actionListener) {
        if (request.isWaitForAck()) {
            this.clusterService.submitStateUpdateTask("update-job-" + request.getJobId(), (ClusterStateTaskConfig)new AckedClusterStateUpdateTask<PutJobAction.Response>((AckedRequest)request, actionListener){
                private AtomicReference<Job> updatedJob;
                {
                    super(x0, x1);
                    this.updatedJob = new AtomicReference();
                }

                protected PutJobAction.Response newResponse(boolean acknowledged) {
                    return new PutJobAction.Response(this.updatedJob.get());
                }

                public ClusterState execute(ClusterState currentState) {
                    Job job = JobManager.getJobOrThrowIfUnknown(request.getJobId(), currentState);
                    this.updatedJob.set(request.getJobUpdate().mergeWithJob(job, JobManager.this.maxModelMemoryLimit));
                    return JobManager.this.updateClusterState(this.updatedJob.get(), true, currentState);
                }

                public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                    JobManager.this.afterClusterStateUpdate(newState, request);
                }
            });
        } else {
            this.clusterService.submitStateUpdateTask("update-job-" + request.getJobId(), (ClusterStateTaskConfig)new ClusterStateUpdateTask(){
                private AtomicReference<Job> updatedJob = new AtomicReference();

                public ClusterState execute(ClusterState currentState) throws Exception {
                    Job job = JobManager.getJobOrThrowIfUnknown(request.getJobId(), currentState);
                    this.updatedJob.set(request.getJobUpdate().mergeWithJob(job, JobManager.this.maxModelMemoryLimit));
                    return JobManager.this.updateClusterState(this.updatedJob.get(), true, currentState);
                }

                public void onFailure(String source, Exception e) {
                    actionListener.onFailure(e);
                }

                public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                    JobManager.this.afterClusterStateUpdate(newState, request);
                    actionListener.onResponse((Object)new PutJobAction.Response(this.updatedJob.get()));
                }
            });
        }
    }

    private void afterClusterStateUpdate(ClusterState newState, UpdateJobAction.Request request) {
        JobUpdate jobUpdate = request.getJobUpdate();
        boolean processUpdateRequired = jobUpdate.isAutodetectProcessUpdate();
        if (processUpdateRequired && this.isJobOpen(newState, request.getJobId())) {
            this.updateJobProcessNotifier.submitJobUpdate(UpdateParams.fromJobUpdate(jobUpdate), (ActionListener<Boolean>)ActionListener.wrap(isUpdated -> {
                if (isUpdated.booleanValue()) {
                    this.auditJobUpdatedIfNotInternal(request);
                }
            }, e -> {}));
        } else {
            this.logger.debug("[{}] No process update required for job update: {}", new Supplier[]{() -> request.getJobId(), () -> {
                try {
                    XContentBuilder jsonBuilder = XContentFactory.jsonBuilder();
                    jobUpdate.toXContent(jsonBuilder, ToXContent.EMPTY_PARAMS);
                    return Strings.toString((XContentBuilder)jsonBuilder);
                }
                catch (IOException e) {
                    return "(unprintable due to " + e.getMessage() + ")";
                }
            }});
            this.auditJobUpdatedIfNotInternal(request);
        }
    }

    private void auditJobUpdatedIfNotInternal(UpdateJobAction.Request request) {
        if (!request.isInternal()) {
            this.auditor.info(request.getJobId(), Messages.getMessage((String)"Job updated: {0}", (Object[])new Object[]{request.getJobUpdate().getUpdateFields()}));
        }
    }

    private boolean isJobOpen(ClusterState clusterState, String jobId) {
        PersistentTasksCustomMetaData persistentTasks = (PersistentTasksCustomMetaData)clusterState.metaData().custom("persistent_tasks");
        JobState jobState = MlTasks.getJobState((String)jobId, (PersistentTasksCustomMetaData)persistentTasks);
        return jobState == JobState.OPENED;
    }

    private ClusterState updateClusterState(Job job, boolean overwrite, ClusterState currentState) {
        MlMetadata.Builder builder = JobManager.createMlMetadataBuilder(currentState);
        builder.putJob(job, overwrite);
        return JobManager.buildNewClusterState(currentState, builder);
    }

    public void notifyFilterChanged(MlFilter filter, Set<String> addedItems, Set<String> removedItems) {
        if (addedItems.isEmpty() && removedItems.isEmpty()) {
            return;
        }
        ClusterState clusterState = this.clusterService.state();
        QueryPage<Job> jobs = this.expandJobs("*", true, this.clusterService.state());
        for (Job job : jobs.results()) {
            Set jobFilters = job.getAnalysisConfig().extractReferencedFilters();
            if (!jobFilters.contains(filter.getId())) continue;
            if (this.isJobOpen(clusterState, job.getId())) {
                this.updateJobProcessNotifier.submitJobUpdate(UpdateParams.filterUpdate(job.getId(), filter), (ActionListener<Boolean>)ActionListener.wrap(isUpdated -> this.auditFilterChanges(job.getId(), filter.getId(), addedItems, removedItems), e -> {}));
                continue;
            }
            this.auditFilterChanges(job.getId(), filter.getId(), addedItems, removedItems);
        }
    }

    private void auditFilterChanges(String jobId, String filterId, Set<String> addedItems, Set<String> removedItems) {
        StringBuilder auditMsg = new StringBuilder("Filter [");
        auditMsg.append(filterId);
        auditMsg.append("] has been modified; ");
        if (!addedItems.isEmpty()) {
            auditMsg.append("added items: ");
            JobManager.appendCommaSeparatedSet(addedItems, auditMsg);
            if (!removedItems.isEmpty()) {
                auditMsg.append(", ");
            }
        }
        if (!removedItems.isEmpty()) {
            auditMsg.append("removed items: ");
            JobManager.appendCommaSeparatedSet(removedItems, auditMsg);
        }
        this.auditor.info(jobId, auditMsg.toString());
    }

    private static void appendCommaSeparatedSet(Set<String> items, StringBuilder sb) {
        sb.append("[");
        Strings.collectionToDelimitedString(items, (String)", ", (String)"'", (String)"'", (StringBuilder)sb);
        sb.append("]");
    }

    public void updateProcessOnCalendarChanged(List<String> calendarJobIds) {
        ClusterState clusterState = this.clusterService.state();
        MlMetadata mlMetadata = MlMetadata.getMlMetadata((ClusterState)clusterState);
        List<String> existingJobsOrGroups = calendarJobIds.stream().filter(arg_0 -> ((MlMetadata)mlMetadata).isGroupOrJob(arg_0)).collect(Collectors.toList());
        HashSet expandedJobIds = new HashSet();
        existingJobsOrGroups.forEach(jobId -> expandedJobIds.addAll(this.expandJobIds((String)jobId, true, clusterState)));
        for (String jobId2 : expandedJobIds) {
            if (!this.isJobOpen(clusterState, jobId2)) continue;
            this.updateJobProcessNotifier.submitJobUpdate(UpdateParams.scheduledEventsUpdate(jobId2), (ActionListener<Boolean>)ActionListener.wrap(isUpdated -> {
                if (isUpdated.booleanValue()) {
                    this.auditor.info(jobId2, Messages.getMessage((String)"Updated calendars in running process"));
                }
            }, e -> {}));
        }
    }

    public void deleteJob(DeleteJobAction.Request request, JobStorageDeletionTask task, ActionListener<DeleteJobAction.Response> actionListener) {
        final String jobId = request.getJobId();
        this.logger.debug("Deleting job '" + jobId + "'");
        CheckedConsumer apiResponseHandler = jobDeleted -> {
            if (jobDeleted.booleanValue()) {
                this.logger.info("Job [" + jobId + "] deleted");
                this.auditor.info(jobId, Messages.getMessage((String)"Job deleted"));
                actionListener.onResponse((Object)new DeleteJobAction.Response(true));
            } else {
                actionListener.onResponse((Object)new DeleteJobAction.Response(false));
            }
        };
        CheckedConsumer deleteJobStateHandler = response -> this.clusterService.submitStateUpdateTask("delete-job-" + jobId, (ClusterStateTaskConfig)new AckedClusterStateUpdateTask<Boolean>((AckedRequest)request, ActionListener.wrap((CheckedConsumer)apiResponseHandler, arg_0 -> ((ActionListener)actionListener).onFailure(arg_0))){

            protected Boolean newResponse(boolean acknowledged) {
                return acknowledged && response != false;
            }

            public ClusterState execute(ClusterState currentState) {
                MlMetadata currentMlMetadata = MlMetadata.getMlMetadata((ClusterState)currentState);
                if (!currentMlMetadata.getJobs().containsKey(jobId)) {
                    return currentState;
                }
                MlMetadata.Builder builder = new MlMetadata.Builder(currentMlMetadata);
                builder.deleteJob(jobId, (PersistentTasksCustomMetaData)currentState.getMetaData().custom("persistent_tasks"));
                return JobManager.buildNewClusterState(currentState, builder);
            }
        });
        CheckedConsumer removeFromCalendarsHandler = response -> this.jobProvider.removeJobFromCalendars(jobId, (ActionListener<Boolean>)ActionListener.wrap(arg_0 -> ((CheckedConsumer)deleteJobStateHandler).accept(arg_0), arg_0 -> ((ActionListener)actionListener).onFailure(arg_0)));
        task.delete(jobId, this.client, this.clusterService.state(), removeFromCalendarsHandler, arg_0 -> actionListener.onFailure(arg_0));
    }

    public void revertSnapshot(final RevertModelSnapshotAction.Request request, final ActionListener<RevertModelSnapshotAction.Response> actionListener, final ModelSnapshot modelSnapshot) {
        ModelSizeStats modelSizeStats = modelSnapshot.getModelSizeStats();
        JobResultsPersister persister = new JobResultsPersister(this.settings, this.client);
        CheckedConsumer modelSizeStatsResponseHandler = response -> persister.persistQuantiles(modelSnapshot.getQuantiles(), WriteRequest.RefreshPolicy.IMMEDIATE, (ActionListener<IndexResponse>)ActionListener.wrap(quantilesResponse -> {
            ModelSnapshot snapshotWithoutQuantiles = new ModelSnapshot.Builder(modelSnapshot).setQuantiles(null).build();
            actionListener.onResponse((Object)new RevertModelSnapshotAction.Response(snapshotWithoutQuantiles));
        }, arg_0 -> ((ActionListener)actionListener).onFailure(arg_0)));
        CheckedConsumer updateHandler = response -> {
            if (response.booleanValue()) {
                ModelSizeStats revertedModelSizeStats = new ModelSizeStats.Builder(modelSizeStats).setLogTime(new Date()).build();
                persister.persistModelSizeStats(revertedModelSizeStats, WriteRequest.RefreshPolicy.IMMEDIATE, (ActionListener<IndexResponse>)ActionListener.wrap((CheckedConsumer)modelSizeStatsResponseHandler, arg_0 -> ((ActionListener)actionListener).onFailure(arg_0)));
            }
        };
        Consumer<Long> clusterStateHandler = response -> this.clusterService.submitStateUpdateTask("revert-snapshot-" + request.getJobId(), (ClusterStateTaskConfig)new AckedClusterStateUpdateTask<Boolean>((AckedRequest)request, ActionListener.wrap((CheckedConsumer)updateHandler, arg_0 -> ((ActionListener)actionListener).onFailure(arg_0))){

            protected Boolean newResponse(boolean acknowledged) {
                if (acknowledged) {
                    JobManager.this.auditor.info(request.getJobId(), Messages.getMessage((String)"Job model snapshot reverted to ''{0}''", (Object[])new Object[]{modelSnapshot.getDescription()}));
                    return true;
                }
                actionListener.onFailure((Exception)new IllegalStateException("Could not revert modelSnapshot on job [" + request.getJobId() + "], not acknowledged by master."));
                return false;
            }

            public ClusterState execute(ClusterState currentState) {
                Job job = JobManager.getJobOrThrowIfUnknown(request.getJobId(), currentState);
                Job.Builder builder = new Job.Builder(job);
                builder.setModelSnapshotId(modelSnapshot.getSnapshotId());
                builder.setEstablishedModelMemory(response);
                return JobManager.this.updateClusterState(builder.build(), true, currentState);
            }
        });
        this.jobProvider.getEstablishedMemoryUsage(request.getJobId(), modelSizeStats.getTimestamp(), modelSizeStats, clusterStateHandler, arg_0 -> actionListener.onFailure(arg_0));
    }

    private static MlMetadata.Builder createMlMetadataBuilder(ClusterState currentState) {
        return new MlMetadata.Builder(MlMetadata.getMlMetadata((ClusterState)currentState));
    }

    private static ClusterState buildNewClusterState(ClusterState currentState, MlMetadata.Builder builder) {
        XPackPlugin.checkReadyForXPackCustomMetadata((ClusterState)currentState);
        ClusterState.Builder newState = ClusterState.builder((ClusterState)currentState);
        newState.metaData(MetaData.builder((MetaData)currentState.getMetaData()).putCustom("ml", (MetaData.Custom)builder.build()).build());
        return newState.build();
    }

    private /* synthetic */ void lambda$putJob$0(Job job, ClusterState state, ActionListener putJobListener, Boolean response) throws Exception {
        this.jobProvider.createJobResultIndex(job, state, (ActionListener<Boolean>)putJobListener);
    }
}

