/* Copyright (C) 2009-2011  Freescale Semiconductor, Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of Freescale Semiconductor, Inc. nor
 *       the names of its contributors may be used to endorse or promote
 *       products derived from this software without specific prior written
 *       permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef __Z160_H
#define __Z160_H


/* Largest buffer is 2048x2048 */
#define	Z160_MAX_WIDTH			2048
#define	Z160_MAX_HEIGHT			2048

/* Buffer base address alignment */
#define	Z160_ALIGN_OFFSET		4096

/* Buffer pitch alignment */
#define	Z160_ALIGN_PITCH		4

/* Maximum stride in (4-byte) words */
#define	Z160_MAX_STRIDE			4095

/* Convert between stride and pitch */
#define	Z160_STRIDE_FROM_PITCH(_x)	(((_x) / Z160_ALIGN_PITCH) - 1)
#define	Z160_PITCH_FROM_STRIDE(_x)	(((_x)+1) * Z160_ALIGN_PITCH)

/* Maximum pitch in bytes */
#define	Z160_MAX_PITCH_BYTES		Z160_PITCH_FROM_STRIDE(Z160_MAX_STRIDE)


/* Raster ops */
#define	Z160_ROP_NOT			0x01
#define	Z160_ROP_DEST			0x02
#define	Z160_ROP_SOURCE			0x04
#define	Z160_ROP_PATTERN		0x08
#define	Z160_ROP_SOURCE_DEST		0x10
#define	Z160_ROP_DEST_PATTERN		0x20
#define	Z160_ROP_SOURCE_PATTERN		0x40
#define	Z160_ROP_DEST_SOURCE_PATTERN	0x80
#define	Z160_MAKE_ROP(mask,nonmask)	(((mask) << 8) | (nonmask))


/* Buffer formats; id values match those supported by the hardware. */
typedef enum _Z160_FORMAT
{

	Z160_FORMAT_1		= 0x00,		/* foreground and background */
	Z160_FORMAT_1BW		= 0x01,		/* black and white */
	Z160_FORMAT_4		= 0x02,
	Z160_FORMAT_8		= 0x03, 	/* use blue */
	Z160_FORMAT_4444	= 0x04,
	Z160_FORMAT_1555	= 0x05,
	Z160_FORMAT_0565	= 0x06,
	Z160_FORMAT_8888	= 0x07,
	Z160_FORMAT_YUY2	= 0x08,
	Z160_FORMAT_UYVY	= 0x09,
	Z160_FORMAT_YVYU	= 0x0a,
	Z160_FORMAT_4444_RGBA	= 0x0b,
	Z160_FORMAT_5551_RGBA	= 0x0c,
	Z160_FORMAT_8888_RGBA	= 0x0d,
	Z160_FORMAT_A8		= 0x0e,
	Z160_FORMAT_88		= 0x0f		/* alpha and blue */

} Z160_FORMAT;


/* Buffer properties. */
typedef struct Z160Buffer_
{
	void*		base;	/* base address for aligned pixel data */
	unsigned	pitch;	/* num bytes to next line */
	unsigned	width;	/* width of 2D buffer in pixels */
	unsigned	height;	/* height of 2D buffer in pixels */
	unsigned	bpp;	/* bits per pixel */
	Z160_FORMAT	format;	/* format for the pixels */
	int		swapRB;	/* flag set to swap red and blue components */
	int		opaque;	/* flag set if buffer has alpha channel and is assumed opaque */
	int		alpha4;	/* flag set if each channel has alpha */
} Z160Buffer;


/* Blending operations */
typedef enum _Z160_BLEND
{

	Z160_BLEND_UNKNOWN		= -1,

	Z160_BLEND_SRC			= 0,
	Z160_BLEND_OVER			= 1,
	Z160_BLEND_IN			= 2,
	Z160_BLEND_IN_REVERSE		= 3,
	Z160_BLEND_OUT_REVERSE		= 4,
	Z160_BLEND_ADD			= 5,

	Z160_BLEND_OP_COUNT		= 6

} Z160_BLEND;


/* Setup mode */
typedef enum _Z160_SETUP
{

	Z160_SETUP_UNKNOWN,
	Z160_SETUP_FILL_SOLID,
	Z160_SETUP_COPY,
	Z160_SETUP_BLEND_IMAGE,
	Z160_SETUP_BLEND_IMAGE_MASKED,
	Z160_SETUP_BLEND_CONST,
	Z160_SETUP_BLEND_CONST_MASKED,
	Z160_SETUP_BLEND_PATTERN,
	Z160_SETUP_BLEND_PATTERN_MASKED

} Z160_SETUP;


/* Connection */
void* z160_connect();
void z160_disconnect(void* context);

/* Sync waits for GPU to finish operations */
void z160_sync(void* context);

/* Flush sends out any queued GPU requests to the hardware. */
void z160_flush(void* context);

/* Target buffer */
void z160_setup_buffer_target(void* context, Z160Buffer* pTarget);

/* Query setup - depends on which z160_setup_* function was called. */
Z160_SETUP z160_get_setup(void* context);

/* Solid Fill */
void z160_setup_fill_solid(void* context, unsigned color);
void z160_fill_solid_rect(
	void* context,
	int x, int y, 			/* target start */
	int width, int height);		/* pixel area */

/* Copy (target and source have same format) */
void z160_setup_copy(
	void* context,
	Z160Buffer* pSource,	/* source buffer */
	int xdir,		/* pos for left-to-right; neg for R-to-L */
	int ydir);		/* pos for top-to-bottom; neg for B-to-T */
void z160_copy_rect(
	void* context,
	int x, int y, 		/* target start */
	int width, int height,	/* pixel area */
	int sx, int sy);	/* source start */

/* Blend Image (no alpha mask) with Target */
void z160_setup_blend_image(
	void* context,
	Z160_BLEND blendOp,	/* blend operation */
	Z160Buffer* pImage);	/* image buffer */
void z160_blend_image_rect(
	void* context,
	int x, int y,		/* target start */
	int width, int height,	/* pixel area */
	int ix, int iy);	/* image start */

/* Blend Image adjusted by alpha mask with Target */
void z160_setup_blend_image_masked(
	void* context,
	Z160_BLEND blendOp,	/* blend operation */
	Z160Buffer* pImage,	/* image buffer */
	Z160Buffer* pMask);	/* mask buffer */
void z160_blend_image_masked_rect(
	void* context,
	int x, int y,		/* target start */
	int width, int height,	/* pixel area */
	int ix, int iy,		/* image start */
	int mx, int my);	/* mask start */

/* Blend 1x1 Pattern (no alpha mask) with Target */
void z160_setup_blend_const(
	void* context,
	Z160_BLEND blendOp,	/* blend operation */
	Z160Buffer* pPattern);	/* 1x1 pattern buffer */
void z160_blend_const_masked(
	void* context,
	int x, int y,		/* target start */
	int width, int height);	/* pixel area */

/* Blend 1x1 Pattern adjusted by alpha mask with Target */
void z160_setup_blend_const_masked(
	void* context,
	Z160_BLEND blendOp,	/* blend operation */
	Z160Buffer* pPattern,	/* 1x1 pattern buffer */
	Z160Buffer* pMask);	/* mask buffer */
void z160_blend_const_masked_rect(
	void* context,
	int x, int y,		/* target start */
	int width, int height,	/* pixel area */
	int mx, int my);	/* mask start */

/* Blend mxn Pattern (no alpha mask) with Target */
void z160_setup_blend_pattern(
	void* context,
	Z160_BLEND blendOp,	/* blend operation */
	Z160Buffer* pPattern);	/* pattern buffer */
void z160_blend_pattern_rect(
	void* context,
	int x, int y,		/* target start */
	int width, int height,	/* pixel area */
	int px, int py);	/* pattern start */

/* Blend mxn Pattern adjusted by alpha mask with Target */
void z160_setup_blend_pattern_masked(
	void* context,
	Z160_BLEND blendOp,	/* blend operation */
	Z160Buffer* pPattern,	/* pattern buffer */
	Z160Buffer* pMask);	/* mask buffer */
void z160_blend_pattern_masked_rect(
	void* context,
	int x, int y,		/* target start */
	int width, int height,	/* pixel area */
	int px, int py,		/* pattern start */
	int mx, int my);	/* mask start */

#endif
