"use strict";

Object.defineProperty(exports, "__esModule", {
    value: true
});

var _getIterator2 = require("babel-runtime/core-js/get-iterator");

var _getIterator3 = _interopRequireDefault(_getIterator2);

var _slicedToArray2 = require("babel-runtime/helpers/slicedToArray");

var _slicedToArray3 = _interopRequireDefault(_slicedToArray2);

var _bluebird = require("bluebird");

var _bluebird2 = _interopRequireDefault(_bluebird);

var _syncAccumulator = require("../sync-accumulator");

var _syncAccumulator2 = _interopRequireDefault(_syncAccumulator);

var _utils = require("../utils");

var _utils2 = _interopRequireDefault(_utils);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

var VERSION = 1; /*
                 Copyright 2017 Vector Creations Ltd
                 Copyright 2018 New Vector Ltd
                 
                 Licensed under the Apache License, Version 2.0 (the "License");
                 you may not use this file except in compliance with the License.
                 You may obtain a copy of the License at
                 
                     http://www.apache.org/licenses/LICENSE-2.0
                 
                 Unless required by applicable law or agreed to in writing, software
                 distributed under the License is distributed on an "AS IS" BASIS,
                 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
                 See the License for the specific language governing permissions and
                 limitations under the License.
                 */

function createDatabase(db) {
    // Make user store, clobber based on user ID. (userId property of User objects)
    db.createObjectStore("users", { keyPath: ["userId"] });

    // Make account data store, clobber based on event type.
    // (event.type property of MatrixEvent objects)
    db.createObjectStore("accountData", { keyPath: ["type"] });

    // Make /sync store (sync tokens, room data, etc), always clobber (const key).
    db.createObjectStore("sync", { keyPath: ["clobber"] });
}

/**
 * Helper method to collect results from a Cursor and promiseify it.
 * @param {ObjectStore|Index} store The store to perform openCursor on.
 * @param {IDBKeyRange=} keyRange Optional key range to apply on the cursor.
 * @param {Function} resultMapper A function which is repeatedly called with a
 * Cursor.
 * Return the data you want to keep.
 * @return {Promise<T[]>} Resolves to an array of whatever you returned from
 * resultMapper.
 */
function selectQuery(store, keyRange, resultMapper) {
    var query = store.openCursor(keyRange);
    return new _bluebird2.default(function (resolve, reject) {
        var results = [];
        query.onerror = function (event) {
            reject(new Error("Query failed: " + event.target.errorCode));
        };
        // collect results
        query.onsuccess = function (event) {
            var cursor = event.target.result;
            if (!cursor) {
                resolve(results);
                return; // end of results
            }
            results.push(resultMapper(cursor));
            cursor.continue();
        };
    });
}

function promiseifyTxn(txn) {
    return new _bluebird2.default(function (resolve, reject) {
        txn.oncomplete = function (event) {
            resolve(event);
        };
        txn.onerror = function (event) {
            reject(event);
        };
    });
}

function promiseifyRequest(req) {
    return new _bluebird2.default(function (resolve, reject) {
        req.onsuccess = function (event) {
            resolve(event);
        };
        req.onerror = function (event) {
            reject(event);
        };
    });
}

/**
 * Does the actual reading from and writing to the indexeddb
 *
 * Construct a new Indexed Database store backend. This requires a call to
 * <code>connect()</code> before this store can be used.
 * @constructor
 * @param {Object} indexedDBInterface The Indexed DB interface e.g
 * <code>window.indexedDB</code>
 * @param {string=} dbName Optional database name. The same name must be used
 * to open the same database.
 */
var LocalIndexedDBStoreBackend = function LocalIndexedDBStoreBackend(indexedDBInterface, dbName) {
    this.indexedDB = indexedDBInterface;
    this._dbName = "matrix-js-sdk:" + (dbName || "default");
    this.db = null;
    this._disconnected = true;
    this._syncAccumulator = new _syncAccumulator2.default();
};

LocalIndexedDBStoreBackend.prototype = {
    /**
     * Attempt to connect to the database. This can fail if the user does not
     * grant permission.
     * @return {Promise} Resolves if successfully connected.
     */
    connect: function connect() {
        var _this = this;

        if (!this._disconnected) {
            console.log("LocalIndexedDBStoreBackend.connect: already connected or connecting");
            return _bluebird2.default.resolve();
        }

        this._disconnected = false;

        console.log("LocalIndexedDBStoreBackend.connect: connecting...");
        var req = this.indexedDB.open(this._dbName, VERSION);
        req.onupgradeneeded = function (ev) {
            var db = ev.target.result;
            var oldVersion = ev.oldVersion;
            console.log("LocalIndexedDBStoreBackend.connect: upgrading from " + oldVersion);
            if (oldVersion < 1) {
                // The database did not previously exist.
                createDatabase(db);
            }
            // Expand as needed.
        };

        req.onblocked = function () {
            console.log("can't yet open LocalIndexedDBStoreBackend because it is open elsewhere");
        };

        console.log("LocalIndexedDBStoreBackend.connect: awaiting connection...");
        return promiseifyRequest(req).then(function (ev) {
            console.log("LocalIndexedDBStoreBackend.connect: connected");
            _this.db = ev.target.result;

            // add a poorly-named listener for when deleteDatabase is called
            // so we can close our db connections.
            _this.db.onversionchange = function () {
                _this.db.close();
            };

            return _this._init();
        });
    },

    /**
     * Having connected, load initial data from the database and prepare for use
     * @return {Promise} Resolves on success
     */
    _init: function _init() {
        var _this2 = this;

        return _bluebird2.default.all([this._loadAccountData(), this._loadSyncData()]).then(function (_ref) {
            var _ref2 = (0, _slicedToArray3.default)(_ref, 2),
                accountData = _ref2[0],
                syncData = _ref2[1];

            console.log("LocalIndexedDBStoreBackend: loaded initial data");
            _this2._syncAccumulator.accumulate({
                next_batch: syncData.nextBatch,
                rooms: syncData.roomsData,
                groups: syncData.groupsData,
                account_data: {
                    events: accountData
                }
            });
        });
    },

    /**
     * Clear the entire database. This should be used when logging out of a client
     * to prevent mixing data between accounts.
     * @return {Promise} Resolved when the database is cleared.
     */
    clearDatabase: function clearDatabase() {
        var _this3 = this;

        return new _bluebird2.default(function (resolve, reject) {
            console.log("Removing indexeddb instance: " + _this3._dbName);
            var req = _this3.indexedDB.deleteDatabase(_this3._dbName);

            req.onblocked = function () {
                console.log("can't yet delete indexeddb " + _this3._dbName + " because it is open elsewhere");
            };

            req.onerror = function (ev) {
                // in firefox, with indexedDB disabled, this fails with a
                // DOMError. We treat this as non-fatal, so that we can still
                // use the app.
                console.warn("unable to delete js-sdk store indexeddb: " + ev.target.error);
                resolve();
            };

            req.onsuccess = function () {
                console.log("Removed indexeddb instance: " + _this3._dbName);
                resolve();
            };
        });
    },

    /**
     * @param {boolean=} copy If false, the data returned is from internal
     * buffers and must not be mutated. Otherwise, a copy is made before
     * returning such that the data can be safely mutated. Default: true.
     *
     * @return {Promise} Resolves with a sync response to restore the
     * client state to where it was at the last save, or null if there
     * is no saved sync data.
     */
    getSavedSync: function getSavedSync(copy) {
        if (copy === undefined) copy = true;

        var data = this._syncAccumulator.getJSON();
        if (!data.nextBatch) return _bluebird2.default.resolve(null);
        if (copy) {
            // We must deep copy the stored data so that the /sync processing code doesn't
            // corrupt the internal state of the sync accumulator (it adds non-clonable keys)
            return _bluebird2.default.resolve(_utils2.default.deepCopy(data));
        } else {
            return _bluebird2.default.resolve(data);
        }
    },

    getNextBatchToken: function getNextBatchToken() {
        return _bluebird2.default.resolve(this._syncAccumulator.getNextBatchToken());
    },

    setSyncData: function setSyncData(syncData) {
        var _this4 = this;

        return _bluebird2.default.resolve().then(function () {
            _this4._syncAccumulator.accumulate(syncData);
        });
    },

    syncToDatabase: function syncToDatabase(userTuples) {
        var syncData = this._syncAccumulator.getJSON();

        return _bluebird2.default.all([this._persistUserPresenceEvents(userTuples), this._persistAccountData(syncData.accountData), this._persistSyncData(syncData.nextBatch, syncData.roomsData, syncData.groupsData)]);
    },

    /**
     * Persist rooms /sync data along with the next batch token.
     * @param {string} nextBatch The next_batch /sync value.
     * @param {Object} roomsData The 'rooms' /sync data from a SyncAccumulator
     * @param {Object} groupsData The 'groups' /sync data from a SyncAccumulator
     * @return {Promise} Resolves if the data was persisted.
     */
    _persistSyncData: function _persistSyncData(nextBatch, roomsData, groupsData) {
        var _this5 = this;

        console.log("Persisting sync data up to ", nextBatch);
        return _bluebird2.default.try(function () {
            var txn = _this5.db.transaction(["sync"], "readwrite");
            var store = txn.objectStore("sync");
            store.put({
                clobber: "-", // constant key so will always clobber
                nextBatch: nextBatch,
                roomsData: roomsData,
                groupsData: groupsData
            }); // put == UPSERT
            return promiseifyTxn(txn);
        });
    },

    /**
     * Persist a list of account data events. Events with the same 'type' will
     * be replaced.
     * @param {Object[]} accountData An array of raw user-scoped account data events
     * @return {Promise} Resolves if the events were persisted.
     */
    _persistAccountData: function _persistAccountData(accountData) {
        var _this6 = this;

        return _bluebird2.default.try(function () {
            var txn = _this6.db.transaction(["accountData"], "readwrite");
            var store = txn.objectStore("accountData");
            for (var i = 0; i < accountData.length; i++) {
                store.put(accountData[i]); // put == UPSERT
            }
            return promiseifyTxn(txn);
        });
    },

    /**
     * Persist a list of [user id, presence event] they are for.
     * Users with the same 'userId' will be replaced.
     * Presence events should be the event in its raw form (not the Event
     * object)
     * @param {Object[]} tuples An array of [userid, event] tuples
     * @return {Promise} Resolves if the users were persisted.
     */
    _persistUserPresenceEvents: function _persistUserPresenceEvents(tuples) {
        var _this7 = this;

        return _bluebird2.default.try(function () {
            var txn = _this7.db.transaction(["users"], "readwrite");
            var store = txn.objectStore("users");
            var _iteratorNormalCompletion = true;
            var _didIteratorError = false;
            var _iteratorError = undefined;

            try {
                for (var _iterator = (0, _getIterator3.default)(tuples), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
                    var tuple = _step.value;

                    store.put({
                        userId: tuple[0],
                        event: tuple[1]
                    }); // put == UPSERT
                }
            } catch (err) {
                _didIteratorError = true;
                _iteratorError = err;
            } finally {
                try {
                    if (!_iteratorNormalCompletion && _iterator.return) {
                        _iterator.return();
                    }
                } finally {
                    if (_didIteratorError) {
                        throw _iteratorError;
                    }
                }
            }

            return promiseifyTxn(txn);
        });
    },

    /**
     * Load all user presence events from the database. This is not cached.
     * FIXME: It would probably be more sensible to store the events in the
     * sync.
     * @return {Promise<Object[]>} A list of presence events in their raw form.
     */
    getUserPresenceEvents: function getUserPresenceEvents() {
        var _this8 = this;

        return _bluebird2.default.try(function () {
            var txn = _this8.db.transaction(["users"], "readonly");
            var store = txn.objectStore("users");
            return selectQuery(store, undefined, function (cursor) {
                return [cursor.value.userId, cursor.value.event];
            });
        });
    },

    /**
     * Load all the account data events from the database. This is not cached.
     * @return {Promise<Object[]>} A list of raw global account events.
     */
    _loadAccountData: function _loadAccountData() {
        var _this9 = this;

        console.log("LocalIndexedDBStoreBackend: loading account data...");
        return _bluebird2.default.try(function () {
            var txn = _this9.db.transaction(["accountData"], "readonly");
            var store = txn.objectStore("accountData");
            return selectQuery(store, undefined, function (cursor) {
                return cursor.value;
            }).then(function (result) {
                console.log("LocalIndexedDBStoreBackend: loaded account data");
                return result;
            });
        });
    },

    /**
     * Load the sync data from the database.
     * @return {Promise<Object>} An object with "roomsData" and "nextBatch" keys.
     */
    _loadSyncData: function _loadSyncData() {
        var _this10 = this;

        console.log("LocalIndexedDBStoreBackend: loading sync data...");
        return _bluebird2.default.try(function () {
            var txn = _this10.db.transaction(["sync"], "readonly");
            var store = txn.objectStore("sync");
            return selectQuery(store, undefined, function (cursor) {
                return cursor.value;
            }).then(function (results) {
                console.log("LocalIndexedDBStoreBackend: loaded sync data");
                if (results.length > 1) {
                    console.warn("loadSyncData: More than 1 sync row found.");
                }
                return results.length > 0 ? results[0] : {};
            });
        });
    }
};

exports.default = LocalIndexedDBStoreBackend;
//# sourceMappingURL=indexeddb-local-backend.js.map