############################################################################## 
##
#W  gpdaut.gi              GAP4 package `groupoids'              Chris Wensley
#W                                                                & Emma Moore
#Y  Copyright (C) 2000-2017, Emma Moore and Chris Wensley,  
#Y  School of Computer Science, Bangor University, U.K. 
##  

#############################################################################
##
#M  GroupoidAutomorphismByObjectPermNC  
#M  GroupoidAutomorphismByObjectPerm 
##
InstallMethod( GroupoidAutomorphismByObjectPermNC , 
    "for a single piece groupoid and a permutation of objects", true, 
    [ IsGroupoid and IsSinglePiece, IsHomogeneousList ], 0,
function( gpd, oims ) 

    local obs, gens, ngens, images, i, a, pt, ph, mor, L; 

    obs := gpd!.objects; 
    gens := GeneratorsOfGroupoid( gpd ); 
    ngens := Length( gens ); 
    images := [1..ngens]; 
    for i in [1..ngens] do 
        a := gens[i]; 
        pt := Position( obs, a![2] ); 
        ph := Position( obs, a![3] );
        images[i] := Arrow( gpd, a![1], oims[pt], oims[ph] ); 
    od; 
    mor := GroupoidHomomorphismFromSinglePiece( gpd, gpd, gens, images );
    SetIsInjectiveOnObjects( mor, true ); 
    SetIsSurjectiveOnObjects( mor, true ); 
    L := [1..Length(obs)]; 
    SortParallel( ShallowCopy( oims ), L );  
    SetOrder( mor, Order( PermList( L ) ) ); 
    SetIsGroupoidAutomorphismByObjectPerm( mor, true ); 
    return mor; 
end ); 

InstallMethod( GroupoidAutomorphismByObjectPermNC,  
    "for a hom discrete groupoid and a permutation of objects", true, 
    [ IsHomogeneousDiscreteGroupoid, IsHomogeneousList ], 0,
function( gpd, oims )

    local gpd1, gp, id, homs, mor, L; 

    gpd1 := Pieces( gpd )[1]; 
    gp := gpd1!.magma; 
    id := IdentityMapping( gp ); 
    homs := List( oims, o -> id ); 
    mor := GroupoidHomomorphismFromHomogeneousDiscreteNC
               ( gpd, gpd, homs, oims ); 
    SetIsInjectiveOnObjects( mor, true ); 
    SetIsSurjectiveOnObjects( mor, true ); 
    L := [1..Length(oims)]; 
    SortParallel( ShallowCopy( oims ), L );  
    SetOrder( mor, Order( PermList( L ) ) );
    SetIsGroupoidAutomorphismByObjectPerm( mor, true ); 
    return mor; 
end ); 

InstallMethod( GroupoidAutomorphismByObjectPerm, 
    "for a groupoid and a permutation of objects", true, 
    [ IsGroupoid, IsHomogeneousList ], 0,
function( gpd, oims ) 

    local obs, pos; 

    obs := ObjectList( gpd ); 
    pos := PermList( List( oims, o -> Position( obs, o ) ) ); 
    if ( pos = fail ) then 
        Error( "object images not a permutation of the objects" ); 
    fi; 
    return GroupoidAutomorphismByObjectPermNC( gpd, oims ); 
end ); 

#############################################################################
##
#M  GroupoidAutomorphismByGroupAutoNC  
#M  GroupoidAutomorphismByGroupAuto 
##
InstallMethod( GroupoidAutomorphismByGroupAutoNC , 
    "for a groupoid and an automorphism of the root group", true, 
    [ IsGroupoid and IsSinglePiece, IsGroupHomomorphism and IsBijective ], 0,
function( gpd, hom ) 

    local gens, images, mor; 

    gens := GeneratorsOfGroupoid( gpd ); 
    images := List( gens, 
                    a -> Arrow( gpd, ImageElm( hom, a![1] ), a![2], a![3] ) ); 
    mor := GroupoidHomomorphismFromSinglePiece( gpd, gpd, gens, images ); 
    SetIsInjectiveOnObjects( mor, true ); 
    SetIsSurjectiveOnObjects( mor, true ); 
    SetOrder( mor, Order( hom ) ); 
    SetIsGroupoidAutomorphismByGroupAuto( mor, true ); 
    return mor; 
end ); 

InstallMethod( GroupoidAutomorphismByGroupAuto, 
    "for a groupoid and an automorphism of the root group", true, 
    [ IsGroupoid and IsSinglePiece, IsGroupHomomorphism ], 0,
function( gpd, hom ) 

    local rgp; 

    rgp := gpd!.magma; 
    if not ( (Source(hom) = rgp) and (Range(hom) = rgp) ) then 
        Error( "hom not an endomorphism of the root group" ); 
    fi; 
    if not IsBijective( hom ) then 
        Error( "hom is not an automorphism" ); 
    fi;
    return GroupoidAutomorphismByGroupAutoNC( gpd, hom ); 
end ); 

#############################################################################
##
#M  GroupoidAutomorphismByGroupAutosNC  
#M  GroupoidAutomorphismByGroupAutos 
##
InstallMethod( GroupoidAutomorphismByGroupAutosNC , 
    "for homogeneous discrete groupoid and automorphism list", true, 
    [ IsHomogeneousDiscreteGroupoid, IsHomogeneousList ], 0,
function( gpd, homs ) 

    local obs, orders, m; 

    obs := ObjectList( gpd );
    m := GroupoidHomomorphismFromHomogeneousDiscreteNC( gpd, gpd, homs, obs ); 
    SetIsEndoGeneralMapping( m, true ); 
    SetIsInjectiveOnObjects( m, true ); 
    SetIsSurjectiveOnObjects( m, true ); 
    orders := List( homs, h -> Order( h ) ); 
    SetOrder( m, Lcm( orders ) ); 
    return m; 
end ); 

InstallMethod( GroupoidAutomorphismByGroupAutos, 
    "for homogeneous discrete groupoid and automorphism list", true, 
    [ IsHomogeneousDiscreteGroupoid, IsHomogeneousList ], 0,
function( gpd, homs )  

    local pieces, len, i, p, g, h; 

    pieces := Pieces( gpd ); 
    len := Length( pieces ); 
    if not ( len = Length( homs ) ) then 
        Error( "length of <homs> not equal to number of objects on <gpd>," ); 
    fi; 
    for i in [1..len] do 
        p := pieces[i];  
        g := p!.magma; 
        h := homs[i]; 
        if not ( (Source(h) = g) and (Range(h) = g) ) then 
            Error( "<h> not an endomorphism of group <g> at object <i>," ); 
        fi; 
        if not IsBijective( h ) then 
            Error( "<h> not an automorphism of group <g> at object <i>," ); 
        fi; 
    od; 
    return GroupoidAutomorphismByGroupAutosNC( gpd, homs ); 
end ); 

#############################################################################
##
#M  GroupoidAutomorphismByRayShiftsNC  
#M  GroupoidAutomorphismByRayShifts 
##
InstallMethod( GroupoidAutomorphismByRayShiftsNC , 
    "for a groupoid and a list of elements of the root group", true, 
    [ IsGroupoid and IsSinglePiece, IsHomogeneousList ], 0,
function( gpd, shifts ) 

    local gens, ngens, nobs, images, i, k, a, mor; 

    gens := GeneratorsOfGroupoid( gpd ); 
    ngens := Length( gens ); 
    nobs := Length( gpd!.objects ); 
    images := ShallowCopy( gens ); 
    k := ngens - nobs;
    for i in [2..nobs] do 
        a := gens[i+k]; 
        images[i+k] := Arrow( gpd, a![1]*shifts[i], a![2], a![3] ); 
    od; 
    mor := GroupoidHomomorphismFromSinglePiece( gpd, gpd, gens, images ); 
    SetOrder( mor, Lcm( List( shifts, i -> Order(i) ) ) ); 
    SetIsInjectiveOnObjects( mor, true ); 
    SetIsSurjectiveOnObjects( mor, true ); 
    SetIsGroupoidAutomorphismByRayShifts( mor, true ); 
    return mor;
end ); 

InstallMethod( GroupoidAutomorphismByRayShifts, 
    "for a groupoid and a list of elements of the root group", true, 
    [ IsGroupoid and IsSinglePiece, IsHomogeneousList ], 0,
function( gpd, shifts ) 

    local rgp, rays, nobs, conj; 

    rgp := gpd!.magma; 
    rays := gpd!.rays; 
    nobs := Length( gpd!.objects );
    if not ForAll( shifts, s -> s in rgp ) then  
        Error( "ray shifts not all in the relevant homsets" ); 
    fi; 
    if not ( shifts[1] = One( rgp ) ) then 
        Error( "the first ray shift is not the identity" ); 
    fi; 
    return GroupoidAutomorphismByRayShiftsNC( gpd, shifts ); 
end ); 

#############################################################################
##
#M  GroupoidInnerAutomorphism 
##
InstallMethod( GroupoidInnerAutomorphism, 
    "for a groupoid and an element", true, 
    [ IsGroupoid, IsGroupoidElement ], 0,
function( gpd, e ) 
    Error( "not yet implemented for unions of groupoids" );
end );

InstallMethod( GroupoidInnerAutomorphism, 
    "for a groupoid and an element", true, 
    [ IsGroupoid and IsSinglePieceDomain, IsGroupoidElement ], 0,
function( gpd, e ) 

    local gens, images; 

    Info( InfoGroupoids, 3, "GroupoidInnerAutomorphism from single piece" );  
    gens := GeneratorsOfGroupoid( gpd ); 
    images := List( gens, g -> g^e ); 
    return GroupoidHomomorphism( gpd, gpd, gens, images );
end );

InstallMethod( GroupoidInnerAutomorphism, 
    "for a groupoid and an element", true, 
    [ IsGroupoid and IsDiscreteDomainWithObjects, IsGroupoidElement ], 0,
function( gpd, e ) 

    local obs, nobs, pos, id, auts; 

    Info( InfoGroupoids, 3, "GroupoidInnerAutomorphism from discrete domain" );  
    obs := gpd!.objects; 
    nobs := Length( obs );
    pos := Position( obs, e![2] ); 
    id := IdentityMapping( gpd!.magma );
    auts := List( [1..nobs], i -> id ); 
    auts[pos] := InnerAutomorphism( gpd!.magma, e![1] ); 
    return GroupoidAutomorphismByGroupAutosNC( gpd, auts ); 
end );

InstallOtherMethod( GroupoidInnerAutomorphism, 
    "for a groupoid, a subgroupoid, and an element", true, 
    [ IsGroupoid and IsSinglePieceDomain, IsGroupoid, IsGroupoidElement ], 0,
function( gpd, sub, e ) 

    local gens, images, inn, res; 

    Info( InfoGroupoids, 3, "GroupoidInnerAutomorphism for a subgroupoid" ); 
    if not IsSubgroupoid( gpd, sub ) then 
        Error( "sub is not a subgroupoid of gpd" ); 
    fi; 
    gens := GeneratorsOfGroupoid( gpd ); 
    images := List( gens, g -> g^e ); 
    inn := GroupoidHomomorphism( gpd, gpd, gens, images ); 
    res := RestrictedMappingGroupoids( inn, sub ); 
    return res;
end );

#############################################################################
##
#M  Size( <agpd> ) . . . . . . . . . . . . . . . . . for a connected groupoid
##
InstallMethod( Size, "for a groupoid automorphism group", true,  
    [ IsAutomorphismGroupOfGroupoid ], 0,
function( agpd ) 

    local gpd, gp, n, aut; 

    gpd := AutomorphismDomain( agpd ); 
    gp := gpd!.magma; 
    if IsSinglePieceDomain( gpd ) then  
        n := Length( ObjectList( gpd ) ); 
        aut := AutomorphismGroup( gpd!.magma ); 
        return Factorial( n ) * Size( aut ) * Size( gp )^(n-1); 
    elif IsDiscreteDomainWithObjects( gpd ) 
             and IsHomogeneousDomainWithObjects( gpd ) then 
        n := Length( ObjectList( gpd ) ); 
        aut := AutomorphismGroup( gpd!.magma ); 
        return Factorial( n ) * Size( aut )^n;             
    fi;
    return fail; 
end ); 


#############################################################################
##
#M  NiceObjectAutoGroupGroupoid( <gpd>, <aut> ) . . create a nice monomorphism 
##
InstallMethod( NiceObjectAutoGroupGroupoid, "for a single piece groupoid", 
    true, [ IsGroupoid and IsSinglePieceDomain, IsAutomorphismGroup ], 0,
function( gpd, aut ) 

    local genaut, rgp, genrgp, nrgp, agp, genagp, nagp, iso1, im1, iso2, 
          pagp, iso, obs, n, L, symm, gensymm, nsymm, ngp, genngp, nngp, 
          ninfo, nemb, gens1, i, pi, imi, j, gens2, k, krgp, ikrgp, 
          pasy, esymm, epagp, genpasy, gens12, actgp, ok, action, sdp, 
          sinfo, siso, ssdp, epasy, engp, gennorm, norm, a, c, c1, c2, c12, 
          agens1, agens2, agens3, agens, nat, autgen, eno, gim1, gim2, gim3, 
          niceob, nicemap; 

    genaut := GeneratorsOfGroup( aut );
    ## first: automorphisms of the root group 
    rgp := gpd!.magma; 
    genrgp := GeneratorsOfGroup( rgp );
    nrgp := Length( genrgp ); 
    agp := AutomorphismGroup( rgp ); 
    genagp := SmallGeneratingSet( agp ); 
    nagp := Length( genagp ); 
    iso1 := IsomorphismPermGroup( agp ); 
    im1 := Image( iso1 );
    iso2 := SmallerDegreePermutationRepresentation( im1 );
    pagp := Image( iso2 ); 
    iso := iso1*iso2; 
    agens1 := List( genagp, a -> ImageElm( iso, a ) ); 
    ## second: permutations of the objects 
    obs := gpd!.objects; 
    n := Length( obs );
    if ( n = 1 ) then 
        return pagp; 
    elif ( n = 2 ) then 
        gensymm := [ (1,2) ]; 
    else 
        L := [2..n]; 
        Append( L, [1] ); 
        gensymm := [ PermList(L), (1,2) ];  #? replace (1,2) with (n-1,n) ?? 
    fi;
    symm := Group( gensymm ); 
    nsymm := Length( gensymm ); 
    ## third: ray products 
    ngp := DirectProduct( ListWithIdenticalEntries( n, rgp ) ); 
    genngp := GeneratorsOfGroup( ngp ); 
    nngp := Length( genngp );
    ninfo := DirectProductInfo( ngp ); 
    ## force construction of the n embeddings 
    for i in [1..n] do 
        k := Embedding( ngp, i ); 
    od; 
    nemb := ninfo!.embeddings; 
    # action of agp on ngp 
    gens1 := [1..nagp]; 
    for i in [1..nagp] do 
        k := genagp[i]; 
        krgp := List( genrgp, r -> ImageElm( k, r ) ); 
        ikrgp := [ ]; 
        for j in [1..n] do 
            Append( ikrgp, List( krgp, x -> ImageElm( nemb[j], x ) ) ); 
        od; 
        gens1[i] := GroupHomomorphismByImages( ngp, ngp, genngp, ikrgp ); 
    od; 
    ## action of symm on ngp = rgp^n 
    gens2 := [1..nsymm]; 
    for i in [1..nsymm] do 
        pi := gensymm[i]^-1; 
        gens2[i] := GroupHomomorphismByImages( ngp, ngp, genngp,  
            List( [1..nngp], j -> genngp[ RemInt( j-1, nrgp ) + 1  
            + nrgp * (( QuoInt( j-1, nrgp ) + 1 )^pi - 1 ) ] ) ); 
    od; 
    pasy := DirectProduct( pagp, symm ); 
    epagp := Embedding( pasy, 1 ); 
    agens1 := List( agens1, g -> ImageElm( epagp, g ) ); 
    esymm := Embedding( pasy, 2 ); 
    agens2 := List( gensymm, g -> ImageElm( esymm, g ) ); 
    ## genpasy := GeneratorsOfGroup( pasy ); 
    genpasy := Concatenation( agens1, agens2 );
    #  construct the semidirect product 
    gens12 := Concatenation( gens1, gens2 ); 
    actgp := Group( gens12 ); 
    ok := IsGroupOfAutomorphisms( actgp ); 
    action := GroupHomomorphismByImages( pasy, actgp, genpasy, gens12 ); 
    sdp := SemidirectProduct( pasy, action, ngp ); 
    Info( InfoGroupoids, 2, "sdp has ", 
          Length(GeneratorsOfGroup(sdp)), " gens." ); 
    sinfo := SemidirectProductInfo( sdp ); 
    siso := SmallerDegreePermutationRepresentation( sdp ); 
    ssdp := Image( siso ); 
    epasy := Embedding( sdp, 1 ) * siso; 
    agens1 := List( agens1, g -> ImageElm( epasy, g ) );  
    agens2 := List( agens2, g -> ImageElm( epasy, g ) ); 
    engp := Embedding( sdp, 2 ) * siso; 
    #  why [3..nngp]?  no doubt because Sn has two generators? 
    agens3 := List( genngp{[3..nngp]}, g -> ImageElm( engp, g ) ); 
    #  construct the normal subgroup 
    gennorm := [1..nrgp ]; 
    for i in [1..nrgp] do 
        c := genrgp[i]; 
        a := GroupHomomorphismByImages( rgp, rgp, genrgp, List(genrgp,x->x^c) );
        c1 := ImageElm( epasy, ImageElm( epagp, ImageElm( iso, a ) ) );
        c := c^-1; 
        c2 := ImageElm( engp, 
            Product( List( [1..n], j->ImageElm( nemb[j], c ) ) ) ); 
        gennorm[i] := c1 * c2; 
    od; 
    norm := Subgroup( ssdp, gennorm ); 
    ok := IsNormal( ssdp, norm ); 
    if not ok then 
        Error( "norm should be a normal subgroup of ssdp" ); 
    fi; 
    nat := NaturalHomomorphismByNormalSubgroup( ssdp, norm ); 
    niceob := Image( nat ); 
    eno := ListWithIdenticalEntries( n+2, 0 );
    eno[1] := iso * epagp * epasy * nat;   ## agp->pagp->pasy->ssdp->niceob
    eno[2] := esymm * epasy * nat;         ##      symm->pasy->ssdp->niceob
    for i in [1..n] do 
        gim1 := List( genrgp, g -> ImageElm( nemb[i], g ) ); 
        gim2 := List( gim1, g -> ImageElm( engp, g ) ); 
        gim3 := List( gim2, g -> ImageElm( nat, g ) ); 
        eno[i+2] := GroupHomomorphismByImages( rgp, niceob, genrgp, gim3 ); 
    od; 
    autgen := Concatenation( agens1, agens2, agens3 ); 
    agens := List( autgen, g -> ImageElm( nat, g ) ); 
    return [ niceob, eno ]; 
end );

InstallMethod( NiceObjectAutoGroupGroupoid, "for a hom discrete groupoid", 
    true, [ IsHomogeneousDiscreteGroupoid, IsAutomorphismGroup ], 0,
function( gpd, aut ) 

    local pieces, obs, m, p1, g1, geng1, ng1, ag1, genag1, nag, 
          iso1, ag2, genag2, mag2, genmag2, nmag2, minfo, i, k, memb, 
          K, L, symm, gensymm, imact, pi, actgp, ok, action, sdp, sinfo, 
          siso, ssdp, esymm, egensymm, emag2, egenmag2, agens, 
          im1, im2, im3, eno;  

    pieces := Pieces( gpd ); 
    obs := ObjectList( gpd ); 
    m := Length( obs );
    ## first: automorphisms of the object groups 
    p1 := pieces[1]; 
    g1 := p1!.magma; 
    geng1 := GeneratorsOfGroup( g1 );
    ng1 := Length( geng1 ); 
    ag1 := AutomorphismGroup( g1 ); 
    genag1 := SmallGeneratingSet( ag1 ); 
    nag := Length( genag1 ); 
    iso1 := IsomorphismPermGroup( ag1 ); 
    genag2 := List( genag1, g -> ImageElm( iso1, g ) ); 
    ag2 := Group( genag2 ); 
    mag2 := DirectProduct( ListWithIdenticalEntries( m, ag2 ) ); 
    genmag2 := GeneratorsOfGroup( mag2 ); 
    nmag2 := Length( genmag2 );
    minfo := DirectProductInfo( mag2 ); 
    ## force construction of the m embeddings 
    for i in [1..m] do 
        k := Embedding( mag2, i ); 
    od; 
    memb := minfo!.embeddings; 
    ## second: permutations of the objects 
    if ( m = 1 ) then 
        Error( "only one object, so no permutations" ); 
    elif ( m = 2 ) then 
        K := [1]; 
        gensymm := [ (1,2) ]; 
    else 
        K := [1,2]; 
        L := [2..m]; 
        Append( L, [1] ); 
        gensymm := [ PermList(L), (1,2) ];  #? replace (1,2) with (m-1,m) ?? 
    fi;
    symm := Group( gensymm ); 
    ## action of symm on mag2 = ag2^m 
    imact := ShallowCopy( K ); 
    for i in K do 
        pi := gensymm[i]^-1; 
        imact[i] := GroupHomomorphismByImages( mag2, mag2, genmag2,  
            List( [1..nmag2], j -> genmag2[ RemInt( j-1, nag ) + 1  
            + nag * (( QuoInt( j-1, nag ) + 1 )^pi - 1 ) ] ) ); 
    od; 
    #  construct the semidirect product: symm acting on mag2
    actgp := Group( imact ); 
    ok := IsGroupOfAutomorphisms( actgp ); 
    action := GroupHomomorphismByImages( symm, actgp, gensymm, imact ); 
    sdp := SemidirectProduct( symm, action, mag2 ); 
    Info( InfoGroupoids, 2, "sdp has ", 
          Length(GeneratorsOfGroup(sdp)), " gens." ); 
    sinfo := SemidirectProductInfo( sdp ); 
    ## (13/04/16) comment this out for the moment and replace with identity 
    ## siso := SmallerDegreePermutationRepresentation( sdp ); 
    siso := IdentityMapping( sdp ); 
    ssdp := Image( siso ); 
    esymm := Embedding( sdp, 1 ) * siso; 
    egensymm := List( gensymm, g -> ImageElm( esymm, g ) );  
    emag2 := Embedding( sdp, 2 ) * siso; 
    egenmag2 := List( genmag2{[1..nag]}, g -> ImageElm( emag2, g ) ); 
    eno := ListWithIdenticalEntries( m+1, 0 );
    eno[1] := esymm; 
    for i in [1..m] do 
        im1 := List( genag1, g -> ImageElm( iso1, g ) ); 
        im2 := List( im1, g -> ImageElm( memb[i], g ) ); 
        im3 := List( im2, g -> ImageElm( emag2, g ) ); 
        eno[i+1] := GroupHomomorphismByImages( ag1, ssdp, genag1, im3 ); 
    od; 
    agens := Concatenation( egensymm, egenmag2 ); 
    return [ ssdp, agens, eno ]; 
end );

#############################################################################
##
#M  AutomorphismGroupOfGroupoid( <gpd> )
##
InstallMethod( AutomorphismGroupOfGroupoid, "for one-object groupoid", true, 
    [ IsGroupoid and IsSinglePieceDomain and IsDiscreteDomainWithObjects ], 0,
function( gpd ) 

    local autgen, rgp, agp, genagp, a, a0, ok, id, aut;  

    Info( InfoGroupoids, 2, 
          "AutomorphismGroupOfGroupoid for one-object groupoids" ); 
    autgen := [ ]; 
    rgp := gpd!.magma; 
    agp := AutomorphismGroup( rgp ); 
    genagp := SmallGeneratingSet( agp ); 
    for a in genagp do  
        a0 := GroupoidAutomorphismByGroupAutoNC( gpd, a ); 
        ok := IsAutomorphismWithObjects( a0 ); 
        Add( autgen, a0 );  
    od; 
    id := IdentityMapping( gpd ); 
    aut := GroupWithGenerators( autgen, id ); 
    SetIsAutomorphismGroup( aut, true ); 
    SetIsFinite( aut, true ); 
    SetIsAutomorphismGroupOfGroupoid( aut, true ); 
    SetAutomorphismGroup( gpd, aut );
    Info( InfoGroupoids, 2, "nice object not yet coded in this case" ); 
    return aut; 
end ); 

InstallMethod( AutomorphismGroupOfGroupoid, "for a single piece groupoid", 
    true, [ IsGroupoid and IsSinglePieceDomain ], 0,
function( gpd ) 

    local autgen, nautgen, rgp, genrgp, agp, genagp, a, obs, n, imobs, 
          L, ok, id, ids, cids, i, c, aut, niceob, nicemap, rgh, ioo, ior;  

    Info( InfoGroupoids, 2, 
          "AutomorphismGroupOfGroupoid for single piece groupoids" ); 
    ##  first: automorphisms of the root group 
    autgen := [ ]; 
    rgp := gpd!.magma; 
    genrgp := GeneratorsOfGroup( rgp ); 
    agp := AutomorphismGroup( rgp ); 
    genagp := SmallGeneratingSet( agp ); 
    for a in genagp do  
        Add( autgen, GroupoidAutomorphismByGroupAutoNC( gpd, a ) );  
    od; 
    ##  second: permutations of the objects 
    obs := gpd!.objects; 
    n := Length( obs );
    if ( n = 2 ) then 
        imobs := [ obs[2], obs[1] ]; 
        Add( autgen, GroupoidAutomorphismByObjectPerm( gpd, imobs ) );
    else 
        L := [2..n]; 
        Append( L, [1] ); 
        imobs := List( L, i -> obs[i] ); 
        Add( autgen, GroupoidAutomorphismByObjectPerm( gpd, imobs ) );
        imobs := ShallowCopy( obs ); 
        imobs[1] := obs[2]; 
        imobs[2] := obs[1]; 
        Add( autgen, GroupoidAutomorphismByObjectPerm( gpd, imobs ) );
    fi; 
    ##  third: add in the ray prods 
    ids := List( obs, o -> One( rgp ) ); 
    for i in [2..n] do 
        for c in genrgp do 
            cids := ShallowCopy( ids ); 
            cids[i] := c; 
            Add( autgen, GroupoidAutomorphismByRayShifts( gpd, cids ) ); 
        od; 
    od; 
    nautgen := Length( autgen ); 
    ##  generating set for the automorphism group now complete 
    for a in autgen do 
        ok := IsAutomorphismWithObjects( a ); 
    od; 
    id := IdentityMapping( gpd ); 
    ## imobs := L[0]; 
    aut := GroupWithGenerators( autgen, id ); 
    SetIsAutomorphismGroup( aut, true ); 
    SetIsFinite( aut, true ); 
    SetIsAutomorphismGroupOfGroupoid( aut, true );
    niceob := NiceObjectAutoGroupGroupoid( gpd, aut ); 
    SetNiceObject( aut, niceob[1] ); 
    SetEmbeddingsInNiceObject( aut, niceob[2] ); 
    #?  SetInnerAutomorphismsAutomorphismGroup( aut, ?? ); 
    SetAutomorphismGroup( gpd, aut );

    ## now construct nicemap using GroupHomomorphismByFunction 
    nicemap := GroupHomomorphismByFunction( aut, niceob[1], 
        function( alpha ) 
            local G, autG, q, eno, obG, nobG, oha, j, ioa, Lpos, Lmap; 
            G := Source( alpha ); 
            autG := AutomorphismGroup( G ); 
            q := One( NiceObject( autG ) );
            eno := EmbeddingsInNiceObject( autG ); 
            obG := ObjectList( G ); 
            nobG := Length( obG ); 
            rgh := RootGroupHomomorphism( alpha ); 
            q := ImageElm( eno[1], rgh ); 
            ior := ImageElementsOfRays( alpha ); 
            for j in [1..nobG] do 
                q := q * ImageElm( eno[j+2], ior[j] ); 
            od;
            ioo := ImagesOfObjects( alpha ); 
            Lpos := List( ioo, j -> Position( obG, j ) ); 
            Lmap := MappingPermListList( [1..nobG], Lpos ); 
            q := q * ImageElm( eno[2], Lmap ); 
            return q;             
        end); 

    SetNiceMonomorphism( aut, nicemap ); 
    ## SetIsHandledByNiceMonomorphism( aut, true ); 
    SetIsCommutative( aut, IsCommutative( niceob[1] ) );
    return aut; 
end ); 

InstallMethod( AutomorphismGroupOfGroupoid, "for a hom. discrete groupoid", 
    true, [ IsHomogeneousDiscreteGroupoid ], 0,
function( gpd ) 

    local pieces, autgen, nautgen, p1, g1, ag1, id1, genag1, a, b, auts, 
          obs, n, imobs, L, ok, id, ids, aut, niceob, nicemap;  

    Info( InfoGroupoids, 2, 
          "AutomorphismGroupOfGroupoid for homogeneous discrete groupoids" ); 
    pieces := Pieces( gpd ); 
    obs := ObjectList( gpd ); 
    n := Length( obs );
    ##  first: automorphisms of the first object group 
    autgen := [ ]; 
    p1 := pieces[1]; 
    g1 := p1!.magma; 
    ag1 := AutomorphismGroup( g1 ); 
    id1 := IdentityMapping( g1 ); 
    auts := ListWithIdenticalEntries( n, id1 ); 
    genag1 := SmallGeneratingSet( ag1 ); 
    for a in genag1 do 
        auts[1] := a; 
        b := GroupoidAutomorphismByGroupAutos( gpd, auts ); 
        SetIsGroupWithObjectsHomomorphism( b, true ); 
        Add( autgen, b );  
    od; 
    ##  second: permutations of the objects 
    if ( n = 2 ) then 
        imobs := [ obs[2], obs[1] ]; 
        Add( autgen, GroupoidAutomorphismByObjectPerm( gpd, imobs ) );
    else 
        L := [2..n]; 
        Append( L, [1] ); 
        imobs := List( L, i -> obs[i] ); 
        Add( autgen, GroupoidAutomorphismByObjectPerm( gpd, imobs ) );
        imobs := ShallowCopy( obs ); 
        imobs[1] := obs[2]; 
        imobs[2] := obs[1]; 
        Add( autgen, GroupoidAutomorphismByObjectPerm( gpd, imobs ) );
    fi; 
    nautgen := Length( autgen ); 
    ##  generating set for the automorphism group now complete 
    for a in autgen do 
        ok := IsAutomorphismWithObjects( a ); 
    od; 
    id := IdentityMapping( gpd ); 
    ## imobs := L[0];
    aut := GroupWithGenerators( autgen, id ); 
    SetIsAutomorphismGroup( aut, true ); 
    SetIsFinite( aut, true ); 
    SetIsAutomorphismGroupOfGroupoid( aut, true );
    niceob := NiceObjectAutoGroupGroupoid( gpd, aut ); 
    SetNiceObject( aut, niceob[1] ); 
    SetEmbeddingsInNiceObject( aut, niceob[3] ); 
    SetAutomorphismGroup( gpd, aut ); 

    ## now construct nicemap using GroupHomomorphismByFunction 
    nicemap := GroupHomomorphismByFunction( aut, niceob[1], 
        function( alpha ) 
            local G, autG, q, eno, obG, nobG, oha, j, ioa, Lpos, Lmap; 
            G := Source( alpha ); 
            autG := AutomorphismGroup( G ); 
            q := One( NiceObject( autG ) );
            eno := EmbeddingsInNiceObject( autG ); 
            obG := ObjectList( G ); 
            nobG := Length( obG ); 
            oha := ObjectHomomorphisms( alpha ); 
            for j in [1..nobG] do 
                q := q * ImageElm( eno[j+1], oha[j] ); 
            od;
            ioa := ImagesOfObjects( alpha ); 
            Lpos := List( ioa, j -> Position( obG, j ) ); 
            Lmap := MappingPermListList( Lpos, [1..nobG] ); 
            q := q * ImageElm( eno[1], Lmap ); 
            return q;             
        end); 

    SetNiceMonomorphism( aut, nicemap ); 
    ## SetIsHandledByNiceMonomorphism( aut, true ); 
    #?  SetInnerAutomorphismsAutomorphismGroup( aut, ?? );  
    SetIsCommutative( aut, IsCommutative( niceob[1] ) );
    return aut; 
end ); 

## ========================================================================
##                     Homogeneous groupoid automorphisms                  
## ======================================================================== ##

InstallMethod( \in,
    "method for an automorphism of a single piece groupoid", true,
    [ IsGroupWithObjectsHomomorphism, IsAutomorphismGroupOfGroupoidAsGroupoid ], 
    0,
function( arr, aut0 )

    local o;

    o := ObjectList( aut0 )[1]; 
    if not ( ( o = arr![2] ) and ( o = arr![3] ) ) then 
        return false; 
    fi; 
    return ( arr![1] in aut0!.magma ); 
end ); 

InstallMethod( \in,
    "method for an automorphism of a single piece groupoid", true,
    [ IsGroupWithObjectsHomomorphism, IsAutomorphismGroupOfGroupoid ], 
    0,
function( a, aut )

    local gpd, gp, data;

    gpd := AutomorphismDomain( aut ); 
    gp := gpd!.magma; 
    data := MappingToSinglePieceData( a )[1];
    if not ( data[1] in AutomorphismGroup( gp ) ) then 
        return false; 
    fi; 
    if not ForAll( data[2], o -> o in ObjectList( gpd ) ) then 
        return false; 
    fi; 
    if not ForAll( data[3], e -> e in gp ) then 
        return false; 
    fi;
    return true; 
end ); 

InstallMethod( \in,
    "for groupoid hom and automorphism group : discrete", true, 
    [ IsGroupoidHomomorphismFromHomogeneousDiscrete and IsGroupoidHomomorphism,
      IsAutomorphismGroupOfGroupoid ], 0, 
function( a, aut ) 

    local gens, g1, gpd, obs, imobs, G, AG; 

    gens := GeneratorsOfGroup( aut ); 
    gpd := AutomorphismDomain( aut ); 
    if not ( ( Source(a) = gpd ) and ( Range(a) = gpd ) ) then 
        Info( InfoGroupoids, 2, "require Source(a) = Range(a) = gpd" ); 
        return false; 
    fi; 
    obs := ObjectList( gpd );
    imobs := ShallowCopy( ImagesOfObjects( a ) ); 
    Sort( imobs ); 
    if not ( obs = imobs ) then 
        Info( InfoGroupoids, 2, "incorrect images of objects" ); 
        return false;  
    fi; 
    G := Source(a)!.magma; 
    AG := AutomorphismGroup( G ); 
    if not ForAll( ObjectHomomorphisms(a), h -> h in AG ) then 
        Info( InfoGroupoids, 2, "object homomorphisms incorrect" ); 
        return false; 
    fi; 
    #? is there anything else to test? 
    return true; 
end ); 
    
##############################################################################
##
#E  gpdaut.gi  . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
##
