"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const util = require("util");
const semver = require("semver");
const json_1 = require("../utils/json");
const configuration_1 = require("../configuration");
const transport_1 = require("../utils/transport");
const debug_1 = require("debug");
const debug = debug_1.default('axm:notify');
class NotifyOptions {
}
exports.NotifyOptions = NotifyOptions;
exports.NotifyOptionsDefault = {
    level: 'info',
    catchExceptions: true
};
class NotifyFeature {
    constructor() {
        this.options = exports.NotifyOptionsDefault;
        this.levels = ['fatal', 'error', 'warn', 'info', 'debug', 'trace'];
    }
    init(options) {
        if (options) {
            this.options = options;
        }
        configuration_1.default.configureModule({
            error: true
        });
        if (process.env.CATCH_CONTEXT_ON_ERROR === 'true' && (semver.satisfies(process.version, '< 8.0.0') ||
            (semver.satisfies(process.version, '< 10.0.0') && !process.env.FORCE_INSPECTOR))) {
            debug(`Inspector is not available on node version ${process.version} !`);
        }
        if (process.env.CATCH_CONTEXT_ON_ERROR === 'true' && semver.satisfies(process.version, '>= 10.0.0') ||
            (semver.satisfies(process.version, '>= 8.0.0') && process.env.FORCE_INSPECTOR)) {
            debug('Enabling inspector based error reporting');
            const NotifyInspector = require('./notifyInspector').default;
            this.feature = new NotifyInspector();
            this.feature.init(options);
        }
        else {
            this.catchAll();
        }
        return {
            notifyError: this.notifyError
        };
    }
    destroy() {
        if (this.feature) {
            this.feature.destroy();
        }
    }
    notifyError(err, level) {
        if (!(err instanceof Error)) {
            console.trace('[PM2-IO-APM] You should use notify with an Error object');
            return -1;
        }
        if (!level || this.levels.indexOf(level) === -1) {
            return transport_1.default.send({
                type: 'process:exception',
                data: json_1.default.jsonize(err)
            });
        }
        if (this.levels.indexOf(this.options.level) >= this.levels.indexOf(level)) {
            return transport_1.default.send({
                type: 'process:exception',
                data: json_1.default.jsonize(err)
            });
        }
        return null;
    }
    catchAll(opts) {
        if (opts === undefined) {
            opts = { errors: true };
        }
        if (process.env.exec_mode === 'cluster_mode') {
            return false;
        }
        const self = this;
        function getUncaughtExceptionListener(listener) {
            return function uncaughtListener(err) {
                let error = err && err.stack ? err.stack : err;
                if (listener === 'unhandledRejection') {
                    console.log('You have triggered an unhandledRejection, you may have forgotten to catch a Promise rejection:');
                }
                console.error(error);
                let errObj;
                if (err) {
                    errObj = self._interpretError(err);
                }
                transport_1.default.send({
                    type: 'process:exception',
                    data: errObj !== undefined ? errObj : { message: 'No error but ' + listener + ' was caught!' }
                });
                if (!process.listeners(listener).filter(function (listener) {
                    return listener !== uncaughtListener;
                }).length) {
                    if (listener === 'uncaughtException') {
                        process.exit(1);
                    }
                }
            };
        }
        if (opts.errors === true && util.inspect(process.listeners('uncaughtException')).length === 2) {
            process.once('uncaughtException', getUncaughtExceptionListener('uncaughtException'));
            process.once('unhandledRejection', getUncaughtExceptionListener('unhandledRejection'));
        }
        else if (opts.errors === false
            && util.inspect(process.listeners('uncaughtException')).length !== 2) {
            process.removeAllListeners('uncaughtException');
            process.removeAllListeners('unhandledRejection');
        }
    }
    expressErrorHandler() {
        configuration_1.default.configureModule({
            error: true
        });
        return function errorHandler(err, req, res, next) {
            if (res.statusCode < 400)
                res.statusCode = 500;
            err.url = req.url;
            err.component = req.url;
            err.action = req.method;
            err.params = req.body;
            err.session = req.session;
            transport_1.default.send({
                type: 'process:exception',
                data: json_1.default.jsonize(err)
            });
            return next(err);
        };
    }
    _interpretError(err) {
        let sErr = {
            message: null,
            stack: null
        };
        if (err instanceof Error) {
            // Error object type processing
            sErr = err;
        }
        else {
            // JSON processing
            sErr.message = err;
            sErr.stack = err;
        }
        return json_1.default.jsonize(sErr);
    }
}
exports.NotifyFeature = NotifyFeature;
//# sourceMappingURL=data:application/json;base64,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