/*
 *  Phusion Passenger - https://www.phusionpassenger.com/
 *  Copyright (c) 2014-2017 Phusion Holding B.V.
 *
 *  "Passenger", "Phusion Passenger" and "Union Station" are registered
 *  trademarks of Phusion Holding B.V.
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to deal
 *  in the Software without restriction, including without limitation the rights
 *  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 *  THE SOFTWARE.
 */
#ifndef _PASSENGER_SPAWNING_KIT_RESULT_H_
#define _PASSENGER_SPAWNING_KIT_RESULT_H_

#include <string>
#include <vector>

#include <sys/types.h>

#include <FileDescriptor.h>
#include <Exceptions.h>
#include <Utils/SystemTime.h>
#include <ConfigKit/ConfigKit.h>
#include <Core/SpawningKit/Context.h>
#include <Core/SpawningKit/Config.h>

namespace Passenger {
namespace SpawningKit {

using namespace std;


/**
 * Represents the result of a spawning operation.
 *
 * - begin hinted parseable class -
 */
class Result {
public:
	struct Socket {
		struct Schema: public ConfigKit::Schema {
			Schema() {
				using namespace Passenger::ConfigKit;

				add("address", STRING_TYPE, REQUIRED);
				add("protocol", STRING_TYPE, REQUIRED);
				add("description", STRING_TYPE, OPTIONAL);
				add("concurrency", INT_TYPE, OPTIONAL, -1);
				add("accept_http_requests", BOOL_TYPE, OPTIONAL, false);

				finalize();
			}
		};

		string address;
		string protocol;
		string description;
		/**
		 * Special values:
		 * 0 = unlimited concurrency
		 * < 0 = unknown
		 */
		int concurrency;
		bool acceptHttpRequests;

		Socket()
			: concurrency(-1),
			  acceptHttpRequests(false)
			{ }

		Socket(const Schema &schema, const Json::Value &values) {
			ConfigKit::Store store(schema);
			vector<ConfigKit::Error> errors;

			if (!store.update(values, errors)) {
				throw ArgumentException("Invalid initial values: "
					+ toString(errors));
			}

			address = store["address"].asString();
			protocol = store["protocol"].asString();
			if (!store["description"].isNull()) {
				description = store["description"].asString();
			}
			concurrency = store["concurrency"].asInt();
			acceptHttpRequests = store["accept_http_requests"].asBool();
		}

		Json::Value inspectAsJson() const {
			Json::Value doc;
			doc["address"] = address;
			doc["protocol"] = protocol;
			if (!description.empty()) {
				doc["description"] = description;
			}
			doc["concurrency"] = concurrency;
			doc["accept_http_requests"] = acceptHttpRequests;
			return doc;
		}
	};

private:
	void validate_autoGeneratedCode(vector<StaticString> &internalFieldErrors,
		vector<StaticString> &appSuppliedFieldErrors) const;

public:
	/****** Fields supplied by HandshakePrepare and HandshakePerform ******/

	/**
	 * @hinted_parseable
	 * @require result.pid != -1
	 */
	pid_t pid;

	/**
	 * If true, then indicates that this Process does not refer to a real OS
	 * process. The sockets in the socket list are fake and need not be deleted.
	 * Set to true by DummySpawner, used during unit tests.
	 *
	 * @hinted_parseable
	 */
	bool dummy;

	/**
	 * @hinted_parseable
	 * @require_non_empty
	 */
	string gupid;

	/**
	 * @hinted_parseable
	 */
	string codeRevision;

	/**
	 * @hinted_parseable
	 */
	FileDescriptor stdinFd;

	/**
	 * @hinted_parseable
	 */
	FileDescriptor stdoutAndErrFd;

	/**
	 * @hinted_parseable
	 * @require result.spawnStartTime != 0
	 */
	unsigned long long spawnStartTime;

	/**
	 * @hinted_parseable
	 * @require result.spawnEndTime != 0
	 */
	unsigned long long spawnEndTime;

	/**
	 * @hinted_parseable
	 * @require result.spawnStartTimeMonotonic != 0
	 */
	MonotonicTimeUsec spawnStartTimeMonotonic;

	/**
	 * @hinted_parseable
	 * @require result.spawnEndTimeMonotonic != 0
	 */
	MonotonicTimeUsec spawnEndTimeMonotonic;


	/****** Fields supplied by the app ******/

	vector<Socket> sockets;


	Result()
		: pid(-1),
		  dummy(false),
		  spawnStartTime(0),
		  spawnEndTime(0),
		  spawnStartTimeMonotonic(0),
		  spawnEndTimeMonotonic(0)
		{ }

	void initialize(const Context &context, const Config * const config) {
		gupid = integerToHex(SystemTime::get() / 60) + "-" +
			context.randomGenerator->generateAsciiString(10);
		spawnStartTime = SystemTime::getUsec();
		spawnStartTimeMonotonic = SystemTime::getMonotonicUsec();
	}

	bool validate(vector<StaticString> &internalFieldErrors,
		vector<StaticString> &appSuppliedFieldErrors) const
	{
		validate_autoGeneratedCode(internalFieldErrors, appSuppliedFieldErrors);

		if (sockets.empty()) {
			appSuppliedFieldErrors.push_back(P_STATIC_STRING("sockets are not supplied"));
		}

		return internalFieldErrors.empty() && appSuppliedFieldErrors.empty();
	}
};
// - end hinted parseable class -


} // namespace SpawningKit
} // namespace Passenger

#include <Core/SpawningKit/Result/AutoGeneratedCode.h>

#endif /* _PASSENGER_SPAWNING_KIT_RESULT_H_ */
