/*
 * Copyright 2018 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.internal.fingerprint;

import org.gradle.api.internal.changedetection.state.mirror.FileSystemSnapshot;
import org.gradle.api.internal.changedetection.state.mirror.PhysicalSnapshot;
import org.gradle.internal.fingerprint.impl.EmptyCurrentFileCollectionFingerprint;
import org.gradle.internal.fingerprint.impl.FingerprintCompareStrategy;
import org.gradle.internal.scan.UsedByScanPlugin;

import java.util.Map;

/**
 * Strategy for converting a sequence of {@link PhysicalSnapshot}s into a {@link FileCollectionFingerprint}.
 */
public interface FingerprintingStrategy {

    /**
     * Converts the roots into the {@link NormalizedFileSnapshot}s used by the {@link FileCollectionFingerprint}.
     */
    Map<String, NormalizedFileSnapshot> collectSnapshots(Iterable<FileSystemSnapshot> roots);

    /**
     * Used by the {@link FileCollectionFingerprint} to compare/hash two maps of snapshots generated by {@link #collectSnapshots(Iterable)}
     */
    FingerprintCompareStrategy getCompareStrategy();

    Identifier getIdentifier();

    @UsedByScanPlugin("not linked directly - name are expected as part of org.gradle.api.internal.tasks.SnapshotTaskInputsBuildOperationType.Result.VisitState.getPropertyNormalizationStrategyName")
    enum Identifier {
        IGNORED_PATH,
        NAME_ONLY,
        RELATIVE_PATH,
        ABSOLUTE_PATH,
        COMPILE_CLASSPATH,
        CLASSPATH;

        private final EmptyCurrentFileCollectionFingerprint emptyFingerprint;

        Identifier() {
            emptyFingerprint = new EmptyCurrentFileCollectionFingerprint(this);
        }

        public EmptyCurrentFileCollectionFingerprint getEmptyFingerprint() {
            return emptyFingerprint;
        }
    }
}
