##########################################################################
#
# pgAdmin 4 - PostgreSQL Tools
#
# Copyright (C) 2013 - 2018, The pgAdmin Development Team
# This software is released under the PostgreSQL Licence
#
##########################################################################

from __future__ import print_function
import sys
import random

from selenium.webdriver import ActionChains
from selenium.webdriver.common.keys import Keys
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
from regression.python_test_utils import test_utils
from regression.feature_utils.base_feature_test import BaseFeatureTest


class QueryToolAutoCompleteFeatureTest(BaseFeatureTest):
    """
        This feature test will test the query tool auto complete feature.
    """

    first_table_name = ""
    second_table_name = ""

    scenarios = [
        ("Query tool auto complete feature test", dict())
    ]

    def before(self):
        self.page.wait_for_spinner_to_disappear()

        self.page.add_server(self.server)
        self.first_table_name = "auto_comp_" + \
                                str(random.randint(1000, 3000))
        test_utils.create_table(self.server, self.test_db,
                                self.first_table_name)

        self.second_table_name = "auto_comp_" + \
                                 str(random.randint(1000, 3000))
        test_utils.create_table(self.server, self.test_db,
                                self.second_table_name)

        self._locate_database_tree_node()
        self.page.open_query_tool()
        self.page.wait_for_spinner_to_disappear()

    def runTest(self):
        # Test case for keywords
        print("\nAuto complete ALTER keyword... ", file=sys.stderr, end="")
        self._auto_complete("A", "ALTER")
        print("OK.", file=sys.stderr)
        self._clear_query_tool()

        print("Auto complete BEGIN keyword... ", file=sys.stderr, end="")
        self._auto_complete("BE", "BEGIN")
        print("OK.", file=sys.stderr)
        self._clear_query_tool()

        print("Auto complete CASCADED keyword... ", file=sys.stderr, end="")
        self._auto_complete("CAS", "CASCADED")
        print("OK.", file=sys.stderr)
        self._clear_query_tool()

        print("Auto complete SELECT keyword... ", file=sys.stderr, end="")
        self._auto_complete("SE", "SELECT")
        print("OK.", file=sys.stderr)
        self._clear_query_tool()

        print("Auto complete pg_backend_pid() function ... ",
              file=sys.stderr, end="")
        self._auto_complete("SELECT pg_", "pg_backend_pid()")
        print("OK.", file=sys.stderr)
        self._clear_query_tool()

        print("Auto complete current_query() function ... ",
              file=sys.stderr, end="")
        self._auto_complete("SELECT current_", "current_query()")
        print("OK.", file=sys.stderr)
        self._clear_query_tool()

        print("Auto complete function with argument ... ",
              file=sys.stderr, end="")
        self._auto_complete("SELECT pg_st", "pg_stat_file(filename)")
        print("OK.", file=sys.stderr)
        self._clear_query_tool()

        print("Auto complete first table in public schema ... ",
              file=sys.stderr, end="")
        self._auto_complete("SELECT * FROM public.", self.first_table_name)
        print("OK.", file=sys.stderr)
        self._clear_query_tool()

        print("Auto complete second table in public schema ... ",
              file=sys.stderr, end="")
        self._auto_complete("SELECT * FROM public.", self.second_table_name)
        print("OK.", file=sys.stderr)
        self._clear_query_tool()

        print("Auto complete JOIN second table with after schema name ... ",
              file=sys.stderr, end="")
        query = "SELECT * FROM public." + self.first_table_name + \
                " JOIN public."
        self._auto_complete(query, self.second_table_name)
        print("OK.", file=sys.stderr)
        self._clear_query_tool()

        print("Auto complete JOIN ON some columns ... ",
              file=sys.stderr, end="")
        query = "SELECT * FROM public." + self.first_table_name + \
                " JOIN public." + self.second_table_name + " ON " + \
                self.second_table_name + "."
        expected_string = "some_column = " + self.first_table_name + \
                          ".some_column"
        self._auto_complete(query, expected_string)
        print("OK.", file=sys.stderr)
        self._clear_query_tool()

        print("Auto complete JOIN ON some columns using tabel alias ... ",
              file=sys.stderr, end="")
        query = "SELECT * FROM public." + self.first_table_name + \
                " t1 JOIN public." + self.second_table_name + " t2 ON t2."
        self._auto_complete(query, "some_column = t1.some_column")
        print("OK.", file=sys.stderr)
        self._clear_query_tool()

    def after(self):
        self.page.remove_server(self.server)

    def _locate_database_tree_node(self):
        self.page.toggle_open_tree_item(self.server['name'])
        self.page.toggle_open_tree_item('Databases')
        self.page.toggle_open_tree_item(self.test_db)

    def _clear_query_tool(self):
        self.page.click_element(
            self.page.find_by_xpath("//*[@id='btn-clear-dropdown']")
        )
        ActionChains(self.driver) \
            .move_to_element(self.page.find_by_xpath("//*[@id='btn-clear']")) \
            .perform()
        self.page.click_element(
            self.page.find_by_xpath("//*[@id='btn-clear']")
        )
        self.page.click_modal('Yes')

    def _auto_complete(self, word, expected_string):
        self.page.fill_codemirror_area_with(word)
        ActionChains(self.page.driver).key_down(
            Keys.CONTROL).send_keys(Keys.SPACE).key_up(Keys.CONTROL).perform()
        self.page.find_by_xpath(
            "//ul[contains(@class, 'CodeMirror-hints') and "
            "contains(., '" + expected_string + "')]")
