"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const async_1 = require("async");
const json_1 = require("../utils/json");
const serviceManager_1 = require("../serviceManager");
const transport_1 = require("../utils/transport");
class NotifyInspector {
    constructor() {
        this.inspectorService = serviceManager_1.ServiceManager.get('inspector');
        this.exceptionsTrapped = [];
    }
    init() {
        this.inspectorService.createSession();
        this.inspectorService.connect();
        this.catchAllDebugger();
    }
    destroy() {
        this.inspectorService.disconnect();
    }
    trapException(listener) {
        return (error) => {
            // log it
            if (listener === 'unhandledRejection') {
                console.log('You have triggered an unhandledRejection, you may have forgotten to catch a Promise rejection:');
            }
            console.error(error);
            // create object to be send
            const context = this.exceptionsTrapped.find((exception) => {
                return !!exception.error.description.match(error.message);
            });
            error = json_1.default.jsonize(error);
            // inject async stackframe
            if (context && context.asyncStackTrace) {
                const fetchFrames = (entry) => {
                    return entry.parent ? entry.callFrames.concat(fetchFrames(entry.parent)) : entry.callFrames;
                };
                const asyncStack = fetchFrames(context.asyncStackTrace)
                    .filter(frame => frame.url.indexOf('internal') === -1)
                    .map(frame => {
                    return `    at ${frame.functionName || '<anonymous>'} (${frame.url}:${frame.lineNumber}:${frame.columnNumber})`;
                });
                asyncStack.unshift('');
                error.stack = error.stack.concat(asyncStack.join('\n'));
            }
            error.frame = context ? context.frame : undefined;
            // send it
            transport_1.default.send({
                type: 'process:exception',
                data: error
            });
            // at this point the process should exit
            process.exit(1);
        };
    }
    isObjectInteresting(entry) {
        if (!entry.value)
            return false;
        if (!entry.value.objectId)
            return false;
        if (entry.value.type !== 'object')
            return false;
        switch (entry.value.description) {
            case 'IncomingMessage': {
                return true;
            }
        }
        switch (entry.name) {
            case 'headers': {
                return true;
            }
            case 'user': {
                return true;
            }
            case 'token': {
                return true;
            }
            case 'body': {
                return true;
            }
            case 'params': {
                return true;
            }
            case 'query': {
                return true;
            }
        }
        return false;
    }
    isPropertyIntesting(entry, parent) {
        if (!entry.value)
            return false;
        if (entry.value.type === 'object' && entry.properties)
            return true;
        if (parent && parent.name === 'headers')
            return true;
        if (parent && parent.name === 'body')
            return true;
        if (parent && parent.name === 'params')
            return true;
        if (parent && parent.name === 'query')
            return true;
        if (entry.name === '__proto__')
            return false;
        switch (entry.name) {
            case 'url': {
                return true;
            }
            case 'user': {
                return true;
            }
            case 'token': {
                return true;
            }
            case 'method': {
                return true;
            }
            case 'ip': {
                return true;
            }
            case 'query': {
                return true;
            }
            case 'path': {
                return true;
            }
            case 'body': {
                return true;
            }
            case 'params': {
                return true;
            }
        }
        return false;
    }
    formatProperty(property) {
        const value = property.value && property.value.value ? property.value.value : null;
        const description = property.value && property.value.description ? property.value.description : null;
        return {
            name: property.name,
            value: value || description || property.value,
            properties: property.properties
        };
    }
    fetchObjectProperties(session, object, cb) {
        session.post('Runtime.getProperties', {
            objectId: object,
            ownProperties: true
        }, (err, data) => {
            if (err)
                return cb(err, undefined);
            async_1.default.map(data.result, (entry, next) => {
                if (entry.value && entry.value.objectId && this.isObjectInteresting(entry)) {
                    return this.fetchObjectProperties(session, entry.value.objectId, (err, properties) => {
                        if (err)
                            return next(err);
                        // if some properties has been dumped, attach them
                        if (properties) {
                            entry.properties = properties
                                .filter(property => {
                                return this.isPropertyIntesting(property, entry);
                            })
                                .map(this.formatProperty);
                        }
                        return next(undefined, this.formatProperty(entry));
                    });
                }
                else {
                    return next(undefined, this.formatProperty(entry));
                }
            }, cb);
        });
    }
    catchAllDebugger() {
        const session = this.inspectorService.createSession();
        this.inspectorService.connect();
        // trap exception so we can re-use them with the debugger
        process.on('uncaughtException', this.trapException('uncaughtException'));
        process.on('unhandledRejection', this.trapException('unhandledRejection'));
        // enable all the debugger options
        session.post('Debugger.enable');
        const maxDepth = parseInt(process.env.PM2_APM_ASYNC_STACK_DEPTH || '', 10);
        if (!isNaN(maxDepth)) {
            session.post('Debugger.setAsyncCallStackDepth', { maxDepth });
        }
        session.post('Debugger.setPauseOnExceptions', { state: 'uncaught' });
        // register handler for paused event
        session.on('Debugger.paused', ({ params }) => {
            // should not happen but anyway
            if (params.reason !== 'exception' && params.reason !== 'promiseRejection') {
                return session.post('Debugger.resume');
            }
            if (!params.data)
                return session.post('Debugger.resume');
            const error = params.data;
            // get only the current frame
            const frame = params.callFrames[0];
            // on each frame, dump all scopes
            async_1.default.map(frame.scopeChain, (scope, nextScope) => {
                if (scope.type === 'global')
                    return nextScope();
                if (!scope.object.objectId)
                    return nextScope();
                // get context of the scope
                return this.fetchObjectProperties(session, scope.object.objectId, (error, context) => {
                    return nextScope(error, Object.assign(scope, {
                        context,
                        object: undefined
                    }));
                });
            }, (err, scopes) => {
                if (err) {
                    console.error(err);
                    return session.post('Debugger.resume');
                }
                // we can remove some scope so we want to remove null entry
                scopes = scopes.filter(scope => !!scope);
                // inspect each scope to retrieve his context
                this.exceptionsTrapped.push({
                    error,
                    asyncStackTrace: params.asyncStackTrace,
                    frame: Object.assign(frame, {
                        scopeChain: scopes
                    })
                });
                // continue execution
                return session.post('Debugger.resume');
            });
        });
    }
}
exports.default = NotifyInspector;
//# sourceMappingURL=data:application/json;base64,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