"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const notify_1 = require("./features/notify");
const metrics_1 = require("./features/metrics");
const actions_1 = require("./features/actions");
const events_1 = require("./features/events");
const merge = require("deepmerge");
const configuration_1 = require("./configuration");
const metricConfig_1 = require("./utils/metricConfig");
const debug_1 = require("debug");
const fs = require("fs");
const cluster = require("cluster");
const serviceManager_1 = require("./serviceManager");
const debug = debug_1.default('PM2-IO-APM');
class TransactionConfig {
}
class MetricsConfig {
}
class ActionsConfig {
}
class IOConfig {
}
class PMX {
    constructor() {
        this.notifyFeature = new notify_1.NotifyFeature();
        this.metricsFeature = new metrics_1.default();
        this.actionsFeature = new actions_1.default(!cluster.isWorker);
        this.eventsFeature = new events_1.default();
        const eventLoopInspector = require('event-loop-inspector')(true);
        serviceManager_1.ServiceManager.set('eventLoopService', {
            inspector: eventLoopInspector
        });
    }
    getInitialConfig() {
        return this.initialConfig;
    }
    init(config, force) {
        let notifyOptions = notify_1.NotifyOptionsDefault;
        let configMetrics = {};
        if (!config) {
            config = new IOConfig();
        }
        if (process.env.PMX_FORCE_UPDATE) {
            const IO_KEY = Symbol.for('@pm2/io');
            const globalSymbols = Object.getOwnPropertySymbols(global);
            const alreadyInstanciated = (globalSymbols.indexOf(IO_KEY) > -1);
            if (alreadyInstanciated) {
                global[IO_KEY].destroy();
            }
            global[IO_KEY] = this;
        }
        if (!config) {
            config = new IOConfig();
        }
        if (config.level) {
            notifyOptions.level = config.level;
        }
        if (config.catchExceptions) {
            notifyOptions.catchExceptions = config.catchExceptions;
        }
        if (config.metrics) {
            configMetrics = config.metrics;
        }
        this.backwardConfigConversion(config);
        this.notifyFeature.init(notifyOptions);
        this.metricsFeature.init(config.metrics, force);
        this.actionsFeature.init(config.actions, force);
        configuration_1.default.init(config);
        this.initialConfig = config;
        return this;
    }
    destroy() {
        if (this.metricsFeature)
            this.metricsFeature.destroy();
        if (this.actionsFeature)
            this.actionsFeature.destroy();
        if (this.notifyFeature)
            this.notifyFeature.destroy();
    }
    notifyError(err, context) {
        let level = 'info';
        if (context && context.level) {
            level = context.level;
        }
        this.notifyFeature.notifyError(err, level);
    }
    metrics(metrics) {
        const res = {};
        let allMetrics = [];
        if (!Array.isArray(metrics)) {
            allMetrics[0] = metrics;
        }
        else {
            allMetrics = metrics;
        }
        for (let i = 0; i < allMetrics.length; i++) {
            const currentMetric = allMetrics[i];
            if (!currentMetric || !currentMetric.hasOwnProperty('name') || !currentMetric.hasOwnProperty('type')) {
                console.warn(`Metric can't be initialized : missing some properties !`);
                console.warn('name => required');
                console.warn('type => required');
                console.warn('id => optional');
                console.warn('unit => optional');
                console.warn('value => optional');
                console.warn('historic => optional');
                console.warn('agg_type => optional');
                console.warn('measurement => optional');
                continue;
            }
            // escape spaces and special characters from metric's name
            const metricKey = currentMetric.name.replace(/ /g, '_').replace(/[^\w\s]/gi, '');
            const type = currentMetric.type;
            currentMetric.type = currentMetric.id;
            delete currentMetric.id;
            if (typeof this.metricsFeature[type] !== 'function') {
                console.warn(`Metric ${currentMetric.name} cant be initialized : unknown type ${type} !`);
                continue;
            }
            res[metricKey] = this.metricsFeature[type](currentMetric);
        }
        return res;
    }
    histogram(config) {
        config = metricConfig_1.default.buildConfig(config);
        return this.metricsFeature['histogram'](config);
    }
    metric(config) {
        config = metricConfig_1.default.buildConfig(config);
        return this.metricsFeature['metric'](config);
    }
    counter(config) {
        config = metricConfig_1.default.buildConfig(config);
        return this.metricsFeature['counter'](config);
    }
    meter(config) {
        config = metricConfig_1.default.buildConfig(config);
        return this.metricsFeature['meter'](config);
    }
    action(name, opts, fn) {
        if (typeof name === 'object') {
            opts = name.opts;
            fn = name.action;
            name = name.name;
        }
        this.actionsFeature.action(name, opts, fn);
    }
    scopedAction(name, fn) {
        this.actionsFeature.scopedAction(name, fn);
    }
    transpose(variableName, reporter) {
        this.metricsFeature.transpose(variableName, reporter);
    }
    onExit(callback) {
        if (callback && typeof callback === 'function') {
            const onExit = require('signal-exit');
            return onExit(callback);
        }
    }
    // -----------------------------------------------------------
    // Retro compatibility
    // -----------------------------------------------------------
    probe() {
        return {
            histogram: (histogram) => {
                return this.genericBackwardConversion(histogram, 'histogram');
            },
            meter: (meter) => {
                return this.genericBackwardConversion(meter, 'meter');
            },
            metric: (metric) => {
                return this.genericBackwardConversion(metric, 'metric');
            },
            counter: (counter) => {
                return this.genericBackwardConversion(counter, 'counter');
            },
            transpose: (variableName, reporter) => {
                this.transpose(variableName, reporter);
            }
        };
    }
    emit(name, data) {
        this.eventsFeature.emit(name, data);
    }
    emitEvent(name, data) {
        this.eventsFeature.emit(name, data);
    }
    notify(notification) {
        if (!(notification instanceof Error)) {
            notification = new Error(notification);
        }
        this.notifyFeature.notifyError(notification);
    }
    getPID(file) {
        if (typeof (file) === 'number')
            return file;
        return parseInt(fs.readFileSync(file).toString(), 10);
    }
    initModule(opts, cb) {
        if (!opts)
            opts = {};
        if (opts.reference) {
            opts.name = opts.reference;
            delete opts.reference;
        }
        opts = merge({
            widget: {}
        }, opts);
        opts.widget = merge({
            type: 'generic',
            logo: 'https://app.keymetrics.io/img/logo/keymetrics-300.png',
            theme: ['#111111', '#1B2228', '#807C7C', '#807C7C']
        }, opts.widget);
        opts.isModule = true;
        opts = configuration_1.default.init(opts);
        if (cb && typeof (cb) === 'function')
            return cb(null, opts);
        return opts;
    }
    expressErrorHandler() {
        return this.notifyFeature.expressErrorHandler();
    }
    genericBackwardConversion(object, type) {
        if (typeof object !== 'object') {
            console.error('Parameter should be an object');
            return null;
        }
        object.type = type;
        // escape spaces and special characters from metric's name
        const metricKey = object.name.replace(/ /g, '_').replace(/[^\w\s]/gi, '');
        return this.metrics(object)[metricKey];
    }
    backwardConfigConversion(config) {
        // ------------------------------------------
        // Network
        // ------------------------------------------
        if (config.hasOwnProperty('network') || config.hasOwnProperty('ports')) {
            const networkConf = {};
            if (config.hasOwnProperty('network')) {
                networkConf.traffic = Boolean(config.network);
                delete config.network;
            }
            if (config.hasOwnProperty('ports')) {
                networkConf.ports = Boolean(config.ports);
                delete config.ports;
            }
            this.initMetricsConf(config);
            config.metrics.network = networkConf;
        }
        // ------------------------------------------
        // V8
        // ------------------------------------------
        if (config.hasOwnProperty('v8')) {
            this.initMetricsConf(config);
            config.metrics.v8 = config.v8;
            delete config.v8;
        }
        // ------------------------------------------
        // transactions
        // ------------------------------------------
        if (config.hasOwnProperty('transactions') || config.hasOwnProperty('http')) {
            this.initMetricsConf(config);
            config.metrics.transaction = new TransactionConfig();
            if (config.hasOwnProperty('transactions')) {
                config.metrics.transaction.tracing = config.transactions;
                delete config.transactions;
            }
            if (config.hasOwnProperty('http')) {
                config.metrics.transaction.http = config.http;
                delete config.http;
            }
        }
        // ------------------------------------------
        // Deep metrics
        // ------------------------------------------
        if (config.hasOwnProperty('deep_metrics')) {
            this.initMetricsConf(config);
            config.metrics.deepMetrics = config.deep_metrics;
            delete config.deep_metrics;
        }
        // ------------------------------------------
        // Event Loop action
        // ------------------------------------------
        if (config.hasOwnProperty('event_loop_dump')) {
            this.initActionsConf(config);
            config.actions.eventLoopDump = config.event_loop_dump;
            delete config.event_loop_dump;
        }
        // ------------------------------------------
        // Profiling action
        // ------------------------------------------
        if (config.hasOwnProperty('profiling')) {
            this.initActionsConf(config);
            config.actions.profilingHeap = config.profiling;
            config.actions.profilingHeap = config.profiling;
            delete config.profiling;
        }
    }
    initMetricsConf(config) {
        if (!config.metrics) {
            config.metrics = new MetricsConfig();
        }
    }
    initActionsConf(config) {
        if (!config.actions) {
            config.actions = new ActionsConfig();
        }
    }
}
exports.default = PMX;
//# sourceMappingURL=data:application/json;base64,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