# 1 "src/dsexp/dune_lexer.mll"
 
open! Stdune
open Lexer_shared

type block_string_line_kind =
  | With_escape_sequences
  | Raw

module Template = struct
  include Template

  let dummy_loc =
    { Loc.
      start = Lexing.dummy_pos
    ; stop = Lexing.dummy_pos
    }

  let add_text parts s =
    match parts with
    | Template.Text s' :: parts -> Template.Text (s' ^ s) :: parts
    | _ -> Template.Text s :: parts

  let token parts ~quoted ~start (lexbuf : Lexing.lexbuf) =
    lexbuf.lex_start_p <- start;
    match parts with
    | [] | [Text ""] ->
      error lexbuf "Internal error in the S-expression parser, \
                    please report upstream."
    | [Text s] ->
      Token.Atom (Atom.of_string s)
    | _ ->
      Token.Template
        { quoted
        ; loc = dummy_loc
        ; parts = List.rev parts
        }

  module Buffer : sig
    val new_token : unit -> unit
    val get : unit -> Token.t
    val add_var : part -> unit
    val add_text : string -> unit
    val add_text_c : char -> unit
  end = struct
    type state =
      | String
      | Template of Template.part list

    let text_buf = Buffer.create 256

    let new_token () = Buffer.clear text_buf

    let take_buf () =
      let contents = Buffer.contents text_buf in
      Buffer.clear text_buf;
      contents

    let state = ref String

    let add_buf_to_parts parts =
      match take_buf () with
      | "" -> parts
      | t -> add_text parts t

    let get () =
      match !state with
      | String -> Token.Quoted_string (take_buf ())
      | Template parts ->
        state := String;
        begin match add_buf_to_parts parts with
        | [] -> assert false
        | [Text s] -> Quoted_string s
        | parts ->
          Token.Template
            { quoted = true
            ; loc = dummy_loc
            ; parts = List.rev parts
            }
        end

    let add_var v =
      match !state with
      | String ->
        state := Template (v :: add_buf_to_parts []);
      | Template parts ->
        let parts = add_buf_to_parts parts in
        state := Template (v::parts)

    let add_text   = Buffer.add_string text_buf
    let add_text_c = Buffer.add_char text_buf
  end
end

# 96 "src/dsexp/dune_lexer.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base =
   "\000\000\250\255\251\255\252\255\253\255\001\000\006\000\255\255\
    \006\000\018\000\022\000\113\000\254\255\055\000\178\000\254\255\
    \255\255\007\000\252\255\253\255\254\255\255\255\009\000\231\000\
    \251\255\252\255\027\000\254\255\255\255\011\000\253\255\013\000\
    \061\000\014\000\180\000\254\255\255\255\014\000\253\255\254\255\
    \255\255\018\000\235\000\250\255\251\255\252\255\019\000\031\000\
    \254\255\255\255\253\255\237\000\246\255\247\255\054\001\001\001\
    \253\255\049\000\255\255\020\000\254\255\032\001\077\001\087\001\
    \109\001\147\001\173\001\026\000\243\001\253\255\254\255\081\002\
    \255\255\184\002\022\003\255\255";
  Lexing.lex_backtrk =
   "\006\000\255\255\255\255\255\255\255\255\001\000\001\000\255\255\
    \255\255\003\000\002\000\000\000\255\255\002\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\003\000\255\255\
    \255\255\255\255\003\000\255\255\255\255\003\000\255\255\002\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\001\000\255\255\255\255\255\255\255\255\004\000\004\000\
    \255\255\255\255\255\255\005\000\255\255\255\255\007\000\005\000\
    \255\255\008\000\255\255\008\000\255\255\005\000\003\000\005\000\
    \007\000\006\000\007\000\000\000\255\255\255\255\255\255\002\000\
    \255\255\255\255\255\255\255\255";
  Lexing.lex_default =
   "\255\255\000\000\000\000\000\000\000\000\005\000\255\255\000\000\
    \255\255\255\255\255\255\255\255\000\000\255\255\255\255\000\000\
    \000\000\018\000\000\000\000\000\000\000\000\000\255\255\025\000\
    \000\000\000\000\255\255\000\000\000\000\255\255\000\000\255\255\
    \255\255\255\255\255\255\000\000\000\000\039\000\000\000\000\000\
    \000\000\255\255\044\000\000\000\000\000\000\000\255\255\255\255\
    \000\000\000\000\000\000\053\000\000\000\000\000\255\255\255\255\
    \000\000\255\255\000\000\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\069\000\000\000\000\000\255\255\
    \000\000\255\255\255\255\000\000";
  Lexing.lex_trans =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\006\000\007\000\255\255\006\000\008\000\255\255\006\000\
    \007\000\021\000\006\000\021\000\022\000\028\000\033\000\033\000\
    \040\000\033\000\033\000\041\000\040\000\045\000\058\000\000\000\
    \006\000\000\000\002\000\067\000\000\000\000\000\006\000\020\000\
    \004\000\003\000\000\000\000\000\000\000\033\000\033\000\032\000\
    \032\000\000\000\000\000\011\000\000\000\011\000\011\000\010\000\
    \011\000\011\000\067\000\005\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\000\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\012\000\011\000\014\000\011\000\011\000\030\000\011\000\
    \011\000\034\000\050\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\060\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \015\000\028\000\035\000\000\000\029\000\045\000\000\000\058\000\
    \046\000\000\000\059\000\000\000\000\000\000\000\000\000\000\000\
    \001\000\255\255\000\000\000\000\000\000\000\000\000\000\019\000\
    \000\000\000\000\000\000\000\000\026\000\049\000\038\000\056\000\
    \047\000\000\000\057\000\000\000\056\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\016\000\000\000\
    \036\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\027\000\000\000\000\000\000\000\048\000\
    \000\000\056\000\000\000\000\000\000\000\000\000\000\000\056\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\000\000\056\000\000\000\000\000\000\000\056\000\
    \000\000\056\000\000\000\000\000\000\000\054\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\000\000\000\000\000\000\000\000\000\000\000\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\000\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\000\000\000\000\000\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\024\000\
    \000\000\000\000\000\000\043\000\000\000\052\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\071\000\000\000\071\000\071\000\
    \000\000\071\000\071\000\000\000\000\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\000\000\000\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\000\000\071\000\
    \070\000\071\000\074\000\000\000\074\000\074\000\000\000\074\000\
    \074\000\000\000\000\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\073\000\000\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\000\000\074\000\072\000\074\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\073\000\000\000\073\000\073\000\000\000\073\000\073\000\
    \000\000\000\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\070\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\000\000\073\000\075\000\073\000\074\000\
    \000\000\074\000\074\000\000\000\074\000\074\000\000\000\000\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \073\000\000\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\000\000\074\000\072\000\074\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_check =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\005\000\000\000\000\000\005\000\006\000\
    \008\000\017\000\006\000\022\000\017\000\029\000\031\000\033\000\
    \037\000\031\000\033\000\037\000\041\000\046\000\059\000\255\255\
    \000\000\255\255\000\000\067\000\255\255\255\255\006\000\017\000\
    \000\000\000\000\255\255\255\255\255\255\031\000\033\000\031\000\
    \033\000\255\255\255\255\009\000\255\255\009\000\009\000\009\000\
    \009\000\009\000\067\000\000\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\009\000\009\000\255\255\009\000\009\000\
    \009\000\009\000\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\009\000\009\000\009\000\009\000\009\000\009\000\009\000\
    \009\000\010\000\011\000\013\000\011\000\011\000\026\000\011\000\
    \011\000\032\000\047\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\057\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \014\000\023\000\034\000\255\255\023\000\042\000\255\255\051\000\
    \042\000\255\255\051\000\255\255\255\255\255\255\255\255\255\255\
    \000\000\005\000\255\255\255\255\255\255\255\255\255\255\017\000\
    \255\255\255\255\255\255\255\255\023\000\042\000\037\000\051\000\
    \042\000\255\255\051\000\255\255\051\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\014\000\255\255\
    \034\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\023\000\255\255\255\255\255\255\042\000\
    \255\255\051\000\255\255\255\255\255\255\255\255\255\255\051\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\255\255\051\000\255\255\255\255\255\255\051\000\
    \255\255\051\000\255\255\255\255\255\255\051\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\054\000\
    \054\000\054\000\054\000\054\000\054\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\255\255\255\255\255\255\255\255\255\255\255\255\054\000\
    \054\000\054\000\054\000\054\000\054\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\064\000\064\000\
    \064\000\064\000\064\000\064\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\255\255\064\000\064\000\
    \064\000\064\000\064\000\064\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\255\255\255\255\255\255\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\023\000\
    \255\255\255\255\255\255\042\000\255\255\051\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\066\000\066\000\
    \066\000\066\000\066\000\066\000\068\000\255\255\068\000\068\000\
    \255\255\068\000\068\000\255\255\255\255\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\255\255\255\255\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\255\255\068\000\
    \068\000\068\000\071\000\255\255\071\000\071\000\255\255\071\000\
    \071\000\255\255\255\255\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\255\255\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\255\255\071\000\071\000\071\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\073\000\255\255\073\000\073\000\255\255\073\000\073\000\
    \255\255\255\255\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\068\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\255\255\073\000\073\000\073\000\074\000\
    \255\255\074\000\074\000\255\255\074\000\074\000\255\255\255\255\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\255\255\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\255\255\074\000\074\000\074\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255";
  Lexing.lex_base_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\092\000\
    \009\000\184\000\020\001\022\000";
  Lexing.lex_backtrk_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000";
  Lexing.lex_default_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000";
  Lexing.lex_trans_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\001\000\000\000\001\000\001\000\000\000\001\000\001\000\
    \000\000\000\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\000\000\000\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\000\000\001\000\001\000\001\000\001\000\
    \001\000\000\000\001\000\001\000\000\000\000\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\004\000\000\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\000\000\
    \001\000\014\000\001\000\014\000\014\000\000\000\014\000\014\000\
    \000\000\000\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\000\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\000\000\014\000\001\000\014\000\001\000\
    \001\000\000\000\001\000\001\000\000\000\000\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\017\000\000\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\000\000\
    \001\000\000\000\001\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_check_code =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\068\000\255\255\068\000\068\000\255\255\068\000\068\000\
    \255\255\255\255\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\255\255\255\255\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\255\255\068\000\071\000\068\000\071\000\
    \071\000\255\255\071\000\071\000\255\255\255\255\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\255\255\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\255\255\
    \071\000\073\000\071\000\073\000\073\000\255\255\073\000\073\000\
    \255\255\255\255\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\255\255\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\255\255\073\000\074\000\073\000\074\000\
    \074\000\255\255\074\000\074\000\255\255\255\255\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\255\255\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\255\255\
    \074\000\255\255\074\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255";
  Lexing.lex_code =
   "\255\003\255\255\005\255\004\255\255\002\255\000\003\255\005\255\
    \255\004\255\005\255\255\002\004\001\005\000\003\255";
}

let rec token lexbuf =
   __ocaml_lex_token_rec lexbuf 0
and __ocaml_lex_token_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 106 "src/dsexp/dune_lexer.mll"
    ( Lexing.new_line lexbuf; token lexbuf )
# 576 "src/dsexp/dune_lexer.ml"

  | 1 ->
# 108 "src/dsexp/dune_lexer.mll"
    ( token lexbuf )
# 581 "src/dsexp/dune_lexer.ml"

  | 2 ->
# 110 "src/dsexp/dune_lexer.mll"
    ( Token.Lparen )
# 586 "src/dsexp/dune_lexer.ml"

  | 3 ->
# 112 "src/dsexp/dune_lexer.mll"
    ( Rparen )
# 591 "src/dsexp/dune_lexer.ml"

  | 4 ->
# 114 "src/dsexp/dune_lexer.mll"
    ( let start = Lexing.lexeme_start_p lexbuf in
      Template.Buffer.new_token ();
      let token = start_quoted_string lexbuf in
      lexbuf.lex_start_p <- start;
      token
    )
# 601 "src/dsexp/dune_lexer.ml"

  | 5 ->
# 121 "src/dsexp/dune_lexer.mll"
    ( Eof )
# 606 "src/dsexp/dune_lexer.ml"

  | 6 ->
# 123 "src/dsexp/dune_lexer.mll"
    ( atom [] (Lexing.lexeme_start_p lexbuf) lexbuf )
# 611 "src/dsexp/dune_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_token_rec lexbuf __ocaml_lex_state

and atom acc start lexbuf =
   __ocaml_lex_atom_rec acc start lexbuf 9
and __ocaml_lex_atom_rec acc start lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 126 "src/dsexp/dune_lexer.mll"
                          s
# 624 "src/dsexp/dune_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 127 "src/dsexp/dune_lexer.mll"
    ( atom (Template.add_text acc s) start lexbuf )
# 628 "src/dsexp/dune_lexer.ml"

  | 1 ->
# 129 "src/dsexp/dune_lexer.mll"
    ( atom ((template_variable lexbuf) :: acc) start lexbuf )
# 633 "src/dsexp/dune_lexer.ml"

  | 2 ->
# 131 "src/dsexp/dune_lexer.mll"
    ( atom (Template.add_text acc "%") start lexbuf )
# 638 "src/dsexp/dune_lexer.ml"

  | 3 ->
# 133 "src/dsexp/dune_lexer.mll"
    ( Template.token acc ~quoted:false ~start lexbuf )
# 643 "src/dsexp/dune_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_atom_rec acc start lexbuf __ocaml_lex_state

and start_quoted_string lexbuf =
   __ocaml_lex_start_quoted_string_rec lexbuf 13
and __ocaml_lex_start_quoted_string_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 137 "src/dsexp/dune_lexer.mll"
    ( block_string_start With_escape_sequences lexbuf )
# 655 "src/dsexp/dune_lexer.ml"

  | 1 ->
# 139 "src/dsexp/dune_lexer.mll"
    ( block_string_start Raw lexbuf )
# 660 "src/dsexp/dune_lexer.ml"

  | 2 ->
# 141 "src/dsexp/dune_lexer.mll"
    ( quoted_string lexbuf )
# 665 "src/dsexp/dune_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_start_quoted_string_rec lexbuf __ocaml_lex_state

and block_string_start kind lexbuf =
   __ocaml_lex_block_string_start_rec kind lexbuf 17
and __ocaml_lex_block_string_start_rec kind lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 144 "src/dsexp/dune_lexer.mll"
               s
# 678 "src/dsexp/dune_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 145 "src/dsexp/dune_lexer.mll"
    ( Lexing.new_line lexbuf;
      Template.Buffer.add_text s;
      block_string_after_newline lexbuf
    )
# 685 "src/dsexp/dune_lexer.ml"

  | 1 ->
# 150 "src/dsexp/dune_lexer.mll"
    ( match kind with
      | With_escape_sequences -> block_string lexbuf
      | Raw -> raw_block_string lexbuf
    )
# 693 "src/dsexp/dune_lexer.ml"

  | 2 ->
# 155 "src/dsexp/dune_lexer.mll"
    ( Template.Buffer.get () )
# 698 "src/dsexp/dune_lexer.ml"

  | 3 ->
# 157 "src/dsexp/dune_lexer.mll"
    ( error lexbuf "There must be at least one space after \"\\|"
    )
# 704 "src/dsexp/dune_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_block_string_start_rec kind lexbuf __ocaml_lex_state

and block_string lexbuf =
   __ocaml_lex_block_string_rec lexbuf 23
and __ocaml_lex_block_string_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 161 "src/dsexp/dune_lexer.mll"
               s
# 717 "src/dsexp/dune_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 162 "src/dsexp/dune_lexer.mll"
    ( Lexing.new_line lexbuf;
      Template.Buffer.add_text s;
      block_string_after_newline lexbuf
    )
# 724 "src/dsexp/dune_lexer.ml"

  | 1 ->
# 167 "src/dsexp/dune_lexer.mll"
    ( match escape_sequence lexbuf with
      | Newline -> block_string_after_newline lexbuf
      | Other   -> block_string               lexbuf
    )
# 732 "src/dsexp/dune_lexer.ml"

  | 2 ->
# 171 "src/dsexp/dune_lexer.mll"
         (
      let var = template_variable lexbuf in
      Template.Buffer.add_var var;
      block_string lexbuf
    )
# 741 "src/dsexp/dune_lexer.ml"

  | 3 ->
let
# 176 "src/dsexp/dune_lexer.mll"
         c
# 747 "src/dsexp/dune_lexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 177 "src/dsexp/dune_lexer.mll"
    ( Template.Buffer.add_text_c c;
      block_string lexbuf
    )
# 753 "src/dsexp/dune_lexer.ml"

  | 4 ->
# 181 "src/dsexp/dune_lexer.mll"
    ( Template.Buffer.get ()
    )
# 759 "src/dsexp/dune_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_block_string_rec lexbuf __ocaml_lex_state

and block_string_after_newline lexbuf =
   __ocaml_lex_block_string_after_newline_rec lexbuf 31
and __ocaml_lex_block_string_after_newline_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 186 "src/dsexp/dune_lexer.mll"
    ( block_string_start With_escape_sequences lexbuf )
# 771 "src/dsexp/dune_lexer.ml"

  | 1 ->
# 188 "src/dsexp/dune_lexer.mll"
    ( block_string_start Raw lexbuf )
# 776 "src/dsexp/dune_lexer.ml"

  | 2 ->
# 190 "src/dsexp/dune_lexer.mll"
    ( Template.Buffer.get ()
    )
# 782 "src/dsexp/dune_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_block_string_after_newline_rec lexbuf __ocaml_lex_state

and raw_block_string lexbuf =
   __ocaml_lex_raw_block_string_rec lexbuf 37
and __ocaml_lex_raw_block_string_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 194 "src/dsexp/dune_lexer.mll"
               s
# 795 "src/dsexp/dune_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 195 "src/dsexp/dune_lexer.mll"
    ( Lexing.new_line lexbuf;
      Template.Buffer.add_text s;
      block_string_after_newline lexbuf
    )
# 802 "src/dsexp/dune_lexer.ml"

  | 1 ->
let
# 199 "src/dsexp/dune_lexer.mll"
         c
# 808 "src/dsexp/dune_lexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 200 "src/dsexp/dune_lexer.mll"
    ( Template.Buffer.add_text_c c;
      raw_block_string lexbuf
    )
# 814 "src/dsexp/dune_lexer.ml"

  | 2 ->
# 204 "src/dsexp/dune_lexer.mll"
    ( Template.Buffer.get ()
    )
# 820 "src/dsexp/dune_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_raw_block_string_rec lexbuf __ocaml_lex_state

and quoted_string lexbuf =
   __ocaml_lex_quoted_string_rec lexbuf 42
and __ocaml_lex_quoted_string_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 209 "src/dsexp/dune_lexer.mll"
    ( Template.Buffer.get () )
# 832 "src/dsexp/dune_lexer.ml"

  | 1 ->
# 211 "src/dsexp/dune_lexer.mll"
    ( match escape_sequence lexbuf with
      | Newline -> quoted_string_after_escaped_newline lexbuf
      | Other   -> quoted_string                       lexbuf
    )
# 840 "src/dsexp/dune_lexer.ml"

  | 2 ->
# 216 "src/dsexp/dune_lexer.mll"
    ( Template.Buffer.add_var (template_variable lexbuf);
      quoted_string lexbuf
    )
# 847 "src/dsexp/dune_lexer.ml"

  | 3 ->
let
# 219 "src/dsexp/dune_lexer.mll"
               s
# 853 "src/dsexp/dune_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 220 "src/dsexp/dune_lexer.mll"
    ( Lexing.new_line lexbuf;
      Template.Buffer.add_text s;
      quoted_string lexbuf
    )
# 860 "src/dsexp/dune_lexer.ml"

  | 4 ->
let
# 224 "src/dsexp/dune_lexer.mll"
         c
# 866 "src/dsexp/dune_lexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 225 "src/dsexp/dune_lexer.mll"
    ( Template.Buffer.add_text_c c;
      quoted_string lexbuf
    )
# 872 "src/dsexp/dune_lexer.ml"

  | 5 ->
# 229 "src/dsexp/dune_lexer.mll"
    ( error lexbuf "unterminated quoted string"
    )
# 878 "src/dsexp/dune_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_quoted_string_rec lexbuf __ocaml_lex_state

and escape_sequence lexbuf =
   __ocaml_lex_escape_sequence_rec lexbuf 51
and __ocaml_lex_escape_sequence_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 234 "src/dsexp/dune_lexer.mll"
    ( Lexing.new_line lexbuf;
      Newline )
# 891 "src/dsexp/dune_lexer.ml"

  | 1 ->
let
# 236 "src/dsexp/dune_lexer.mll"
            s
# 897 "src/dsexp/dune_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos (lexbuf.Lexing.lex_start_pos + 2) in
# 237 "src/dsexp/dune_lexer.mll"
    ( Template.Buffer.add_text s;
      Other
    )
# 903 "src/dsexp/dune_lexer.ml"

  | 2 ->
let
# 240 "src/dsexp/dune_lexer.mll"
                                       c
# 909 "src/dsexp/dune_lexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 241 "src/dsexp/dune_lexer.mll"
    ( let c =
        match c with
        | 'n' -> '\n'
        | 'r' -> '\r'
        | 'b' -> '\b'
        | 't' -> '\t'
        | _   -> c
      in
      Template.Buffer.add_text_c c;
      Other
    )
# 923 "src/dsexp/dune_lexer.ml"

  | 3 ->
let
# 252 "src/dsexp/dune_lexer.mll"
              c1
# 929 "src/dsexp/dune_lexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos
and
# 252 "src/dsexp/dune_lexer.mll"
                            c2
# 934 "src/dsexp/dune_lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1)
and
# 252 "src/dsexp/dune_lexer.mll"
                                          c3
# 939 "src/dsexp/dune_lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 2) in
# 253 "src/dsexp/dune_lexer.mll"
    ( let v = eval_decimal_escape c1 c2 c3 in
      if v > 255 then
        error lexbuf "escape sequence in quoted string out of range"
          ~delta:(-1);
      Template.Buffer.add_text_c (Char.chr v);
      Other
    )
# 949 "src/dsexp/dune_lexer.ml"

  | 4 ->
# 261 "src/dsexp/dune_lexer.mll"
    ( error lexbuf "escape sequence in quoted string out of range" ~delta:(-1);
    )
# 955 "src/dsexp/dune_lexer.ml"

  | 5 ->
# 264 "src/dsexp/dune_lexer.mll"
    ( error lexbuf "unterminated decimal escape sequence" ~delta:(-1);
    )
# 961 "src/dsexp/dune_lexer.ml"

  | 6 ->
let
# 266 "src/dsexp/dune_lexer.mll"
                     c1
# 967 "src/dsexp/dune_lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 1)
and
# 266 "src/dsexp/dune_lexer.mll"
                                      c2
# 972 "src/dsexp/dune_lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 2) in
# 267 "src/dsexp/dune_lexer.mll"
    ( let v = eval_hex_escape c1 c2 in
      Template.Buffer.add_text_c (Char.chr v);
      Other
    )
# 979 "src/dsexp/dune_lexer.ml"

  | 7 ->
# 272 "src/dsexp/dune_lexer.mll"
    ( error lexbuf "unterminated hexadecimal escape sequence" ~delta:(-1);
    )
# 985 "src/dsexp/dune_lexer.ml"

  | 8 ->
# 275 "src/dsexp/dune_lexer.mll"
    ( error lexbuf "unknown escape sequence" ~delta:(-1);
    )
# 991 "src/dsexp/dune_lexer.ml"

  | 9 ->
# 278 "src/dsexp/dune_lexer.mll"
    ( error lexbuf "unterminated escape sequence" ~delta:(-1);
    )
# 997 "src/dsexp/dune_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_escape_sequence_rec lexbuf __ocaml_lex_state

and quoted_string_after_escaped_newline lexbuf =
   __ocaml_lex_quoted_string_after_escaped_newline_rec lexbuf 67
and __ocaml_lex_quoted_string_after_escaped_newline_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 283 "src/dsexp/dune_lexer.mll"
    ( quoted_string lexbuf )
# 1009 "src/dsexp/dune_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_quoted_string_after_escaped_newline_rec lexbuf __ocaml_lex_state

and template_variable lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 6 (-1); __ocaml_lex_template_variable_rec lexbuf 68
and __ocaml_lex_template_variable_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 286 "src/dsexp/dune_lexer.mll"
                      name
# 1022 "src/dsexp/dune_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_mem.(0)
and
# 286 "src/dsexp/dune_lexer.mll"
                                                           payload
# 1027 "src/dsexp/dune_lexer.ml"
= Lexing.sub_lexeme_opt lexbuf lexbuf.Lexing.lex_mem.(2) lexbuf.Lexing.lex_mem.(1) in
# 287 "src/dsexp/dune_lexer.mll"
    ( let payload =
        match payload with
        | Some "" -> error lexbuf "payload after : in variable cannot be empty"
        | p -> p
      in
      Template.Var
        { loc =
            { start = Lexing.lexeme_start_p lexbuf
            ; stop = Lexing.lexeme_end_p lexbuf
            }
        ; name
        ; payload
        ; syntax = Percent
        }
  )
# 1045 "src/dsexp/dune_lexer.ml"

  | 1 ->
# 303 "src/dsexp/dune_lexer.mll"
    ( error lexbuf "%{...} forms cannot be empty" )
# 1050 "src/dsexp/dune_lexer.ml"

  | 2 ->
# 304 "src/dsexp/dune_lexer.mll"
      ( error lexbuf "This character is not allowed inside %{...} forms" )
# 1055 "src/dsexp/dune_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_template_variable_rec lexbuf __ocaml_lex_state

;;

