"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const netModule = require("net");
const metricConfig_1 = require("../utils/metricConfig");
const debug_1 = require("debug");
const debug = debug_1.default('axm:network');
class NetworkMetric {
    constructor(metricFeature) {
        this.defaultConf = {
            ports: false,
            traffic: true
        };
        this.metricFeature = metricFeature;
    }
    init(config) {
        config = metricConfig_1.default.getConfig(config, this.defaultConf);
        if (config.traffic) {
            this.catchTraffic(config.traffic);
        }
        if (config.ports) {
            this.catchPorts();
        }
    }
    destroy() {
        clearTimeout(this.timer);
        debug('NetworkMetric destroyed !');
    }
    catchPorts() {
        const portsList = [];
        let openedPorts = 'N/A';
        this.metricFeature.metric({
            name: 'Open ports',
            type: 'internal/network/open-ports',
            value: function () { return openedPorts; }
        });
        const originalListen = netModule.Server.prototype.listen;
        netModule.Server.prototype.listen = function () {
            const port = parseInt(arguments[0], 10);
            if (!isNaN(port) && portsList.indexOf(port) === -1) {
                portsList.push(port);
                openedPorts = portsList.sort().join();
            }
            this.once('close', function () {
                if (portsList.indexOf(port) > -1) {
                    portsList.splice(portsList.indexOf(port), 1);
                    openedPorts = portsList.sort().join();
                }
            });
            return originalListen.apply(this, arguments);
        };
    }
    catchTraffic(config) {
        let download = 0;
        let upload = 0;
        let up = '0 B/sec';
        let down = '0 B/sec';
        const filter = function (bytes) {
            let toFixed = 0;
            if (bytes < 1024) {
                toFixed = 6;
            }
            else if (bytes < (1024 * 1024)) {
                toFixed = 3;
            }
            else if (bytes !== 0) {
                toFixed = 2;
            }
            bytes = (bytes / (1024 * 1024)).toFixed(toFixed);
            let cutZeros = 0;
            for (let i = (bytes.length - 1); i > 0; --i) {
                if (bytes[i] === '.') {
                    ++cutZeros;
                    break;
                }
                if (bytes[i] !== '0')
                    break;
                ++cutZeros;
            }
            if (cutZeros > 0) {
                bytes = bytes.slice(0, -(cutZeros));
            }
            return (bytes + ' MB/s');
        };
        const interval = setInterval(function () {
            up = filter(upload);
            down = filter(download);
            upload = 0;
            download = 0;
        }, 999);
        interval.unref();
        if (config === true || config.download === true) {
            this.metricFeature.metric({
                name: 'Network In',
                type: 'internal/network/in',
                agg_type: 'sum',
                value: function () {
                    return down;
                }
            });
        }
        if (config === true || config.upload === true) {
            this.metricFeature.metric({
                name: 'Network Out',
                type: 'internal/network/out',
                agg_type: 'sum',
                value: function () {
                    return up;
                }
            });
        }
        if (config === true || config.upload === true) {
            const originalWrite = netModule.Socket.prototype.write;
            netModule.Socket.prototype.write = function (data) {
                if (data.length) {
                    upload += data.length;
                }
                return originalWrite.apply(this, arguments);
            };
        }
        if (config === true || config.download === true || Number.isInteger(config.download)) {
            const delay = Number.isInteger(config.download) ? config.download : 500;
            this.timer = setTimeout(() => {
                const originalRead = netModule.Socket.prototype.read;
                netModule.Socket.prototype.read = function () {
                    if (!this.monitored) {
                        this.monitored = true;
                        this.on('data', function (data) {
                            if (data.length) {
                                download += data.length;
                            }
                        });
                    }
                    return originalRead.apply(this, arguments);
                };
            }, delay);
            this.timer.unref();
        }
    }
}
exports.default = NetworkMetric;
//# sourceMappingURL=data:application/json;base64,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