#ifndef _EFL_UI_CURSOR_EO_H_
#define _EFL_UI_CURSOR_EO_H_

#ifndef _EFL_UI_CURSOR_EO_CLASS_TYPE
#define _EFL_UI_CURSOR_EO_CLASS_TYPE

typedef Eo Efl_Ui_Cursor;

#endif

#ifndef _EFL_UI_CURSOR_EO_TYPES
#define _EFL_UI_CURSOR_EO_TYPES


#endif
/**
 * @brief An interface to define mouse cursors to use when hovering over UI
 * widgets.
 *
 * The Elementary cursor is an internal smart object used to customize the
 * mouse cursor displayed over objects (or widgets). In the most common
 * scenario, the cursor decoration comes from the graphical engine Elementary
 * is running on. These engines may provide different decorations for cursors,
 * and Elementary provides functions to choose them for instance X11 cursors.
 *
 * By default, Elementary searches cursors only from the engine. Besides using
 * cursors provided by the you can also use those coming from Edje theme files.
 * Both globally and per widget, Elementary can help you store cursors in
 * engines only or in Elementary's theme file. To set cursor's hot spot two
 * data items must be added to the cursor's theme: "hot_x" and "hot_y" are the
 * offset from the upper-left corner of the cursor (coordinates 0,0).
 *
 * @ingroup Efl_Ui_Cursor
 */
#define EFL_UI_CURSOR_INTERFACE efl_ui_cursor_interface_get()

EWAPI const Efl_Class *efl_ui_cursor_interface_get(void);

/**
 * @brief The cursor to be shown when mouse is over the object
 *
 * This is the cursor that will be displayed when mouse is over the object. The
 * object can have only one cursor set to it so if @ref efl_ui_cursor_set is
 * called twice for an object, the previous set will be unset.
 *
 * If using X cursors, a definition of all the valid cursor names is listed on
 * Elementary_Cursors.h. If an invalid name is set the default cursor will be
 * used.
 *
 * Sets or unsets the current cursor.
 *
 * If @c cursor is @c null this function will reset the cursor to the default
 * one.
 *
 * @param[in] obj The object.
 * @param[in] cursor The cursor name, defined either by the display system or
 * the theme.
 *
 * @return @c true if successful.
 *
 * @ingroup Efl_Ui_Cursor
 */
EOAPI Eina_Bool efl_ui_cursor_set(Eo *obj, const char *cursor);

/**
 * @brief The cursor to be shown when mouse is over the object
 *
 * This is the cursor that will be displayed when mouse is over the object. The
 * object can have only one cursor set to it so if @ref efl_ui_cursor_set is
 * called twice for an object, the previous set will be unset.
 *
 * If using X cursors, a definition of all the valid cursor names is listed on
 * Elementary_Cursors.h. If an invalid name is set the default cursor will be
 * used.
 *
 * Returns the current cursor name.
 *
 * @param[in] obj The object.
 *
 * @return The cursor name, defined either by the display system or the theme.
 *
 * @ingroup Efl_Ui_Cursor
 */
EOAPI const char *efl_ui_cursor_get(const Eo *obj);

/**
 * @brief A different style for the cursor.
 *
 * This only makes sense if theme cursors are used. The cursor should be set
 * with @ref efl_ui_cursor_set first before setting its style with this
 * property.
 *
 * Sets a style for the current cursor. Call after @ref efl_ui_cursor_set.
 *
 * @param[in] obj The object.
 * @param[in] style A specific style to use, eg. default, transparent, ....
 *
 * @return @c true if successful.
 *
 * @ingroup Efl_Ui_Cursor
 */
EOAPI Eina_Bool efl_ui_cursor_style_set(Eo *obj, const char *style);

/**
 * @brief A different style for the cursor.
 *
 * This only makes sense if theme cursors are used. The cursor should be set
 * with @ref efl_ui_cursor_set first before setting its style with this
 * property.
 *
 * Returns the current cursor style name.
 *
 * @param[in] obj The object.
 *
 * @return A specific style to use, eg. default, transparent, ....
 *
 * @ingroup Efl_Ui_Cursor
 */
EOAPI const char *efl_ui_cursor_style_get(const Eo *obj);

/**
 * @brief Whether the cursor may be looked in the theme or not.
 *
 * If @c false, the cursor may only come from the render engine, i.e. from the
 * display manager.
 *
 * Enables or disables theme cursors.
 *
 * @param[in] obj The object.
 * @param[in] allow Whether to use theme cursors.
 *
 * @return @c true if successful.
 *
 * @ingroup Efl_Ui_Cursor
 */
EOAPI Eina_Bool efl_ui_cursor_theme_search_enabled_set(Eo *obj, Eina_Bool allow);

/**
 * @brief Whether the cursor may be looked in the theme or not.
 *
 * If @c false, the cursor may only come from the render engine, i.e. from the
 * display manager.
 *
 * Returns the current state of theme cursors search.
 *
 * @param[in] obj The object.
 *
 * @return Whether to use theme cursors.
 *
 * @ingroup Efl_Ui_Cursor
 */
EOAPI Eina_Bool efl_ui_cursor_theme_search_enabled_get(const Eo *obj);

#endif
