#setup: extraspecial group r^(1+2n) in GL(r^n,q), r prime, r|q-1
#
#the functions in this file:
#
#    extraspecial(r,n,q) 
#    output: the extraspecial group r^(1+2n)
#    (in the case r=2, the output is + type)
#
#    minusextraspecial(r,n,q)
#    output: the group 2^(1+2n), of - type
#
#    c6(r,n,q)
#    output: the normalizer of r^(1+2n) in GL(r^n,q)
#    (in the case r=2, the extraspecial is + type
#
#    minusc6(r,n,q)
#    output: the normalizer of 2^(1+2n) of - type in GL(r^n,q)
#
#the groups in this file: maximal subgroups of Sp(4,p) for p=5,11,13
#as 4-dimensional groups, in lists maximals5, maximals11, maximals13


#constructs r^(1+2n) in GL(r^n,q)
#when r=2, the group is + type (central product of dihedrals)
extraspecial:=function(r,n,q)
local a,b,rho,i,perm,id,id2,gens;

a:=NullMat(r,r,GF(q));
rho:=Z(q)^((q-1)/r);
for i in [1..r] do 
   a[i][i]:=rho^(i-1);
od;
perm:=Concatenation([2..r],[1]);
perm:=PermList(perm);
b:=PermutationMat(perm,r,GF(q));
id:=a^0;
gens:=[a,b];
for i in [1..n-1] do 
   id2:=gens[1]^0;
   gens:=List(gens,x->KroneckerProduct(x,id));
   Add(gens,KroneckerProduct(id2,a));
   Add(gens,KroneckerProduct(id2,b));
od;
return Group(gens);
end;

#constructs the group 2^(1+2n) of minus type
minusextraspecial:=function(r,n,q)
local a,b,c,d,mark,x,y,diff,elmts,i,j,id,id2,gens;

c:=[ [0*Z(q),-Z(q)^0], [Z(q)^0, 0*Z(q)] ];

#this is really inefficient
diff:=Difference(Elements(GF(q)),[0*Z(q)]);
elmts:=Elements(GF(q));
mark:=false;
j:=0;
repeat 
  j:=j+1;
  x:=diff[j];
  i:=0;
  repeat
    i:=i+1;
    y:=elmts[i];
    if x^2+y^2=-Z(q)^0 
       then mark:=true;
    fi;
  until mark or i=q;
until j=q-1 or mark;
d:=[ [x,y], [y,-x] ];

if n=1 then 
  return Group(c,d);
else 
  a:=[ [Z(q)^0, 0*Z(q)], [0*Z(q),-Z(q)^0] ];
  b:=[ [0*Z(q), Z(q)^0], [Z(q)^0, 0*Z(q)] ];
  id:=a^0;

  gens:=[a,b];
  for i in [1..n-2] do 
    id2:=gens[1]^0;
    gens:=List(gens,x->KroneckerProduct(x,id));
    Add(gens,KroneckerProduct(id2,a));
    Add(gens,KroneckerProduct(id2,b));
  od;
  id2:=gens[1]^0;
  gens:=List(gens,x->KroneckerProduct(x,id));
  Add(gens,KroneckerProduct(id2,c));
  Add(gens,KroneckerProduct(id2,d));
  return Group(gens);
fi;

end;

#constructs the normalizer of r^(1+2n) in GL(r^n,q) 
#when r=2, the top group is O^+(2n,2)
c6:=function(r,n,q)
local g,gens,i,j,rho,a,b,c,id,id2,id3,gens2;

g:=extraspecial(r,n,q);
gens:=[];
for i in [1..2*n] do
  gens[i]:=GeneratorsOfGroup(g)[i];
od;

a:=NullMat(r,r,GF(q));
rho:=Z(q)^((q-1)/r);
for i in [1..r] do 
   a[i][i]:=rho^(i*(i-1)/2);
od;

b:=List([1..r],x->[]);
for i in [1..r] do 
  for j in [1..r] do
    b[i][j]:=rho^((i-1)*(j-1));
  od;
od;

c:=NullMat(r^2,r^2,GF(q));
for i in [0..r^2-1] do
  c[i+1][((i+( (i-1) mod r )*r) mod r^2)+1]:=Z(q)^0;
od;

id:=IdentityMat(r,GF(q));
gens2:=[a,b];
for i in [1..n-1] do 
   gens2:=List(gens2,x->KroneckerProduct(x,id));
   id2:=IdentityMat(r^i,GF(q));
   Add(gens2,KroneckerProduct(id2,a));
   Add(gens2,KroneckerProduct(id2,b));
   id3:=IdentityMat(r^(i-1),GF(q));
   Add(gens2,KroneckerProduct(id3,c));
od;

return Group(Concatenation(gens,gens2));
end;

#constructs the normalizer of 2^(1+2n) of - type in GL(2^n,q)
minusc6:=function(r,n,q)
local diff,elmts,mark,x,y,g,gens,c,d,e,u,v,w,i,j,id,id2,id3;

c:=[ [0*Z(q),-Z(q)^0], [Z(q)^0, 0*Z(q)] ];

#this is really inefficient
diff:=Difference(Elements(GF(q)),[0*Z(q)]);
elmts:=Elements(GF(q));
mark:=false;
j:=0;
repeat 
  j:=j+1;
  x:=diff[j];
  i:=0;
  repeat
    i:=i+1;
    y:=elmts[i];
    if x^2+y^2=-Z(q)^0 
       then mark:=true;
    fi;
  until mark or i=q;
until j=q-1 or mark;
d:=[ [x,y], [y,-x] ];

e:=Z(q)^0;
u:=[ [e,e], [-e,e] ];
v:=[ [ e+x+y, e-x+y ], [ -e-x+y, e-x-y ] ];

if n=1 then return Group(c,d,u,v); fi;

w:=[ [e,0*e,e,0*e], [0*e,e,0*e,e], [0*e,e,0*e,-e], [-e,0*e,e,0*e] ];
g:=c6(r,n-1,q);
id:=IdentityMat(r,GF(q));
id2:=IdentityMat(r^(n-1),GF(q));
id3:=IdentityMat(r^(n-2),GF(q));
gens:=List(GeneratorsOfGroup(g),z->KroneckerProduct(z,id));
Add(gens,KroneckerProduct(id2,c));
Add(gens,KroneckerProduct(id2,d));
Add(gens,KroneckerProduct(id2,u));
Add(gens,KroneckerProduct(id2,v));
Add(gens,KroneckerProduct(id3,w));

return(Group(gens));

end;




	f := GF(5);

maximals5 := [
    Group([ [
        [3,4,3,1],
        [0,2,3,3],
        [2,0,1,1],
        [3,2,0,0] ]*One(f), [

        [0,3,3,2],
        [1,3,2,3],
        [1,0,4,2],
        [4,1,4,2] ]*One(f), [

        [0,2,2,4],
        [3,0,1,3],
        [3,4,0,3],
        [1,3,3,2] ]*One(f), [

        [0,0,4,4],
        [1,4,1,3],
        [0,0,4,0],
        [1,0,3,1] ]*One(f), [

        [1,1,3,2],
        [3,3,2,3],
        [4,2,0,4],
        [3,4,2,2] ]*One(f), [

        [4,0,0,0],
        [0,4,0,0],
        [0,0,4,0],
        [0,0,0,4] ]*One(f), [

        [0,0,1,1],
        [4,1,3,0],
        [0,0,1,0],
        [4,0,0,0] ]*One(f) ] ),

    Group([ [
        [3,4,3,1],
        [0,2,3,3],
        [2,0,1,1],
        [3,2,0,0] ]*One(f), [

        [0,2,2,4],
        [3,0,1,3],
        [3,4,0,3],
        [1,3,3,2] ]*One(f), [

        [1,1,3,2],
        [3,3,2,3],
        [4,2,0,4],
        [3,4,2,2] ]*One(f), [

        [4,0,0,0],
        [0,4,0,0],
        [0,0,4,0],
        [0,0,0,4] ]*One(f), [

        [3,0,4,2],
        [3,4,3,4],
        [0,0,4,0],
        [2,0,2,0] ]*One(f), [

        [1,4,4,1],
        [2,1,3,4],
        [2,1,0,1],
        [3,2,3,0] ]*One(f), [

        [0,0,1,1],
        [4,1,3,0],
        [0,0,1,0],
        [4,0,0,0] ]*One(f) ]),

    Group([ [
        [0,1,1,1],
        [3,1,4,2],
        [4,4,3,4],
        [3,1,0,0] ]*One(f), [

        [0,1,2,0],
        [3,0,0,3],
        [4,0,0,1],
        [0,1,3,0] ]*One(f), [

        [1,1,3,1],
        [3,1,4,2],
        [3,4,1,4],
        [0,1,4,0] ]*One(f), [

        [4,0,0,0],
        [0,4,0,0],
        [0,0,4,0],
        [0,0,0,4] ]*One(f) ] ),

    Group([ [
        [0,2,2,4],
        [3,0,1,3],
        [3,4,0,3],
        [1,3,3,2] ]*One(f), [

        [4,2,2,3],
        [4,1,3,2],
        [0,0,4,3],
        [0,0,1,1] ]*One(f), [

        [4,0,0,0],
        [0,4,0,0],
        [0,0,4,0],
        [0,0,0,4] ]*One(f), [

        [1,4,4,1],
        [2,1,3,4],
        [2,1,0,1],
        [3,2,3,0] ]*One(f) ] ),

    Group([ [
        [1,1,1,4],
        [3,0,1,1],
        [3,3,1,4],
        [2,3,2,0] ]*One(f), [

        [3,4,1,0],
        [0,2,0,4],
        [2,0,2,4],
        [0,3,0,3] ]*One(f), [

        [4,0,0,0],
        [0,4,0,0],
        [0,0,4,0],
        [0,0,0,4] ]*One(f), [

        [0,3,2,1],
        [3,3,3,2],
        [2,3,1,2],
        [1,2,2,4] ]*One(f) ] ),

    Group([ [
        [1,1,1,4],
        [1,1,4,1],
        [1,4,4,4],
        [4,1,4,4] ]*One(f), [

        [4,3,2,1],
        [3,3,2,2],
        [2,0,3,2],
        [4,2,2,2] ]*One(f), [

        [4,0,0,0],
        [0,4,0,0],
        [0,0,4,0],
        [0,0,0,4] ]*One(f), [

        [0,1,2,3],
        [3,1,1,3],
        [4,2,1,3],
        [2,3,3,2] ]*One(f)  ]),

    Group([ [
        [2,1,4,4],
        [0,4,3,4],
        [3,4,0,4],
        [4,3,0,2] ]*One(f), [

        [3,2,3,4],
        [4,0,3,0],
        [4,4,1,2],
        [4,4,0,3] ]*One(f), [

        [2,3,3,2],
        [2,1,0,1],
        [3,3,3,3],
        [3,3,4,2] ]*One(f), [

        [2,3,4,3],
        [2,3,0,4],
        [3,1,3,2],
        [3,3,3,4] ]*One(f), [

        [0,4,3,0],
        [4,0,0,2],
        [0,0,0,4],
        [0,0,4,0] ]*One(f), [

        [4,0,0,0],
        [0,4,0,0],
        [0,0,4,0],
        [0,0,0,4] ]*One(f), [

        [4,2,4,2],
        [3,3,1,4],
        [0,4,2,3],
        [1,0,2,1] ]*One(f), [

        [4,1,4,2],
        [0,3,1,4],
        [3,3,2,4],
        [3,3,0,1] ]*One(f), [

        [0,0,4,3],
        [4,2,2,4],
        [3,4,3,0],
        [4,3,1,0] ]*One(f) ] ),

    Group([ [
        [2,3,1,1],
        [1,4,1,1],
        [4,4,0,4],
        [4,2,4,0] ]*One(f), [

        [3,2,3,3],
        [0,0,2,2],
        [3,4,3,0],
        [4,1,1,3] ]*One(f), [

        [4,0,0,0],
        [0,4,0,0],
        [0,0,4,0],
        [0,0,0,4] ]*One(f) ] )
];

magmasp45:=Group(
 [ [2, 0, 0, 0],
    [0, 1, 0, 0],
    [0, 0, 1, 0],
    [0, 0, 0, 3]] *One(f),

  [  [1, 0, 1, 0],
    [1, 0, 0, 0],
    [0, 1, 0, 1],
    [0, 4, 0, 0] ]*One(f));

  f := GF(7);
maximals7 := [
  Group([[ 
    [4,3,1,0],
    [2,6,0,5],
    [1,0,6,6],
    [2,3,1,2] 
]*One(f),
[
    [1,0,1,5],
    [3,3,4,4],
    [2,2,3,5],
    [2,0,1,4] 
]*One(f),
[
    [6,0,0,0],
    [0,6,0,0],
    [0,0,6,0],
    [0,0,0,6] 
]*One(f),
[
    [4,4,2,2],
    [5,6,0,2],
    [4,0,6,3],
    [5,4,2,1] 
]*One(f),
[
    [2,3,5,6],
    [2,0,3,5],
    [3,2,2,4],
    [1,3,5,0] 
]*One(f),
[
    [2,2,3,0],
    [6,3,5,3],
    [5,5,6,5],
    [2,5,1,0] 
]*One(f),
[
    [5,3,1,1],
    [0,0,2,3],
    [0,3,2,5],
    [4,1,5,1] 
]*One(f)]),
  Group([[ 
    [4,3,1,0],
    [2,6,0,5],
    [1,0,6,6],
    [2,3,1,2]
]*One(f),
[
    [1,0,1,5],
    [3,3,4,4],
    [2,2,3,5],
    [2,0,1,4]
]*One(f),
[
    [6,0,0,0],
    [0,6,0,0],
    [0,0,6,0],
    [0,0,0,6]
]*One(f),
[
    [6,0,0,0],
    [3,5,2,0],
    [5,3,0,0],
    [6,5,4,6]
]*One(f),
[
    [2,3,5,6],
    [2,0,3,5],
    [3,2,2,4],
    [1,3,5,0]
]*One(f),
[
    [2,2,3,0],
    [6,3,5,3],
    [5,5,6,5],
    [2,5,1,0]
]*One(f),
[
    [2,4,0,4],
    [1,0,1,0],
    [6,2,1,3],
    [0,6,6,6]
]*One(f)]) ,
  Group([[ 
    [0,0,2,0],
    [2,0,0,5],
    [4,0,0,0],
    [0,3,2,0]
]*One(f),
[
    [6,0,0,0],
    [0,6,0,0],
    [0,0,6,0],
    [0,0,0,6]
]*One(f),
[
    [0,4,6,1],
    [3,6,1,2],
    [6,0,5,6],
    [5,3,6,4]
]*One(f),
[
    [4,1,5,2],
    [5,4,0,4],
    [4,2,5,5],
    [0,5,6,5]
]*One(f)]),
  Group([[ 
    [4,0,3,2],
    [2,5,2,3],
    [4,2,2,0],
    [3,4,5,3]
]*One(f),
[
    [1,0,1,5],
    [3,3,4,4],
    [2,2,3,5],
    [2,0,1,4]
]*One(f),
[
    [6,0,0,0],
    [0,6,0,0],
    [0,0,6,0],
    [0,0,0,6]
]*One(f),
[
    [2,4,0,4],
    [1,0,1,0],
    [6,2,1,3],
    [0,6,6,6]
]*One(f)]),
  Group([[ 
    [2,2,3,5],
    [3,2,4,3],
    [0,4,1,5],
    [6,0,4,1]
]*One(f),
[
    [6,0,0,0],
    [0,6,0,0],
    [0,0,6,0],
    [0,0,0,6]
]*One(f),
[
    [0,2,2,3],
    [4,3,2,2],
    [6,1,3,5],
    [0,6,3,6]
]*One(f),
[
    [6,2,4,0],
    [4,1,0,3],
    [5,0,1,2],
    [0,2,4,6]
]*One(f)]),
  Group([[ 
    [4,1,0,5],
    [2,6,3,0],
    [5,1,1,6],
    [6,5,5,3]
]*One(f),
[
    [6,0,0,0],
    [0,6,0,0],
    [0,0,6,0],
    [0,0,0,6]
]*One(f),
[
    [1,2,4,6],
    [6,1,6,0],
    [6,2,5,6],
    [6,6,2,1]
]*One(f),
[
    [3,2,4,3],
    [4,4,1,4],
    [2,6,4,5],
    [4,2,3,5]
]*One(f)]),
  Group([[ 
    [5,3,3,0],
    [5,2,0,4],
    [1,0,2,3],
    [0,6,5,5]
]*One(f),
[
    [2,2,6,4],
    [1,6,3,3],
    [2,2,2,1],
    [3,1,4,2]
]*One(f),
[
    [5,2,4,1],
    [2,4,6,4],
    [5,1,6,5],
    [0,5,5,5]
]*One(f),
[
    [0,1,2,4],
    [1,3,3,2],
    [6,4,4,6],
    [0,6,6,0]
]*One(f),
[
    [3,2,5,5],
    [0,3,0,5],
    [5,1,4,5],
    [0,5,0,4]
]*One(f),
[
    [4,0,2,2],
    [5,5,2,5],
    [5,3,1,4],
    [2,6,4,6]
]*One(f),
[
    [6,0,0,0],
    [0,6,0,0],
    [0,0,6,0],
    [0,0,0,6]
]*One(f),
[
    [5,5,2,0],
    [2,2,0,5],
    [4,0,2,5],
    [0,3,2,5]
]*One(f),
[
    [3,5,2,6],
    [6,5,4,4],
    [4,4,2,5],
    [4,3,4,6]
]*One(f),
[
    [5,5,3,6],
    [5,6,1,3],
    [2,6,4,2],
    [0,2,2,5]
]*One(f),
[
    [0,6,2,2],
    [5,2,0,2],
    [1,3,6,1],
    [1,1,2,1]
]*One(f),
[
    [4,3,6,4],
    [3,4,4,6],
    [4,5,3,4],
    [5,4,4,3]
]*One(f),
[
    [1,3,5,0],
    [3,6,0,2],
    [1,0,6,3],
    [0,6,3,1]
]*One(f)]),
  Group([[ 
    [6,1,6,4],
    [0,4,2,6],
    [2,5,4,1],
    [6,0,2,5]
]*One(f),
[
    [6,0,0,0],
    [0,6,0,0],
    [0,0,6,0],
    [0,0,0,6]
]*One(f),
[
    [4,2,0,1],
    [2,0,1,0],
    [0,2,4,5],
    [2,0,5,0]
]*One(f),
[
    [1,6,4,6],
    [3,0,0,4],
    [4,3,0,1],
    [1,4,4,6]
]*One(f),
[
    [5,6,3,2],
    [2,1,1,3],
    [6,1,0,3],
    [5,4,0,6]
]*One(f),
[
    [6,2,1,0],
    [2,1,0,6],
    [3,0,1,2],
    [0,4,2,6]
]*One(f),
[
    [2,5,0,0],
    [5,5,0,0],
    [5,0,5,5],
    [0,2,5,2]
]*One(f),
[
    [2,0,0,3],
    [4,6,6,6],
    [6,4,3,2],
    [3,3,5,5]
]*One(f),
[
    [6,6,5,0],
    [6,1,0,2],
    [4,0,1,6],
    [0,3,6,6]
]*One(f),
[
    [0,4,1,5],
    [5,3,0,1],
    [1,1,5,3],
    [4,1,2,1]
]*One(f),
[
    [1,1,0,4],
    [1,6,4,0],
    [0,1,1,6],
    [1,0,6,6]
]*One(f),
[
    [1,1,6,6],
    [0,4,4,6],
    [6,6,3,6],
    [3,6,0,6]
]*One(f),
[
    [0,5,0,6],
    [5,4,6,0],
    [0,5,0,2],
    [5,0,2,4]
]*One(f)]),
  Group([[ 
    [2,1,4,6],
    [5,0,0,1],
    [5,2,2,4],
    [3,0,1,2]
]*One(f),
[
    [6,0,0,0],
    [0,6,0,0],
    [0,0,6,0],
    [0,0,0,6]
]*One(f),
[
    [3,1,5,0],
    [4,6,0,3],
    [2,0,6,5],
    [5,1,5,5]
]*One(f)]),
  Group([[ 
    [6,0,0,0],
    [0,6,0,0],
    [0,0,6,0],
    [0,0,0,6]
]*One(f),
[
    [4,0,6,3],
    [1,4,0,5],
    [2,3,3,6],
    [6,3,3,4]
]*One(f),
[
    [1,3,4,5],
    [6,2,3,1],
    [3,1,1,2],
    [1,3,0,4]
]*One(f)]),
  Group([[ 
    [2,0,0,2],
    [6,1,2,6],
    [1,4,2,1],
    [4,6,2,1]
]*One(f),
[
    [6,0,0,0],
    [0,6,0,0],
    [0,0,6,0],
    [0,0,0,6]
]*One(f),
[
    [5,0,0,0],
    [0,5,0,0],
    [0,5,3,0],
    [2,0,0,3]
]*One(f)])];

  f := GF(11);
maximals11 := [
Group([[ 
    [10,0,0,0],
    [0,10,0,0],
    [0,0,10,0],
    [0,0,0,10]
]*One(f),
[
    [7,8,3,7],
    [4,5,3,9],
    [1,5,7,3],
    [7,9,1,2]
]*One(f),
[
    [6,9,9,1],
    [1,5,4,9],
    [10,7,8,2],
    [8,10,10,7]
]*One(f),
[
    [9,9,2,8],
    [6,8,4,8],
    [8,5,5,2],
    [0,5,10,1]
]*One(f),
[
    [7,3,7,1],
    [7,8,1,7],
    [2,5,1,8],
    [0,2,4,2]
]*One(f),
[
    [9,8,1,1],
    [4,0,7,4],
    [2,2,4,3],
    [5,6,4,10]
]*One(f),
[
    [10,9,1,3],
    [6,2,6,1],
    [9,4,0,2],
    [7,9,5,3]
]*One(f)]),
  Group([[ 
    [10,0,0,0],
    [0,10,0,0],
    [0,0,10,0],
    [0,0,0,10]
]*One(f),
[
    [7,8,3,7],
    [4,5,3,9],
    [1,5,7,3],
    [7,9,1,2]
]*One(f),
[
    [6,9,9,1],
    [1,5,4,9],
    [10,7,8,2],
    [8,10,10,7]
]*One(f),
[
    [9,9,2,8],
    [6,8,4,8],
    [8,5,5,2],
    [0,5,10,1]
]*One(f),
[
    [7,3,7,1],
    [7,8,1,7],
    [2,5,1,8],
    [0,2,4,2]
]*One(f),
[
    [1,2,4,3],
    [0,0,9,4],
    [0,6,2,9],
    [0,0,0,1]
]*One(f),
[
    [2,5,3,3],
    [7,2,2,3],
    [5,1,10,6],
    [8,5,4,10]
]*One(f)]),
  Group([[ 
    [10,0,0,0],
    [0,10,0,0],
    [0,0,10,0],
    [0,0,0,10]
]*One(f),
[
    [6,0,3,8],
    [3,0,4,4],
    [7,4,5,4],
    [0,9,4,1]
]*One(f),
[
    [9,6,8,7],
    [0,8,4,2],
    [7,5,3,3],
    [7,6,2,1]
]*One(f),
[
    [10,4,6,0],
    [8,1,0,5],
    [2,0,1,4],
    [0,9,8,10]
]*One(f)]),
  Group([[ 
    [10,0,0,0],
    [0,10,0,0],
    [0,0,10,0],
    [0,0,0,10]
]*One(f),
[
    [7,8,3,7],
    [4,5,3,9],
    [1,5,7,3],
    [7,9,1,2]
]*One(f),
[
    [2,5,3,3],
    [7,2,2,3],
    [5,1,10,6],
    [8,5,4,10]
]*One(f),
[
    [9,5,10,9],
    [6,7,9,10],
    [1,1,4,6],
    [6,1,5,2]
]*One(f)]),
  Group([[ 
    [10,0,0,0],
    [0,10,0,0],
    [0,0,10,0],
    [0,0,0,10]
]*One(f),
[
    [5,1,1,0],
    [3,3,9,1],
    [1,9,3,10],
    [7,1,8,1]
]*One(f),
[
    [1,10,10,0],
    [1,10,0,1],
    [10,0,10,10],
    [0,1,1,1]
]*One(f),
[
    [8,2,4,2],
    [10,10,3,4],
    [3,0,0,9],
    [8,3,1,2]
]*One(f)]),
  Group([[ 
    [10,0,0,0],
    [0,10,0,0],
    [0,0,10,0],
    [0,0,0,10]
]*One(f),
[
    [9,5,10,8],
    [2,3,3,1],
    [8,2,0,2],
    [8,3,3,4]
]*One(f),
[
    [10,5,6,5],
    [3,2,0,6],
    [8,8,10,6],
    [0,8,8,2]
]*One(f),
[
    [9,6,9,7],
    [8,2,8,9],
    [6,10,9,5],
    [2,6,3,2]
]*One(f)]),
  Group([[ 
    [6,1,8,8],
    [7,9,5,4],
    [5,1,6,7],
    [3,5,3,3]
]*One(f),
[
    [7,9,3,3],
    [2,10,9,3],
    [2,2,1,2],
    [1,2,9,4]
]*One(f),
[
    [9,9,8,7],
    [1,8,3,0],
    [4,5,4,6],
    [3,6,6,3]
]*One(f),
[
    [0,7,3,0],
    [4,0,0,8],
    [2,0,0,7],
    [0,9,4,0]
]*One(f),
[
    [10,0,0,0],
    [0,10,0,0],
    [0,0,10,0],
    [0,0,0,10]
]*One(f),
[
    [7,5,7,0],
    [2,1,7,7],
    [3,3,9,6],
    [7,3,9,3]
]*One(f),
[
    [8,8,4,0],
    [10,3,0,7],
    [0,0,3,8],
    [0,0,10,8]
]*One(f),
[
    [2,4,9,7],
    [9,7,9,1],
    [5,2,5,0],
    [2,7,9,10]
]*One(f),
[
    [10,9,6,9],
    [8,1,2,6],
    [1,4,10,2],
    [7,1,3,1]
]*One(f)]),
  Group([[ 
    [10,0,0,0],
    [0,10,0,0],
    [0,0,10,0],
    [0,0,0,10]
]*One(f),
[
    [4,9,9,1],
    [1,3,6,10],
    [6,7,8,2],
    [6,8,4,8]
]*One(f),
[
    [3,0,1,4],
    [5,4,9,8],
    [5,2,1,0],
    [1,7,10,4]
]*One(f)]),
  Group([[ 
    [10,0,0,0],
    [0,10,0,0],
    [0,0,10,0],
    [0,0,0,10]
]*One(f),
[
    [8,10,7,0],
    [0,3,0,4],
    [2,0,3,10],
    [0,9,0,8]
]*One(f),
[
    [7,3,10,3],
    [10,6,5,6],
    [0,1,8,9],
    [5,7,2,2]
]*One(f)]),
  Group([[ 
    [10,0,0,0],
    [0,10,0,0],
    [0,0,10,0],
    [0,0,0,10]
]*One(f),
[
    [4,9,4,0],
    [9,7,0,7],
    [9,0,7,9],
    [0,2,9,4]
]*One(f),
[
    [9,7,7,4],
    [6,2,7,0],
    [10,1,6,3],
    [5,3,4,4]
] * One(f)])];
 
  f := GF(13);
maximals13 := [
  Group([[ 
    [7,0,7,3],
    [3,2,3,7],
    [2,1,9,0],
    [0,2,10,4]
]*One(f),
[
    [12,0,0,0],
    [0,12,0,0],
    [0,0,12,0],
    [0,0,0,12]
]*One(f),
[
    [12,10,6,1],
    [9,7,7,6],
    [6,1,8,3],
    [0,6,4,3]
]*One(f),
[
    [1,1,5,9],
    [0,3,10,5],
    [0,10,12,12],
    [0,0,0,1]
]*One(f),
[
    [4,9,7,8],
    [3,1,1,12],
    [12,5,4,1],
    [6,3,4,2]
]*One(f),
[
    [11,11,0,9],
    [7,6,6,8],
    [9,11,0,5],
    [0,5,12,7]
]*One(f),
[
    [10,10,6,2],
    [0,9,11,10],
    [9,1,7,11],
    [8,7,3,12]
]*One(f)]),
  Group([[ 
    [3,7,4,3],
    [2,3,12,4],
    [2,9,9,6],
    [10,2,11,9]
]*One(f),
[
    [12,0,0,0],
    [0,12,0,0],
    [0,0,12,0],
    [0,0,0,12]
]*One(f),
[
    [12,10,6,1],
    [9,7,7,6],
    [6,1,8,3],
    [0,6,4,3]
]*One(f),
[
    [1,1,5,9],
    [0,3,10,5],
    [0,10,12,12],
    [0,0,0,1]
]*One(f),
[
    [4,9,7,8],
    [3,1,1,12],
    [12,5,4,1],
    [6,3,4,2]
]*One(f),
[
    [2,10,12,9],
    [4,8,3,12],
    [1,12,3,3],
    [12,1,9,9]
]*One(f),
[
    [11,11,0,9],
    [7,6,6,8],
    [9,11,0,5],
    [0,5,12,7]
]*One(f)]),
  Group([[ 
    [12,0,0,0],
    [0,12,0,0],
    [0,0,12,0],
    [0,0,0,12]
]*One(f),
[
    [7,11,9,9],
    [1,9,9,8],
    [12,7,1,8],
    [5,0,5,8]
]*One(f),
[
    [10,9,0,9],
    [7,9,2,11],
    [12,9,0,3],
    [12,1,5,9]
]*One(f),
[
    [5,3,0,0],
    [5,8,0,0],
    [5,0,8,3],
    [0,8,5,5]
]*One(f)]),
  Group([[ 
    [3,7,4,3],
    [2,3,12,4],
    [2,9,9,6],
    [10,2,11,9]
]*One(f),
[
    [12,0,0,0],
    [0,12,0,0],
    [0,0,12,0],
    [0,0,0,12]
]*One(f),
[
    [2,2,10,1],
    [6,7,2,10],
    [2,11,6,11],
    [2,2,7,11]
]*One(f),
[
    [11,11,0,9],
    [7,6,6,8],
    [9,11,0,5],
    [0,5,12,7]
]*One(f)]),
  Group([[ 
    [3,3,12,8],
    [5,11,2,12],
    [3,3,10,10],
    [10,3,8,5]
]*One(f),
[
    [12,0,0,0],
    [0,12,0,0],
    [0,0,12,0],
    [0,0,0,12]
]*One(f),
[
    [12,6,4,0],
    [9,1,0,9],
    [6,0,1,6],
    [0,7,9,12]
]*One(f),
[
    [12,10,11,7],
    [5,8,3,11],
    [12,7,6,3],
    [7,12,8,2]
]*One(f)]),
  Group([[ 
    [12,0,0,0],
    [0,12,0,0],
    [0,0,12,0],
    [0,0,0,12]
]*One(f),
[
    [12,7,5,6],
    [6,3,12,7],
    [4,0,5,11],
    [6,11,4,1]
]*One(f),
[
    [0,2,5,5],
    [9,6,5,5],
    [11,0,8,11],
    [6,11,4,1]
]*One(f),
[
    [12,2,7,0],
    [6,6,5,7],
    [11,6,7,11],
    [12,11,7,1]
]*One(f)]),
  Group([[ 
    [12,7,5,2],
    [5,7,7,4],
    [3,1,12,8],
    [4,8,6,6]
]*One(f),
[
    [11,6,4,4],
    [3,1,4,4],
    [1,7,12,7],
    [3,1,10,2]
]*One(f),
[
    [12,0,0,0],
    [0,12,0,0],
    [0,0,12,0],
    [0,0,0,12]
]*One(f),
[
    [10,8,1,1],
    [4,0,9,1],
    [10,1,0,5],
    [0,10,9,3]
]*One(f),
[
    [4,0,0,0],
    [5,1,10,0],
    [11,9,0,0],
    [8,11,8,10]
]*One(f),
[
    [3,5,12,12],
    [10,10,7,12],
    [1,12,3,8],
    [7,1,3,10]
]*One(f),
[
    [1,0,2,7],
    [11,1,5,1],
    [7,9,5,2],
    [10,12,0,4]
]*One(f),
[
    [7,11,12,0],
    [2,6,0,1],
    [5,0,6,11],
    [0,8,2,7]
]*One(f),
[
    [9,2,10,10],
    [7,2,0,10],
    [10,1,12,11],
    [6,10,6,5]
]*One(f)]),
  Group([[ 
    [12,0,0,0],
    [0,12,0,0],
    [0,0,12,0],
    [0,0,0,12]
]*One(f),
[
    [2,9,5,0],
    [12,12,0,4],
    [5,1,3,2],
    [6,12,10,5]
]*One(f),
[
    [4,9,10,6],
    [1,3,9,5],
    [6,10,12,7],
    [0,2,4,6]
]*One(f)]),
  Group([[ 
    [1,9,12,0],
    [0,12,0,1],
    [0,0,12,9],
    [0,0,0,1]
]*One(f),
[
    [12,0,0,0],
    [0,12,0,0],
    [0,0,12,0],
    [0,0,0,12]
]*One(f),
[
    [10,12,4,8],
    [2,5,10,2],
    [6,5,0,12],
    [9,5,7,10]
]*One(f)]),
  Group([[ 
    [12,0,0,0],
    [0,12,0,0],
    [0,0,12,0],
    [0,0,0,12]
]*One(f),
[
    [11,8,10,0],
    [1,2,0,3],
    [8,0,2,8],
    [0,5,1,11]
]*One(f),
[
    [7,2,11,8],
    [10,0,12,8],
    [12,1,3,7],
    [0,6,7,2]
]*One(f)] )];
