# Copyright (c) 2014-2018 Cedric Bellegarde <cedric.bellegarde@adishatz.org>
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.

from gi.repository import Gtk, GLib, Gio

from gettext import gettext as _

from lollypop.define import App, ResponsiveType, Type
from lollypop.helper_task import TaskHelper
from lollypop.view_albums_list import AlbumsListView
from lollypop.search import Search


class SearchPopover(Gtk.Popover):
    """
        Popover allowing user to search for tracks/albums
    """

    def __init__(self):
        """
            Init Popover
        """
        Gtk.Popover.__init__(self)
        self.set_position(Gtk.PositionType.BOTTOM)
        self.connect("map", self.__on_map)
        self.connect("unmap", self.__on_unmap)
        self.__timeout_id = None
        self.__current_search = ""
        self.__cancellable = Gio.Cancellable()
        self.__history = []
        self.__width = 0

        builder = Gtk.Builder()
        builder.add_from_resource("/org/gnome/Lollypop/SearchPopover.ui")
        self.__widget = builder.get_object("widget")
        self.__new_btn = builder.get_object("new_btn")
        self.__entry = builder.get_object("entry")

        self.__spinner = builder.get_object("spinner")
        self.__header_stack = builder.get_object("header_stack")
        self.__stack = builder.get_object("stack")
        self.__placeholder = builder.get_object("placeholder")

        # Connect here because we don"t want previous switch.set_state()
        # to emit a signal on init
        builder.connect_signals(self)

        self.__view = AlbumsListView(ResponsiveType.SEARCH)
        self.__view.show()
        self.__stack.add_named(self.__view, "view")

        self.__set_default_placeholder()

        self.add(self.__widget)

    def set_text(self, text):
        """
            Set search text
        """
        self.__entry.set_text(text)

    def do_get_preferred_width(self):
        if self.__width == 0:
            return Gtk.Popover.do_get_preferred_width(self)
        else:
            return (self.__width, self.__width)

#######################
# PROTECTED           #
#######################
    def _on_new_btn_clicked(self, button):
        """
            Create a new playlist based on search
            @param button as Gtk.Button
        """
        button.set_sensitive(False)
        helper = TaskHelper()
        helper.run(self.__search_to_playlist)

    def _on_search_changed(self, widget):
        """
            Timeout filtering
            @param widget as Gtk.TextEntry
        """
        if self.__timeout_id:
            GLib.source_remove(self.__timeout_id)
            self.__timeout_id = None
        self.__cancellable.cancel()
        self.__view.stop()
        self.__current_search = widget.get_text().strip()
        self.__timeout_id = GLib.timeout_add(
                500,
                self.__on_search_changed_timeout)

#######################
# PRIVATE             #
#######################
    def __set_no_result_placeholder(self):
        """
            Set placeholder for no result
        """
        self.__placeholder.set_markup(
            _("<big>No results for this search</big>"))

    def __set_default_placeholder(self):
        """
            Set placeholder for no result
        """
        self.__placeholder.set_markup(
            _("<big>Search for artists, albums and tracks</big>"))

    def __populate(self):
        """
            Populate searching items
            in db based on text entry current text
        """
        self.__cancellable.reset()
        self.__header_stack.set_visible_child(self.__spinner)
        self.__spinner.start()
        self.__history = []
        if self.__current_search:
            search = Search()
            search.get(self.__current_search,
                       self.__cancellable,
                       callback=(self.__on_search_get,))
        else:
            self.__stack.set_visible_child_name("placeholder")
            self.__set_default_placeholder()
            self.__header_stack.set_visible_child(self.__new_btn)
            GLib.idle_add(self.__spinner.stop)

    def __search_to_playlist(self):
        """
            Create a new playlist based on search
        """
        tracks = []
        for child in self.__view.children:
            tracks += child.album.tracks
        if tracks:
            playlist_id = App().playlists.get_id(self.__current_search)
            if playlist_id == Type.NONE:
                App().playlists.add(self.__current_search)
                playlist_id = App().playlists.get_id(self.__current_search)
            App().playlists.add_tracks(playlist_id, tracks)

    def __on_search_get(self, result):
        """
            Add rows for internal results
            @param result as [(int, Album, bool)]
        """
        if result:
            albums = []
            reveal_albums = []
            for (score, album, in_tracks) in result:
                albums.append(album)
                if in_tracks:
                    reveal_albums.append(album.id)
            self.__view.set_reveal(reveal_albums)
            self.__view.populate(albums)
            self.__stack.set_visible_child_name("view")
        else:
            self.__stack.set_visible_child_name("placeholder")
            self.__set_no_result_placeholder()
        self.__header_stack.set_visible_child(self.__new_btn)
        GLib.idle_add(self.__spinner.stop)

    def __on_map(self, widget):
        """
            Disable global shortcuts and resize
            @param widget as Gtk.Widget
        """
        window_size = App().window.get_size()
        height = window_size[1]
        self.__width = min(500, window_size[0])
        self.set_size_request(self.__width, height * 0.7)

    def __on_unmap(self, widget):
        """
            Enable global shortcuts
            @param widget as Gtk.Widget
        """
        self.__cancellable.cancel()
        self.__view.stop()
        self.__header_stack.set_visible_child(self.__new_btn)
        self.__spinner.stop()

    def __on_search_changed_timeout(self):
        """
            Populate widget
        """
        if self.__view.children:
            self.__view.stop()
            self.__view.clear()
            return True
        self.__timeout_id = None
        self.__populate()
        if self.__current_search != "":
            self.__new_btn.set_sensitive(True)
        else:
            self.__new_btn.set_sensitive(False)
