"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const debug_1 = require("debug");
const debug = debug_1.default('axm:profilingaction');
const profiling_1 = require("../features/profiling");
const miscellaneous_1 = require("../utils/miscellaneous");
const serviceManager_1 = require("../serviceManager");
class ProfilingCPUAction {
    constructor(actionFeature) {
        this.actionFeature = actionFeature;
    }
    init() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            this.profilingFeature = new profiling_1.default();
            this.profilings = this.profilingFeature.init();
            try {
                yield this.profilings.cpuProfiling.init();
                // expose actions only if the feature is available
                this.exposeActions();
            }
            catch (err) {
                debug(`Failed to load cpu profiler: ${err.message}`);
            }
        });
    }
    destroy() {
        if (this.profilingFeature)
            this.profilingFeature.destroy();
    }
    stopProfiling(reply) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const data = yield this.profilings.cpuProfiling.stop();
                return reply({
                    success: true,
                    cpuprofile: true,
                    dump_file: data,
                    dump_file_size: data.length,
                    uuid: this.uuid
                });
            }
            catch (err) {
                return reply({
                    success: false,
                    err: err,
                    uuid: this.uuid
                });
            }
        });
    }
    exposeActions() {
        const profilingReply = (data) => serviceManager_1.ServiceManager.get('transport').send('profilings', {
            data: data.dump_file,
            at: data.at,
            initiated: data.initiated || 'manual',
            duration: data.duration || null,
            type: 'cpuprofile'
        });
        let startTime = null;
        this.actionFeature.action('km:cpu:profiling:start', (opts, reply) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            startTime = new Date();
            if (!reply) {
                reply = opts;
                opts = {};
            }
            if (!opts)
                opts = {};
            try {
                this.uuid = miscellaneous_1.default.generateUUID();
                yield this.profilings.cpuProfiling.start();
                reply({ success: true, uuid: this.uuid });
                if (opts.timeout && typeof opts.timeout === 'number') {
                    setTimeout((_) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                        yield this.stopProfiling(data => profilingReply(Object.assign({ at: startTime, initiated: opts.initiated, duration: startTime ? new Date().getTime() - startTime.getTime() : null }, data)));
                    }), opts.timeout);
                }
            }
            catch (err) {
                return reply({
                    success: false,
                    err: err,
                    uuid: this.uuid
                });
            }
        }));
        this.actionFeature.action('km:cpu:profiling:stop', this.stopProfiling.bind(this, data => profilingReply(Object.assign({ at: startTime, initiated: 'manual', duration: startTime ? new Date().getTime() - startTime.getTime() : null }, data))));
    }
}
exports.default = ProfilingCPUAction;
//# sourceMappingURL=data:application/json;base64,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