/**
 * Nextcloud - contacts
 *
 * This file is licensed under the Affero General Public License version 3 or
 * later. See the COPYING file.
 *
 * @author Hendrik Leppelsack <hendrik@leppelsack.de>
 * @copyright Hendrik Leppelsack 2015
 */

angular.module('contactsApp', ['uuid4', 'angular-cache', 'ngRoute', 'ui.bootstrap', 'ui.select', 'ngSanitize', 'angular-click-outside', 'ngclipboard'])
.config(['$routeProvider', function($routeProvider) {

	$routeProvider.when('/:gid', {
		template: '<contactdetails></contactdetails>'
	});

	$routeProvider.when('/contact/:uid', {
		redirectTo: function(parameters) {
			return '/' + t('contacts', 'All contacts') + '/' + parameters.uid;
		}
	});

	$routeProvider.when('/:gid/:uid', {
		template: '<contactdetails></contactdetails>'
	});

	$routeProvider.otherwise('/' + t('contacts', 'All contacts'));

}]);

angular.module('contactsApp')
.directive('datepicker', ['$timeout', function($timeout) {
	var loadDatepicker = function (scope, element, attrs, ngModelCtrl) {
		$timeout(function() {
			element.datepicker({
				dateFormat:'yy-mm-dd',
				minDate: null,
				maxDate: null,
				constrainInput: false,
				onSelect:function (date, dp) {
					if (dp.selectedYear < 1000) {
						date = '0' + date;
					}
					if (dp.selectedYear < 100) {
						date = '0' + date;
					}
					if (dp.selectedYear < 10) {
						date = '0' + date;
					}
					ngModelCtrl.$setViewValue(date);
					scope.$apply();
				}
			});
		});
	};
	return {
		restrict: 'A',
		require : 'ngModel',
		transclude: true,
		link : loadDatepicker
	};
}]);

angular.module('contactsApp')
.directive('focusExpression', ['$timeout', function ($timeout) {
	return {
		restrict: 'A',
		link: {
			post: function postLink(scope, element, attrs) {
				scope.$watch(attrs.focusExpression, function () {
					if (attrs.focusExpression) {
						if (scope.$eval(attrs.focusExpression)) {
							$timeout(function () {
								if (element.is('input')) {
									element.focus();
								} else {
									element.find('input').focus();
								}
							}, 100); //need some delay to work with ng-disabled
						}
					}
				});
			}
		}
	};
}]);

angular.module('contactsApp')
.directive('inputresize', function() {
	return {
		restrict: 'A',
		link : function (scope, element) {
			var elInput = element.val();
			element.bind('keydown keyup load focus', function() {
				elInput = element.val();
				// If set to 0, the min-width css data is ignored
				var length = elInput.length > 1 ? elInput.length : 1;
				element.attr('size', length);
			});
		}
	};
});

angular.module('contactsApp')
.directive('selectExpression', ['$timeout', function ($timeout) {
	return {
		restrict: 'A',
		link: {
			post: function postLink(scope, element, attrs) {
				scope.$watch(attrs.selectExpression, function () {
					if (attrs.selectExpression) {
						if (scope.$eval(attrs.selectExpression)) {
							$timeout(function () {
								if (element.is('input')) {
									element.select();
								} else {
									element.find('input').select();
								}
							}, 100); //need some delay to work with ng-disabled
						}
					}
				});
			}
		}
	};
}]);

angular.module('contactsApp')
.controller('addressbookCtrl', ['$scope', 'AddressBookService', function($scope, AddressBookService) {
	var ctrl = this;

	ctrl.t = {
		download: t('contacts', 'Download'),
		copyURL: t('contacts', 'Copy link'),
		clickToCopy: t('contacts', 'Click to copy the link to your clipboard'),
		shareAddressbook: t('contacts', 'Toggle sharing'),
		deleteAddressbook: t('contacts', 'Delete'),
		renameAddressbook: t('contacts', 'Rename'),
		shareInputPlaceHolder: t('contacts', 'Share with users or groups'),
		delete: t('contacts', 'Delete'),
		canEdit: t('contacts', 'can edit'),
		close: t('contacts', 'Close'),
		enabled: t('contacts', 'Enabled'),
		disabled: t('contacts', 'Disabled')
	};

	ctrl.editing = false;
	ctrl.enabled = ctrl.addressBook.enabled;

	ctrl.tooltipIsOpen = false;
	ctrl.tooltipTitle = ctrl.t.clickToCopy;
	ctrl.showInputUrl = false;

	ctrl.clipboardSuccess = function() {
		ctrl.tooltipIsOpen = true;
		ctrl.tooltipTitle = t('core', 'Copied!');
		_.delay(function() {
			ctrl.tooltipIsOpen = false;
			ctrl.tooltipTitle = ctrl.t.clickToCopy;
		}, 3000);
	};

	ctrl.clipboardError = function() {
		ctrl.showInputUrl = true;
		if (/iPhone|iPad/i.test(navigator.userAgent)) {
			ctrl.InputUrlTooltip = t('core', 'Not supported!');
		} else if (/Mac/i.test(navigator.userAgent)) {
			ctrl.InputUrlTooltip = t('core', 'Press ⌘-C to copy.');
		} else {
			ctrl.InputUrlTooltip = t('core', 'Press Ctrl-C to copy.');
		}
		$('#addressBookUrl_'+ctrl.addressBook.ctag).select();
	};

	ctrl.renameAddressBook = function() {
		AddressBookService.rename(ctrl.addressBook, ctrl.addressBook.displayName);
		ctrl.editing = false;
	};

	ctrl.edit = function() {
		ctrl.editing = true;
	};

	ctrl.closeMenus = function() {
		$scope.$parent.ctrl.openedMenu = false;
	};

	ctrl.openMenu = function(index) {
		ctrl.closeMenus();
		$scope.$parent.ctrl.openedMenu = index;
	};

	ctrl.toggleMenu = function(index) {
		if ($scope.$parent.ctrl.openedMenu === index) {
			ctrl.closeMenus();
		} else {
			ctrl.openMenu(index);
		}
	};

	ctrl.toggleSharesEditor = function() {
		ctrl.editingShares = !ctrl.editingShares;
		ctrl.selectedSharee = null;
	};

	/* From Calendar-Rework - js/app/controllers/calendarlistcontroller.js */
	ctrl.findSharee = function (val) {
		return $.get(
			OC.linkToOCS('apps/files_sharing/api/v1') + 'sharees',
			{
				format: 'json',
				search: val.trim(),
				perPage: 200,
				itemType: 'principals'
			}
		).then(function(result) {
			var users   = result.ocs.data.exact.users.concat(result.ocs.data.users);
			var groups  = result.ocs.data.exact.groups.concat(result.ocs.data.groups);

			var userShares = ctrl.addressBook.sharedWith.users;
			var userSharesLength = userShares.length;

			var groupsShares = ctrl.addressBook.sharedWith.groups;
			var groupsSharesLength = groupsShares.length;
			var i, j;

			// Filter out current user
			for (i = 0 ; i < users.length; i++) {
				if (users[i].value.shareWith === OC.currentUser) {
					users.splice(i, 1);
					break;
				}
			}

			// Now filter out all sharees that are already shared with
			for (i = 0; i < userSharesLength; i++) {
				var shareUser = userShares[i];
				for (j = 0; j < users.length; j++) {
					if (users[j].value.shareWith === shareUser.id) {
						users.splice(j, 1);
						break;
					}
				}
			}

			// Now filter out all groups that are already shared with
			for (i = 0; i < groupsSharesLength; i++) {
				var sharedGroup = groupsShares[i];
				for (j = 0; j < groups.length; j++) {
					if (groups[j].value.shareWith === sharedGroup.id) {
						groups.splice(j, 1);
						break;
					}
				}
			}

			// Combine users and groups
			users = users.map(function(item) {
				return {
					display: _.escape(item.value.shareWith),
					type: OC.Share.SHARE_TYPE_USER,
					identifier: item.value.shareWith
				};
			});

			groups = groups.map(function(item) {
				return {
					display: _.escape(item.value.shareWith) + ' (group)',
					type: OC.Share.SHARE_TYPE_GROUP,
					identifier: item.value.shareWith
				};
			});

			return groups.concat(users);
		});
	};

	ctrl.onSelectSharee = function (item) {
		// Prevent settings to slide down
		$('#app-settings-header > button').data('apps-slide-toggle', false);
		_.delay(function() {
			$('#app-settings-header > button').data('apps-slide-toggle', '#app-settings-content');
		}, 500);

		ctrl.selectedSharee = null;
		AddressBookService.share(ctrl.addressBook, item.type, item.identifier, false, false).then(function() {
			$scope.$apply();
		});

	};

	ctrl.updateExistingUserShare = function(userId, writable) {
		AddressBookService.share(ctrl.addressBook, OC.Share.SHARE_TYPE_USER, userId, writable, true).then(function() {
			$scope.$apply();
		});
	};

	ctrl.updateExistingGroupShare = function(groupId, writable) {
		AddressBookService.share(ctrl.addressBook, OC.Share.SHARE_TYPE_GROUP, groupId, writable, true).then(function() {
			$scope.$apply();
		});
	};

	ctrl.unshareFromUser = function(userId) {
		AddressBookService.unshare(ctrl.addressBook, OC.Share.SHARE_TYPE_USER, userId).then(function() {
			$scope.$apply();
		});
	};

	ctrl.unshareFromGroup = function(groupId) {
		AddressBookService.unshare(ctrl.addressBook, OC.Share.SHARE_TYPE_GROUP, groupId).then(function() {
			$scope.$apply();
		});
	};

	ctrl.deleteAddressBook = function() {
		AddressBookService.delete(ctrl.addressBook).then(function() {
			$scope.$apply();
		});
	};

	ctrl.toggleState = function() {
		AddressBookService.toggleState(ctrl.addressBook).then(function(addressBook) {
			ctrl.enabled = addressBook.enabled;
			$scope.$apply();
		});
	};

}]);

angular.module('contactsApp')
.directive('addressbook', function() {
	return {
		restrict: 'A', // has to be an attribute to work with core css
		scope: {},
		controller: 'addressbookCtrl',
		controllerAs: 'ctrl',
		bindToController: {
			addressBook: '=data',
			list: '='
		},
		templateUrl: OC.linkTo('contacts', 'templates/addressBook.html')
	};
});

angular.module('contactsApp')
.controller('addressbooklistCtrl', ['$scope', 'AddressBookService', function($scope, AddressBookService) {
	var ctrl = this;

	ctrl.loading = true;
	ctrl.openedMenu = false;
	ctrl.addressBookRegex = /^[a-zA-Z0-9À-ÿ\s-_.!?#|()]+$/i;

	AddressBookService.getAll().then(function(addressBooks) {
		ctrl.addressBooks = addressBooks;
		ctrl.loading = false;
		if(ctrl.addressBooks.length === 0) {
			AddressBookService.create(t('contacts', 'Contacts')).then(function() {
				AddressBookService.getAddressBook(t('contacts', 'Contacts')).then(function(addressBook) {
					ctrl.addressBooks.push(addressBook);
					$scope.$apply();
				});
			});
		}
	});

	ctrl.t = {
		addressBookName : t('contacts', 'Address book name'),
		regexError : t('contacts', 'Only these special characters are allowed: -_.!?#|()')
	};

	ctrl.createAddressBook = function() {
		if(ctrl.newAddressBookName) {
			AddressBookService.create(ctrl.newAddressBookName).then(function() {
				AddressBookService.getAddressBook(ctrl.newAddressBookName).then(function(addressBook) {
					ctrl.addressBooks.push(addressBook);
					$scope.$apply();
				});
			}).catch(function() {
				OC.Notification.showTemporary(t('contacts', 'Address book could not be created.'));
			});
		}
	};
}]);

angular.module('contactsApp')
.directive('addressbooklist', function() {
	return {
		restrict: 'EA', // has to be an attribute to work with core css
		scope: {},
		controller: 'addressbooklistCtrl',
		controllerAs: 'ctrl',
		bindToController: {},
		templateUrl: OC.linkTo('contacts', 'templates/addressBookList.html')
	};
});

angular.module('contactsApp')
.controller('avatarCtrl', ['ContactService', function(ContactService) {
	var ctrl = this;

	ctrl.import = ContactService.import.bind(ContactService);

	ctrl.removePhoto = function() {
		ctrl.contact.removeProperty('photo', ctrl.contact.getProperty('photo'));
		ContactService.update(ctrl.contact);
		$('avatar').removeClass('maximized');
	};

	ctrl.downloadPhoto = function() {
		/* globals ArrayBuffer, Uint8Array */
		var img = document.getElementById('contact-avatar');
		// atob to base64_decode the data-URI
		var imageSplit = img.src.split(',');
		// "data:image/png;base64" -> "png"
		var extension = '.' + imageSplit[0].split(';')[0].split('/')[1];
		var imageData = atob(imageSplit[1]);
		// Use typed arrays to convert the binary data to a Blob
		var arrayBuffer = new ArrayBuffer(imageData.length);
		var view = new Uint8Array(arrayBuffer);
		for (var i=0; i<imageData.length; i++) {
			view[i] = imageData.charCodeAt(i) & 0xff;
		}
		var blob = new Blob([arrayBuffer], {type: 'application/octet-stream'});

		// Use the URL object to create a temporary URL
		var url = (window.webkitURL || window.URL).createObjectURL(blob);

		var a = document.createElement('a');
		document.body.appendChild(a);
		a.style = 'display: none';
		a.href = url;
		a.download = ctrl.contact.uid() + extension;
		a.click();
		window.URL.revokeObjectURL(url);
		a.remove();
	};

	ctrl.openPhoto = function() {
		$('avatar').toggleClass('maximized');
	};

	ctrl.t = {
		uploadNewPhoto : t('contacts', 'Upload new image'),
		deletePhoto : t('contacts', 'Delete'),
		closePhoto : t('contacts', 'Close'),
		downloadPhoto : t('contacts', 'Download')
	};

	// Quit avatar preview
	$('avatar').click(function() {
		$('avatar').removeClass('maximized');
	});
	$('avatar img, avatar .avatar-options').click(function(e) {
		e.stopPropagation();
	});
	$(document).keyup(function(e) {
		if (e.keyCode === 27) {
			$('avatar').removeClass('maximized');
		}
	});

}]);

angular.module('contactsApp')
.directive('avatar', ['ContactService', function(ContactService) {
	return {
		scope: {
			contact: '=data'
		},
		controller: 'avatarCtrl',
		controllerAs: 'ctrl',
		bindToController: {
			contact: '=data'
		},
		link: function(scope, element) {
			var input = element.find('input');
			input.bind('change', function() {
				var file = input.get(0).files[0];
				if (file.size > 1024*1024) { // 1 MB
					OC.Notification.showTemporary(t('contacts', 'The selected image is too big (max 1MB)'));
				} else {
					var reader = new FileReader();

					reader.addEventListener('load', function () {
						scope.$apply(function() {
							scope.contact.photo(reader.result);
							ContactService.update(scope.contact);
						});
					}, false);

					if (file) {
						reader.readAsDataURL(file);
					}
				}
			});
		},
		templateUrl: OC.linkTo('contacts', 'templates/avatar.html')
	};
}]);

angular.module('contactsApp')
.controller('contactCtrl', ['$route', '$routeParams', 'SortByService', function($route, $routeParams, SortByService) {
	var ctrl = this;

	ctrl.t = {
		errorMessage : t('contacts', 'This card is corrupted and has been fixed. Please check the data and trigger a save to make the changes permanent.'),
	};

	ctrl.getName = function() {
		// If lastName equals to firstName then none of them is set
		if (ctrl.contact.lastName() === ctrl.contact.firstName()) {
			return ctrl.contact.displayName();
		}

		if (SortByService.getSortByKey() === 'sortLastName') {
			return (
				ctrl.contact.lastName()
				+ (ctrl.contact.firstName() ? ', ' : '')
				+ ctrl.contact.firstName() + ' '
				+ ctrl.contact.additionalNames()
			).trim();
		}

		if (SortByService.getSortByKey() === 'sortFirstName') {
			return (
				ctrl.contact.firstName() + ' '
				+ ctrl.contact.additionalNames() + ' '
				+ ctrl.contact.lastName()
			).trim();
		}

		return ctrl.contact.displayName();
	};
}]);


angular.module('contactsApp')
.directive('contact', function() {
	return {
		scope: {},
		controller: 'contactCtrl',
		controllerAs: 'ctrl',
		bindToController: {
			contact: '=data'
		},
		templateUrl: OC.linkTo('contacts', 'templates/contact.html')
	};
});

angular.module('contactsApp')
.controller('contactdetailsCtrl', ['ContactService', 'AddressBookService', 'vCardPropertiesService', '$route', '$routeParams', '$scope', function(ContactService, AddressBookService, vCardPropertiesService, $route, $routeParams, $scope) {

	var ctrl = this;

	ctrl.init = true;
	ctrl.loading = false;
	ctrl.show = false;

	ctrl.clearContact = function() {
		$route.updateParams({
			gid: $routeParams.gid,
			uid: undefined
		});
		ctrl.show = false;
		ctrl.contact = undefined;
	};

	ctrl.uid = $routeParams.uid;
	ctrl.t = {
		noContacts : t('contacts', 'No contacts in here'),
		placeholderName : t('contacts', 'Name'),
		placeholderOrg : t('contacts', 'Organization'),
		placeholderTitle : t('contacts', 'Title'),
		selectField : t('contacts', 'Add field …'),
		download : t('contacts', 'Download'),
		delete : t('contacts', 'Delete'),
		save : t('contacts', 'Save changes'),
		addressBook : t('contacts', 'Address book'),
		loading : t('contacts', 'Loading contacts …')
	};

	ctrl.fieldDefinitions = vCardPropertiesService.fieldDefinitions;
	ctrl.focus = undefined;
	ctrl.field = undefined;
	ctrl.addressBooks = [];

	AddressBookService.getAll().then(function(addressBooks) {
		ctrl.addressBooks = addressBooks;

		if (!angular.isUndefined(ctrl.contact)) {
			ctrl.addressBook = _.find(ctrl.addressBooks, function(book) {
				return book.displayName === ctrl.contact.addressBookId;
			});
		}
		ctrl.init = false;
		// Start watching for ctrl.uid when we have addressBooks, as they are needed for fetching
		// full details.
		$scope.$watch('ctrl.uid', function(newValue) {
			ctrl.changeContact(newValue);
		});
	});


	ctrl.changeContact = function(uid) {
		if (typeof uid === 'undefined') {
			ctrl.show = false;
			$('#app-navigation-toggle').removeClass('showdetails');
			$('.app-content-list').removeClass('showdetails');
			return;
		}
		ctrl.loading = true;
		ContactService.getById(ctrl.addressBooks, uid).then(function(contact) {
			if (angular.isUndefined(contact)) {
				ctrl.clearContact();
				return;
			}
			ctrl.contact = contact;
			ctrl.show = true;
			ctrl.loading = false;
			$('#app-navigation-toggle').addClass('showdetails');
			$('.app-content-list').addClass('showdetails');

			ctrl.addressBook = _.find(ctrl.addressBooks, function(book) {
				return book.displayName === ctrl.contact.addressBookId;
			});
		});
	};

	ctrl.deleteContact = function() {
		ContactService.delete(ctrl.addressBook, ctrl.contact);
	};

	ctrl.addField = function(field) {
		var defaultValue = vCardPropertiesService.getMeta(field).defaultValue || {value: ''};
		ctrl.contact.addProperty(field, defaultValue);
		ctrl.focus = field;
		ctrl.field = '';
	};

	ctrl.deleteField = function (field, prop) {
		ctrl.contact.removeProperty(field, prop);
		ctrl.focus = undefined;
	};

	ctrl.changeAddressBook = function (addressBook, oldAddressBook) {
		ContactService.moveContact(ctrl.contact, addressBook, oldAddressBook);
	};

	ctrl.updateContact = function() {
		ContactService.queueUpdate(ctrl.contact);
	};

	ctrl.closeMenus = function() {
		ctrl.openedMenu = false;
	};

	ctrl.openMenu = function(index) {
		ctrl.closeMenus();
		ctrl.openedMenu = index;
	};

	ctrl.toggleMenu = function(index) {
		if (ctrl.openedMenu === index) {
			ctrl.closeMenus();
		} else {
			ctrl.openMenu(index);
		}
	};
}]);

angular.module('contactsApp')
.directive('contactdetails', function() {
	return {
		priority: 1,
		scope: {},
		controller: 'contactdetailsCtrl',
		controllerAs: 'ctrl',
		bindToController: {},
		templateUrl: OC.linkTo('contacts', 'templates/contactDetails.html')
	};
});

angular.module('contactsApp')
.controller('contactfilterCtrl', function() {
	// eslint-disable-next-line no-unused-vars
	var ctrl = this;
});

angular.module('contactsApp')
.directive('contactFilter', function() {
	return {
		restrict: 'A', // has to be an attribute to work with core css
		scope: {},
		controller: 'contactfilterCtrl',
		controllerAs: 'ctrl',
		bindToController: {
			contactFilter: '=contactFilter'
		},
		templateUrl: OC.linkTo('contacts', 'templates/contactFilter.html')
	};
});

angular.module('contactsApp')
.controller('contactimportCtrl', ['ContactService', 'AddressBookService', '$timeout', '$scope', function(ContactService, AddressBookService, $timeout, $scope) {
	var ctrl = this;

	ctrl.t = {
		importText : t('contacts', 'Import into'),
		importingText : t('contacts', 'Importing...'),
		selectAddressbook : t('contacts', 'Select your addressbook'),
		importdisabled : t('contacts', 'Import is disabled because no writable address book had been found.')
	};

	ctrl.import = ContactService.import.bind(ContactService);
	ctrl.loading = true;
	ctrl.importText = ctrl.t.importText;
	ctrl.importing = false;
	ctrl.loadingClass = 'icon-upload';

	AddressBookService.getAll().then(function(addressBooks) {
		ctrl.addressBooks = addressBooks;
		ctrl.loading = false;
		ctrl.selectedAddressBook = AddressBookService.getDefaultAddressBook();
	});

	AddressBookService.registerObserverCallback(function() {
		$timeout(function() {
			$scope.$apply(function() {
				ctrl.selectedAddressBook = AddressBookService.getDefaultAddressBook();
			});
		});
	});

	ctrl.stopHideMenu = function(isOpen) {
		if(isOpen) {
			// disabling settings bind
			$('#app-settings-header > button').data('apps-slide-toggle', false);
		} else {
			// reenabling it
			$('#app-settings-header > button').data('apps-slide-toggle', '#app-settings-content');
		}
	};

}]);

angular.module('contactsApp')
.directive('contactimport', ['ContactService', 'ImportService', '$rootScope', function(ContactService, ImportService, $rootScope) {
	return {
		link: function(scope, element, attrs, ctrl) {
			var input = element.find('input');
			input.bind('change', function() {
				angular.forEach(input.get(0).files, function(file) {
					var reader = new FileReader();

					reader.addEventListener('load', function () {
						scope.$apply(function () {
							// Indicate the user we started something
							ctrl.importText = ctrl.t.importingText;
							ctrl.loadingClass = 'icon-loading-small';
							ctrl.importing = true;
							$rootScope.importing = true;

							ContactService.import.call(ContactService, reader.result, file.type, ctrl.selectedAddressBook, function (progress, user) {
								if (progress === 1) {
									ctrl.importText = ctrl.t.importText;
									ctrl.loadingClass = 'icon-upload';
									ctrl.importing = false;
									$rootScope.importing = false;
									ImportService.importPercent = 0;
									ImportService.importing = false;
									ImportService.importedUser = '';
									ImportService.selectedAddressBook = '';
								} else {
									// Ugly hack, hide sidebar on import & mobile
									// Simulate click since we can't directly access snapper
									if($(window).width() <= 768 && $('body').hasClass('snapjs-left')) {
										$('#app-navigation-toggle').click();
										$('body').removeClass('snapjs-left');
									}

									ImportService.importPercent = parseInt(Math.floor(progress * 100));
									ImportService.importing = true;
									ImportService.importedUser = user;
									ImportService.selectedAddressBook = ctrl.selectedAddressBook.displayName;
								}
								scope.$apply();

								/* Broadcast service update */
								$rootScope.$broadcast('importing', true);
							});
						});
					}, false);

					if (file) {
						reader.readAsText(file);
					}
				});
				input.get(0).value = '';
			});
		},
		templateUrl: OC.linkTo('contacts', 'templates/contactImport.html'),
		controller: 'contactimportCtrl',
		controllerAs: 'ctrl'
	};
}]);

angular.module('contactsApp')
.controller('contactlistCtrl', ['$scope', '$filter', '$route', '$routeParams', '$timeout', 'AddressBookService', 'ContactService', 'SortByService', 'vCardPropertiesService', 'SearchService', function($scope, $filter, $route, $routeParams, $timeout, AddressBookService, ContactService, SortByService, vCardPropertiesService, SearchService) {
	var ctrl = this;

	ctrl.routeParams = $routeParams;

	ctrl.filteredContacts = []; // the displayed contacts list
	ctrl.searchTerm = '';
	ctrl.show = true;
	ctrl.invalid = false;
	ctrl.limitTo = 25;

	ctrl.sortBy = SortByService.getSortBy();

	ctrl.t = {
		emptySearch : t('contacts', 'No search result for {query}', {query: ctrl.searchTerm})
	};

	ctrl.resetLimitTo = function () {
		ctrl.limitTo = 25;
		clearInterval(ctrl.intervalId);
		ctrl.intervalId = setInterval(
			function () {
				if (!ctrl.loading && ctrl.contactList && ctrl.contactList.length > ctrl.limitTo) {
					ctrl.limitTo += 25;
					$scope.$apply();
				}
			}, 300);
	};

	$scope.query = function(contact) {
		return contact.matches(SearchService.getSearchTerm());
	};

	SortByService.subscribe(function(newValue) {
		ctrl.sortBy = newValue;
	});

	SearchService.registerObserverCallback(function(ev) {
		if (ev.event === 'submitSearch') {
			var uid = !_.isEmpty(ctrl.filteredContacts) ? ctrl.filteredContacts[0].uid() : undefined;
			ctrl.setSelectedId(uid);
			$scope.$apply();
		}
		if (ev.event === 'changeSearch') {
			ctrl.resetLimitTo();
			ctrl.searchTerm = ev.searchTerm;
			ctrl.t.emptySearch = t('contacts',
								   'No search result for {query}',
								   {query: ctrl.searchTerm}
								  );
			$scope.$apply();
		}
	});

	ctrl.loading = true;

	ContactService.registerObserverCallback(function(ev) {
		/* after import at first refresh the contactList */
		if (ev.event === 'importend') {
			$scope.$apply(function() {
				ctrl.contactList = ev.contacts;
			});
		}
		/* update route parameters */
		$timeout(function() {
			$scope.$apply(function() {
				switch(ev.event) {
				case 'delete':
					ctrl.selectNearestContact(ev.uid);
					break;
				case 'create':
					$route.updateParams({
						gid: $routeParams.gid,
						uid: ev.uid
					});
					break;
				case 'importend':
					/* after import select 'All contacts' group and first contact */
					$route.updateParams({
						gid: t('contacts', 'All contacts'),
						uid: ctrl.filteredContacts.length !== 0 ? ctrl.filteredContacts[0].uid() : undefined
					});
					return;
				case 'getFullContacts' || 'update':
					break;
				default:
					// unknown event -> leave callback without action
					return;
				}
				ctrl.contactList = ev.contacts;
			});
		});
	});

	AddressBookService.registerObserverCallback(function(ev) {
		$timeout(function() {
			$scope.$apply(function() {
				switch (ev.event) {
				case 'delete':
				case 'disable':
					ctrl.loading = true;
					ContactService.removeContactsFromAddressbook(ev.addressBook, function() {
						ContactService.getAll().then(function(contacts) {
							ctrl.contactList = contacts;
							ctrl.loading = false;
							// Only change contact if the selectd one is not in the list anymore
							if(ctrl.contactList.findIndex(function(contact) {
								return contact.uid() === ctrl.getSelectedId();
							}) === -1) {
								ctrl.selectNearestContact(ctrl.getSelectedId());
							}
						});
					});
					break;
				case 'enable':
					ctrl.loading = true;
					ContactService.appendContactsFromAddressbook(ev.addressBook, function() {
						ContactService.getAll().then(function(contacts) {
							ctrl.contactList = contacts;
							ctrl.loading = false;
						});
					});
					break;
				default:
						// unknown event -> leave callback without action
					return;

				}
			});
		});
	});

	// Get contacts
	ContactService.getAll().then(function(contacts) {
		if(contacts.length>0) {
			$scope.$apply(function() {
				ctrl.contactList = contacts;
			});
		} else {
			ctrl.loading = false;
		}
	});

	var getVisibleContacts = function() {
		var scrolled = $('.app-content-list').scrollTop();
		var elHeight = $('.contacts-list').children().outerHeight(true);
		var listHeight = $('.app-content-list').height();

		var topContact = Math.round(scrolled/elHeight);
		var contactsCount = Math.round(listHeight/elHeight);

		return ctrl.filteredContacts.slice(topContact-1, topContact+contactsCount+1);
	};

	var timeoutId = null;
	document.querySelector('.app-content-list').addEventListener('scroll', function () {
		clearTimeout(timeoutId);
		timeoutId = setTimeout(function () {
			var contacts = getVisibleContacts();
			ContactService.getFullContacts(contacts);
		}, 250);
	});

	// Wait for ctrl.filteredContacts to be updated, load the contact requested in the URL if any, and
	// load full details for the probably initially visible contacts.
	// Then kill the watch.
	var unbindListWatch = $scope.$watch('ctrl.filteredContacts', function() {
		if(ctrl.filteredContacts && ctrl.filteredContacts.length > 0) {
			// Check if a specific uid is requested
			if($routeParams.uid && $routeParams.gid) {
				ctrl.filteredContacts.forEach(function(contact) {
					if(contact.uid() === $routeParams.uid) {
						ctrl.setSelectedId($routeParams.uid);
						ctrl.loading = false;
					}
				});
			}
			// No contact previously loaded, let's load the first of the list if not in mobile mode
			if(ctrl.loading && $(window).width() > 768) {
				ctrl.setSelectedId(ctrl.filteredContacts[0].uid());
			}
			// Get full data for the first 20 contacts of the list
			ContactService.getFullContacts(ctrl.filteredContacts.slice(0, 20));
			ctrl.loading = false;
			unbindListWatch();
		}
	});

	$scope.$watch('ctrl.routeParams.uid', function(newValue, oldValue) {
		// Used for mobile view to clear the url
		if(typeof oldValue != 'undefined' && typeof newValue == 'undefined' && $(window).width() <= 768) {
			// no contact selected
			ctrl.show = true;
			return;
		}
		if(newValue === undefined) {
			// we might have to wait until ng-repeat filled the contactList
			if(ctrl.filteredContacts && ctrl.filteredContacts.length > 0) {
				$route.updateParams({
					gid: $routeParams.gid,
					uid: ctrl.filteredContacts[0].uid()
				});
			} else {
				// watch for next contactList update
				var unbindWatch = $scope.$watch('ctrl.filteredContacts', function() {
					if(ctrl.filteredContacts && ctrl.filteredContacts.length > 0) {
						$route.updateParams({
							gid: $routeParams.gid,
							uid: ctrl.filteredContacts[0].uid()
						});
					}
					unbindWatch(); // unbind as we only want one update
				});
			}
		} else {
			// displaying contact details
			ctrl.show = false;
		}
	});

	$scope.$watch('ctrl.routeParams.gid', function() {
		// we might have to wait until ng-repeat filled the contactList
		ctrl.filteredContacts = [];
		ctrl.resetLimitTo();
		// not in mobile mode
		if($(window).width() > 768) {
			// watch for next contactList update
			var unbindWatch = $scope.$watch('ctrl.filteredContacts', function() {
				if(ctrl.filteredContacts && ctrl.filteredContacts.length > 0) {
					$route.updateParams({
						gid: $routeParams.gid,
						uid: $routeParams.uid || ctrl.filteredContacts[0].uid()
					});
				}
				unbindWatch(); // unbind as we only want one update
			});
		}
	});

	// Watch if we have an invalid contact
	$scope.$watch('ctrl.filteredContacts[0].displayName()', function(displayName) {
		ctrl.invalid = (displayName === '');
	});

	ctrl.hasContacts = function () {
		if (!ctrl.contactList) {
			return false;
		}
		return ctrl.contactList.length > 0;
	};

	ctrl.setSelectedId = function (contactId) {
		$route.updateParams({
			uid: contactId
		});
	};

	ctrl.getSelectedId = function() {
		return $routeParams.uid;
	};

	ctrl.selectNearestContact = function(contactId) {
		if (ctrl.filteredContacts.length === 1) {
			$route.updateParams({
				gid: t('contacts', 'All contacts'),
				uid: ctrl.filteredContacts.length !== 0 ? ctrl.filteredContacts[0].uid() : undefined
			});
		} else {
			for (var i = 0, length = ctrl.filteredContacts.length; i < length; i++) {
				// Get nearest contact
				if (ctrl.filteredContacts[i].uid() === contactId) {
					$route.updateParams({
						gid: $routeParams.gid,
						uid: (ctrl.filteredContacts[i+1]) ? ctrl.filteredContacts[i+1].uid() : ctrl.filteredContacts[i-1].uid()
					});
					break;
				}
			}
		}
	};

}]);

angular.module('contactsApp')
.directive('contactlist', function() {
	return {
		priority: 1,
		scope: {},
		controller: 'contactlistCtrl',
		controllerAs: 'ctrl',
		bindToController: {
			addressbook: '=adrbook'
		},
		templateUrl: OC.linkTo('contacts', 'templates/contactList.html')
	};
});

angular.module('contactsApp')
.controller('detailsItemCtrl', ['$templateRequest', '$filter', 'vCardPropertiesService', 'ContactService', function($templateRequest, $filter, vCardPropertiesService, ContactService) {
	var ctrl = this;

	ctrl.meta = vCardPropertiesService.getMeta(ctrl.name);
	ctrl.type = undefined;
	ctrl.isPreferred = false;
	ctrl.t = {
		poBox : t('contacts', 'Post office box'),
		postalCode : t('contacts', 'Postal code'),
		city : t('contacts', 'City'),
		state : t('contacts', 'State or province'),
		country : t('contacts', 'Country'),
		address: t('contacts', 'Address'),
		newGroup: t('contacts', '(new group)'),
		familyName: t('contacts', 'Last name'),
		firstName: t('contacts', 'First name'),
		additionalNames: t('contacts', 'Additional names'),
		honorificPrefix: t('contacts', 'Prefix'),
		honorificSuffix: t('contacts', 'Suffix'),
		delete: t('contacts', 'Delete')
	};

	ctrl.availableOptions = ctrl.meta.options || [];
	if (!_.isUndefined(ctrl.data) && !_.isUndefined(ctrl.data.meta) && !_.isUndefined(ctrl.data.meta.type)) {
		// parse type of the property
		var array = ctrl.data.meta.type[0].split(',');
		array = array.map(function (elem) {
			return elem.trim().replace(/\/+$/, '').replace(/\\+$/, '').trim().toUpperCase();
		});
		// the pref value is handled on its own so that we can add some favorite icon to the ui if we want
		if (array.indexOf('PREF') >= 0) {
			ctrl.isPreferred = true;
			array.splice(array.indexOf('PREF'), 1);
		}
		// simply join the upper cased types together as key
		ctrl.type = array.join(',');
		var displayName = array.map(function (element) {
			return element.charAt(0).toUpperCase() + element.slice(1).toLowerCase();
		}).join(' ');
		// in case the type is not yet in the default list of available options we add it
		if (!ctrl.availableOptions.some(function(e) { return e.id === ctrl.type; } )) {
			ctrl.availableOptions = ctrl.availableOptions.concat([{id: ctrl.type, name: displayName}]);
		}

		// Remove duplicate entry
		ctrl.availableOptions = _.uniq(ctrl.availableOptions, function(option) { return option.name; });
		if (ctrl.availableOptions.filter(function(option) { return option.id === ctrl.type; }).length === 0) {
			// Our default value has been thrown out by the uniq function, let's find a replacement
			var optionName = ctrl.meta.options.filter(function(option) { return option.id === ctrl.type; })[0].name;
			ctrl.type = ctrl.availableOptions.filter(function(option) { return option.name === optionName; })[0].id;
			// We don't want to override the default keys. Compatibility > standardization
			// ctrl.data.meta.type[0] = ctrl.type;
			// ctrl.model.updateContact();
		}
	}
	if (!_.isUndefined(ctrl.data) && !_.isUndefined(ctrl.data.namespace)) {
		if (!_.isUndefined(ctrl.contact.props['X-ABLABEL'])) {
			var val = _.find(this.contact.props['X-ABLABEL'], function(x) { return x.namespace === ctrl.data.namespace; });
			ctrl.type = val.value.toUpperCase();
			if (!_.isUndefined(val)) {
				// in case the type is not yet in the default list of available options we add it
				if (!ctrl.availableOptions.some(function(e) { return e.id === val.value; } )) {
					ctrl.availableOptions = ctrl.availableOptions.concat([{id: val.value.toUpperCase(), name: val.value.toUpperCase()}]);
				}
			}
		}
	}

	ctrl.availableGroups = [];

	ContactService.getGroups().then(function(groups) {
		ctrl.availableGroups = _.unique(groups);
	});

	ctrl.changeType = function (val) {
		if (ctrl.isPreferred) {
			val += ',PREF';
		}
		ctrl.data.meta = ctrl.data.meta || {};
		ctrl.data.meta.type = ctrl.data.meta.type || [];
		ctrl.data.meta.type[0] = val;
		ContactService.queueUpdate(ctrl.contact);
	};

	ctrl.dateInputChanged = function () {
		ctrl.data.meta = ctrl.data.meta || {};

		var match = ctrl.data.value.match(/^(\d{4})-(\d{2})-(\d{2})$/);
		if (match) {
			ctrl.data.meta.value = [];
		} else {
			ctrl.data.meta.value = ctrl.data.meta.value || [];
			ctrl.data.meta.value[0] = 'text';
		}
		ContactService.queueUpdate(ctrl.contact);
	};

	ctrl.updateDetailedName = function () {
		var fn = '';
		if (ctrl.data.value[3]) {
			fn += ctrl.data.value[3] + ' ';
		}
		if (ctrl.data.value[1]) {
			fn += ctrl.data.value[1] + ' ';
		}
		if (ctrl.data.value[2]) {
			fn += ctrl.data.value[2] + ' ';
		}
		if (ctrl.data.value[0]) {
			fn += ctrl.data.value[0] + ' ';
		}
		if (ctrl.data.value[4]) {
			fn += ctrl.data.value[4];
		}

		ctrl.contact.fullName(fn);
		ContactService.queueUpdate(ctrl.contact);
	};

	ctrl.updateContact = function() {
		ContactService.queueUpdate(ctrl.contact);
	};

	ctrl.getTemplate = function() {
		var templateUrl = OC.linkTo('contacts', 'templates/detailItems/' + ctrl.meta.template + '.html');
		return $templateRequest(templateUrl);
	};

	ctrl.deleteField = function () {
		ctrl.contact.removeProperty(ctrl.name, ctrl.data);
		ContactService.queueUpdate(ctrl.contact);
	};
}]);

angular.module('contactsApp')
.directive('detailsitem', ['$compile', function($compile) {
	return {
		scope: {},
		controller: 'detailsItemCtrl',
		controllerAs: 'ctrl',
		bindToController: {
			name: '=',
			data: '=',
			contact: '=model',
			index: '='
		},
		link: function(scope, element, attrs, ctrl) {
			ctrl.getTemplate().then(function(html) {
				var template = angular.element(html);
				element.append(template);
				$compile(template)(scope);
			});
		}
	};
}]);

angular.module('contactsApp')
.controller('groupCtrl', function() {
	// eslint-disable-next-line no-unused-vars
	var ctrl = this;
});

angular.module('contactsApp')
.directive('group', function() {
	return {
		restrict: 'A', // has to be an attribute to work with core css
		scope: {},
		controller: 'groupCtrl',
		controllerAs: 'ctrl',
		bindToController: {
			group: '=group'
		},
		templateUrl: OC.linkTo('contacts', 'templates/group.html')
	};
});

angular.module('contactsApp')
.controller('grouplistCtrl', ['$scope', '$timeout', 'ContactService', 'SearchService', '$routeParams', function($scope, $timeout, ContactService, SearchService, $routeParams) {
	var ctrl = this;

	ctrl.groups = [];
	ctrl.contactFilters = [];

	ContactService.getGroupList().then(function(groups) {
		ctrl.groups = groups;
	});

	ContactService.getContactFilters().then(function(contactFilters) {
		ctrl.contactFilters = contactFilters;
	});

	ctrl.getSelected = function() {
		return $routeParams.gid;
	};

	// Update groupList on contact add/delete/update/groupsUpdate
	ContactService.registerObserverCallback(function(ev) {
		if (ev.event !== 'getFullContacts') {
			$timeout(function () {
				$scope.$apply(function() {
					ContactService.getGroupList().then(function(groups) {
						ctrl.groups = groups;
					});
					ContactService.getContactFilters().then(function(contactFilters) {
						ctrl.contactFilters = contactFilters;
					});
				});
			});
		}
	});

	ctrl.setSelected = function (selectedGroup) {
		SearchService.cleanSearch();
		$routeParams.gid = selectedGroup;
	};
}]);

angular.module('contactsApp')
.directive('grouplist', function() {
	return {
		restrict: 'EA', // has to be an attribute to work with core css
		scope: {},
		controller: 'grouplistCtrl',
		controllerAs: 'ctrl',
		bindToController: {},
		templateUrl: OC.linkTo('contacts', 'templates/groupList.html')
	};
});

angular.module('contactsApp')
.controller('importscreenCtrl', ['$scope', 'ImportService', function($scope, ImportService) {
	var ctrl = this;

	ctrl.t = {
		importingTo : t('contacts', 'Importing into'),
		selectAddressbook : t('contacts', 'Select your addressbook')
	};

	// Broadcast update
	$scope.$on('importing', function () {
		ctrl.selectedAddressBook = ImportService.selectedAddressBook;
		ctrl.importedUser = ImportService.importedUser;
		ctrl.importing = ImportService.importing;
		ctrl.importPercent = ImportService.importPercent;
	});

}]);

angular.module('contactsApp')
.directive('importscreen', function() {
	return {
		restrict: 'EA', // has to be an attribute to work with core css
		scope: {},
		controller: 'importscreenCtrl',
		controllerAs: 'ctrl',
		bindToController: {},
		templateUrl: OC.linkTo('contacts', 'templates/importScreen.html')
	};
});

angular.module('contactsApp')
.controller('newContactButtonCtrl', ['$scope', 'ContactService', '$routeParams', 'vCardPropertiesService', function($scope, ContactService, $routeParams, vCardPropertiesService) {
	var ctrl = this;

	ctrl.t = {
		addContact : t('contacts', 'New contact')
	};

	ctrl.createContact = function() {
		ContactService.create().then(function(contact) {
			['tel', 'adr', 'email'].forEach(function(field) {
				var defaultValue = vCardPropertiesService.getMeta(field).defaultValue || {value: ''};
				contact.addProperty(field, defaultValue);
			} );
			if ([t('contacts', 'All contacts'), t('contacts', 'Not grouped')].indexOf($routeParams.gid) === -1) {
				contact.categories([ $routeParams.gid ]);
			} else {
				contact.categories([]);
			}
			$('#details-fullName').focus();
		});
	};
}]);

angular.module('contactsApp')
.directive('newcontactbutton', function() {
	return {
		restrict: 'EA', // has to be an attribute to work with core css
		scope: {},
		controller: 'newContactButtonCtrl',
		controllerAs: 'ctrl',
		bindToController: {},
		templateUrl: OC.linkTo('contacts', 'templates/newContactButton.html')
	};
});

angular.module('contactsApp')
.directive('telModel', function() {
	return{
		restrict: 'A',
		require: 'ngModel',
		link: function(scope, element, attr, ngModel) {
			ngModel.$formatters.push(function(value) {
				return value;
			});
			ngModel.$parsers.push(function(value) {
				return value;
			});
		}
	};
});

angular.module('contactsApp')
.controller('propertyGroupCtrl', ['vCardPropertiesService', function(vCardPropertiesService) {
	var ctrl = this;

	ctrl.meta = vCardPropertiesService.getMeta(ctrl.name);

	this.isHidden = function() {
		return ctrl.meta.hasOwnProperty('hidden') && ctrl.meta.hidden === true;
	};

	this.getIconClass = function() {
		return ctrl.meta.icon || 'icon-contacts-dark';
	};

	this.getInfoClass = function() {
		if (ctrl.meta.hasOwnProperty('info')) {
			return 'icon-info';

		}
	};

	this.getInfoText = function() {
		return ctrl.meta.info;
	};

	this.getReadableName = function() {
		return ctrl.meta.readableName;
	};
}]);

angular.module('contactsApp')
.directive('propertygroup', function() {
	return {
		scope: {},
		controller: 'propertyGroupCtrl',
		controllerAs: 'ctrl',
		bindToController: {
			properties: '=data',
			name: '=',
			contact: '=model'
		},
		templateUrl: OC.linkTo('contacts', 'templates/propertyGroup.html'),
		link: function(scope, element, attrs, ctrl) {
			if(ctrl.isHidden()) {
				// TODO replace with class
				element.css('display', 'none');
			}
		}
	};
});

angular.module('contactsApp')
.controller('sortbyCtrl', ['SortByService', function(SortByService) {
	var ctrl = this;

	var sortText = t('contacts', 'Sort by');
	ctrl.sortText = sortText;

	var sortList = SortByService.getSortByList();
	ctrl.sortList = sortList;

	ctrl.defaultOrder = SortByService.getSortByKey();

	ctrl.updateSortBy = function() {
		SortByService.setSortBy(ctrl.defaultOrder);
	};
}]);

angular.module('contactsApp')
.directive('sortby', function() {
	return {
		priority: 1,
		scope: {},
		controller: 'sortbyCtrl',
		controllerAs: 'ctrl',
		bindToController: {},
		templateUrl: OC.linkTo('contacts', 'templates/sortBy.html')
	};
});

angular.module('contactsApp')
.factory('AddressBook', function()
{
	return function AddressBook(data) {
		angular.extend(this, {

			displayName: '',
			contacts: [],
			groups: data.data.props.groups,
			readOnly: data.data.props.readOnly === '1',
			// In case of not defined
			enabled: data.data.props.enabled !== '0',

			sharedWith: {
				users: [],
				groups: []
			}

		});
		angular.extend(this, data);
		angular.extend(this, {
			owner: data.data.props.owner.split('/').slice(-2, -1)[0]
		});

		var shares = this.data.props.invite;
		if (typeof shares !== 'undefined') {
			for (var j = 0; j < shares.length; j++) {
				var href = shares[j].href;
				if (href.length === 0) {
					continue;
				}
				var access = shares[j].access;
				if (access.length === 0) {
					continue;
				}

				var readWrite = (typeof access.readWrite !== 'undefined');

				if (href.startsWith('principal:principals/users/')) {
					this.sharedWith.users.push({
						id: href.substr(27),
						displayname: href.substr(27),
						writable: readWrite
					});
				} else if (href.startsWith('principal:principals/groups/')) {
					this.sharedWith.groups.push({
						id: href.substr(28),
						displayname: href.substr(28),
						writable: readWrite
					});
				}
			}
		}
	};
});

angular.module('contactsApp')
	.factory('ContactFilter', function()
	{
		return function ContactFilter(data) {
			angular.extend(this, {
				name: '',
				count: 0
			});

			angular.extend(this, data);
		};
	});

angular.module('contactsApp')
.factory('Contact', ['$filter', 'MimeService', 'uuid4', function($filter, MimeService, uuid4) {
	return function Contact(addressBook, vCard) {
		angular.extend(this, {

			data: {},
			props: {},
			failedProps: [],

			dateProperties: ['bday', 'anniversary', 'deathdate'],

			addressBookId: addressBook.displayName,
			readOnly: addressBook.readOnly,

			version: function() {
				var property = this.getProperty('version');
				if(property) {
					return property.value;
				}

				return undefined;
			},

			uid: function(value) {
				var model = this;
				if (angular.isDefined(value)) {
					// setter
					return model.setProperty('uid', { value: value });
				} else {
					// getter
					var uid = model.getProperty('uid').value;
					/* global md5 */
					return uuid4.validate(uid) ? uid : md5(uid);
				}
			},

			displayName: function() {
				var displayName = this.fullName() || this.org() || '';
				if(angular.isArray(displayName)) {
					return displayName.join(' ');
				}
				return displayName;
			},

			readableFilename: function() {
				if(this.displayName()) {
					return (this.displayName()) + '.vcf';
				} else {
					// fallback to default filename (see download attribute)
					return '';
				}

			},

			firstName: function() {
				var property = this.getProperty('n');
				if (property) {
					return property.value[1];
				} else {
					return this.displayName();
				}
			},

			lastName: function() {
				var property = this.getProperty('n');
				if (property) {
					return property.value[0];
				} else {
					return this.displayName();
				}
			},

			additionalNames: function() {
				var property = this.getProperty('n');
				if (property) {
					return property.value[2];
				} else {
					return '';
				}
			},

			fullName: function(value) {
				var model = this;
				if (angular.isDefined(value)) {
					// setter
					return this.setProperty('fn', { value: value });
				} else {
					// getter
					var property = model.getProperty('fn');
					if(property) {
						return property.value;
					}
					property = model.getProperty('n');
					if(property) {
						return property.value.filter(function(elem) {
							return elem;
						}).join(' ');
					}
					return undefined;
				}
			},

			title: function(value) {
				if (angular.isDefined(value)) {
					// setter
					return this.setProperty('title', { value: value });
				} else {
					// getter
					var property = this.getProperty('title');
					if(property) {
						return property.value;
					} else {
						return undefined;
					}
				}
			},

			org: function(value) {
				var property = this.getProperty('org');
				if (angular.isDefined(value)) {
					var val = value;
					// setter
					if(property && Array.isArray(property.value)) {
						val = property.value;
						val[0] = value;
					}
					return this.setProperty('org', { value: val });
				} else {
					// getter
					if(property) {
						if (Array.isArray(property.value)) {
							return property.value[0];
						}
						return property.value;
					} else {
						return undefined;
					}
				}
			},

			email: function() {
				// getter
				var property = this.getProperty('email');
				if(property) {
					return property.value;
				} else {
					return undefined;
				}
			},

			photo: function(value) {
				if (angular.isDefined(value)) {
					// setter
					// splits image data into "data:image/jpeg" and base 64 encoded image
					var imageData = value.split(';base64,');
					var imageType = imageData[0].slice('data:'.length);
					if (!imageType.startsWith('image/')) {
						return;
					}
					imageType = imageType.substring(6).toUpperCase();

					return this.setProperty('photo', { value: imageData[1], meta: {type: [imageType], encoding: ['b']} });
				} else {
					var property = this.getProperty('photo');
					if(property) {
						var type = property.meta.type;
						if (angular.isArray(type)) {
							type = type[0];
						}
						if (!type.startsWith('image/')) {
							type = 'image/' + type.toLowerCase();
						}
						return 'data:' + type + ';base64,' + property.value;
					} else {
						return undefined;
					}
				}
			},

			categories: function(value) {
				if (angular.isDefined(value)) {
					// setter
					if (angular.isString(value)) {
						/* check for empty string */
						this.setProperty('categories', { value: !value.length ? [] : [value] });
					} else if (angular.isArray(value)) {
						this.setProperty('categories', { value: value });
					}
				} else {
					// getter
					var property = this.getProperty('categories');
					if(!property) {
						return [];
					}
					if (angular.isArray(property.value)) {
						return property.value;
					}
					return [property.value];
				}
			},

			formatDateAsRFC6350: function(name, data) {
				if (angular.isUndefined(data) || angular.isUndefined(data.value)) {
					return data;
				}
				if (this.dateProperties.indexOf(name) !== -1) {
					var match = data.value.match(/^(\d{4})-(\d{2})-(\d{2})$/);
					if (match) {
						data.value = match[1] + match[2] + match[3];
					}
				}

				return data;
			},

			formatDateForDisplay: function(name, data) {
				if (angular.isUndefined(data) || angular.isUndefined(data.value)) {
					return data;
				}
				if (this.dateProperties.indexOf(name) !== -1) {
					var match = data.value.match(/^(\d{4})(\d{2})(\d{2})$/);
					if (match) {
						data.value = match[1] + '-' + match[2] + '-' + match[3];
					}
				}

				return data;
			},

			getProperty: function(name) {
				if (this.props[name]) {
					return this.formatDateForDisplay(name, this.validate(name, this.props[name][0]));
				} else {
					return undefined;
				}
			},
			addProperty: function(name, data) {
				data = angular.copy(data);
				data = this.formatDateAsRFC6350(name, data);
				if(!this.props[name]) {
					this.props[name] = [];
				}
				var idx = this.props[name].length;
				this.props[name][idx] = data;

				// keep vCard in sync
				this.data.addressData = $filter('JSON2vCard')(this.props);
				return idx;
			},
			setProperty: function(name, data) {
				if(!this.props[name]) {
					this.props[name] = [];
				}
				data = this.formatDateAsRFC6350(name, data);
				this.props[name][0] = data;

				// keep vCard in sync
				this.data.addressData = $filter('JSON2vCard')(this.props);
			},
			removeProperty: function (name, prop) {
				angular.copy(_.without(this.props[name], prop), this.props[name]);
				if(this.props[name].length === 0) {
					delete this.props[name];
				}
				this.data.addressData = $filter('JSON2vCard')(this.props);
			},
			setETag: function(etag) {
				this.data.etag = etag;
			},
			setUrl: function(addressBook, uid) {
				this.data.url = addressBook.url + uid + '.vcf';
			},
			setAddressBook: function(addressBook) {
				this.addressBookId = addressBook.displayName;
				this.data.url = addressBook.url + this.uid() + '.vcf';
			},

			getISODate: function(date) {
				function pad(number) {
					if (number < 10) {
						return '0' + number;
					}
					return '' + number;
				}

				return date.getUTCFullYear() + '' +
						pad(date.getUTCMonth() + 1) +
						pad(date.getUTCDate()) +
						'T' + pad(date.getUTCHours()) +
						pad(date.getUTCMinutes()) +
						pad(date.getUTCSeconds()) + 'Z';
			},

			syncVCard: function() {

				this.setProperty('rev', { value: this.getISODate(new Date()) });
				var self = this;

				_.each(this.dateProperties, function(name) {
					if (!angular.isUndefined(self.props[name]) && !angular.isUndefined(self.props[name][0])) {
						// Set dates again to make sure they are in RFC-6350 format
						self.setProperty(name, self.props[name][0]);
					}
				});
				// force fn to be set
				this.fullName(this.fullName());

				// keep vCard in sync
				self.data.addressData = $filter('JSON2vCard')(self.props);

				// Revalidate all props
				_.each(self.failedProps, function(name, index) {
					if (!angular.isUndefined(self.props[name]) && !angular.isUndefined(self.props[name][0])) {
						// Reset previously failed properties
						self.failedProps.splice(index, 1);
						// And revalidate them again
						self.validate(name, self.props[name][0]);

					} else if(angular.isUndefined(self.props[name]) || angular.isUndefined(self.props[name][0])) {
						// Property has been removed
						self.failedProps.splice(index, 1);
					}
				});

			},

			matches: function(pattern) {
				if (angular.isUndefined(pattern) || pattern.length === 0) {
					return true;
				}
				var model = this;
				var matchingProps = ['fn', 'title', 'org', 'email', 'nickname', 'note', 'url', 'cloud', 'adr', 'impp', 'tel', 'gender', 'relationship', 'related'].filter(function (propName) {
					if (model.props[propName]) {
						return model.props[propName].filter(function (property) {
							if (!property.value) {
								return false;
							}
							if (angular.isString(property.value)) {
								return property.value.toLowerCase().indexOf(pattern.toLowerCase()) !== -1;
							}
							if (angular.isArray(property.value)) {
								return property.value.filter(function(v) {
									return v.toLowerCase().indexOf(pattern.toLowerCase()) !== -1;
								}).length > 0;
							}
							return false;
						}).length > 0;
					}
					return false;
				});
				return matchingProps.length > 0;
			},

			/* eslint-disable no-console */
			validate: function(prop, property) {
				switch(prop) {
				case 'rev':
				case 'prodid':
				case 'version':
					if (!angular.isUndefined(this.props[prop]) && this.props[prop].length > 1) {
						this.props[prop] = [this.props[prop][0]];
						console.warn(this.uid()+': Too many '+prop+' fields. Saving this one only: ' + this.props[prop][0].value);
						this.failedProps.push(prop);
					}
					break;

				case 'categories':
					// Avoid unescaped commas
					if (angular.isArray(property.value)) {
						if(property.value.join(';').indexOf(',') !== -1) {
							this.failedProps.push(prop);
							property.value = property.value.join(',').split(',');
							//console.warn(this.uid()+': Categories split: ' + property.value);
						}
					} else if (angular.isString(property.value)) {
						if(property.value.indexOf(',') !== -1) {
							this.failedProps.push(prop);
							property.value = property.value.split(',');
							//console.warn(this.uid()+': Categories split: ' + property.value);
						}
					}
					// Remove duplicate categories on array
					if(property.value.length !== 0 && angular.isArray(property.value)) {
						var uniqueCategories = _.unique(property.value);
						if(!angular.equals(uniqueCategories, property.value)) {
							this.failedProps.push(prop);
							property.value = uniqueCategories;
							//console.warn(this.uid()+': Categories duplicate: ' + property.value);
						}
					}
					break;
				case 'photo':
					// Avoid undefined photo type
					if (angular.isDefined(property)) {
						if (angular.isUndefined(property.meta.type)) {
							var mime = MimeService.b64mime(property.value);
							if (mime) {
								this.failedProps.push(prop);
								property.meta.type=[mime];
								this.setProperty('photo', {
									value:property.value,
									meta: {
										type:property.meta.type,
										encoding:property.meta.encoding
									}
								});
								console.warn(this.uid()+': Photo detected as ' + property.meta.type);
							} else {
								this.failedProps.push(prop);
								this.removeProperty('photo', property);
								property = undefined;
								console.warn(this.uid()+': Photo removed');
							}
						}
					}
					break;
				}
				return property;
			},
			/* eslint-enable no-console */

			fix: function() {
				this.validate('rev');
				this.validate('version');
				this.validate('prodid');
				return this.failedProps.indexOf('rev') !== -1
					|| this.failedProps.indexOf('prodid') !== -1
					|| this.failedProps.indexOf('version') !== -1;
			}

		});

		if(angular.isDefined(vCard)) {
			angular.extend(this.data, vCard);
			angular.extend(this.props, $filter('vCard2JSON')(this.data.addressData));
			// We do not want to store our addressbook within contacts
			delete this.data.addressBook;
		} else {
			angular.extend(this.props, {
				version: [{value: '3.0'}],
				fn: [{value: t('contacts', 'New contact')}]
			});
			this.data.addressData = $filter('JSON2vCard')(this.props);
		}

		var property = this.getProperty('categories');
		if(!property) {
			// categories should always have the same type (an array)
			this.categories([]);
		} else {
			if (angular.isString(property.value)) {
				this.categories([property.value]);
			}
		}
	};
}]);

angular.module('contactsApp')
	.factory('Group', function()
	{
		return function Group(data) {
			angular.extend(this, {
				name: '',
				count: 0
			});

			angular.extend(this, data);
		};
	});

angular.module('contactsApp')
.factory('AddressBookService', ['DavClient', 'DavService', 'SettingsService', 'AddressBook', '$q', function(DavClient, DavService, SettingsService, AddressBook, $q) {

	var addressBooks = [];
	var loadPromise = undefined;

	var observerCallbacks = [];

	var notifyObservers = function(eventName, addressBook) {
		var ev = {
			event: eventName,
			addressBooks: addressBooks,
			addressBook: addressBook,
		};
		angular.forEach(observerCallbacks, function(callback) {
			callback(ev);
		});
	};

	var loadAll = function() {
		if (addressBooks.length > 0) {
			return $q.when(addressBooks);
		}
		if (_.isUndefined(loadPromise)) {
			loadPromise = DavService.then(function(account) {
				loadPromise = undefined;
				addressBooks = account.addressBooks.map(function(addressBook) {
					return new AddressBook(addressBook);
				});
			});
		}
		return loadPromise;
	};

	return {
		registerObserverCallback: function(callback) {
			observerCallbacks.push(callback);
		},

		getAll: function() {
			return loadAll().then(function() {
				return addressBooks;
			});
		},

		getGroups: function() {
			return this.getAll().then(function(addressBooks) {
				return addressBooks.map(function (element) {
					return element.groups;
				}).reduce(function(a, b) {
					return a.concat(b);
				});
			});
		},

		getDefaultAddressBook: function(throwOC) {
			var i = addressBooks.findIndex(function(addressBook) {
				return addressBook.enabled && !addressBook.readOnly;
			});
			if (i !== -1) {
				return addressBooks[i];
			} else if(throwOC) {
				OC.Notification.showTemporary(t('contacts', 'There is no address book available to create a contact.'));
			}
			return false;
		},

		getAddressBook: function(displayName) {
			return DavService.then(function(account) {
				return DavClient.getAddressBook({displayName:displayName, url:account.homeUrl}).then(function(res) {
					var addressBook = new AddressBook({
						account: account,
						ctag: res[0].props.getctag,
						url: account.homeUrl+displayName+'/',
						data: res[0],
						displayName: res[0].props.displayname,
						resourcetype: res[0].props.resourcetype,
						syncToken: res[0].props.syncToken
					});
					notifyObservers('create', addressBook);
					return addressBook;
				});
			});
		},

		create: function(displayName) {
			return DavService.then(function(account) {
				return DavClient.createAddressBook({displayName:displayName, url:account.homeUrl});
			});
		},

		delete: function(addressBook) {
			return DavService.then(function() {
				return DavClient.deleteAddressBook(addressBook).then(function() {
					var index = addressBooks.indexOf(addressBook);
					addressBooks.splice(index, 1);
					notifyObservers('delete', addressBook);
				});
			});
		},

		rename: function(addressBook, displayName) {
			return DavService.then(function(account) {
				return DavClient.renameAddressBook(addressBook, {displayName:displayName, url:account.homeUrl});
			});
		},

		get: function(displayName) {
			return this.getAll().then(function(addressBooks) {
				return addressBooks.filter(function (element) {
					return element.displayName === displayName;
				})[0];
			});
		},

		sync: function(addressBook) {
			return DavClient.syncAddressBook(addressBook);
		},

		addContact: function(addressBook, contact) {
			// We don't want to add the same contact again
			if (addressBook.contacts.indexOf(contact) === -1) {
				return addressBook.contacts.push(contact);
			}
		},

		removeContact: function(addressBook, contact) {
			// We can't remove an undefined object
			if (addressBook.contacts.indexOf(contact) !== -1) {
				return addressBook.contacts.splice(addressBook.contacts.indexOf(contact), 1);
			}
		},

		toggleState: function(addressBook) {
			var xmlDoc = document.implementation.createDocument('', '', null);
			var dPropUpdate = xmlDoc.createElement('d:propertyupdate');
			dPropUpdate.setAttribute('xmlns:d', 'DAV:');
			dPropUpdate.setAttribute('xmlns:o', 'http://owncloud.org/ns');
			xmlDoc.appendChild(dPropUpdate);

			var dSet = xmlDoc.createElement('d:set');
			dPropUpdate.appendChild(dSet);

			var dProp = xmlDoc.createElement('d:prop');
			dSet.appendChild(dProp);

			var oEnabled = xmlDoc.createElement('o:enabled');
			// Revert state to toggle
			oEnabled.textContent = !addressBook.enabled ? '1' : '0';
			dProp.appendChild(oEnabled);

			var body = dPropUpdate.outerHTML;

			return DavClient.xhr.send(
				dav.request.basic({method: 'PROPPATCH', data: body}),
				addressBook.url
			).then(function(response) {
				if (response.status === 207) {
					addressBook.enabled = !addressBook.enabled;
					notifyObservers(
						addressBook.enabled ? 'enable' : 'disable',
						addressBook
					);
				}
				return addressBook;
			});
		},

		share: function(addressBook, shareType, shareWith, writable, existingShare) {
			var xmlDoc = document.implementation.createDocument('', '', null);
			var oShare = xmlDoc.createElement('o:share');
			oShare.setAttribute('xmlns:d', 'DAV:');
			oShare.setAttribute('xmlns:o', 'http://owncloud.org/ns');
			xmlDoc.appendChild(oShare);

			var oSet = xmlDoc.createElement('o:set');
			oShare.appendChild(oSet);

			var dHref = xmlDoc.createElement('d:href');
			if (shareType === OC.Share.SHARE_TYPE_USER) {
				dHref.textContent = 'principal:principals/users/';
			} else if (shareType === OC.Share.SHARE_TYPE_GROUP) {
				dHref.textContent = 'principal:principals/groups/';
			}
			dHref.textContent += shareWith;
			oSet.appendChild(dHref);

			var oSummary = xmlDoc.createElement('o:summary');
			oSummary.textContent = t('contacts', '{addressbook} shared by {owner}', {
				addressbook: addressBook.displayName,
				owner: addressBook.owner
			});
			oSet.appendChild(oSummary);

			if (writable) {
				var oRW = xmlDoc.createElement('o:read-write');
				oSet.appendChild(oRW);
			}

			var body = oShare.outerHTML;

			return DavClient.xhr.send(
				dav.request.basic({method: 'POST', data: body}),
				addressBook.url
			).then(function(response) {
				if (response.status === 200) {
					if (!existingShare) {
						if (shareType === OC.Share.SHARE_TYPE_USER) {
							addressBook.sharedWith.users.push({
								id: shareWith,
								displayname: shareWith,
								writable: writable
							});
						} else if (shareType === OC.Share.SHARE_TYPE_GROUP) {
							addressBook.sharedWith.groups.push({
								id: shareWith,
								displayname: shareWith,
								writable: writable
							});
						}
					}
				}
			});

		},

		unshare: function(addressBook, shareType, shareWith) {
			var xmlDoc = document.implementation.createDocument('', '', null);
			var oShare = xmlDoc.createElement('o:share');
			oShare.setAttribute('xmlns:d', 'DAV:');
			oShare.setAttribute('xmlns:o', 'http://owncloud.org/ns');
			xmlDoc.appendChild(oShare);

			var oRemove = xmlDoc.createElement('o:remove');
			oShare.appendChild(oRemove);

			var dHref = xmlDoc.createElement('d:href');
			if (shareType === OC.Share.SHARE_TYPE_USER) {
				dHref.textContent = 'principal:principals/users/';
			} else if (shareType === OC.Share.SHARE_TYPE_GROUP) {
				dHref.textContent = 'principal:principals/groups/';
			}
			dHref.textContent += shareWith;
			oRemove.appendChild(dHref);
			var body = oShare.outerHTML;


			return DavClient.xhr.send(
				dav.request.basic({method: 'POST', data: body}),
				addressBook.url
			).then(function(response) {
				if (response.status === 200) {
					if (shareType === OC.Share.SHARE_TYPE_USER) {
						addressBook.sharedWith.users = addressBook.sharedWith.users.filter(function(user) {
							return user.id !== shareWith;
						});
					} else if (shareType === OC.Share.SHARE_TYPE_GROUP) {
						addressBook.sharedWith.groups = addressBook.sharedWith.groups.filter(function(groups) {
							return groups.id !== shareWith;
						});
					}
					//todo - remove entry from addressbook object
					return true;
				} else {
					return false;
				}
			});

		}


	};

}]);

angular.module('contactsApp')
.service('ContactService', ['DavClient', 'AddressBookService', 'Contact', 'Group', 'ContactFilter', '$q', 'CacheFactory', 'uuid4', function(DavClient, AddressBookService, Contact, Group, ContactFilter, $q, CacheFactory, uuid4) {

	var contactService = this;

	var cacheFilled = false;
	var contactsCache = CacheFactory('contacts');
	var observerCallbacks = [];
	var loadPromise = undefined;

	var allUpdates = $q.when();
	this.queueUpdate = function(contact) {
		allUpdates = allUpdates.then(function() {
			return contactService.update(contact);
		});
	};

	this.registerObserverCallback = function(callback) {
		observerCallbacks.push(callback);
	};

	var notifyObservers = function(eventName, uid) {
		var ev = {
			event: eventName,
			uid: uid,
			contacts: contactsCache.values()
		};
		angular.forEach(observerCallbacks, function(callback) {
			callback(ev);
		});
	};

	this.getFullContacts = function(contacts) {
		AddressBookService.getAll().then(function(addressBooks) {
			var promises = [];
			var xhrAddressBooks = [];
			contacts.forEach(function(contact) {
				// Regroup urls by addressbooks
				if(addressBooks.indexOf(contact.addressBook) !== -1) {
					// Initiate array if no exists
					xhrAddressBooks[contact.addressBookId] = xhrAddressBooks[contact.addressBookId] || [];
					xhrAddressBooks[contact.addressBookId].push(contact.data.url);
				}
			});
			// Get our full vCards
			addressBooks.forEach(function(addressBook) {
				// Only go through enabled addressbooks
				// Though xhrAddressBooks does not contains contacts from disabled ones
				if(addressBook.enabled) {
					if(angular.isArray(xhrAddressBooks[addressBook.displayName])) {
						var promise = DavClient.getContacts(addressBook, {}, xhrAddressBooks[addressBook.displayName]).then(
							function(vcards) {
								return vcards.map(function(vcard) {
									return new Contact(addressBook, vcard);
								});
							}).then(function(contacts_) {
								contacts_.map(function(contact) {
									// Validate some fields
									if(contact.fix()) {
										// Can't use `this` in those nested functions
										contactService.update(contact);
									}
									contactsCache.put(contact.uid(), contact);
									addressBook.contacts.push(contact);
								});
							});
						promises.push(promise);
					}
				}
			});
			$q.all(promises).then(function() {
				notifyObservers('getFullContacts', '');
			});
		});
	};

	this.fillCache = function() {
		if (_.isUndefined(loadPromise)) {
			loadPromise = AddressBookService.getAll().then(function(addressBooks) {
				var promises = [];
				addressBooks.forEach(function(addressBook) {
					// Only go through enabled addressbooks
					if(addressBook.enabled) {
						promises.push(
							AddressBookService.sync(addressBook).then(function(addressBook) {
								contactService.appendContactsFromAddressbook(addressBook);
							})
						);
					}
				});
				return $q.all(promises).then(function() {
					cacheFilled = true;
				});
			});
		}
		return loadPromise;
	};

	this.getAll = function() {
		if(cacheFilled === false) {
			return this.fillCache().then(function() {
				return contactsCache.values();
			});
		} else {
			return $q.when(contactsCache.values());
		}
	};

	this.getContactFilters = function() {
		return this.getAll().then(function(contacts) {
			var allContacts = new ContactFilter({
				name: t('contacts', 'All contacts'),
				count: contacts.length
			});
			var notGrouped = new ContactFilter({
				name: t('contacts', 'Not grouped'),
				count: contacts.filter(
					function(contact) {
						return contact.categories().length === 0;
					}).length
			});
			var filters = [allContacts];
			// Only have Not Grouped if at least one contact in it
			if(notGrouped.count !== 0) {
				filters.push(notGrouped);
			}

			return filters;
		});
	};

	// get list of groups and the count of contacts in said groups
	this.getGroupList = function() {
		return this.getAll().then(function(contacts) {
			// allow groups with names such as toString
			var groups = Object.create(null);

			// collect categories and their associated counts
			contacts.forEach(function(contact) {
				contact.categories().forEach(function(category) {
					groups[category] = groups[category] ? groups[category] + 1 : 1;
				});
			});
			return _.keys(groups).map(
				function(key) {
					return new Group({
						name: key,
						count: groups[key]
					});
				});
		});
	};

	this.getGroups = function() {
		return this.getAll().then(function(contacts) {
			return _.uniq(contacts.map(function(element) {
				return element.categories();
			}).reduce(function(a, b) {
				return a.concat(b);
			}, []).sort(), true);
		});
	};

	this.getById = function(addressBooks, uid) {
		return (function() {
			if(cacheFilled === false) {
				return this.fillCache().then(function() {
					return contactsCache.get(uid);
				});
			} else {
				return $q.when(contactsCache.get(uid));
			}
		}).call(this)
			.then(function(contact) {
				if(angular.isUndefined(contact)) {
					OC.Notification.showTemporary(t('contacts', 'Contact not found.'));
					return;
				} else {
					var addressBook = addressBooks.find(function(book) {
						return book.displayName === contact.addressBookId;
					});
					// Fetch and return full contact vcard
					return addressBook
						? DavClient.getContacts(addressBook, {}, [ contact.data.url ]).then(function(vcards) {
							return new Contact(addressBook, vcards[0]);
						}).then(function(newContact) {
							contactsCache.put(contact.uid(), newContact);
							var contactIndex = addressBook.contacts.findIndex(function(testedContact) {
								return testedContact.uid() === contact.uid();
							});
							addressBook.contacts[contactIndex] = newContact;
							notifyObservers('getFullContacts', contact.uid());
							return newContact;
						}) : contact;
				}
			});
	};

	this.create = function(newContact, addressBook, uid, fromImport) {
		addressBook = addressBook || AddressBookService.getDefaultAddressBook(true);

		// No addressBook available
		if(!addressBook) {
			return;
		}

		if(addressBook.readOnly) {
			OC.Notification.showTemporary(t('contacts', 'You don\'t have permission to write to this addressbook.'));
			return;
		}
		try {
			newContact = newContact || new Contact(addressBook);
		} catch(error) {
			OC.Notification.showTemporary(t('contacts', 'Contact could not be created.'));
			return;
		}
		var newUid = '';
		if(uuid4.validate(uid)) {
			newUid = uid;
		} else {
			newUid = uuid4.generate();
		}
		newContact.uid(newUid);
		newContact.setUrl(addressBook, newUid);
		newContact.addressBookId = addressBook.displayName;
		if (_.isUndefined(newContact.fullName()) || newContact.fullName() === '') {
			newContact.fullName(newContact.displayName());
		}

		return DavClient.createCard(
			addressBook,
			{
				data: newContact.data.addressData,
				filename: newUid + '.vcf'
			}
		).then(function(xhr) {
			newContact.setETag(xhr.getResponseHeader('OC-ETag') || xhr.getResponseHeader('ETag'));
			contactsCache.put(newUid, newContact);
			AddressBookService.addContact(addressBook, newContact);
			if (fromImport !== true) {
				notifyObservers('create', newUid);
				$('#details-fullName').select();
			}
			return newContact;
		}).catch(function() {
			OC.Notification.showTemporary(t('contacts', 'Contact could not be created.'));
			return false;
		});
	};

	this.import = function(data, type, addressBook, progressCallback) {
		addressBook = addressBook || AddressBookService.getDefaultAddressBook(true);

		// No addressBook available
		if(!addressBook) {
			return;
		}

		var regexp = /BEGIN:VCARD[\s\S]*?END:VCARD/mgi;
		var singleVCards = data.match(regexp);

		if (!singleVCards) {
			OC.Notification.showTemporary(t('contacts', 'No contacts in file. Only vCard files are allowed.'));
			if (progressCallback) {
				progressCallback(1);
			}
			return;
		}

		notifyObservers('importstart');

		var num = 1;
		for(var i in singleVCards) {
			var newContact = new Contact(addressBook, {addressData: singleVCards[i]});
			if (['3.0', '4.0'].indexOf(newContact.version()) < 0) {
				if (progressCallback) {
					progressCallback(num / singleVCards.length);
				}
				OC.Notification.showTemporary(t('contacts', 'Only vCard version 4.0 (RFC6350) or version 3.0 (RFC2426) are supported.'));
				num++;
				continue;
			}
			// eslint-disable-next-line no-loop-func
			this.create(newContact, addressBook, '', true).then(function(xhrContact) {
				if (xhrContact !== false) {
					var xhrContactName = xhrContact.displayName();
				}
				// Update the progress indicator
				if (progressCallback) {
					progressCallback(num / singleVCards.length, xhrContactName);
				}
				num++;
				/* Import is over, let's notify */
				if (num === singleVCards.length + 1) {
					notifyObservers('importend');
				}
			});
		}
	};

	this.moveContact = function(contact, addressBook, oldAddressBook) {
		if (addressBook !== null && contact.addressBookId === addressBook.displayName) {
			return;
		}
		if (addressBook.readOnly) {
			OC.Notification.showTemporary(t('contacts', 'You don\'t have permission to write to this addressbook.'));
			return;
		}
		contact.syncVCard();

		DavClient.xhr.send(
			dav.request.basic({method: 'MOVE', destination: addressBook.url + contact.data.url.split('/').pop(-1)}),
			contact.data.url
		).then(function(response) {
			if (response.status === 201 || response.status === 204) {
				contact.setAddressBook(addressBook);
				AddressBookService.addContact(addressBook, contact);
				AddressBookService.removeContact(oldAddressBook, contact);
				notifyObservers('groupsUpdate');
			} else {
				OC.Notification.showTemporary(t('contacts', 'Contact could not be moved.'));
			}
		});
	};

	this.update = function(contact) {
		// update rev field
		contact.syncVCard();

		// update contact on server
		return DavClient.updateCard(contact.data, {json: true}).then(function(xhr) {
			var newEtag = xhr.getResponseHeader('OC-ETag') || xhr.getResponseHeader('ETag');
			contact.setETag(newEtag);
			notifyObservers('update', contact.uid());
		}).catch(function() {
			OC.Notification.showTemporary(t('contacts', 'Contact could not be saved.'));
		});
	};

	this.delete = function(addressBook, contact) {
		// delete contact from server
		return DavClient.deleteCard(contact.data).then(function() {
			contactsCache.remove(contact.uid());
			AddressBookService.removeContact(addressBook, contact);
			notifyObservers('delete', contact.uid());
		});
	};

	/*
	 * Delete all contacts present in the addressBook from the cache
	 */
	this.removeContactsFromAddressbook = function(addressBook, callback) {
		angular.forEach(addressBook.contacts, function(contact) {
			contactsCache.remove(contact.uid());
		});
		callback();
		notifyObservers('groupsUpdate');
	};

	/*
	 * Create and append contacts to the addressBook
	 */
	this.appendContactsFromAddressbook = function(addressBook, callback) {
		// Addressbook has been initiated but contacts have not been fetched
		if (addressBook.objects === null) {
			AddressBookService.sync(addressBook).then(function(addressBook) {
				contactService.appendContactsFromAddressbook(addressBook, callback);
			});
		} else if (addressBook.contacts.length === 0) {
			// Only add contact if the addressBook doesn't already have it
			addressBook.objects.forEach(function(vcard) {
				try {
					// Only add contact if the addressBook doesn't already have it
					var contact = new Contact(addressBook, vcard);
					contactsCache.put(contact.uid(), contact);
					AddressBookService.addContact(addressBook, contact);
				} catch(error) {
					// eslint-disable-next-line no-console
					console.log('Invalid contact received: ', vcard, error);
				}
			});
		} else {
			// Contact are already present in the addressBook
			angular.forEach(addressBook.contacts, function(contact) {
				contactsCache.put(contact.uid(), contact);
			});
		}
		notifyObservers('groupsUpdate');
		if (typeof callback === 'function') {
			callback();
		}
	};

}]);

angular.module('contactsApp')
.service('DavClient', function() {
	var xhr = new dav.transport.Basic(
		new dav.Credentials()
	);
	return new dav.Client(xhr);
});

angular.module('contactsApp')
.service('DavService', ['DavClient', function(DavClient) {
	return DavClient.createAccount({
		server: OC.linkToRemote('dav/addressbooks'),
		accountType: 'carddav',
		useProvidedPath: true
	});
}]);

angular.module('contactsApp')
.service('ImportService', function() {

	this.importing = false;
	this.selectedAddressBook = t('contacts', 'Import into');
	this.importedUser = t('contacts', 'Waiting for the server to be ready…');
	this.importPercent = 0;

	this.t = {
		importText : t('contacts', 'Import into'),
		importingText : t('contacts', 'Importing…')
	};

});

angular.module('contactsApp')
	.service('MimeService', function() {
		var magicNumbers = {
			'/9j/' : 'JPEG',
			'R0lGOD' : 'GIF',
			'iVBORw0KGgo' : 'PNG'
		};

		this.b64mime = function(b64string) {
			for (var mn in magicNumbers) {
				if(b64string.startsWith(mn)) return magicNumbers[mn];
			}
			return null;
		};
	});

angular.module('contactsApp').service('SearchService', function() {
	var searchTerm = '';

	var observerCallbacks = [];

	this.registerObserverCallback = function(callback) {
		observerCallbacks.push(callback);
	};

	var notifyObservers = function(eventName) {
		var ev = {
			event: eventName,
			searchTerm: searchTerm
		};
		angular.forEach(observerCallbacks, function(callback) {
			callback(ev);
		});
	};

	var SearchProxy = {
		attach: function(search) {
			search.setFilter('contacts', this.filterProxy);
		},
		filterProxy: function(query) {
			searchTerm = query;
			notifyObservers('changeSearch');
		}
	};

	this.getSearchTerm = function() {
		return searchTerm;
	};

	this.cleanSearch = function() {
		searchTerm = '';
		notifyObservers('changeSearch');
	};

	this.search = function(search) {
		searchTerm = search;
		notifyObservers('submitSearch');
	};

	if (!_.isUndefined(OC.Plugins)) {
		OC.Plugins.register('OCA.Search', SearchProxy);
		if (!_.isUndefined(OCA.Search)) {
			OC.Search = new OCA.Search(this.search, this.cleanSearch);
		}
	}
});

angular.module('contactsApp')
.service('SettingsService', function() {
	var settings = {
		addressBooks: [
			'testAddr'
		]
	};

	this.set = function(key, value) {
		settings[key] = value;
	};

	this.get = function(key) {
		return settings[key];
	};

	this.getAll = function() {
		return settings;
	};
});

angular.module('contactsApp')
.service('SortByService', function () {
	var subscriptions = [];

	// Array of keys to sort by. Ordered by priorities.
	var sortOptions = {
		sortFirstName: ['firstName', 'lastName', 'uid'],
		sortLastName: ['lastName', 'firstName', 'uid'],
		sortDisplayName: ['displayName', 'uid']
	};

	// Key
	var sortBy = 'sortDisplayName';

	var defaultOrder = window.localStorage.getItem('contacts_default_order');
	if (defaultOrder) {
		sortBy = defaultOrder;
	}

	function notifyObservers() {
		angular.forEach(subscriptions, function (subscription) {
			if (typeof subscription === 'function') {
				subscription(sortOptions[sortBy]);
			}
		});
	}

	return {
		subscribe: function (callback) {
			subscriptions.push(callback);
		},
		setSortBy: function (value) {
			sortBy = value;
			window.localStorage.setItem('contacts_default_order', value);
			notifyObservers();
		},
		getSortBy: function () {
			return sortOptions[sortBy];
		},
		getSortByKey: function () {
			return sortBy;
		},
		getSortByList: function () {
			return {
				sortDisplayName: t('contacts', 'Display name'),
				sortFirstName: t('contacts', 'First name'),
				sortLastName: t('contacts', 'Last name')
			};
		}
	};
});

angular.module('contactsApp')
.service('vCardPropertiesService', function() {
	/**
	 * map vCard attributes to internal attributes
	 *
	 * propName: {
	 * 		multiple: [Boolean], // is this prop allowed more than once? (default = false)
	 * 		readableName: [String], // internationalized readable name of prop
	 * 		template: [String], // template name found in /templates/detailItems
	 * 		[...] // optional additional information which might get used by the template
	 *
	 *		options: If multiple options have the same name, the first will be used as default.
	 *				 Others will be merge, but still supported. Order is important!
	 * }
	 */
	this.vCardMeta = {
		nickname: {
			readableName: t('contacts', 'Nickname'),
			template: 'text',
			icon: 'icon-user'
		},
		n: {
			readableName: t('contacts', 'Detailed name'),
			defaultValue: {
				value:['', '', '', '', '']
			},
			template: 'n',
			icon: 'icon-user'
		},
		note: {
			readableName: t('contacts', 'Notes'),
			template: 'textarea',
			icon: 'icon-rename'
		},
		url: {
			multiple: true,
			readableName: t('contacts', 'Website'),
			template: 'url',
			icon: 'icon-public'
		},
		cloud: {
			multiple: true,
			readableName: t('contacts', 'Federated Cloud ID'),
			template: 'text',
			defaultValue: {
				value:[''],
				meta:{type:['HOME']}
			},
			options: [
				{id: 'HOME', name: t('contacts', 'Home')},
				{id: 'WORK', name: t('contacts', 'Work')},
				{id: 'OTHER', name: t('contacts', 'Other')}
			]		},
		adr: {
			multiple: true,
			readableName: t('contacts', 'Address'),
			template: 'adr',
			icon: 'icon-address',
			defaultValue: {
				value:['', '', '', '', '', '', ''],
				meta:{type:['HOME']}
			},
			options: [
				{id: 'HOME', name: t('contacts', 'Home')},
				{id: 'WORK', name: t('contacts', 'Work')},
				{id: 'OTHER', name: t('contacts', 'Other')}
			]
		},
		categories: {
			readableName: t('contacts', 'Groups'),
			template: 'groups'
		},
		bday: {
			readableName: t('contacts', 'Birthday'),
			template: 'date',
			icon: 'icon-calendar-dark'
		},
		anniversary: {
			readableName: t('contacts', 'Anniversary'),
			template: 'date',
			icon: 'icon-calendar-dark'
		},
		deathdate: {
			readableName: t('contacts', 'Date of death'),
			template: 'date',
			icon: 'icon-calendar-dark'
		},
		email: {
			multiple: true,
			readableName: t('contacts', 'Email'),
			template: 'email',
			icon: 'icon-mail',
			defaultValue: {
				value:'',
				meta:{type:['HOME']}
			},
			options: [
				{id: 'HOME', name: t('contacts', 'Home')},
				{id: 'WORK', name: t('contacts', 'Work')},
				{id: 'OTHER', name: t('contacts', 'Other')}
			]
		},
		impp: {
			multiple: true,
			readableName: t('contacts', 'Instant messaging'),
			template: 'username',
			icon: 'icon-comment',
			defaultValue: {
				value:[''],
				meta:{type:['SKYPE']}
			},
			options: [
				{id: 'IRC', name: 'IRC'},
				{id: 'KIK', name: 'KiK'},
				{id: 'SKYPE', name: 'Skype'},
				{id: 'TELEGRAM', name: 'Telegram'},
				{id: 'XMPP', name:'XMPP'}
			]
		},
		tel: {
			multiple: true,
			readableName: t('contacts', 'Phone'),
			template: 'tel',
			icon: 'icon-comment',
			defaultValue: {
				value:'',
				meta:{type:['HOME,VOICE']}
			},
			options: [
				{id: 'HOME,VOICE', name: t('contacts', 'Home')},
				{id: 'HOME', name: t('contacts', 'Home')},
				{id: 'WORK,VOICE', name: t('contacts', 'Work')},
				{id: 'WORK', name: t('contacts', 'Work')},
				{id: 'CELL', name: t('contacts', 'Mobile')},
				{id: 'CELL,VOICE', name: t('contacts', 'Mobile')},
				{id: 'WORK,CELL', name: t('contacts', 'Work mobile')},
				{id: 'FAX', name: t('contacts', 'Fax')},
				{id: 'HOME,FAX', name: t('contacts', 'Fax home')},
				{id: 'WORK,FAX', name: t('contacts', 'Fax work')},
				{id: 'PAGER', name: t('contacts', 'Pager')},
				{id: 'VOICE', name: t('contacts', 'Voice')},
				{id: 'CAR', name: t('contacts', 'Car')},
				{id: 'PAGER', name: t('contacts', 'Pager')},
				{id: 'WORK,PAGER', name: t('contacts', 'Work pager')}
			]
		},
		'X-SOCIALPROFILE': {
			multiple: true,
			readableName: t('contacts', 'Social network'),
			template: 'username',
			defaultValue: {
				value:[''],
				meta:{type:['facebook']}
			},
			options: [
				{id: 'FACEBOOK', name: 'Facebook'},
				{id: 'GITHUB', name: 'GitHub'},
				{id: 'GOOGLEPLUS', name: 'Google+'},
				{id: 'INSTAGRAM', name: 'Instagram'},
				{id: 'LINKEDIN', name: 'LinkedIn'},
				{id: 'PINTEREST', name: 'Pinterest'},
				{id: 'QZONE', name: 'QZone'},
				{id: 'TUMBLR', name: 'Tumblr'},
				{id: 'TWITTER', name: 'Twitter'},
				{id: 'WECHAT', name: 'WeChat'},
				{id: 'YOUTUBE', name: 'YouTube'}


			]
		},
		relationship: {
			readableName: t('contacts', 'Relationship'),
			template: 'select',
			info: t('contacts', 'Specify a relationship between you and the entity represented by this vCard.'),
			options: [
				{id: 'SPOUSE', name: t('contacts', 'Spouse')},
				{id: 'CHILD', name: t('contacts', 'Child')},
				{id: 'MOTHER', name: t('contacts', 'Mother')},
				{id: 'FATHER', name: t('contacts', 'Father')},
				{id: 'PARENT', name: t('contacts', 'Parent')},
				{id: 'BROTHER', name: t('contacts', 'Brother')},
				{id: 'SISTER', name: t('contacts', 'Sister')},
				{id: 'RELATIVE', name: t('contacts', 'Relative')},
				{id: 'FRIEND', name: t('contacts', 'Friend')},
				{id: 'COLLEAGUE', name: t('contacts', 'Colleague')},
				{id: 'MANAGER', name: t('contacts', 'Manager')},
				{id: 'ASSISTANT', name: t('contacts', 'Assistant')},
			]
		},
		related: {
			multiple: true,
			readableName: t('contacts', 'Related'),
			template: 'text',
			info: t('contacts', 'Specify a relationship between another entity and the entity represented by this vCard.'),
			defaultValue: {
				value:[''],
				meta:{type:['CONTACT']}
			},
			options: [
				{id: 'CONTACT', name: t('contacts', 'Contact')},
				{id: 'AGENT', name: t('contacts', 'Agent')},
				{id: 'EMERGENCY', name: t('contacts', 'Emergency')},
				{id: 'FRIEND', name: t('contacts', 'Friend')},
				{id: 'COLLEAGUE', name: t('contacts', 'Colleague')},
				{id: 'COWORKER', name: t('contacts', 'Co-worker')},
				{id: 'MANAGER', name: t('contacts', 'Manager')},
				{id: 'ASSISTANT', name: t('contacts', 'Assistant')},
				{id: 'SPOUSE', name: t('contacts', 'Spouse')},
				{id: 'CHILD', name: t('contacts', 'Child')},
				{id: 'MOTHER', name: t('contacts', 'Mother')},
				{id: 'FATHER', name: t('contacts', 'Father')},
				{id: 'PARENT', name: t('contacts', 'Parent')},
				{id: 'BROTHER', name: t('contacts', 'Brother')},
				{id: 'SISTER', name: t('contacts', 'Sister')},
				{id: 'RELATIVE', name: t('contacts', 'Relative')}
			]
		},
		gender: {
			readableName: t('contacts', 'Gender'),
			template: 'select',
			options: [
				{id: 'F', name: t('contacts', 'Female')},
				{id: 'M', name: t('contacts', 'Male')},
				{id: 'O', name: t('contacts', 'Other')}
			]
		}
	};

	this.fieldOrder = [
		'org',
		'title',
		'tel',
		'email',
		'adr',
		'impp',
		'nick',
		'bday',
		'anniversary',
		'deathdate',
		'url',
		'X-SOCIALPROFILE',
		'relationship',
		'related',
		'note',
		'categories',
		'role',
		'gender'
	];

	this.fieldDefinitions = [];
	for (var prop in this.vCardMeta) {
		this.fieldDefinitions.push({id: prop, name: this.vCardMeta[prop].readableName, multiple: !!this.vCardMeta[prop].multiple});
	}

	this.fallbackMeta = function(property) {
		function capitalize(string) { return string.charAt(0).toUpperCase() + string.slice(1); }
		return {
			name: 'unknown-' + property,
			readableName: capitalize(property),
			template: 'hidden',
			necessity: 'optional',
			hidden: true
		};
	};

	this.getMeta = function(property) {
		return this.vCardMeta[property] || this.fallbackMeta(property);
	};

});

angular.module('contactsApp')
.filter('JSON2vCard', function() {
	return function(input) {
		return vCard.generate(input);
	};
});

angular.module('contactsApp')
.filter('contactColor', function() {
	return function(input) {
		// Check if core has the new color generator
		if(typeof input.toRgb === 'function') {
			var rgb = input.toRgb();
			return 'rgb('+rgb['r']+', '+rgb['g']+', '+rgb['b']+')';
		} else if(typeof input.toHsl === 'function') {
			var hsl = input.toHsl();
			return 'hsl('+hsl[0]+', '+hsl[1]+'%, '+hsl[2]+'%)';
		} else {
			// If not, we use the old one
			/* global md5 */
			var hash = md5(input).substring(0, 4),
				maxRange = parseInt('ffff', 16),
				hue = parseInt(hash, 16) / maxRange * 256;
			return 'hsl(' + hue + ', 90%, 65%)';
		}
	};
});
angular.module('contactsApp')
.filter('contactGroupFilter', function() {
	'use strict';
	return function (contacts, group) {
		if (typeof contacts === 'undefined') {
			return contacts;
		}
		if (typeof group === 'undefined' || group.toLowerCase() === t('contacts', 'All contacts').toLowerCase()) {
			return contacts;
		}
		var filter = [];
		if (contacts.length > 0) {
			for (var i = 0; i < contacts.length; i++) {
				if (group.toLowerCase() === t('contacts', 'Not grouped').toLowerCase()) {
					if (contacts[i].categories().length === 0) {
						filter.push(contacts[i]);
					}
				} else {
					if (contacts[i].categories().indexOf(group) >= 0) {
						filter.push(contacts[i]);
					}
				}
			}
		}
		return filter;
	};
});

// from https://docs.nextcloud.com/server/11/developer_manual/app/css.html#menus
angular.module('contactsApp')
.filter('counterFormatter', function () {
	'use strict';
	return function (count) {
		if (count > 9999) {
			return '9999+';
		}
		if (count === 0) {
			return '';
		}
		return count;
	};
});


angular.module('contactsApp')
.filter('counterTooltipDisplay', function () {
	'use strict';
	return function (count) {
		if (count > 9999) {
			return count;
		}
		return '';
	};
});



angular.module('contactsApp')
.filter('fieldFilter', function() {
	'use strict';
	return function (fields, contact) {
		if (typeof fields === 'undefined') {
			return fields;
		}
		if (typeof contact === 'undefined') {
			return fields;
		}
		var filter = [];
		if (fields.length > 0) {
			for (var i = 0; i < fields.length; i++) {
				if (fields[i].multiple ) {
					filter.push(fields[i]);
					continue;
				}
				if (_.isUndefined(contact.getProperty(fields[i].id))) {
					filter.push(fields[i]);
				}
			}
		}
		return filter;
	};
});

angular.module('contactsApp')
.filter('firstCharacter', function() {
	return function(input) {
		return input.charAt(0);
	};
});

angular.module('contactsApp')
.filter('localeOrderBy', [function () {
	return function (array, sortPredicate, reverseOrder) {
		if (!Array.isArray(array)) return array;
		if (!sortPredicate) return array;

		var arrayCopy = [];
		angular.forEach(array, function (item) {
			arrayCopy.push(item);
		});

		arrayCopy.sort(function (a, b) {


			// Did we pass multiple sorting options? If not, create an array anyway.
			sortPredicate = angular.isArray(sortPredicate) ? sortPredicate: [sortPredicate];
			// Let's test the first sort and continue if no sort occured
			for(var i=0; i<sortPredicate.length; i++) {
				var sortBy = sortPredicate[i];

				var valueA = a[sortBy];
				if (angular.isFunction(valueA)) {
					valueA = a[sortBy]();
				}
				var valueB = b[sortBy];
				if (angular.isFunction(valueB)) {
					valueB = b[sortBy]();
				}

				// Start sorting
				if (angular.isString(valueA)) {
					if(valueA !== valueB) {
						return reverseOrder ? valueB.localeCompare(valueA) : valueA.localeCompare(valueB);
					}
				}

				if (angular.isNumber(valueA) || typeof valueA === 'boolean') {
					if(valueA !== valueB) {
						return reverseOrder ? valueB - valueA : valueA - valueB;
					}
				}
			}

			return 0;
		});

		return arrayCopy;
	};
}]);

angular.module('contactsApp')
.filter('newContact', function() {
	return function(input) {
		return input !== '' ? input : t('contacts', 'New contact');
	};
});

angular.module('contactsApp')
.filter('orderDetailItems', ['vCardPropertiesService', function(vCardPropertiesService) {
	'use strict';
	return function(items, field, reverse) {

		var filtered = [];
		angular.forEach(items, function(item) {
			filtered.push(item);
		});

		var fieldOrder = angular.copy(vCardPropertiesService.fieldOrder);
		// reverse to move custom items to the end (indexOf == -1)
		fieldOrder.reverse();

		filtered.sort(function (a, b) {
			if(fieldOrder.indexOf(a[field]) < fieldOrder.indexOf(b[field])) {
				return 1;
			}
			if(fieldOrder.indexOf(a[field]) > fieldOrder.indexOf(b[field])) {
				return -1;
			}
			return 0;
		});

		if(reverse) filtered.reverse();
		return filtered;
	};
}]);

angular.module('contactsApp')
.filter('toArray', function() {
	return function(obj) {
		if (!(obj instanceof Object)) return obj;
		return _.map(obj, function(val, key) {
			return Object.defineProperty(val, '$key', {value: key});
		});
	};
});

angular.module('contactsApp')
.filter('vCard2JSON', function() {
	return function(input) {
		return vCard.parse(input);
	};
});

//# sourceMappingURL=data:application/json;charset=utf8;base64,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
