/*
 * Decompiled with CFR 0.152.
 */
package org.apache.lucene.util.bkd;

import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.function.IntFunction;
import org.apache.lucene.codecs.CodecUtil;
import org.apache.lucene.codecs.MutablePointValues;
import org.apache.lucene.index.MergeState;
import org.apache.lucene.index.PointValues;
import org.apache.lucene.store.ChecksumIndexInput;
import org.apache.lucene.store.DataOutput;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.GrowableByteArrayDataOutput;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexOutput;
import org.apache.lucene.store.RAMOutputStream;
import org.apache.lucene.store.TrackingDirectoryWrapper;
import org.apache.lucene.util.ArrayUtil;
import org.apache.lucene.util.BytesRef;
import org.apache.lucene.util.BytesRefBuilder;
import org.apache.lucene.util.BytesRefComparator;
import org.apache.lucene.util.FixedBitSet;
import org.apache.lucene.util.FutureArrays;
import org.apache.lucene.util.IOUtils;
import org.apache.lucene.util.LongBitSet;
import org.apache.lucene.util.MSBRadixSorter;
import org.apache.lucene.util.NumericUtils;
import org.apache.lucene.util.OfflineSorter;
import org.apache.lucene.util.PriorityQueue;
import org.apache.lucene.util.bkd.BKDReader;
import org.apache.lucene.util.bkd.DocIdsWriter;
import org.apache.lucene.util.bkd.HeapPointWriter;
import org.apache.lucene.util.bkd.MutablePointsReaderUtils;
import org.apache.lucene.util.bkd.OfflinePointWriter;
import org.apache.lucene.util.bkd.PointReader;
import org.apache.lucene.util.bkd.PointWriter;

public class BKDWriter
implements Closeable {
    public static final String CODEC_NAME = "BKD";
    public static final int VERSION_START = 0;
    public static final int VERSION_COMPRESSED_DOC_IDS = 1;
    public static final int VERSION_COMPRESSED_VALUES = 2;
    public static final int VERSION_IMPLICIT_SPLIT_DIM_1D = 3;
    public static final int VERSION_PACKED_INDEX = 4;
    public static final int VERSION_LEAF_STORES_BOUNDS = 5;
    public static final int VERSION_CURRENT = 5;
    private final int bytesPerDoc;
    public static final int DEFAULT_MAX_POINTS_IN_LEAF_NODE = 1024;
    public static final float DEFAULT_MAX_MB_SORT_IN_HEAP = 16.0f;
    public static final int MAX_DIMS = 8;
    protected final int numDims;
    protected final int bytesPerDim;
    protected final int packedBytesLength;
    final TrackingDirectoryWrapper tempDir;
    final String tempFileNamePrefix;
    final double maxMBSortInHeap;
    final byte[] scratchDiff;
    final byte[] scratch1;
    final byte[] scratch2;
    final BytesRef scratchBytesRef1 = new BytesRef();
    final BytesRef scratchBytesRef2 = new BytesRef();
    final int[] commonPrefixLengths;
    protected final FixedBitSet docsSeen;
    private OfflinePointWriter offlinePointWriter;
    private HeapPointWriter heapPointWriter;
    private IndexOutput tempInput;
    protected final int maxPointsInLeafNode;
    private final int maxPointsSortInHeap;
    protected final byte[] minPackedValue;
    protected final byte[] maxPackedValue;
    protected long pointCount;
    protected final boolean longOrds;
    private final long totalPointCount;
    protected final boolean singleValuePerDoc;
    protected final OfflineSorter.BufferSize offlineSorterBufferMB;
    protected final int offlineSorterMaxTempFiles;
    private final int maxDoc;
    private final GrowableByteArrayDataOutput scratchOut = new GrowableByteArrayDataOutput(32768);

    public BKDWriter(int maxDoc, Directory tempDir, String tempFileNamePrefix, int numDims, int bytesPerDim, int maxPointsInLeafNode, double maxMBSortInHeap, long totalPointCount, boolean singleValuePerDoc) throws IOException {
        this(maxDoc, tempDir, tempFileNamePrefix, numDims, bytesPerDim, maxPointsInLeafNode, maxMBSortInHeap, totalPointCount, singleValuePerDoc, totalPointCount > Integer.MAX_VALUE, Math.max(1L, (long)maxMBSortInHeap), 10);
    }

    protected BKDWriter(int maxDoc, Directory tempDir, String tempFileNamePrefix, int numDims, int bytesPerDim, int maxPointsInLeafNode, double maxMBSortInHeap, long totalPointCount, boolean singleValuePerDoc, boolean longOrds, long offlineSorterBufferMB, int offlineSorterMaxTempFiles) throws IOException {
        BKDWriter.verifyParams(numDims, maxPointsInLeafNode, maxMBSortInHeap, totalPointCount);
        this.tempDir = new TrackingDirectoryWrapper(tempDir);
        this.tempFileNamePrefix = tempFileNamePrefix;
        this.maxPointsInLeafNode = maxPointsInLeafNode;
        this.numDims = numDims;
        this.bytesPerDim = bytesPerDim;
        this.totalPointCount = totalPointCount;
        this.maxDoc = maxDoc;
        this.offlineSorterBufferMB = OfflineSorter.BufferSize.megabytes(offlineSorterBufferMB);
        this.offlineSorterMaxTempFiles = offlineSorterMaxTempFiles;
        this.docsSeen = new FixedBitSet(maxDoc);
        this.packedBytesLength = numDims * bytesPerDim;
        this.scratchDiff = new byte[bytesPerDim];
        this.scratch1 = new byte[this.packedBytesLength];
        this.scratch2 = new byte[this.packedBytesLength];
        this.commonPrefixLengths = new int[numDims];
        this.minPackedValue = new byte[this.packedBytesLength];
        this.maxPackedValue = new byte[this.packedBytesLength];
        this.longOrds = longOrds;
        this.singleValuePerDoc = singleValuePerDoc;
        if (singleValuePerDoc) {
            assert (!longOrds);
            this.bytesPerDoc = this.packedBytesLength + 4;
        } else {
            this.bytesPerDoc = longOrds ? this.packedBytesLength + 8 + 4 : this.packedBytesLength + 4 + 4;
        }
        this.maxPointsSortInHeap = (int)(0.5 * (maxMBSortInHeap * 1024.0 * 1024.0) / (double)(this.bytesPerDoc * numDims));
        if (this.maxPointsSortInHeap < maxPointsInLeafNode) {
            throw new IllegalArgumentException("maxMBSortInHeap=" + maxMBSortInHeap + " only allows for maxPointsSortInHeap=" + this.maxPointsSortInHeap + ", but this is less than maxPointsInLeafNode=" + maxPointsInLeafNode + "; either increase maxMBSortInHeap or decrease maxPointsInLeafNode");
        }
        this.heapPointWriter = new HeapPointWriter(16, this.maxPointsSortInHeap, this.packedBytesLength, longOrds, singleValuePerDoc);
        this.maxMBSortInHeap = maxMBSortInHeap;
    }

    public static void verifyParams(int numDims, int maxPointsInLeafNode, double maxMBSortInHeap, long totalPointCount) {
        if (numDims < 1 || numDims > 8) {
            throw new IllegalArgumentException("numDims must be 1 .. 8 (got: " + numDims + ")");
        }
        if (maxPointsInLeafNode <= 0) {
            throw new IllegalArgumentException("maxPointsInLeafNode must be > 0; got " + maxPointsInLeafNode);
        }
        if (maxPointsInLeafNode > ArrayUtil.MAX_ARRAY_LENGTH) {
            throw new IllegalArgumentException("maxPointsInLeafNode must be <= ArrayUtil.MAX_ARRAY_LENGTH (= " + ArrayUtil.MAX_ARRAY_LENGTH + "); got " + maxPointsInLeafNode);
        }
        if (maxMBSortInHeap < 0.0) {
            throw new IllegalArgumentException("maxMBSortInHeap must be >= 0.0 (got: " + maxMBSortInHeap + ")");
        }
        if (totalPointCount < 0L) {
            throw new IllegalArgumentException("totalPointCount must be >=0 (got: " + totalPointCount + ")");
        }
    }

    private void spillToOffline() throws IOException {
        this.offlinePointWriter = new OfflinePointWriter(this.tempDir, this.tempFileNamePrefix, this.packedBytesLength, this.longOrds, "spill", 0L, this.singleValuePerDoc);
        this.tempInput = this.offlinePointWriter.out;
        PointReader reader = this.heapPointWriter.getReader(0L, this.pointCount);
        int i = 0;
        while ((long)i < this.pointCount) {
            boolean hasNext = reader.next();
            assert (hasNext);
            this.offlinePointWriter.append(reader.packedValue(), i, this.heapPointWriter.docIDs[i]);
            ++i;
        }
        this.heapPointWriter = null;
    }

    public void add(byte[] packedValue, int docID) throws IOException {
        if (packedValue.length != this.packedBytesLength) {
            throw new IllegalArgumentException("packedValue should be length=" + this.packedBytesLength + " (got: " + packedValue.length + ")");
        }
        if (this.pointCount >= (long)this.maxPointsSortInHeap) {
            if (this.offlinePointWriter == null) {
                this.spillToOffline();
            }
            this.offlinePointWriter.append(packedValue, this.pointCount, docID);
        } else {
            this.heapPointWriter.append(packedValue, this.pointCount, docID);
        }
        if (this.pointCount == 0L) {
            System.arraycopy(packedValue, 0, this.minPackedValue, 0, this.packedBytesLength);
            System.arraycopy(packedValue, 0, this.maxPackedValue, 0, this.packedBytesLength);
        } else {
            for (int dim = 0; dim < this.numDims; ++dim) {
                int offset = dim * this.bytesPerDim;
                if (FutureArrays.compareUnsigned(packedValue, offset, offset + this.bytesPerDim, this.minPackedValue, offset, offset + this.bytesPerDim) < 0) {
                    System.arraycopy(packedValue, offset, this.minPackedValue, offset, this.bytesPerDim);
                }
                if (FutureArrays.compareUnsigned(packedValue, offset, offset + this.bytesPerDim, this.maxPackedValue, offset, offset + this.bytesPerDim) <= 0) continue;
                System.arraycopy(packedValue, offset, this.maxPackedValue, offset, this.bytesPerDim);
            }
        }
        ++this.pointCount;
        if (this.pointCount > this.totalPointCount) {
            throw new IllegalStateException("totalPointCount=" + this.totalPointCount + " was passed when we were created, but we just hit " + this.pointCount + " values");
        }
        this.docsSeen.set(docID);
    }

    public long getPointCount() {
        return this.pointCount;
    }

    public long writeField(IndexOutput out, String fieldName, MutablePointValues reader) throws IOException {
        if (this.numDims == 1) {
            return this.writeField1Dim(out, fieldName, reader);
        }
        return this.writeFieldNDims(out, fieldName, reader);
    }

    private long writeFieldNDims(IndexOutput out, String fieldName, MutablePointValues values) throws IOException {
        if (this.pointCount != 0L) {
            throw new IllegalStateException("cannot mix add and writeField");
        }
        if (this.heapPointWriter == null && this.tempInput == null) {
            throw new IllegalStateException("already finished");
        }
        this.heapPointWriter = null;
        long countPerLeaf = this.pointCount = values.size();
        long innerNodeCount = 1L;
        while (countPerLeaf > (long)this.maxPointsInLeafNode) {
            countPerLeaf = (countPerLeaf + 1L) / 2L;
            innerNodeCount *= 2L;
        }
        int numLeaves = Math.toIntExact(innerNodeCount);
        this.checkMaxLeafNodeCount(numLeaves);
        byte[] splitPackedValues = new byte[numLeaves * (this.bytesPerDim + 1)];
        long[] leafBlockFPs = new long[numLeaves];
        Arrays.fill(this.minPackedValue, (byte)-1);
        Arrays.fill(this.maxPackedValue, (byte)0);
        for (int i = 0; i < Math.toIntExact(this.pointCount); ++i) {
            values.getValue(i, this.scratchBytesRef1);
            for (int dim = 0; dim < this.numDims; ++dim) {
                int offset = dim * this.bytesPerDim;
                if (FutureArrays.compareUnsigned(this.scratchBytesRef1.bytes, this.scratchBytesRef1.offset + offset, this.scratchBytesRef1.offset + offset + this.bytesPerDim, this.minPackedValue, offset, offset + this.bytesPerDim) < 0) {
                    System.arraycopy(this.scratchBytesRef1.bytes, this.scratchBytesRef1.offset + offset, this.minPackedValue, offset, this.bytesPerDim);
                }
                if (FutureArrays.compareUnsigned(this.scratchBytesRef1.bytes, this.scratchBytesRef1.offset + offset, this.scratchBytesRef1.offset + offset + this.bytesPerDim, this.maxPackedValue, offset, offset + this.bytesPerDim) <= 0) continue;
                System.arraycopy(this.scratchBytesRef1.bytes, this.scratchBytesRef1.offset + offset, this.maxPackedValue, offset, this.bytesPerDim);
            }
            this.docsSeen.set(values.getDocID(i));
        }
        int[] parentSplits = new int[this.numDims];
        this.build(1, numLeaves, values, 0, Math.toIntExact(this.pointCount), out, this.minPackedValue, this.maxPackedValue, parentSplits, splitPackedValues, leafBlockFPs, new int[this.maxPointsInLeafNode]);
        assert (Arrays.equals(parentSplits, new int[this.numDims]));
        long indexFP = out.getFilePointer();
        this.writeIndex(out, Math.toIntExact(countPerLeaf), leafBlockFPs, splitPackedValues);
        return indexFP;
    }

    private long writeField1Dim(IndexOutput out, String fieldName, MutablePointValues reader) throws IOException {
        MutablePointsReaderUtils.sort(this.maxDoc, this.packedBytesLength, reader, 0, Math.toIntExact(reader.size()));
        OneDimensionBKDWriter oneDimWriter = new OneDimensionBKDWriter(out);
        reader.intersect(new PointValues.IntersectVisitor(){

            @Override
            public void visit(int docID, byte[] packedValue) throws IOException {
                oneDimWriter.add(packedValue, docID);
            }

            @Override
            public void visit(int docID) throws IOException {
                throw new IllegalStateException();
            }

            @Override
            public PointValues.Relation compare(byte[] minPackedValue, byte[] maxPackedValue) {
                return PointValues.Relation.CELL_CROSSES_QUERY;
            }
        });
        return oneDimWriter.finish();
    }

    public long merge(IndexOutput out, List<MergeState.DocMap> docMaps, List<BKDReader> readers) throws IOException {
        assert (docMaps == null || readers.size() == docMaps.size());
        BKDMergeQueue queue = new BKDMergeQueue(this.bytesPerDim, readers.size());
        for (int i = 0; i < readers.size(); ++i) {
            MergeState.DocMap docMap;
            BKDReader bkd = readers.get(i);
            MergeReader reader = new MergeReader(bkd, docMap = docMaps == null ? null : docMaps.get(i));
            if (!reader.next()) continue;
            queue.add(reader);
        }
        OneDimensionBKDWriter oneDimWriter = new OneDimensionBKDWriter(out);
        while (queue.size() != 0) {
            MergeReader reader = (MergeReader)queue.top();
            oneDimWriter.add(reader.state.scratchPackedValue1, reader.docID);
            if (reader.next()) {
                queue.updateTop();
                continue;
            }
            queue.pop();
        }
        return oneDimWriter.finish();
    }

    private void rotateToTree(int nodeID, int offset, int count, byte[] index, List<byte[]> leafBlockStartValues) {
        if (count == 1) {
            System.arraycopy(leafBlockStartValues.get(offset), 0, index, nodeID * (1 + this.bytesPerDim) + 1, this.bytesPerDim);
        } else {
            if (count > 1) {
                int countAtLevel = 1;
                int totalCount = 0;
                while (true) {
                    int countLeft;
                    if ((countLeft = count - totalCount) <= countAtLevel) {
                        int lastLeftCount = Math.min(countAtLevel / 2, countLeft);
                        assert (lastLeftCount >= 0);
                        int leftHalf = (totalCount - 1) / 2 + lastLeftCount;
                        int rootOffset = offset + leftHalf;
                        System.arraycopy(leafBlockStartValues.get(rootOffset), 0, index, nodeID * (1 + this.bytesPerDim) + 1, this.bytesPerDim);
                        this.rotateToTree(2 * nodeID, offset, leftHalf, index, leafBlockStartValues);
                        this.rotateToTree(2 * nodeID + 1, rootOffset + 1, count - leftHalf - 1, index, leafBlockStartValues);
                        return;
                    }
                    totalCount += countAtLevel;
                    countAtLevel *= 2;
                }
            }
            assert (count == 0);
        }
    }

    private void sortHeapPointWriter(HeapPointWriter writer, int dim) {
        int pointCount = Math.toIntExact(this.pointCount);
        new MSBRadixSorter(this.bytesPerDim + 4){

            @Override
            protected int byteAt(int i, int k) {
                assert (k >= 0);
                if (k < BKDWriter.this.bytesPerDim) {
                    int block = i / writer.valuesPerBlock;
                    int index = i % writer.valuesPerBlock;
                    return writer.blocks.get(block)[index * BKDWriter.this.packedBytesLength + dim * BKDWriter.this.bytesPerDim + k] & 0xFF;
                }
                int s = 3 - (k - BKDWriter.this.bytesPerDim);
                return writer.docIDs[i] >>> s * 8 & 0xFF;
            }

            @Override
            protected void swap(int i, int j) {
                int docID = writer.docIDs[i];
                writer.docIDs[i] = writer.docIDs[j];
                writer.docIDs[j] = docID;
                if (!BKDWriter.this.singleValuePerDoc) {
                    if (BKDWriter.this.longOrds) {
                        long ord = writer.ordsLong[i];
                        writer.ordsLong[i] = writer.ordsLong[j];
                        writer.ordsLong[j] = ord;
                    } else {
                        int ord = writer.ords[i];
                        writer.ords[i] = writer.ords[j];
                        writer.ords[j] = ord;
                    }
                }
                byte[] blockI = writer.blocks.get(i / writer.valuesPerBlock);
                int indexI = i % writer.valuesPerBlock * BKDWriter.this.packedBytesLength;
                byte[] blockJ = writer.blocks.get(j / writer.valuesPerBlock);
                int indexJ = j % writer.valuesPerBlock * BKDWriter.this.packedBytesLength;
                System.arraycopy(blockI, indexI, BKDWriter.this.scratch1, 0, BKDWriter.this.packedBytesLength);
                System.arraycopy(blockJ, indexJ, blockI, indexI, BKDWriter.this.packedBytesLength);
                System.arraycopy(BKDWriter.this.scratch1, 0, blockJ, indexJ, BKDWriter.this.packedBytesLength);
            }
        }.sort(0, pointCount);
    }

    private PointWriter sort(int dim) throws IOException {
        assert (dim >= 0 && dim < this.numDims);
        if (this.heapPointWriter != null) {
            HeapPointWriter sorted;
            assert (this.tempInput == null);
            if (dim == 0) {
                sorted = this.heapPointWriter;
            } else {
                sorted = new HeapPointWriter((int)this.pointCount, (int)this.pointCount, this.packedBytesLength, this.longOrds, this.singleValuePerDoc);
                sorted.copyFrom(this.heapPointWriter);
            }
            this.sortHeapPointWriter(sorted, dim);
            sorted.close();
            return sorted;
        }
        assert (this.tempInput != null);
        int offset = this.bytesPerDim * dim;
        BytesRefComparator cmp = dim == this.numDims - 1 ? new BytesRefComparator(this.bytesPerDim + 4){

            @Override
            protected int byteAt(BytesRef ref, int i) {
                return ref.bytes[ref.offset + offset + i] & 0xFF;
            }
        } : new BytesRefComparator(this.bytesPerDim + 4){

            @Override
            protected int byteAt(BytesRef ref, int i) {
                if (i < BKDWriter.this.bytesPerDim) {
                    return ref.bytes[ref.offset + offset + i] & 0xFF;
                }
                return ref.bytes[ref.offset + BKDWriter.this.packedBytesLength + i - BKDWriter.this.bytesPerDim] & 0xFF;
            }
        };
        OfflineSorter sorter = new OfflineSorter(this.tempDir, this.tempFileNamePrefix + "_bkd" + dim, cmp, this.offlineSorterBufferMB, this.offlineSorterMaxTempFiles, this.bytesPerDoc, null, 0){

            @Override
            protected OfflineSorter.ByteSequencesWriter getWriter(IndexOutput out, long count) {
                return new OfflineSorter.ByteSequencesWriter(out){

                    @Override
                    public void write(byte[] bytes, int off, int len) throws IOException {
                        assert (len == BKDWriter.this.bytesPerDoc) : "len=" + len + " bytesPerDoc=" + BKDWriter.access$1300(BKDWriter.this);
                        this.out.writeBytes(bytes, off, len);
                    }
                };
            }

            @Override
            protected OfflineSorter.ByteSequencesReader getReader(ChecksumIndexInput in, String name) throws IOException {
                return new OfflineSorter.ByteSequencesReader(in, name){
                    final BytesRef scratch;
                    {
                        this.scratch = new BytesRef(new byte[BKDWriter.this.bytesPerDoc]);
                    }

                    @Override
                    public BytesRef next() throws IOException {
                        if (this.in.getFilePointer() >= this.end) {
                            return null;
                        }
                        this.in.readBytes(this.scratch.bytes, 0, BKDWriter.this.bytesPerDoc);
                        return this.scratch;
                    }
                };
            }
        };
        String name = sorter.sort(this.tempInput.getName());
        return new OfflinePointWriter(this.tempDir, name, this.packedBytesLength, this.pointCount, this.longOrds, this.singleValuePerDoc);
    }

    private void checkMaxLeafNodeCount(int numLeaves) {
        if ((long)(1 + this.bytesPerDim) * (long)numLeaves > (long)ArrayUtil.MAX_ARRAY_LENGTH) {
            throw new IllegalStateException("too many nodes; increase maxPointsInLeafNode (currently " + this.maxPointsInLeafNode + ") and reindex");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long finish(IndexOutput out) throws IOException {
        if (this.heapPointWriter == null && this.tempInput == null) {
            throw new IllegalStateException("already finished");
        }
        if (this.offlinePointWriter != null) {
            this.offlinePointWriter.close();
        }
        if (this.pointCount == 0L) {
            throw new IllegalStateException("must index at least one point");
        }
        LongBitSet ordBitSet = this.numDims > 1 ? (this.singleValuePerDoc ? new LongBitSet(this.maxDoc) : new LongBitSet(this.pointCount)) : null;
        long countPerLeaf = this.pointCount;
        long innerNodeCount = 1L;
        while (countPerLeaf > (long)this.maxPointsInLeafNode) {
            countPerLeaf = (countPerLeaf + 1L) / 2L;
            innerNodeCount *= 2L;
        }
        int numLeaves = (int)innerNodeCount;
        this.checkMaxLeafNodeCount(numLeaves);
        byte[] splitPackedValues = new byte[Math.toIntExact(numLeaves * (1 + this.bytesPerDim))];
        long[] leafBlockFPs = new long[numLeaves];
        assert (this.pointCount / (long)numLeaves <= (long)this.maxPointsInLeafNode) : "pointCount=" + this.pointCount + " numLeaves=" + numLeaves + " maxPointsInLeafNode=" + this.maxPointsInLeafNode;
        PathSlice[] sortedPointWriters = new PathSlice[this.numDims];
        ArrayList<Closeable> toCloseHeroically = new ArrayList<Closeable>();
        boolean success = false;
        try {
            for (int dim = 0; dim < this.numDims; ++dim) {
                sortedPointWriters[dim] = new PathSlice(this.sort(dim), 0L, this.pointCount);
            }
            if (this.tempInput != null) {
                this.tempDir.deleteFile(this.tempInput.getName());
                this.tempInput = null;
            } else {
                assert (this.heapPointWriter != null);
                this.heapPointWriter = null;
            }
            int[] parentSplits = new int[this.numDims];
            this.build(1, numLeaves, sortedPointWriters, ordBitSet, out, this.minPackedValue, this.maxPackedValue, parentSplits, splitPackedValues, leafBlockFPs, toCloseHeroically);
            assert (Arrays.equals(parentSplits, new int[this.numDims]));
            for (PathSlice slice : sortedPointWriters) {
                slice.writer.destroy();
            }
            assert (this.tempDir.getCreatedFiles().isEmpty());
            success = true;
        }
        finally {
            if (!success) {
                IOUtils.deleteFilesIgnoringExceptions((Directory)this.tempDir, this.tempDir.getCreatedFiles());
                IOUtils.closeWhileHandlingException(toCloseHeroically);
            }
        }
        long indexFP = out.getFilePointer();
        this.writeIndex(out, Math.toIntExact(countPerLeaf), leafBlockFPs, splitPackedValues);
        return indexFP;
    }

    private byte[] packIndex(long[] leafBlockFPs, byte[] splitPackedValues) throws IOException {
        int numLeaves = leafBlockFPs.length;
        if (this.numDims == 1 && numLeaves > 1) {
            int levelCount = 2;
            while (true) {
                if (numLeaves >= levelCount && numLeaves <= 2 * levelCount) {
                    int lastLevel = 2 * (numLeaves - levelCount);
                    assert (lastLevel >= 0);
                    if (lastLevel == 0) break;
                    long[] newLeafBlockFPs = new long[numLeaves];
                    System.arraycopy(leafBlockFPs, lastLevel, newLeafBlockFPs, 0, leafBlockFPs.length - lastLevel);
                    System.arraycopy(leafBlockFPs, 0, newLeafBlockFPs, leafBlockFPs.length - lastLevel, lastLevel);
                    leafBlockFPs = newLeafBlockFPs;
                    break;
                }
                levelCount *= 2;
            }
        }
        RAMOutputStream writeBuffer = new RAMOutputStream();
        ArrayList<byte[]> blocks = new ArrayList<byte[]>();
        byte[] lastSplitValues = new byte[this.bytesPerDim * this.numDims];
        int totalSize = this.recursePackIndex(writeBuffer, leafBlockFPs, splitPackedValues, 0L, blocks, 1, lastSplitValues, new boolean[this.numDims], false);
        byte[] index = new byte[totalSize];
        int upto = 0;
        for (byte[] block : blocks) {
            System.arraycopy(block, 0, index, upto, block.length);
            upto += block.length;
        }
        assert (upto == totalSize);
        return index;
    }

    private int appendBlock(RAMOutputStream writeBuffer, List<byte[]> blocks) throws IOException {
        int pos = Math.toIntExact(writeBuffer.getFilePointer());
        byte[] bytes = new byte[pos];
        writeBuffer.writeTo(bytes, 0);
        writeBuffer.reset();
        blocks.add(bytes);
        return pos;
    }

    private int recursePackIndex(RAMOutputStream writeBuffer, long[] leafBlockFPs, byte[] splitPackedValues, long minBlockFP, List<byte[]> blocks, int nodeID, byte[] lastSplitValues, boolean[] negativeDeltas, boolean isLeft) throws IOException {
        int firstDiffByteDelta;
        int prefix;
        long leftBlockFP;
        if (nodeID >= leafBlockFPs.length) {
            int leafID = nodeID - leafBlockFPs.length;
            if (leafID < leafBlockFPs.length) {
                long delta = leafBlockFPs[leafID] - minBlockFP;
                if (isLeft) {
                    assert (delta == 0L);
                    return 0;
                }
                assert (nodeID == 1 || delta > 0L) : "nodeID=" + nodeID;
                writeBuffer.writeVLong(delta);
                return this.appendBlock(writeBuffer, blocks);
            }
            return 0;
        }
        if (!isLeft) {
            leftBlockFP = this.getLeftMostLeafBlockFP(leafBlockFPs, nodeID);
            long delta = leftBlockFP - minBlockFP;
            assert (nodeID == 1 || delta > 0L);
            writeBuffer.writeVLong(delta);
        } else {
            leftBlockFP = minBlockFP;
        }
        int address = nodeID * (1 + this.bytesPerDim);
        int splitDim = splitPackedValues[address++] & 0xFF;
        for (prefix = 0; prefix < this.bytesPerDim && splitPackedValues[address + prefix] == lastSplitValues[splitDim * this.bytesPerDim + prefix]; ++prefix) {
        }
        if (prefix < this.bytesPerDim) {
            firstDiffByteDelta = (splitPackedValues[address + prefix] & 0xFF) - (lastSplitValues[splitDim * this.bytesPerDim + prefix] & 0xFF);
            if (negativeDeltas[splitDim]) {
                firstDiffByteDelta = -firstDiffByteDelta;
            }
            assert (firstDiffByteDelta > 0);
        } else {
            firstDiffByteDelta = 0;
        }
        int code = (firstDiffByteDelta * (1 + this.bytesPerDim) + prefix) * this.numDims + splitDim;
        writeBuffer.writeVInt(code);
        int suffix = this.bytesPerDim - prefix;
        byte[] savSplitValue = new byte[suffix];
        if (suffix > 1) {
            writeBuffer.writeBytes(splitPackedValues, address + prefix + 1, suffix - 1);
        }
        byte[] cmp = (byte[])lastSplitValues.clone();
        System.arraycopy(lastSplitValues, splitDim * this.bytesPerDim + prefix, savSplitValue, 0, suffix);
        System.arraycopy(splitPackedValues, address + prefix, lastSplitValues, splitDim * this.bytesPerDim + prefix, suffix);
        int numBytes = this.appendBlock(writeBuffer, blocks);
        int idxSav = blocks.size();
        blocks.add(null);
        boolean savNegativeDelta = negativeDeltas[splitDim];
        negativeDeltas[splitDim] = true;
        int leftNumBytes = this.recursePackIndex(writeBuffer, leafBlockFPs, splitPackedValues, leftBlockFP, blocks, 2 * nodeID, lastSplitValues, negativeDeltas, true);
        if (nodeID * 2 < leafBlockFPs.length) {
            writeBuffer.writeVInt(leftNumBytes);
        } else assert (leftNumBytes == 0) : "leftNumBytes=" + leftNumBytes;
        int numBytes2 = Math.toIntExact(writeBuffer.getFilePointer());
        byte[] bytes2 = new byte[numBytes2];
        writeBuffer.writeTo(bytes2, 0);
        writeBuffer.reset();
        blocks.set(idxSav, bytes2);
        negativeDeltas[splitDim] = false;
        int rightNumBytes = this.recursePackIndex(writeBuffer, leafBlockFPs, splitPackedValues, leftBlockFP, blocks, 2 * nodeID + 1, lastSplitValues, negativeDeltas, false);
        negativeDeltas[splitDim] = savNegativeDelta;
        System.arraycopy(savSplitValue, 0, lastSplitValues, splitDim * this.bytesPerDim + prefix, suffix);
        assert (Arrays.equals(lastSplitValues, cmp));
        return numBytes + numBytes2 + leftNumBytes + rightNumBytes;
    }

    private long getLeftMostLeafBlockFP(long[] leafBlockFPs, int nodeID) {
        int nodeIDIn = nodeID;
        while (nodeID < leafBlockFPs.length) {
            nodeID *= 2;
        }
        int leafID = nodeID - leafBlockFPs.length;
        long result = leafBlockFPs[leafID];
        if (result < 0L) {
            throw new AssertionError((Object)(result + " for leaf " + leafID));
        }
        return result;
    }

    private void writeIndex(IndexOutput out, int countPerLeaf, long[] leafBlockFPs, byte[] splitPackedValues) throws IOException {
        byte[] packedIndex = this.packIndex(leafBlockFPs, splitPackedValues);
        this.writeIndex(out, countPerLeaf, leafBlockFPs.length, packedIndex);
    }

    private void writeIndex(IndexOutput out, int countPerLeaf, int numLeaves, byte[] packedIndex) throws IOException {
        CodecUtil.writeHeader(out, CODEC_NAME, 5);
        out.writeVInt(this.numDims);
        out.writeVInt(countPerLeaf);
        out.writeVInt(this.bytesPerDim);
        assert (numLeaves > 0);
        out.writeVInt(numLeaves);
        out.writeBytes(this.minPackedValue, 0, this.packedBytesLength);
        out.writeBytes(this.maxPackedValue, 0, this.packedBytesLength);
        out.writeVLong(this.pointCount);
        out.writeVInt(this.docsSeen.cardinality());
        out.writeVInt(packedIndex.length);
        out.writeBytes(packedIndex, 0, packedIndex.length);
    }

    private void writeLeafBlockDocs(DataOutput out, int[] docIDs, int start, int count) throws IOException {
        assert (count > 0) : "maxPointsInLeafNode=" + this.maxPointsInLeafNode;
        out.writeVInt(count);
        DocIdsWriter.writeDocIds(docIDs, start, count, out);
    }

    private void writeLeafBlockPackedValues(DataOutput out, int[] commonPrefixLengths, int count, int sortedDim, IntFunction<BytesRef> packedValues) throws IOException {
        int prefixLenSum = Arrays.stream(commonPrefixLengths).sum();
        if (prefixLenSum == this.packedBytesLength) {
            out.writeByte((byte)-1);
        } else {
            if (this.numDims != 1) {
                this.writeActualBounds(out, commonPrefixLengths, count, packedValues);
            }
            assert (commonPrefixLengths[sortedDim] < this.bytesPerDim);
            out.writeByte((byte)sortedDim);
            int compressedByteOffset = sortedDim * this.bytesPerDim + commonPrefixLengths[sortedDim];
            int n = sortedDim;
            commonPrefixLengths[n] = commonPrefixLengths[n] + 1;
            int i = 0;
            while (i < count) {
                int runLen = BKDWriter.runLen(packedValues, i, Math.min(i + 255, count), compressedByteOffset);
                assert (runLen <= 255);
                BytesRef first = packedValues.apply(i);
                byte prefixByte = first.bytes[first.offset + compressedByteOffset];
                out.writeByte(prefixByte);
                out.writeByte((byte)runLen);
                this.writeLeafBlockPackedValuesRange(out, commonPrefixLengths, i, i + runLen, packedValues);
                assert ((i += runLen) <= count);
            }
        }
    }

    private void writeActualBounds(DataOutput out, int[] commonPrefixLengths, int count, IntFunction<BytesRef> packedValues) throws IOException {
        for (int dim = 0; dim < this.numDims; ++dim) {
            int commonPrefixLength = commonPrefixLengths[dim];
            int suffixLength = this.bytesPerDim - commonPrefixLength;
            if (suffixLength <= 0) continue;
            BytesRef[] minMax = BKDWriter.computeMinMax(count, packedValues, dim * this.bytesPerDim + commonPrefixLength, suffixLength);
            BytesRef min = minMax[0];
            BytesRef max = minMax[1];
            out.writeBytes(min.bytes, min.offset, min.length);
            out.writeBytes(max.bytes, max.offset, max.length);
        }
    }

    private static BytesRef[] computeMinMax(int count, IntFunction<BytesRef> packedValues, int offset, int length) {
        assert (length > 0);
        BytesRefBuilder min = new BytesRefBuilder();
        BytesRefBuilder max = new BytesRefBuilder();
        BytesRef first = packedValues.apply(0);
        min.copyBytes(first.bytes, first.offset + offset, length);
        max.copyBytes(first.bytes, first.offset + offset, length);
        for (int i = 1; i < count; ++i) {
            BytesRef candidate = packedValues.apply(i);
            if (FutureArrays.compareUnsigned(min.bytes(), 0, length, candidate.bytes, candidate.offset + offset, candidate.offset + offset + length) > 0) {
                min.copyBytes(candidate.bytes, candidate.offset + offset, length);
                continue;
            }
            if (FutureArrays.compareUnsigned(max.bytes(), 0, length, candidate.bytes, candidate.offset + offset, candidate.offset + offset + length) >= 0) continue;
            max.copyBytes(candidate.bytes, candidate.offset + offset, length);
        }
        return new BytesRef[]{min.get(), max.get()};
    }

    private void writeLeafBlockPackedValuesRange(DataOutput out, int[] commonPrefixLengths, int start, int end, IntFunction<BytesRef> packedValues) throws IOException {
        for (int i = start; i < end; ++i) {
            BytesRef ref = packedValues.apply(i);
            assert (ref.length == this.packedBytesLength);
            for (int dim = 0; dim < this.numDims; ++dim) {
                int prefix = commonPrefixLengths[dim];
                out.writeBytes(ref.bytes, ref.offset + dim * this.bytesPerDim + prefix, this.bytesPerDim - prefix);
            }
        }
    }

    private static int runLen(IntFunction<BytesRef> packedValues, int start, int end, int byteOffset) {
        BytesRef first = packedValues.apply(start);
        byte b = first.bytes[first.offset + byteOffset];
        for (int i = start + 1; i < end; ++i) {
            BytesRef ref = packedValues.apply(i);
            byte b2 = ref.bytes[ref.offset + byteOffset];
            assert (Byte.toUnsignedInt(b2) >= Byte.toUnsignedInt(b));
            if (b == b2) continue;
            return i - start;
        }
        return end - start;
    }

    private void writeCommonPrefixes(DataOutput out, int[] commonPrefixes, byte[] packedValue) throws IOException {
        for (int dim = 0; dim < this.numDims; ++dim) {
            out.writeVInt(commonPrefixes[dim]);
            out.writeBytes(packedValue, dim * this.bytesPerDim, commonPrefixes[dim]);
        }
    }

    @Override
    public void close() throws IOException {
        if (this.tempInput != null) {
            try {
                this.tempInput.close();
            }
            finally {
                this.tempDir.deleteFile(this.tempInput.getName());
                this.tempInput = null;
            }
        }
    }

    private Error verifyChecksum(Throwable priorException, PointWriter writer) throws IOException {
        assert (priorException != null);
        if (writer instanceof OfflinePointWriter) {
            String tempFileName = ((OfflinePointWriter)writer).name;
            try (ChecksumIndexInput in = this.tempDir.openChecksumInput(tempFileName, IOContext.READONCE);){
                CodecUtil.checkFooter(in, priorException);
            }
        }
        throw IOUtils.rethrowAlways(priorException);
    }

    private byte[] markRightTree(long rightCount, int splitDim, PathSlice source, LongBitSet ordBitSet) throws IOException {
        try (PointReader reader = source.writer.getReader(source.start + source.count - rightCount, rightCount);){
            boolean result = reader.next();
            assert (result) : "rightCount=" + rightCount + " source.count=" + source.count + " source.writer=" + source.writer;
            System.arraycopy(reader.packedValue(), splitDim * this.bytesPerDim, this.scratch1, 0, this.bytesPerDim);
            if (this.numDims > 1) {
                assert (!ordBitSet.get(reader.ord()));
                ordBitSet.set(reader.ord());
                reader.markOrds(rightCount - 1L, ordBitSet);
            }
        }
        catch (Throwable t) {
            throw this.verifyChecksum(t, source.writer);
        }
        return this.scratch1;
    }

    private boolean valueInBounds(BytesRef packedValue, byte[] minPackedValue, byte[] maxPackedValue) {
        for (int dim = 0; dim < this.numDims; ++dim) {
            int offset = this.bytesPerDim * dim;
            if (FutureArrays.compareUnsigned(packedValue.bytes, packedValue.offset + offset, packedValue.offset + offset + this.bytesPerDim, minPackedValue, offset, offset + this.bytesPerDim) < 0) {
                return false;
            }
            if (FutureArrays.compareUnsigned(packedValue.bytes, packedValue.offset + offset, packedValue.offset + offset + this.bytesPerDim, maxPackedValue, offset, offset + this.bytesPerDim) <= 0) continue;
            return false;
        }
        return true;
    }

    protected int split(byte[] minPackedValue, byte[] maxPackedValue, int[] parentSplits) {
        int maxNumSplits = 0;
        for (int numSplits : parentSplits) {
            maxNumSplits = Math.max(maxNumSplits, numSplits);
        }
        for (int dim = 0; dim < this.numDims; ++dim) {
            int offset = dim * this.bytesPerDim;
            if (parentSplits[dim] >= maxNumSplits / 2 || FutureArrays.compareUnsigned(minPackedValue, offset, offset + this.bytesPerDim, maxPackedValue, offset, offset + this.bytesPerDim) == 0) continue;
            return dim;
        }
        int splitDim = -1;
        for (int dim = 0; dim < this.numDims; ++dim) {
            NumericUtils.subtract(this.bytesPerDim, dim, maxPackedValue, minPackedValue, this.scratchDiff);
            if (splitDim != -1 && FutureArrays.compareUnsigned(this.scratchDiff, 0, this.bytesPerDim, this.scratch1, 0, this.bytesPerDim) <= 0) continue;
            System.arraycopy(this.scratchDiff, 0, this.scratch1, 0, this.bytesPerDim);
            splitDim = dim;
        }
        return splitDim;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private PathSlice switchToHeap(PathSlice source, List<Closeable> toCloseHeroically) throws IOException {
        int count = Math.toIntExact(source.count);
        PointReader reader = source.writer.getSharedReader(source.start, source.count, toCloseHeroically);
        try (HeapPointWriter writer = new HeapPointWriter(count, count, this.packedBytesLength, this.longOrds, this.singleValuePerDoc);){
            for (int i = 0; i < count; ++i) {
                boolean hasNext = reader.next();
                assert (hasNext);
                writer.append(reader.packedValue(), reader.ord(), reader.docID());
            }
            PathSlice pathSlice = new PathSlice(writer, 0L, count);
            return pathSlice;
        }
        catch (Throwable t) {
            throw this.verifyChecksum(t, source.writer);
        }
    }

    private void build(int nodeID, int leafNodeOffset, MutablePointValues reader, int from, int to, IndexOutput out, byte[] minPackedValue, byte[] maxPackedValue, int[] parentSplits, byte[] splitPackedValues, long[] leafBlockFPs, int[] spareDocIds) throws IOException {
        if (nodeID >= leafNodeOffset) {
            int dim;
            int count = to - from;
            assert (count <= this.maxPointsInLeafNode);
            Arrays.fill(this.commonPrefixLengths, this.bytesPerDim);
            reader.getValue(from, this.scratchBytesRef1);
            for (int i = from + 1; i < to; ++i) {
                reader.getValue(i, this.scratchBytesRef2);
                block1: for (dim = 0; dim < this.numDims; ++dim) {
                    int offset = dim * this.bytesPerDim;
                    for (int j = 0; j < this.commonPrefixLengths[dim]; ++j) {
                        if (this.scratchBytesRef1.bytes[this.scratchBytesRef1.offset + offset + j] == this.scratchBytesRef2.bytes[this.scratchBytesRef2.offset + offset + j]) continue;
                        this.commonPrefixLengths[dim] = j;
                        continue block1;
                    }
                }
            }
            FixedBitSet[] usedBytes = new FixedBitSet[this.numDims];
            for (dim = 0; dim < this.numDims; ++dim) {
                if (this.commonPrefixLengths[dim] >= this.bytesPerDim) continue;
                usedBytes[dim] = new FixedBitSet(256);
            }
            for (int i = from + 1; i < to; ++i) {
                for (int dim2 = 0; dim2 < this.numDims; ++dim2) {
                    if (usedBytes[dim2] == null) continue;
                    byte b = reader.getByteAt(i, dim2 * this.bytesPerDim + this.commonPrefixLengths[dim2]);
                    usedBytes[dim2].set(Byte.toUnsignedInt(b));
                }
            }
            int sortedDim = 0;
            int sortedDimCardinality = Integer.MAX_VALUE;
            for (int dim3 = 0; dim3 < this.numDims; ++dim3) {
                int cardinality;
                if (usedBytes[dim3] == null || (cardinality = usedBytes[dim3].cardinality()) >= sortedDimCardinality) continue;
                sortedDim = dim3;
                sortedDimCardinality = cardinality;
            }
            MutablePointsReaderUtils.sortByDim(sortedDim, this.bytesPerDim, this.commonPrefixLengths, reader, from, to, this.scratchBytesRef1, this.scratchBytesRef2);
            leafBlockFPs[nodeID - leafNodeOffset] = out.getFilePointer();
            assert (this.scratchOut.getPosition() == 0);
            int[] docIDs = spareDocIds;
            for (int i = from; i < to; ++i) {
                docIDs[i - from] = reader.getDocID(i);
            }
            this.writeLeafBlockDocs(this.scratchOut, docIDs, 0, count);
            reader.getValue(from, this.scratchBytesRef1);
            System.arraycopy(this.scratchBytesRef1.bytes, this.scratchBytesRef1.offset, this.scratch1, 0, this.packedBytesLength);
            this.writeCommonPrefixes(this.scratchOut, this.commonPrefixLengths, this.scratch1);
            IntFunction<BytesRef> packedValues = new IntFunction<BytesRef>(){

                @Override
                public BytesRef apply(int i) {
                    reader.getValue(from + i, BKDWriter.this.scratchBytesRef1);
                    return BKDWriter.this.scratchBytesRef1;
                }
            };
            assert (this.valuesInOrderAndBounds(count, sortedDim, minPackedValue, maxPackedValue, packedValues, docIDs, 0));
            this.writeLeafBlockPackedValues(this.scratchOut, this.commonPrefixLengths, count, sortedDim, packedValues);
            out.writeBytes(this.scratchOut.getBytes(), 0, this.scratchOut.getPosition());
            this.scratchOut.reset();
        } else {
            int splitDim = this.split(minPackedValue, maxPackedValue, parentSplits);
            int mid = from + to + 1 >>> 1;
            int commonPrefixLen = this.bytesPerDim;
            for (int i = 0; i < this.bytesPerDim; ++i) {
                if (minPackedValue[splitDim * this.bytesPerDim + i] == maxPackedValue[splitDim * this.bytesPerDim + i]) continue;
                commonPrefixLen = i;
                break;
            }
            MutablePointsReaderUtils.partition(this.maxDoc, splitDim, this.bytesPerDim, commonPrefixLen, reader, from, to, mid, this.scratchBytesRef1, this.scratchBytesRef2);
            int address = nodeID * (1 + this.bytesPerDim);
            splitPackedValues[address] = (byte)splitDim;
            reader.getValue(mid, this.scratchBytesRef1);
            System.arraycopy(this.scratchBytesRef1.bytes, this.scratchBytesRef1.offset + splitDim * this.bytesPerDim, splitPackedValues, address + 1, this.bytesPerDim);
            byte[] minSplitPackedValue = ArrayUtil.copyOfSubArray(minPackedValue, 0, this.packedBytesLength);
            byte[] maxSplitPackedValue = ArrayUtil.copyOfSubArray(maxPackedValue, 0, this.packedBytesLength);
            System.arraycopy(this.scratchBytesRef1.bytes, this.scratchBytesRef1.offset + splitDim * this.bytesPerDim, minSplitPackedValue, splitDim * this.bytesPerDim, this.bytesPerDim);
            System.arraycopy(this.scratchBytesRef1.bytes, this.scratchBytesRef1.offset + splitDim * this.bytesPerDim, maxSplitPackedValue, splitDim * this.bytesPerDim, this.bytesPerDim);
            int n = splitDim;
            parentSplits[n] = parentSplits[n] + 1;
            this.build(nodeID * 2, leafNodeOffset, reader, from, mid, out, minPackedValue, maxSplitPackedValue, parentSplits, splitPackedValues, leafBlockFPs, spareDocIds);
            this.build(nodeID * 2 + 1, leafNodeOffset, reader, mid, to, out, minSplitPackedValue, maxPackedValue, parentSplits, splitPackedValues, leafBlockFPs, spareDocIds);
            int n2 = splitDim;
            parentSplits[n2] = parentSplits[n2] - 1;
        }
    }

    private void build(int nodeID, int leafNodeOffset, PathSlice[] slices, LongBitSet ordBitSet, IndexOutput out, byte[] minPackedValue, byte[] maxPackedValue, int[] parentSplits, byte[] splitPackedValues, long[] leafBlockFPs, List<Closeable> toCloseHeroically) throws IOException {
        for (PathSlice slice : slices) {
            assert (slice.count == slices[0].count);
        }
        if (this.numDims == 1 && slices[0].writer instanceof OfflinePointWriter && slices[0].count <= (long)this.maxPointsSortInHeap) {
            slices[0] = this.switchToHeap(slices[0], toCloseHeroically);
        }
        if (nodeID >= leafNodeOffset) {
            int sortedDim = 0;
            int sortedDimCardinality = Integer.MAX_VALUE;
            for (int dim = 0; dim < this.numDims; ++dim) {
                int prefix;
                if (!(slices[dim].writer instanceof HeapPointWriter)) {
                    slices[dim] = this.switchToHeap(slices[dim], toCloseHeroically);
                }
                PathSlice source = slices[dim];
                HeapPointWriter heapSource = (HeapPointWriter)source.writer;
                heapSource.readPackedValue(Math.toIntExact(source.start), this.scratch1);
                heapSource.readPackedValue(Math.toIntExact(source.start + source.count - 1L), this.scratch2);
                int offset = dim * this.bytesPerDim;
                this.commonPrefixLengths[dim] = this.bytesPerDim;
                for (int j = 0; j < this.bytesPerDim; ++j) {
                    if (this.scratch1[offset + j] == this.scratch2[offset + j]) continue;
                    this.commonPrefixLengths[dim] = j;
                    break;
                }
                if ((prefix = this.commonPrefixLengths[dim]) >= this.bytesPerDim) continue;
                int cardinality = 1;
                byte previous = this.scratch1[offset + prefix];
                for (long i = 1L; i < source.count; ++i) {
                    heapSource.readPackedValue(Math.toIntExact(source.start + i), this.scratch2);
                    byte b = this.scratch2[offset + prefix];
                    assert (Byte.toUnsignedInt(previous) <= Byte.toUnsignedInt(b));
                    if (b == previous) continue;
                    ++cardinality;
                    previous = b;
                }
                assert (cardinality <= 256);
                if (cardinality >= sortedDimCardinality) continue;
                sortedDim = dim;
                sortedDimCardinality = cardinality;
            }
            PathSlice source = slices[sortedDim];
            HeapPointWriter heapSource = (HeapPointWriter)source.writer;
            leafBlockFPs[nodeID - leafNodeOffset] = out.getFilePointer();
            int count = Math.toIntExact(source.count);
            assert (count > 0) : "nodeID=" + nodeID + " leafNodeOffset=" + leafNodeOffset;
            this.writeLeafBlockDocs(out, heapSource.docIDs, Math.toIntExact(source.start), count);
            this.writeCommonPrefixes(out, this.commonPrefixLengths, this.scratch1);
            IntFunction<BytesRef> packedValues = new IntFunction<BytesRef>(){
                final BytesRef scratch = new BytesRef();
                {
                    this.scratch.length = BKDWriter.this.packedBytesLength;
                }

                @Override
                public BytesRef apply(int i) {
                    heapSource.getPackedValueSlice(Math.toIntExact(source.start + (long)i), this.scratch);
                    return this.scratch;
                }
            };
            assert (this.valuesInOrderAndBounds(count, sortedDim, minPackedValue, maxPackedValue, packedValues, heapSource.docIDs, Math.toIntExact(source.start)));
            this.writeLeafBlockPackedValues(out, this.commonPrefixLengths, count, sortedDim, packedValues);
        } else {
            int dim;
            int splitDim = this.numDims > 1 ? this.split(minPackedValue, maxPackedValue, parentSplits) : 0;
            PathSlice source = slices[splitDim];
            assert (nodeID < splitPackedValues.length) : "nodeID=" + nodeID + " splitValues.length=" + splitPackedValues.length;
            long rightCount = source.count / 2L;
            long leftCount = source.count - rightCount;
            byte[] splitValue = this.markRightTree(rightCount, splitDim, source, ordBitSet);
            int address = nodeID * (1 + this.bytesPerDim);
            splitPackedValues[address] = (byte)splitDim;
            System.arraycopy(splitValue, 0, splitPackedValues, address + 1, this.bytesPerDim);
            PathSlice[] leftSlices = new PathSlice[this.numDims];
            PathSlice[] rightSlices = new PathSlice[this.numDims];
            byte[] minSplitPackedValue = new byte[this.packedBytesLength];
            System.arraycopy(minPackedValue, 0, minSplitPackedValue, 0, this.packedBytesLength);
            byte[] maxSplitPackedValue = new byte[this.packedBytesLength];
            System.arraycopy(maxPackedValue, 0, maxSplitPackedValue, 0, this.packedBytesLength);
            int dimToClear = this.numDims - 1 == splitDim ? this.numDims - 2 : this.numDims - 1;
            for (dim = 0; dim < this.numDims; ++dim) {
                if (dim == splitDim) {
                    leftSlices[dim] = new PathSlice(source.writer, source.start, leftCount);
                    rightSlices[dim] = new PathSlice(source.writer, source.start + leftCount, rightCount);
                    System.arraycopy(splitValue, 0, minSplitPackedValue, dim * this.bytesPerDim, this.bytesPerDim);
                    System.arraycopy(splitValue, 0, maxSplitPackedValue, dim * this.bytesPerDim, this.bytesPerDim);
                    continue;
                }
                PointReader reader = slices[dim].writer.getSharedReader(slices[dim].start, slices[dim].count, toCloseHeroically);
                try (PointWriter leftPointWriter = this.getPointWriter(leftCount, "left" + dim);
                     PointWriter rightPointWriter = this.getPointWriter(source.count - leftCount, "right" + dim);){
                    long nextRightCount = reader.split(source.count, ordBitSet, leftPointWriter, rightPointWriter, dim == dimToClear);
                    if (rightCount != nextRightCount) {
                        throw new IllegalStateException("wrong number of points in split: expected=" + rightCount + " but actual=" + nextRightCount);
                    }
                    leftSlices[dim] = new PathSlice(leftPointWriter, 0L, leftCount);
                    rightSlices[dim] = new PathSlice(rightPointWriter, 0L, rightCount);
                    continue;
                }
                catch (Throwable t) {
                    throw this.verifyChecksum(t, slices[dim].writer);
                }
            }
            int n = splitDim;
            parentSplits[n] = parentSplits[n] + 1;
            this.build(2 * nodeID, leafNodeOffset, leftSlices, ordBitSet, out, minPackedValue, maxSplitPackedValue, parentSplits, splitPackedValues, leafBlockFPs, toCloseHeroically);
            for (dim = 0; dim < this.numDims; ++dim) {
                if (dim == splitDim) continue;
                leftSlices[dim].writer.destroy();
            }
            this.build(2 * nodeID + 1, leafNodeOffset, rightSlices, ordBitSet, out, minSplitPackedValue, maxPackedValue, parentSplits, splitPackedValues, leafBlockFPs, toCloseHeroically);
            for (dim = 0; dim < this.numDims; ++dim) {
                if (dim == splitDim) continue;
                rightSlices[dim].writer.destroy();
            }
            int n2 = splitDim;
            parentSplits[n2] = parentSplits[n2] - 1;
        }
    }

    private boolean valuesInOrderAndBounds(int count, int sortedDim, byte[] minPackedValue, byte[] maxPackedValue, IntFunction<BytesRef> values, int[] docs, int docsOffset) throws IOException {
        byte[] lastPackedValue = new byte[this.packedBytesLength];
        int lastDoc = -1;
        for (int i = 0; i < count; ++i) {
            BytesRef packedValue = values.apply(i);
            assert (packedValue.length == this.packedBytesLength);
            assert (this.valueInOrder(i, sortedDim, lastPackedValue, packedValue.bytes, packedValue.offset, docs[docsOffset + i], lastDoc));
            lastDoc = docs[docsOffset + i];
            assert (this.valueInBounds(packedValue, minPackedValue, maxPackedValue));
        }
        return true;
    }

    private boolean valueInOrder(long ord, int sortedDim, byte[] lastPackedValue, byte[] packedValue, int packedValueOffset, int doc, int lastDoc) {
        int dimOffset = sortedDim * this.bytesPerDim;
        if (ord > 0L) {
            int cmp = FutureArrays.compareUnsigned(lastPackedValue, dimOffset, dimOffset + this.bytesPerDim, packedValue, packedValueOffset + dimOffset, packedValueOffset + dimOffset + this.bytesPerDim);
            if (cmp > 0) {
                throw new AssertionError((Object)("values out of order: last value=" + new BytesRef(lastPackedValue) + " current value=" + new BytesRef(packedValue, packedValueOffset, this.packedBytesLength) + " ord=" + ord));
            }
            if (cmp == 0 && doc < lastDoc) {
                throw new AssertionError((Object)("docs out of order: last doc=" + lastDoc + " current doc=" + doc + " ord=" + ord));
            }
        }
        System.arraycopy(packedValue, packedValueOffset, lastPackedValue, 0, this.packedBytesLength);
        return true;
    }

    PointWriter getPointWriter(long count, String desc) throws IOException {
        if (count <= (long)this.maxPointsSortInHeap) {
            int size = Math.toIntExact(count);
            return new HeapPointWriter(size, size, this.packedBytesLength, this.longOrds, this.singleValuePerDoc);
        }
        return new OfflinePointWriter(this.tempDir, this.tempFileNamePrefix, this.packedBytesLength, this.longOrds, desc, count, this.singleValuePerDoc);
    }

    private static final class PathSlice {
        final PointWriter writer;
        final long start;
        final long count;

        public PathSlice(PointWriter writer, long start, long count) {
            this.writer = writer;
            this.start = start;
            this.count = count;
        }

        public String toString() {
            return "PathSlice(start=" + this.start + " count=" + this.count + " writer=" + this.writer + ")";
        }
    }

    private class OneDimensionBKDWriter {
        final IndexOutput out;
        final List<Long> leafBlockFPs = new ArrayList<Long>();
        final List<byte[]> leafBlockStartValues = new ArrayList<byte[]>();
        final byte[] leafValues;
        final int[] leafDocs;
        private long valueCount;
        private int leafCount;
        final byte[] lastPackedValue;
        private int lastDocID;

        OneDimensionBKDWriter(IndexOutput out) {
            this.leafValues = new byte[BKDWriter.this.maxPointsInLeafNode * BKDWriter.this.packedBytesLength];
            this.leafDocs = new int[BKDWriter.this.maxPointsInLeafNode];
            if (BKDWriter.this.numDims != 1) {
                throw new UnsupportedOperationException("numDims must be 1 but got " + BKDWriter.this.numDims);
            }
            if (BKDWriter.this.pointCount != 0L) {
                throw new IllegalStateException("cannot mix add and merge");
            }
            if (BKDWriter.this.heapPointWriter == null && BKDWriter.this.tempInput == null) {
                throw new IllegalStateException("already finished");
            }
            BKDWriter.this.heapPointWriter = null;
            this.out = out;
            this.lastPackedValue = new byte[BKDWriter.this.packedBytesLength];
        }

        void add(byte[] packedValue, int docID) throws IOException {
            assert (BKDWriter.this.valueInOrder(this.valueCount + (long)this.leafCount, 0, this.lastPackedValue, packedValue, 0, docID, this.lastDocID));
            System.arraycopy(packedValue, 0, this.leafValues, this.leafCount * BKDWriter.this.packedBytesLength, BKDWriter.this.packedBytesLength);
            this.leafDocs[this.leafCount] = docID;
            BKDWriter.this.docsSeen.set(docID);
            ++this.leafCount;
            if (this.valueCount > BKDWriter.this.totalPointCount) {
                throw new IllegalStateException("totalPointCount=" + BKDWriter.this.totalPointCount + " was passed when we were created, but we just hit " + BKDWriter.this.pointCount + " values");
            }
            if (this.leafCount == BKDWriter.this.maxPointsInLeafNode) {
                this.writeLeafBlock();
                this.leafCount = 0;
            }
            assert ((this.lastDocID = docID) >= 0);
        }

        public long finish() throws IOException {
            if (this.leafCount > 0) {
                this.writeLeafBlock();
                this.leafCount = 0;
            }
            if (this.valueCount == 0L) {
                return -1L;
            }
            BKDWriter.this.pointCount = this.valueCount;
            long indexFP = this.out.getFilePointer();
            int numInnerNodes = this.leafBlockStartValues.size();
            byte[] index = new byte[(1 + numInnerNodes) * (1 + BKDWriter.this.bytesPerDim)];
            BKDWriter.this.rotateToTree(1, 0, numInnerNodes, index, this.leafBlockStartValues);
            long[] arr = new long[this.leafBlockFPs.size()];
            for (int i = 0; i < this.leafBlockFPs.size(); ++i) {
                arr[i] = this.leafBlockFPs.get(i);
            }
            BKDWriter.this.writeIndex(this.out, BKDWriter.this.maxPointsInLeafNode, arr, index);
            return indexFP;
        }

        private void writeLeafBlock() throws IOException {
            assert (this.leafCount != 0);
            if (this.valueCount == 0L) {
                System.arraycopy(this.leafValues, 0, BKDWriter.this.minPackedValue, 0, BKDWriter.this.packedBytesLength);
            }
            System.arraycopy(this.leafValues, (this.leafCount - 1) * BKDWriter.this.packedBytesLength, BKDWriter.this.maxPackedValue, 0, BKDWriter.this.packedBytesLength);
            this.valueCount += (long)this.leafCount;
            if (this.leafBlockFPs.size() > 0) {
                this.leafBlockStartValues.add(ArrayUtil.copyOfSubArray(this.leafValues, 0, BKDWriter.this.packedBytesLength));
            }
            this.leafBlockFPs.add(this.out.getFilePointer());
            BKDWriter.this.checkMaxLeafNodeCount(this.leafBlockFPs.size());
            int prefix = BKDWriter.this.bytesPerDim;
            int offset = (this.leafCount - 1) * BKDWriter.this.packedBytesLength;
            for (int j = 0; j < BKDWriter.this.bytesPerDim; ++j) {
                if (this.leafValues[j] == this.leafValues[offset + j]) continue;
                prefix = j;
                break;
            }
            BKDWriter.this.commonPrefixLengths[0] = prefix;
            assert (BKDWriter.this.scratchOut.getPosition() == 0);
            BKDWriter.this.writeLeafBlockDocs(BKDWriter.this.scratchOut, this.leafDocs, 0, this.leafCount);
            BKDWriter.this.writeCommonPrefixes(BKDWriter.this.scratchOut, BKDWriter.this.commonPrefixLengths, this.leafValues);
            BKDWriter.this.scratchBytesRef1.length = BKDWriter.this.packedBytesLength;
            BKDWriter.this.scratchBytesRef1.bytes = this.leafValues;
            IntFunction<BytesRef> packedValues = new IntFunction<BytesRef>(){

                @Override
                public BytesRef apply(int i) {
                    BKDWriter.this.scratchBytesRef1.offset = BKDWriter.this.packedBytesLength * i;
                    return BKDWriter.this.scratchBytesRef1;
                }
            };
            assert (BKDWriter.this.valuesInOrderAndBounds(this.leafCount, 0, ArrayUtil.copyOfSubArray(this.leafValues, 0, BKDWriter.this.packedBytesLength), ArrayUtil.copyOfSubArray(this.leafValues, (this.leafCount - 1) * BKDWriter.this.packedBytesLength, this.leafCount * BKDWriter.this.packedBytesLength), packedValues, this.leafDocs, 0));
            BKDWriter.this.writeLeafBlockPackedValues(BKDWriter.this.scratchOut, BKDWriter.this.commonPrefixLengths, this.leafCount, 0, packedValues);
            this.out.writeBytes(BKDWriter.this.scratchOut.getBytes(), 0, BKDWriter.this.scratchOut.getPosition());
            BKDWriter.this.scratchOut.reset();
        }
    }

    private static class BKDMergeQueue
    extends PriorityQueue<MergeReader> {
        private final int bytesPerDim;

        public BKDMergeQueue(int bytesPerDim, int maxSize) {
            super(maxSize);
            this.bytesPerDim = bytesPerDim;
        }

        @Override
        public boolean lessThan(MergeReader a, MergeReader b) {
            assert (a != b);
            int cmp = FutureArrays.compareUnsigned(a.state.scratchPackedValue1, 0, this.bytesPerDim, b.state.scratchPackedValue1, 0, this.bytesPerDim);
            if (cmp < 0) {
                return true;
            }
            if (cmp > 0) {
                return false;
            }
            return a.docID < b.docID;
        }
    }

    private static class MergeReader {
        final BKDReader bkd;
        final BKDReader.IntersectState state;
        final MergeState.DocMap docMap;
        public int docID;
        private int docBlockUpto;
        private int docsInBlock;
        private int blockID;
        private final byte[] packedValues;

        public MergeReader(BKDReader bkd, MergeState.DocMap docMap) throws IOException {
            this.bkd = bkd;
            this.state = new BKDReader.IntersectState(bkd.in.clone(), bkd.numDims, bkd.packedBytesLength, bkd.maxPointsInLeafNode, null, null);
            this.docMap = docMap;
            this.state.in.seek(bkd.getMinLeafBlockFP());
            this.packedValues = new byte[bkd.maxPointsInLeafNode * bkd.packedBytesLength];
        }

        public boolean next() throws IOException {
            int index;
            int oldDocID;
            int mappedDocID;
            do {
                if (this.docBlockUpto == this.docsInBlock) {
                    if (this.blockID == this.bkd.leafNodeOffset) {
                        return false;
                    }
                    this.docsInBlock = this.bkd.readDocIDs(this.state.in, this.state.in.getFilePointer(), this.state.scratchDocIDs);
                    assert (this.docsInBlock > 0);
                    this.docBlockUpto = 0;
                    this.bkd.visitDocValues(this.state.commonPrefixLengths, this.state.scratchPackedValue1, this.state.scratchPackedValue2, this.state.in, this.state.scratchDocIDs, this.docsInBlock, new PointValues.IntersectVisitor(){
                        int i = 0;

                        @Override
                        public void visit(int docID) {
                            throw new UnsupportedOperationException();
                        }

                        @Override
                        public void visit(int docID, byte[] packedValue) {
                            assert (docID == state.scratchDocIDs[this.i]);
                            System.arraycopy(packedValue, 0, packedValues, this.i * bkd.packedBytesLength, bkd.packedBytesLength);
                            ++this.i;
                        }

                        @Override
                        public PointValues.Relation compare(byte[] minPackedValue, byte[] maxPackedValue) {
                            return PointValues.Relation.CELL_CROSSES_QUERY;
                        }
                    });
                    ++this.blockID;
                }
                ++this.docBlockUpto;
                oldDocID = this.state.scratchDocIDs[index];
            } while ((mappedDocID = this.docMap == null ? oldDocID : this.docMap.get(oldDocID)) == -1);
            this.docID = mappedDocID;
            System.arraycopy(this.packedValues, index * this.bkd.packedBytesLength, this.state.scratchPackedValue1, 0, this.bkd.packedBytesLength);
            return true;
        }
    }
}

