/*
 * Decompiled with CFR 0.152.
 */
package org.openstreetmap.josm.data.cache;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URL;
import java.security.SecureRandom;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import org.apache.commons.jcs.access.behavior.ICacheAccess;
import org.apache.commons.jcs.engine.behavior.ICacheElement;
import org.openstreetmap.josm.data.cache.CacheEntry;
import org.openstreetmap.josm.data.cache.CacheEntryAttributes;
import org.openstreetmap.josm.data.cache.ICachedLoaderJob;
import org.openstreetmap.josm.data.cache.ICachedLoaderListener;
import org.openstreetmap.josm.data.imagery.TileJobOptions;
import org.openstreetmap.josm.data.preferences.IntegerProperty;
import org.openstreetmap.josm.tools.CheckParameterUtil;
import org.openstreetmap.josm.tools.HttpClient;
import org.openstreetmap.josm.tools.Logging;
import org.openstreetmap.josm.tools.Utils;

public abstract class JCSCachedTileLoaderJob<K, V extends CacheEntry>
implements ICachedLoaderJob<K> {
    protected static final long DEFAULT_EXPIRE_TIME = TimeUnit.DAYS.toMillis(7L);
    protected static final long EXPIRE_TIME_SERVER_LIMIT = TimeUnit.DAYS.toMillis(28L);
    protected static final long ABSOLUTE_EXPIRE_TIME_LIMIT = TimeUnit.DAYS.toMillis(365L);
    public static final IntegerProperty THREAD_LIMIT = new IntegerProperty("cache.jcs.max_threads", 10);
    private static final ThreadPoolExecutor DEFAULT_DOWNLOAD_JOB_DISPATCHER = new ThreadPoolExecutor(1, (int)THREAD_LIMIT.get(), 30L, TimeUnit.SECONDS, new LinkedBlockingDeque<Runnable>(), Utils.newThreadFactory("JCS-downloader-%d", 5));
    private static final ConcurrentMap<String, Set<ICachedLoaderListener>> inProgress = new ConcurrentHashMap<String, Set<ICachedLoaderListener>>();
    private static final ConcurrentMap<String, Boolean> useHead = new ConcurrentHashMap<String, Boolean>();
    protected final long now;
    private final ICacheAccess<K, V> cache;
    private ICacheElement<K, V> cacheElement;
    protected V cacheData;
    protected CacheEntryAttributes attributes;
    private final int connectTimeout;
    private final int readTimeout;
    private final Map<String, String> headers;
    private final ThreadPoolExecutor downloadJobExecutor;
    private Runnable finishTask;
    private boolean force;
    private long minimumExpiryTime;

    public JCSCachedTileLoaderJob(ICacheAccess<K, V> cache, TileJobOptions options, ThreadPoolExecutor downloadJobExecutor) {
        CheckParameterUtil.ensureParameterNotNull(cache, "cache");
        this.cache = cache;
        this.now = System.currentTimeMillis();
        this.connectTimeout = options.getConnectionTimeout();
        this.readTimeout = options.getReadTimeout();
        this.headers = options.getHeaders();
        this.downloadJobExecutor = downloadJobExecutor;
        this.minimumExpiryTime = TimeUnit.SECONDS.toMillis(options.getMinimumExpiryTime());
    }

    public JCSCachedTileLoaderJob(ICacheAccess<K, V> cache, TileJobOptions options) {
        this(cache, options, DEFAULT_DOWNLOAD_JOB_DISPATCHER);
    }

    private void ensureCacheElement() {
        if (this.cacheElement == null && this.getCacheKey() != null) {
            this.cacheElement = this.cache.getCacheElement(this.getCacheKey());
            if (this.cacheElement != null) {
                this.attributes = (CacheEntryAttributes)this.cacheElement.getElementAttributes();
                this.cacheData = (CacheEntry)this.cacheElement.getVal();
            }
        }
    }

    public V get() {
        this.ensureCacheElement();
        return this.cacheData;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void submit(ICachedLoaderListener listener, boolean force) throws IOException {
        this.force = force;
        boolean first = false;
        URL url = this.getUrl();
        String deduplicationKey = null;
        if (url != null) {
            deduplicationKey = url.toString();
        }
        if (deduplicationKey == null) {
            Logging.warn("No url returned for: {0}, skipping", this.getCacheKey());
            throw new IllegalArgumentException("No url returned");
        }
        ConcurrentMap<String, Set<ICachedLoaderListener>> concurrentMap = inProgress;
        synchronized (concurrentMap) {
            HashSet<ICachedLoaderListener> newListeners = (HashSet<ICachedLoaderListener>)inProgress.get(deduplicationKey);
            if (newListeners == null) {
                newListeners = new HashSet<ICachedLoaderListener>();
                inProgress.put(deduplicationKey, newListeners);
                first = true;
            }
            newListeners.add(listener);
        }
        if (first || force) {
            Logging.debug("JCS - Submitting job for execution for url: {0}", this.getUrlNoException());
            this.downloadJobExecutor.execute(this);
        }
    }

    protected void executionFinished() {
        if (this.finishTask != null) {
            this.finishTask.run();
        }
    }

    protected boolean isObjectLoadable() {
        if (this.cacheData == null) {
            return false;
        }
        return ((CacheEntry)this.cacheData).getContent().length > 0;
    }

    protected boolean cacheAsEmpty() {
        return this.attributes.getResponseCode() < 500;
    }

    protected String getServerKey() {
        try {
            return this.getUrl().getHost();
        }
        catch (IOException e) {
            Logging.trace(e);
            return null;
        }
    }

    @Override
    public void run() {
        Thread currentThread = Thread.currentThread();
        String oldName = currentThread.getName();
        currentThread.setName("JCS Downloading: " + this.getUrlNoException());
        Logging.debug("JCS - starting fetch of url: {0} ", this.getUrlNoException());
        this.ensureCacheElement();
        try {
            if (!this.force && this.cacheElement != null && this.isCacheElementValid() && this.isObjectLoadable()) {
                Logging.debug("JCS - Returning object from cache: {0}", this.getCacheKey());
                this.finishLoading(ICachedLoaderListener.LoadResult.SUCCESS);
                return;
            }
            if (this.loadObject()) {
                this.finishLoading(ICachedLoaderListener.LoadResult.SUCCESS);
            } else if (this.isObjectLoadable()) {
                this.finishLoading(ICachedLoaderListener.LoadResult.SUCCESS);
                Logging.debug("JCS - found stale object in cache: {0}", this.getUrlNoException());
            } else {
                this.finishLoading(ICachedLoaderListener.LoadResult.FAILURE);
            }
        }
        finally {
            this.executionFinished();
            currentThread.setName(oldName);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void finishLoading(ICachedLoaderListener.LoadResult result) {
        Set listeners;
        ConcurrentMap<String, Set<ICachedLoaderListener>> concurrentMap = inProgress;
        synchronized (concurrentMap) {
            try {
                listeners = (Set)inProgress.remove(this.getUrl().toString());
            }
            catch (IOException e) {
                listeners = null;
                Logging.trace(e);
            }
        }
        if (listeners == null) {
            Logging.warn("Listener not found for URL: {0}. Listener not notified!", this.getUrlNoException());
            return;
        }
        for (ICachedLoaderListener l : listeners) {
            l.loadingFinished((CacheEntry)this.cacheData, this.attributes, result);
        }
    }

    protected boolean isCacheElementValid() {
        long expires = this.attributes.getExpirationTime();
        if (expires != 0L) {
            if (this.now > (expires = Math.min(expires, this.attributes.getCreateTime() + Math.max(EXPIRE_TIME_SERVER_LIMIT, this.minimumExpiryTime)))) {
                Logging.debug("JCS - Object {0} has expired -> valid to {1}, now is: {2}", this.getUrlNoException(), Long.toString(expires), Long.toString(this.now));
                return false;
            }
        } else {
            if (this.attributes.getLastModification() > 0L && this.now - this.attributes.getLastModification() > Math.max(DEFAULT_EXPIRE_TIME, this.minimumExpiryTime)) {
                Logging.debug("JCS - Object has expired, maximum file age reached {0}", this.getUrlNoException());
                return false;
            }
            if (this.now - this.attributes.getCreateTime() > Math.max(DEFAULT_EXPIRE_TIME, this.minimumExpiryTime)) {
                Logging.debug("JCS - Object has expired, maximum time since object creation reached {0}", this.getUrlNoException());
                return false;
            }
        }
        return true;
    }

    private boolean loadObject() {
        if (this.attributes == null) {
            this.attributes = new CacheEntryAttributes();
        }
        try {
            HttpClient.Response urlConn;
            if (this.isObjectLoadable() && Boolean.TRUE.equals(useHead.get(this.getServerKey())) && this.isCacheValidUsingHead()) {
                Logging.debug("JCS - cache entry verified using HEAD request: {0}", this.getUrl());
                return true;
            }
            Logging.debug("JCS - starting HttpClient GET request for URL: {0}", this.getUrl());
            HttpClient request = this.getRequest("GET", true);
            if (this.isObjectLoadable() && this.now - this.attributes.getLastModification() <= ABSOLUTE_EXPIRE_TIME_LIMIT) {
                request.setIfModifiedSince(this.attributes.getLastModification());
            }
            if (this.isObjectLoadable() && this.attributes.getEtag() != null) {
                request.setHeader("If-None-Match", this.attributes.getEtag());
            }
            if ((urlConn = request.connect()).getResponseCode() == 304) {
                Logging.debug("JCS - If-Modified-Since/ETag test: local version is up to date: {0}", this.getUrl());
                this.attributes = this.parseHeaders(urlConn);
                this.cache.put(this.getCacheKey(), this.cacheData, this.attributes);
                return true;
            }
            if (this.isObjectLoadable() && (this.attributes.getEtag() != null && this.attributes.getEtag().equals(urlConn.getHeaderField("ETag")) || this.attributes.getLastModification() == urlConn.getLastModified())) {
                String serverKey = this.getServerKey();
                Logging.info("JCS - Host: {0} found not to return 304 codes for If-Modified-Since or If-None-Match headers", serverKey);
                useHead.put(serverKey, Boolean.TRUE);
            }
            this.attributes = this.parseHeaders(urlConn);
            for (int i = 0; i < 5; ++i) {
                if (urlConn.getResponseCode() != 503) {
                    byte[] raw;
                    this.attributes.setResponseCode(urlConn.getResponseCode());
                    if (urlConn.getResponseCode() == 200) {
                        raw = Utils.readBytesFromStream(urlConn.getContent());
                    } else {
                        raw = new byte[]{};
                        try {
                            String detectErrorMessage;
                            String data = urlConn.fetchContent();
                            if (!data.isEmpty() && (detectErrorMessage = this.detectErrorMessage(data)) != null) {
                                this.attributes.setErrorMessage(detectErrorMessage);
                            }
                        }
                        catch (IOException e) {
                            Logging.warn(e);
                        }
                    }
                    if (this.isResponseLoadable(urlConn.getHeaderFields(), urlConn.getResponseCode(), raw)) {
                        this.cacheData = this.createCacheEntry(raw);
                        this.cache.put(this.getCacheKey(), this.cacheData, this.attributes);
                        Logging.debug("JCS - downloaded key: {0}, length: {1}, url: {2}", this.getCacheKey(), raw.length, this.getUrl());
                        return true;
                    }
                    if (this.cacheAsEmpty()) {
                        this.cacheData = this.createCacheEntry(new byte[0]);
                        this.cache.put(this.getCacheKey(), this.cacheData, this.attributes);
                        Logging.debug("JCS - Caching empty object {0}", this.getUrl());
                        return true;
                    }
                    Logging.debug("JCS - failure during load - response is not loadable nor cached as empty");
                    return false;
                }
                Thread.sleep(5000L + (long)new SecureRandom().nextInt(5000));
            }
        }
        catch (FileNotFoundException e) {
            boolean doCache;
            Logging.debug("JCS - Caching empty object as server returned 404 for: {0}", this.getUrlNoException());
            this.attributes.setResponseCode(404);
            this.attributes.setError(e);
            this.attributes.setException(e);
            boolean bl = doCache = this.isResponseLoadable(null, 404, null) || this.cacheAsEmpty();
            if (doCache) {
                this.cacheData = this.createCacheEntry(new byte[0]);
                this.cache.put(this.getCacheKey(), this.cacheData, this.attributes);
            }
            return doCache;
        }
        catch (IOException e) {
            Logging.debug("JCS - IOException during communication with server for: {0}", this.getUrlNoException());
            if (this.isObjectLoadable()) {
                return true;
            }
            this.attributes.setError(e);
            this.attributes.setException(e);
            this.attributes.setResponseCode(599);
            return false;
        }
        catch (InterruptedException e) {
            this.attributes.setError(e);
            this.attributes.setException(e);
            Logging.logWithStackTrace(Logging.LEVEL_WARN, e, "JCS - Exception during download {0}", this.getUrlNoException());
            Thread.currentThread().interrupt();
        }
        Logging.warn("JCS - Silent failure during download: {0}", this.getUrlNoException());
        return false;
    }

    protected String detectErrorMessage(String data) {
        Matcher m = HttpClient.getTomcatErrorMatcher(data);
        return m.matches() ? m.group(1).replace("'", "''") : null;
    }

    protected boolean isResponseLoadable(Map<String, List<String>> headerFields, int responseCode, byte[] raw) {
        return raw != null && raw.length != 0 && responseCode < 400;
    }

    protected abstract V createCacheEntry(byte[] var1);

    protected CacheEntryAttributes parseHeaders(HttpClient.Response urlConn) {
        CacheEntryAttributes ret = new CacheEntryAttributes();
        Long expiration = 0L;
        String cacheControl = urlConn.getHeaderField("Cache-Control");
        if (cacheControl != null) {
            for (String token : cacheControl.split(",")) {
                try {
                    if (token.startsWith("max-age=")) {
                        expiration = Math.max(expiration, TimeUnit.SECONDS.toMillis(Long.parseLong(token.substring("max-age=".length()))) + System.currentTimeMillis());
                    }
                    if (!token.startsWith("s-max-age=")) continue;
                    expiration = Math.max(expiration, TimeUnit.SECONDS.toMillis(Long.parseLong(token.substring("s-max-age=".length()))) + System.currentTimeMillis());
                }
                catch (NumberFormatException e) {
                    Logging.trace(e);
                }
            }
        }
        if (expiration.equals(0L)) {
            expiration = urlConn.getExpiration();
        }
        if (expiration.equals(0L)) {
            expiration = System.currentTimeMillis() + DEFAULT_EXPIRE_TIME;
        }
        ret.setExpirationTime(Math.max(this.minimumExpiryTime + System.currentTimeMillis(), expiration));
        ret.setLastModification(this.now);
        ret.setEtag(urlConn.getHeaderField("ETag"));
        return ret;
    }

    private HttpClient getRequest(String requestMethod, boolean noCache) throws IOException {
        HttpClient urlConn = HttpClient.create(this.getUrl(), requestMethod);
        urlConn.setAccept("text/html, image/png, image/jpeg, image/gif, */*");
        urlConn.setReadTimeout(this.readTimeout);
        urlConn.setConnectTimeout(this.connectTimeout);
        if (this.headers != null) {
            urlConn.setHeaders(this.headers);
        }
        if (this.force || noCache) {
            urlConn.useCache(false);
        }
        return urlConn;
    }

    private boolean isCacheValidUsingHead() throws IOException {
        boolean ret;
        HttpClient.Response urlConn = this.getRequest("HEAD", false).connect();
        long lastModified = urlConn.getLastModified();
        boolean bl = ret = this.attributes.getEtag() != null && this.attributes.getEtag().equals(urlConn.getHeaderField("ETag")) || lastModified != 0L && lastModified <= this.attributes.getLastModification();
        if (ret) {
            this.attributes = this.parseHeaders(urlConn);
            this.cache.put(this.getCacheKey(), this.cacheData, this.attributes);
        }
        return ret;
    }

    public void cancelOutstandingTasks() {
        for (Runnable r : this.downloadJobExecutor.getQueue()) {
            if (!this.downloadJobExecutor.remove(r) || !(r instanceof JCSCachedTileLoaderJob)) continue;
            ((JCSCachedTileLoaderJob)r).handleJobCancellation();
        }
    }

    public void setFinishedTask(Runnable runnable) {
        this.finishTask = runnable;
    }

    public void handleJobCancellation() {
        this.finishLoading(ICachedLoaderListener.LoadResult.CANCELED);
    }

    private URL getUrlNoException() {
        try {
            return this.getUrl();
        }
        catch (IOException e) {
            return null;
        }
    }
}

