/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.time;

import java.time.DateTimeException;
import java.time.DayOfWeek;
import java.time.Instant;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoField;
import java.time.temporal.TemporalAccessor;
import java.time.temporal.TemporalAdjusters;
import java.time.temporal.TemporalField;
import java.time.temporal.TemporalQueries;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.function.LongSupplier;
import org.elasticsearch.ElasticsearchParseException;
import org.elasticsearch.common.time.DateFormatter;
import org.elasticsearch.common.time.DateFormatters;
import org.elasticsearch.common.time.DateMathParser;

public class JavaDateMathParser
implements DateMathParser {
    private static final Map<TemporalField, Long> ROUND_UP_BASE_FIELDS = new HashMap<TemporalField, Long>(6);
    private final DateFormatter formatter;
    private final DateFormatter roundUpFormatter;

    public JavaDateMathParser(DateFormatter formatter) {
        ROUND_UP_BASE_FIELDS.put(ChronoField.MONTH_OF_YEAR, 1L);
        ROUND_UP_BASE_FIELDS.put(ChronoField.DAY_OF_MONTH, 1L);
        ROUND_UP_BASE_FIELDS.put(ChronoField.HOUR_OF_DAY, 23L);
        ROUND_UP_BASE_FIELDS.put(ChronoField.MINUTE_OF_HOUR, 59L);
        ROUND_UP_BASE_FIELDS.put(ChronoField.SECOND_OF_MINUTE, 59L);
        ROUND_UP_BASE_FIELDS.put(ChronoField.MILLI_OF_SECOND, 999L);
        Objects.requireNonNull(formatter);
        this.formatter = formatter;
        this.roundUpFormatter = formatter.parseDefaulting(ROUND_UP_BASE_FIELDS);
    }

    @Override
    public long parse(String text, LongSupplier now, boolean roundUp, ZoneId timeZone) {
        String mathString;
        long time;
        if (text.startsWith("now")) {
            try {
                time = now.getAsLong();
            }
            catch (Exception e) {
                throw new ElasticsearchParseException("could not read the current timestamp", (Throwable)e, new Object[0]);
            }
            mathString = text.substring("now".length());
        } else {
            int index = text.indexOf("||");
            if (index == -1) {
                return this.parseDateTime(text, timeZone, roundUp);
            }
            time = this.parseDateTime(text.substring(0, index), timeZone, false);
            mathString = text.substring(index + 2);
        }
        return this.parseMath(mathString, time, roundUp, timeZone);
    }

    private long parseMath(String mathString, long time, boolean roundUp, ZoneId timeZone) throws ElasticsearchParseException {
        if (timeZone == null) {
            timeZone = ZoneOffset.UTC;
        }
        ZonedDateTime dateTime = ZonedDateTime.ofInstant(Instant.ofEpochMilli(time), timeZone);
        int i = 0;
        while (i < mathString.length()) {
            int num;
            int sign;
            boolean round;
            char c;
            if ((c = mathString.charAt(i++)) == '/') {
                round = true;
                sign = 1;
            } else {
                round = false;
                if (c == '+') {
                    sign = 1;
                } else if (c == '-') {
                    sign = -1;
                } else {
                    throw new ElasticsearchParseException("operator not supported for date math [{}]", mathString);
                }
            }
            if (i >= mathString.length()) {
                throw new ElasticsearchParseException("truncated date math [{}]", mathString);
            }
            if (!Character.isDigit(mathString.charAt(i))) {
                num = 1;
            } else {
                int numFrom = i;
                while (i < mathString.length() && Character.isDigit(mathString.charAt(i))) {
                    ++i;
                }
                if (i >= mathString.length()) {
                    throw new ElasticsearchParseException("truncated date math [{}]", mathString);
                }
                num = Integer.parseInt(mathString.substring(numFrom, i));
            }
            if (round && num != 1) {
                throw new ElasticsearchParseException("rounding `/` can only be used on single unit types [{}]", mathString);
            }
            char unit = mathString.charAt(i++);
            switch (unit) {
                case 'y': {
                    dateTime = round ? dateTime.withDayOfYear(1).with(LocalTime.MIN) : dateTime.plusYears(sign * num);
                    if (!roundUp) break;
                    dateTime = dateTime.plusYears(1L);
                    break;
                }
                case 'M': {
                    dateTime = round ? dateTime.withDayOfMonth(1).with(LocalTime.MIN) : dateTime.plusMonths(sign * num);
                    if (!roundUp) break;
                    dateTime = dateTime.plusMonths(1L);
                    break;
                }
                case 'w': {
                    dateTime = round ? dateTime.with(TemporalAdjusters.previousOrSame(DayOfWeek.MONDAY)).with(LocalTime.MIN) : dateTime.plusWeeks(sign * num);
                    if (!roundUp) break;
                    dateTime = dateTime.plusWeeks(1L);
                    break;
                }
                case 'd': {
                    dateTime = round ? dateTime.with(LocalTime.MIN) : dateTime.plusDays(sign * num);
                    if (!roundUp) break;
                    dateTime = dateTime.plusDays(1L);
                    break;
                }
                case 'H': 
                case 'h': {
                    dateTime = round ? dateTime.withMinute(0).withSecond(0).withNano(0) : dateTime.plusHours(sign * num);
                    if (!roundUp) break;
                    dateTime = dateTime.plusHours(1L);
                    break;
                }
                case 'm': {
                    dateTime = round ? dateTime.withSecond(0).withNano(0) : dateTime.plusMinutes(sign * num);
                    if (!roundUp) break;
                    dateTime = dateTime.plusMinutes(1L);
                    break;
                }
                case 's': {
                    dateTime = round ? dateTime.withNano(0) : dateTime.plusSeconds(sign * num);
                    if (!roundUp) break;
                    dateTime = dateTime.plusSeconds(1L);
                    break;
                }
                default: {
                    throw new ElasticsearchParseException("unit [{}] not supported for date math [{}]", Character.valueOf(unit), mathString);
                }
            }
            if (!roundUp) continue;
            dateTime = dateTime.minus(1L, ChronoField.MILLI_OF_SECOND.getBaseUnit());
        }
        return dateTime.toInstant().toEpochMilli();
    }

    private long parseDateTime(String value, ZoneId timeZone, boolean roundUpIfNoTime) {
        DateFormatter formatter = roundUpIfNoTime ? this.roundUpFormatter : this.formatter;
        try {
            if (timeZone == null) {
                return DateFormatters.toZonedDateTime(formatter.parse(value)).toInstant().toEpochMilli();
            }
            TemporalAccessor accessor = formatter.parse(value);
            ZoneId zoneId = TemporalQueries.zone().queryFrom(accessor);
            if (zoneId != null) {
                timeZone = zoneId;
            }
            return DateFormatters.toZonedDateTime(accessor).withZoneSameLocal(timeZone).toInstant().toEpochMilli();
        }
        catch (IllegalArgumentException | DateTimeException e) {
            throw new ElasticsearchParseException("failed to parse date field [{}]: [{}]", (Throwable)e, value, e.getMessage());
        }
    }
}

