/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.license;

import java.time.Clock;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.logging.log4j.util.Supplier;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.Version;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.cluster.AckedClusterStateUpdateTask;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.ClusterStateTaskConfig;
import org.elasticsearch.cluster.ack.AckedRequest;
import org.elasticsearch.cluster.ack.ClusterStateUpdateResponse;
import org.elasticsearch.cluster.metadata.MetaData;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.component.Lifecycle;
import org.elasticsearch.common.joda.FormatDateTimeFormatter;
import org.elasticsearch.common.joda.Joda;
import org.elasticsearch.common.logging.LoggerMessageFormat;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.discovery.DiscoveryModule;
import org.elasticsearch.env.Environment;
import org.elasticsearch.gateway.GatewayService;
import org.elasticsearch.license.ExpirationCallback;
import org.elasticsearch.license.License;
import org.elasticsearch.license.LicenseUtils;
import org.elasticsearch.license.LicenseVerifier;
import org.elasticsearch.license.LicensesMetaData;
import org.elasticsearch.license.OperationModeFileWatcher;
import org.elasticsearch.license.PostStartBasicRequest;
import org.elasticsearch.license.PostStartBasicResponse;
import org.elasticsearch.license.PostStartTrialRequest;
import org.elasticsearch.license.PostStartTrialResponse;
import org.elasticsearch.license.PutLicenseRequest;
import org.elasticsearch.license.SelfGeneratedLicense;
import org.elasticsearch.license.StartBasicClusterTask;
import org.elasticsearch.license.StartTrialClusterTask;
import org.elasticsearch.license.StartupSelfGeneratedLicenseTask;
import org.elasticsearch.license.XPackLicenseState;
import org.elasticsearch.protocol.xpack.XPackInfoResponse;
import org.elasticsearch.protocol.xpack.license.DeleteLicenseRequest;
import org.elasticsearch.protocol.xpack.license.LicensesStatus;
import org.elasticsearch.protocol.xpack.license.PutLicenseResponse;
import org.elasticsearch.watcher.ResourceWatcherService;
import org.elasticsearch.xpack.core.XPackPlugin;
import org.elasticsearch.xpack.core.XPackSettings;
import org.elasticsearch.xpack.core.scheduler.SchedulerEngine;

public class LicenseService
extends AbstractLifecycleComponent
implements ClusterStateListener,
SchedulerEngine.Listener {
    public static final Setting<String> SELF_GENERATED_LICENSE_TYPE = new Setting("xpack.license.self_generated.type", s -> "basic", s -> {
        if (SelfGeneratedLicense.validSelfGeneratedType(s)) {
            return s;
        }
        throw new IllegalArgumentException("Illegal self generated license type [" + s + "]. Must be trial or basic.");
    }, new Setting.Property[]{Setting.Property.NodeScope});
    static final TimeValue NON_BASIC_SELF_GENERATED_LICENSE_DURATION = TimeValue.timeValueHours((long)720L);
    static final Set<String> VALID_TRIAL_TYPES = new HashSet<String>(Arrays.asList("trial", "platinum", "gold"));
    static final TimeValue GRACE_PERIOD_DURATION = LicenseService.days(7);
    public static final long BASIC_SELF_GENERATED_LICENSE_EXPIRATION_MILLIS = XPackInfoResponse.BASIC_SELF_GENERATED_LICENSE_EXPIRATION_MILLIS;
    private final ClusterService clusterService;
    private final XPackLicenseState licenseState;
    private final AtomicReference<License> currentLicense = new AtomicReference();
    private SchedulerEngine scheduler;
    private final Clock clock;
    private final OperationModeFileWatcher operationModeFileWatcher;
    private List<ExpirationCallback> expirationCallbacks = new ArrayList<ExpirationCallback>();
    static final int SELF_GENERATED_LICENSE_MAX_NODES = 1000;
    public static final String LICENSE_JOB = "licenseJob";
    private static final FormatDateTimeFormatter DATE_FORMATTER = Joda.forPattern((String)"EEEE, MMMMM dd, yyyy", (Locale)Locale.ROOT);
    private static final String ACKNOWLEDGEMENT_HEADER = "This license update requires acknowledgement. To acknowledge the license, please read the following messages and update the license again, this time with the \"acknowledge=true\" parameter:";

    public LicenseService(Settings settings, ClusterService clusterService, Clock clock, Environment env, ResourceWatcherService resourceWatcherService, XPackLicenseState licenseState) {
        super(settings);
        this.clusterService = clusterService;
        this.clock = clock;
        this.scheduler = new SchedulerEngine(settings, clock);
        this.licenseState = licenseState;
        this.operationModeFileWatcher = new OperationModeFileWatcher(resourceWatcherService, XPackPlugin.resolveConfigFile(env, "license_mode"), this.logger, () -> this.updateLicenseState(this.getLicensesMetaData()));
        this.scheduler.register(this);
        this.populateExpirationCallbacks();
    }

    private void logExpirationWarning(long expirationMillis, boolean expired) {
        String expiredMsg = expired ? "expired" : "will expire";
        String general = LoggerMessageFormat.format(null, (String)"License [{}] on [{}].\n# If you have a new license, please update it. Otherwise, please reach out to\n# your support contact.\n# ", (Object[])new Object[]{expiredMsg, DATE_FORMATTER.printer().print(expirationMillis)});
        if (expired) {
            general = general.toUpperCase(Locale.ROOT);
        }
        StringBuilder builder = new StringBuilder(general);
        builder.append(System.lineSeparator());
        if (expired) {
            builder.append("# COMMERCIAL PLUGINS OPERATING WITH REDUCED FUNCTIONALITY");
        } else {
            builder.append("# Commercial plugins operate with reduced functionality on license expiration:");
        }
        XPackLicenseState.EXPIRATION_MESSAGES.forEach((feature, messages) -> {
            if (((String[])messages).length > 0) {
                builder.append(System.lineSeparator());
                builder.append("# - ");
                builder.append((String)feature);
                for (String message : messages) {
                    builder.append(System.lineSeparator());
                    builder.append("#  - ");
                    builder.append(message);
                }
            }
        });
        this.logger.warn("{}", (Object)builder);
    }

    private void populateExpirationCallbacks() {
        this.expirationCallbacks.add(new ExpirationCallback.Pre(LicenseService.days(7), LicenseService.days(25), LicenseService.days(1)){

            @Override
            public void on(License license) {
                LicenseService.this.logExpirationWarning(license.expiryDate(), false);
            }
        });
        this.expirationCallbacks.add(new ExpirationCallback.Pre(LicenseService.days(0), LicenseService.days(7), TimeValue.timeValueMinutes((long)10L)){

            @Override
            public void on(License license) {
                LicenseService.this.logExpirationWarning(license.expiryDate(), false);
            }
        });
        this.expirationCallbacks.add(new ExpirationCallback.Post(LicenseService.days(0), null, TimeValue.timeValueMinutes((long)10L)){

            @Override
            public void on(License license) {
                LicenseService.this.logExpirationWarning(license.expiryDate(), true);
            }
        });
    }

    public void registerLicense(PutLicenseRequest request, ActionListener<PutLicenseResponse> listener) {
        final License newLicense = request.license();
        long now = this.clock.millis();
        if (!LicenseVerifier.verifyLicense(newLicense) || newLicense.issueDate() > now || newLicense.startDate() > now) {
            listener.onResponse((Object)new PutLicenseResponse(true, LicensesStatus.INVALID));
        } else if (newLicense.type().equals("basic")) {
            listener.onFailure((Exception)new IllegalArgumentException("Registering basic licenses is not allowed."));
        } else if (newLicense.expiryDate() < now) {
            listener.onResponse((Object)new PutLicenseResponse(true, LicensesStatus.EXPIRED));
        } else {
            Map<String, String[]> acknowledgeMessages;
            License currentLicense;
            if (!request.acknowledged() && (currentLicense = this.getLicense()) != null && !(acknowledgeMessages = LicenseService.getAckMessages(newLicense, currentLicense)).isEmpty()) {
                listener.onResponse((Object)new PutLicenseResponse(false, LicensesStatus.VALID, ACKNOWLEDGEMENT_HEADER, acknowledgeMessages));
                return;
            }
            if (((Boolean)XPackSettings.SECURITY_ENABLED.get(this.settings)).booleanValue()) {
                if (newLicense.isProductionLicense() && !((Boolean)XPackSettings.TRANSPORT_SSL_ENABLED.get(this.settings)).booleanValue() && LicenseService.isProductionMode(this.settings, this.clusterService.localNode())) {
                    throw new IllegalStateException("Cannot install a [" + (Object)((Object)newLicense.operationMode()) + "] license unless TLS is configured or security is disabled");
                }
                if (((Boolean)XPackSettings.FIPS_MODE_ENABLED.get(this.settings)).booleanValue() && newLicense.operationMode() != License.OperationMode.PLATINUM && newLicense.operationMode() != License.OperationMode.TRIAL) {
                    throw new IllegalStateException("Cannot install a [" + (Object)((Object)newLicense.operationMode()) + "] license unless FIPS mode is disabled");
                }
            }
            this.clusterService.submitStateUpdateTask("register license [" + newLicense.uid() + "]", (ClusterStateTaskConfig)new AckedClusterStateUpdateTask<PutLicenseResponse>((AckedRequest)request, listener){

                protected PutLicenseResponse newResponse(boolean acknowledged) {
                    return new PutLicenseResponse(acknowledged, LicensesStatus.VALID);
                }

                public ClusterState execute(ClusterState currentState) throws Exception {
                    XPackPlugin.checkReadyForXPackCustomMetadata(currentState);
                    MetaData currentMetadata = currentState.metaData();
                    LicensesMetaData licensesMetaData = (LicensesMetaData)currentMetadata.custom("licenses");
                    Version trialVersion = null;
                    if (licensesMetaData != null) {
                        trialVersion = licensesMetaData.getMostRecentTrialVersion();
                    }
                    MetaData.Builder mdBuilder = MetaData.builder((MetaData)currentMetadata);
                    mdBuilder.putCustom("licenses", (MetaData.Custom)new LicensesMetaData(newLicense, trialVersion));
                    return ClusterState.builder((ClusterState)currentState).metaData(mdBuilder).build();
                }
            });
        }
    }

    public static Map<String, String[]> getAckMessages(License newLicense, License currentLicense) {
        HashMap<String, String[]> acknowledgeMessages = new HashMap<String, String[]>();
        if (!License.isAutoGeneratedLicense(currentLicense.signature()) && currentLicense.issueDate() > newLicense.issueDate()) {
            acknowledgeMessages.put("license", new String[]{"The new license is older than the currently installed license. Are you sure you want to override the current license?"});
        }
        XPackLicenseState.ACKNOWLEDGMENT_MESSAGES.forEach((feature, ackMessages) -> {
            String[] messages = (String[])ackMessages.apply(currentLicense.operationMode(), newLicense.operationMode());
            if (messages.length > 0) {
                acknowledgeMessages.put((String)feature, messages);
            }
        });
        return acknowledgeMessages;
    }

    private static TimeValue days(int days) {
        return TimeValue.timeValueHours((long)(days * 24));
    }

    @Override
    public void triggered(SchedulerEngine.Event event) {
        LicensesMetaData licensesMetaData = this.getLicensesMetaData();
        if (licensesMetaData != null) {
            License license = licensesMetaData.getLicense();
            if (event.getJobName().equals(LICENSE_JOB)) {
                this.updateLicenseState(license, licensesMetaData.getMostRecentTrialVersion());
            } else if (event.getJobName().startsWith(".license_expiration_job_")) {
                this.expirationCallbacks.stream().filter(expirationCallback -> expirationCallback.getId().equals(event.getJobName())).forEach(expirationCallback -> expirationCallback.on(license));
            }
        }
    }

    public void removeLicense(DeleteLicenseRequest request, ActionListener<ClusterStateUpdateResponse> listener) {
        this.clusterService.submitStateUpdateTask("delete license", (ClusterStateTaskConfig)new AckedClusterStateUpdateTask<ClusterStateUpdateResponse>((AckedRequest)request, listener){

            protected ClusterStateUpdateResponse newResponse(boolean acknowledged) {
                return new ClusterStateUpdateResponse(acknowledged);
            }

            public ClusterState execute(ClusterState currentState) throws Exception {
                MetaData metaData = currentState.metaData();
                LicensesMetaData currentLicenses = (LicensesMetaData)metaData.custom("licenses");
                if (currentLicenses.getLicense() != LicensesMetaData.LICENSE_TOMBSTONE) {
                    MetaData.Builder mdBuilder = MetaData.builder((MetaData)currentState.metaData());
                    LicensesMetaData newMetadata = new LicensesMetaData(LicensesMetaData.LICENSE_TOMBSTONE, currentLicenses.getMostRecentTrialVersion());
                    mdBuilder.putCustom("licenses", (MetaData.Custom)newMetadata);
                    return ClusterState.builder((ClusterState)currentState).metaData(mdBuilder).build();
                }
                return currentState;
            }
        });
    }

    public License getLicense() {
        License license = LicenseService.getLicense(this.clusterService.state().metaData());
        return license == LicensesMetaData.LICENSE_TOMBSTONE ? null : license;
    }

    private LicensesMetaData getLicensesMetaData() {
        return (LicensesMetaData)this.clusterService.state().metaData().custom("licenses");
    }

    void startTrialLicense(PostStartTrialRequest request, ActionListener<PostStartTrialResponse> listener) {
        if (!VALID_TRIAL_TYPES.contains(request.getType())) {
            throw new IllegalArgumentException("Cannot start trial of type [" + request.getType() + "]. Valid trial types are " + VALID_TRIAL_TYPES + ".");
        }
        StartTrialClusterTask task = new StartTrialClusterTask(this.logger, this.clusterService.getClusterName().value(), this.clock, request, listener);
        this.clusterService.submitStateUpdateTask("started trial license", (ClusterStateTaskConfig)task);
    }

    void startBasicLicense(PostStartBasicRequest request, ActionListener<PostStartBasicResponse> listener) {
        StartBasicClusterTask task = new StartBasicClusterTask(this.logger, this.clusterService.getClusterName().value(), this.clock, request, listener);
        this.clusterService.submitStateUpdateTask("start basic license", (ClusterStateTaskConfig)task);
    }

    private void registerOrUpdateSelfGeneratedLicense() {
        this.clusterService.submitStateUpdateTask("maybe generate license for cluster", (ClusterStateTaskConfig)new StartupSelfGeneratedLicenseTask(this.settings, this.clock, this.clusterService));
    }

    protected void doStart() throws ElasticsearchException {
        ClusterState clusterState;
        this.clusterService.addListener((ClusterStateListener)this);
        this.scheduler.start(Collections.emptyList());
        this.logger.debug("initializing license state");
        if (this.clusterService.lifecycleState() == Lifecycle.State.STARTED && !(clusterState = this.clusterService.state()).blocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK) && clusterState.nodes().getMasterNode() != null && XPackPlugin.isReadyForXPackCustomMetadata(clusterState)) {
            boolean noLicense;
            LicensesMetaData currentMetaData = (LicensesMetaData)clusterState.metaData().custom("licenses");
            boolean bl = noLicense = currentMetaData == null || currentMetaData.getLicense() == null;
            if (clusterState.getNodes().isLocalNodeElectedMaster() && (noLicense || LicenseUtils.licenseNeedsExtended(currentMetaData.getLicense()))) {
                this.registerOrUpdateSelfGeneratedLicense();
            }
        }
    }

    protected void doStop() throws ElasticsearchException {
        this.clusterService.removeListener((ClusterStateListener)this);
        this.scheduler.stop();
        this.currentLicense.set(null);
    }

    protected void doClose() throws ElasticsearchException {
    }

    public void clusterChanged(ClusterChangedEvent event) {
        ClusterState previousClusterState = event.previousState();
        ClusterState currentClusterState = event.state();
        if (!currentClusterState.blocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK)) {
            boolean noLicense;
            boolean noLicenseInCurrentMetadata;
            boolean noLicenseInPrevMetadata;
            if (!XPackPlugin.isReadyForXPackCustomMetadata(currentClusterState)) {
                this.logger.debug("cannot add license to cluster as the following nodes might not understand the license metadata: {}", new Supplier[]{() -> XPackPlugin.nodesNotReadyForXPackCustomMetadata(currentClusterState)});
                return;
            }
            LicensesMetaData prevLicensesMetaData = (LicensesMetaData)previousClusterState.getMetaData().custom("licenses");
            LicensesMetaData currentLicensesMetaData = (LicensesMetaData)currentClusterState.getMetaData().custom("licenses");
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("previous [{}]", (Object)prevLicensesMetaData);
                this.logger.debug("current [{}]", (Object)currentLicensesMetaData);
            }
            if (previousClusterState.blocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK) || prevLicensesMetaData == null) {
                if (currentLicensesMetaData != null) {
                    this.onUpdate(currentLicensesMetaData);
                }
            } else if (!prevLicensesMetaData.equals(currentLicensesMetaData)) {
                this.onUpdate(currentLicensesMetaData);
            }
            License currentLicense = null;
            boolean bl = noLicenseInPrevMetadata = prevLicensesMetaData == null || prevLicensesMetaData.getLicense() == null;
            if (!noLicenseInPrevMetadata) {
                currentLicense = prevLicensesMetaData.getLicense();
            }
            boolean bl2 = noLicenseInCurrentMetadata = currentLicensesMetaData == null || currentLicensesMetaData.getLicense() == null;
            if (!noLicenseInCurrentMetadata) {
                currentLicense = currentLicensesMetaData.getLicense();
            }
            boolean bl3 = noLicense = noLicenseInPrevMetadata && noLicenseInCurrentMetadata;
            if (currentClusterState.getNodes().isLocalNodeElectedMaster() && (noLicense || LicenseUtils.licenseNeedsExtended(currentLicense) || LicenseUtils.signatureNeedsUpdate(currentLicense, currentClusterState.nodes()))) {
                this.registerOrUpdateSelfGeneratedLicense();
            }
        } else if (this.logger.isDebugEnabled()) {
            this.logger.debug("skipped license notifications reason: [{}]", (Object)GatewayService.STATE_NOT_RECOVERED_BLOCK);
        }
    }

    private void updateLicenseState(LicensesMetaData licensesMetaData) {
        if (licensesMetaData != null) {
            this.updateLicenseState(LicenseService.getLicense(licensesMetaData), licensesMetaData.getMostRecentTrialVersion());
        }
    }

    protected void updateLicenseState(License license, Version mostRecentTrialVersion) {
        if (license == LicensesMetaData.LICENSE_TOMBSTONE) {
            this.licenseState.update(License.OperationMode.MISSING, false, mostRecentTrialVersion);
            return;
        }
        if (license != null) {
            long time = this.clock.millis();
            boolean active = license.expiryDate() == BASIC_SELF_GENERATED_LICENSE_EXPIRATION_MILLIS ? true : time >= license.issueDate() && time - GRACE_PERIOD_DURATION.getMillis() < license.expiryDate();
            this.licenseState.update(license.operationMode(), active, mostRecentTrialVersion);
            if (active) {
                if (time < license.expiryDate()) {
                    this.logger.debug("license [{}] - valid", (Object)license.uid());
                } else {
                    this.logger.warn("license [{}] - grace", (Object)license.uid());
                }
            } else {
                this.logger.warn("license [{}] - expired", (Object)license.uid());
            }
        }
    }

    private void onUpdate(LicensesMetaData currentLicensesMetaData) {
        License license = LicenseService.getLicense(currentLicensesMetaData);
        if (license != null) {
            License previousLicense = this.currentLicense.get();
            if (!license.equals(previousLicense)) {
                this.currentLicense.set(license);
                license.setOperationModeFileWatcher(this.operationModeFileWatcher);
                this.scheduler.add(new SchedulerEngine.Job(LICENSE_JOB, LicenseService.nextLicenseCheck(license)));
                for (ExpirationCallback expirationCallback : this.expirationCallbacks) {
                    this.scheduler.add(new SchedulerEngine.Job(expirationCallback.getId(), (startTime, now) -> expirationCallback.nextScheduledTimeForExpiry(license.expiryDate(), startTime, now)));
                }
                if (previousLicense != null) {
                    previousLicense.removeOperationModeFileWatcher();
                }
                this.logger.info("license [{}] mode [{}] - valid", (Object)license.uid(), (Object)license.operationMode().name().toLowerCase(Locale.ROOT));
            }
            this.updateLicenseState(license, currentLicensesMetaData.getMostRecentTrialVersion());
        }
    }

    static SchedulerEngine.Schedule nextLicenseCheck(License license) {
        return (startTime, time) -> {
            if (time < license.issueDate()) {
                return license.issueDate();
            }
            if (time < license.expiryDate()) {
                return license.expiryDate();
            }
            if (time < license.expiryDate() + GRACE_PERIOD_DURATION.getMillis()) {
                return license.expiryDate() + GRACE_PERIOD_DURATION.getMillis();
            }
            return -1L;
        };
    }

    public static License getLicense(MetaData metaData) {
        LicensesMetaData licensesMetaData = (LicensesMetaData)metaData.custom("licenses");
        return LicenseService.getLicense(licensesMetaData);
    }

    static License getLicense(LicensesMetaData metaData) {
        if (metaData != null) {
            boolean autoGeneratedLicense;
            License license = metaData.getLicense();
            if (license == LicensesMetaData.LICENSE_TOMBSTONE) {
                return license;
            }
            if (license != null && ((autoGeneratedLicense = License.isAutoGeneratedLicense(license.signature())) && SelfGeneratedLicense.verify(license) || !autoGeneratedLicense && LicenseVerifier.verifyLicense(license))) {
                return license;
            }
        }
        return null;
    }

    private static boolean isProductionMode(Settings settings, DiscoveryNode localNode) {
        boolean singleNodeDisco = "single-node".equals(DiscoveryModule.DISCOVERY_TYPE_SETTING.get(settings));
        return !singleNodeDisco && !LicenseService.isBoundToLoopback(localNode);
    }

    private static boolean isBoundToLoopback(DiscoveryNode localNode) {
        return localNode.getAddress().address().getAddress().isLoopbackAddress();
    }
}

