'use strict';

var _typeof2 = require('babel-runtime/helpers/typeof');

var _typeof3 = _interopRequireDefault(_typeof2);

var _getIterator2 = require('babel-runtime/core-js/get-iterator');

var _getIterator3 = _interopRequireDefault(_getIterator2);

var _keys = require('babel-runtime/core-js/object/keys');

var _keys2 = _interopRequireDefault(_keys);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
Copyright 2015, 2016 OpenMarket Ltd
Copyright 2017 New Vector Ltd

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
/**
 * @module pushprocessor
 */

var RULEKINDS_IN_ORDER = ['override', 'content', 'room', 'sender', 'underride'];

/**
 * Construct a Push Processor.
 * @constructor
 * @param {Object} client The Matrix client object to use
 */
function PushProcessor(client) {
    var _this = this;

    var escapeRegExp = function escapeRegExp(string) {
        return string.replace(/[.*+?^${}()|[\]\\]/g, "\\$&");
    };

    var cachedGlobToRegex = {
        // $glob: RegExp,
    };

    var matchingRuleFromKindSet = function matchingRuleFromKindSet(ev, kindset, device) {
        for (var ruleKindIndex = 0; ruleKindIndex < RULEKINDS_IN_ORDER.length; ++ruleKindIndex) {
            var kind = RULEKINDS_IN_ORDER[ruleKindIndex];
            var ruleset = kindset[kind];

            for (var ruleIndex = 0; ruleIndex < ruleset.length; ++ruleIndex) {
                var rule = ruleset[ruleIndex];
                if (!rule.enabled) {
                    continue;
                }

                var rawrule = templateRuleToRaw(kind, rule, device);
                if (!rawrule) {
                    continue;
                }

                if (_this.ruleMatchesEvent(rawrule, ev)) {
                    rule.kind = kind;
                    return rule;
                }
            }
        }
        return null;
    };

    var templateRuleToRaw = function templateRuleToRaw(kind, tprule, device) {
        var rawrule = {
            'rule_id': tprule.rule_id,
            'actions': tprule.actions,
            'conditions': []
        };
        switch (kind) {
            case 'underride':
            case 'override':
                rawrule.conditions = tprule.conditions;
                break;
            case 'room':
                if (!tprule.rule_id) {
                    return null;
                }
                rawrule.conditions.push({
                    'kind': 'event_match',
                    'key': 'room_id',
                    'value': tprule.rule_id
                });
                break;
            case 'sender':
                if (!tprule.rule_id) {
                    return null;
                }
                rawrule.conditions.push({
                    'kind': 'event_match',
                    'key': 'user_id',
                    'value': tprule.rule_id
                });
                break;
            case 'content':
                if (!tprule.pattern) {
                    return null;
                }
                rawrule.conditions.push({
                    'kind': 'event_match',
                    'key': 'content.body',
                    'pattern': tprule.pattern
                });
                break;
        }
        if (device) {
            rawrule.conditions.push({
                'kind': 'device',
                'profile_tag': device
            });
        }
        return rawrule;
    };

    var eventFulfillsCondition = function eventFulfillsCondition(cond, ev) {
        var condition_functions = {
            "event_match": eventFulfillsEventMatchCondition,
            "device": eventFulfillsDeviceCondition,
            "contains_display_name": eventFulfillsDisplayNameCondition,
            "room_member_count": eventFulfillsRoomMemberCountCondition,
            "sender_notification_permission": eventFulfillsSenderNotifPermCondition
        };
        if (condition_functions[cond.kind]) {
            return condition_functions[cond.kind](cond, ev);
        }
        // unknown conditions: we previously matched all unknown conditions,
        // but given that rules can be added to the base rules on a server,
        // it's probably better to not match unknown conditions.
        return false;
    };

    var eventFulfillsSenderNotifPermCondition = function eventFulfillsSenderNotifPermCondition(cond, ev) {
        var notifLevelKey = cond['key'];
        if (!notifLevelKey) {
            return false;
        }

        var room = client.getRoom(ev.getRoomId());
        if (!room || !room.currentState) {
            return false;
        }

        // Note that this should not be the current state of the room but the state at
        // the point the event is in the DAG. Unfortunately the js-sdk does not store
        // this.
        return room.currentState.mayTriggerNotifOfType(notifLevelKey, ev.getSender());
    };

    var eventFulfillsRoomMemberCountCondition = function eventFulfillsRoomMemberCountCondition(cond, ev) {
        if (!cond.is) {
            return false;
        }

        var room = client.getRoom(ev.getRoomId());
        if (!room || !room.currentState || !room.currentState.members) {
            return false;
        }

        var memberCount = room.currentState.getJoinedMemberCount();

        var m = cond.is.match(/^([=<>]*)([0-9]*)$/);
        if (!m) {
            return false;
        }
        var ineq = m[1];
        var rhs = parseInt(m[2]);
        if (isNaN(rhs)) {
            return false;
        }
        switch (ineq) {
            case '':
            case '==':
                return memberCount == rhs;
            case '<':
                return memberCount < rhs;
            case '>':
                return memberCount > rhs;
            case '<=':
                return memberCount <= rhs;
            case '>=':
                return memberCount >= rhs;
            default:
                return false;
        }
    };

    var eventFulfillsDisplayNameCondition = function eventFulfillsDisplayNameCondition(cond, ev) {
        var content = ev.getContent();
        if (!content || !content.body || typeof content.body != 'string') {
            return false;
        }

        var room = client.getRoom(ev.getRoomId());
        if (!room || !room.currentState || !room.currentState.members || !room.currentState.getMember(client.credentials.userId)) {
            return false;
        }

        var displayName = room.currentState.getMember(client.credentials.userId).name;

        // N.B. we can't use \b as it chokes on unicode. however \W seems to be okay
        // as shorthand for [^0-9A-Za-z_].
        var pat = new RegExp("(^|\\W)" + escapeRegExp(displayName) + "(\\W|$)", 'i');
        return content.body.search(pat) > -1;
    };

    var eventFulfillsDeviceCondition = function eventFulfillsDeviceCondition(cond, ev) {
        return false; // XXX: Allow a profile tag to be set for the web client instance
    };

    var eventFulfillsEventMatchCondition = function eventFulfillsEventMatchCondition(cond, ev) {
        if (!cond.key) {
            return false;
        }

        var val = valueForDottedKey(cond.key, ev);
        if (!val || typeof val != 'string') {
            return false;
        }

        if (cond.value) {
            return cond.value === val;
        }

        var regex = void 0;

        if (cond.key == 'content.body') {
            regex = createCachedRegex('(^|\\W)', cond.pattern, '(\\W|$)');
        } else {
            regex = createCachedRegex('^', cond.pattern, '$');
        }

        return !!val.match(regex);
    };

    var createCachedRegex = function createCachedRegex(prefix, glob, suffix) {
        if (cachedGlobToRegex[glob]) {
            return cachedGlobToRegex[glob];
        }
        cachedGlobToRegex[glob] = new RegExp(prefix + globToRegexp(glob) + suffix, 'i');
        return cachedGlobToRegex[glob];
    };

    var globToRegexp = function globToRegexp(glob) {
        // From
        // https://github.com/matrix-org/synapse/blob/abbee6b29be80a77e05730707602f3bbfc3f38cb/synapse/push/__init__.py#L132
        // Because micromatch is about 130KB with dependencies,
        // and minimatch is not much better.
        var pat = escapeRegExp(glob);
        pat = pat.replace(/\\\*/g, '.*');
        pat = pat.replace(/\?/g, '.');
        pat = pat.replace(/\\\[(!|)(.*)\\]/g, function (match, p1, p2, offset, string) {
            var first = p1 && '^' || '';
            var second = p2.replace(/\\\-/, '-');
            return '[' + first + second + ']';
        });
        return pat;
    };

    var valueForDottedKey = function valueForDottedKey(key, ev) {
        var parts = key.split('.');
        var val = void 0;

        // special-case the first component to deal with encrypted messages
        var firstPart = parts[0];
        if (firstPart == 'content') {
            val = ev.getContent();
            parts.shift();
        } else if (firstPart == 'type') {
            val = ev.getType();
            parts.shift();
        } else {
            // use the raw event for any other fields
            val = ev.event;
        }

        while (parts.length > 0) {
            var thispart = parts.shift();
            if (!val[thispart]) {
                return null;
            }
            val = val[thispart];
        }
        return val;
    };

    var matchingRuleForEventWithRulesets = function matchingRuleForEventWithRulesets(ev, rulesets) {
        if (!rulesets || !rulesets.device) {
            return null;
        }
        if (ev.getSender() == client.credentials.userId) {
            return null;
        }

        var allDevNames = (0, _keys2.default)(rulesets.device);
        for (var i = 0; i < allDevNames.length; ++i) {
            var devname = allDevNames[i];
            var devrules = rulesets.device[devname];

            var matchingRule = matchingRuleFromKindSet(devrules, devname);
            if (matchingRule) {
                return matchingRule;
            }
        }
        return matchingRuleFromKindSet(ev, rulesets.global);
    };

    var pushActionsForEventAndRulesets = function pushActionsForEventAndRulesets(ev, rulesets) {
        var rule = matchingRuleForEventWithRulesets(ev, rulesets);
        if (!rule) {
            return {};
        }

        var actionObj = PushProcessor.actionListToActionsObject(rule.actions);

        // Some actions are implicit in some situations: we add those here
        if (actionObj.tweaks.highlight === undefined) {
            // if it isn't specified, highlight if it's a content
            // rule but otherwise not
            actionObj.tweaks.highlight = rule.kind == 'content';
        }

        return actionObj;
    };

    this.ruleMatchesEvent = function (rule, ev) {
        var ret = true;
        for (var i = 0; i < rule.conditions.length; ++i) {
            var cond = rule.conditions[i];
            ret &= eventFulfillsCondition(cond, ev);
        }
        //console.log("Rule "+rule.rule_id+(ret ? " matches" : " doesn't match"));
        return ret;
    };

    /**
     * Get the user's push actions for the given event
     *
     * @param {module:models/event.MatrixEvent} ev
     *
     * @return {PushAction}
     */
    this.actionsForEvent = function (ev) {
        return pushActionsForEventAndRulesets(ev, client.pushRules);
    };

    /**
     * Get one of the users push rules by its ID
     *
     * @param {string} ruleId The ID of the rule to search for
     * @return {object} The push rule, or null if no such rule was found
     */
    this.getPushRuleById = function (ruleId) {
        var _arr = ['device', 'global'];

        for (var _i = 0; _i < _arr.length; _i++) {
            var scope = _arr[_i];
            if (client.pushRules[scope] === undefined) continue;

            var _iteratorNormalCompletion = true;
            var _didIteratorError = false;
            var _iteratorError = undefined;

            try {
                for (var _iterator = (0, _getIterator3.default)(RULEKINDS_IN_ORDER), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
                    var kind = _step.value;

                    if (client.pushRules[scope][kind] === undefined) continue;

                    var _iteratorNormalCompletion2 = true;
                    var _didIteratorError2 = false;
                    var _iteratorError2 = undefined;

                    try {
                        for (var _iterator2 = (0, _getIterator3.default)(client.pushRules[scope][kind]), _step2; !(_iteratorNormalCompletion2 = (_step2 = _iterator2.next()).done); _iteratorNormalCompletion2 = true) {
                            var rule = _step2.value;

                            if (rule.rule_id === ruleId) return rule;
                        }
                    } catch (err) {
                        _didIteratorError2 = true;
                        _iteratorError2 = err;
                    } finally {
                        try {
                            if (!_iteratorNormalCompletion2 && _iterator2.return) {
                                _iterator2.return();
                            }
                        } finally {
                            if (_didIteratorError2) {
                                throw _iteratorError2;
                            }
                        }
                    }
                }
            } catch (err) {
                _didIteratorError = true;
                _iteratorError = err;
            } finally {
                try {
                    if (!_iteratorNormalCompletion && _iterator.return) {
                        _iterator.return();
                    }
                } finally {
                    if (_didIteratorError) {
                        throw _iteratorError;
                    }
                }
            }
        }
        return null;
    };
}

/**
 * Convert a list of actions into a object with the actions as keys and their values
 * eg. [ 'notify', { set_tweak: 'sound', value: 'default' } ]
 *     becomes { notify: true, tweaks: { sound: 'default' } }
 * @param {array} actionlist The actions list
 *
 * @return {object} A object with key 'notify' (true or false) and an object of actions
 */
PushProcessor.actionListToActionsObject = function (actionlist) {
    var actionobj = { 'notify': false, 'tweaks': {} };
    for (var i = 0; i < actionlist.length; ++i) {
        var action = actionlist[i];
        if (action === 'notify') {
            actionobj.notify = true;
        } else if ((typeof action === 'undefined' ? 'undefined' : (0, _typeof3.default)(action)) === 'object') {
            if (action.value === undefined) {
                action.value = true;
            }
            actionobj.tweaks[action.set_tweak] = action.value;
        }
    }
    return actionobj;
};

/**
 * @typedef {Object} PushAction
 * @type {Object}
 * @property {boolean} notify Whether this event should notify the user or not.
 * @property {Object} tweaks How this event should be notified.
 * @property {boolean} tweaks.highlight Whether this event should be highlighted
 * on the UI.
 * @property {boolean} tweaks.sound Whether this notification should produce a
 * noise.
 */

/** The PushProcessor class. */
module.exports = PushProcessor;
//# sourceMappingURL=pushprocessor.js.map