<?php
declare(strict_types=1);

/**
 *
 * @copyright Copyright (c) 2018, Daniel Calviño Sánchez (danxuliu@gmail.com)
 *
 * @license GNU AGPL version 3 or any later version
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

namespace OCA\Spreed\Share\Helper;

use OCA\Spreed\Exceptions\RoomNotFoundException;
use OCA\Spreed\Manager;
use OCA\Spreed\Room;
use OCP\IUserManager;
use OCP\Share\IShare;

/**
 * Helper of OCA\Files_Sharing\Controller\DeletedShareAPIController for room
 * shares.
 *
 * The methods of this class are called from the DeletedShareAPIController to
 * perform actions or checks specific to room shares.
 */
class DeletedShareAPIController {

	/** @var string */
	private $userId;

	/** @var IUserManager */
	private $userManager;

	/** @var Manager */
	private $manager;

	/**
	 * DeletedShareAPIController constructor.
	 *
	 * @param string $UserId
	 * @param IUserManager $userManager
	 * @param Manager $manager
	 */
	public function __construct(
			string $UserId,
			IUserManager $userManager,
			Manager $manager
	) {
		$this->userId = $UserId;
		$this->userManager = $userManager;
		$this->manager = $manager;
	}

	/**
	 * Formats the specific fields of a room share for OCS output.
	 *
	 * The returned fields override those set by the main
	 * DeletedShareAPIController.
	 *
	 * @param IShare $share
	 * @return array
	 */
	public function formatShare(IShare $share): array {
		$result = [];

		try {
			$room = $this->manager->getRoomByToken($share->getSharedWith());
		} catch (RoomNotFoundException $e) {
			return $result;
		}

		// The display name of one-to-one rooms is set to the display name of
		// the other participant.
		$roomName = $room->getName();
		if ($room->getType() === Room::ONE_TO_ONE_CALL) {
			$participantsList = $room->getParticipants()['users'];
			unset($participantsList[$this->userId]);

			$roomName = $this->userManager->get(key($participantsList))->getDisplayName();
		}

		$result['share_with_displayname'] = $roomName;

		return $result;
	}

}
