/****************************************************************************
**
*W  lists.h                     GAP source                   Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file declares the functions of the generic list package.
**
**  This package provides a uniform   interface to the functions that  access
**  lists and their elements  for the other packages  in the GAP kernel.  For
**  example, 'ExecFor' can loop over the elements  in a list using the macros
**  'LEN_LIST' and 'ELM_LIST' independently of the type of the list.
*/

#ifndef libGAP_GAP_LISTS_H
#define libGAP_GAP_LISTS_H



extern  libGAP_Obj             libGAP_TYPE_LIST_EMPTY_MUTABLE;
extern  libGAP_Obj             libGAP_TYPE_LIST_EMPTY_IMMUTABLE;

extern  libGAP_Obj             libGAP_TYPE_LIST_HOM;

/****************************************************************************
**
*F  IS_LIST(<obj>)  . . . . . . . . . . . . . . . . . . . is an object a list
*V  IsListFuncs[<type>] . . . . . . . . . . . . . . . . . table for list test
**
**  'IS_LIST' returns a nonzero value if  the object <obj> is a list and zero
**  otherwise.
**
**  Note that 'IS_LIST'  is a macro,  so do not  call it with arguments  that
**  have side effects.
**
**  A package implementing  an ordinary list type <type>   must set the  flag
**  'IsListFlag[<type>]'  for  this type to '1'.   A  package  implementing a
**  vector type must set  it to '2'.  A  package implementing an matrix  type
**  must set it to '3'.
*/
#define libGAP_IS_LIST(obj)    ((*libGAP_IsListFuncs[ libGAP_TNUM_OBJ( obj ) ])( obj ))

extern  libGAP_Int             (*libGAP_IsListFuncs [libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj obj );

/****************************************************************************
**
*F  IS_SMALL_LIST(<obj>)  . . . . . . . . . . . . . is an object a small list
*V  IsSmallListFuncs[<type>] . . . . . . . . . . .. table for small list test
**
**  'IS_SMALL_LIST' returns a nonzero value if  the object <obj> is a small
**  list (meaning that its length will fit in 28 bits and zero otherwise.
**  in particular, it returns zero if <obj> is not a list at all
**
**  If <obj> is an external object it does not trigger length computation
**  in general
**  instead it will check if the object HasIsSmallList and IsSmallList, or
**  HasLength in which case Length will be checked
*/

#define libGAP_IS_SMALL_LIST(obj)    ((*libGAP_IsSmallListFuncs[ libGAP_TNUM_OBJ( obj ) ])( obj ))

extern  libGAP_Int                (*libGAP_IsSmallListFuncs [libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj obj );


/****************************************************************************
**
*F  LEN_LIST(<list>)  . . . . . . . . . . . . . . . . . . .  length of a list
*V  LenListFuncs[<type>]  . . . . . . . . . . . . . table of length functions
**
**  'LEN_LIST' returns the logical length of the list <list>  as a C integer.
**  An error is signalled if <list> is not a small list.
**
**  Note that  'LEN_LIST' is a  macro, so do  not call it with arguments that
**  have side effects.
**
**  A package  implementing a list type <type>  must  provide such a function
**  and install it in 'LenListFuncs[<type>]'.
*/
#define libGAP_LEN_LIST(list)  ((*libGAP_LenListFuncs[ libGAP_TNUM_OBJ(list) ])( list ))

extern  libGAP_Int             (*libGAP_LenListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list );


/****************************************************************************
**
*F  LENGTH(<list>)  . . . . . . . . . . . . . . . . . . .  length of a list
*V  LengthFuncs[<type>]  . . . . . . . . . . . . . table of length functions
**
**  'LENGTH' returns the logical length of the list <list>  as a GAP object
**  An error is signalled if <list> is not a list.
**
**  Note that  'LENGTH' is a  macro, so do  not call it with arguments that
**  have side effects.
**
**  A package  implementing a list type <type>  must  provide such a function
**  and install it in 'LengthFuncs[<type>]'.
*/
#define libGAP_LENGTH(list)    ((*libGAP_LengthFuncs[ libGAP_TNUM_OBJ(list) ])( list ))

extern  libGAP_Obj             (*libGAP_LengthFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list );


/****************************************************************************
**
*F  ISB_LIST(<list>,<pos>)  . . . . . . . . . .  test for element from a list
*F  ISBV_LIST(<list>,<pos>) . . . . . . . . . .  test for element from a list
*V  IsbListFuncs[<type>]  . . . . . . . . . . . . . . table of test functions
*V  IsbvListFuncs[<type>] . . . . . . . . . . . . . . table of test functions
**
**  'ISB_LIST' returns 1  if the list <list>  has an entry at  position <pos>
**  and 0 otherwise.  An error is signalled  if <list> is not  a list.  It is
**  the  responsibility of  the  caller to  ensure that  <pos> is a  positive
**  integer.
**
**  'ISBV_LIST'  does the same as  'ISB_LIST', but the caller also guarantees
**  that <list> ist a list and that <pos> is less than or equal to the length
**  of <list>.
**
**  Note that 'ISB_LIST' and 'ISBV_LIST are macros, so  do not call them with
**  arguments that have side effects.
**
**  A  package implementing a  list type <type>  must  provide a function for
**  'ISB_LIST' and install it in 'IsbListFuncs[<type>]'.
**
**  A package  implementing  a list type  <type> must  provide a function for
**  'ISBV_LIST' and  install  it in 'IsbvListFuncs[<type>]'.   This  function
**  need not  test whether  <pos> is less  than   or equal to the   length of
**  <list>.
*/
#define libGAP_ISB_LIST(list,pos) \
                        ((*libGAP_IsbListFuncs[libGAP_TNUM_OBJ(list)])(list,pos))

#define libGAP_ISBV_LIST(list,pos) \
                        ((*libGAP_IsbvListFuncs[libGAP_TNUM_OBJ(list)])(list,pos))

extern  libGAP_Int             (*libGAP_IsbListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list, libGAP_Int pos );

extern  libGAP_Int             (*libGAP_IsbvListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list, libGAP_Int pos );

extern libGAP_Int libGAP_ISBB_LIST( libGAP_Obj list, libGAP_Obj pos );


/****************************************************************************
**

*F * * * * * * * * * * * * list access functions  * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*V  Elm0ListFuncs[ <type> ] . . . . . . . . . .  table of selection functions
**
**  A package  implementing a  list type <type>  must provide  a function for
**  'ELM0_LIST' and install it in 'Elm0ListFuncs[<type>]'.
*/
extern libGAP_Obj (*libGAP_Elm0ListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list, libGAP_Int pos );


/****************************************************************************
**
*F  ELM0_LIST( <list>, <pos> )  . . . . . . . . select an element from a list
**
**  'ELM0_LIST' returns the element at the position <pos> in the list <list>,
**  or 0 if <list>  has no assigned  object at position  <pos>.  An  error is
**  signalled if <list>  is  not a list.  It   is the responsibility   of the
**  caller to ensure that <pos> is a positive integer.
**
**  Note that 'ELM0_LIST' is a  macro, so do  not call it with arguments that
**  have side effects.
*/
#define libGAP_ELM0_LIST(list,pos)     ((*libGAP_Elm0ListFuncs[libGAP_TNUM_OBJ(list)])(list,pos))


/****************************************************************************
**
*V  Elmv0ListFuncs[ <type> ]  . . . . . . . . .  table of selection functions
**
**  A package implementing  a lists type  <type> must provide a function  for
**  'ELMV0_LIST' and install it  in 'Elmv0ListFuncs[<type>]'.   This function
**  need not test   whether <pos> is less  than   or equal to  the  length of
**  <list>.
*/
extern  libGAP_Obj (*libGAP_Elm0vListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list, libGAP_Int pos );


/****************************************************************************
**
*F  ELMV0_LIST( <list>, <pos> ) . . . . . . . . select an element from a list
**
**  'ELMV0_LIST' does the same as 'ELM0_LIST', but the caller also guarantees
**  that <list> is a list and that <pos> is less than  or equal to the length
**  of <list>.
**
**  Note that 'ELMV0_LIST' is a macro, so do not call  it with arguments that
**  have side effects.
*/
#define libGAP_ELMV0_LIST(list,pos)    ((*libGAP_Elm0vListFuncs[libGAP_TNUM_OBJ(list)])(list,pos))


/****************************************************************************
**
*V  ElmListFuncs[ <type> ]  . . . . . . . . . .  table of selection functions
**
**  A package implementing a  list  type <type> must  provide a  function for
**  'ELM_LIST' and install it  in 'ElmListFuncs[<type>]'.  This function must
**  signal an error if <pos> is larger than the length of <list> or if <list>
**  has no assigned object at <pos>.
*/
extern libGAP_Obj (*libGAP_ElmListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list, libGAP_Int pos );


/****************************************************************************
**
*F  ELM_LIST( <list>, <pos> ) . . . . . . . . . select an element from a list
*F  ELMB_LIST( <list>, <pos> ) . . . . . . . . . select an element from a list
**
**  'ELM_LIST' returns the element at the position  <pos> in the list <list>.
**  An  error is signalled if  <list> is not a list,  if <pos> is larger than
**  the length of <list>, or if <list>  has no assigned  object at <pos>.  It
**  is the responsibility  of the caller to  ensure that <pos>  is a positive
**  integer.
**
**  Note that 'ELM_LIST', 'ELMV_LIST', and  'ELMW_LIST' are macros, so do not
**  call them with arguments that have side effects.
**
**  The difference between ELM_LIST and ELMB_LIST is that ELMB_LIST accepts
**  an object as the second argument
**  It is intended as an interface for access to elements of large external
**  lists, on the rare occasions when the kernel needs to do this.
*/
#define libGAP_ELM_LIST(list,pos)      ((*libGAP_ElmListFuncs[libGAP_TNUM_OBJ(list)])(list,pos))

extern libGAP_Obj libGAP_ELMB_LIST( libGAP_Obj list, libGAP_Obj pos );

/****************************************************************************
**
*F  ELM2_LIST( <list>, <pos1>, <pos2> ) . . . . select an element from a list
*/

extern libGAP_Obj libGAP_Elm2List(libGAP_Obj list, libGAP_Obj pos1, libGAP_Obj pos2);

#define libGAP_ELM2_LIST(list, pos1, pos2) libGAP_Elm2List(list, pos1, pos2)

extern void libGAP_Ass2List(libGAP_Obj list, libGAP_Obj pos1, libGAP_Obj pos2, libGAP_Obj obj);

#define libGAP_ASS2_LIST(list, pos1, pos2, obj) libGAP_Ass2List(list, pos1, pos2, obj)


/****************************************************************************
**
*V  ElmvListFuncs[ <type> ] . . . . . . . . . .  table of selection functions
**
**  A package implementing  a list  type  <type> must provide a  function for
**  'ELMV_LIST' and  install  it  in 'ElmvListFuncs[<type>]'.   This function
**  need not check that <pos> is less than or equal to  the length of <list>,
**  but it must signal an error if <list> has no assigned object at <pos>.
**
*/
extern libGAP_Obj (*libGAP_ElmvListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list, libGAP_Int pos );


/****************************************************************************
**
*F  ELMV_LIST( <list>, <pos> )  . . . . . . . . select an element from a list
**
**  'ELMV_LIST' does  the same as 'ELM_LIST', but  the caller also guarantees
**  that <list> is a list and that <pos> is less  than or equal to the length
**  of <list>.
**
**  Note that 'ELM_LIST', 'ELMV_LIST', and  'ELMW_LIST' are macros, so do not
**  call them with arguments that have side effects.
*/
#define libGAP_ELMV_LIST(list,pos)     ((*libGAP_ElmvListFuncs[libGAP_TNUM_OBJ(list)])(list,pos))


/****************************************************************************
**
*V  ElmwListFuncs[ <type> ] . . . . . . . . . .  table of selection functions
**
**  A package implementing a  list type  <type>  must provide a  function for
**  'ELMW_LIST' and install them  in 'ElmwListFuncs[<type>]'.  This  function
**  need not check that <pos> is  less than or equal  to the length of <list>
**  or that <list> has an assigned object at <pos>.
*/
extern libGAP_Obj (*libGAP_ElmwListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list, libGAP_Int pos );


/****************************************************************************
**
*F  ELMW_LIST( <list>, <pos> )  . . . . . . . . select an element from a list
**
**  'ELMW_LIST' does the same as 'ELMV_LIST', but  the caller also guarantees
**  that <list> has an assigned object at the position <pos>.
**
**  Note that 'ELM_LIST', 'ELMV_LIST', and  'ELMW_LIST' are macros, so do not
**  call them with arguments that have side effects.
*/
#define libGAP_ELMW_LIST(list,pos)     ((*libGAP_ElmwListFuncs[libGAP_TNUM_OBJ(list)])(list,pos))


/****************************************************************************
**

*V  ElmsListFuncs[ <type> ] . . . . . . . . . .  table of selection functions
**
**  A package implementing a list  type <type> must  provide such a  function
**  and install it in 'ElmsListFuncs[<type>]'.  This  function must signal an
**  error if any of the positions  is larger than the length  of <list> or if
**  <list>  has no assigned object at any of the positions  (and thus it will
**  always return a dense list).  It *must* create a new list, even if <poss>
**  is equal to  '[1..Length(<list>)]', 'EvalElmListLevel' depends on this so
**  that it can call 'ElmListLevel', which  overwrites this new list.  If the
**  result is a list of lists, then it also *must* create a new list that has
**  the same representation as a plain list.
*/
extern libGAP_Obj (*libGAP_ElmsListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list, libGAP_Obj poss );



/****************************************************************************
**
*F  ELMS_LIST(<list>,<poss>)  . . . . . . select several elements from a list
**
**  'ELMS_LIST' returns a  new list containing the  elements at the positions
**  given in the list <poss> from the list <list>.  An  error is signalled if
**  <list> is not a list, if  any of the positions  is larger than the length
**  of  <list>, or if <list> has  no assigned object at any of the positions.
**  It is the responsibility of the  caller to ensure  that <poss> is a dense
**  list of positive integers.
**
**  Note that 'ELMS_LIST' is a  macro, so do not call it with arguments  that
**  have side effects.
*/
#define libGAP_ELMS_LIST(list,poss)    ((*libGAP_ElmsListFuncs[libGAP_TNUM_OBJ(list)])(list,poss))



/****************************************************************************
**
*F  ElmsListDefault( <list>, <poss> ) . . .  default function for `ELMS_LIST'
*/
extern libGAP_Obj libGAP_ElmsListDefault (
            libGAP_Obj                 list,
            libGAP_Obj                 poss );


/****************************************************************************
**
*F  ElmsListCheck( <list>, <poss> ) . . . . . . . . .  `ELMS_LIST' with check
*/
extern libGAP_Obj libGAP_ElmsListCheck (
    libGAP_Obj                 list,
    libGAP_Obj                 poss );


/****************************************************************************
**
*F  ElmsListLevelCheck( <lists>, <poss>, <level> ) `ElmsListLevel' with check
*/
extern void libGAP_ElmsListLevelCheck (
    libGAP_Obj                 lists,
    libGAP_Obj                 poss,
    libGAP_Int                 level );


/****************************************************************************
**

*F  UNB_LIST(<list>,<pos>)  . . . . . . . . . . .  unbind element from a list
*V  UnbListFuncs[<type>]  . . . . . . . . . . . . . table of unbind functions
**
**  'UNB_LiST' unbinds the element at the position <pos> in the list  <list>.
**  Note that the unbinding may change the length of  the  representation  of
**  <list>.  An error is signalled if  <list>  is  not  a  list.  It  is  the
**  responsibility of the caller to ensure that <pos> is a positive integer.
**
**  Note that 'UNB_LIST' is a macro, so do not call it  with  arguments  that
**  have side effects.
**
**  A package implementing a list type <type> must provide  such  a  function
**  and install it in 'UnbListFuncs[<type>]'.  This function must change  the
**  representation of <list> to that of a plain list if necessary.
*/
#define libGAP_UNB_LIST(list,pos) \
                        ((*libGAP_UnbListFuncs[libGAP_TNUM_OBJ(list)])(list,pos))

extern void             (*libGAP_UnbListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list, libGAP_Int pos );

extern void libGAP_UNBB_LIST( libGAP_Obj list, libGAP_Obj pos );

extern void libGAP_UnbListDefault( libGAP_Obj list, libGAP_Int  pos );


/****************************************************************************
**
*F  ASS_LIST(<list>,<pos>,<obj>)  . . . . . . . . assign an element to a list
*V  AssListFuncs[<type>]  . . . . . . . . . . . table of assignment functions
**
**  'ASS_LIST' assigns the object <obj> to the list <list> at position <pos>.
**  Note that  the assignment may change  the length or the representation of
**  <list>.  An error   is signalled if  <list>  is not a  list.    It is the
**  responsibility of the caller to ensure that <pos>  is a positive integer,
**  and that <obj> is not 0.
**
**  Note that 'ASS_LIST' is a macro,  so do not  call it with arguments  that
**  have side effects.
**
**  A package  implementing a list type  <type> must provide  such a function
**  and   install it in  'AssListFuncs[<type>]'.   This  function must extend
**  <list> if <pos> is larger than the length of  <list> and must also change
**  the representation of <list> to that of a plain list if necessary.
*/
#define libGAP_ASS_LIST(list,pos,obj) \
                        ((*libGAP_AssListFuncs[libGAP_TNUM_OBJ(list)])(list,pos,obj))

extern  void            (*libGAP_AssListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list, libGAP_Int pos, libGAP_Obj obj );

extern void libGAP_ASSB_LIST( libGAP_Obj list, libGAP_Obj pos, libGAP_Obj obj );


/****************************************************************************
**
*F  ASSS_LIST(<list>,<poss>,<objs>) . . . . assign several elements to a list
*V  AsssListFuncs[<type>] . . . . . . . . . . .  table of assignment function
**
**  'ASSS_LIST'  assignes the objects from  the list  <objs> at the positions
**  given in the  list <poss> to the list  <list>.  Note that  the assignment
**  may  change  the length or  the  representation of  <list>.   An error is
**  signalled if  <list> is  not  a list.   It  is the  responsibility of the
**  caller to ensure that  <poss> is a dense  list  of positive  integers and
**  that <objs> is a dense list of the same length as <poss>.
**
**  Note that 'ASSS_LIST' is a macro, so do not call it with  arguments  that
**  have side effects.
**
**  A package implementing  a list type <type> must  provide  such a function
**  and install it in 'AsssListFuncs[<type>]'.  This function must extend the
**  <list> if any of the  positions is larger than the  length of <list>  and
**  must also change the representation of <list> to that  of a plain list if
**  necessary.
*/
#define libGAP_ASSS_LIST(list,poss,objs) \
                        ((*libGAP_AsssListFuncs[libGAP_TNUM_OBJ(list)])(list,poss,objs))

extern  void            (*libGAP_AsssListFuncs[libGAP_LAST_REAL_TNUM+1]) (libGAP_Obj list, libGAP_Obj poss, libGAP_Obj objs);

extern  void            libGAP_AsssListDefault (
            libGAP_Obj                 list,
            libGAP_Obj                 poss,
            libGAP_Obj                 objs );


/****************************************************************************
**
*F  AssListObject( <list>, <pos>, <obj> ) . . . . . . . assign to list object
*/
extern void libGAP_AssListObject (
    libGAP_Obj                 list,
    libGAP_Int                 pos,
    libGAP_Obj                 obj );


/****************************************************************************
**

*F  IS_DENSE_LIST(<list>) . . . . . . . . . . . . . . .  test for dense lists
*V  IsDenseListFuncs[<type>]  . . . . . .  table of dense list test functions
**
**  'IS_DENSE_LIST'  returns 1 if the   list <list> is a   dense  list and  0
**  otherwise, i.e., if either <list> is not a list, or if it is not dense.
**
**  Note that  'IS_DENSE_LIST' is a macro, so  do not call it  with arguments
**  that have side effects.
**
**  A package  implementing a list type  <type> must provide such  a function
**  and  install it in  'IsDenseListFuncs[<type>]'.   This function must loop
**  over the list and test for holes, unless  the type of the list guarantees
**  already that the list is dense (e.g. for sets).
*/
#define libGAP_IS_DENSE_LIST(list) \
                        ((*libGAP_IsDenseListFuncs[libGAP_TNUM_OBJ(list)])(list))

extern  libGAP_Int             (*libGAP_IsDenseListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list );

extern  libGAP_Int             libGAP_IsDenseListDefault (
            libGAP_Obj                 list );


/****************************************************************************
**
*F  IS_HOMOG_LIST(<list>) . . . . . . . . . . . .  test for homogeneous lists
*V  IsHomogListFuncs[<type>]  . . .  table of homogeneous list test functions
**
**  'IS_HOMOG_LIST' returns 1 if  the list <list>  is a  homogeneous list and
**  0 otherwise, i.e., if either <list> is not  a  list,  or  if  it  is  not
**  homogeneous.
**
**  'IS_HOMOG_LIST' is a macro, so do not call it with  arguments  that  have
**  side effects.
**
**  A  package implementing a list  type <type> must  provide such a function
**  and install  it  in 'IsHomogListFuncs[<type>]'.  This function  must loop
**  over the list   and test whether all  elements  lie in  the  same family,
**  unless  the type  of   the list  guarantees    already that the  list  is
**  homogeneous (e.g. for sets).
*/
#define libGAP_IS_HOMOG_LIST(list) \
                        ((*libGAP_IsHomogListFuncs[libGAP_TNUM_OBJ(list)])(list))

extern  libGAP_Int             (*libGAP_IsHomogListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list );

extern  libGAP_Int             libGAP_IsHomogListDefault (
            libGAP_Obj                 list );


/****************************************************************************
**
*F  IS_TABLE_LIST(<list>) . . . . . . . . . . . . . . .  test for table lists
*V  IsTableListFuncs[<type>]  . . . . . .  table of table list test functions
**
**  'IS_TABLE_LIST'  returns  1 if  the  list  <list>  is  a  table, i.e.,  a
**  homogeneous list of homogeneous lists of equal length, and 0 otherwise.
**
**  'IS_TABLE_LIST' is a macro, so do not call it with  arguments  that  have
**  side effects.
**
**  A  package implementing a list  type <type> must  provide such a function
**  and install it in  'IsTableListFuncs[<type>]'.   This function must  loop
**  over the list and test whether  all elements lie  in the same family, are
**  homogenous lists, and have  the same length, unless the  type of the list
**  guarantees already that the list has this property.
*/
#define libGAP_IS_TABLE_LIST(list) \
                        ((*libGAP_IsTableListFuncs[libGAP_TNUM_OBJ(list)])(list))

extern  libGAP_Int             (*libGAP_IsTableListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list );

extern  libGAP_Int             libGAP_IsTableListDefault (
            libGAP_Obj                 list );


/****************************************************************************
**
*F  IS_SSORT_LIST(<list>) . . . . . . . . . .  test for strictly sorted lists
*V  IsSSortListFuncs[<type>]  .  table of strictly sorted list test functions
**
**  'IS_SSORT_LIST' returns 2 if the list <list> is  a strictly  sorted  list
**  and 0 otherwise,  i.e., if either <list>  is not a list,  or if it is not
**  strictly sorted.
**
**  'IS_SSORT_LIST' is a macro, so do not call it  with arguments  that  have
**  side effects.
**
**  A  package implementing a  list type <type>  must provide such a function
**  and install it  in  'IsSSortListFuncs[<type>]'.  This function must  loop
**  over the list and compare each element with the next one, unless the type
**  of the list guarantees already that the list is strictly sorted.
*/
#define libGAP_IS_SSORT_LIST(list) \
                        ((*libGAP_IsSSortListFuncs[libGAP_TNUM_OBJ(list)])(list))

extern  libGAP_Int             (*libGAP_IsSSortListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list );

extern  libGAP_Int             libGAP_IsSSortListDefault (
            libGAP_Obj                 list );


/****************************************************************************
**
*F  IsSSortListProp
*/
extern libGAP_Obj libGAP_IsSSortListProp;


/****************************************************************************
**
*F  IsNSortListProp
*/
extern libGAP_Obj libGAP_IsNSortListProp;


/****************************************************************************
**
*F  IS_POSS_LIST(<list>)  . . . . . . . . . . . . .  test for positions lists
*V  IsPossListFuncs[<type>] . . . . . . table of positions list test function
**
**  'IS_POSS_LIST' returns  1 if the list  <list> is  a dense list containing
**  only positive  integers and 0 otherwise, i.e.,  if either <list> is not a
**  list, or if it is not dense,  or if it contains  an element that is not a
**  positive integer.
**
**  Note that  'IS_POSS_LIST' is a macro,  so  do not  call it with arguments
**  that have side effects.
**
**  A package  implementing a list type  <type> must provide such  a function
**  and install  it  in 'IsPossListFuncs[<type>]'.   This function  must loop
**  over the list  and  test for holes   and elements that are  not  positive
**  integers, unless the type of the list guarantees already that the list is
**  acceptable (e.g. a range with positive <low> and <high> values).
*/
#define libGAP_IS_POSS_LIST(list) \
                        ((*libGAP_IsPossListFuncs[libGAP_TNUM_OBJ(list)])(list))

extern  libGAP_Int             (*libGAP_IsPossListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list );

extern  libGAP_Int             libGAP_IsPossListDefault (
            libGAP_Obj                 list );


/****************************************************************************
**
*F  POS_LIST(<list>,<obj>,<start>)  . . . . . . . . find an element in a list
*V  PosListFuncs[<type>]  . . . . . . . . . . .  table of searching functions
**
**  'POS_LIST' returns  the  position of the  first  occurrence of  the object
**  <obj>,  which may be an object of any type, in the  list <list> after the
**  position  <start> as GAP Integer.  Fail is returned if  <obj> is not in the
**  list after <start>.  An error is signalled if <list> is not a list.
**
**  Note that 'POS_LIST'  is a macro,  so do  not call it with arguments that
**  have side effects.
**
**  A package implementing a list  type <type> must  provide such a  function
**  and install it in 'PosListFuncs[<type>]'.
*/
#define libGAP_POS_LIST(list,obj,start) \
                        ((*libGAP_PosListFuncs[libGAP_TNUM_OBJ(list)])(list,obj,start))

extern  libGAP_Obj             (*libGAP_PosListFuncs[libGAP_LAST_REAL_TNUM+1]) (libGAP_Obj list, libGAP_Obj obj, libGAP_Obj start);

extern  libGAP_Obj             libGAP_PosListDefault (
            libGAP_Obj                 list,
            libGAP_Obj                 obj,
            libGAP_Obj                 start );


/****************************************************************************
**
*F  ElmListLevel(<lists>,<pos>,<level>) . . . . . . . . . . . . . . . . . . .
*F  . . . . . . . . . . . . .  select an element of several lists in parallel
**
**  'ElmListLevel' assigns to '<lists>[<p_1>][<p_2>]...[<p_level>]' the value
**  '<lists>[<p_1>][<p_2>]...[<p_level>][<pos>]' for all  appropriate  tuples
**  of positions <p_1>,<p_2>,...,<p_level>.  An error is signalled if for any
**  tuple of positions  '<list> = <lists>[<p_1>][<p_2>]...[<p_level>]' is not
**  a list, <pos>  is larger  than  the length of <list>,   or <list> has  no
**  assigned object  at <pos>.   It is the   responsibility of the  caller to
**  ensure that <pos> is a positive integer.
**
**  It  is also  the responsibility of   the caller  to ensure  that <lists>,
**  '<lists>[<p_1>]', ...,   '<lists>[<p_1>][<p_2>]...[<p_level-1>]'  are all
**  dense lists with the same representation as plain lists.  Usually <lists>
**  is the   result of <level>  nested applications   of 'ELMS_LIST',  so  we
**  require 'ELMS_LIST' (resp.  the   functions implementing 'ELMS_LIST')  to
**  satisfy this requirements.
*/
extern  void            libGAP_ElmListLevel (
            libGAP_Obj                 lists,
            libGAP_Obj                 pos,
            libGAP_Int                 level );


/****************************************************************************
**
*F  ElmsListLevel(<lists>,<poss>,<level>) . . . . . . . . . . . . . . . . . .
*F  . . . . . . . . . .  select several elements of several lists in parallel
**
**  'ElmsListLevel'    assigns  to '<lists>[<p_1>][<p_2>]...[<p_level>]'  the
**  objects '<lists>[<p_1>][<p_2>]...[<p_level>]{<poss>}' for all appropriate
**  tuples of positions <p_1>,<p_2>,...,<p_level>.   An error is signalled if
**  for any tuple of positions '<list> = <lists>[<p_1>][<p_2>]...[<p_level>]'
**  is not a list, any  of the positions of  <poss> is larger than the length
**  of <list>, or <list> has  no assigned object at any of the positions.  It
**  is also the responsibility of the caller to ensure that <poss> is a dense
**  list of positive integers.
**
**  It  is  also the  responsibility  of the  caller  to ensure that <lists>,
**  '<lists>[<p_1>]',  ...,   '<lists>[<p_1>][<p_2>]...[<p_level-1>]' are all
**  dense lists with the same representation as plain lists.  Usually <lists>
**  is the result   of  <level> nested applications   of 'ELMS_LIST',  so  we
**  require 'ELMS_LIST' (resp.  the   functions implementing 'ELMS_LIST')  to
**  satisfy this requirements.
*/
extern  void            libGAP_ElmsListLevel (
            libGAP_Obj                 lists,
            libGAP_Obj                 poss,
            libGAP_Int                 level );


/****************************************************************************
**
*F  AssListLevel(<lists>,<pos>,<objs>,<level>)  . . . . . . . . . . . . . . .
*F  . . . . . . . . . . . . .  assign an element to several lists in parallel
**
**  'AssListLevel' assigns to    '<lists>[<p_1>][<p_2>]...[<p_level>][<pos>]'
**  the value '<objs>[<p_1>][<p_2>]...[<p_level>]' for all appropriate tuples
**  of positions <p_1>,<p_2>,...,<p_level>.  An error is signalled if for any
**  tuple of positions '<list>  = <lists>[<p_1>][<p_2>]...[<p_level>]' is not
**  a list, '<obj> =  <objs>[<p_1>][<p_2>]...[<p_i-1>]' is not a  dense list,
**  or  <obj> has not  the same length as '<list>[<p_1>][<p_2>]...[<p_i-1>]'.
**  It is the responsibility of the caller to ensure that <pos> is a positive
**  integer.
**
**  It is  also  the responsibility of   the caller  to  ensure that <lists>,
**  '<lists>[<p_1>]',   ...,  '<lists>[<p_1>][<p_2>]...[<p_level-1>]' are all
**  dense lists with the same representation as plain lists.  Usually <lists>
**  is the  result of   <level> nested applications  of  'ELMS_LIST',   so we
**  require  'ELMS_LIST'  (resp.  the functions implementing  'ELMS_LIST') to
**  satisfy this requirements.
*/
extern  void            libGAP_AssListLevel (
            libGAP_Obj                 lists,
            libGAP_Obj                 pos,
            libGAP_Obj                 objs,
            libGAP_Int                 level );


/****************************************************************************
**
*F  AsssListLevel(<lists>,<poss>,<objs>,<level>)  . . . . . . . . . . . . . .
*F  . . . . . . . . . .  assign several elements to several lists in parallel
**
**  'AsssListLevel'  assigns to '<lists>[<p_1>][<p_2>]...[<p_level>]{<poss>}'
**  the  objects  '<objs>[<p_1>][<p_2>]...[<p_level>]' for  all   appropriate
**  tuples of positions <p_1>,<p_2>,...,<p_level>.   An error is signalled if
**  for any tuple of positions '<list> = <lists>[<p_1>][<p_2>]...[<p_level>]'
**  is not a list, '<obj> = <objs>[<p_1>][<p_2>]...[<p_i-1>]'  is not a dense
**  list, <obj> has not the same length as '<list>[<p_1>][<p_2>]...[<p_i-1>]'
**  or  '<objs>[<p_1>][<p_2>]...[<p_level>]' is not a  dense list of the same
**  length as <poss>.  It is the responsibility  of the caller to ensure that
**  <poss> is a dense list of positive integers.
**
**  It is also the   responsibility of the caller   to ensure that   <lists>,
**  '<lists>[<p_1>]',    ..., '<lists>[<p_1>][<p_2>]...[<p_level-1>]' are all
**  dense lists with the same representation as plain lists.  Usually <lists>
**  is the   result  of <level> nested applications   of   'ELMS_LIST', so we
**  require 'ELMS_LIST' (resp.   the functions  implementing 'ELMS_LIST')  to
**  satisfy this requirements.
*/
extern  void            libGAP_AsssListLevel (
            libGAP_Obj                 lists,
            libGAP_Obj                 poss,
            libGAP_Obj                 objs,
            libGAP_Int                 lev );


/****************************************************************************
**
*F  PLAIN_LIST(<list>)  . . . . . . . . . . .  convert a list to a plain list
*V  PlainListFuncs[<type>]  . . . . . . . . . . table of conversion functions
**
**  'PLAIN_LIST' changes  the representation of the  list <list> to that of a
**  plain list Note that the type of <list> need not be 'T_PLIST' afterwards,
**  it could also be 'T_SET' or 'T_VECTOR'.  An  error is signalled if <list>
**  is not a list.
**
**  Note that 'PLAIN_LIST' is a macro, so do not call  it with arguments that
**  have side effects.
**
**  A package implementing a  list type <type>  must provide such  a function
**  and install it in 'PlainListFuncs[<type>]'.
*/
#define libGAP_PLAIN_LIST(list) \
                        ((*libGAP_PlainListFuncs[libGAP_TNUM_OBJ(list)])(list))

extern  void            (*libGAP_PlainListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list );


/****************************************************************************
**
*F  TYPES_LIST_FAM(<fam>) . . . . . . .  list of types of lists over a family
*/
extern  libGAP_Obj             libGAP_TYPES_LIST_FAM (
            libGAP_Obj                 fam );


/****************************************************************************
**

*V  SetFiltListTNums[ <tnum> ][ <fnum> ]  . . . . . new tnum after filter set
**
**  If a list  with type number <tnum>  gains  the filter  with filter number
**  <fnum>, then the new type number is stored in:
**
**  `SetFiltListTNums[<tnum>][<fnum>]'
**
**  The macro  `SET_FILT_LIST' is  used  to  set  the filter  for a  list  by
**  changing its type number.
*/
extern libGAP_UInt libGAP_SetFiltListTNums [ libGAP_LAST_REAL_TNUM ] [ libGAP_LAST_FN + 1 ];


/****************************************************************************
**
*F  SET_FILT_LIST( <list>, <fnum> ) . . . . . . . . . . . . . .  set a filter
*/
#define libGAP_SET_FILT_LIST(list,fn) \
  do { \
    libGAP_UInt     new; \
    new = libGAP_SetFiltListTNums[libGAP_TNUM_OBJ(list)][fn]; \
    if ( new != (libGAP_UInt)-1 ) \
      libGAP_RetypeBagIfWritable( list, new ); \
     else { \
      libGAP_Pr( "#E  SET_FILT_LIST[%s][%d] in ", (libGAP_Int)libGAP_TNAM_OBJ(list), fn ); \
      libGAP_Pr( "%s line %d\n", (libGAP_Int)__FILE__, (libGAP_Int)__LINE__); \
      }  \
  } while (0)

/****************************************************************************
**
*F  FuncSET_FILTER_LIST( <self>, <list>, <filter> ) . . . . . . .  set filter
*/
extern libGAP_Obj libGAP_FuncSET_FILTER_LIST ( libGAP_Obj self, libGAP_Obj list, libGAP_Obj filter );

/****************************************************************************
**
*V  ResetFiltListTNums[ <tnum> ][ <fnum> ]  . . . new tnum after filter reset
**
**  If a list  with type number <tnum>  loses  the filter  with filter number
**  <fnum>, then the new type number is stored in:
**
**  `ResetFiltListTNums[<tnum>][<fnum>]'
**
**  The macro `RESET_FILT_LIST' is used  to  set  the filter  for a  list  by
**  changing its type number.
*/
extern libGAP_UInt libGAP_ResetFiltListTNums [ libGAP_LAST_REAL_TNUM ] [ libGAP_LAST_FN + 1 ];


/****************************************************************************
**
*F  RESET_FILT_LIST( <list>, <fnum> ) . . . . . . . . . . . .  reset a filter
*/
#define libGAP_RESET_FILT_LIST(list,fn) \
  do { \
    libGAP_UInt     new; \
    new = libGAP_ResetFiltListTNums[libGAP_TNUM_OBJ(list)][fn]; \
    if ( new != (libGAP_UInt) -1 ) \
      libGAP_RetypeBag( list, new ); \
    else  { \
      libGAP_Pr( "#E  RESET_FILT_LIST[%s][%d] in ", (libGAP_Int)libGAP_TNAM_OBJ(list), fn ); \
      libGAP_Pr( "%s line %d\n", (libGAP_Int)__FILE__, (libGAP_Int)__LINE__); \
      }  \
  } while (0)


/****************************************************************************
**
*V  HasFiltListTNums[ <tnum> ][ <fnum> ]  . . . . . . . . . . . .  has filter
*/
extern libGAP_Int libGAP_HasFiltListTNums [ libGAP_LAST_REAL_TNUM ] [ libGAP_LAST_FN + 1 ];


/****************************************************************************
**
*F  HAS_FILT_LIST( <list>, <fnum> ) . . . . . . . . . . . . . . .  has filter
*/
#define libGAP_HAS_FILT_LIST(list,fn)   libGAP_HasFiltListTNums[libGAP_TNUM_OBJ(list)][fn]


/****************************************************************************
**
*V  ClearFiltsTNums[ <tnum> ] . clear all list filters except `FN_IS_MUTABLE'
**
**  The type  number without any  known properties  of a  list of type number
**  <tnum> is stored in:
**
**  `ClearPropsTNums[<tnum>]'
**
**  The macro `CLEAR_PROPS_LIST' is used to clear all properties of a list.
*/
extern libGAP_UInt libGAP_ClearFiltsTNums [ libGAP_LAST_REAL_TNUM ];


/****************************************************************************
**
*F  CLEAR_FILTS_LIST( <list> )  . . . . . . . . . . . . . .  clear properties
*/
#define libGAP_CLEAR_FILTS_LIST(list) \
  do { \
    libGAP_UInt     new; \
    new = libGAP_ClearFiltsTNums[libGAP_TNUM_OBJ(list)]; \
    if ( new > 0 ) \
      libGAP_RetypeBag( list, new ); \
/*    else if ( new < 0 ) { \
      Pr( "#E  CLEAR_FILTS_LIST[%s] in ", (Int)TNAM_OBJ(list), 0 ); \
      Pr( "%s line %d\n", (Int)__FILE__, (Int)__LINE__); \
      } */ \
  } while (0)

/****************************************************************************
**
*F  MARK_LIST( <list>, <what> ) . . . . . . . . . . . . . . . . . . mark list
*/
#define libGAP_MARK_LIST( list, what ) \
    libGAP_RetypeBag( list, libGAP_TNUM_OBJ(list) + what );


/****************************************************************************
**
*F  UNMARK_LIST( <list>, <what> ) . . . . . . . . . . . . . . . . unmark list
*/
#define libGAP_UNMARK_LIST( list, what ) \
    libGAP_RetypeBag( list, libGAP_TNUM_OBJ(list) - what );


/****************************************************************************
**

*F * * * * * * * * * * * functions with checking  * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*F  AsssListCheck( <list>, <poss>, <rhss> ) . . . . . . . . . . . . ASSS_LIST
*/
extern void libGAP_AsssListCheck (
    libGAP_Obj                 list,
    libGAP_Obj                 poss,
    libGAP_Obj                 rhss );


/****************************************************************************
**
*F  AsssPosObjCheck( <list>, <poss>, <rhss> ) . . . . . . . . . . . ASSS_LIST
*/
extern void libGAP_AsssPosObjCheck (
    libGAP_Obj                 list,
    libGAP_Obj                 poss,
    libGAP_Obj                 rhss );


/****************************************************************************
**
*F  AsssListLevelCheck( <lists>, <poss>, <rhss>, <level> )  . . AsssListLevel
*/
extern void libGAP_AsssListLevelCheck (
    libGAP_Obj                 lists,
    libGAP_Obj                 poss,
    libGAP_Obj                 rhss,
    libGAP_Int                 level );


/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*F  InitInfoLists() . . . . . . . . . . . . . . . . . table of init functions
*/
libGAP_StructInitInfo * libGAP_InitInfoLists ( void );


#endif // GAP_LISTS_H

/****************************************************************************
**

*E  lists.h . . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
