"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const proxy_1 = require("../utils/proxy");
class HttpWrapper {
    constructor(metricFeature) {
        this.metricFeature = metricFeature;
    }
    init(opts, http) {
        let glMeter;
        let glLatency;
        if (!opts.name)
            opts.name = 'http';
        glMeter = this.metricFeature.meter({
            name: opts.name.toUpperCase(),
            samples: 60,
            type: 'internal/http/builtin/reqs',
            unit: 'req/min'
        });
        glLatency = this.metricFeature.histogram({
            measurement: 'mean',
            name: `pmx:${opts.name}:latency`,
            type: 'internal/http/builtin/latency',
            unit: 'ms'
        });
        const ignoreRoutes = function (url) {
            for (let i = 0; i < opts.ignore_routes.length; ++i) {
                if (url.match(opts.ignore_routes[i]) !== null) {
                    return true;
                }
            }
            return false;
        };
        proxy_1.default.wrap(http.Server.prototype, ['on', 'addListener'], function (addListener) {
            return function (event, listener) {
                const self = this;
                const overloadedFunction = function (request, response) {
                    glMeter.mark();
                    let httpStart = {
                        url: request.url,
                        start: Date.now()
                    };
                    response.once('finish', function () {
                        if (!ignoreRoutes(httpStart.url)) {
                            glLatency.update(Date.now() - httpStart.start);
                        }
                    });
                };
                if (!(event === 'request' && typeof listener === 'function')) {
                    return addListener.apply(self, arguments);
                }
                if (self.__overloaded !== true) {
                    self.on('removeListener', function onRemoveListener() {
                        setTimeout(function () {
                            if (self.listeners('request').length === 1) {
                                self.removeListener('request', overloadedFunction);
                                self.removeListener('removeListener', onRemoveListener);
                                self.__overloaded = false;
                            }
                        }, 200);
                    });
                    addListener.call(self, event, overloadedFunction);
                    self.__overloaded = true;
                }
                return addListener.apply(self, arguments);
            };
        });
        return http;
    }
}
exports.default = HttpWrapper;
//# sourceMappingURL=data:application/json;base64,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