"use strict";

var _getIterator2 = require("babel-runtime/core-js/get-iterator");

var _getIterator3 = _interopRequireDefault(_getIterator2);

var _keys = require("babel-runtime/core-js/object/keys");

var _keys2 = _interopRequireDefault(_keys);

var _regenerator = require("babel-runtime/regenerator");

var _regenerator2 = _interopRequireDefault(_regenerator);

var _bluebird = require("bluebird");

var _initialiseAccount = function () {
    var _ref2 = (0, _bluebird.coroutine)(_regenerator2.default.mark(function _callee2(sessionStore, cryptoStore, pickleKey, account) {
        return _regenerator2.default.wrap(function _callee2$(_context2) {
            while (1) {
                switch (_context2.prev = _context2.next) {
                    case 0:
                        _context2.next = 2;
                        return (0, _bluebird.resolve)(cryptoStore.doTxn('readwrite', [_indexeddbCryptoStore2.default.STORE_ACCOUNT], function (txn) {
                            cryptoStore.getAccount(txn, function (pickledAccount) {
                                if (pickledAccount !== null) {
                                    account.unpickle(pickleKey, pickledAccount);
                                } else {
                                    account.create();
                                    pickledAccount = account.pickle(pickleKey);
                                    cryptoStore.storeAccount(txn, pickledAccount);
                                }
                            });
                        }));

                    case 2:
                    case "end":
                        return _context2.stop();
                }
            }
        }, _callee2, this);
    }));

    return function _initialiseAccount(_x, _x2, _x3, _x4) {
        return _ref2.apply(this, arguments);
    };
}();

/**
 * @return {array} The version of Olm.
 */


var _indexeddbCryptoStore = require("./store/indexeddb-crypto-store");

var _indexeddbCryptoStore2 = _interopRequireDefault(_indexeddbCryptoStore);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * olm.js wrapper
 *
 * @module crypto/OlmDevice
 */
var Olm = global.Olm; /*
                      Copyright 2016 OpenMarket Ltd
                      Copyright 2017 New Vector Ltd
                      
                      Licensed under the Apache License, Version 2.0 (the "License");
                      you may not use this file except in compliance with the License.
                      You may obtain a copy of the License at
                      
                          http://www.apache.org/licenses/LICENSE-2.0
                      
                      Unless required by applicable law or agreed to in writing, software
                      distributed under the License is distributed on an "AS IS" BASIS,
                      WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
                      See the License for the specific language governing permissions and
                      limitations under the License.
                      */

if (!Olm) {
    throw new Error("global.Olm is not defined");
}

// The maximum size of an event is 65K, and we base64 the content, so this is a
// reasonable approximation to the biggest plaintext we can encrypt.
var MAX_PLAINTEXT_LENGTH = 65536 * 3 / 4;

function checkPayloadLength(payloadString) {
    if (payloadString === undefined) {
        throw new Error("payloadString undefined");
    }

    if (payloadString.length > MAX_PLAINTEXT_LENGTH) {
        // might as well fail early here rather than letting the olm library throw
        // a cryptic memory allocation error.
        //
        // Note that even if we manage to do the encryption, the message send may fail,
        // because by the time we've wrapped the ciphertext in the event object, it may
        // exceed 65K. But at least we won't just fail with "abort()" in that case.
        throw new Error("Message too long (" + payloadString.length + " bytes). " + "The maximum for an encrypted message is " + MAX_PLAINTEXT_LENGTH + " bytes.");
    }
}

/**
 * The type of object we use for importing and exporting megolm session data.
 *
 * @typedef {Object} module:crypto/OlmDevice.MegolmSessionData
 * @property {String} sender_key  Sender's Curve25519 device key
 * @property {String[]} forwarding_curve25519_key_chain Devices which forwarded
 *     this session to us (normally empty).
 * @property {Object<string, string>} sender_claimed_keys Other keys the sender claims.
 * @property {String} room_id     Room this session is used in
 * @property {String} session_id  Unique id for the session
 * @property {String} session_key Base64'ed key data
 */

/**
 * Manages the olm cryptography functions. Each OlmDevice has a single
 * OlmAccount and a number of OlmSessions.
 *
 * Accounts and sessions are kept pickled in a sessionStore.
 *
 * @constructor
 * @alias module:crypto/OlmDevice
 *
 * @param {Object} sessionStore A store to be used for data in end-to-end
 *    crypto. This is deprecated and being replaced by cryptoStore.
 * @param {Object} cryptoStore A store for crypto data
 *
 * @property {string} deviceCurve25519Key   Curve25519 key for the account
 * @property {string} deviceEd25519Key      Ed25519 key for the account
 */
function OlmDevice(sessionStore, cryptoStore) {
    this._sessionStore = sessionStore;
    this._cryptoStore = cryptoStore;
    this._pickleKey = "DEFAULT_KEY";

    // don't know these until we load the account from storage in init()
    this.deviceCurve25519Key = null;
    this.deviceEd25519Key = null;
    this._maxOneTimeKeys = null;

    // we don't bother stashing outboundgroupsessions in the sessionstore -
    // instead we keep them here.
    this._outboundGroupSessionStore = {};

    // Store a set of decrypted message indexes for each group session.
    // This partially mitigates a replay attack where a MITM resends a group
    // message into the room.
    //
    // When we decrypt a message and the message index matches a previously
    // decrypted message, one possible cause of that is that we are decrypting
    // the same event, and may not indicate an actual replay attack.  For
    // example, this could happen if we receive events, forget about them, and
    // then re-fetch them when we backfill.  So we store the event ID and
    // timestamp corresponding to each message index when we first decrypt it,
    // and compare these against the event ID and timestamp every time we use
    // that same index.  If they match, then we're probably decrypting the same
    // event and we don't consider it a replay attack.
    //
    // Keys are strings of form "<senderKey>|<session_id>|<message_index>"
    // Values are objects of the form "{id: <event id>, timestamp: <ts>}"
    this._inboundGroupSessionMessageIndexes = {};
}

/**
 * Initialise the OlmAccount. This must be called before any other operations
 * on the OlmDevice.
 *
 * Attempts to load the OlmAccount from the crypto store, or creates one if none is
 * found.
 *
 * Reads the device keys from the OlmAccount object.
 */
OlmDevice.prototype.init = (0, _bluebird.coroutine)(_regenerator2.default.mark(function _callee() {
    var e2eKeys, account;
    return _regenerator2.default.wrap(function _callee$(_context) {
        while (1) {
            switch (_context.prev = _context.next) {
                case 0:
                    _context.next = 2;
                    return (0, _bluebird.resolve)(this._migrateFromSessionStore());

                case 2:
                    e2eKeys = void 0;
                    account = new Olm.Account();
                    _context.prev = 4;
                    _context.next = 7;
                    return (0, _bluebird.resolve)(_initialiseAccount(this._sessionStore, this._cryptoStore, this._pickleKey, account));

                case 7:
                    e2eKeys = JSON.parse(account.identity_keys());

                    this._maxOneTimeKeys = account.max_number_of_one_time_keys();

                case 9:
                    _context.prev = 9;

                    account.free();
                    return _context.finish(9);

                case 12:

                    this.deviceCurve25519Key = e2eKeys.curve25519;
                    this.deviceEd25519Key = e2eKeys.ed25519;

                case 14:
                case "end":
                    return _context.stop();
            }
        }
    }, _callee, this, [[4,, 9, 12]]);
}));

OlmDevice.getOlmVersion = function () {
    return Olm.get_library_version();
};

OlmDevice.prototype._migrateFromSessionStore = (0, _bluebird.coroutine)(_regenerator2.default.mark(function _callee4() {
    var _this = this;

    var sessions, ibGroupSessions;
    return _regenerator2.default.wrap(function _callee4$(_context4) {
        while (1) {
            switch (_context4.prev = _context4.next) {
                case 0:
                    _context4.next = 2;
                    return (0, _bluebird.resolve)(this._cryptoStore.doTxn('readwrite', [_indexeddbCryptoStore2.default.STORE_ACCOUNT], function (txn) {
                        _this._cryptoStore.getAccount(txn, function (pickledAccount) {
                            if (pickledAccount === null) {
                                // Migrate from sessionStore
                                pickledAccount = _this._sessionStore.getEndToEndAccount();
                                if (pickledAccount !== null) {
                                    console.log("Migrating account from session store");
                                    _this._cryptoStore.storeAccount(txn, pickledAccount);
                                }
                            }
                        });
                    }));

                case 2:

                    // remove the old account now the transaction has completed. Either we've
                    // migrated it or decided not to, either way we want to blow away the old data.
                    this._sessionStore.removeEndToEndAccount();

                    // sessions
                    sessions = this._sessionStore.getAllEndToEndSessions();

                    if (!((0, _keys2.default)(sessions).length > 0)) {
                        _context4.next = 8;
                        break;
                    }

                    _context4.next = 7;
                    return (0, _bluebird.resolve)(this._cryptoStore.doTxn('readwrite', [_indexeddbCryptoStore2.default.STORE_SESSIONS], function (txn) {
                        // Don't migrate sessions from localstorage if we already have sessions
                        // in indexeddb, since this means we've already migrated and an old version
                        // has run against the same localstorage and created some spurious sessions.
                        _this._cryptoStore.countEndToEndSessions(txn, function (count) {
                            if (count) {
                                console.log("Crypto store already has sessions: not migrating");
                                return;
                            }
                            var numSessions = 0;
                            var _iteratorNormalCompletion = true;
                            var _didIteratorError = false;
                            var _iteratorError = undefined;

                            try {
                                for (var _iterator = (0, _getIterator3.default)((0, _keys2.default)(sessions)), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
                                    var deviceKey = _step.value;
                                    var _iteratorNormalCompletion2 = true;
                                    var _didIteratorError2 = false;
                                    var _iteratorError2 = undefined;

                                    try {
                                        for (var _iterator2 = (0, _getIterator3.default)((0, _keys2.default)(sessions[deviceKey])), _step2; !(_iteratorNormalCompletion2 = (_step2 = _iterator2.next()).done); _iteratorNormalCompletion2 = true) {
                                            var sessionId = _step2.value;

                                            numSessions++;
                                            _this._cryptoStore.storeEndToEndSession(deviceKey, sessionId, sessions[deviceKey][sessionId], txn);
                                        }
                                    } catch (err) {
                                        _didIteratorError2 = true;
                                        _iteratorError2 = err;
                                    } finally {
                                        try {
                                            if (!_iteratorNormalCompletion2 && _iterator2.return) {
                                                _iterator2.return();
                                            }
                                        } finally {
                                            if (_didIteratorError2) {
                                                throw _iteratorError2;
                                            }
                                        }
                                    }
                                }
                            } catch (err) {
                                _didIteratorError = true;
                                _iteratorError = err;
                            } finally {
                                try {
                                    if (!_iteratorNormalCompletion && _iterator.return) {
                                        _iterator.return();
                                    }
                                } finally {
                                    if (_didIteratorError) {
                                        throw _iteratorError;
                                    }
                                }
                            }

                            console.log("Migrating " + numSessions + " sessions from session store");
                        });
                    }));

                case 7:

                    this._sessionStore.removeAllEndToEndSessions();

                case 8:

                    // inbound group sessions
                    ibGroupSessions = this._sessionStore.getAllEndToEndInboundGroupSessionKeys();

                    if (!((0, _keys2.default)(ibGroupSessions).length > 0)) {
                        _context4.next = 11;
                        break;
                    }

                    return _context4.delegateYield(_regenerator2.default.mark(function _callee3() {
                        var numIbSessions;
                        return _regenerator2.default.wrap(function _callee3$(_context3) {
                            while (1) {
                                switch (_context3.prev = _context3.next) {
                                    case 0:
                                        numIbSessions = 0;
                                        _context3.next = 3;
                                        return (0, _bluebird.resolve)(_this._cryptoStore.doTxn('readwrite', [_indexeddbCryptoStore2.default.STORE_INBOUND_GROUP_SESSIONS], function (txn) {
                                            // We always migrate inbound group sessions, even if we already have some
                                            // in the new store. They should be be safe to migrate.
                                            var _iteratorNormalCompletion3 = true;
                                            var _didIteratorError3 = false;
                                            var _iteratorError3 = undefined;

                                            try {
                                                for (var _iterator3 = (0, _getIterator3.default)(ibGroupSessions), _step3; !(_iteratorNormalCompletion3 = (_step3 = _iterator3.next()).done); _iteratorNormalCompletion3 = true) {
                                                    var s = _step3.value;

                                                    try {
                                                        _this._cryptoStore.addEndToEndInboundGroupSession(s.senderKey, s.sessionId, JSON.parse(_this._sessionStore.getEndToEndInboundGroupSession(s.senderKey, s.sessionId)), txn);
                                                    } catch (e) {
                                                        console.warn("Failed to migrate session " + s.senderKey + "/" + s.sessionId + ": " + e.stack || e);
                                                    }
                                                    ++numIbSessions;
                                                }
                                            } catch (err) {
                                                _didIteratorError3 = true;
                                                _iteratorError3 = err;
                                            } finally {
                                                try {
                                                    if (!_iteratorNormalCompletion3 && _iterator3.return) {
                                                        _iterator3.return();
                                                    }
                                                } finally {
                                                    if (_didIteratorError3) {
                                                        throw _iteratorError3;
                                                    }
                                                }
                                            }

                                            console.log("Migrated " + numIbSessions + " inbound group sessions from session store");
                                        }));

                                    case 3:
                                        _this._sessionStore.removeAllEndToEndInboundGroupSessions();

                                    case 4:
                                    case "end":
                                        return _context3.stop();
                                }
                            }
                        }, _callee3, _this);
                    })(), "t0", 11);

                case 11:
                case "end":
                    return _context4.stop();
            }
        }
    }, _callee4, this);
}));

/**
 * extract our OlmAccount from the crypto store and call the given function
 * with the account object
 * The `account` object is useable only within the callback passed to this
 * function and will be freed as soon the callback returns. It is *not*
 * useable for the rest of the lifetime of the transaction.
 * This function requires a live transaction object from cryptoStore.doTxn()
 * and therefore may only be called in a doTxn() callback.
 *
 * @param {*} txn Opaque transaction object from cryptoStore.doTxn()
 * @param {function} func
 * @private
 */
OlmDevice.prototype._getAccount = function (txn, func) {
    var _this2 = this;

    this._cryptoStore.getAccount(txn, function (pickledAccount) {
        var account = new Olm.Account();
        try {
            account.unpickle(_this2._pickleKey, pickledAccount);
            func(account);
        } finally {
            account.free();
        }
    });
};

/*
 * Saves an account to the crypto store.
 * This function requires a live transaction object from cryptoStore.doTxn()
 * and therefore may only be called in a doTxn() callback.
 *
 * @param {*} txn Opaque transaction object from cryptoStore.doTxn()
 * @param {object} Olm.Account object
 * @private
 */
OlmDevice.prototype._storeAccount = function (txn, account) {
    this._cryptoStore.storeAccount(txn, account.pickle(this._pickleKey));
};

/**
 * extract an OlmSession from the session store and call the given function
 * The session is useable only within the callback passed to this
 * function and will be freed as soon the callback returns. It is *not*
 * useable for the rest of the lifetime of the transaction.
 *
 * @param {string} deviceKey
 * @param {string} sessionId
 * @param {*} txn Opaque transaction object from cryptoStore.doTxn()
 * @param {function} func
 * @private
 */
OlmDevice.prototype._getSession = function (deviceKey, sessionId, txn, func) {
    var _this3 = this;

    this._cryptoStore.getEndToEndSession(deviceKey, sessionId, txn, function (pickledSession) {
        _this3._unpickleSession(pickledSession, func);
    });
};

/**
 * Creates a session object from a session pickle and executes the given
 * function with it. The session object is destroyed once the function
 * returns.
 *
 * @param {string} pickledSession
 * @param {function} func
 * @private
 */
OlmDevice.prototype._unpickleSession = function (pickledSession, func) {
    var session = new Olm.Session();
    try {
        session.unpickle(this._pickleKey, pickledSession);
        func(session);
    } finally {
        session.free();
    }
};

/**
 * store our OlmSession in the session store
 *
 * @param {string} deviceKey
 * @param {OlmSession} session
 * @param {*} txn Opaque transaction object from cryptoStore.doTxn()
 * @private
 */
OlmDevice.prototype._saveSession = function (deviceKey, session, txn) {
    var pickledSession = session.pickle(this._pickleKey);
    this._cryptoStore.storeEndToEndSession(deviceKey, session.session_id(), pickledSession, txn);
};

/**
 * get an OlmUtility and call the given function
 *
 * @param {function} func
 * @return {object} result of func
 * @private
 */
OlmDevice.prototype._getUtility = function (func) {
    var utility = new Olm.Utility();
    try {
        return func(utility);
    } finally {
        utility.free();
    }
};

/**
 * Signs a message with the ed25519 key for this account.
 *
 * @param {string} message  message to be signed
 * @return {Promise<string>} base64-encoded signature
 */
OlmDevice.prototype.sign = function () {
    var _ref4 = (0, _bluebird.coroutine)(_regenerator2.default.mark(function _callee5(message) {
        var _this4 = this;

        var result;
        return _regenerator2.default.wrap(function _callee5$(_context5) {
            while (1) {
                switch (_context5.prev = _context5.next) {
                    case 0:
                        result = void 0;
                        _context5.next = 3;
                        return (0, _bluebird.resolve)(this._cryptoStore.doTxn('readonly', [_indexeddbCryptoStore2.default.STORE_ACCOUNT], function (txn) {
                            _this4._getAccount(txn, function (account) {
                                result = account.sign(message);
                            });
                        }));

                    case 3:
                        return _context5.abrupt("return", result);

                    case 4:
                    case "end":
                        return _context5.stop();
                }
            }
        }, _callee5, this);
    }));

    return function (_x5) {
        return _ref4.apply(this, arguments);
    };
}();

/**
 * Get the current (unused, unpublished) one-time keys for this account.
 *
 * @return {object} one time keys; an object with the single property
 * <tt>curve25519</tt>, which is itself an object mapping key id to Curve25519
 * key.
 */
OlmDevice.prototype.getOneTimeKeys = (0, _bluebird.coroutine)(_regenerator2.default.mark(function _callee6() {
    var _this5 = this;

    var result;
    return _regenerator2.default.wrap(function _callee6$(_context6) {
        while (1) {
            switch (_context6.prev = _context6.next) {
                case 0:
                    result = void 0;
                    _context6.next = 3;
                    return (0, _bluebird.resolve)(this._cryptoStore.doTxn('readonly', [_indexeddbCryptoStore2.default.STORE_ACCOUNT], function (txn) {
                        _this5._getAccount(txn, function (account) {
                            result = JSON.parse(account.one_time_keys());
                        });
                    }));

                case 3:
                    return _context6.abrupt("return", result);

                case 4:
                case "end":
                    return _context6.stop();
            }
        }
    }, _callee6, this);
}));

/**
 * Get the maximum number of one-time keys we can store.
 *
 * @return {number} number of keys
 */
OlmDevice.prototype.maxNumberOfOneTimeKeys = function () {
    return this._maxOneTimeKeys;
};

/**
 * Marks all of the one-time keys as published.
 */
OlmDevice.prototype.markKeysAsPublished = (0, _bluebird.coroutine)(_regenerator2.default.mark(function _callee7() {
    var _this6 = this;

    return _regenerator2.default.wrap(function _callee7$(_context7) {
        while (1) {
            switch (_context7.prev = _context7.next) {
                case 0:
                    _context7.next = 2;
                    return (0, _bluebird.resolve)(this._cryptoStore.doTxn('readwrite', [_indexeddbCryptoStore2.default.STORE_ACCOUNT], function (txn) {
                        _this6._getAccount(txn, function (account) {
                            account.mark_keys_as_published();
                            _this6._storeAccount(txn, account);
                        });
                    }));

                case 2:
                case "end":
                    return _context7.stop();
            }
        }
    }, _callee7, this);
}));

/**
 * Generate some new one-time keys
 *
 * @param {number} numKeys number of keys to generate
 * @return {Promise} Resolved once the account is saved back having generated the keys
 */
OlmDevice.prototype.generateOneTimeKeys = function (numKeys) {
    var _this7 = this;

    return this._cryptoStore.doTxn('readwrite', [_indexeddbCryptoStore2.default.STORE_ACCOUNT], function (txn) {
        _this7._getAccount(txn, function (account) {
            account.generate_one_time_keys(numKeys);
            _this7._storeAccount(txn, account);
        });
    });
};

/**
 * Generate a new outbound session
 *
 * The new session will be stored in the cryptoStore.
 *
 * @param {string} theirIdentityKey remote user's Curve25519 identity key
 * @param {string} theirOneTimeKey  remote user's one-time Curve25519 key
 * @return {string} sessionId for the outbound session.
 */
OlmDevice.prototype.createOutboundSession = function () {
    var _ref7 = (0, _bluebird.coroutine)(_regenerator2.default.mark(function _callee8(theirIdentityKey, theirOneTimeKey) {
        var _this8 = this;

        var newSessionId;
        return _regenerator2.default.wrap(function _callee8$(_context8) {
            while (1) {
                switch (_context8.prev = _context8.next) {
                    case 0:
                        newSessionId = void 0;
                        _context8.next = 3;
                        return (0, _bluebird.resolve)(this._cryptoStore.doTxn('readwrite', [_indexeddbCryptoStore2.default.STORE_ACCOUNT, _indexeddbCryptoStore2.default.STORE_SESSIONS], function (txn) {
                            _this8._getAccount(txn, function (account) {
                                var session = new Olm.Session();
                                try {
                                    session.create_outbound(account, theirIdentityKey, theirOneTimeKey);
                                    newSessionId = session.session_id();
                                    _this8._storeAccount(txn, account);
                                    _this8._saveSession(theirIdentityKey, session, txn);
                                } finally {
                                    session.free();
                                }
                            });
                        }));

                    case 3:
                        return _context8.abrupt("return", newSessionId);

                    case 4:
                    case "end":
                        return _context8.stop();
                }
            }
        }, _callee8, this);
    }));

    return function (_x6, _x7) {
        return _ref7.apply(this, arguments);
    };
}();

/**
 * Generate a new inbound session, given an incoming message
 *
 * @param {string} theirDeviceIdentityKey remote user's Curve25519 identity key
 * @param {number} messageType  messageType field from the received message (must be 0)
 * @param {string} ciphertext base64-encoded body from the received message
 *
 * @return {{payload: string, session_id: string}} decrypted payload, and
 *     session id of new session
 *
 * @raises {Error} if the received message was not valid (for instance, it
 *     didn't use a valid one-time key).
 */
OlmDevice.prototype.createInboundSession = function () {
    var _ref8 = (0, _bluebird.coroutine)(_regenerator2.default.mark(function _callee9(theirDeviceIdentityKey, messageType, ciphertext) {
        var _this9 = this;

        var result;
        return _regenerator2.default.wrap(function _callee9$(_context9) {
            while (1) {
                switch (_context9.prev = _context9.next) {
                    case 0:
                        if (!(messageType !== 0)) {
                            _context9.next = 2;
                            break;
                        }

                        throw new Error("Need messageType == 0 to create inbound session");

                    case 2:
                        result = void 0;
                        _context9.next = 5;
                        return (0, _bluebird.resolve)(this._cryptoStore.doTxn('readwrite', [_indexeddbCryptoStore2.default.STORE_ACCOUNT, _indexeddbCryptoStore2.default.STORE_SESSIONS], function (txn) {
                            _this9._getAccount(txn, function (account) {
                                var session = new Olm.Session();
                                try {
                                    session.create_inbound_from(account, theirDeviceIdentityKey, ciphertext);
                                    account.remove_one_time_keys(session);
                                    _this9._storeAccount(txn, account);

                                    var payloadString = session.decrypt(messageType, ciphertext);

                                    _this9._saveSession(theirDeviceIdentityKey, session, txn);

                                    result = {
                                        payload: payloadString,
                                        session_id: session.session_id()
                                    };
                                } finally {
                                    session.free();
                                }
                            });
                        }));

                    case 5:
                        return _context9.abrupt("return", result);

                    case 6:
                    case "end":
                        return _context9.stop();
                }
            }
        }, _callee9, this);
    }));

    return function (_x8, _x9, _x10) {
        return _ref8.apply(this, arguments);
    };
}();

/**
 * Get a list of known session IDs for the given device
 *
 * @param {string} theirDeviceIdentityKey Curve25519 identity key for the
 *     remote device
 * @return {Promise<string[]>}  a list of known session ids for the device
 */
OlmDevice.prototype.getSessionIdsForDevice = function () {
    var _ref9 = (0, _bluebird.coroutine)(_regenerator2.default.mark(function _callee10(theirDeviceIdentityKey) {
        var _this10 = this;

        var sessionIds;
        return _regenerator2.default.wrap(function _callee10$(_context10) {
            while (1) {
                switch (_context10.prev = _context10.next) {
                    case 0:
                        sessionIds = void 0;
                        _context10.next = 3;
                        return (0, _bluebird.resolve)(this._cryptoStore.doTxn('readonly', [_indexeddbCryptoStore2.default.STORE_SESSIONS], function (txn) {
                            _this10._cryptoStore.getEndToEndSessions(theirDeviceIdentityKey, txn, function (sessions) {
                                sessionIds = (0, _keys2.default)(sessions);
                            });
                        }));

                    case 3:
                        return _context10.abrupt("return", sessionIds);

                    case 4:
                    case "end":
                        return _context10.stop();
                }
            }
        }, _callee10, this);
    }));

    return function (_x11) {
        return _ref9.apply(this, arguments);
    };
}();

/**
 * Get the right olm session id for encrypting messages to the given identity key
 *
 * @param {string} theirDeviceIdentityKey Curve25519 identity key for the
 *     remote device
 * @return {Promise<?string>}  session id, or null if no established session
 */
OlmDevice.prototype.getSessionIdForDevice = function () {
    var _ref10 = (0, _bluebird.coroutine)(_regenerator2.default.mark(function _callee11(theirDeviceIdentityKey) {
        var sessionIds;
        return _regenerator2.default.wrap(function _callee11$(_context11) {
            while (1) {
                switch (_context11.prev = _context11.next) {
                    case 0:
                        _context11.next = 2;
                        return (0, _bluebird.resolve)(this.getSessionIdsForDevice(theirDeviceIdentityKey));

                    case 2:
                        sessionIds = _context11.sent;

                        if (!(sessionIds.length === 0)) {
                            _context11.next = 5;
                            break;
                        }

                        return _context11.abrupt("return", null);

                    case 5:
                        // Use the session with the lowest ID.
                        sessionIds.sort();
                        return _context11.abrupt("return", sessionIds[0]);

                    case 7:
                    case "end":
                        return _context11.stop();
                }
            }
        }, _callee11, this);
    }));

    return function (_x12) {
        return _ref10.apply(this, arguments);
    };
}();

/**
 * Get information on the active Olm sessions for a device.
 * <p>
 * Returns an array, with an entry for each active session. The first entry in
 * the result will be the one used for outgoing messages. Each entry contains
 * the keys 'hasReceivedMessage' (true if the session has received an incoming
 * message and is therefore past the pre-key stage), and 'sessionId'.
 *
 * @param {string} deviceIdentityKey Curve25519 identity key for the device
 * @return {Array.<{sessionId: string, hasReceivedMessage: Boolean}>}
 */
OlmDevice.prototype.getSessionInfoForDevice = function () {
    var _ref11 = (0, _bluebird.coroutine)(_regenerator2.default.mark(function _callee12(deviceIdentityKey) {
        var _this11 = this;

        var info;
        return _regenerator2.default.wrap(function _callee12$(_context12) {
            while (1) {
                switch (_context12.prev = _context12.next) {
                    case 0:
                        info = [];
                        _context12.next = 3;
                        return (0, _bluebird.resolve)(this._cryptoStore.doTxn('readonly', [_indexeddbCryptoStore2.default.STORE_SESSIONS], function (txn) {
                            _this11._cryptoStore.getEndToEndSessions(deviceIdentityKey, txn, function (sessions) {
                                var sessionIds = (0, _keys2.default)(sessions).sort();
                                var _iteratorNormalCompletion4 = true;
                                var _didIteratorError4 = false;
                                var _iteratorError4 = undefined;

                                try {
                                    var _loop = function _loop() {
                                        var sessionId = _step4.value;

                                        _this11._unpickleSession(sessions[sessionId], function (session) {
                                            info.push({
                                                hasReceivedMessage: session.has_received_message(),
                                                sessionId: sessionId
                                            });
                                        });
                                    };

                                    for (var _iterator4 = (0, _getIterator3.default)(sessionIds), _step4; !(_iteratorNormalCompletion4 = (_step4 = _iterator4.next()).done); _iteratorNormalCompletion4 = true) {
                                        _loop();
                                    }
                                } catch (err) {
                                    _didIteratorError4 = true;
                                    _iteratorError4 = err;
                                } finally {
                                    try {
                                        if (!_iteratorNormalCompletion4 && _iterator4.return) {
                                            _iterator4.return();
                                        }
                                    } finally {
                                        if (_didIteratorError4) {
                                            throw _iteratorError4;
                                        }
                                    }
                                }
                            });
                        }));

                    case 3:
                        return _context12.abrupt("return", info);

                    case 4:
                    case "end":
                        return _context12.stop();
                }
            }
        }, _callee12, this);
    }));

    return function (_x13) {
        return _ref11.apply(this, arguments);
    };
}();

/**
 * Encrypt an outgoing message using an existing session
 *
 * @param {string} theirDeviceIdentityKey Curve25519 identity key for the
 *     remote device
 * @param {string} sessionId  the id of the active session
 * @param {string} payloadString  payload to be encrypted and sent
 *
 * @return {Promise<string>} ciphertext
 */
OlmDevice.prototype.encryptMessage = function () {
    var _ref12 = (0, _bluebird.coroutine)(_regenerator2.default.mark(function _callee13(theirDeviceIdentityKey, sessionId, payloadString) {
        var _this12 = this;

        var res;
        return _regenerator2.default.wrap(function _callee13$(_context13) {
            while (1) {
                switch (_context13.prev = _context13.next) {
                    case 0:
                        checkPayloadLength(payloadString);

                        res = void 0;
                        _context13.next = 4;
                        return (0, _bluebird.resolve)(this._cryptoStore.doTxn('readwrite', [_indexeddbCryptoStore2.default.STORE_SESSIONS], function (txn) {
                            _this12._getSession(theirDeviceIdentityKey, sessionId, txn, function (session) {
                                res = session.encrypt(payloadString);
                                _this12._saveSession(theirDeviceIdentityKey, session, txn);
                            });
                        }));

                    case 4:
                        return _context13.abrupt("return", res);

                    case 5:
                    case "end":
                        return _context13.stop();
                }
            }
        }, _callee13, this);
    }));

    return function (_x14, _x15, _x16) {
        return _ref12.apply(this, arguments);
    };
}();

/**
 * Decrypt an incoming message using an existing session
 *
 * @param {string} theirDeviceIdentityKey Curve25519 identity key for the
 *     remote device
 * @param {string} sessionId  the id of the active session
 * @param {number} messageType  messageType field from the received message
 * @param {string} ciphertext base64-encoded body from the received message
 *
 * @return {Promise<string>} decrypted payload.
 */
OlmDevice.prototype.decryptMessage = function () {
    var _ref13 = (0, _bluebird.coroutine)(_regenerator2.default.mark(function _callee14(theirDeviceIdentityKey, sessionId, messageType, ciphertext) {
        var _this13 = this;

        var payloadString;
        return _regenerator2.default.wrap(function _callee14$(_context14) {
            while (1) {
                switch (_context14.prev = _context14.next) {
                    case 0:
                        payloadString = void 0;
                        _context14.next = 3;
                        return (0, _bluebird.resolve)(this._cryptoStore.doTxn('readwrite', [_indexeddbCryptoStore2.default.STORE_SESSIONS], function (txn) {
                            _this13._getSession(theirDeviceIdentityKey, sessionId, txn, function (session) {
                                payloadString = session.decrypt(messageType, ciphertext);
                                _this13._saveSession(theirDeviceIdentityKey, session, txn);
                            });
                        }));

                    case 3:
                        return _context14.abrupt("return", payloadString);

                    case 4:
                    case "end":
                        return _context14.stop();
                }
            }
        }, _callee14, this);
    }));

    return function (_x17, _x18, _x19, _x20) {
        return _ref13.apply(this, arguments);
    };
}();

/**
 * Determine if an incoming messages is a prekey message matching an existing session
 *
 * @param {string} theirDeviceIdentityKey Curve25519 identity key for the
 *     remote device
 * @param {string} sessionId  the id of the active session
 * @param {number} messageType  messageType field from the received message
 * @param {string} ciphertext base64-encoded body from the received message
 *
 * @return {Promise<boolean>} true if the received message is a prekey message which matches
 *    the given session.
 */
OlmDevice.prototype.matchesSession = function () {
    var _ref14 = (0, _bluebird.coroutine)(_regenerator2.default.mark(function _callee15(theirDeviceIdentityKey, sessionId, messageType, ciphertext) {
        var _this14 = this;

        var matches;
        return _regenerator2.default.wrap(function _callee15$(_context15) {
            while (1) {
                switch (_context15.prev = _context15.next) {
                    case 0:
                        if (!(messageType !== 0)) {
                            _context15.next = 2;
                            break;
                        }

                        return _context15.abrupt("return", false);

                    case 2:
                        matches = void 0;
                        _context15.next = 5;
                        return (0, _bluebird.resolve)(this._cryptoStore.doTxn('readonly', [_indexeddbCryptoStore2.default.STORE_SESSIONS], function (txn) {
                            _this14._getSession(theirDeviceIdentityKey, sessionId, txn, function (session) {
                                matches = session.matches_inbound(ciphertext);
                            });
                        }));

                    case 5:
                        return _context15.abrupt("return", matches);

                    case 6:
                    case "end":
                        return _context15.stop();
                }
            }
        }, _callee15, this);
    }));

    return function (_x21, _x22, _x23, _x24) {
        return _ref14.apply(this, arguments);
    };
}();

// Outbound group session
// ======================

/**
 * store an OutboundGroupSession in _outboundGroupSessionStore
 *
 * @param {Olm.OutboundGroupSession} session
 * @private
 */
OlmDevice.prototype._saveOutboundGroupSession = function (session) {
    var pickledSession = session.pickle(this._pickleKey);
    this._outboundGroupSessionStore[session.session_id()] = pickledSession;
};

/**
 * extract an OutboundGroupSession from _outboundGroupSessionStore and call the
 * given function
 *
 * @param {string} sessionId
 * @param {function} func
 * @return {object} result of func
 * @private
 */
OlmDevice.prototype._getOutboundGroupSession = function (sessionId, func) {
    var pickled = this._outboundGroupSessionStore[sessionId];
    if (pickled === null) {
        throw new Error("Unknown outbound group session " + sessionId);
    }

    var session = new Olm.OutboundGroupSession();
    try {
        session.unpickle(this._pickleKey, pickled);
        return func(session);
    } finally {
        session.free();
    }
};

/**
 * Generate a new outbound group session
 *
 * @return {string} sessionId for the outbound session.
 */
OlmDevice.prototype.createOutboundGroupSession = function () {
    var session = new Olm.OutboundGroupSession();
    try {
        session.create();
        this._saveOutboundGroupSession(session);
        return session.session_id();
    } finally {
        session.free();
    }
};

/**
 * Encrypt an outgoing message with an outbound group session
 *
 * @param {string} sessionId  the id of the outboundgroupsession
 * @param {string} payloadString  payload to be encrypted and sent
 *
 * @return {string} ciphertext
 */
OlmDevice.prototype.encryptGroupMessage = function (sessionId, payloadString) {
    var self = this;

    checkPayloadLength(payloadString);

    return this._getOutboundGroupSession(sessionId, function (session) {
        var res = session.encrypt(payloadString);
        self._saveOutboundGroupSession(session);
        return res;
    });
};

/**
 * Get the session keys for an outbound group session
 *
 * @param {string} sessionId  the id of the outbound group session
 *
 * @return {{chain_index: number, key: string}} current chain index, and
 *     base64-encoded secret key.
 */
OlmDevice.prototype.getOutboundGroupSessionKey = function (sessionId) {
    return this._getOutboundGroupSession(sessionId, function (session) {
        return {
            chain_index: session.message_index(),
            key: session.session_key()
        };
    });
};

// Inbound group session
// =====================

/**
 * data stored in the session store about an inbound group session
 *
 * @typedef {Object} InboundGroupSessionData
 * @property {string} room_Id
 * @property {string} session   pickled Olm.InboundGroupSession
 * @property {Object<string, string>} keysClaimed
 * @property {Array<string>} forwardingCurve25519KeyChain  Devices involved in forwarding
 *     this session to us (normally empty).
 */

/**
 * Unpickle a session from a sessionData object and invoke the given function.
 * The session is valid only until func returns.
 *
 * @param {Object} sessionData Object describing the session.
 * @param {function(Olm.InboundGroupSession)} func Invoked with the unpickled session
 * @return {*} result of func
 */
OlmDevice.prototype._unpickleInboundGroupSession = function (sessionData, func) {
    var session = new Olm.InboundGroupSession();
    try {
        session.unpickle(this._pickleKey, sessionData.session);
        return func(session);
    } finally {
        session.free();
    }
};

/**
 * extract an InboundGroupSession from the crypto store and call the given function
 *
 * @param {string} roomId The room ID to extract the session for, or null to fetch
 *     sessions for any room.
 * @param {string} senderKey
 * @param {string} sessionId
 * @param {*} txn Opaque transaction object from cryptoStore.doTxn()
 * @param {function(Olm.InboundGroupSession, InboundGroupSessionData)} func
 *   function to call.
 *
 * @private
 */
OlmDevice.prototype._getInboundGroupSession = function (roomId, senderKey, sessionId, txn, func) {
    var _this15 = this;

    this._cryptoStore.getEndToEndInboundGroupSession(senderKey, sessionId, txn, function (sessionData) {
        if (sessionData === null) {
            func(null);
            return;
        }

        // if we were given a room ID, check that the it matches the original one for the session. This stops
        // the HS pretending a message was targeting a different room.
        if (roomId !== null && roomId !== sessionData.room_id) {
            throw new Error("Mismatched room_id for inbound group session (expected " + sessionData.room_id + ", was " + roomId + ")");
        }

        _this15._unpickleInboundGroupSession(sessionData, function (session) {
            func(session, sessionData);
        });
    });
};

/**
 * Add an inbound group session to the session store
 *
 * @param {string} roomId     room in which this session will be used
 * @param {string} senderKey  base64-encoded curve25519 key of the sender
 * @param {Array<string>} forwardingCurve25519KeyChain  Devices involved in forwarding
 *     this session to us.
 * @param {string} sessionId  session identifier
 * @param {string} sessionKey base64-encoded secret key
 * @param {Object<string, string>} keysClaimed Other keys the sender claims.
 * @param {boolean} exportFormat true if the megolm keys are in export format
 *    (ie, they lack an ed25519 signature)
 */
OlmDevice.prototype.addInboundGroupSession = function () {
    var _ref15 = (0, _bluebird.coroutine)(_regenerator2.default.mark(function _callee16(roomId, senderKey, forwardingCurve25519KeyChain, sessionId, sessionKey, keysClaimed, exportFormat) {
        var _this16 = this;

        return _regenerator2.default.wrap(function _callee16$(_context16) {
            while (1) {
                switch (_context16.prev = _context16.next) {
                    case 0:
                        _context16.next = 2;
                        return (0, _bluebird.resolve)(this._cryptoStore.doTxn('readwrite', [_indexeddbCryptoStore2.default.STORE_INBOUND_GROUP_SESSIONS], function (txn) {
                            /* if we already have this session, consider updating it */
                            _this16._getInboundGroupSession(roomId, senderKey, sessionId, txn, function (existingSession, existingSessionData) {
                                if (existingSession) {
                                    console.log("Update for megolm session " + senderKey + "/" + sessionId);
                                    // for now we just ignore updates. TODO: implement something here
                                    return;
                                }

                                // new session.
                                var session = new Olm.InboundGroupSession();
                                try {
                                    if (exportFormat) {
                                        session.import_session(sessionKey);
                                    } else {
                                        session.create(sessionKey);
                                    }
                                    if (sessionId != session.session_id()) {
                                        throw new Error("Mismatched group session ID from senderKey: " + senderKey);
                                    }

                                    var sessionData = {
                                        room_id: roomId,
                                        session: session.pickle(_this16._pickleKey),
                                        keysClaimed: keysClaimed,
                                        forwardingCurve25519KeyChain: forwardingCurve25519KeyChain
                                    };

                                    _this16._cryptoStore.addEndToEndInboundGroupSession(senderKey, sessionId, sessionData, txn);
                                } finally {
                                    session.free();
                                }
                            });
                        }));

                    case 2:
                    case "end":
                        return _context16.stop();
                }
            }
        }, _callee16, this);
    }));

    return function (_x25, _x26, _x27, _x28, _x29, _x30, _x31) {
        return _ref15.apply(this, arguments);
    };
}();

/**
 * Decrypt a received message with an inbound group session
 *
 * @param {string} roomId    room in which the message was received
 * @param {string} senderKey base64-encoded curve25519 key of the sender
 * @param {string} sessionId session identifier
 * @param {string} body      base64-encoded body of the encrypted message
 * @param {string} eventId   ID of the event being decrypted
 * @param {Number} timestamp timestamp of the event being decrypted
 *
 * @return {null} the sessionId is unknown
 *
 * @return {Promise<{result: string, senderKey: string,
 *    forwardingCurve25519KeyChain: Array<string>,
 *    keysClaimed: Object<string, string>}>}
 */
OlmDevice.prototype.decryptGroupMessage = function () {
    var _ref16 = (0, _bluebird.coroutine)(_regenerator2.default.mark(function _callee17(roomId, senderKey, sessionId, body, eventId, timestamp) {
        var _this17 = this;

        var result;
        return _regenerator2.default.wrap(function _callee17$(_context17) {
            while (1) {
                switch (_context17.prev = _context17.next) {
                    case 0:
                        result = void 0;
                        _context17.next = 3;
                        return (0, _bluebird.resolve)(this._cryptoStore.doTxn('readwrite', [_indexeddbCryptoStore2.default.STORE_INBOUND_GROUP_SESSIONS], function (txn) {
                            _this17._getInboundGroupSession(roomId, senderKey, sessionId, txn, function (session, sessionData) {
                                if (session === null) {
                                    result = null;
                                    return;
                                }
                                var res = session.decrypt(body);

                                var plaintext = res.plaintext;
                                if (plaintext === undefined) {
                                    // Compatibility for older olm versions.
                                    plaintext = res;
                                } else {
                                    // Check if we have seen this message index before to detect replay attacks.
                                    // If the event ID and timestamp are specified, and the match the event ID
                                    // and timestamp from the last time we used this message index, then we
                                    // don't consider it a replay attack.
                                    var messageIndexKey = senderKey + "|" + sessionId + "|" + res.message_index;
                                    if (messageIndexKey in _this17._inboundGroupSessionMessageIndexes) {
                                        var msgInfo = _this17._inboundGroupSessionMessageIndexes[messageIndexKey];
                                        if (msgInfo.id !== eventId || msgInfo.timestamp !== timestamp) {
                                            throw new Error("Duplicate message index, possible replay attack: " + messageIndexKey);
                                        }
                                    }
                                    _this17._inboundGroupSessionMessageIndexes[messageIndexKey] = {
                                        id: eventId,
                                        timestamp: timestamp
                                    };
                                }

                                sessionData.session = session.pickle(_this17._pickleKey);
                                _this17._cryptoStore.storeEndToEndInboundGroupSession(senderKey, sessionId, sessionData, txn);
                                result = {
                                    result: plaintext,
                                    keysClaimed: sessionData.keysClaimed || {},
                                    senderKey: senderKey,
                                    forwardingCurve25519KeyChain: sessionData.forwardingCurve25519KeyChain || []
                                };
                            });
                        }));

                    case 3:
                        return _context17.abrupt("return", result);

                    case 4:
                    case "end":
                        return _context17.stop();
                }
            }
        }, _callee17, this);
    }));

    return function (_x32, _x33, _x34, _x35, _x36, _x37) {
        return _ref16.apply(this, arguments);
    };
}();

/**
 * Determine if we have the keys for a given megolm session
 *
 * @param {string} roomId    room in which the message was received
 * @param {string} senderKey base64-encoded curve25519 key of the sender
 * @param {sring} sessionId session identifier
 *
 * @returns {Promise<boolean>} true if we have the keys to this session
 */
OlmDevice.prototype.hasInboundSessionKeys = function () {
    var _ref17 = (0, _bluebird.coroutine)(_regenerator2.default.mark(function _callee18(roomId, senderKey, sessionId) {
        var _this18 = this;

        var result;
        return _regenerator2.default.wrap(function _callee18$(_context18) {
            while (1) {
                switch (_context18.prev = _context18.next) {
                    case 0:
                        result = void 0;
                        _context18.next = 3;
                        return (0, _bluebird.resolve)(this._cryptoStore.doTxn('readonly', [_indexeddbCryptoStore2.default.STORE_INBOUND_GROUP_SESSIONS], function (txn) {
                            _this18._cryptoStore.getEndToEndInboundGroupSession(senderKey, sessionId, txn, function (sessionData) {
                                if (sessionData === null) {
                                    result = false;
                                    return;
                                }

                                if (roomId !== sessionData.room_id) {
                                    console.warn("requested keys for inbound group session " + senderKey + "|" + (sessionId + ", with incorrect room_id ") + ("(expected " + sessionData.room_id + ", ") + ("was " + roomId + ")"));
                                    result = false;
                                } else {
                                    result = true;
                                }
                            });
                        }));

                    case 3:
                        return _context18.abrupt("return", result);

                    case 4:
                    case "end":
                        return _context18.stop();
                }
            }
        }, _callee18, this);
    }));

    return function (_x38, _x39, _x40) {
        return _ref17.apply(this, arguments);
    };
}();

/**
 * Extract the keys to a given megolm session, for sharing
 *
 * @param {string} roomId    room in which the message was received
 * @param {string} senderKey base64-encoded curve25519 key of the sender
 * @param {string} sessionId session identifier
 *
 * @returns {Promise<{chain_index: number, key: string,
 *        forwarding_curve25519_key_chain: Array<string>,
 *        sender_claimed_ed25519_key: string
 *    }>}
 *    details of the session key. The key is a base64-encoded megolm key in
 *    export format.
 */
OlmDevice.prototype.getInboundGroupSessionKey = function () {
    var _ref18 = (0, _bluebird.coroutine)(_regenerator2.default.mark(function _callee19(roomId, senderKey, sessionId) {
        var _this19 = this;

        var result;
        return _regenerator2.default.wrap(function _callee19$(_context19) {
            while (1) {
                switch (_context19.prev = _context19.next) {
                    case 0:
                        result = void 0;
                        _context19.next = 3;
                        return (0, _bluebird.resolve)(this._cryptoStore.doTxn('readonly', [_indexeddbCryptoStore2.default.STORE_INBOUND_GROUP_SESSIONS], function (txn) {
                            _this19._getInboundGroupSession(roomId, senderKey, sessionId, txn, function (session, sessionData) {
                                if (session === null) {
                                    result = null;
                                    return;
                                }
                                var messageIndex = session.first_known_index();

                                var claimedKeys = sessionData.keysClaimed || {};
                                var senderEd25519Key = claimedKeys.ed25519 || null;

                                result = {
                                    "chain_index": messageIndex,
                                    "key": session.export_session(messageIndex),
                                    "forwarding_curve25519_key_chain": sessionData.forwardingCurve25519KeyChain || [],
                                    "sender_claimed_ed25519_key": senderEd25519Key
                                };
                            });
                        }));

                    case 3:
                        return _context19.abrupt("return", result);

                    case 4:
                    case "end":
                        return _context19.stop();
                }
            }
        }, _callee19, this);
    }));

    return function (_x41, _x42, _x43) {
        return _ref18.apply(this, arguments);
    };
}();

/**
 * Export an inbound group session
 *
 * @param {string} senderKey base64-encoded curve25519 key of the sender
 * @param {string} sessionId session identifier
 * @param {string} sessionData The session object from the store
 * @return {module:crypto/OlmDevice.MegolmSessionData} exported session data
 */
OlmDevice.prototype.exportInboundGroupSession = function (senderKey, sessionId, sessionData) {
    return this._unpickleInboundGroupSession(sessionData, function (session) {
        var messageIndex = session.first_known_index();

        return {
            "sender_key": senderKey,
            "sender_claimed_keys": sessionData.keysClaimed,
            "room_id": sessionData.room_id,
            "session_id": sessionId,
            "session_key": session.export_session(messageIndex),
            "forwarding_curve25519_key_chain": session.forwardingCurve25519KeyChain || []
        };
    });
};

// Utilities
// =========

/**
 * Verify an ed25519 signature.
 *
 * @param {string} key ed25519 key
 * @param {string} message message which was signed
 * @param {string} signature base64-encoded signature to be checked
 *
 * @raises {Error} if there is a problem with the verification. If the key was
 * too small then the message will be "OLM.INVALID_BASE64". If the signature
 * was invalid then the message will be "OLM.BAD_MESSAGE_MAC".
 */
OlmDevice.prototype.verifySignature = function (key, message, signature) {
    this._getUtility(function (util) {
        util.ed25519_verify(key, message, signature);
    });
};

/** */
module.exports = OlmDevice;
//# sourceMappingURL=OlmDevice.js.map