"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const debug_1 = require("debug");
const debug = debug_1.default('axm:profilingaction');
const profiling_1 = require("../features/profiling");
const miscellaneous_1 = require("../utils/miscellaneous");
const serviceManager_1 = require("../serviceManager");
class ProfilingHeapAction {
    constructor(actionFeature, config) {
        this.config = config;
        if (!config) {
            this.config = {};
        }
        this.actionFeature = actionFeature;
    }
    init() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            this.profilingFeature = new profiling_1.default();
            this.profilings = this.profilingFeature.init();
            try {
                yield this.profilings.heapProfiling.init(this.config.heap);
                this.exposeActions();
            }
            catch (err) {
                debug(`Failed to load heap profiler: ${err.message}`);
            }
        });
    }
    destroy() {
        if (this.profilingFeature)
            this.profilingFeature.destroy();
    }
    stopProfiling(reply) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const data = yield this.profilings.heapProfiling.stop();
                return reply({
                    success: true,
                    heapprofile: true,
                    dump_file: data,
                    dump_file_size: data.length,
                    uuid: this.uuid
                });
            }
            catch (err) {
                return reply({
                    success: false,
                    err: err,
                    uuid: this.uuid
                });
            }
        });
    }
    exposeActions() {
        // -------------------------------------
        // Heap sampling
        // -------------------------------------
        const profilingReply = (data) => serviceManager_1.ServiceManager.get('transport').send('profilings', {
            data: data.dump_file,
            at: data.at,
            initiated: data.initiated || 'manual',
            duration: data.duration || null,
            type: 'heapprofile'
        });
        let startTime = null;
        this.actionFeature.action('km:heap:sampling:start', (opts, reply) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            startTime = new Date();
            if (!reply) {
                reply = opts;
                opts = {};
            }
            if (!opts)
                opts = {};
            try {
                this.uuid = miscellaneous_1.default.generateUUID();
                yield this.profilings.heapProfiling.start();
                reply({ success: true, uuid: this.uuid });
                if (opts.timeout && typeof opts.timeout === 'number') {
                    setTimeout((_) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                        yield this.stopProfiling(data => profilingReply(Object.assign({ at: startTime, initiated: opts.initiated, duration: startTime ? new Date().getTime() - startTime.getTime() : null }, data)));
                    }), opts.timeout);
                }
            }
            catch (err) {
                return reply({
                    success: false,
                    err: err,
                    uuid: this.uuid
                });
            }
        }));
        this.actionFeature.action('km:heap:sampling:stop', this.stopProfiling.bind(this, data => profilingReply(Object.assign({ at: startTime, initiated: 'manual', duration: startTime ? new Date().getTime() - startTime.getTime() : null }, data))));
        // -------------------------------------
        // Heap dump
        // -------------------------------------
        this.actionFeature.action('km:heapdump', (opts, reply) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (!reply) {
                reply = opts;
                opts = {};
            }
            if (!opts)
                opts = {};
            const startTime = new Date();
            try {
                const data = yield this.profilings.heapProfiling.takeSnapshot();
                return serviceManager_1.ServiceManager.get('transport').send('profilings', {
                    data,
                    at: startTime,
                    initiated: opts.initiated || 'manual',
                    duration: new Date().getTime() - startTime.getTime(),
                    type: 'heapdump'
                });
            }
            catch (err) {
                return reply({
                    success: false,
                    err: err
                });
            }
        }));
    }
}
exports.default = ProfilingHeapAction;
//# sourceMappingURL=data:application/json;base64,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