"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var BinaryHeap_1 = require("./BinaryHeap");
var units_1 = require("./units");
var ExponentiallyDecayingSample = (function () {
    function ExponentiallyDecayingSample(options) {
        this.RESCALE_INTERVAL = 1 * units_1.default.HOURS;
        this.ALPHA = 0.015;
        this.SIZE = 1028;
        options = options || {};
        this._elements = new BinaryHeap_1.default({
            score: function (element) {
                return -element.priority;
            }
        });
        this._rescaleInterval = options.rescaleInterval || this.RESCALE_INTERVAL;
        this._alpha = options.alpha || this.ALPHA;
        this._size = options.size || this.SIZE;
        this._random = options.random || this._random;
        this._landmark = null;
        this._nextRescale = null;
        this._mean = null;
    }
    ExponentiallyDecayingSample.prototype.update = function (value, timestamp) {
        var now = Date.now();
        if (!this._landmark) {
            this._landmark = now;
            this._nextRescale = this._landmark + this._rescaleInterval;
        }
        timestamp = timestamp || now;
        var newSize = this._elements.size() + 1;
        var element = {
            priority: this._priority(timestamp - this._landmark),
            value: value
        };
        if (newSize <= this._size) {
            this._elements.add(element);
        }
        else if (element.priority > this._elements.first().priority) {
            this._elements.removeFirst();
            this._elements.add(element);
        }
        if (now >= this._nextRescale)
            this._rescale(now);
    };
    ExponentiallyDecayingSample.prototype.toSortedArray = function () {
        return this._elements
            .toSortedArray()
            .map(function (element) {
            return element.value;
        });
    };
    ExponentiallyDecayingSample.prototype.toArray = function () {
        return this._elements
            .toArray()
            .map(function (element) {
            return element.value;
        });
    };
    ExponentiallyDecayingSample.prototype._weight = function (age) {
        return Math.exp(this._alpha * (age / 1000));
    };
    ExponentiallyDecayingSample.prototype._priority = function (age) {
        return this._weight(age) / this._random();
    };
    ExponentiallyDecayingSample.prototype._random = function () {
        return Math.random();
    };
    ExponentiallyDecayingSample.prototype._rescale = function (now) {
        now = now || Date.now();
        var self = this;
        var oldLandmark = this._landmark;
        this._landmark = now || Date.now();
        this._nextRescale = now + this._rescaleInterval;
        var factor = self._priority(-(self._landmark - oldLandmark));
        this._elements
            .toArray()
            .forEach(function (element) {
            element.priority *= factor;
        });
    };
    ExponentiallyDecayingSample.prototype.avg = function (now) {
        var sum = 0;
        this._elements
            .toArray()
            .forEach(function (element) {
            sum += element.value;
        });
        return (sum / this._elements.size());
    };
    return ExponentiallyDecayingSample;
}());
exports.default = ExponentiallyDecayingSample;
//# sourceMappingURL=data:application/json;base64,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