"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var tslib_1 = require("tslib");
var configuration_1 = require("../configuration");
var serviceManager_1 = require("../serviceManager");
var miscellaneous_1 = require("../utils/miscellaneous");
var Debug = require("debug");
var semver = require("semver");
var CurrentProfile = (function () {
    function CurrentProfile() {
    }
    return CurrentProfile;
}());
var InspectorProfiler = (function () {
    function InspectorProfiler() {
        this.profiler = undefined;
        this.currentProfile = null;
        this.logger = Debug('axm:features:profiling:inspector');
        this.isNode11 = semver.satisfies(semver.clean(process.version), '>11.x');
    }
    InspectorProfiler.prototype.init = function () {
        this.profiler = serviceManager_1.ServiceManager.get('inspector');
        if (this.profiler === undefined) {
            configuration_1.default.configureModule({
                heapdump: false,
                'feature.profiler.heap_snapshot': false,
                'feature.profiler.heap_sampling': false,
                'feature.profiler.cpu_js': false
            });
            return console.error("Failed to require the profiler via inspector, disabling profiling ...");
        }
        this.profiler.getSession().post('Profiler.enable');
        this.profiler.getSession().post('HeapProfiler.enable');
        this.logger('init');
        this.actionService = serviceManager_1.ServiceManager.get('actions');
        if (this.actionService === undefined) {
            return this.logger("Fail to get action service");
        }
        this.transport = serviceManager_1.ServiceManager.get('transport');
        if (this.transport === undefined) {
            return this.logger("Fail to get transport service");
        }
        configuration_1.default.configureModule({
            heapdump: true,
            'feature.profiler.heapsnapshot': !this.isNode11,
            'feature.profiler.heapsampling': true,
            'feature.profiler.cpu_js': true
        });
        this.register();
    };
    InspectorProfiler.prototype.register = function () {
        if (this.actionService === undefined) {
            return this.logger("Fail to get action service");
        }
        this.logger('register');
        this.actionService.registerAction('km:heapdump', this.onHeapdump.bind(this));
        this.actionService.registerAction('km:cpu:profiling:start', this.onCPUProfileStart.bind(this));
        this.actionService.registerAction('km:cpu:profiling:stop', this.onCPUProfileStop.bind(this));
        this.actionService.registerAction('km:heap:sampling:start', this.onHeapProfileStart.bind(this));
        this.actionService.registerAction('km:heap:sampling:stop', this.onHeapProfileStop.bind(this));
    };
    InspectorProfiler.prototype.destroy = function () {
        this.logger('Inspector Profiler destroyed !');
        if (this.profiler === undefined)
            return;
        this.profiler.getSession().post('Profiler.disable');
        this.profiler.getSession().post('HeapProfiler.disable');
    };
    InspectorProfiler.prototype.onHeapProfileStart = function (opts, cb) {
        var _this = this;
        if (typeof cb !== 'function') {
            cb = opts;
            opts = {};
        }
        if (typeof opts !== 'object' || opts === null) {
            opts = {};
        }
        if (this.profiler === undefined) {
            return cb({
                err: 'Profiler not available',
                success: false
            });
        }
        if (this.currentProfile !== null) {
            return cb({
                err: 'A profiling is already running',
                success: false
            });
        }
        this.currentProfile = new CurrentProfile();
        this.currentProfile.uuid = miscellaneous_1.default.generateUUID();
        this.currentProfile.startTime = Date.now();
        this.currentProfile.initiated = typeof opts.initiated === 'string'
            ? opts.initiated : 'manual';
        cb({ success: true, uuid: this.currentProfile.uuid });
        var defaultSamplingInterval = 16384;
        this.profiler.getSession().post('HeapProfiler.startSampling', {
            samplingInterval: typeof opts.samplingInterval === 'number'
                ? opts.samplingInterval : defaultSamplingInterval
        });
        if (isNaN(parseInt(opts.timeout, 10)))
            return;
        var duration = parseInt(opts.timeout, 10);
        setTimeout(function (_) {
            _this.onHeapProfileStop(function (_) {
                return;
            });
        }, duration);
    };
    InspectorProfiler.prototype.onHeapProfileStop = function (cb) {
        var _this = this;
        if (this.currentProfile === null) {
            return cb({
                err: 'No profiling are already running',
                success: false
            });
        }
        if (this.profiler === undefined) {
            return cb({
                err: 'Profiler not available',
                success: false
            });
        }
        cb({ success: true, uuid: this.currentProfile.uuid });
        this.profiler.getSession().post('HeapProfiler.stopSampling', function (_, _a) {
            var profile = _a.profile;
            if (_this.currentProfile === null)
                return;
            if (_this.transport === undefined)
                return;
            var data = JSON.stringify(profile);
            _this.transport.send('profilings', {
                uuid: _this.currentProfile.uuid,
                duration: Date.now() - _this.currentProfile.startTime,
                at: _this.currentProfile.startTime,
                data: data,
                success: true,
                initiated: _this.currentProfile.initiated,
                type: 'heapprofile',
                heapprofile: true
            });
            _this.currentProfile = null;
        });
    };
    InspectorProfiler.prototype.onCPUProfileStart = function (opts, cb) {
        var _this = this;
        if (typeof cb !== 'function') {
            cb = opts;
            opts = {};
        }
        if (typeof opts !== 'object' || opts === null) {
            opts = {};
        }
        if (this.profiler === undefined) {
            return cb({
                err: 'Profiler not available',
                success: false
            });
        }
        if (this.currentProfile !== null) {
            return cb({
                err: 'A profiling is already running',
                success: false
            });
        }
        this.currentProfile = new CurrentProfile();
        this.currentProfile.uuid = miscellaneous_1.default.generateUUID();
        this.currentProfile.startTime = Date.now();
        this.currentProfile.initiated = typeof opts.initiated === 'string'
            ? opts.initiated : 'manual';
        cb({ success: true, uuid: this.currentProfile.uuid });
        if (process.hasOwnProperty('_startProfilerIdleNotifier') === true) {
            process._startProfilerIdleNotifier();
        }
        this.profiler.getSession().post('Profiler.start');
        if (isNaN(parseInt(opts.timeout, 10)))
            return;
        var duration = parseInt(opts.timeout, 10);
        setTimeout(function (_) {
            _this.onCPUProfileStop(function (_) {
                return;
            });
        }, duration);
    };
    InspectorProfiler.prototype.onCPUProfileStop = function (cb) {
        var _this = this;
        if (this.currentProfile === null) {
            return cb({
                err: 'No profiling are already running',
                success: false
            });
        }
        if (this.profiler === undefined) {
            return cb({
                err: 'Profiler not available',
                success: false
            });
        }
        cb({ success: true, uuid: this.currentProfile.uuid });
        if (process.hasOwnProperty('_stopProfilerIdleNotifier') === true) {
            process._stopProfilerIdleNotifier();
        }
        this.profiler.getSession().post('Profiler.stop', function (_, res) {
            if (_this.currentProfile === null)
                return;
            if (_this.transport === undefined)
                return;
            var profile = res.profile;
            var data = JSON.stringify(profile);
            _this.transport.send('profilings', {
                uuid: _this.currentProfile.uuid,
                duration: Date.now() - _this.currentProfile.startTime,
                at: _this.currentProfile.startTime,
                data: data,
                success: true,
                initiated: _this.currentProfile.initiated,
                type: 'cpuprofile',
                cpuprofile: true
            });
            _this.currentProfile = null;
        });
    };
    InspectorProfiler.prototype.onHeapdump = function (opts, cb) {
        var _this = this;
        if (typeof cb !== 'function') {
            cb = opts;
            opts = {};
        }
        if (typeof opts !== 'object' || opts === null) {
            opts = {};
        }
        if (this.profiler === undefined) {
            return cb({
                err: 'Profiler not available',
                success: false
            });
        }
        cb({ success: true });
        setTimeout(function () {
            var startTime = Date.now();
            _this.takeSnapshot()
                .then(function (data) {
                return _this.transport.send('profilings', {
                    data: data,
                    at: startTime,
                    initiated: typeof opts.initiated === 'string' ? opts.initiated : 'manual',
                    duration: Date.now() - startTime,
                    type: 'heapdump'
                });
            }).catch(function (err) {
                return cb({
                    success: err.message,
                    err: err
                });
            });
        }, 200);
    };
    InspectorProfiler.prototype.takeSnapshot = function () {
        var _this = this;
        return new Promise(function (resolve, reject) { return tslib_1.__awaiter(_this, void 0, void 0, function () {
            var chunks, chunkHandler;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        if (this.profiler === undefined)
                            return [2, reject(new Error("Profiler not available"))];
                        chunks = [];
                        chunkHandler = function (raw) {
                            var data = raw.params;
                            chunks.push(data.chunk);
                        };
                        this.profiler.getSession().on('HeapProfiler.addHeapSnapshotChunk', chunkHandler);
                        return [4, this.profiler.getSession().post('HeapProfiler.takeHeapSnapshot', {
                                reportProgress: false
                            })];
                    case 1:
                        _a.sent();
                        this.profiler.getSession().removeListener('HeapProfiler.addHeapSnapshotChunk', chunkHandler);
                        return [2, resolve(chunks.join(''))];
                }
            });
        }); });
    };
    return InspectorProfiler;
}());
exports.default = InspectorProfiler;
//# sourceMappingURL=data:application/json;base64,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