"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var EDS_1 = require("../EDS");
var Histogram = (function () {
    function Histogram(opts) {
        this._sample = new EDS_1.default();
        this._count = 0;
        this._sum = 0;
        this._varianceM = 0;
        this._varianceS = 0;
        this._ema = 0;
        this.used = false;
        opts = opts || {};
        this._measurement = opts.measurement;
        this._callFn = null;
        var methods = {
            min: this.getMin,
            max: this.getMax,
            sum: this.getSum,
            count: this.getCount,
            variance: this._calculateVariance,
            mean: this._calculateMean,
            ema: this.getEma()
        };
        if (methods.hasOwnProperty(this._measurement)) {
            this._callFn = methods[this._measurement];
        }
        else {
            this._callFn = function () {
                var percentiles = this.percentiles([0.5, 0.75, 0.95, 0.99, 0.999]);
                var medians = {
                    median: percentiles[0.5],
                    p75: percentiles[0.75],
                    p95: percentiles[0.95],
                    p99: percentiles[0.99],
                    p999: percentiles[0.999]
                };
                return medians[this._measurement];
            };
        }
    }
    Histogram.prototype.update = function (value) {
        this.used = true;
        this._count++;
        this._sum += value;
        this._sample.update(value);
        this._updateMin(value);
        this._updateMax(value);
        this._updateVariance(value);
        this._updateEma(value);
    };
    Histogram.prototype.percentiles = function (percentiles) {
        var values = this._sample
            .toArray()
            .sort(function (a, b) {
            return (a === b)
                ? 0
                : a - b;
        });
        var results = {};
        for (var i = 0; i < percentiles.length; i++) {
            var percentile = percentiles[i];
            if (!values.length) {
                results[percentile] = null;
                continue;
            }
            var pos = percentile * (values.length + 1);
            if (pos < 1) {
                results[percentile] = values[0];
            }
            else if (pos >= values.length) {
                results[percentile] = values[values.length - 1];
            }
            else {
                var lower = values[Math.floor(pos) - 1];
                var upper = values[Math.ceil(pos) - 1];
                results[percentile] = lower + (pos - Math.floor(pos)) * (upper - lower);
            }
        }
        return results;
    };
    Histogram.prototype.val = function () {
        if (typeof (this._callFn) === 'function') {
            return this._callFn();
        }
        else {
            return this._callFn;
        }
    };
    Histogram.prototype.getMin = function () {
        return this._min;
    };
    Histogram.prototype.getMax = function () {
        return this._max;
    };
    Histogram.prototype.getSum = function () {
        return this._sum;
    };
    Histogram.prototype.getCount = function () {
        return this._count;
    };
    Histogram.prototype.getEma = function () {
        return this._ema;
    };
    Histogram.prototype.fullResults = function () {
        var percentiles = this.percentiles([0.5, 0.75, 0.95, 0.99, 0.999]);
        return {
            min: this._min,
            max: this._max,
            sum: this._sum,
            variance: this._calculateVariance(),
            mean: this._calculateMean(),
            count: this._count,
            median: percentiles[0.5],
            p75: percentiles[0.75],
            p95: percentiles[0.95],
            p99: percentiles[0.99],
            p999: percentiles[0.999],
            ema: this._ema
        };
    };
    Histogram.prototype._updateMin = function (value) {
        if (this._min === undefined || value < this._min) {
            this._min = value;
        }
    };
    Histogram.prototype._updateMax = function (value) {
        if (this._max === undefined || value > this._max) {
            this._max = value;
        }
    };
    Histogram.prototype._updateVariance = function (value) {
        if (this._count === 1)
            return this._varianceM = value;
        var oldM = this._varianceM;
        this._varianceM += ((value - oldM) / this._count);
        this._varianceS += ((value - oldM) * (value - this._varianceM));
    };
    Histogram.prototype._updateEma = function (value) {
        if (this._count <= 1)
            return this._ema = this._calculateMean();
        var alpha = 2 / (1 + this._count);
        this._ema = value * alpha + this._ema * (1 - alpha);
    };
    Histogram.prototype._calculateMean = function () {
        return (this._count === 0)
            ? 0
            : this._sum / this._count;
    };
    Histogram.prototype._calculateVariance = function () {
        return (this._count <= 1)
            ? null
            : this._varianceS / (this._count - 1);
    };
    Histogram.prototype.isUsed = function () {
        return this.used;
    };
    return Histogram;
}());
exports.default = Histogram;
//# sourceMappingURL=data:application/json;base64,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