/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.gradle.api.reporting.dependencies.internal;

import com.googlecode.jatl.Html;
import org.gradle.api.Project;
import org.gradle.api.Transformer;
import org.gradle.reporting.HtmlPageBuilder;
import org.gradle.reporting.ReportRenderer;
import org.gradle.util.GradleVersion;

import java.io.IOException;
import java.io.Writer;
import java.util.Date;

public class ProjectPageRenderer extends ReportRenderer<Project, HtmlPageBuilder<Writer>> {
    private final Transformer<String, Project> namingScheme;

    public ProjectPageRenderer(Transformer<String, Project> namingScheme) {
        this.namingScheme = namingScheme;
    }

    @Override
    public void render(final Project project, final HtmlPageBuilder<Writer> builder) throws IOException {
        final String baseCssLink = builder.requireResource(getClass().getResource("/org/gradle/reporting/base-style.css"));
        final String cssLink = builder.requireResource(getClass().getResource(getReportResourcePath("style.css")));
        final String jqueryLink = builder.requireResource(getClass().getResource("/org/gradle/reporting/jquery.min-1.11.0.js"));
        final String jtreeLink = builder.requireResource(getClass().getResource(getReportResourcePath("jquery.jstree.js")));
        final String scriptLink = builder.requireResource(getClass().getResource(getReportResourcePath("script.js")));
        builder.requireResource(getClass().getResource(getReportResourcePath("tree.css")));
        builder.requireResource(getClass().getResource(getReportResourcePath("d.gif")));
        builder.requireResource(getClass().getResource(getReportResourcePath("d.png")));
        builder.requireResource(getClass().getResource(getReportResourcePath("throbber.gif")));

        new Html(builder.getOutput()) {{
            html();
                head();
                    meta().httpEquiv("Content-Type").content("text/html; charset=utf-8");
                    meta().httpEquiv("x-ua-compatible").content("IE=edge");
                    link().rel("stylesheet").type("text/css").href(baseCssLink).end();
                    link().rel("stylesheet").type("text/css").href(cssLink).end();
                    script().src(jqueryLink).charset("utf-8").end();
                    script().src(jtreeLink).charset("utf-8").end();
                    script().src(namingScheme.transform(project)).charset("utf-8").end();
                    script().src(scriptLink).charset("utf-8").end();
                    title().text("Dependency reports").end();
                end();
                body();
                    div().id("content");
                        h1().text("Dependency Report").end();
                        div().classAttr("breadcrumbs");
                            a().href("index.html").text("Projects").end();
                            text(" > ");
                            span().id("projectBreadcrumb").end();
                        end();
                        div().id("insight").end();
                        div().id("dependencies").end();
                        div().id("footer");
                            p();
                                text("Generated by ");
                                a().href("http://www.gradle.org").text(GradleVersion.current().toString()).end();
                                text(" at " + builder.formatDate(new Date()));
                            end();
                        end();
                    end();
                end();
            endAll();
        }};
    }

    private String getReportResourcePath(String fileName) {
        return "/org/gradle/api/tasks/diagnostics/htmldependencyreport/" + fileName;
    }
}
